(*
 *  dcl/dChan.ml
 *
 *  Distributed, type-safe channels
 *
 *  (c) Copyright 2006, John N. Billings <jnb26@cam.ac.uk>
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

let debug = Debug.debug "DChan"
let error = Debug.error "DChan"
let warn = Debug.warn "DChan"

type com_handle_t = LChan.com_handle_t
type recv_handle_t = LChan.recv_handle_t

let do_finally f g = 
  (try f () with e -> g (); raise e); g ()

let wrap_error f = 
  try
    f ()
  with Unix.Unix_error (err, _, _) as e ->
    error (Unix.error_message err);
    raise e

(* Messages *)
type 'a msg = {
  msg_chan : 'a name;
  msg_dat : 'a;
}

module Msg = Exist.Make(struct type 'a t = 'a msg end)
open Msg

(* Invoked by server upon reception of message *)
let recv com_handle fd =
  debug "Received message";
  (* Run in separate thread to prevent exception from bringing down the
   * server *)
  let f () =
    let cin = Unix.in_channel_of_descr fd in
    let msg = Marshal.from_channel cin in
    let g = { f = fun m -> LChan.send com_handle m.msg_chan m.msg_dat } in
    Msg.use g msg in
  let g () = 
    (* Note that it's _really_ important to close each fd after use,
     * otherwise we run out of them rather quickly *)
    Unix.shutdown fd Unix.SHUTDOWN_ALL;
    Unix.close fd in
  ignore (Thread.create wrap_error (fun () -> do_finally f g))

let init ?(inet_addr = Unix.inet_addr_any) ~port () =
  let f () =
    debug "Initialising";
    let com_handle = LChan.init () in
    let tid =
        Server.server ~servlet:(recv com_handle) ~inet_addr:inet_addr
        ~port:port () in
    (tid, com_handle) in
  wrap_error f

let send ~inet_addr ~port ~chan ~dat =
  debug "Sending";
  let sock_addr = Unix.ADDR_INET (inet_addr, port) in
  let fd = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0 in
  let msg = {
    msg_chan = chan;
    msg_dat = dat;
  } in
  let f () =
    Unix.connect fd sock_addr;
    let cout = Unix.out_channel_of_descr fd in
    (* Existentials are functional values *)
    Marshal.to_channel cout (Msg.pack msg) [Marshal.Closures];
    (* Must flush because Unix.close won't. However, we don't call
     * Pervasives.close_out because then Unix.close complains. *)
    flush cout in
  let g () =
    Unix.shutdown fd Unix.SHUTDOWN_ALL;
    Unix.close fd in
  wrap_error (fun () -> do_finally f g)

let register_recv ~com_h ~chan ~recv ~replicate ~timeout =
  LChan.register_recv com_h chan recv replicate timeout

let unregister_recv ~com_h ~recv_h =
  LChan.unregister_recv com_h recv_h
