(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * Type checker                                                   *
 *                                                                *
 * Version: $Id: tysupp.ml,v 1.543 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Ast       (* abstract syntax tree *)
open Util
open Pretty
open Basecon
open Baseconty

(* == EQUALITY FUNCTIONS == *)

(* compare hashes for equality *)
let h_eq : hash -> hash -> bool
  = fun h1 h2
 -> hash_of_hash h1 = hash_of_hash h2

(* compare hashes for inequality, by the lexicographic order on their
hashed representations *)
let h_lteq : hash -> hash -> bool
    = fun h1 h2 ->
      hash_of_hash h1 <= hash_of_hash h2

(* compare hashes *)
let h_cmp : hash -> hash -> int
  = fun h1 h2
 -> compare (hash_of_hash h1) (hash_of_hash h2)

(* compare X for equality *)
let x_eq : hash_or_modname -> hash_or_modname -> bool
  = fun x x'
 -> match (x,x') with
   (Hash(h)    ,Hash(h')    ) -> h_eq h h'
 | (Modname(mn),Modname(mn')) -> mn_eq mn mn'
 | _ -> false

(* compare Xs *)
let x_cmp : hash_or_modname -> hash_or_modname -> int
  = fun x x'
 -> match (x,x') with
   (Hash(h)    ,Hash(h')    ) -> h_cmp h h'
 | (Modname(mn),Modname(mn')) -> mn_compare mn mn'
 | (Hash _     ,Modname _   ) -> -1
 | (Modname _  ,Hash _      ) -> 1


(* compare types for syntactic equality *)
(* maps are substitutions to apply to ty1 and ty2 before comparison *)
let rec typ_eq0 : internal_typname ITypnameMap.t -> internal_typname ITypnameMap.t -> typ -> typ -> bool
  = fun m1 m2 ty1 ty2
 -> match (ty1,ty2) with
    | (TTyCon0 c      , TTyCon0 c'       ) -> c = c'
    | (TTyCon1(c,ty)  , TTyCon1(c',ty')  ) -> c = c' && typ_eq0 m1 m2 ty ty'
    | (TTup(tys)      , TTup(tys')       )
    | (TSum(tys)      , TSum(tys')       ) -> List.length tys = List.length tys'
                                              && List.for_all2 (typ_eq0 m1 m2) tys tys'
    | (TFunc(ty1,ty2) , TFunc(ty1',ty2') ) -> typ_eq0 m1 m2 ty1 ty1' && typ_eq0 m1 m2 ty2 ty2'
    | (TVar(tn0)      , TVar(tn0')       ) -> let tn  = try ITypnameMap.find tn0  m1 with Not_found -> tn0 in
                                              let tn' = try ITypnameMap.find tn0' m2 with Not_found -> tn0' in
                                              tn_eq tn tn'
    | (TXDot(x,etn)   , TXDot(x',etn')   ) -> x_eq x x' && etn = etn'
    | (TTyName(n)     , TTyName(n')      ) -> n = n'
    | (TForall(tn,ty) , TForall(tn',ty') )
    | (TExists(tn,ty) , TExists(tn',ty') ) -> let tn'' = fresh_prettyname tn in
                                              typ_eq0 (ITypnameMap.add tn tn'' m1) (ITypnameMap.add tn' tn'' m2) ty ty'
    | (_              , _                ) -> false

let typ_eq : typ -> typ -> bool
  = fun ty1 ty2 -> typ_eq0 ITypnameMap.empty ITypnameMap.empty ty1 ty2


(* compare types syntactically *)
(* maps are substitutions to apply to ty1 and ty2 before comparison *)
let rec typ_cmp0 : internal_typname ITypnameMap.t -> internal_typname ITypnameMap.t -> typ -> typ -> int
  = fun m1 m2 ty1 ty2
 -> match (ty1,ty2) with
    | (TTyCon0 c      , TTyCon0 c'       ) -> compare c c'
    | (TTyCon1(c,ty)  , TTyCon1(c',ty')  ) -> compare c c'
                                              |==|
                                              (fun () -> typ_cmp0 m1 m2 ty ty')
    | (TTup(tys)      , TTup(tys')       )
    | (TSum(tys)      , TSum(tys')       ) -> compare (List.length tys) (List.length tys')
                                              |==|
                                              (fun () -> try compare_list (List.map2 (typ_cmp0 m1 m2) tys tys') with
                                                Invalid_argument _ -> raise (Never_happen "typ_cmp"))
    | (TFunc(ty1,ty2) , TFunc(ty1',ty2') ) -> typ_cmp0 m1 m2 ty1 ty1' |=| typ_cmp0 m1 m2 ty2 ty2'
    | (TVar(tn0)      , TVar(tn0')       ) -> let tn  = try ITypnameMap.find tn0  m1 with Not_found -> tn0 in
                                              let tn' = try ITypnameMap.find tn0' m2 with Not_found -> tn0' in
                                              tn_compare tn tn'
    | (TXDot(x,etn)   , TXDot(x',etn')   ) -> x_cmp x x' |==| (fun () -> compare etn etn')
    | (TTyName(n)     , TTyName(n')      ) -> compare n n'
    | (TForall(tn,ty) , TForall(tn',ty') )
    | (TExists(tn,ty) , TExists(tn',ty') ) -> let tn'' = fresh_prettyname tn in
                                              typ_cmp0 (ITypnameMap.add tn tn'' m1) (ITypnameMap.add tn' tn'' m2) ty ty'
    | (_              , _                ) -> compare ty1 ty2  (* assume OCaml checks constructor ordering first before args *)

let rec typ_cmp : typ -> typ -> int
  = fun ty1 ty2 -> typ_cmp0 ITypnameMap.empty ITypnameMap.empty ty1 ty2

(* compare kinds for syntactic equality *)
let k_eq : kind -> kind -> bool
  = fun k1 k2
 -> match (k1,k2) with
      (KType   ,KType   ) -> true
    | (KEq(ty1),KEq(ty2)) -> typ_eq ty1 ty2
    | (_       ,_       ) -> false


(* compare types for syntactic equality *)
(* maps are substitutions to apply to ty1 and ty2 before comparison *)
let rec typ_eq0_with : tysubst -> internal_typname ITypnameMap.t -> internal_typname ITypnameMap.t -> typ -> typ -> bool
  = fun tenv m1 m2 ty1 ty2 ->
    let follow m tenv ty =
      match ty with
        TVar tn ->
          (let tn' =
            match maybe (ITypnameMap.find tn) m with
              Some tn' -> tn'
            | None     -> tn
          in
          match maybe (ITypnameMap.find tn') tenv with
            Some ty' -> ty'
          | None     -> TVar tn')
      | _ -> ty
    in
    match (follow m1 tenv ty1, follow m2 tenv ty2) with
    | (TTyCon0 c      , TTyCon0 c'       ) -> c = c'
    | (TTyCon1(c,ty)  , TTyCon1(c',ty')  ) -> c = c' && typ_eq0_with tenv m1 m2 ty ty'
    | (TTup(tys)      , TTup(tys')       )
    | (TSum(tys)      , TSum(tys')       ) -> List.length tys = List.length tys'
                                              && List.for_all2 (typ_eq0_with tenv m1 m2) tys tys'
    | (TFunc(ty1,ty2) , TFunc(ty1',ty2') ) -> typ_eq0_with tenv m1 m2 ty1 ty1' && typ_eq0_with tenv m1 m2 ty2 ty2'
    | (TVar(tn)       , TVar(tn')        ) -> tn_eq tn tn'
    | (TXDot(x,etn)   , TXDot(x',etn')   ) -> x_eq x x' && etn = etn'
    | (TTyName(n)     , TTyName(n')      ) -> n = n'
    | (TForall(tn,ty) , TForall(tn',ty') )
    | (TExists(tn,ty) , TExists(tn',ty') ) -> let tn'' = fresh_prettyname tn in
                                              typ_eq0_with tenv (ITypnameMap.add tn tn'' m1) (ITypnameMap.add tn' tn'' m2) ty ty'
    | (_              , _                ) -> false

let rec typ_eq_with : tysubst -> typ -> typ -> bool
  = fun tenv ty1 ty2 ->
    typ_eq0_with tenv ITypnameMap.empty ITypnameMap.empty ty1 ty2


(* compare MM_M.t for equality *)
let mnetn_eq : modname * external_typname -> modname * external_typname -> bool
  = fun (mn,etn) (mn',etn')
 -> mn_eq mn mn' && etn = etn'

(* compare h.t for equality *)
let hetn_eq : hash * external_typname -> hash * external_typname -> bool
  = fun (h,etn) (h',etn')
 -> h_eq h h' && etn = etn'

(* compare X.t for equality *)
let xetn_eq : hash_or_modname * external_typname -> hash_or_modname * external_typname -> bool
  = fun (x,etn) (x',etn')
 -> match (x,x') with
      (Hash(h),    Hash(h'))     -> h_eq  h  h'  && etn = etn'
    | (Modname(mn),Modname(mn')) -> mn_eq mn mn' && etn = etn'
    | (_,_) -> false

(* test for X.t membership in eqs *)
let xetn_mem_eqs : hash_or_modname * external_typname -> eqs -> bool
  = fun (x,etn) eqs
 -> List.exists (function EHash(h',etn',_) -> xetn_eq (x,etn) (Hash(h')    ,etn')
                        | EMod(mn',etn',_) -> xetn_eq (x,etn) (Modname(mn'),etn')) eqs

let lookup_eqs_xetn : eqs -> hash_or_modname * external_typname -> typ option
  = fun eqs (x,etn)
 -> findfirst (function EHash(h',etn',ty) -> if xetn_eq (x,etn) (Hash(h')    ,etn') then Some ty else None
                      | EMod(mn',etn',ty) -> if xetn_eq (x,etn) (Modname(mn'),etn') then Some ty else None) eqs

(* compare signatures for syntactic equality *)
let rec s_eq
  = fun sign1 sign2
 -> match (sign1.desc,sign2.desc) with
      ([],[])
        -> true
 | ([],_) | (_,[]) -> false
 | (s1::sign1', s2::sign2') ->
     (match (s1.desc, s2.desc) with
    | (SVal(id1,ty1),SVal(id2,ty2))
        -> ident_ext id1 = ident_ext id2
           &&
           (let iid = fresh in
            let (id1,sign1') = swap (fst (ident_int id1)) and iid in (id1,sign1') in
            let (id2,sign2') = swap (fst (ident_int id2)) and iid in (id2,sign2') in
            typ_eq ty1 ty2
            && s_eq {sign1 with desc=sign1'} {sign2 with desc=sign2'})
    | (STyp(tn1,k1),STyp(tn2,k2))
        -> typname_ext tn1 = typname_ext tn2
           &&
           (let itn = fresh in
            let (tn1,sign1') = swap (fst (typname_int tn1)) and itn in (tn1,sign1') in
            let (tn2,sign2') = swap (fst (typname_int tn2)) and itn in (tn2,sign2') in
            k_eq k1 k2
            && s_eq {sign1 with desc=sign1'} {sign2 with desc=sign2'})
    | (_,_)
        -> false)

let eqn_eq : eqn -> eqn -> bool
  = fun eq1 eq2
 -> match (eq1,eq2) with
      (EHash(h1,etn1,ty1),EHash(h2,etn2,ty2)) -> h_eq h1 h2 && etn1 = etn2 && typ_eq ty1 ty2
    | (EMod(mn1,etn1,ty1),EMod(mn2,etn2,ty2)) -> mn_eq mn1 mn2 && etn1 = etn2 && typ_eq ty1 ty2
    | (_,_) -> false

let eqn_cmp : eqn -> eqn -> int
  = fun eq1 eq2
 -> match (eq1,eq2) with
    | (EHash(h1,etn1,ty1),EHash(h2,etn2,ty2))
         -> h_cmp h1 h2 |=| compare etn1 etn2 |=| typ_cmp ty1 ty2
    | (EHash _, EMod _) -> -1
    | (EMod _, EHash _) -> 1
    | (EMod(mn1,etn1,ty1),EMod(mn2,etn2,ty2))
         -> mn_compare mn1 mn2 |=| compare etn1 etn2 |=| typ_cmp ty1 ty2

let canonicalise_eqs : eqs -> eqs
  = fun eqs
 -> uniq eqn_eq (List.sort eqn_cmp eqs)

let _ = Pretty.canonicalise_eqs := Some canonicalise_eqs  (* fill in recursive dependency *)

let eqs_eq : eqs -> eqs -> bool
  = fun eqs1 eqs2
 -> let fast eqs1 eqs2
      = try
          List.for_all2 eqn_eq eqs1 eqs2
        with
          Invalid_argument _ -> false
    in
    (* XXX if this becomes a bottleneck, it can be speeded up by
       not recomputing the hash_of_hash values (etc) each time;
       instead, convert each equation to a big string or number,
       and just sort them normally *)
    let full eqs1 eqs2 = fast (canonicalise_eqs eqs1) (canonicalise_eqs eqs2) in
    eqs1 == eqs2    (* physical equality *)
    ||
    fast eqs1 eqs2  (* without commutativity and idempotence *)
    ||
    full eqs1 eqs2  (* doing it properly *)


(* == SIGNATURE MANIPULATION == *)

(* selfifies only, but does not substitute out references to internal typenames of fields *)
let rec selfify_signature : hash_or_modname -> signature -> signature
  = fun x sign
 -> {sign with desc=List.map (function si ->
              match si.desc with
                SVal(_,_)      -> si
              | STyp(_,KEq(_)) -> si
              | STyp(tn,KType) -> {desc = (STyp(tn,KEq(TXDot(x,typname_ext tn)))); loc = si.loc})
             sign.desc}

(* for every concrete type t:EQ(t'), push through the substitution t'/t *)
let rec type_flatten_sign' = function
  | [] -> []
  | s ::sign' -> (match s.desc with
  | (SVal _) as signitem ->
      s :: type_flatten_sign' sign'
  | (STyp ((ext, inter), kind)) as signitem ->
      let sign'' = begin match kind with
      | KType -> sign'
      | KEq(t) -> tmap_signature' (tsub_typ [(inter, t)]) sign'
      end in
      s :: type_flatten_sign' sign''
)

let type_flatten_sign sign = {sign with desc=type_flatten_sign' sign.desc}


let rec type_flatten_str' = function
  | [] -> []
  | s::str' -> (match s.desc with
  | (StrVal _) as stritem  ->
      s :: type_flatten_str' str'
  | (StrValMulti _) as stritem ->
      s :: type_flatten_str' str'
  | (StrTyp ((ext, inter), t)) as stritem ->
      let str'' = tmap_structure' (tsub_typ [(inter,t)]) str' in
      s :: type_flatten_str' str'')

let type_flatten_str str = {str with desc=type_flatten_str' str.desc}

(* ensure that all types named in signature are globally meaningful:
   abstract types remain abstract and concrete types concrete, but every
   t becomes X.t, and every concrete type's substitution is pushed through.
   This is very similar to type_flatten_sign o selfify_signature, except that
   abstract types remain abstract. *)
let type_really_flatten_sign'
  = fun x sign
 -> let rec go sign = match sign with
  | [] -> []
  | s :: sign -> (match s.desc with
  | (SVal _) as si ->
      s :: go sign
  | (STyp (tn, k)) as si ->
      let t = match k with
        KType  -> txdot x (typname_ext tn)
      | KEq(t) -> t
      in
      s :: go (tmap_signature' (tsub_typ [(typname_int tn, t)]) sign)
 )in
 go sign;;

let type_really_flatten_sign  : hash_or_modname -> signature -> signature
    = fun x sign -> {sign with desc=type_really_flatten_sign' x sign.desc}

(* == VALUE TEST == *)

let collect_args : expr -> (expr * expr list)
  = fun e
 -> let rec go args e =
    match e.desc with
      LocApp(e1,e2) -> go (e2::args) e1
    | _          -> (e,args)
    in
    go [] e


let apply_args : expr * expr list -> expr
  = fun (e,es)
 -> List.fold_left (fun a b -> {desc=LocApp(a,b); loc={a.loc with Location.loc_ghost=true}}) e es


let prim_collect_args : prim_expr -> (prim_expr * prim_expr list)
  = fun e
 -> let rec go args e =
    match e with
      App(e1,e2) -> go (e2::args) e1
    | _          -> (e,args)
    in
    go [] e


let prim_apply_args : prim_expr * prim_expr list -> prim_expr
  = fun (e,es)
 -> List.fold_left (fun a b -> App(a,b)) e es


(* Is a given syntactic expression an eqs-value? *)
(*
   either:
    *  e is a value && if e is a guard, aft (e) is true
   or :
   *  pred e is true
*)

let rec is_value_like pred ((clos, tclos) as closures) eqs e =
  match pred(e) with
    | Some t -> t
    | None ->
        (match e with
             C0(c0)          -> true
           | C1(c1,e)        -> is_value_like pred closures eqs e
           | Cons(e1,e2)     -> is_value_like pred closures eqs e1 && is_value_like pred closures eqs e2
           | Tup(es)         -> List.for_all (is_value_like pred closures eqs) es
           | Op(oe,es)       -> opx_value closures eqs (e,[])
           | Loc(loc)        -> true
           | Fn(mtch)        -> not clos
           | Fun(ps,e1)      -> not clos
           | InEnv(env,e)    -> false
           | Clos _          -> clos || raise (Never_happen "is_value_like closures not expecting a closure")
           | TClos _         -> tclos || raise (Never_happen "is_value_like closures not expecting a tclosure")
           | Id(iid)         -> false
           | Dot(mn,eid)     -> false
           | HashDot(h,eid)  -> false
           | If(e1,e2,e3)    -> false
           | While(e1,e2)    -> false
           | LazyOp(lo,el)   -> opx_value closures eqs (prim_collect_args e)
           | Seq(e1,e2)      -> false
           | App(e1,e2)      -> opx_value closures eqs (prim_collect_args e)
           | Match(e,m)      -> false
           | Let(e1,pe2)     -> false
           | LetMulti(ty,pse,ie) -> false
           | Letrec(ty,ime)  -> false
           | LetrecMulti(ty,psiee) -> false
           | Raise(e)        -> false
           | Try(e,mtch)     -> false
           | Marshal(mk,e,ty)-> false
           | Marshalz(mk,e,ty)-> false
           | Unmarshal(e,ty) -> false
           | RET(ty)         -> false
           | Col(e,eqs',TXDot(x,etn))
             -> xetn_mem_eqs (x,etn) eqs' && not (xetn_mem_eqs (x,etn) eqs) && is_value_like pred closures eqs' e
           | Col(e, eqs', TTyCon1(TRef,t))
           | Col(e, eqs', TTyCon1(TName,t)) -> is_value_like pred closures eqs' e
           | Col(e,eqs',_)   -> false
           | OP(n,oe,es)     -> false
           | Resolve(e,mn,rs)
           | Resolve_blocked(e,mn,rs)       -> false
           | ValOfTie _                     -> false
           | NameOfTie _                    -> false
           | Tie (_,_)                      -> false
           | Support (_, _)                 -> false
           | Freshfor (_, _)                -> false
           | Swap (_, _, _)                 -> false
           | HashHts (_,e1,e2,_)            -> false
           | HashTs (_, e1, _)              -> false
           | HashMvf (_,_,_)                -> false
           | NameValue _                    -> true
           | CFresh _                       -> false
           | Fresh _                        -> false
           | Par (_, _)                     -> false
           | SLOWRET _                      -> false
           | Namecase (_,_, _, _, _, _, _)  -> false
           | Unpack (_, _, _, _)            -> false
           | Pack (_,e1,_)                  -> is_value_like pred closures eqs e1
           | TApp (_, _)                    -> false
           | TAbs(_, _)                     -> not tclos)

and opx_value closures eqs (e,args) =
  let (arity,args') =
    match e with
    | Op(oe,es)     -> (Some (arity_of_op_or_econst oe), List.rev_append es args)
    | LazyOp(lo,es) -> (Some 2, List.rev_append es args)
    | _             -> (None, args)
  in
  match arity with
    None -> false
  | Some(n) ->
      List.length args' < n && List.for_all (is_value_like (fun x -> None) closures eqs) args'

let isvalue closures eqs e = is_value_like (fun x -> None) closures eqs e;;

(* == MISCELLANEOUS == *)

(* bizarrely, in OCaml constructors are not functions: make them so *)
let tTup    ty = TTup ty
let tSum    ty = TSum ty
let tFunc tya tyb = TFunc(tya,tyb)
let eMod (mn,etn,ty) = EMod(mn,etn,ty)

(* collect the first n argument types from function type ty *)
let collect_argtys : int -> typ -> (typ list * typ) option
  = fun n ty
 -> let rec go n tys0 ty =
    if n = 0 then
      Some (List.rev tys0, ty)
    else
      match ty with
        TFunc(tya,tyb) -> go (n-1) (tya :: tys0) tyb
      | _              -> None
    in
    go n [] ty


(* == BASIC TYPE FUNCTIONS == *)

let typeof_C0 : con0 -> typ
(* NB: doesn't check that type is well-formed *)
  = fun c0
 -> match c0 with
    | Nil(ty)   -> TTyCon1(TList,ty)
    | NONE(ty)  -> TTyCon1(TOption,ty)
    | B0(b0)    -> typeof_B0 b0

let typeof_C1 : con1 -> (typ * typ)
(* NB: doesn't check that type is well-formed *)
  = fun c1
 -> match c1 with
    | Inj(i,TSum(tys)) when 1 <= i && i <= List.length tys
                      -> (List.nth tys (i-1)      , TSum tys)
    | Inj _           -> raise (Never_happen "typeof_C1: Inj with non-sum type annotation or index out of range")
    | SOME
    | TieCon
    | Node            -> raise (Never_happen "typeof_C1: expression insufficiently type-annotated; time for a trip to Delphi")
    | B1(b1)          -> typeof_B1 b1

let typeof_Op : op -> typ
  = fun op
 -> match op_info op with
      (tys,ty) -> List.fold_right tFunc tys ty


let typeof_op_or_econst : op_or_econst -> typ
  = function
      OEOp op      -> typeof_Op op
    | OEEconst iid -> Econst.type_of_ident iid


let info_of_op_or_econst : op_or_econst -> typ list * typ
  = function
      OEOp op      -> op_info op
    | OEEconst iid -> the (collect_argtys (Econst.arity_of_ident iid) (Econst.type_of_ident iid))


(* == TYPE VALIDATION FUNCTIONS == *)

(* dead code, but may reincarnate shortly -K
 * (* is type free of arrows? *)
 * let rec arrowfree_p : typ -> bool
 *   = function
 *     | TTyCon0 _
 *     | TVar _
 *       -> true
 *     | TTyCon1(_,ty)
 *       -> arrowfree_p ty
 *     | TTup(tys)
 *       -> List.for_all arrowfree_p tys
 *     | TXDot(_,_)
 *       -> raise (Never_happen "arrowfree_p on X.t - it depends on what you mean!!")
 *           (* TODO: there are three options:
 *              - abstract types disallowed (=false)
 *              - abstract types allowed if their rep type is arrow free (need extra args
 *                to get at rep type, and can't do it with numeric hashes)
 *              - abstract types allowed always, and we rely on a runtime check *)
 *     | TFunc(_,_)
 *       -> false
 *)


(* == TYPING AN EXPRESSION WITHOUT CHECKING == *)

let rec strip_arg_tys : int -> typ -> typ
  = fun n ty
 -> if n <= 0 then ty
    else match ty with
      TFunc(_,ty') -> strip_arg_tys (n-1) ty'
    | _            -> raise (Never_happen "strip_arg_tys: not enough arrows")

let rec patty : pat -> typ
  = fun p
 -> match p with
      PWild(ty)              -> ty
    | PVar(iid,ty)           -> ty
    | PC0(c0)                -> typeof_C0 c0
    | PC1(SOME,p)            -> TTyCon1(TOption,patty p)
    | PC1(Node,PTup [p1;p2]) -> TTyCon1(TTree,patty p1)
    | PC1(Node,_)            -> raise (Never_happen "patty: node")
    | PC1(c1,p)              -> snd (typeof_C1 c1)
    | PCons(p1,p2)           -> TTyCon1(TList,patty p1)
    | PTup(ps)               -> TTup (List.map patty ps)
    | PTyped(p,ty)           -> ty

let matchty : (pat * 'a) list -> typ  (* mtch or prim_mtch *)
  = fun mtch
 -> match mtch with
      [] -> raise (Never_happen "matchty: empty match")
    | ((p,_)::_) -> patty p


(* == SYNTAX CHECKS == *)

let sugared : prim_expr -> bool
  = fun e0 ->
    let r = ref false in
    let complexmatch = function
        [(PVar(_,_),_)] -> false
      | _ -> true
    in
    let f _ e =
      (match e with
        Fn m when complexmatch m.desc
                                 -> r := true
      | Fun _                    -> r := true
      | Let _                    -> r := true
      | Letrec(_,(_,(m,_))) when complexmatch m.desc
                                 -> r := true
      | Op(oe,es) when List.length es <> arity_of_op_or_econst oe
                                 -> r := true
      | LetMulti _               -> r := true
      | LetrecMulti _            -> r := true
      | Par _                    -> r := true
      | _ -> ());
      None
    in
    let (_:prim_expr) = early_emap_prim_expr false f None e0 in
    !r


(* --------------------------------------------------------------  *)
(* version constraint inclusion and version-in-constraint checking *)


(* This is a bit ugly - partly as the Ast version constraint grammar is a bit wierd, but also due to the irregular (but useful?) nature of * *)

exception Bad_version_test





let vlit_eq : Ast.vliteral -> Ast.vliteral -> bool
 = fun vlit vlit' ->
   match (vlit,vlit') with
     (VNumber(n),VNumber(n')) -> (n=n')
   | (VHash(h),VHash(h')) ->  h_eq h h'
   | _ -> false

(* to avoid combinatorics of tail_version_constraint comparisons, map
into nonstandard vliterals *)

(* XXX this code could be simplified now that we never actually try to compare
   naturals and hashes *)

type ns_vliteral = Std of vliteral | Infty

let ns_range_of_tvc : Ast.tail_version_constraint -> ns_vliteral*ns_vliteral
 = fun tvc -> match tvc with
   TVCAtomic(AVCNumber(n)) -> (Std(VNumber(n)),Std(VNumber(n)))
 | TVCAtomic(AVCHashVersion(AVCHHash(h))) -> (Std(VHash(h)),Std(VHash(h)))
 | TVCAtomic(AVCHashVersion(AVCHModname(_))) -> raise Bad_version_test
 | TVCBetween(n1,n2) -> (Std(VNumber(n1)),Std(VNumber(n2)))
 | TVCBefore(n2) -> (Std(VNumber(0)),Std(VNumber(n2)))
 | TVCAfter(n1) -> (Std(VNumber(n1)), Infty)
 | TVCAnything -> (Std(VNumber(0)),Infty)


(* NB: All hashes are bigger than all OCaml ints *)

let ns_lteq : ns_vliteral -> ns_vliteral -> bool
    = fun ns ns' -> match (ns,ns') with
      (_,Infty) -> true
    | (Infty,Std(_)) -> false
    | (Std(VNumber(n)),Std(VNumber(n'))) -> n<=n'
    | (Std(VNumber(n)),Std(VHash(h'))) -> true
    | (Std(VHash(h)),Std(VNumber(n'))) -> false
    | (Std(VHash(h)),Std(VHash(h'))) -> h_lteq h h'

let ns_range_subseteq : (ns_vliteral*ns_vliteral) -> (ns_vliteral*ns_vliteral) -> bool
    = fun (ns1,ns2) (ns1',ns2') -> (ns_lteq ns1' ns1) && (ns_lteq ns2 ns2')


(* raises Bad_version_test for some (not guaranteed to be all) cases
of non-value version constraints, ie VCNameEqual .  *)

let rec vc_subseteq_vc : Ast.version_constraint -> Ast.version_constraint -> bool
    = fun vc vc' -> match (vc,vc') with
      (VCDotted(avcs,tvc),VCDotted(avcs',tvc')) -> (
        match avcs with
          (AVCNumber(n)::avcs0) ->
            (match avcs' with
              AVCNumber(n')::avcs0' ->
                n = n' && vc_subseteq_vc (VCDotted(avcs0,tvc))
                                         (VCDotted(avcs0',tvc'))
            | AVCHashVersion(AVCHModname(_))::_ -> raise Bad_version_test
            | AVCHashVersion(AVCHHash(_))::_ -> false
            | [] -> (match tvc' with
                TVCAnything -> true
              | TVCAtomic(AVCHashVersion(AVCHModname(_))) -> raise Bad_version_test
              | _ -> false))
        | (AVCHashVersion(AVCHModname(_))::avcs0) -> raise Bad_version_test
        | (AVCHashVersion(AVCHHash(h))::avcs0) ->
            (match avcs' with
              AVCNumber(_)::avcs0' -> false
            | AVCHashVersion(AVCHModname(_))::_ -> raise Bad_version_test
            | AVCHashVersion(AVCHHash(h'))::avcs0' ->
                h_eq h h' && vc_subseteq_vc (VCDotted(avcs0,tvc))
                                            (VCDotted(avcs0',tvc'))
            | [] -> (match tvc' with
                TVCAnything -> true
              | TVCAtomic(AVCHashVersion(AVCHModname(_))) -> raise Bad_version_test
              | _ -> false))
        | [] -> (match avcs' with
            _::_ -> false
          | [] -> ns_range_subseteq (ns_range_of_tvc tvc)
                                    (ns_range_of_tvc tvc')))
    | (VCNameEqual(ahvc),VCNameEqual(ahvc')) ->  (
        match (ahvc,ahvc') with
          (AVCHHash(h),AVCHHash(h')) -> h_eq h h'
        | _ -> raise Bad_version_test )
    | _ -> false



let rec vn_isin_vc_p : Ast.version -> atomic_version_constraint list -> tail_version_constraint -> bool
    = fun vn avcs tvc -> match vn with
      VAtomic(VLiteral(vlit)) -> (
        match avcs with
          [] ->
            let ns= Std(vlit) in
            let (ns1,ns2)=ns_range_of_tvc tvc in
            ns_lteq ns1 ns && ns_lteq ns ns2
        | avc::avcs0 -> (match avc with
            AVCHashVersion(AVCHHash(h')) ->
              vlit_eq vlit (VHash(h')) && avcs0=[] && tvc=TVCAnything
          | AVCHashVersion(AVCHModname(_)) -> raise Bad_version_test
          | AVCNumber(n') ->
              vlit_eq vlit (VNumber(n')) && avcs0=[] && tvc=TVCAnything))

    | VCons(VLiteral(vlit),vn0) -> (
        match avcs with
          [] -> (tvc = TVCAnything)
        | avc::avcs0 -> (match avc with
            AVCHashVersion(AVCHHash(h')) ->
              vlit_eq vlit (VHash(h')) && vn_isin_vc_p vn0 avcs0 tvc
          | AVCHashVersion(AVCHModname(_)) -> raise Bad_version_test
          | AVCNumber(n') ->
              vlit_eq vlit (VNumber(n')) && vn_isin_vc_p vn0 avcs0 tvc))

    | VAtomic(VMyname) -> raise Bad_version_test
    | VCons(VMyname,_) -> raise Bad_version_test



(* this is only sensible for vc of the form VCDotted( ) *)
let vn_isin_vc : Ast.version -> Ast.version_constraint -> bool
    = fun vn vc -> match vc with
      VCDotted(avcs,tvc) -> vn_isin_vc_p vn avcs tvc
    | VCNameEqual(_) -> raise Bad_version_test



let vn_h_satisfies_vc : Ast.version -> Ast.hash -> Ast.version_constraint -> bool
     = fun vn h vc ->
       match vc with
         VCDotted(_) -> vn_isin_vc vn vc
       | VCNameEqual(AVCHHash(h')) -> h_eq h h'
       | VCNameEqual(AVCHModname(_)) -> raise Bad_version_test


(* -=-- ---------- ------------------------------------ ---------- --=- *)

(* jamey's TT function and inverse *)

exception Bad_Index

  let rec geti t i =
	if i = 1 then
	  match t with TFunc(t1,t2) -> t1
	  | _ -> raise Bad_Index
	else
	  match t with TFunc(t1,t2) -> geti t2 (i-1)
	  | _ -> raise Bad_Index



let tt : op -> int -> typ =
  fun e i ->
    let ty = typeof_Op e in
    if i=0 then
      let rec getlast t =
	match t with TFunc(t1, t2) -> getlast t2
	| t -> t
      in getlast ty
    else
      geti ty i


let tt_inv : typ -> op -> typ =
  fun t e ->
  match e with
    Ref _ -> geti t 1
  | Deref _ -> geti t 0
  | Assign _ -> geti t 2
  | Equal _ -> geti t 1
  | _ -> raise (Never_happen "tt_inv on non-type-indexed ops")


(* == TYPE ENVIRONMENTS == *)


(* -- Constructors -- *)

(* LIST VERSION *)
(*
 * let addtypeenv0 : typeenv_entry -> typeenv -> typeenv
 *   = fun ei env
 *  -> env
 *)

(* MAP VERSION *)
let addtypeenv0 : typeenv_entry -> typeenv -> typeenv
  = fun ei env
 -> match ei with
      Env_val(iid,ty) -> { env with te_iidmap = IIdentMap.add iid ty env.te_iidmap }
    | Env_loc(loc,ty) -> { env with te_locmap = LocMap.add loc ty env.te_locmap }
    | Env_mod(mn,sign)->
        let sign' = type_really_flatten_sign' (Modname mn) sign.desc in
        let rec build eidmap etnmap = function
            ({ desc=SVal (id,ty) }::s) -> build (EIdentMap.add (ident_ext id) ty eidmap) etnmap s
          | ({ desc=STyp (tn,k)  }::s) -> build eidmap (ETypnameMap.add (typname_ext tn) k etnmap) s
          | [] -> (eidmap,etnmap) in
        let (eidmap,etnmap) = build EIdentMap.empty ETypnameMap.empty sign' in
        { env with
          te_modmap = ModnameMap.add mn
                        { mod_sign = sign;
                          mod_eidmap = eidmap;
                          mod_etnmap = etnmap;
                        }
                        env.te_modmap
        }
    | Env_typ(itn,k)  -> { env with te_itnmap = ITypnameMap.add itn k env.te_itnmap }


(* common *)

let addtypeenv : typeenv_entry -> typeenv -> typeenv
  = fun ei env
 -> let env' = addtypeenv0 ei env in
    { env' with te_list = ei :: env.te_list }

let emptytypeenv : typeenv
  = { te_list = [];
      te_iidmap = IIdentMap.empty;
      te_locmap = LocMap.empty;
      te_modmap = ModnameMap.empty;
      te_itnmap = ITypnameMap.empty }

let emptynametypeenv : nametypeenv
  = (emptynameenv,
     { te_list = [];
      te_iidmap = IIdentMap.empty;
      te_locmap = LocMap.empty;
      te_modmap = ModnameMap.empty;
      te_itnmap = ITypnameMap.empty })

(* LIST VERSIONS *)
(*
 * let mktypeenv : typeenv_list -> typeenv
 *     = fun eis
 *    -> { emptytypeenv with te_list = eis }
 *
 * let appendtypeenv : typeenv -> typeenv -> typeenv
 *     = fun env1 env2
 *    -> { emptytypeenv with te_list = env1.te_list @ env2.te_list }
 *)

(* MAP VERSIONS *)
let mktypeenv : typeenv_list -> typeenv
    = fun eis
   -> (* env is in reverse order; head should shadow tail *)
      List.fold_right addtypeenv0 eis { emptytypeenv with te_list = eis }

let appendtypeenv : typeenv -> typeenv -> typeenv
    = fun env1 env2
   -> (* env1 shadows env2 *)
      List.fold_right addtypeenv env1.te_list env2


(* common *)

let concattypeenv : typeenv list -> typeenv
    = fun envs
   -> List.fold_right appendtypeenv envs emptytypeenv


(* -- Lookups -- *)

let ff_typeenv_modname_matcher mn0
    = function
        Env_val(id,ty)   -> None
      | Env_loc(l,ty)    -> None
      | Env_mod(mn,sign) -> if mn_eq mn mn0 then Some(sign) else None
      | Env_typ(tn,k)    -> None


(* LIST VERSIONS *)
(*
 * let lookup_typeenv_iident : typeenv -> internal_ident -> typ option
 *   = fun env id0
 *  -> findfirst (function tee -> match tee with
 *                                  Env_val(id,ty)   -> if id_eq id id0 then Some (ty) else None
 *                                | Env_loc(l,ty)    -> None
 *                                | Env_mod(mn,sign) -> None
 *                                | Env_typ(tn,k)    -> None) env.te_list
 *
 * let lookup_typeenv_location : typeenv -> location -> typ option
 *   = fun env l0
 *  -> findfirst (function tee -> match tee with
 *                                  Env_val(id,ty)   -> None
 *                                | Env_loc(l,ty)    -> if l_eq l l0 then Some(ty) else None
 *                                | Env_mod(mn,sign) -> None
 *                                | Env_typ(tn,k)    -> None) env.te_list
 *
 * let lookup_typeenv_modname : typeenv -> modname -> signature option
 *   = fun env mn0
 *  -> findfirst (ff_typeenv_modname_matcher mn0) env.te_list
 *
 * let lookup_typeenv_itypname : typeenv -> internal_typname -> kind option
 *   = fun env tn0
 *  -> findfirst (function tee -> match tee with
 *                                  Env_val(id,ty)   -> None
 *                                | Env_loc(l,ty)    -> None
 *                                | Env_mod(mn,sign) -> None
 *                                | Env_typ(tn,k)    -> if tn_eq tn tn0 then Some(k) else None) env.te_list
 *)

(* MAP VERSIONS *)
let lookup_typeenv_iident : typeenv -> internal_ident -> typ option
  = fun env id0
 -> maybe (IIdentMap.find id0) env.te_iidmap

let lookup_typeenv_location : typeenv -> location -> typ option
  = fun env l0
 -> maybe (LocMap.find l0) env.te_locmap

let lookup_typeenv_modname : typeenv -> modname -> signature option
  = fun env mn0
 -> maybe (fun () -> (ModnameMap.find mn0 env.te_modmap).mod_sign) ()

(* None => no module; Some None => no field; Some (Some ty) => success *)
let lookup_typeenv_modname_val : typeenv -> modname -> external_ident -> typ option option
  = fun env mn0 id0
 -> option_lift (fun mi -> maybe (EIdentMap.find id0) mi.mod_eidmap) (maybe (ModnameMap.find mn0) env.te_modmap)

(* None => no module; Some None => no field; Some (Some k) => success *)
let lookup_typeenv_modname_typ : typeenv -> modname -> external_typname -> kind option option
  = fun env mn0 tn0
 -> option_lift (fun mi -> maybe (ETypnameMap.find tn0) mi.mod_etnmap) (maybe (ModnameMap.find mn0) env.te_modmap)

let lookup_typeenv_itypname : typeenv -> internal_typname -> kind option
  = fun env tn0
 -> maybe (ITypnameMap.find tn0) env.te_itnmap



(* common *)

(* returns environment prefix as well *)
let lookup_typeenv_modname_ext : typeenv -> modname -> (signature * typeenv) option
  = fun env mn0
 -> match findfirst_ext (ff_typeenv_modname_matcher mn0) env.te_list with
      Some (s,e) -> Some (s, mktypeenv e)
    | None       -> None

let modname_of_typeenv_entry : typeenv_entry -> modname option
  = function
      Env_mod(mn,_) -> Some mn
    | _             -> None


(* -- Membership tests -- *)

let iident_mem_typeenv : internal_ident -> typeenv -> bool
  = fun id0 env -> is_Some (lookup_typeenv_iident env id0)
let location_mem_typeenv : location -> typeenv -> bool
  = fun l0 env -> is_Some (lookup_typeenv_location env l0)
let modname_mem_typeenv : modname -> typeenv -> bool
  = fun mn0 env -> is_Some (lookup_typeenv_modname env mn0)
let itypname_mem_typeenv : internal_typname -> typeenv -> bool
  = fun tn0 env -> is_Some (lookup_typeenv_itypname env tn0)


(* -- Constructing from definitions -- *)

(* note that the argument list get reversed when making the output
list (following the convention for type environments) *)

(* semicompiled environment *)
let env0_of_defs defs =
  let rec f env = function
    | [] -> List.rev env
    | def :: defs ->
        begin match def with
        | Mod_compile _
        | Mod_imod _
        | Import_compile _ ->
            let env_entry =
              Env_mod (modname_of_definition def,
                       sign0_of_definition def) in
            f (env_entry :: env) defs
        | Mod_fresh _
        | Import_fresh _ ->
            let env_entry =
              Env_mod (modname_of_definition def,
                       sign_of_definition def) in
            f (env_entry :: env) defs
        | Mark_compile _ -> f env defs end
  in
  mktypeenv (f [] defs)

(* compiled environment *)
let env1_of_defs defs =
  let rec f env = function
    | [] -> List.rev env
    | def :: defs ->
        begin match def with
        | Mod_compile _
        | Mod_imod _
        | Import_compile _ ->
            let env_entry =
              Env_mod (modname_of_definition def,
                       sign1_of_definition def) in
            f (env_entry :: env) defs
        | Mod_fresh _
        | Import_fresh _ ->
            let env_entry =
              Env_mod (modname_of_definition def,
                       sign_of_definition def) in
            f (env_entry :: env) defs
        | Mark_compile _ -> f env defs end
  in
  mktypeenv (f [] defs)

(* -=-- ---------- ------------------------------------ ---------- --=- *)


let econst_env : typeenv
  = mktypeenv Econst.econst_env_list

