(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * Acute Systematic Tester                                        *
 *                                                                *
 * Version: $Id: runtest_pretty.ml,v 1.513 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)


open Runtest_db
open Unix

let ppmode = ref false

let set_pp_mode b = ppmode := b

type out_type = Color | Unix | Overstrike | Html

let black   = 0
let red     = 1
let green   = 2
let yellow  = 3
let blue    = 4
let magenta = 5
let cyan    = 6
let white   = 7

let out_type = ref (match (fstat stdout).st_kind with
| S_CHR -> Color
| S_REG -> Html
| _     -> Unix)

let set_out_type s = try
  out_type := List.assoc s
    [("color",Color);
     ("colour",Color);
     ("unix",Unix);
     ("html",Html);
     ("overstrike",Overstrike)]
with
  Not_found -> raise (Failure ("Unrecognised output type specified: "^s))

let _reset = "\x1b[0m"

let _color fg br = Printf.sprintf "\x1b[%u;%um" br (fg+30)

let _w = "\x1b[0;1;4m"

let _r = _color red 0
let _b = _color blue 1
let _ok = _color green 0

let rec indent = function
| 0 -> ""
| i -> "| " ^ (indent (i-1))

let escape s =
  let s = Str.global_replace (Str.regexp_string "&") "&amp;" s in
  let s = Str.global_replace (Str.regexp_string "<") "&lt;" s in
  Str.global_replace (Str.regexp_string ">") "&gt;" s

let print s = print_endline s

let ovs_bold s = Str.global_replace (Str.regexp "\\(.\\)") "\\1\b\\1" s
let ovs_ul s = Str.global_replace (Str.regexp "\\(.\\)") "_\b\\1" s

let col_name s =
  match !out_type with
  | Color -> _w ^ s ^ _reset
  | Unix -> "*" ^ s ^ "*"
  | Overstrike -> ovs_bold s
  | _ -> s

let col_ok s =
  match !out_type with
  | Color -> _ok ^ s ^ _reset
  | Unix -> s
  | Overstrike -> ovs_ul s
  | _ -> s

let col_bad s =
  match !out_type with
  | Color -> _r ^ s ^ _reset
  | Unix -> "_" ^ s ^ "_"
  | Overstrike -> ovs_ul s
  | _ -> s

let col_doc s =
  match !out_type with
  | Color -> _b ^ s ^ _reset
  | Unix -> s
  | Overstrike -> ovs_bold s
  | _ -> s



let print_res t_name res level =
  let msg = (
    match res,!out_type with
    | true,Html ->
        if Runtest_db.is_grouped t_name then
          "  <tr><td class=\"test_ok\" colspan=\"3\">OK</td>\n  </tr>"
        else
          "    <td class=\"test_ok\">OK</td>\n  </tr>"
    | false,Html ->
        if Runtest_db.is_grouped t_name then
          "  <tr><td class=\"test_bad\" colspan=\"3\">BAD</td>\n  </tr>"
        else
          "    <td class=\"test_bad\">BAD</td>\n  </tr>"
    | true,_ ->
        Printf.sprintf "%s| %s" (indent level) (col_ok "OK!")
    | false,_ ->
        Printf.sprintf "%s| %s" (indent level) (col_bad "BAD")
  ) in print msg;
  res

let print_title t_name t_doc level =
  let msg =
  if !out_type = Html then (
    if Runtest_db.is_grouped t_name then
      Printf.sprintf "  <tr class=\"l%u\">\n    <td class=\"test\" colspan=\"3\">\n      <span class=\"test\">%s</span> <span class=\"doc\">%s</span>\n    </td></tr>"
        level t_name (escape t_doc)
    else
      Printf.sprintf "  <tr class=\"l%u\">
    <td class=\"test\">
        <span class=\"test\"><a href='%s.ac'>%s</a> [<a href='%s.err'>debug</a>]</span>
        <span class=\"doc\">%s</span>
</td>"
        level t_name t_name t_name (escape t_doc)
  ) else (
    print (indent level);
    Printf.sprintf "%s* TEST %s \t(%s)" (indent level) (col_name t_name) (col_doc t_doc)
  ) in
  print msg

let rec print_indent l level =
  if !out_type=Html then (
    print "    <td class=\"sry\">";
    List.iter (function s -> print ("      "^(escape s)^"<br />")) l;
    print "    </td>";
  ) else (
    match l with
    | []   -> ()
    | h::t -> print ((indent (level+1)) ^ h); print_indent t level
  )

let rec print_lines i = function
  | []   -> ()
  | h::t ->
      print (Printf.sprintf "| %3u: %s" i h);
      print_lines (i+1) t

let print_body t_name s f_name =
  if !out_type != Html then (
    print (Printf.sprintf "\n* CODE %s (%s)" (col_name t_name) (col_doc f_name));
    let l = (Str.split (Str.regexp_string "\n") s) in
    print_lines 1 l
  )

let dump_opts chan = function
  | [] -> ()
  | opts ->
      output_string chan "(* OPTS";
      List.iter (function s -> output_string chan (" "^s)) opts;
      output_string chan " *)\n"

let dump_expect chan = function
  | Exits l ->
      output_string chan "(* ";
      List.iter (function None -> output_string chan ("RETX\n   ") | Some s ->  output_string chan ("RET  "^s^"\n   ")) l;
      output_string chan "*)\n"
  | Fail s ->
      output_string chan ("(* FAIL  "^s^" *)\n")
  | Group g ->
      output_string chan "(* GROUP  ";
      List.iter (function s ->  output_string chan (s^" ")) g;
      output_string chan "*)\n"
  | Killed i ->
      output_string chan ("(* KILLED "^(string_of_int i)^" *)\n")
  | Compile ->
      output_string chan "(* COMPILE *)\n"
  | Empty -> ()

let print_killed level =
  if !out_type = Html then
    print "    <td class=\"test_killed\">KILLED</td>\n"
  else
    print (Printf.sprintf "%s| %s" (indent level) (col_bad "KILLED !"))

let acpp_body body =
  let add s = "  " ^ (String.sub s 2 (String.length s - 2)) ^ "\n" in
  let check c s = match String.length s with
  | 0 | 1 -> false
  | _ -> s.[0] = '#' && s.[1] = c
  in
  let rec preproc = function
    | [] -> ""
    | h::t when check '+' h -> if !ppmode then (add h) ^ (preproc t) else "" ^ (preproc t)
    | h::t when check '-' h -> if !ppmode then "" ^ (preproc t)    else (add h) ^ (preproc t)
    | h::t -> h ^ "\n" ^ (preproc t)
  in
  preproc ( Str.split (Str.regexp_string "\n") body )

let dump_body t dir = match t.body, dir with
  | _, "" | None, _ -> ()
  | Some s, dir ->
      let chan = open_out (dir ^ "/" ^ t.name ^ ".ac") in
      output_string chan (Printf.sprintf "(* TEST %s *)\n" t.name );
      dump_opts chan t.options;
      dump_expect chan t.expect;
      output_string chan (acpp_body s);
      close_out chan

