(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * Acute Systematic Tester Database Handler                       *
 *                                                                *
 * Version: $Id: runtest_library.ml,v 1.514 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)


(* types *)

exception This_test_has_no_Body of string

open Runtest_db

(******************************************************************************)
(* RUNTEST INTERNALS AND THEIR PUBLIC INTERFACE                               *)
(******************************************************************************)

let lazy_group = ref true
let set_lazy b = lazy_group := b

(* the default directory to write files *)
let test_dir = ref "tests_out"

let dump_src_dir = ref ""
let set_src_dump s = dump_src_dir := s

(* change the above setting *)
let set_runtest_dir s = test_dir := s

(* hash table of results, to avoid running the same test twice *)
let tests_db = Hashtbl.create 127

(******************************************************************************)
(* TEST RUNNING                                                               *)
(******************************************************************************)

let write_to_file f_name s =
  let out_d = Unix.openfile f_name [ Unix.O_CREAT; Unix.O_WRONLY; Unix.O_TRUNC ] 0o664 in
  let out = Unix.out_channel_of_descr out_d in
  output_string out s;
  close_out out

(* return the result of a previously runned test, or run it *)
let do_test test opts tee =
  try
    Hashtbl.find tests_db test.name
  with Not_found ->
    let (out,trace,ret_val) = Runtest_shell.run_compile opts test tee in
    if !dump_src_dir <> "" then (
      Runtest_pretty.dump_body test !dump_src_dir;
      write_to_file (!dump_src_dir ^"/"^test.name^".out") out;
      write_to_file (!dump_src_dir ^"/"^test.name^".err") trace
    );
    Hashtbl.add tests_db test.name (out,trace,ret_val);
    (out,trace,ret_val)

(* exec a test, in regards of his .expect field *)
let rec exec_test verb opts test level =
  ( try
      Unix.mkdir !test_dir 0o777
    with Unix.Unix_error (Unix.EEXIST,_,_) -> ()
    | e -> (
          prerr_endline ("Couldn't create test directory " ^ !test_dir);
          raise e
        )
  );

  let opts' = (test.options @ opts) in

  match test.expect with
  | Empty ->
      Runtest_pretty.print_title test.name test.doc level;
      Runtest_pretty.print_indent [ "runtest: NO GOAL !!!" ] level;
      Runtest_pretty.print_res test.name false level
  | Group l ->
      Runtest_pretty.print_title test.name test.doc level;
      let t_list = List.map find_test l in
      let fold_fun acc test = if !lazy_group then
        acc && (exec_test verb opts' test (level+1))
      else
        (exec_test verb opts' test (level+1)) && acc
      in

      let test_res = List.fold_left fold_fun true t_list in
      Runtest_pretty.print_res test.name test_res level
  | _ ->
      Runtest_pretty.print_title test.name test.doc level;
      try
        let (out,trace,ret_val) = do_test test opts' (verb = 2) in
        if verb = 1 then
          Runtest_pretty.print_indent (Str.split (Str.regexp "\n") out) level;
        Runtest_pretty.print_res test.name (ret_val = 0) level
      with Runtest_shell.Killed -> (
        Runtest_pretty.print_killed level;
        match test.expect with
        | Killed _ -> Runtest_pretty.print_res test.name true level
        | _ -> Runtest_pretty.print_res test.name false level
      )

(******************************************************************************)
(* PUBLIC INTERFACE                                                           *)
(******************************************************************************)

let runtest verb force_opts test_name =
  let test = find_test test_name in
  ignore( exec_test verb force_opts test 0)

let runfamily verb force_opts fam_name =
  let fam = find_family fam_name in
  List.iter (fun test -> ignore(exec_test verb force_opts (find_test test) 0)) fam

let get_test_code t_name =
  let test = find_test t_name in
  match test.body with
  | Some s -> (Runtest_pretty.acpp_body s,test.file)
  | None   -> raise (This_test_has_no_Body t_name)

