(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * pipeline.ml                                                    *
 *                                                                *
 * Version: $Id: pipeline.ml,v 1.551 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* Compile a program *)


(* TODO: the non-program compile code doesn't pay attention to all the
dump/typecheck etc options. Should rearrange so it does. *)

exception Typecheck_of_compiled of string
(* exception Fatal of string *)
exception Include_cycle
exception CheckfinalFail of string * int * string
exception Runtime_mismatch
exception Nonfinal

open Ast
open Util
open Exceptions
open Tysupp
open Basecon


(* The intention is that a program will either return a Result with some expr or some error will occur;
   a library will either compile successfully or some error will occur *)

(* type return_value = Success of Ast.definitions | Result of Ast.configuration | Error of exn *)
exception Library_error of string

let defs_start_mark = "\"Mark_definitions_lib_start\""
let defs_end_mark = "\"StdLib\""

(* one function per pass: deal with command line, parse, infertypes, compile program, compile library, run program, typecheck, finish *)
(* better with monads to carry state around? *)

(* can all functions return a return_value ? No ... can monads help things behave here? (eg feeding from one function into the next?) *)

(* reorganise so that typechecking etc can deal with library or program, when told which *)




(* let fatal s = prerr_endline ("Fatal: "^s); exit 1 *)

let print_err level skipdefs f =
  (match f with
    Compile ce ->
      begin
	match ce with
	  Lex (s) -> "Lexer error: " ^ s (* ^ Location.print () l *)
	| Parse s -> "Parse error: " ^ s (* ^ Location.print () l *)
	| Type (s) -> "Type error: " ^ s
	| Include ie -> "Include failure: " ^
	    (match ie with
	      Cycle -> "Cycle"
	    | System_error s -> "File not found: " ^ s
						      )
	| Nonfinal_expression -> "Non final expression"
	| Hashify he -> "Hashify failure: " ^
	    (match he with
	      Withspec_equation_from_import s -> "Withspec_equation_from_import: " ^ s
	    | Withspec_wrt_bad_type_field s -> "Withspec_wrt_bad_type_field: " ^ s
	    | Withspec_types_not_equal s -> "Withspec_types_not_equal: " ^ s
	    | Likespec_missing_type_fields s -> "Likespec_missing_type_fields: " ^ s
	    | Linkok_not s -> "Linkok failed: " ^ s
	    | Bad_sourcedef_valuability s -> "Bad_sourcedef_valuability " ^ s
	    | Nenv_merge_of_compiledunit -> "Nenv_merge_of_compiledunit"
						    )

	| Typecheck_of_compiledunit s -> "Typecheck of compiled unit failed: " ^ s
	| Runtime_mismatch s -> "Runtime mismatch: " ^ s
      end
  | Run re ->
      begin
	match re with
	  Typecheck_of_configuration c -> "Typechecking configuration failed: " ^ Pretty.print_configuration (Dump.fresh_printer_state level skipdefs false) c
	| Typecheck_on_unmarshal v -> "Fail typecheck on unmarshal of " ^ Pretty.print_marshalled_value (Dump.fresh_printer_state level skipdefs false) (Ast.Marshalled_value v)
	| Typecheck_on_marshal -> "Fail typecheck on marshal"
	| Typecheck_on_get_URI s -> "Fail typecheck on get uri. URI: " ^ s
      end
  | Internal ie ->
      begin
	match ie with
	  Never_happen s -> "Never_happen: " ^ s
	| Stuck s -> "Stuck: " ^ s
	| Unimplemented s -> "Unimplemented: " ^ s
      end
	) ^ "\n"

 let emitobject skip cu file =
      begin
        let ps = Pretty.initial_printer_state
            (Some Econst.string_of_ident)
            (Pretty.pm_objectcode skip) in
        let s = Pretty.print_compiled_unit ps cu in
        let ch = open_out file in
        try
          (* print_string "\naaaaa\n"; *)
          output ch s 0 (String.length s);
          flush ch;
          close_out ch
        with
          e -> close_out ch; raise e
      end


let run_program : int -> Ast.nameenv -> Ast.program -> Exceptions.top_exn
   = fun skipdefs ne (alldefs, expr) ->
      (* run *)
     Eval.skipdefs_ref := skipdefs;
     if !Opts.run then (
       Dump.showpass "Executing program";
       let em = () in (* (Evaltypes.Eval_typecheck_mode () in (!Opts.rttc,!Opts.showdefns,!Opts.terminate_on_rttc)) in *)
       (* try *)
       let (final_result) = Eval.execute_program (ne,(alldefs,expr)) in
       let final_typecheck config =
	  begin
	       if !Opts.rttc then (
		 Dump.showpass "Typechecking final state";
	      try
	      Typecheck.tcheck_configuration
		  {config with
            cfg_store = List.map (function (l,expr) -> (l, Evalsupp.flatten_all_closures [] expr))
              config.cfg_store;
            cfg_threads = NameValueMap.map (function ti -> {ti with ti_expr = Evalsupp.flatten_all_closures [] ti.ti_expr; ti_defs = List.map Evalsupp.def_flatten_all_closures ti.ti_defs})
              config.cfg_threads;
            cfg_defs = List.map Evalsupp.def_flatten_all_closures config.cfg_defs
          }

	      with
	      Typecheck.TCFail(s) ->
	       begin
                  if (!Opts.dumptrace >= !Opts.dumptypefail) then
                    Debug.print_string_really ("\nruntime typechecking failed with error: "^s^"\n")
                  else
                    Debug.print_string_really
                      ("\nRuntime typechecking failed for configuration\n"
                       ^ Pretty.print_configuration (Dump.fresh_printer_state !Opts.dumptypefail !Eval.skipdefs_ref false) config
                       ^ "\nwith error: "^s^"\n");
                  if (!Opts.terminate_on_rttc) then raise (Eval.Typecheck_of_configuration config)
               end
		  );
	  end
       in
	 match final_result with
	   (* Value(final_config) ->
	      final_typecheck final_config
	     final_result
	 | Acute_exception(st,e) ->
	      final_typecheck (st,Raise(e)); (* pity we have to put the raise back to typecheck :/ *)
	      final_result
	       *)
	 | Success (final_config) ->
	     final_typecheck final_config;
	     final_result
	 | Deadlock (final_config) ->
	     final_typecheck final_config;
	     final_result

	 | Failure(f) ->  (
	     final_result
	       )

	     (*
		with
		Eval.TerminateOnRTTC config -> (
		Debug.print_string_really (":::::::: Terminated on failure of runtime type check ::\n");
		if !Opts.dumptypefail>0 then
		Dump.dump_final "Final state" config !Opts.dumptypefail skipdefs  !Opts.dumptex true ;
		raise (Eval.TerminateOnRTTC config))

		| Eval.TerminateOnUnmarshalFailure (s,config) -> (
		Debug.print_string_really (":::::::: Terminated on unmarshal failure ::\n");
		Debug.print_string_really (s^"\n");
		if !Opts.dumptypefail>0 then
		Dump.dump_final "Final state" config !Opts.dumptypefail skipdefs !Opts.dumptex true ;
		raise (Eval.TerminateOnUnmarshalFailure (s,config)))
		| e -> Debug.print_string_really("Unexpected Error!: " ^ print_exception 4  e); raise e
		      *)
	 | Library _ -> raise (Util.Never_happen "Running program returned a library")
         | Emitsource b -> final_result
	       )
     else Library (alldefs) (* do this better? *)


(* convert return value into a string, based on the level options given *)
let string_of_result : int -> top_exn -> string
    = fun skipdefs result ->
      if (!Opts.dumpfinal=0) then
	""
      else
	let ps0 =
	try
	  Dump.fresh_printer_state !Opts.dumpfinal skipdefs !Opts.dumptex
	with Util.Never_happen e -> raise (Util.Never_happen ("String_of_result[1]: " ^ e))
	in
	let ps =
	  try
	  Pretty.add_typname_binders ps0 (List.map fst [])
	      with Util.Never_happen e -> raise (Util.Never_happen ("String_of_result[2]: " ^ e))
	in
	try
	  let s1 =
	    match result with
	      Library defs -> "Library:\n"
	(* | Value config -> "Value:\n"
	   | Acute_exception config -> "Exception:\n" *)
	    | Deadlock config -> "*** DEADLOCK *** \n"
	    | Success config -> "Success:\n"
	    | Failure f -> "Error:\n"
            | Emitsource b -> "Emit source:\n"
	  in let s2 =
	   match result with
	  Library defs -> Pretty.print_definitions ps defs
	(* | Value config ->  Pretty.print_configuration ps config
	| Acute_exception config -> "Exception:\n" ^ (Pretty.print_configuration ps config) *)
	| Deadlock config ->  Pretty.print_configuration ps config
	| Success config -> Pretty.print_configuration ps config
	| Failure f -> (print_err !Opts.dumpfinal skipdefs f)
        | Emitsource b -> if b then "OK\n" else "Error\n"
	   in
	   if (!Opts.dumpfinal = 1) then s1 else s1 ^ s2
	with Util.Never_happen e -> raise (Util.Never_happen ("String_of_result[3]: " ^ e))

let writefinal : int -> top_exn -> unit
    = fun skipdefs result_value ->
      (* (if not (!Opts.final_output_filename_opt = None) then *)
	let str =
	  string_of_result skipdefs result_value
	in
        let out_to_ch ch =
	  try
	    output ch str 0 (String.length str);
	    output ch "\n" 0 1;
	    flush ch;
	    Dump.safe_close_final_out ch
	  with
	    e -> Dump.safe_close_final_out ch; raise e
	in
	out_to_ch (Dump.safe_open_dumpfinal_out true);
	if not (!Opts.final_output_filename_opt = None) then out_to_ch (Dump.safe_open_final_out true)



let checkfinal : int -> top_exn -> unit
    = fun skipdefs result_value ->
      (if not (!Opts.final_check_filename_opt = None) then
	try (
	  let str =
	    string_of_result skipdefs result_value
	  in
	  let ch = Dump.safe_open_check_in true in
	    try
	      let buf = String.create (String.length str) in
	      let in_length = input ch buf 0 (String.length str) in
	      (if (in_length <> String.length str) || ((String.compare buf str) <> 0)  then raise (CheckfinalFail (buf, in_length, str))
	      else  Debug.print_string_really(":::::::: Final checking passed ::\n");
	       );
	      Dump.safe_close_check_in ch
	    with
	      e -> Dump.safe_close_check_in ch; raise e
		  )
	with
	  CheckfinalFail (buf, in_length, str) -> (
	    Debug.print_string_really(":::::::: Final checking failed ::\n");
	    let buf_sub = String.sub buf 0 in_length in
	    Debug.print_string_really (":: " ^ "Result: '" ^ str ^ "' did not match saved: '" ^ buf_sub ^ "' ::\n")
	      )
		    (* A line saying 'blah did not match blah'? But need to get rid of any random characters from buf first. *)

	    );()


(* deal with lib somehow *)
(* what's different about lib?  it magics up the parser state, and
   uses a different one; it returns some funny marks for some reason;
   ... what else? *)
(* use some records *)


(* generate a filename from a root filename
	may later want to modify this with a list of root paths,
   and check for the existences of the file?
*)
let abs_filename filename root_filename =
  if String.get filename 0 = '/' then filename else
  let root_path = (* assume no / *)
    String.sub root_filename 0
      (
    try (String.rindex root_filename '/')+1 with
      Not_found -> 0 (* (String.length root_filename) *)
	  )
  in root_path ^ filename

type pretty_states = {
    ps_parsed   : Pretty.printer_state ref option;
    ps_preinf   : Pretty.printer_state ref option;
    ps_postinf  : Pretty.printer_state ref option;
    ps_desugared: Pretty.printer_state ref option;
    ps_compiled : Pretty.printer_state ref option;
}

type compile_state = {
    ps   : Parsertypes.parser_state;
    ppss : pretty_states;
    defs : definitions;
    cm   : Hashifysupp.compilation_map;
    en   : Ast.nameenv;      (* nameenv *)
    env0 : typeenv;          (* source or semicompiled environment *)
    env1 : typeenv;          (* compiled environment *)
 };;

type compile_env = {
    seenfiles : string list;
    prefix : string;   (* for visual indication of recursion depth *)
    islib : bool;
    filename : string option(* current filename *)
  };;

(* in lieu of using a monad (which is not well supported in pure
   OCaml), we pass around two records.  This gives us most of the
   advantages of a monad (we can add and remove fields without
   touching the code) but not all.

   ce:compile_env are values that are passed downwards only (inherited
   attributes, if you like).  cs:compile_state are values that are
   threaded through (state, like it says).  Thus env things are
   visible only within a scope, whereas state things are visible to
   the entire remainder of the computation.

   We don't have any write-only attributes (dual to env) but we could
   have.
*)

(* XXX should thread the pretty-state from the dumps through compiled imports as well *)
let rec compile : compile_env -> compile_state -> Parsertypes.parser_state compilation_unit -> compile_state * prim_expr option
  = fun ce cs cu
 -> let skipdefs = if ce.islib then max_int else 0 in
    match cu with
      ([],None) ->
        (cs, None)

    | ([],Some e_fun) ->
        Dump.showpass (ce.prefix ^ "Scope-resolve expression");
        let (e,ps') = e_fun cs.ps in
        let metas = Typecheck.freetyvars_expr [] e in  (* expr is black *)
        Dump.dump_expr cs.ppss.ps_parsed "Parsed and scope-resolved" e (*metas*);
          (* (since we couldn't print it right after parsing as requested, we print it here,
              at the first opportunity) *)
        Dump.showpass (ce.prefix ^ "Infer expression");
        Dump.dump_expr cs.ppss.ps_preinf "Pre-inference" e (*metas*);
        let (e,ty) = Typecheck.tyinf_expr (if !Opts.default then Some(TTyCon0 TUnit) else None) metas (cs.en, cs.env0) [] e
                                          (if !Opts.nonunitthread then None else Some (TTyCon0 TUnit)) in
        Dump.dump_expr cs.ppss.ps_postinf "Inferred" e;
        Dump.showpass (ce.prefix ^ "Desugar expression");
        let e = if !Opts.desugar then Desugar.desugar_expr e else e in
        Dump.dump_expr cs.ppss.ps_desugared "Desugared" e;
        Dump.showpass (ce.prefix ^ "Compile expression");

        let en' = cs.en in
	(* XXX get rid of the conversion from expr to prim_expr *)
	(* TODO: More tasteful to have the parser to do this check *)
	let ep = exprtoprim e in
        if not (Hashifysupp.no_cfresh None ep) then raise (Misc.Syntax_error "cfresh in main expression");
	if not (Hashifysupp.no_hashMvf None ep) then raise (Misc.Syntax_error "hash(M.x) in main expression");
	if not (Hashifysupp.no_tie None ep) then raise (Misc.Syntax_error "tie in main expression");


        let e' = Hashify.rho_expr cs.cm e in
        let ty = Hashify.rho_type cs.cm ty in
        Dump.dump_expr cs.ppss.ps_compiled "Compiled" e';
        if !Opts.typecheckcompiled then begin
          Dump.showpass (ce.prefix ^ "Typecheck compiled expression");
          (try
            let (_:typ) = Typecheck.tcheck_expr CompiledForm (en',cs.env1) [] e' (Some ty) in
            ()
          with
            Typecheck.TCFail(s) ->
              if (!Opts.terminate_on_rttc) then (
                raise (Typecheck_of_compiled s)
		  )
		  )
        end;
        Dump.showpass (ce.prefix ^ "Strip location information");
        let e' = exprtoprim e' in
        Dump.showpass (ce.prefix ^ "Done expression");
        ({ cs with
           ps = ps'; en = en' },
         Some e')

    | (CUDef sdef_fun::cu,eo) ->
        Dump.showpass (ce.prefix ^ "Scope-resolve source definition");
        let (sdef,ps') = sdef_fun cs.ps in
        let metas = Typecheck.freetyvars_source_definition sdef in
        Dump.dump_source_definitions cs.ppss.ps_parsed  "Parsed and scope-resolved" [sdef] (*metas*) ce.islib;
          (* (since we couldn't print it right after parsing as requested, we print it here,
              at the first opportunity) *)
        Dump.showpass (ce.prefix ^ "Infer source definition");
        Dump.dump_source_definitions cs.ppss.ps_preinf "Pre-inference" [sdef] (*metas*) ce.islib;
        let (sdef,env0') = Typecheck.tyinf_source_definition (if !Opts.default then Some(TTyCon0 TUnit) else None) metas (cs.en,cs.env0) sdef in
        Dump.dump_source_definitions cs.ppss.ps_postinf "Inferred" [sdef] ce.islib;
        Dump.showpass (ce.prefix ^ "Desugar source definition");
        let sdef = if !Opts.desugar then Desugar.desugar_sdef sdef else sdef in
        Dump.dump_source_definitions cs.ppss.ps_desugared "Desugared" [sdef] ce.islib;
        Dump.showpass (ce.prefix ^ "Compile source definition");

        let (en',def',cm') =
	  (match sdef with
	    Mark_user m ->  (cs.en, Mark_compile m, cs.cm)
	  | _ ->


	          let vubs = Hashifysupp.derive_valuabilities cs.cm sdef in
                  Dump.showpass (ce.prefix ^ "Valuabilities derived: "
				 	       );
                  ( match mode_of_source_def(sdef) with
                      | MHash | MBangHash | MCfresh | MBangCfresh ->
                          let (en',def') = Hashify.hashify_sdef cs.en cs.cm sdef vubs in
                          let (cm') = Hashifysupp.add_compiled_modules cs.cm [def'] in
 	                    (en', def', cm')
                      | MFresh ->
                          let def' = sdef_to_fresh_def sdef in
                          let (cm') = Hashifysupp.add_compiled_modules cs.cm [def'] in
                            (cs.en,def',cm') )) in
        (* let env' = Hashify.hashify_env cm' env0' in      FZ THIS IS NO LONGER USED: VERY SUSPICIOUS!!! *)
        let env1' = env1_of_defs [def'] in
        Dump.dump_definitions cs.ppss.ps_compiled "Compiled" [def'] ce.islib;
        if !Opts.typecheckcompiled then begin
          Dump.showpass (ce.prefix ^ "Typecheck compiled definition");
          (try
            let (_:typeenv) = Typecheck.tcheck_definition (en',cs.env1) def' (Some env1') in
            ()
          with
            Typecheck.TCFail(s) ->
              (* Debug.print_string_really ("Typecheck failed: "^s^"\n"); *)
              if (!Opts.terminate_on_rttc) then (
   (* Debug.print_string_really (":::::::: Terminating on failure of compiled definition type check ::\n"); *)
                raise (Typecheck_of_compiled s)
		)
	    )
        end;
        Dump.showpass (ce.prefix ^ "Done source definition");
        compile ce { cs with
                     ps   = ps';
                     defs = cs.defs @ [def']; (* XXX inefficient @ *)
                     cm   = cm';
                     en   = en';
                     env0 = appendtypeenv env0' cs.env0;  (* XXX or perhaps take the sign0? *)
                     env1 = appendtypeenv env1' cs.env1 } (cu,eo)

    | (CUSourceString str::cu,eo) ->
        Dump.showpass (ce.prefix ^ "Parse internal source string");
        let cu' = Parsetools.parse_compilation_unit_from_string str in
        let metas = [] in  (* [] is wrong, but this will change soon anyway *)
        Dump.dump_cu "Parsed" cu' metas !Opts.dumpparse skipdefs !Opts.dumptex false;
        Dump.showpass (ce.prefix ^ "Process internal source string");
        let (cs', eo') = compile { ce with
                                   prefix = ce.prefix ^ "+" }
                                 cs cu' in
        Dump.showpass (ce.prefix ^ "Done internal source string");
        compile_rest ce cs' eo' (cu,eo)

    | (CUIncludeSource(filename,object_filename)::cu,eo) ->
        let filename' = (
          match ce.filename with Some root_filename ->
            abs_filename filename root_filename | None -> filename) in
	Dump.showpass (ce.prefix ^ "Include source file " ^ filename');
        if List.mem filename' ce.seenfiles then
	    raise Include_cycle;
        Dump.showpass (ce.prefix ^ "Parse source file");
        let cu' =
          try
            Parsetools.parse_compilation_unit_from_filename filename'
          with Myparsing.Runtime_mismatch e' -> raise (Myparsing.Runtime_mismatch ("Include: " ^ e')) (* TODO: Can we get lithashes in an uncompiled file anyway? *)
        in
	let metas = [] in  (* [] is wrong, but this will change soon anyway TODO: erm, how soon? *)
        Dump.dump_cu "Parsed" cu' metas !Opts.dumpparse skipdefs !Opts.dumptex false;
        Dump.showpass (ce.prefix ^ "Process source file");
        let n_old_defs = List.length cs.defs in
        let (cs', eo') = compile { ce with
				   seenfiles = filename :: ce.seenfiles;
                                   prefix = ce.prefix ^ "+";
				   filename = Some filename'
				 }
                                 cs cu' in
        (match object_filename with
          None -> ()
        | Some object_filename ->
            Dump.showpass (ce.prefix ^ "Emitting object file");
            let eo'' = match eo' with None -> None | Some e -> Some (primtoexpr e) in
            emitobject n_old_defs (cs'.en,cs'.defs,eo'') object_filename
        );
        Dump.showpass (ce.prefix ^ "Done source file");
        compile_rest ce cs' eo' (cu,eo)

    | (CUIncludeCompiled filename::cu,eo) ->
	  let filename' = (
          match ce.filename with Some root_filename ->
            abs_filename filename root_filename | None -> filename) in
        Dump.showpass (ce.prefix ^ "Parse compiled file " ^ filename');

        let ((en', defs', eo'),ps') =
	  try
	    Parsetools.parse_compiled_unit_from_filename filename' cs.ps
          with Myparsing.Runtime_mismatch e' -> raise (Myparsing.Runtime_mismatch ("Include: " ^ e'))
	in
        Dump.dump_defs_eo "Parsed" (defs',eo') [] !Opts.dumpparse skipdefs !Opts.dumptex false;
        let en'' = Evalsupp.merge_nameenv (cs.en,en') in (* TODO:
							    catch error and convert to compilation error *)
        let env0' = env0_of_defs defs' in
        let env1' = env1_of_defs defs' in
        if (!Opts.typecheckcompiled) then begin
          Dump.showpass (ce.prefix ^ "Typecheck compiled file");
	  Debug.print' Opts.DBC_namevalues (fun () -> Pretty.print_nameenv (Dump.fresh_printer_state 3 0 false) en'');
          let (_,tyo) = Typecheck.tcheck_cdefs_eo (en'',cs.env1) (defs',eo') (Some (env1',(if is_None eo' || !Opts.nonunitthread then None else Some (TTyCon0 TUnit)))) in
          ()
        end;
        Dump.showpass (ce.prefix ^ "Strip location information");
        let eo' = option_lift exprtoprim eo' in
        Dump.showpass (ce.prefix ^ "Done compiled file");
        let cm' = Hashifysupp.add_compiled_modules cs.cm defs' in
        let cs' =  { cs with
                     ps   = ps';
                     defs = cs.defs @ defs'; (* XXX inefficient @ *)
                     cm   = cm';
                     en   = en'';
                     env0 = appendtypeenv env0' cs.env0;
                     env1 = appendtypeenv env1' cs.env1 } in
        compile_rest ce cs' eo' (cu,eo)

and compile_rest : compile_env -> compile_state -> prim_expr option -> Parsertypes.parser_state compilation_unit -> compile_state * prim_expr option
  = fun ce cs eo' (cu,eo)
 -> match (eo',(cu,eo)) with
      (Some e',([],None)) -> (cs, Some e')  (* expr must be last *)
    | (None,_)            -> compile ce cs (cu,eo)
    | _                   -> raise Nonfinal



let include_of_file : bool -> Opts.filename -> 'a compilation_unit_definition
  = fun emit fi
 -> if fi.Opts.fi_compiled then
      CUIncludeCompiled fi.Opts.fi_filename
     else
      CUIncludeSource (fi.Opts.fi_filename,
                       if emit then
                         if !Opts.emitobject then
                           Some (fi.Opts.fi_basename ^ ".aco")
                         else
                           !Opts.emitobjectfile
                       else
                         None)
;;

let dotestparser ce0 cs0 cu =
  let ps = Pretty.initial_printer_state (Some Econst.string_of_ident) (Pretty.pm_marshal ()) in
  let s = Pretty.print_compiled_unit ps cu in
  try
     let (cu', ps1') = Parsetools.parse_compiled_unit_from_string s cs0.ps in
  let s' =  Pretty.print_compiled_unit ps cu' in
  if s = s' then (prerr_string "OK:\n"; exit(0))
            else (prerr_string "Fail (1):\n";
                  prerr_string s;
                  print_string "\n(2):\n";
                  print_string s';
                  exit(1))
  with
     exc -> prerr_string "Fail (1):\n";
            prerr_string s;
            prerr_string "\n";
            raise exc
;;

let emitsource ifile ofile =
  Emitsource (Sys.command ("cp \"" ^ String.escaped ifile.Opts.fi_filename ^ "\" \"" ^ String.escaped ofile ^ "\"") = 0)
;;

let mk_printer_state dumplevel =
  if dumplevel = 0 then
       None
  else Some (ref (Dump.fresh_printer_state dumplevel (*skipdefs=*)0 !Opts.dumptex))

let main () =
   match !Opts.emitsourcefile, !Opts.input_filenames with
     Some ofile, [ifile] -> emitsource ifile ofile
   | Some _, _  ->
                      let errstring = "Unexpected error: Bad use of -emitsourcefile" in (
                      let out_ch = Dump.safe_open_dumpfinal_out true in
                      try
                        output out_ch errstring 0 (String.length errstring);
                        output out_ch "\n" 0 1;
                        flush out_ch;
                        Dump.safe_close_final_out out_ch
                      with e -> Dump.safe_close_final_out out_ch; raise e);
                      exit(4)
   | None,  _   ->

         Dump.showpass "Begin";

  try (

    let dslib_parser_mode = (Parsertypes.Program_mode
        (Parsertypes.Source, Parsertypes.Econsts_allowed, Parsertypes.InStdLib), []) in

    let ce0 = { seenfiles = [];
                prefix = "";
                islib = false;
                filename = None
	      } in
    let cs0 = { ps = Parsertypes.new_parser_state
                       dslib_parser_mode (Some (Econst.ident_of_string, Econst.string_of_ident));
                ppss = { ps_parsed    = mk_printer_state !Opts.dumpparse;
                         ps_preinf    = mk_printer_state !Opts.dumppreinf;
                         ps_postinf   = mk_printer_state !Opts.dumppostinf;
                         ps_desugared = mk_printer_state !Opts.dumpdesugared;
                         ps_compiled  = mk_printer_state !Opts.dumpcompiled;
                       };
                defs = [];
                cm   = Hashifysupp.empty_cm;
               	en   = emptynameenv;
                env0 = Tysupp.econst_env;
                env1 = Tysupp.econst_env } in

    Dump.showpass "Process library";

    let lib_cu = ([CUSourceString ("mark " ^ defs_start_mark)]
                  @ (match !Opts.dslib_filename_opt with
                    None -> []
                  | Some dslib_filename ->
		      [include_of_file false dslib_filename])
                  @ [CUSourceString ("mark " ^ defs_end_mark)],
                  None) in

    let (cs1, eo) = compile { ce0 with prefix = "Lib: "; islib = true } cs0 lib_cu in

    (* record the size of definitions_lib for dump-printing purposes *)
    let definitions_lib_size = List.length (cs1.defs) in
    let result =
      try
	Dump.showpass "Process input files";

	let main_parser_mode = (Parsertypes.Program_mode
            (Parsertypes.Source, Parsertypes.Econsts_not_allowed, Parsertypes.InUserCode),
				if (!Opts.lithash) then [Parsertypes.Lithash] else []) in

	let cs2 = { cs1 with
                    ps = Parsertypes.replace_mode_opts
                      cs1.ps main_parser_mode; } in

	let main_cu = (List.map (include_of_file (!Opts.emitobject || !Opts.emitobjectfile <> None)) !Opts.input_filenames,
                       None) in

	let (cs3, eo) = compile ce0 cs2 main_cu in
        (if !Opts.parsetest then
         let eo' = match eo with None -> None | Some e -> Some (primtoexpr e) in
         dotestparser ce0 cs0 (cs3.en,cs3.defs, eo'));

	Dump.showpass "Run";

	match eo with
          Some expr ->  (* program *)
	    run_program definitions_lib_size cs3.en (cs3.defs, expr)
	| None ->  (* library only *)
	    Library cs3.defs
      with
	Lextools.Lex_error (s, l) -> Failure (Compile (Lex (s
							    ^ "\n" ^ Location.print () l)))
      |	Parsetools.Wrapped_parse_error s -> Failure(Compile (Parse s))
      |	Misc.Syntax_error s -> Failure(Compile (Parse (s)))
      |	Myparsing.Runtime_mismatch s -> Failure (Compile (Runtime_mismatch s))
      | Myparsing.Unbound_identifier s -> Failure (Compile (Parse ("Unbound identifier: " ^ s)))
      |	Typecheck.TCFail s -> Failure(Compile (Type s))
      |	Hashify.Hashify he -> Failure(Compile (Hashify he))
      |	Hashifysupp.Hashify he -> Failure(Compile (Hashify he))
      |	Util.Never_happen s -> Failure (Internal (Never_happen s))
      | Eval.Stuck s  	     -> Failure (Exceptions.Internal (Stuck s))
      | Util.Unimplemented s  -> Failure (Internal (Unimplemented s))
      | Eval.Typecheck_of_configuration c 	-> Failure (Run (Typecheck_of_configuration c))
      | Eval.Typecheck_on_unmarshal v  -> Failure (Run (Typecheck_on_unmarshal v))
      | Eval.Typecheck_on_marshal -> Failure (Run (Typecheck_on_marshal))
      | Eval.Typecheck_on_get_URI s -> Failure(Run(Typecheck_on_get_URI s))
      | Eval.Runtime_mismatch s -> Failure (Compile (Runtime_mismatch s))
      |	Include_cycle -> Failure (Compile (Include (Cycle)))
      |	Sys_error s -> Failure (Compile (Include (System_error s)))
      |	Typecheck_of_compiled s -> Failure (Compile (Typecheck_of_compiledunit s))
      |	Nonfinal -> Failure (Compile (Nonfinal_expression))
      |	Ast.Filter_sign_missing_fields -> Failure (Compile (Hashify (Likespec_missing_type_fields "")))

      | e -> Failure(Internal(Never_happen ("Unknown error: " ^ Printexc.to_string e)) )

    in
    (match result with Success _ | Deadlock _ | Library _  ->
    Dump.showpass "Done";
  | _ -> ());
    writefinal definitions_lib_size result;
    checkfinal definitions_lib_size result;
    result
        )
	      with
		e -> ((* should never happen *) (
                      let errstring = "Unexpected error: " ^ Printexc.to_string e in (
		      let out_ch = Dump.safe_open_dumpfinal_out true in
		      try
			output out_ch errstring 0 (String.length errstring);
			output out_ch "\n" 0 1;
			flush out_ch;
			Dump.safe_close_final_out out_ch
		      with e -> Dump.safe_close_final_out out_ch; raise e);


         (* do something with that string... which chan to print on? *)
                      exit(3)))

  ;;

(* --- REPLING --- *)

let rec scontains s str =
  try
  let start = String.get str 0 in
  let ssub = String.sub s (String.index s start) (String.length str)
  in if (String.compare ssub str)=0 then true
  else scontains
      (String.sub s ((String.index s start)+1) ((String.length s) - (String.index s start) - 1))
      str
  with _ -> false


(* loop 2: read characters until two ;; are read *)

let getbuf () =
  let rec loop prompt buf =
    let s =(match Editline.editline prompt with
      Some s -> s
           | None -> raise End_of_file) in
    if scontains s ";;" then
      buf ^ "\n" ^ s
    else loop "  " (buf ^ "\n" ^ s)
  in
  loop "# " ""


(* loop 1 : collect a string,
   run compile, add defs to collection *)


let rec repl () =
   try
   let s = (* (match Editline.editline "# " with
             Some s -> s
           | None -> raise End_of_file) *) getbuf () in
   let (f_name,chan) = Filename.open_temp_file "repl_" ".ac" in
   output_string chan s;
   close_out chan;
   let basename = String.sub f_name 0 ((String.length f_name) - 2) in
   Opts.input_filenames := [{Opts.fi_filename=f_name;Opts.fi_basename=basename;Opts.fi_compiled=false}];
   let result = main() in (); (* store defs, create an 'it' variable? *)
   Eval.reset();
   repl ();
   Unix.unlink f_name
   with End_of_file -> (
     print_string "\n";
     flush stdout;
     exit(0)
    )

