(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * high-level pretty-printing used by pipeline.ml and eval.ml.    *
 *                                                                *
 * Version: $Id: dump.ml,v 1.507 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)


(* Can't go in pretty.ml as it needs Econst, and in any case is
logically distinct. *)

open Util

let fresh_printer_state dumplevel skipdefs dumptex =
  Pretty.initial_printer_state (Some Econst.string_of_ident) (Pretty.pm_user dumplevel skipdefs dumptex)

let fresh_printer_state_all () =
  fresh_printer_state 4 0 false

let safe_open_out to_outfile =
  let ch_out = match to_outfile, !Opts.output_filename_opt with
  | (false,_)
  | (_,None) -> stderr
  | (true,Some output_filename) -> open_out output_filename in
  ch_out

let safe_open_dumpfinal_out to_outfile =
  let ch_out = match to_outfile, !Opts.final_filename_opt with
  | (false,_)
  | (_,None) -> stdout
  | (true,Some output_filename) -> open_out output_filename in
  ch_out


let safe_close_out ch = if ch <> stderr then close_out ch

let safe_open_final_out to_outfile =
  let ch_out = match to_outfile, !Opts.final_output_filename_opt with
  | (false,_)
  | (_,None) -> raise (Util.Never_happen "Trying to dump result without a filename")
  | (true,Some output_filename) -> open_out output_filename in
  ch_out

let safe_close_final_out ch = if ch <> stdout then close_out ch

let safe_open_check_in to_infile =
  let ch_in = match to_infile, !Opts.final_check_filename_opt with
  | (false,_)
  | (_,None) -> raise (Util.Never_happen "Trying to check result against contents of nothing!")
  | (true,Some input_filename) -> open_in input_filename in
  ch_in

let safe_close_check_in ch = if ch <> stdin then close_in ch



let last_utime = ref None

let showpass s =
  if !Opts.showpasses then
    (if !Opts.showtimes then
      (let t1 = (Unix.times ()).Unix.tms_utime in
      (match !last_utime with
        None    -> ()
      | Some t0 -> let s = Printf.sprintf ":: %3.3fs user time\n" (t1 -. t0) in Debug.print_string_really s);
      last_utime := Some t1)
    else
      ();
     Debug.print_string_really (":::::::: " ^ s ^ " ::\n"))
  else
    ()

(* NB: metas argument is (internal_typname * 'a), where 'a is ignored,
   to allow for the colour info returned by Typecheck.freetypnames *)
let dump_generic title f x metas dumplevel skipdefs dumptex to_outfile =
  if dumplevel=0 then () else
  (* let ch_out = safe_open_out to_outfile in *)
  try
    let ps0 = fresh_printer_state dumplevel skipdefs dumptex in
    let ps = Pretty.add_typname_binders ps0 (List.map fst metas) in
    let s_preamble = title ^ " dump:\n" in
    let s_postamble = "\n*****\n" in
    (* output ch_out s 0 (String.length s); *)
    (* flush ch_out; *)
    let s = f ps x in
    (* output ch_out s 0 (String.length s); *)
    (* flush ch_out; *)

    if s = "" then (* horrid hack to be quiet for skipped definitions *)
      ()
    else
      let s = if dumptex then
        Str.global_replace (Str.regexp "_") "\\_" s
      else
        (s_preamble ^ s ^ s_postamble)
      in
      Debug.print_string_really s;

      (* safe_close_out ch_out *)
  with
    e -> (* safe_close_out ch_out; *) raise e

let dump title alldefs expr metas dumplevel skipdefs locs dumptex to_outfile =
  dump_generic title (fun ps () ->
    let (s1,ps) = match alldefs with
                    Inl sdefs -> Pretty.print_source_definitions_ps ps sdefs
                  | Inr cdefs -> Pretty.print_definitions_ps ps cdefs in
    let  s2     = match expr with
                    Some e ->
		      if (locs) then (Pretty.print_expr_loc ps e)
		      else Pretty.print_expr ps e
                  | None   -> "" in
    s1^s2) () metas dumplevel skipdefs dumptex to_outfile


let dump_generic2 myps title f =
  match myps with
    None -> ()
  | Some psr ->
        let s_preamble = title ^ " dump:\n" in
        let s_postamble = "\n*****\n" in
        let (s,ps') = f !psr in
        (* Update the pretty-printing state if necessary *)
        (match ps' with None -> () | Some ps -> psr := ps);

        if s = "" then (* horrid hack to be quiet for skipped definitions *)
          ()
        else (
          let s = if !Opts.dumptex then
            Str.global_replace (Str.regexp "_") "\\_" s
          else
            (s_preamble ^ s ^ s_postamble)
          in
          Debug.print_string_really s
        )


let dump_expr myps title e =
  dump_generic2 myps title
    (fun ps ->
       if !Opts.showlocs then (Pretty.print_expr_loc ps e, None)
                         else (Pretty.print_expr     ps e, None)
    )

let dump_source_definitions myps title ds skip =
  dump_generic2 myps title
    (fun ps ->
       let ps = if skip then Pretty.set_ignore_defns ps None
                        else Pretty.set_ignore_defns ps (Some 0) in
       let (s1,ps') = Pretty.print_source_definitions_ps ps ds in (s1, Some ps')
    )

let dump_definitions myps title ds skip =
  dump_generic2 myps title
    (fun ps ->
       let ps = if skip then Pretty.set_ignore_defns ps None
                        else Pretty.set_ignore_defns ps (Some 0) in
       let (s1,ps') = Pretty.print_definitions_ps ps ds in (s1, Some ps')
    )


let dump_prim title alldefs expr metas dumplevel skipdefs dumptex to_outfile =
  dump_generic title (fun ps () ->
    let (s1,ps) = match alldefs with
                    Inl sdefs -> Pretty.print_source_definitions_ps ps sdefs
                  | Inr cdefs -> Pretty.print_definitions_ps ps cdefs in
    let  s2     = match expr with
                    Some e -> Pretty.print_prim_expr ps e
                  | None   -> "" in
    s1^s2) () metas dumplevel skipdefs dumptex to_outfile


let dump_cu title cu metas dumplevel skipdefs dumptex to_outfile =
  dump_generic title Pretty.print_compilation_unit cu metas dumplevel skipdefs dumptex to_outfile

let dump_defs_eo title (ds,eo) metas dumplevel skipdefs dumptex to_outfile =
  dump_generic title (fun ps () ->
    let (s1,ps) = Pretty.print_definitions_ps ps ds in
    let  s2     = match eo with
                    Some e -> Pretty.print_expr ps e
                  | None   -> "" in
    s1^s2) () metas dumplevel skipdefs dumptex to_outfile

let dump_final title x dumplevel skipdefs dumptex to_outfile =
  let f = Pretty.print_configuration in
  if (dumplevel = 0) then () else
  let ch_out = safe_open_dumpfinal_out to_outfile in
  try
    let ps0 = fresh_printer_state dumplevel skipdefs dumptex in
    let ps = Pretty.add_typname_binders ps0 (List.map fst []) in
    let s = title ^ " dump:\n" in
    output ch_out s 0 (String.length s);
    flush ch_out;
    let s = f ps x ^ "\n*****\n" in
    output ch_out s 0 (String.length s);
    flush ch_out;
    safe_close_out ch_out
  with
    e -> safe_close_out ch_out; raise e

