(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * ast.mli                                                        *
 *                                                                *
 * Datatypes for MiniCaml abstract syntax trees.                  *
 * (Returned by the parser and used everywhere).                  *
 *                                                                *
 * Version: $Id: ast.mli,v 1.616 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Basecon

type 'a prettyname = 'a name * string

val string_of_prettyname : 'a prettyname -> string
val fresh_prettyname : 'a prettyname -> 'a prettyname
val swap_prettyname : 'a prettyname -> 'a prettyname -> 'b -> 'b
val name_of_prettyname : 'a prettyname -> 'a name
val string_of_name : 'a name -> string
val debug_string_of_prettyname : 'a prettyname -> string

type external_typname
type internal_typname_T
type internal_typname = internal_typname_T prettyname
type typname = external_typname * internal_typname

type external_ident
type internal_ident_T
type internal_ident = internal_ident_T prettyname
type ident = external_ident * internal_ident

type external_modname = string
type internal_modname_T
type internal_modname = internal_modname_T prettyname
type modname = external_modname * internal_modname

val eid_compare : external_ident -> external_ident -> int

module IIdentMap :
  sig
    type key = internal_ident
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

module ITypnameMap :
  sig
    type key = internal_typname
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

module ModnameMap :
  sig
    type key = modname
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

module ModnameSet :
    sig
      type elt = modname
      type t
      val empty : t
      val is_empty : t -> bool
      val mem : elt -> t -> bool
      val add : elt -> t -> t
      val singleton : elt -> t
      val remove : elt -> t -> t
      val union : t -> t -> t
      val inter : t -> t -> t
      val diff : t -> t -> t
      val compare : t -> t -> int
      val equal : t -> t -> bool
      val subset : t -> t -> bool
      val iter : (elt -> unit) -> t -> unit
      val fold : (elt -> 'a -> 'a) -> t -> 'a -> 'a
      val for_all : (elt -> bool) -> t -> bool
      val exists : (elt -> bool) -> t -> bool
      val filter : (elt -> bool) -> t -> t
      val partition : (elt -> bool) -> t -> t * t
      val cardinal : t -> int
      val elements : t -> elt list
      val min_elt : t -> elt
      val max_elt : t -> elt
      val choose : t -> elt
    end


module EModnameMap :
  sig
    type key = external_modname
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

type location = int  (* reference cell names *)

val fresh_location : unit -> location

module LocMap :
  sig
    type key = location
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

module EIdentMap :
  sig
    type key = external_ident
    and 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

module ETypnameMap :
  sig
    type key = external_typname
    and 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

type 'a located = { desc : 'a; loc : Location.t }
val gh_loc : Location.t -> 'a -> 'a located
val at_loc : Location.t -> 'a -> 'a located
val no_loc : 'a -> 'a located

type typ  =
  | TTyCon0 of tycon0
  | TTyCon1 of tycon1 * typ
  | TTup of typ list
  | TSum of typ list
  | TTyName of abstract_name
  | TXDot of hash_or_modname * external_typname
  | TFunc of typ * typ
  | TVar of internal_typname
  | TForall of internal_typname * typ
  | TExists of internal_typname * typ

and kind = KType | KEq of typ
and hash = (* use Pretty.mkHashM and Pretty.mkHashI *)
    HashM of Digest.t * external_modname * mod_hash_body
  | HashI of Digest.t * external_modname * import_hash_body
  | HashLit of Digest.t
  | HashName of abstract_name

and hash_or_modname = Hash of hash
                    | Modname of modname
and eqn =
    EHash of hash * external_typname * typ
  | EMod of modname * external_typname * typ
and eqs = eqn list

and abstract_name = Digest.t

and con0 =
        | Nil of typ   (* the type of the whole expression is: typ list *)
        | NONE of typ  (* the type of the whole expression is: typ option *)
        | B0 of basecon0

and con1 =
  | Inj of int * typ
  | SOME
  | TieCon
  | Node
  | B1 of basecon1

and bracket_seq = (eqs * typ) list

and op_or_econst = OEOp of op
                 | OEEconst of internal_ident

and lazy_op = LoAnd | LoOr

and name_value =
    VHashMvf  of abstract_name * hash * external_ident * typ
  | VHashTs   of abstract_name * typ * string
  | VHashHts  of abstract_name * typ * string * name_value
  | VHashName of abstract_name * typ


(* Expressions *)
and expr_desc =
          LocC0 of con0
        | LocC1 of con1 * expr
        | LocCons of expr * expr
        | LocTup of expr list
        | LocOp of op_or_econst * expr list
              (* this is a sugared form: the source form requires the
                 list to be empty, and the core form requries that the
                 list has length n, where n is the arity of the op *)
        | LocLoc of location
        | LocFn  of mtch  (* this is a sugared form; the core form has mtch = [(PVar(x,t),e)] *)
        | LocFun  of pat list * expr   (* this is a sugared form; fun p1 .. pn -> e *)
        | LocTAbs of internal_typname * expr  (* type abstraction *)
        | LocInEnv of etysubst * expr
              (* this is a runtime-only form *)
              (* it is a delayed substitution, binding *all* vars in expr *)
              (* See flattenclos for the precise meaning of the term *)
        | LocClos of etysubst Lazy.t
              * internal_ident * typ
              * bracket_seq * expr
              * (internal_ident * typ * bracket_seq) option
              (* this is a runtime-only form *)
              (* remember to freshen_clos *)
              (* Clos(env,x2,ty2,bs2,e1,None)              - nonrecursive closure
                 Clos(env,x2,ty2,bs2,e1,Some(x1,ty1,bs1))  - recursive closure
                 env      - substitution environment
                 x2 : ty2 - argument
                 bs2      - brackets to apply around x2 in e1
                 e1       - body
                 x1 : ty1 - name of this closure in env
                 bs1      - brackets to apply around x1 in e1
              *)
              (* Closures carry an environment for their free variables.
                 This is (often) hidden from the printed
                 representation and from the typechecker.
                 See flattenclos for the precise meaning of the term. *)
        | LocTClos of etysubst
              * internal_typname
              * expr
              (* this is a runtime-only form *)
              (* remember to freshen *)
              (* TClos(menv,tenv,itn,e)   (type closures are never recursive!)
                 (menv,tenv)  - substitution environment
                 itn          - argument (always of kind Type)
                 e            - body
                 NB: bracket-pushing through closure is *weird*
                 because tyvar ends up at different colour. Hmm.
              *)
        | LocId of internal_ident
        | LocDot of modname * external_ident  (* module projection *)
        | LocHashDot of hash * external_ident (* hash projection *)
        | LocIf of expr * expr * expr
        | LocWhile of expr * expr
        | LocLazyOp of lazy_op * expr list
        | LocSeq of expr * expr
        | LocApp of expr * expr
        | LocTApp of expr * typ               (* type application *)
        | LocPack of typ * expr * typ       (* existential package *)
        | LocNamecase of expr * internal_typname * internal_ident * internal_ident * expr * expr * expr  (* unpackaging and name equality *)
        | LocUnpack of internal_typname * internal_ident * expr * expr (* unpackaging *)
        | LocMatch of expr * mtch
        | LocLet of expr * (pat * expr)
             (* this is a sugared-only form; we have that
                  let p = e' in e''
                is roughly represented by
                  Let (e', (p, e'')) *)
             (* remember to freshen_pat_expr *)
        | LocLetMulti of typ * (pat list * expr) * (internal_ident(*rtfif*) * expr)
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetMulti (T, ([p1;..; pn], e'), (x,e'')) *)
             (* remember to freshen_pats_foo *)
        | LocLetrec of typ * (internal_ident(*rtfif*) * (mtch * expr))
             (* this includes a sugared form;
                the core form has mtch = [(PVar(x,t),e)] *)
        | LocLetrecMulti of typ * (pat list * (internal_ident(*rtfif*) * (expr * expr)))
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let rec (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetrecMulti (T, ([p1;..; pn], (x,(e',e'')))) *)
             (* remember to freshen_pats_foo (ps,e') *)
        | LocRaise of expr  (* not fully type annotated, but it's too hard to maintain the type annotation :-( *)
        | LocTry of expr * mtch
        | LocMarshal of expr * expr * typ
        | LocMarshalz of string * expr * typ
        | LocUnmarshal of expr * typ
        | LocRET of typ
        | LocSLOWRET of typ
        | LocCol of expr * eqs * typ       (* coloured brackets *)
        | LocOP of int * op_or_econst * expr list  (* int is n, the total number of args.  I think (-K). *)
        | LocResolve of expr * modname * resolvespec  (* NB: expr is more general than the semantics allows *)
        | LocResolve_blocked of expr * modname * resolvespec  (* ditto *)
        | LocPar of expr * expr  (* e1 | e2 : sugared form *)
        | LocFresh of typ
        | LocCFresh of typ
        | LocHashMvf of hash_or_modname * external_ident * typ
        | LocHashTs of typ * expr * typ
        | LocHashHts of typ * expr * expr * typ
        | LocNameValue of name_value
        | LocSwap of expr * expr * expr
        | LocFreshfor of expr * expr
        | LocSupport of typ * expr
        | LocTie of modname * external_ident
        | LocNameOfTie of expr
        | LocValOfTie of expr


and prim_expr =
          C0 of con0
        | C1 of con1 * prim_expr
        | Cons of prim_expr * prim_expr
        | Tup of prim_expr list
        | Op of op_or_econst * prim_expr list
              (* this is a sugared form: the source form requires the
                 list to be empty, and the core form requries that the
                 list has length n, where n is the arity of the op *)
        | Loc of location
        | Fn  of prim_mtch  (* this is a sugared form; the core form has mtch = [(PVar(x,t),e)] *)
        | Fun of pat list * prim_expr (* this is a sugared form; fun p1 .. pn -> e *)
        | TAbs of internal_typname * prim_expr  (* type abstraction *)
        | InEnv of etysubst * prim_expr
              (* this is a runtime-only form *)
              (* it is a delayed substitution, binding *all* vars in expr *)
              (* See flattenclos for the precise meaning of the term *)
        | Clos of etysubst Lazy.t
              * internal_ident * typ
              * bracket_seq * prim_expr
              * (internal_ident * typ * bracket_seq) option
              (* this is a runtime-only form *)
              (* remember to freshen_clos *)
              (* Clos(env,x2,ty2,bs2,e1,None)              - nonrecursive closure
                 Clos(env,x2,ty2,bs2,e1,Some(x1,ty1,bs1))  - recursive closure
                 env      - substitution environment
                 x2 : ty2 - argument
                 bs2      - brackets to apply around x2 in e1
                 e1       - body
                 x1 : ty1 - name of this closure in env
                 bs1      - brackets to apply around x1 in e1
              *)
              (* Closures carry an environment for their free variables.
                 This is (often) hidden from the printed
                 representation and from the typechecker.
                 See flattenclos for the precise meaning of the term. *)
        | TClos of etysubst
              * internal_typname
              * prim_expr
              (* this is a runtime-only form *)
              (* remember to freshen *)
              (* TClos(menv,tenv,itn,e)   (type closures are never recursive!)
                 (menv,tenv)  - substitution environment
                 itn          - argument (always of kind Type)
                 e            - body
                 NB: bracket-pushing through closure is *weird*
                 because tyvar ends up at different colour. Hmm.
              *)
        | Id of internal_ident
        | Dot of modname * external_ident  (* module projection *)
        | HashDot of hash * external_ident (* hash projection *)
        | If of prim_expr * prim_expr * prim_expr
        | While of prim_expr * prim_expr
        | LazyOp of lazy_op * prim_expr list
        | Seq of prim_expr * prim_expr
        | App of prim_expr * prim_expr
        | TApp of prim_expr * typ               (* type application *)
        | Pack of typ * prim_expr * typ         (* existential package *)
        | Unpack of internal_typname * internal_ident * prim_expr * prim_expr (* unpackaging *)
        | Namecase of prim_expr * internal_typname * internal_ident * internal_ident * prim_expr * prim_expr * prim_expr  (* unpackaging and name equality *)
        | Match of prim_expr * prim_mtch
        | Let of prim_expr * (pat * prim_expr)
             (* this is a sugared-only form; we have that
                  let p = e' in e''
                is roughly represented by
                  Let (e', (p, e'')) *)
             (* remember to freshen_pat_expr *)
        | LetMulti of typ * (pat list * prim_expr) * (internal_ident (*rtfif*) * prim_expr)
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetMulti (T, ([p1;..; pn], e'), (x,e'')) *)
             (* remember to freshen_pats_foo *)
        | Letrec of typ * (internal_ident(*rtfif*) * (prim_mtch * prim_expr))
             (* this includes a sugared form;
                the core form has mtch = [(PVar(x,t),e)] *)
        | LetrecMulti of typ * (pat list * (internal_ident (*rtfif*) * (prim_expr * prim_expr)))
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let rec (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetrecMulti (T, ([p1;..; pn], (x,(e',e'')))) *)
             (* remember to freshen_pats_foo (ps,e') *)
        | Raise of prim_expr  (* not fully type annotated, but it's too hard to maintain the type annotation :-( *)
        | Try of prim_expr * prim_mtch
        | Marshal of prim_expr * prim_expr * typ
        | Marshalz of string * prim_expr * typ
        | Unmarshal of prim_expr * typ
        | RET of typ
        | SLOWRET of typ
        | Col of prim_expr * eqs * typ       (* coloured brackets *)
        | OP of int * op_or_econst * prim_expr list  (* int is n, the total number of args.  I think (-K). *)
        | Resolve of prim_expr * modname * resolvespec  (* NB: expr is more general than the semantics allows *)
        | Resolve_blocked of prim_expr * modname * resolvespec  (* ditto *)
        | Par of prim_expr * prim_expr (* e1 | e2 : sugared form *)
        | Fresh of typ
        | CFresh of typ
        | HashMvf of hash_or_modname * external_ident * typ
        | HashTs of typ * prim_expr * typ
        | HashHts of typ * prim_expr * prim_expr * typ
        | NameValue of name_value
        | Swap of prim_expr * prim_expr * prim_expr
        | Freshfor of prim_expr * prim_expr
        | Support of typ * prim_expr
        | Tie of modname * external_ident
        | NameOfTie of prim_expr
        | ValOfTie of prim_expr

(* and located_prim_mtch = prim_mtch located *)

and expr = expr_desc located

and eval_ctx =  (* ...from the inside *)
    CtxTop
  | CtxC1 of con1 * eval_ctx
  | CtxConsL of eval_ctx * prim_expr
  | CtxConsR of prim_value * eval_ctx
  | CtxTup of prim_expr list * eval_ctx * prim_value list  (* Note 3 *)
  | CtxInEnv of etysubst * eval_ctx  (* Note 2 *)
  | CtxIf of eval_ctx * prim_expr * prim_expr
  | CtxSeq of eval_ctx * prim_expr
  | CtxAppL of eval_ctx * prim_expr
  | CtxAppR of prim_value * eval_ctx
  | CtxTApp of eval_ctx * typ
  | CtxUnpack of internal_typname * internal_ident * eval_ctx * prim_expr
  | CtxNamecaseL of eval_ctx * internal_typname * internal_ident * internal_ident * prim_expr * prim_expr * prim_expr
  | CtxNamecaseR of prim_value * internal_typname * internal_ident * internal_ident * eval_ctx * prim_expr * prim_expr
  | CtxMatch of eval_ctx * prim_mtch
  | CtxRaise of eval_ctx
  | CtxTry of eval_ctx * prim_mtch
  | CtxMarshalL of eval_ctx * prim_expr * typ
  | CtxMarshalR of prim_value * eval_ctx * typ
  | CtxMarshalz of eqs * string * eval_ctx * typ
  | CtxUnmarshal of eval_ctx * typ
  | CtxCol of eval_ctx * eqs * typ       (* coloured brackets *)
  | CtxOp of op_or_econst * prim_value list * eval_ctx * prim_expr list
  | CtxLazyOp of lazy_op * eval_ctx * prim_expr
  | CtxOP of eqs * int * op_or_econst * prim_expr list * eval_ctx * prim_value list  (* Note 3 *)
  | CtxHashTs of eqs * typ * eval_ctx * typ
  | CtxHashHtsL of eqs * typ * eval_ctx * prim_expr * typ
  | CtxHashHtsR of eqs * typ * prim_value * eval_ctx * typ
  | CtxSwapL of eval_ctx * prim_expr * prim_expr
  | CtxSwapM of prim_value * eval_ctx * prim_expr
  | CtxSwapR of prim_value * prim_value * eval_ctx
  | CtxFreshforL of eval_ctx * prim_expr
  | CtxFreshforR of prim_value * eval_ctx
  | CtxSupport of typ * eval_ctx
  | CtxNameOfTie of eval_ctx
  | CtxValOfTie of eval_ctx
  | CtxPack of typ * eval_ctx * typ
(* Module init stuff *)
  | CtxStrVal of ident * eval_ctx * Location.t
  | CtxImodule of eqs * modname * mod_compile_body_ctx * eval_ctx (* Note 4 *)


(* Note 1: in the colour-changing contexts CtxMarshalz, CtxCol, and CtxOP, the eqs records the *outer* colour *)
(* Note 2: in the environment-changing context CtxInEnv, the env records the *outer* environment *)
(* Note 3: elements (e1,e2,e3,__,v5,v6,v7) stored as ([e3;e2;e1],__,[v5;v6;v7]) *)

and language =
  | SugaredSourceInternalForm
  | SourceInternalForm
  | SourceInternalFreshForm
  | CompiledForm

and value = expr  (* just for documentation in the eval_ctx defn *)
and prim_value = prim_expr

and mtch = (pat * expr) list   (* remember to freshen_pat_expr *)
and prim_mtch = (pat * prim_expr) list located   (* remember to freshen_pat_expr *)

and pat =
    PWild of typ
  | PVar of internal_ident * typ
  | PC0 of con0
  | PC1 of con1 * pat
  | PCons of pat * pat
  | PTup of pat list
  | PTyped of pat * typ

and esubst = prim_expr IIdentMap.t
(* simultaneous substitution: all domain idents are distinct and none
   binds in any of the range expressions *)
(* TODO: not sure if this is the semantics currently adopted in the bitrotted eval *)
and tysubst = typ ITypnameMap.t

and etysubst = esubst * tysubst


and op =
        | Ref of typ    (* ref *)
              (* Ref ty has language type: ty -> ty ref *)
        | Deref of typ  (* ! *)
              (* Deref ty has language type: ty ref -> ty *)
        | Assign of typ (* := *)
              (* Assign ty has language type: ty ref -> ty -> unit *)
        | Assign' of typ (* := *)
              (* Assign ty has language type: ty ref -> ty -> unit *)

	| Equal of typ  (* = *)
              (* Equal ty has language type: ty -> ty -> bool *)
        | Less          (* < *)
              (* Less ty has language type: int -> int -> bool *)
        | LessEq        (* <= *)
        | Greater       (* > *)
        | GreaterEq     (* >= *)
        | Mod
        | Land
        | Lor
        | Lxor
        | Lsl
        | Lsr
        | Asr
        | UMinus     (* - *)
        | Plus       (* + *)
        | Minus      (* - *)
        | Times      (* * *)
        | Divide     (* / *)
        | ListConcat of typ (* @ *)
              (* ListConcat ty has language type: ty list -> ty list -> ty list *)
        | StrConcat  (* ^ *)
        | CompareName of typ    (* comparison between names *)
        | NameToString of typ    (* comparison between names *)
        | CreateThread of typ   (* concurrency operators *)
        | Self
        | Kill
        | CreateMutex
        | Lock
        | TryLock
        | Unlock
        | CreateCVar
        | Wait
        | Waiting
        | Signal
        | Broadcast
        | Thunkify
        | Unthunkify
        | Exit of typ

and signature_item_desc= SVal of ident * typ | STyp of typname * kind
and signature_item = signature_item_desc located
and signature = signature_item list located (* remember to freshen_signature *)
and structure_item_desc =
    StrVal of ident * expr   (* more general than the semantics allows *)
  | StrValMulti of ident * pat list * expr
        (* this is a sugared-only form.  pat list is nonempty. *)
        (* we have that
                  let x p1 .. pn = e'
           is roughly represented by
                  StrValMulti (x, [p1;..; pn], e') *)
        (* remember to freshen_pats_expr *)
  | StrTyp of typname * typ
and structure_item = structure_item_desc located
and structure = structure_item list located (* remember to freshen_structure *)


and vliteral =
     | VNumber of int   (* nn *)
     | VHash of hash    (* NN, h *)

and atomic_version =  (* avne *)
     | VLiteral of vliteral
     | VMyname (* compiler will write in name of current module *)

and version =  (* vne *)
     | VAtomic of atomic_version
     | VCons of atomic_version * version

and atomic_hash_version_constraint =  (* ahvce *)
      | AVCHModname of modname
      | AVCHHash of hash

and atomic_version_constraint =  (* avce *)
      | AVCHashVersion of atomic_hash_version_constraint
      | AVCNumber of int

and tail_version_constraint =
        TVCAtomic  of atomic_version_constraint   (* atomic  *)
      | TVCBetween of int * int  (* closed interval *)
      | TVCBefore  of int        (* left-open interval *)
      | TVCAfter   of int        (* right-open interval *)
      | TVCAnything              (* wildcard, matches any version *)

and version_constraint = (* vce *)
      | VCDotted of atomic_version_constraint list * tail_version_constraint
      | VCNameEqual of atomic_hash_version_constraint

and withspec = (modname * external_typname * typ) list
and likespec = LikeNone | LikeMod of modname | LikeStr of structure
and resolvespec_item =
    Resolve_static_link
  | Resolve_here_already
  | Resolve_url of string
and resolvespec = resolvespec_item list

and mode =
       | MHash
       | MFresh
       | MCfresh
       | MBangHash
       | MBangCfresh

and mod_user_body = {
    mu_sign: signature;
  mu_vne: version;
  mu_str: structure;
  mu_withspec: withspec;
}
and mod_compile_body = {
  mc_hash : hash;
  mc_eqs : eqs;
  mc_vubs: valuabilities;
  mc_sign0 : signature;
  mc_sign1 : signature;
  mc_vn : version;
  mc_str : structure;
}

and mod_imod_body = { mi_hash: hash;
                      mi_vubs : valuabilities;
		      mi_eqs: eqs;
                      mi_sign0: signature;
                      mi_sign1: signature;
                      mi_vn: version;
                      mi_str_done: structure_item list;
		      mi_str_todo: structure_item list;
		      mi_str_loc : Location.t
		    }


and mod_compile_body_ctx = { ctx_mc_hash: hash;
                              ctx_mc_vubs : valuabilities;
			     ctx_mc_eqs: eqs;
                             ctx_mc_sign0: signature;
                             ctx_mc_sign1: signature;
                             ctx_mc_vn: version;
			     ctx_mc_str_done: structure_item list;
			     ctx_mc_str_todo: structure_item list;
                             ctx_mc_str_loc: Location.t }


and mod_hash_body = {
  mh_eqs : eqs;
  mh_sign0 : signature;
  mh_vne : version;
  mh_str : structure;
}
and mod_alias_body = { ma_sign : signature; ma_modname : modname; }

and import_user_body = {
  iu_sign : signature;
  iu_vce : version_constraint;
  iu_likespec : likespec;
  iu_resolvespec : resolvespec;
  iu_mo : modname option;
}
and import_compile_body = {
  ic_hash : hash;
  ic_vubs: valuabilities;
  ic_sign0 : signature;
  ic_sign1 : signature;
  ic_vc : version_constraint;
  ic_likestr : structure;
  ic_resolvespec : resolvespec;
  ic_mo : modname option;
}
and import_hash_body = {
  ih_sign0 : signature;
  ih_vce : version_constraint;
  ih_likestr : structure;
}
and source_definition =
    Mod_user of modname * mode * mod_user_body
  | Mod_alias of modname * mod_alias_body
  | Import_user of modname * mode * import_user_body
  | Mark_user of string
and definition =
  | Mod_compile of modname * mod_compile_body
  | Mod_imod of modname * mod_imod_body
  | Import_compile of modname * import_compile_body
  | Mod_fresh of modname * mod_user_body
  | Import_fresh of modname * import_user_body
  | Mark_compile of string
and source_definitions = source_definition list
and definitions = definition list

and 'state compilation_unit_definition =
       | CUDef of ('state -> source_definition * 'state)
       | CUSourceString of string  (* implementation extension; internal use only *)
       | CUIncludeSource of string * string option  (* second argument is file to write object code to *)
       | CUIncludeCompiled of string

and 'state compilation_unit_definitions =  'state compilation_unit_definition list

and 'state compilation_unit = 'state compilation_unit_definitions * ('state -> expr * 'state) option

and compiled_unit = nameenv * definitions * expr option

and prim_compiled_unit = nameenv * definitions * prim_expr option


and program = definitions * prim_expr  (* remember to freshen_compiled_program *)

and store = (location * prim_expr) list  (* no repetitions in the domain *)

(* if (l, ty) is in loctyplist then l has language type ty
   (thus we don't strip the refs) *)
and loctyplist = (location * typ) list (* no repetitions in the domain *)
and namesiglist = (modname * signature) list (* no repetitions in the domain *)

and typeenv_entry =
  Env_val of internal_ident * typ
| Env_loc of location * typ  (* l : T ref is represented as Env_loc(l,T) *)
| Env_mod of modname * signature
| Env_typ of internal_typname * kind

and typeenv_list = typeenv_entry list

and module_info = {  (* for efficiency, we store the following info on a module: *)
    mod_sign : signature;             (* original signature *)
    mod_eidmap : typ EIdentMap.t;     (* _flattened_ type of each value field *)
    mod_etnmap : kind ETypnameMap.t;  (* _flattened_ kind of each type field *)
  }

(* PLEASE TREAT typeenv AS ABSTRACT; it is only concrete because of dependency problems.
   Only the code in Tysupp is allowed direct access. *)
and typeenv = {
    te_list : typeenv_list;   (* stored from right to left, so shadowing identifiers occur before shadowed ones *)
    te_iidmap : typ IIdentMap.t;
    te_locmap : typ LocMap.t;
    te_modmap : module_info ModnameMap.t;
    te_itnmap : kind ITypnameMap.t;
  }

and nameenv_entry =
  | NEnv_nmod of abstract_name * external_modname * mod_hash_body    (* n : nmodule... *)
  | NEnv_nimp of abstract_name * external_modname * import_hash_body (* n : nimport... *)
  | NEnv_type of abstract_name                    (* n : Type *)
  | NEnv_tname of abstract_name * typ             (* n : T name *)

and nameenv_list = nameenv_entry list

(* nameenv *)
and nameenv

(* Valuability *)

and valuability =
  | Valuable
  | CValuable
  | Nonvaluable

and valuabilities = { term_valuability : valuability;
                      type_valuability : valuability;
                    }

val name_of_nameenv_entry : nameenv_entry -> abstract_name

val emptynameenv : nameenv
val lookup_nameenv : abstract_name -> nameenv -> nameenv_entry option
val addnameenv : nameenv_entry -> nameenv -> nameenv

val name_mem_nameenv : abstract_name -> nameenv -> bool

val nameenv_list_of_nameenv : nameenv -> nameenv_list
val nameenv_nameenv_of_list : nameenv_list -> nameenv (* FZ ugly, I know *)
val limitnameenv : nameenv -> abstract_name list -> nameenv

val typeenv_list_of_typeenv : typeenv -> typeenv_list

val names_of_modnames : nameenv -> external_modname list -> abstract_name list

(* often En and E go around together holding hands *)
(* we need to pull this apart and put it together again too often to use a record here *)
type nametypeenv = nameenv * typeenv

(* names *)
module NameValueMap :
    sig
      type key = name_value
      type 'a t
      val empty : 'a t
      val add : key -> 'a -> 'a t -> 'a t
      val find : key -> 'a t -> 'a
      val remove : key -> 'a t -> 'a t
      val mem : key -> 'a t -> bool
      val iter : (key -> 'a -> unit) -> 'a t -> unit
      val map : ('a -> 'b) -> 'a t -> 'b t
      val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
      val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end

val nameValueMap_assocs : 'a NameValueMap.t -> (NameValueMap.key * 'a) list


(* Marshalled data *)
type marshalled_body = { mb_ne : nameenv option;
                         mb_defs : definitions;
                         mb_store : store;
                         mb_storeenv : loctyplist;
                         mb_expr : expr;
                         mb_typ : typ; }

(* the following will be the result of unserialising a message string;
   by contrast, we use Marshalled foo as a subexpression of the
   expression language *)
and marshalled_value = Marshalled_value of marshalled_body




(* Processes *)

and thread_state = TsRunnable
                 | TsSlowcall
		 | TsResolveBlocked
                 | TsMutexBlocked of name_value
                 | TsCVarWaiting of name_value

and threaded_smallstep_inner = {
    tsc_name   : name_value;
    tsc_state  : thread_state;
    tsc_defs   : definitions;
    tsc_ec     : eval_ctx;
    tsc_env    : etysubst;
    tsc_eqs    : eqs;
    tsc_def    : (structure_item list) option;
    tsc_expr   : prim_expr;
    tsc_next_expr : prim_expr option}


and thread_info = {
    ti_name : name_value;
    ti_state: thread_state;
    ti_defs : definitions;
    ti_expr : prim_expr;
  }

and mutex_info = {
    mtxi_name : name_value;
    mtxi_state : locked_mutex_info option;
  }

and locked_mutex_info = {
    lmtxi_owner   : name_value;
    lmtxi_waiting : name_value Util.AQueue.t
  }

and cvar_info = {
    cvi_name : name_value;
    cvi_waiting : name_value Util.AQueue.t;
  }

(* NB: the loctyplist and store should have identical domains, though
not necessarily identically ordered *)
type configuration = { cfg_nenv : nameenv;
                       cfg_defs : definitions;
                       cfg_store : store;
                       cfg_senv : loctyplist;
                       cfg_runnable : name_value Util.AQueue.t;
		       cfg_slowcall : unit NameValueMap.t;
                       cfg_threads : thread_info NameValueMap.t;
                       cfg_mutexes : mutex_info NameValueMap.t;
                       cfg_cvars : cvar_info NameValueMap.t; }



(* a smallstep configuration has:
   - definitions, store environment, store
   - evaluation context
   - current colour (at the hole)
   - expression under evaluation
*)

and smallstep_outer = { scfg_nenv : nameenv;
                       	scfg_defs : definitions;
                        scfg_store : store;
                        scfg_senv : loctyplist;
                        scfg_runnable : name_value Util.AQueue.t;
                        scfg_slowcall : unit NameValueMap.t;
                        scfg_threads : threaded_smallstep_inner NameValueMap.t;
                        scfg_mutexes : mutex_info NameValueMap.t;
                        scfg_cvars : cvar_info NameValueMap.t; }

and collected =
    {
    cdefs : definitions;
    cstore : store;
    csenv : loctyplist;
    ceqs : eqs;
    cenv : etysubst;
    cctx : eval_ctx;
    cexpr : prim_expr
  }


val append_iidmap : 'a IIdentMap.t -> 'a IIdentMap.t -> 'a IIdentMap.t  (* left-priority *)
val iidmap_to_list : 'a IIdentMap.t -> (internal_ident * 'a) list
val append_itnmap : 'a ITypnameMap.t -> 'a ITypnameMap.t -> 'a ITypnameMap.t  (* left-priority *)
val itnmap_to_list : 'a ITypnameMap.t -> (internal_typname * 'a) list

val empty_esubst   :   esubst
val empty_tysubst  :  tysubst
val empty_etysubst : etysubst

val initial_store : store
val initial_store_env : loctyplist


(* operator info *)
val op_info : op -> (typ list * typ)

val isinfix_op : op -> bool
val isinfix_op_or_econst: op_or_econst -> bool

(* arities of primitive ops *)
val arity_of_op : op -> int
val arity_of_econst_ref : (internal_ident -> int) ref  (* filled in by Econst *)
val arity_of_op_or_econst : op_or_econst -> int

(* freshening *)

(* Each will freshen up the top-level binders but not the ones in
subexpressions: for example, after doing freshen_definitions you still
need to freshen_pat_expr for the subexpressions therein. *)

val freshen_iident_foo : internal_ident * 'a -> internal_ident * 'a
val freshen_pat_expr : pat * expr -> pat * expr
val freshen_pat_prim_expr : pat * prim_expr -> pat * prim_expr
val freshen_pats_foo : pat list * 'a -> pat list * 'a
(* bitrot val freshen_clos : esubst * internal_ident * typ * bracket_seq * expr * (internal_ident * typ * bracket_seq) option -> esubst * internal_ident * typ * bracket_seq * expr * (internal_ident * typ * bracket_seq) option *)
val freshen_signature : signature -> signature
val freshen_structure : structure -> structure
val freshen_source_definitions : source_definitions -> source_definitions
val freshen_definitions : definitions -> definitions
val freshen_program : program -> program
(* KW: never used, no longer implementable:  val freshen_compilation_unit : compilation_unit -> compilation_unit *)

(* NB: we don't touch the locations *)
(* val freshen_configuration : configuration -> configuration *)

(* the above function should normally suffice, but here are some more
utilies *)

val binding_vars_of_pat : pat -> internal_ident list
val binding_vars_of_pats : pat list -> internal_ident list
val swap_list : ('a name * 'a name) list -> 'b -> 'b

(* freshen a marshalled body, by giving all top-level definitions
distinct fresh names and locations distinct numbers *)
val freshen_mb : marshalled_body -> marshalled_body


(* constructor/inspector functions *)

val external_typname_of_string : string -> external_typname
val external_typname_to_string : external_typname -> string
val typname_ext : typname -> external_typname
val typname_int : typname -> internal_typname
val fresh_internal_typname : string -> internal_typname

val etn_compare : external_typname -> external_typname -> int

val tn_eq : internal_typname -> internal_typname -> bool
val tn_compare : internal_typname -> internal_typname -> int

val external_ident_of_string : string -> external_ident
val external_ident_to_string : external_ident -> string
val ident_ext : ident -> external_ident
val ident_int : ident -> internal_ident
val fresh_internal_ident : string -> internal_ident

val id_eq : internal_ident -> internal_ident -> bool
val eid_eq : external_ident -> external_ident -> bool
val id_compare : internal_ident -> internal_ident -> int

val external_modname_of_string : string -> external_modname
val external_modname_to_string : external_modname -> string
val modname_ext : modname -> external_modname
val modname_int : modname -> internal_modname
val fresh_internal_modname : unit -> internal_modname
val fresh_modname : external_modname -> modname

val string_of_lithash : Digest.t -> string
val lithash_of_string : string -> Digest.t

val txdot : hash_or_modname -> external_typname -> typ
val xdot : hash_or_modname -> external_ident -> prim_expr

(*
val definition_is_compiled : definition -> bool
val definition_not_compiled : definition -> bool
*)

val modname_of_source_definition : source_definition -> modname (* fails if applied to a Mark_user *)

val modname_of_definition : definition -> modname (* fails if applied to a Mark_compile *)

val sign0_of_definition : definition -> signature (* fails if applied to
                                                     Mark_compile, Mod_fresh, or Import_fresh *)
val sign1_of_definition : definition -> signature (* fails if applied to
                                                     Mark_compile, Mod_fresh, or Import_fresh *)
val sign_of_definition : definition -> signature  (* fails if applied to anything but
                                                     Mod_fresh or Import_fresh *)

val external_modname_of_hash : hash -> external_modname option

val mn_eq : modname -> modname -> bool
val mn_ext_eq : external_modname -> external_modname -> bool
val mn_compare : modname -> modname -> int

val l_eq : location -> location -> bool

val sign_typfieldnames : signature -> typname list
val sign_valfieldnames : signature -> ident list
val lookup_signature_etn : signature -> external_typname -> kind option
val lookup_signature_eid : signature -> external_ident -> typ option

(* for both of these the sig or struct argument should be sufficiently
type flattened that no dangling binders are left over afterwards. *)

val limitdom : signature -> signature
(* val limitdomstr : ats -> structure -> structure *)

val str_typfieldnames : structure -> typname list
val str_valfieldnames : structure -> ident list
val lookup_structure_etn : structure -> external_typname -> typ option
val lookup_structure_eid : structure -> external_ident -> expr option

val lookup_store_loc : store -> location -> prim_expr option

val abstract_name_of_name_value : name_value -> abstract_name
val type_of_name_value : name_value -> typ
val name_value_eq : name_value -> name_value -> bool
val name_value_compare : name_value -> name_value -> int

(* printers (filled in later) *)

val basic_print_expr : (expr -> string) ref
val basic_print_prim_expr : (prim_expr -> string) ref
val basic_print_esubst : (esubst -> string) ref

(* catamorphisms *)

(* tmap_* f find the top-level type nodes and apply f to them; they
don't go inside hashes. *)

val tmap_expr : (typ -> typ) -> expr -> expr
val tmap_kind : (typ -> typ) -> kind -> kind
val tmap_mtch : (typ -> typ) -> mtch -> mtch
val tmap_signature' : (typ -> typ) -> signature_item_desc located list -> signature_item_desc located list
val tmap_signature : (typ -> typ) -> signature -> signature
val tmap_structure' : (typ -> typ) -> structure_item_desc located list -> structure_item_desc located list
val tmap_structure : (typ -> typ) -> structure -> structure

(* emap_* f find all top-level term nodes and apply f them; they
don't go inside hashes. *)

val emap_structure_item : (expr -> expr) -> structure_item -> structure_item
val emap_structure : bool -> (expr -> expr) -> structure -> structure

(* early_tmap_typ ft t tries to apply ft to t; if ft returns Some x
then the result is x; if None, early_tmap_typ traverses through the
smaller type expressions with the same ``quit early if possible''
strategy; it does not go inside hashes. *)

val early_tmap_typ : (typ -> typ option) -> typ -> typ

(* tsub_typ perform a simultaneous internal typname to type expression
substitution on a type; doesn't go inside hashes. *)
(* ASSUMES NO SHADOWING OF TYPE BINDERS!! *)
val tsub_typ : (internal_typname * typ) list -> typ -> typ

(* early_emap_expr fe tries to apply fe to e; if fe returns Some x
then the result is x; if None, early_emap_expr traverses through the
smaller term expressions with the same ``quit early if possible''
strategy.  It does not go inside hashes, Resolve, or
Resolve_blocked.  (To process the inside of Resolve nodes, for
example, make fe explicitly handle Resolve nodes.)  *)

val early_emap_expr : bool -> (eqs option -> expr -> expr option) -> eqs option -> expr -> expr
val early_emap_mtch : bool -> (eqs option -> expr -> expr option) -> eqs option -> mtch -> mtch
val early_emap_pat_expr : bool -> (eqs option -> expr -> expr option) -> eqs option -> (pat * expr) -> (pat * expr)

val early_emap_prim_expr : bool -> (eqs option -> prim_expr -> prim_expr option) -> eqs option -> prim_expr -> prim_expr
val early_emap_prim_mtch : bool -> (eqs option -> prim_expr -> prim_expr option) -> eqs option -> (pat * prim_expr) list -> (pat * prim_expr) list
val early_emap_prim_pat_expr : bool -> (eqs option -> prim_expr -> prim_expr option) -> eqs option -> (pat * prim_expr) -> (pat * prim_expr)
val early_emap_esubst : bool -> (eqs option -> prim_expr -> prim_expr option) -> eqs option -> esubst -> esubst

(* free module variables (with no duplicates in the output) *)
val fmv_typ : typ -> modname list
val fmv_expr : expr -> modname list
val fmv_prim_expr : prim_expr -> modname list
val fmv_str : structure -> modname list
val fmv_mo : modname option -> modname list

(* locations of an expression *)
val prim_free_locs : prim_expr -> location list

(* locations of an expression *)
val free_locs : expr -> location list

val reachable_modules_locs : definitions -> store -> definitions * prim_expr ->
  (modname list * location list)

(* the filter function from the minicaml-rt:
 - the sign may not contain any val fields;
 - the str should be sufficiently type-flattened to avoid dangling
   names
*)
exception Filter_val_fields_in_sign_not_allowed
exception Filter_sign_missing_fields
val filter_str_sign : structure -> signature -> structure


val find_val_field : signature -> external_ident -> signature_item
val find_typ_field : signature -> external_ident -> signature_item

(* helper functions: maintained by K *)

val texists : (typ -> bool) -> typ -> bool

(* context helpers *)

val flatten_context : eval_ctx -> (eval_ctx->eval_ctx) list
val unflatten_context : (eval_ctx->eval_ctx) list -> eval_ctx

(* val print_location : Location.t -> string *) (* defunct *)

val mtchtoprim: Location.t -> mtch -> prim_mtch
val exprtoprim: expr -> prim_expr
val primtoexpr:prim_expr -> expr

val expr_of_loc : Location.t -> prim_expr

val seed : unit -> unit

val new_abstract_name : unit -> abstract_name
val add_name_value : nameenv -> name_value -> nameenv
val new_name_value :  nameenv -> typ -> nameenv * name_value

(* FZ a better place for these? *)
val defs_below_mark : definitions -> definitions
val mode_of_source_def : source_definition -> mode
val sdef_to_fresh_def : source_definition -> definition

val print_hack : prim_expr -> string
val print_hack_all : prim_expr -> string

