// This file is (c) 2016 Peter Sewell
// It is made available under the Creative Commons
// Attribution-NonCommercial 4.0 International Public License 
// (CC BY-NC 4.0) in the file LICENSE

#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <string.h>
#include <inttypes.h>
#include <float.h>
#include <math.h>
#include <unistd.h>
#include <sys/select.h>
#include <signal.h>
#include <fcntl.h>
#include <time.h>
#include <sys/ioctl.h>
#include <termios.h>

#include "colours.h"


#include "arduino-serial-lib.h"

// ***********************************************************************
// interface to astronomy code
#include "pointer_aa_interface.h"


// true to run without device connected
bool dry_run = false;
//bool dry_run = true;


// Setup:
// the pointer should be initialised with:
//   vertical axis (y) (azimuth) rotation such that the pointer is due North   azimuth = azimuth_init clockwise from north
//     with at least half a turn of slack in each direction

#define azimuth_init (0.0)

//   horizontal axis (x) (altitude) rotation with the pointer horizontal       elevation = elevation_init up from horizontal

#define elevation_init (0.0)

// Coordinate systems
// 

// The astronomical topocentric coordinates, indicating a direction
// with respect to a particular point on the Earth's surface, are
// specified as:
//  - an azimuth, measured as an angle clockwise (from above) from North
//  - an elevation, measured as an angle up from horizontal  
//     (this is sometimes known as an altitude)


// Steppers: 
// - a full rotation for a stepper is 'rot' steps:

#define rot      (509*2*2)  // one rotation in steps  (2036) (0.18 deg/step)
#define halfrot  (509*2)    // 0.5 rotation in steps

// - the stepper direction is positive for clockwise motion, looking into the stepper axis
// - the stepper initial position is 'stepper_init' (as set by the Arduino code)

#define stepper_init (1024)

// - for the azimuth (vertical axis) stepper, we use values in [0,rot)
// - for the elevation (horizontal axis) stepper
//    -- for the pointer, we use values in [stepper_init - halfrot, stepper_init]
//    -- for the mirror, we use values in [stepper_init, stepper_init + halfrot]

// fudge factor for backlash: the pointer mass pulls down,
// so the pointing direction for the mirror and the pointer are at
// opposite ends of the slop in the horizontal axis rotation.
// Assuming the pointer is calibrated (initially horizontal), we
// apply this compensating factor to the desired mirror coordinate.

#define backlash  (22)

// for reference:
// - the sun's angular diameter is around 0.5 degrees (approx 2.83 steps)
//    or 0.0087 radians (approx 0.01 radians, so the sun image will have 
//    diameter in cm roughly equal to the mirror-projection distance in m)
// - stepper backlash is (very roughly) 10 sun diameters, maybe 30 steps. 
// - the sun's angular speed is 360/(24*60*60) = 0.0042 degrees/s, or 
//    120 seconds per sun-diameter


// Knobs:
// - reads from [0,knob_range)

#define knob_range 1024

// - azimuth knob should give South (astronomical azimuth 180.0) when centred
// - elevation knob should give horizontal (astronomical elevation 0.0) when centred



// ***********************************************************************
//  coordinate transforms

void azimuth_to_stepper(double a, double *vertical_axis_position_double_p, long *vertical_axis_position_long_p) {
  double vd;
  //printf("aziumuth_to_stepper input: %f  ",a);
  long vl;
  if ((a < 0) || (a >= 360.0)) { printf("warning: azimuth_to_stepper input out of range (%f)\n",a);  }
  vd = a - azimuth_init;            // convert from azimuth wrt North to azimuth wrt pointer init position
  if (vd < 0.0) vd = vd + 360.0;    // normalise to [0, 360) 
  vd = vd / 360.0 * rot;            // convert to stepper units
  vd = vd + stepper_init;           // convert to stepper origin for pointer init position
  if (vd > rot) vd = vd - rot;      // normalise to [0, rot) 
  vl = (long)vd; /* % rot*/;        // convert to integer
  if ((vl < 0) || (vl >= rot)) { printf("warning: azimuth_to_stepper output out of range (%ld)\n",vl);  }
  *vertical_axis_position_double_p = vd;
  *vertical_axis_position_long_p = vl;
}

void elevation_to_stepper_pointer(double e, double *horizontal_axis_position_double_p, long *horizontal_axis_position_long_p) {
  double hd;
  long hl;
  //printf("elevation_to_stepper_pointer input: %f  ",e);
  if ((e < -90.0) || (e >= 90.0)) { printf("warning: elevation_to_stepper_pointer input out of range (%f)\n",e);  }
  hd = e - elevation_init;          // convert from elevation wrt horizontal to elevation wrt pointer init elevation. Now in [-180,0]
  hd = hd / 360.0 * rot;            // convert to stepper units
  hd = hd + stepper_init;           // convert to stepper origin for pointer init position.  
  hl = (long)hd /*% rot*/;          // converto to integer
  //  if ((hl < stepper_init - halfrot) || (hl > stepper_init)) { printf("warning: elevation_to_stepper_pointer output out of range (%ld)\n",hl);  }
  *horizontal_axis_position_double_p = hd;
  *horizontal_axis_position_long_p = hl;
}

void elevation_to_stepper_mirror(double e, double *horizontal_axis_position_double_p, long *horizontal_axis_position_long_p) {
  double hd;
  long hl;
  //printf("elevation_to_stepper_mirror input: %f  ",e);
  if ((e < -90.0) || (e >= 90.0)) { printf("warning: elevation_to_stepper_mirror input out of range (%f)\n",e);  }
  hd = e - elevation_init;          // convert from elevation wrt horizontal to elevation wrt pointer init elevation. Now in [-180,0]
  hd = hd / 360.0 * rot;            // convert to stepper units
  hd = hd + stepper_init;           // convert to stepper origin for pointer init position.  
  hd = hd + halfrot;                // add half rotation for mirror
  hd = hd - backlash;               // adjust for backlash
  hl = ((long)hd) % rot;            // convert to integer
  //  if ((hl < stepper_init - halfrot + halfrot) || (hl > stepper_init + halfrot)) { printf("warning: elevation_to_stepper_mirror output out of range (%ld)\n",hl);  }
  *horizontal_axis_position_double_p = hd;
  *horizontal_axis_position_long_p = hl;
}

double stepper_to_azimuth(long vl) {
  double vd;
  if ((vl < 0) || (vl >= rot)) { printf("warning: stepper_to_azimuth input out of range (%ld)\n",vl);  }

  vd = ((double)vl) - stepper_init; // convert to stepper origin wrt pointer init position
  if (vd < 0) vd = vd + rot;        // normalise to [0,rot)
  vd = vd / rot * 360.0;            // convert to degrees
  vd = vd + azimuth_init;           // convert to azimuth wrt pointer init position
  if (vd > 360.0) vd = vd - 360.0;  // normalise to [0, 360)
  if ((vd < 0) || (vd >= 360.0)) { printf("warning: stepper_to_azimuth output out of range (%f)\n",vd);  }
  return(vd);
}

double stepper_to_elevation_pointer(long hl) {
  double hd;
  //  if ((hl < stepper_init - halfrot) || (hl > stepper_init)) { printf("warning: stepper_to_elevation_pointer input out of range (%ld)\n",hl);  }

  hd = ((double)hl) - stepper_init; // convert to stepper origin wrt pointer init position  (now in [-halfrot,0] )
  hd = hd / rot * 360.0;            // convert to degrees
  hd = hd + elevation_init;         // convert to astronomical elevation, now in [-90,90]
  return(hd);
}

double stepper_to_elevation_mirror(long hl) {
  double hd;
  //  if ((hl < stepper_init - halfrot + halfrot) || (hl > stepper_init + halfrot)) { printf("warning: stepper_to_elevation_mirror input out of range (%ld)\n",hl);  }

  hd = ((double)hl) - stepper_init; // convert to stepper origin wrt pointer init position  (now in [-halfrot,0] )
  hd = hd + backlash;               // adjust for backlash
  hd = hd - halfrot;                // convert to stepper origin wrt mirror init position
  hd = hd / rot * 360.0;            // convert to degrees
  hd = hd + elevation_init;         // convert to astronomical elevation, now in [-90,90]
  return(hd);
}

long knob_azimuth_to_stepper(long ka) {
  double a;
  a = (double)ka / knob_range * 360.0;  
  double vapd;
  long vapl;
  azimuth_to_stepper(a, &vapd, &vapl);
  return(  vapl );   
}

long knob_elevation_to_stepper_pointer(long ke) {
  double e;
  e = ((double)ke / knob_range) * 180.0 - 90.0;
  double hapd;
  long hapl;
  elevation_to_stepper_pointer(e, &hapd, &hapl);
  return( hapl );
  //  return(   stepper_init - halfrot +   (((double)ke) / knob_range * halfrot) );
}


double degrees_to_radians(double d) {
  return( d * 2.0 * 3.1415 / 360.0); 
}

double radians_to_degrees(double r) {
  return( r * 360.0 / (2.0 * 3.1415));
}


// ***********************************************************************
//  control knob values (shared with Arduino code)

#define control_manual    12  
#define control_refl_set  13  
#define control_reflect   14  
#define control_sun        0  
#define control_mercury    1  
#define control_venus      2  
#define control_moon       3  
#define control_mars       4  
#define control_jupiter    5  
#define control_saturn     6  
#define control_uranus     7  
#define control_neptune    8  
#define control_pluto      9  


// ***********************************************************************
// position and knob state 

// current position, as written to arduino, in integer stepper coordinates
long vertical_axis_position = stepper_init;
long horizontal_axis_position = stepper_init;

// previous position, in integer stepper coordinates
long vap_last = stepper_init;
long hap_last = stepper_init;

// current position, as calculated in floating point
double vertical_axis_position_double;
double horizontal_axis_position_double;

// target position for sun-reflect mode, in integer stepper coordinates
long target_vertical_axis_position = stepper_init;  
long target_horizontal_axis_position = stepper_init;


// state of device control knobs
int knob_azimuth;
int knob_elevation;
int knob_control_encoding;
int previous_good_knob_control_encoding;
int knob_valid = false;



// ***********************************************************************
//   sleeping

void my_sleep(int sec, long nsec) {
  struct timespec req, rem;
  req.tv_sec=sec;
  req.tv_nsec=nsec;
  nanosleep(&req,&rem);
}


// ***********************************************************************
//   communication with Arduino

// file descriptor for arduino USB serial connection

int fd; 

// code to catch ^C and exit cleanly, resetting pointer position

void reset_pointer() {
  int n;
  unsigned char b = 'r';
  if (!dry_run) {
    n = write(fd, &b, 1); // write byte 'r' to arduino
    if (n==-1) perror("write");
    if (n==0) printf("write wrote 0\n");
    close(fd);
  }
}

bool volatile controlC = false;

void intHandler(int dummy) {
    controlC = true;
}

void reset_and_exit() {
  signal(SIGINT,SIG_DFL);
  reset_pointer();
  raise(SIGINT);
}

void my_error_and_reset(const char *s) {
  printf("error: %s\n", s);
  reset_pointer();
  exit(1);
}

void perror_exit(const char *s) {
  perror(s);
  exit(1);
}


// serial port for arduino
//   to fix permissions:
//    sudo chmod 666 /dev/ttyACM0  or more permanently
//    add to dialout group:  sudo usermod -a -G dialout USERNAME
//     (and log out and in)

const char *device3 =  "/dev/ttyACM3";
const char *device4 =  "/dev/ttyACM4";
const char *device5 =  "/dev/ttyACM0";
char device[100];

int nread;
int bytesAv;

int baudrate = 115200;  

void init_device() {
  int max_tries = 10;
  int try;
  try = 0;
  int failed;
  failed = false;
  while (try++ < max_tries) {
    printf("trying to connect to device: try %i\n",try);
    fd = serialport_init(device3,baudrate);
    if (fd != -1) { 
      strncpy(device,device3,100);
      break; }
    else {
      fd = serialport_init(device4,baudrate);
      if (fd != -1) {
        strncpy(device,device4,100);
        break;
      }
      else {
        fd = serialport_init(device5,baudrate);
        if (fd != -1) {
          strncpy(device,device5,100);
          break;
        }
        else {
          failed = true;
        }
      }
    }
    my_sleep(0,300000000);
  }
  if (failed) {
    printf("devices \"%s\", \"%s\", \"%s\":\n",device3,device4,device5);
    perror_exit("error opening device");
  }
  serialport_flush(fd);
  printf("opened port %s\n",device);
  //  printf("fd= %i\n",fd);
}


#define read_buffer_max 1000

char read_buffer[read_buffer_max];
int read_buffer_length = 0;
int read_buffer_line_end = false;


// try to read knob values from arduino; return true if there was something 
//  - read repeatedly, to consume all available bytes, and let newer values 
//    supersede old values 
int read_from_arduino() {

  unsigned char b;  
  int n;
  int read_something;
  read_something = 0;
  while (1) {
    if (controlC) reset_and_exit();
    if (!dry_run) 
      n = read(fd, &b, 1);  // read a char 
    else {
      n = 0;
      knob_control_encoding = control_sun;
      knob_valid = true;
    }
    if( n==-1) perror("read");   // couldn't read
    if( n==0 ) 
      break;
    else { 
      read_something = 1;
      //putc(b,stdout);
      if (read_buffer_line_end == true) 
        read_buffer_length = 0;

      if (b=='\n') {
        read_buffer_line_end = true;
        read_buffer[ read_buffer_length ] = 0;
        int i;
        n = fflush(stdout);
        if (n!=0) perror("fflush stdout (from arduino)");
        int tmp_azimuth, tmp_elevation, tmp_control_encoding;
        n = sscanf(read_buffer, "az=%i el=%i ce=%i",&tmp_azimuth, &tmp_elevation, &tmp_control_encoding);
        if (n==3) {
          knob_azimuth = tmp_azimuth;
          knob_elevation = tmp_elevation;
          knob_control_encoding = tmp_control_encoding;
          //printf("read: az=%i el=%i ce=%i\n",knob_azimuth, knob_elevation, knob_control_encoding);
          knob_valid = true;
        } else {
          // for (i=0; i<read_buffer_length; i++)
          //          putc(read_buffer[i],stdout); 
          // putc(b,stdout);
          // printf("read fail: only %i\n",n);
        }
      } else {
        read_buffer_line_end = false;
        read_buffer[ read_buffer_length ] = b;
        read_buffer_length = (read_buffer_length + 1) % read_buffer_max;
      }

    }
  }
  return(read_something);
}


// write position to arduino
void write_position_to_arduino(bool ispointer) {
  int v3,v2,v1,v0,h3,h2,h1,h0;
  long p;

  p=vertical_axis_position;
  v3=p / 1000;
  p=p-1000*v3;
  v2= p / 100;
  p=p-100*v2;
  v1= p / 10;
  v0=p-10*v1;
  p=horizontal_axis_position;
  h3=p / 1000;
  p=p-1000*h3;
  h2= p / 100;
  p=p-100*h2;
  h1= p / 10;
  h0=p-10*h1;

  //printf("vertical axis position %li  %i%i%i%i  %f  ",vertical_axis_position,v3,v2,v1,v0,vertical_axis_position_double-(double)vertical_axis_position);
  //printf("horizontal axis position %li  %i%i%i%i  %f\n",horizontal_axis_position,h3,h2,h1,h0,horizontal_axis_position_double-(double)horizontal_axis_position);

  printf("azimuth %7.2f  vert.axis %li  offset %7.2f  ",stepper_to_azimuth(vertical_axis_position), vertical_axis_position,vertical_axis_position_double-(double)vertical_axis_position);
  if (ispointer)
    printf("elevation (pointer) %7.2f  hor.axis %li  offset %7.2f\n",stepper_to_elevation_pointer(horizontal_axis_position),horizontal_axis_position,horizontal_axis_position_double-(double)horizontal_axis_position);
  else
    printf("elevation (mirror) %7.2f  hor.axis %li  offset %7.2f\n",stepper_to_elevation_mirror(horizontal_axis_position),horizontal_axis_position,horizontal_axis_position_double-(double)horizontal_axis_position);

  char outbuf[11];
  sprintf(outbuf,"%i%i%i%i%i%i%i%i ",h3,h2,h1,h0,v3,v2,v1,v0);
  if (!dry_run) {
    int n;
    n = write(fd, outbuf, 9); // write 9 bytes to arduino
    if (n==-1) perror("write_position_to_arduino");
    if (n!=9) printf("write_position_to_arduino wrote %i bytes (not 9)\n",n);
  }
}



// ***********************************************************************
// terminal IO

void set_terminal_no_echo() {
  struct termios buf;
  tcgetattr(0, &buf);
  buf.c_lflag &= ~(ECHO | ICANON);
  buf.c_cc[VMIN]=1;
  buf.c_cc[VTIME]=0;
  tcsetattr(0, TCSAFLUSH, &buf);
  int savedflags=fcntl(0, F_GETFL, 0);
  fcntl(0, F_SETFL, savedflags | O_NONBLOCK );
  return;
}

int stdinInputAvailable() {
  struct timeval tv;
  fd_set fds;
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  FD_ZERO(&fds);
  FD_SET(STDIN_FILENO, &fds);
  select(STDIN_FILENO+1, &fds, NULL, NULL, &tv);
  return (FD_ISSET(0, &fds));
}

int get_digit() {
  int i;
  while (true) {
    if (controlC) reset_and_exit();
    while (!stdinInputAvailable()) {};
    i=getc(stdin);
    putc(i,stdout); // echo byte to stdout
    fflush(stdout);
    if (i>='0' && i<= '9') return(i - '0');
  }
}

void my_beep() {
  // use sox to make some kind of a beep
  char *system_buffer[200];
  snprintf((char * __restrict__ )system_buffer,200,
           "play -n output.wav synth 0.05 sine 600 vol 0.4 2> /dev/null&");
  int i;
  i=system((char * __restrict__ )system_buffer);
}




// ***********************************************************************
// vectors 

double vector_modulus(double *va) {
  return(sqrt(va[0]*va[0] + va[1]*va[1] + va[2]*va[2]));
}

void print_vec(char *s, double *v) {
  printf("%s  N=%f  E=%f  H=%f\n",s,v[0],v[1],v[2]);
}

void vector_normalise(double *v, double *va) {
  double m = vector_modulus(va);
  v[0]=va[0]/m;
  v[1]=va[1]/m;
  v[2]=va[2]/m;
}

// ***********************************************************************
// sun reflect code

// calculate sun position and set mirror (vertical_axis_position, horizontal_axis_position)
//  to reflect image in the direction of (target_vertical_axis_position, target_horizontal_axis_position).
//  beep if the stepper position changed

void get_sun_reflect_position() {

  // set (azimuth_reference, elevation_reference) to sun position

  set_date_time();
  objnum=0;
  set_elements_planet();
  print_object_name();
  set_azalt();
  printf("JD %.2f  sun azimuth = %f   altitude = %f\n",JD, azimuth_reference, elevation_reference);

  // convert to radians

  double azimuth_rad, elevation_rad;
  azimuth_rad = degrees_to_radians(azimuth_reference);
  elevation_rad = degrees_to_radians(elevation_reference);

  // work in vectors (n,e,h) of the north, east, and height  (nb this is not a right-handed coordinate system)

  double neh_sun[3];
  neh_sun[0] = cos(azimuth_rad) * cos(elevation_rad);
  neh_sun[1] = sin(azimuth_rad) * cos(elevation_rad);
  neh_sun[2] = sin(elevation_rad);

  // intended reflection
  double taz,tel;

  // azimuth angle from North clockwise (from above), in radians
  taz = degrees_to_radians(stepper_to_azimuth(target_vertical_axis_position));

  // elevation angle, in radians
  tel = degrees_to_radians(stepper_to_elevation_pointer(target_horizontal_axis_position));

  printf("target azimuth = %f (%ld)  elevation = %f (%ld)\n", 
         radians_to_degrees(taz), target_vertical_axis_position,
         radians_to_degrees(tel), target_horizontal_axis_position);

  // convert target direction to vector
  double neh_refl[3];
  neh_refl[0] = cos(taz) * cos(tel);
  neh_refl[1] = sin(taz) * cos(tel);
  neh_refl[2] = sin(tel);

  // compute mirror vector by bisecting sun and target vectors
  double neh_mirror[3];
  int i;
  for (i=0; i<3; i++) 
    neh_mirror[i] = /* neh_sun[i]; */ (neh_sun[i] + neh_refl[i]) / 2.0;
  vector_normalise(neh_mirror,neh_mirror);

  // convert mirror vector to azimuth/elevation (in radians)
  double mirror_azimuth_rad, mirror_elevation_rad;
  mirror_azimuth_rad = atan2(neh_mirror[1],neh_mirror[0]);
  if (mirror_azimuth_rad < 0) mirror_azimuth_rad += 2*3.1415;
  mirror_elevation_rad = asin(neh_mirror[2]);
  //atan2(y,x) = arctan(y/x) 
  //http://en.cppreference.com/w/c/numeric/math/atan2

  print_vec("sun vector    ",neh_sun);
  print_vec("refl vector   ",neh_refl);
  print_vec("mirror vector ",neh_mirror);
  printf("mirror az = %f   mirror_el = %f\n",radians_to_degrees(mirror_azimuth_rad), radians_to_degrees(mirror_elevation_rad));

  vap_last = vertical_axis_position;
  hap_last = horizontal_axis_position;

  azimuth_to_stepper(radians_to_degrees(mirror_azimuth_rad), &vertical_axis_position_double, &vertical_axis_position);
  elevation_to_stepper_mirror(radians_to_degrees(mirror_elevation_rad), &horizontal_axis_position_double, &horizontal_axis_position);

  int changed;
  changed = (vertical_axis_position != vap_last) || (horizontal_axis_position !=hap_last);
  if (changed && true) my_beep();

}


// ***********************************************************************

void interact() {
  unsigned char b;  
  int n;
  //printf("type [p] for planet, [s] for star, [o] for orbit, [c] for continuous tracking of last object (planet only), [C] for fast tracking, [r] to reset pointer, [i] to reinitialise pointer, [m] for manual control, [a] for auto-reflect\n");
 
  while (1) {
    if (controlC) reset_and_exit();
    if (read_from_arduino()) {
    } else {
      my_sleep(0,10*1000000); // sleep 10ms
    }

    if (!knob_valid)
      continue;

    switch (knob_control_encoding) {

    case control_manual:
    case control_refl_set:
      
      vertical_axis_position = knob_azimuth_to_stepper(knob_azimuth);
      horizontal_axis_position = knob_elevation_to_stepper_pointer(knob_elevation);

      write_position_to_arduino(true);
      my_sleep(0,90*1000000); // sleep 90ms

      previous_good_knob_control_encoding = knob_control_encoding;
      break;

    case control_sun     :
    case control_mercury :
    case control_venus   :
    case control_moon    :
    case control_mars    :
    case control_jupiter :
    case control_saturn  :
    case control_uranus  :
    case control_neptune :
    case control_pluto   :

      previous_good_knob_control_encoding = knob_control_encoding;

      objnum = knob_control_encoding;
      set_elements_planet();
      print_object_name();
      set_date_time();
      set_azalt();

      //printf("fresh az alt = %f %f\n",azimuth_reference, elevation_reference);

      azimuth_to_stepper(azimuth_reference, &vertical_axis_position_double, &vertical_axis_position);
      elevation_to_stepper_pointer(elevation_reference, &horizontal_axis_position_double, &horizontal_axis_position);

      //printf("v/h axis_position_double = %f %f\n",vertical_axis_position_double, horizontal_axis_position_double);
      //printf("v/h axis_position = %ld %ld\n",vertical_axis_position, horizontal_axis_position);
      
      int changed;
      changed = (vertical_axis_position != vap_last) || (horizontal_axis_position !=hap_last);
      if (changed && false) my_beep();
      
      write_position_to_arduino(true);
      my_sleep(0,90*1000000); // sleep 90ms

      break;

    case control_reflect:

//      if (previous_good_knob_control_encoding != control_reflect) {
//        previous_knob_azimuth = knob_azimuth;
//        previous_knob_elevation = knob_elevation;
//      }
//      previous_good_knob_control_encoding = knob_control_encoding;

      target_vertical_axis_position = knob_azimuth_to_stepper(knob_azimuth);
      target_horizontal_axis_position = knob_elevation_to_stepper_pointer(knob_elevation);

      get_sun_reflect_position();
      write_position_to_arduino(false);
      my_sleep(0,90*1000000); // sleep 90ms
      break;


    default: 
      break;

    }

    continue;

  }
}

// // ********************************************************************
// // ********************************************************************
// // ************   OLD CODE    *****************************************
// // ********************************************************************
// // ********************************************************************
// 
//     // try to read a byte from stdin and write to arduino
//     int i;
//     if (stdinInputAvailable() || (knob_valid && knob_control_encoding != 127)) {
//       i=getc(stdin);
//       if (i!=EOF) { 
//         putc(i,stdout); // echo byte to stdout
//         n = fflush(stdout);
//         if (n!=0) perror("fflush stdout (from console)");
//         b=(unsigned char)i; 
//         if (b == 'p') return(PLANET);  // go off to astro code
//         if (b == 's') return(STAR);  // go off to astro code
//         if (b == 'o') return(ORBIT);  // go off to astro code
//         if (b == 'c') return(CONTINUOUS);  // go off to astro code
// 
//         /* ******************************************************************* */
//         if (b == 'a') {  // start autoreflect
// 
//           printf("autoreflect, ^C to break and reset\n");
//           while (1) {
//             if (controlC) reset_and_exit();
//             if (read_from_arduino()) {
//             } else {
//               my_sleep(0,10000000);
//             }
// 
//             get_sun_reflect_position();
//             write_position_to_arduino(true);
//             my_sleep(0,500000000);
// 
//             // try to read a byte from stdin
//             int i;
//             if (stdinInputAvailable()) {
//               i=getc(stdin);
//               if (i!=EOF) { 
//                 putc(i,stdout); // echo byte to stdout
//                 n = fflush(stdout);
//                 if (n!=0) perror("fflush stdout (from console)");
//                 b=(unsigned char)i; 
//                 int hd,vd;
//                 hd=0;
//                 vd=0;
//                 switch(b) {
//                 case 'u': hd=-100; break;
//                 case 'i': hd=-10; break;
//                 case 'o': hd=-1; break;
//                 case 'p': hd=1; break;
//                 case '[': hd=10; break;
//                 case ']': hd=100; break;
//                 case 'q': vd=-100; break;
//                 case 'w': vd=-10; break;
//                 case 'e': vd=-1; break;
//                 case 'r': vd=1; break;
//                 case 't': vd=10; break;
//                 case 'y': vd=100; break;
//                 }
//                 if (hd==0 && vd==0) continue;
//                 
// 
//                 target_vertical_axis_position = (target_vertical_axis_position + vd);
//                 if (target_vertical_axis_position < 0) target_vertical_axis_position += rot;
//                 target_vertical_axis_position = target_vertical_axis_position % rot;
// 
//                 target_horizontal_axis_position = (target_horizontal_axis_position + hd) ;
//                 if (target_horizontal_axis_position < 0) target_horizontal_axis_position += rot;
//                 target_horizontal_axis_position = target_horizontal_axis_position % rot;
// 
//                 printf("\ntarget vertical axis position %li    ",target_vertical_axis_position);
//                 printf("target horizontal axis position %li\n",target_horizontal_axis_position);
//               }
//             }
//           }
//         }
//         /* ******************************************************************* */
//         if (b == 'm') {  // start manual control
//           printf("manual control: qwertyu for vertical axis; uiop[] for horiztontal  axis, ^C to break and reset\n");
//           while (1) {
//             if (controlC) reset_and_exit();
//             if (read_from_arduino()) {
//             } else {
//               my_sleep(0,10000000);
//             }
// 
// 
//             // try to read a byte from stdin
//             int i;
//             if (stdinInputAvailable()) {
//               i=getc(stdin);
//               if (i!=EOF) { 
//                 putc(i,stdout); // echo byte to stdout
//                 n = fflush(stdout);
//                 if (n!=0) perror("fflush stdout (from console)");
//                 b=(unsigned char)i; 
//                 int hd,vd;
//                 hd=0;
//                 vd=0;
//                 switch(b) {
//                 case 'u': hd=-100; break;
//                 case 'i': hd=-10; break;
//                 case 'o': hd=-1; break;
//                 case 'p': hd=1; break;
//                 case '[': hd=10; break;
//                 case ']': hd=100; break;
//                 case 'q': vd=-100; break;
//                 case 'w': vd=-10; break;
//                 case 'e': vd=-1; break;
//                 case 'r': vd=1; break;
//                 case 't': vd=10; break;
//                 case 'y': vd=100; break;
//                 }
//                 if (hd==0 && vd==0) continue;
//                 
//                 vap_last = vertical_axis_position;
//                 hap_last = horizontal_axis_position;
// 
//                 vertical_axis_position = (vertical_axis_position + vd);
//                 if (vertical_axis_position < 0) vertical_axis_position += rot;
//                 vertical_axis_position = vertical_axis_position % rot;
//                 horizontal_axis_position = (horizontal_axis_position + hd) ;
//                 if (horizontal_axis_position < 0) horizontal_axis_position += rot;
//                 horizontal_axis_position = horizontal_axis_position % rot;
//                 printf("\nvertical axis position %li    ",vertical_axis_position);
//                 printf("horiztontal axis position %li\n",horizontal_axis_position);
//                 write_position_to_arduino(true);
//               }
//             }
//           }
//         }
//         n = write(fd, &b, 1); // write byte to arduino
//         if (n==-1) perror("write");
//         if (n==0) printf("write wrote 0\n");
//       }
//     }
//   }
// }







/* Main program starts here.
 */
int main() {

  vertical_axis_position_double = stepper_to_azimuth(vertical_axis_position);
  horizontal_axis_position_double = stepper_to_elevation_pointer(horizontal_axis_position);

  // set signal handler to record ^C for clean exit
  signal(SIGINT, intHandler);

  // initialise communication with Arduino
  if (!dry_run) init_device();

  // set terminal to not echo
  set_terminal_no_echo();

  // initialise astronomy code
  initialise_aa();

  printf("\nSetup:\n- pointer should be horizontal pointing due North, with at least half a turn of slack in the ribbon cable in each direction\n\n");

  interact();

  return(0); // unreachable
}

//  outer_loop:
//   if (controlC) reset_and_exit();
//  
//   //planet_char = interact();
//   if (interact_mode!=CONTINUOUS ) {
//     interact_mode = interact();
//     printf("\n");
//   }
//   else if (interact_mode==CONTINUOUS) {
//     sleep(10);
//   } else {
//     // cannot happen
//   }
//   
// 
//  loop:
//   if (controlC) reset_and_exit();
// 
// 
// 
// 
//  loop1:
// 
//   //getnum( "Planet number 0-9 or 88 to read star, 99 to read orbit, -1 to return to direct interaction",
//   //	&objnum, intfmt );
// 
//   if (interact_mode==PLANET) {
//     printf("0: Sun, 1:Mercury, 2:Venus, 3:Moon, 4:Mars, 5:Jupiter, 6:Saturn, 7:Uranus, 8:Neptune, 9:Pluto\n");
//     printf("Planet number 0-9: ");
//     fflush(stdout);
//     objnum = get_digit();
//     printf("\n");
//   } else if (interact_mode==STAR) {
//     objnum=88;
//   } else if (interact_mode==ORBIT) {
//     objnum=99;
//   } else // CONTINUOUS 
//     {}
// 
//   
//   if (objnum >=0 && objnum <=10) 
//     set_elements_planet();
//   else if (objnum == 88) 
//     input_and_set_elements_star();
//   else if (objnum == 99)
//     input_and_set_elements_orbit();
//   else
//     goto outer_loop;
// 
//   print_object_name();
// 
//   set_date_time();
// 
//   set_azalt();
// 
// 
//   // this code is for normal (non-auto-reflect) pointing mode
// 
//   vap_last = vertical_axis_position;
//   hap_last = horizontal_axis_position;
// 
//   // for testing
//   //   azimuth_reference = 30.0;
//   //   elevation_reference = 00.0;
// 
//   printf("fresh az alt = %f %f\n",azimuth_reference, elevation_reference);
//   vertical_axis_position_double =  ((azimuth_reference  - 90.0 + 360.0)/ 360.0 * rot) - 1024 + rot ;
//   horizontal_axis_position_double = ((elevation_reference - 90.0 + 360.0) / 360.0 * rot) - 1024 + rot;
// 
//   printf("v/h axis_position_double = %f %f\n",vertical_axis_position_double, horizontal_axis_position_double);
// 
// 
//   vertical_axis_position = (long)vertical_axis_position_double % rot ;
//   horizontal_axis_position = (long)horizontal_axis_position_double % rot;
// 
//   printf("v/h axis_position = %ld %ld\n",vertical_axis_position, horizontal_axis_position);
// 
//   if (vertical_axis_position <0 || vertical_axis_position >=rot) { printf("vertical axis position out of range (%li)\n",vertical_axis_position); goto loop; }
//   if (horizontal_axis_position <0 || horizontal_axis_position >=rot) { printf("horizontal axis position out of range (%li)\n",horizontal_axis_position); goto loop; }
// 
//   int changed;
//   changed = (vertical_axis_position != vap_last) || (horizontal_axis_position !=hap_last);
//   if (interact_mode==CONTINUOUS && changed && false) my_beep();
// 
// 
//   write_position_to_arduino(false);
// 
//   read_from_arduino();
// 
// goto outer_loop;



