/** \file
 *	\brief EMV.h Header file
 *
 *  Contains definitions of functions used to implement the EMV standard
 *  including low-level ISO716-3 based as well as application-level protocol
 *
 *  Copyright (C) 2010 Omar Choudary (osc22@cam.ac.uk)
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */

//------------------------------------------------------------------------
// EMV data structures

/**
 * Structure defining an EMV header command 
 */
typedef struct {
    uint8_t cla;
	uint8_t ins;
	uint8_t p1;
	uint8_t p2;
	uint8_t p3;          
} EMVCommandHeader;


/**
 * Structure defining a command APDU
 */
typedef struct {
	EMVCommandHeader *cmdHeader;
	uint8_t *cmdData;
	uint8_t lenData;
} CAPDU;

/**
 * Structure defining the status bytes
 */
typedef struct {
	uint8_t sw1;
	uint8_t sw2;
} EMVStatus;


/**
 * Structure defining the response APDU
 */
typedef struct {
	EMVStatus *repStatus;
	uint8_t *repData;
	uint8_t lenData;
} RAPDU;

/**
 * Structure defining a command-response pair
 */
typedef struct {
	CAPDU *cmd;
	RAPDU *response;	
} CRP;

/**
 * Structure defining a BER-TLV object
 */
typedef struct {
	uint8_t tag1;
	uint8_t tag2;
	uint8_t len;
	uint8_t *value;
} TLV;

/**
 * Structure defining a record (constructed BER-TLV object)
 */
typedef struct {
	uint8_t count;
	TLV **objects;
} RECORD;


//------------------------------------------------------------------------
// T=0 protocol functions


/// This function will return a command header structure
EMVCommandHeader* MakeCommandHeader(uint8_t cla, uint8_t ins, uint8_t p1, 
						uint8_t p2, uint8_t p3);

/// Initiates the communication with both the ICC and terminal for T=0
uint8_t InitSCDTransaction(uint8_t t_inverse, uint8_t t_TC1, 
	uint8_t *inverse_convention, uint8_t *proto, uint8_t *TC1, 
	uint8_t *TA3, uint8_t *TB3);

/// Returns the command case from the command header
uint8_t GetCommandCase(uint8_t cla, uint8_t ins);

/// Receive a command header from the terminal using protocol T=0
EMVCommandHeader* ReceiveT0CmdHeader(uint8_t inverse_convention, uint8_t TC1);

/// Receive a command data from the terminal using protocol T=0
uint8_t* ReceiveT0CmdData(uint8_t inverse_convention, uint8_t TC1,
	uint8_t len);

/// Receive a command (including data) from the terminal using protocol T=0
CAPDU* ReceiveT0Command(uint8_t inverse_convention, uint8_t TC1);

/// Send a command header to the ICC using protocol T=0
uint8_t SendT0CmdHeader(uint8_t inverse_convention, uint8_t TC1,
	EMVCommandHeader *cmdHeader);

/// Send a command data to the ICC using protocol T=0
uint8_t SendT0CmdData(uint8_t inverse_convention, uint8_t TC1,
	uint8_t *cmdData, uint8_t len);

/// Send a command (including data) to the ICC using protocol T=0
uint8_t SendT0Command(uint8_t inverse_convention, uint8_t TC1, CAPDU *cmd);

/// Forwards a command from the terminal to the ICC for T=0
CAPDU* ForwardCommand(uint8_t tInverse, uint8_t cInverse, 
	uint8_t tTC1, uint8_t cTC1);

/// Serialize a CAPDU structure
uint8_t* SerializeCommand(CAPDU *cmd, uint8_t *len);

/// Receive response from ICC for T=0
RAPDU* ReceiveT0Response(uint8_t inverse_convention, 
	EMVCommandHeader *cmdHeader);

/// Send a response to the terminal
uint8_t SendT0Response(uint8_t inverse_convention, 
	EMVCommandHeader *cmdHeader, RAPDU *response);

/// Forwards a response from the ICC to the terminal
RAPDU* ForwardResponse(uint8_t tInverse, uint8_t cInverse,
	EMVCommandHeader *cmdHeader);

/// Serialize a RAPDU structure
uint8_t* SerializeResponse(RAPDU *response, uint8_t *len);

/// Makes a command-response exchange between terminal and ICC
CRP* ExchangeData(uint8_t tInverse, uint8_t cInverse, 
	uint8_t tTC1, uint8_t cTC1);

/// Makes a complete command-response exchange between terminal and ICC
CRP* ExchangeCompleteData(uint8_t tInverse, uint8_t cInverse, 
	uint8_t tTC1, uint8_t cTC1);

/// Eliberates the memory used by a CAPDU
void freeCAPDU(CAPDU* cmd);

/// Eliberates the memory used by a RAPDU
void freeRAPDU(RAPDU* response);

/// Eliberates the memory used by a CRP
void freeCRP(CRP* data);

//------------------------------------------------------------------------
// Application functions

/// Parse a TLV object from a data stream
TLV* ParseTLV(uint8_t *data, uint8_t lenData, uint8_t includeValue);

/// Parse a record as response to a read record command
RECORD* ParseRecord(RAPDU* response);

/// Eliberates the memory used by a RECORD
void freeRECORD(RECORD *data);

/// Get the position of the Authorized Amount value inside CDOL1 if exists
uint8_t AmountPositionInCDOLRecord(RECORD *record);
