open Camlp4.PreCast (* I ought to be parametric in the location type ... *)

(* The syntax of types *) 

type tp = InT of Camlp4.PreCast.Ast.Loc.t * tp'
and tp' =
    Arrow of tp * tp
  | One
  | Prod of tp * tp
  | Sum of tp * tp 
  | Discrete
  | Next of tp
  | Stream of tp
  | Gui of tp

(* Equality of types needs to ignore source locations. *)

let rec tp_equal (InT(_, tp1)) (InT(_, tp2)) =
  match tp1, tp2 with
  | One, One
  | Discrete, Discrete
      -> true
  | Stream tp1', Stream tp2'
  | Next tp1', Next tp2'
  | Gui tp1', Gui tp2'
      -> tp_equal tp1' tp2'                   
  | Prod (tp1', tp1''), Prod (tp2', tp2'')
  | Arrow (tp1', tp1''), Arrow (tp2', tp2'')
  | Sum(tp1', tp1''), Sum(tp2', tp2'') ->
	tp_equal tp1' tp2' && tp_equal tp1'' tp2''
  | _ -> false
    
(* The syntax of expressions *)

module Var =
struct
  type t = string
  let compare = compare
end

type var = Var.t


type pat =
  | PVar of var
  | PUnit
  | PPair of pat * pat 

type exp = In of Camlp4.PreCast.Ast.Loc.t * exp'
and exp' =
  | Var of var
  | Let of pat * exp * exp
  | App of exp * exp
  | Lam of pat * exp
  | Unit
  | Pair of exp * exp
  | Fst of exp
  | Snd of exp
  | Inl of exp
  | Inr of exp
  | Case of exp * var * exp * var * exp
  | NextE of exp
  | Await of exp
  | Head of exp
  | Tail of exp
  | Cons of exp * exp
  | Fix of var * exp
  | Annot of exp * tp
  | Map of exp * exp
  | Zip of exp * exp 
  | Unfold of exp 
  | Embed of Camlp4.PreCast.Ast.expr
  | Inject of Camlp4.PreCast.Ast.expr
  | LetGui of pat * exp * exp
  | GuiReturn of exp

let loc_exp (In(loc, _)) = loc
let loc_tp (InT(loc, _)) = loc
			       
let rec mk_fun loc vs body =
  match vs with
  | [] -> assert false
  | [x] -> In(loc, Lam(x, body))
  | x :: xs -> In(loc, Lam(x, mk_fun loc xs body))

module Vars = Set.Make(Var)

let rec pvars = function
  | PVar x -> Vars.singleton x
  | PPair(p, p') -> Vars.union (pvars p) (pvars p')
  | PUnit -> Vars.empty 

let rec freevars (In(loc, e)) =
  match e with 
  | Var x -> Vars.singleton x
  | Let(p, e1, e2) 
  | LetGui(p, e1, e2) -> Vars.union (freevars e1) (Vars.diff (freevars e2) (pvars p))
  | Lam(p, e) -> Vars.diff (freevars e) (pvars p)
  | Fix(x, e) -> Vars.remove x (freevars e)
  | Pair(e1, e2)
  | App(e1, e2)
  | Cons(e1, e2)
  | Map(e1, e2)
  | Zip(e1, e2) -> Vars.union (freevars e1) (freevars e2)
  | Unit
  | Embed _
  | Inject _ -> Vars.empty
  | Fst e
  | Snd e
  | NextE e
  | Await e
  | Head e
  | Tail e
  | Annot(e, _)
  | Unfold e
  | GuiReturn e -> freevars e
  | Inl _
  | Inr _
  | Case(_, _, _, _, _) -> assert false
