(* Term formers for changing DSL syntax into ML syntax, with one for each combinator *)

type ast = string -> Camlp4.PreCast.Ast.loc -> Camlp4.PreCast.Ast.expr

val op0 : string -> ast
val op1 : string -> ast -> ast
val op2 : string -> ast -> ast -> ast

val id : ast
val compose : ast -> ast -> ast

val into : ast -> ast                        (* Iso(A,B) -> Hom(A,B) *)
val from : ast -> ast                        (* Iso(A,B) -> Hom(B,A) *)

(* Treatment of delay operators *)

val delay     : int -> Term.tp -> ast        (* delay n A    : Hom(A, o^n(A))                     *)
val map_next  : int -> ast -> ast            (* map_next i   : Hom(A, B) -> Hom(o^i(A), o^i(B))   *)

val next_one  : int -> ast                   (* next_one i   : Iso(o^i(1), 1)                     *)
val next_prod : int -> ast                   (* next_pair i  : Iso(o^i(A x B), o^i(A) x o^i(B))   *)
val next_exp  : int -> ast                   (* next_exp i   : Iso(o^i(A => B), o^i(A) => o^i(B)) *)
    
(* Other combinators *)

(* Tuples *)

val one : ast                                (* one  : Hom(A, 1) *)
val pair : ast -> ast -> ast                 (* pair : Hom(A, B) -> Hom(A,C) -> Hom(A, B x C) *)
val pi1 : ast                                (* pi1  : Hom(A x B, A) *)
val pi2 : ast                                (* pi2  : Hom(A x B, B) *)
val map_prod : ast -> ast -> ast             (* pair : Hom(A, B) -> Hom(C,D) -> Hom(A x C, B x D) *)
val map_prod_iso : ast -> ast -> ast         (* pair : Iso(A, B) -> Iso(C,D) -> Iso(A x C, B x D) *)


(* Sums *)
val inl : ast                                (* inl : Hom(A, A + B) *)
val inr : ast                                (* inr : Hom(B, A + B) *)
val case : ast -> ast -> ast                 (* case : Hom(A, C) -> Hom(B, C) -> Hom(A+B, C) *)
val distrib : ast                            (* distrib : Iso(A x (B+C), AxB + AxC) *)

(* Streams *)
val cons : ast                               (* cons : Hom(A x o(S(A)), S(A)) *)
val head : ast                               (* head : Hom(S(A), A) *)
val tail : ast                               (* tail : Hom(S(A), o(S(A))) *)
val constant : Term.tp -> ast                (* constant A : Hom(A, S(A)) *)
val unfold : ast                             (* unfold : Hom(A => B * oA, A => S(B)) *)
val stream_strength : ast 
val zip : ast 

(* Functions *)
val curry : Term.tp -> ast -> ast            (* curry A : Hom(A x B, C) -> Hom(A, B => C) *)
val eval : ast                               (* eval    : Hom(A => B x A, B) *)

(* GUIs *)

val return : ast
val bind : ast -> ast
val strength : ast

(* Fixed points *)
val fix_stream : Term.tp -> ast -> ast       (* fix_stream : Hom(A x o(S(B)), S(B)) -> Hom(A, S(B)) *)
val fix_fun    : ast -> ast                  (* fix_fun : Hom(A x o(B => C), B => C) -> Hom(A, B => C) *)
val fix_guistream : Term.tp -> ast -> ast

val fix_next : Term.tp -> int -> (ast -> ast) -> ast -> ast 

(* Discrete types *)

val embed : Camlp4.PreCast.Ast.expr -> ast 
val exp_discrete : ast 
val pair_discrete : ast

(* Patterns *)

val pattern : Term.pat -> int -> ast



