//!  Downcast a JavaScript wrapper generated by `wasm-bindgen` back to its original
//! struct.
//!
//! # Examples
//!
//! ```rust
//! use wasm_bindgen::{prelude::wasm_bindgen, JsValue};
//! use wasm_bindgen_downcast::DowncastJS;
//!
//! #[wasm_bindgen]
//! #[derive(DowncastJS)]
//! pub struct Person {
//!     name: String,
//! }
//!
//! /// Try to greet something. If it is the provided value wraps a [`Person`]
//! /// struct, we'll try to greet them by name.
//! #[wasm_bindgen]
//! pub fn greet(maybe_person: &JsValue) -> Option<String> {
//!     let p = Person::downcast_js_ref(maybe_person)?;
//!     Some(format!("Hello, {}!", p.name))
//! }
//! ```

pub use wasm_bindgen_downcast_macros::DowncastJS;

use js_sys::{Function, Reflect};
use wasm_bindgen::{
    convert::{FromWasmAbi, RefFromWasmAbi},
    JsCast, JsValue, UnwrapThrowExt,
};

/// Cast a Rust wrapper class that was generated by `#[wasm_bindgen]` back to
/// the underlying Rust type.
///
/// This is a workaround for
/// [rustwasm/wasm-bindgen#2231](https://github.com/rustwasm/wasm-bindgen/issues/2231)
/// that is robust with respect to minification (something other solutions
/// [can't handle](https://github.com/rustwasm/wasm-bindgen/issues/2231#issuecomment-1167895291)).
///
/// # Safety
///
/// This assumes the wrapper class has a static `__wbgd_downcast_symbol`
/// method which returns the same [`DowncastToken`] as
/// [`DowncastJS::token()`]. Each trait implementation.
///
/// Users should use the custom derive rather than implement this manually.
pub unsafe trait DowncastJS:
    Sized + FromWasmAbi<Abi = u32> + RefFromWasmAbi<Abi = u32>
{
    fn token() -> &'static DowncastToken;

    fn downcast_js(value: JsValue) -> Result<Self, JsValue> {
        // Safety: By using an unsafe trait, we're the responsibility to
        // satisfy downcast_to_ptr()'s invariants is moved to the implementor.
        // Assuming they are satisfied, it should be fine to convert back
        // from the ABI representation.
        unsafe {
            match downcast_to_ptr::<Self>(&value) {
                Some(ptr) => Ok(Self::from_abi(ptr)),
                None => Err(value),
            }
        }
    }

    fn downcast_js_ref(value: &JsValue) -> Option<Self::Anchor> {
        // Safety: By using an unsafe trait, we're the responsibility to
        // satisfy downcast_to_ptr()'s invariants is moved to the implementor.
        // Assuming they are satisfied, it should be fine to convert back
        // from the ABI representation.
        unsafe { downcast_to_ptr::<Self>(value).map(|ptr| Self::ref_from_abi(ptr)) }
    }
}

// This whole mechanism works because the JavaScript wrapper
// class has a static `__wbgd_downcast_symbol()` method which returns the
// same unique Symbol we get in the [`DowncastJS::symbol()`] method.
//
// If we can read that symbol and it matches the symbol we get from
// [`DowncastJS::symbol()`], we know for sure that the pointer is valid and safe
// to cast back to our type.
unsafe fn downcast_to_ptr<T: DowncastJS>(value: &JsValue) -> Option<u32> {
    if !value.is_object() {
        return None;
    }

    let class = Reflect::get_prototype_of(value).ok()?.constructor();
    let key = JsValue::from_str("__wbgd_downcast_token");

    let downcast_symbol_func: Function = Reflect::get(&class, &key)
        .and_then(|v: JsValue| v.dyn_into())
        .ok()?;

    let actual_symbol = downcast_symbol_func.call0(&class).ok()?;

    if *T::token() != actual_symbol {
        return None;
    }

    // Note: this assumes the wrapper class generated by #[wasm_bindgen] will
    // always store the pointer in a field called "ptr".
    let key = JsValue::from_str("ptr");
    let ptr = Reflect::get(value, &key).ok().and_then(|v| v.as_f64())?;

    Some(ptr as u32)
}

/// A token used when downcasting wrapper classes back to their Rust types.
#[derive(Clone)]
pub struct DowncastToken(pub js_sys::Symbol);

impl DowncastToken {
    /// Create a new, unique token.
    pub fn unique() -> Self {
        #[wasm_bindgen::prelude::wasm_bindgen]
        extern "C" {
            #[wasm_bindgen(js_name = "Symbol")]
            static SYMBOL: js_sys::Function;
        }

        let js_symbol = SYMBOL
            .call0(&JsValue::NULL)
            .expect_throw("Unable to call the Symbol() function")
            .dyn_into()
            .expect_throw("Unable to convert the return value of Symbol() into a symbol");

        DowncastToken(js_symbol)
    }
}

impl PartialEq<wasm_bindgen::JsValue> for DowncastToken {
    fn eq(&self, other: &wasm_bindgen::JsValue) -> bool {
        *self.0 == *other
    }
}

// Safety: This should be okay because
// a) JavaScript is single-threaded, and
// b) The Symbol type is only ever stored in an once_cell::sync::Lazy
//    without being mutated.
unsafe impl Sync for DowncastToken {}
unsafe impl Send for DowncastToken {}

#[doc(hidden)]
pub mod internal {
    pub use crate::DowncastToken;
    pub use js_sys::Symbol;
    pub use once_cell::sync::Lazy;
    pub use wasm_bindgen::prelude::wasm_bindgen;
}
