use core::convert::TryFrom;

#[macro_use]
extern crate lazy_static;

use reddsa::*;

#[test]
fn verify_librustzcash_spendauth() {
    for (msg, sig, pk_bytes) in LIBRUSTZCASH_SPENDAUTH_SIGS.iter() {
        assert!(VerificationKey::try_from(*pk_bytes)
            .and_then(|pk| pk.verify(&msg, &sig))
            .is_ok());
    }
}

#[test]
fn verify_librustzcash_binding() {
    for (msg, sig, pk_bytes) in LIBRUSTZCASH_BINDING_SIGS.iter() {
        assert!(VerificationKey::try_from(*pk_bytes)
            .and_then(|pk| pk.verify(&msg, &sig))
            .is_ok());
    }
}

lazy_static! {
    static ref LIBRUSTZCASH_SPENDAUTH_SIGS: [(
        Vec<u8>,
        Signature<sapling::SpendAuth>,
        VerificationKeyBytes<sapling::SpendAuth>
    ); 32] = [
        (
            [
                16, 28, 190, 75, 156, 66, 96, 79, 4, 199, 3, 195, 150, 247, 136, 198, 203, 45, 109,
                125, 88, 244, 84, 48, 177, 46, 178, 237, 214, 64, 7, 108
            ]
            .to_vec(),
            [
                184, 72, 147, 0, 184, 42, 112, 23, 132, 161, 146, 154, 96, 228, 121, 19, 57, 57,
                59, 142, 209, 233, 151, 29, 9, 55, 142, 153, 71, 124, 203, 76, 203, 183, 165, 39,
                20, 49, 230, 24, 162, 232, 156, 176, 115, 184, 191, 122, 27, 103, 243, 15, 226, 72,
                177, 219, 67, 176, 95, 27, 108, 40, 69, 10
            ]
            .into(),
            [
                181, 241, 137, 93, 46, 42, 77, 236, 42, 240, 254, 156, 102, 146, 236, 44, 166, 93,
                85, 228, 118, 90, 109, 138, 25, 70, 46, 202, 61, 255, 216, 4
            ]
            .into(),
        ),
        (
            [
                108, 56, 40, 255, 104, 11, 219, 166, 98, 154, 153, 67, 75, 194, 79, 17, 86, 115,
                26, 175, 150, 173, 228, 209, 66, 119, 33, 94, 87, 187, 19, 49
            ]
            .to_vec(),
            [
                79, 51, 180, 117, 215, 124, 136, 130, 125, 50, 132, 219, 196, 16, 28, 85, 68, 163,
                54, 62, 86, 36, 115, 180, 28, 35, 105, 224, 1, 31, 165, 186, 26, 50, 252, 46, 139,
                219, 234, 182, 96, 126, 97, 94, 221, 61, 19, 90, 143, 166, 200, 7, 185, 134, 183,
                2, 81, 255, 168, 130, 34, 25, 142, 12
            ]
            .into(),
            [
                32, 138, 141, 49, 232, 43, 241, 22, 66, 21, 45, 171, 245, 191, 115, 78, 218, 71,
                232, 16, 49, 246, 189, 199, 239, 171, 119, 186, 146, 179, 252, 155
            ]
            .into(),
        ),
        (
            [
                145, 63, 229, 40, 221, 46, 129, 128, 68, 42, 148, 149, 87, 96, 109, 94, 188, 197,
                117, 95, 73, 255, 11, 214, 198, 41, 249, 226, 22, 130, 163, 215
            ]
            .to_vec(),
            [
                72, 6, 48, 149, 135, 161, 137, 34, 233, 249, 121, 150, 228, 0, 165, 120, 106, 27,
                21, 249, 158, 109, 128, 193, 73, 236, 14, 86, 21, 248, 160, 44, 27, 223, 183, 139,
                19, 121, 238, 236, 194, 40, 243, 249, 223, 15, 191, 65, 182, 7, 23, 61, 76, 70, 37,
                11, 45, 98, 208, 218, 81, 97, 67, 12
            ]
            .into(),
            [
                209, 140, 211, 141, 81, 2, 124, 211, 60, 118, 106, 253, 141, 253, 192, 30, 2, 216,
                130, 239, 243, 55, 48, 65, 33, 4, 110, 71, 247, 172, 6, 65
            ]
            .into(),
        ),
        (
            [
                208, 243, 205, 249, 49, 107, 35, 33, 206, 165, 99, 248, 160, 186, 39, 50, 71, 156,
                226, 164, 9, 17, 162, 158, 198, 154, 10, 123, 189, 221, 247, 116
            ]
            .to_vec(),
            [
                96, 38, 240, 221, 163, 157, 103, 24, 101, 88, 180, 51, 66, 149, 94, 187, 20, 236,
                117, 125, 118, 174, 22, 98, 249, 46, 76, 86, 183, 87, 138, 29, 38, 103, 100, 29,
                156, 175, 232, 144, 186, 10, 98, 237, 220, 98, 116, 83, 230, 58, 38, 91, 241, 162,
                38, 216, 38, 248, 51, 35, 116, 229, 245, 1
            ]
            .into(),
            [
                19, 244, 2, 26, 154, 118, 210, 35, 37, 167, 110, 77, 38, 171, 92, 151, 150, 251,
                63, 160, 138, 203, 225, 171, 44, 47, 186, 127, 26, 94, 145, 69
            ]
            .into(),
        ),
        (
            [
                73, 53, 27, 188, 30, 74, 75, 136, 216, 22, 115, 179, 41, 12, 101, 135, 241, 208,
                177, 226, 3, 109, 120, 48, 2, 120, 150, 27, 29, 59, 61, 180
            ]
            .to_vec(),
            [
                203, 244, 129, 94, 156, 32, 138, 223, 81, 124, 198, 181, 173, 91, 243, 132, 223,
                81, 53, 71, 49, 159, 197, 177, 192, 2, 36, 14, 165, 50, 88, 158, 211, 201, 110,
                119, 247, 80, 67, 71, 53, 170, 20, 67, 1, 67, 47, 73, 19, 253, 251, 175, 121, 94,
                162, 58, 126, 42, 13, 85, 33, 134, 4, 2
            ]
            .into(),
            [
                47, 222, 182, 103, 31, 179, 90, 219, 250, 195, 128, 213, 41, 186, 62, 232, 156, 68,
                205, 52, 43, 71, 73, 212, 6, 108, 217, 73, 156, 166, 145, 102
            ]
            .into(),
        ),
        (
            [
                158, 55, 154, 163, 138, 43, 56, 169, 209, 184, 225, 86, 39, 131, 0, 194, 62, 122,
                84, 176, 197, 115, 88, 216, 47, 127, 36, 131, 215, 205, 251, 69
            ]
            .to_vec(),
            [
                253, 159, 166, 4, 58, 37, 220, 26, 159, 123, 69, 114, 202, 2, 113, 136, 103, 251,
                181, 27, 143, 135, 117, 89, 251, 177, 67, 86, 26, 39, 64, 194, 194, 16, 19, 242,
                189, 157, 128, 9, 17, 6, 148, 194, 43, 164, 238, 156, 15, 135, 62, 99, 247, 58,
                246, 104, 162, 24, 205, 152, 193, 214, 118, 5
            ]
            .into(),
            [
                17, 132, 1, 251, 36, 233, 78, 198, 122, 35, 231, 247, 175, 15, 214, 37, 37, 18,
                247, 20, 16, 165, 66, 133, 243, 75, 69, 78, 5, 101, 42, 174
            ]
            .into(),
        ),
        (
            [
                217, 101, 76, 11, 40, 202, 129, 190, 187, 84, 92, 68, 181, 190, 66, 89, 233, 122,
                72, 49, 98, 38, 40, 129, 112, 67, 119, 39, 61, 230, 140, 107
            ]
            .to_vec(),
            [
                94, 211, 95, 97, 114, 31, 97, 88, 94, 137, 189, 186, 206, 228, 157, 10, 148, 94,
                114, 109, 178, 182, 138, 199, 223, 55, 230, 174, 76, 199, 187, 28, 250, 221, 85,
                96, 100, 217, 24, 129, 219, 27, 10, 97, 195, 112, 148, 210, 62, 188, 61, 93, 90,
                185, 107, 58, 177, 123, 81, 220, 219, 199, 119, 0
            ]
            .into(),
            [
                244, 173, 81, 29, 254, 24, 62, 207, 198, 165, 40, 87, 251, 222, 99, 74, 26, 190,
                240, 181, 176, 132, 16, 109, 149, 143, 130, 140, 15, 151, 79, 108
            ]
            .into(),
        ),
        (
            [
                10, 212, 76, 139, 47, 234, 73, 233, 131, 223, 100, 101, 161, 23, 26, 161, 12, 107,
                27, 76, 50, 232, 165, 234, 34, 170, 126, 169, 179, 7, 168, 216
            ]
            .to_vec(),
            [
                139, 199, 128, 202, 224, 132, 196, 210, 244, 72, 85, 142, 184, 248, 176, 86, 173,
                70, 205, 224, 161, 234, 51, 253, 239, 182, 81, 171, 192, 201, 180, 134, 212, 132,
                101, 145, 250, 244, 18, 252, 14, 150, 26, 140, 197, 241, 209, 226, 217, 98, 82,
                135, 131, 167, 91, 146, 190, 125, 147, 152, 175, 221, 232, 3
            ]
            .into(),
            [
                68, 179, 107, 23, 199, 215, 237, 215, 12, 78, 92, 244, 133, 177, 115, 46, 167, 114,
                174, 10, 70, 38, 232, 102, 4, 27, 229, 238, 5, 128, 86, 155
            ]
            .into(),
        ),
        (
            [
                7, 121, 157, 250, 188, 252, 108, 217, 56, 87, 183, 45, 98, 129, 233, 210, 121, 143,
                173, 195, 172, 2, 154, 125, 176, 148, 3, 234, 3, 250, 127, 116
            ]
            .to_vec(),
            [
                200, 10, 84, 163, 98, 111, 226, 140, 225, 250, 208, 165, 104, 197, 0, 68, 118, 236,
                77, 160, 142, 38, 137, 192, 73, 143, 48, 192, 131, 2, 212, 217, 190, 212, 114, 106,
                232, 178, 156, 18, 25, 116, 101, 150, 208, 248, 222, 183, 159, 19, 181, 83, 211,
                153, 122, 105, 92, 118, 10, 208, 109, 14, 14, 10
            ]
            .into(),
            [
                111, 10, 31, 231, 161, 58, 10, 220, 2, 11, 155, 212, 150, 68, 103, 109, 227, 82,
                36, 22, 102, 4, 3, 56, 64, 4, 34, 171, 198, 46, 102, 161
            ]
            .into(),
        ),
        (
            [
                161, 98, 15, 93, 184, 31, 193, 208, 84, 234, 38, 229, 208, 207, 27, 172, 152, 170,
                39, 138, 187, 88, 195, 82, 253, 108, 101, 128, 110, 86, 114, 242
            ]
            .to_vec(),
            [
                18, 41, 243, 39, 192, 12, 24, 229, 74, 197, 174, 162, 214, 152, 159, 92, 220, 52,
                175, 178, 170, 145, 240, 21, 161, 10, 133, 49, 216, 192, 232, 28, 119, 245, 163,
                58, 63, 207, 253, 42, 134, 12, 204, 25, 169, 223, 66, 152, 132, 178, 190, 255, 1,
                193, 120, 237, 17, 164, 1, 172, 146, 11, 176, 2
            ]
            .into(),
            [
                29, 23, 116, 183, 78, 100, 121, 77, 61, 83, 68, 115, 247, 127, 102, 90, 216, 110,
                86, 250, 43, 110, 250, 34, 97, 132, 99, 182, 146, 45, 84, 75
            ]
            .into(),
        ),
        (
            [
                123, 7, 11, 119, 243, 237, 229, 160, 242, 163, 173, 216, 117, 94, 8, 29, 112, 178,
                23, 50, 180, 75, 18, 189, 215, 221, 21, 8, 86, 209, 177, 87
            ]
            .to_vec(),
            [
                86, 236, 148, 81, 183, 70, 209, 165, 2, 184, 111, 25, 206, 21, 195, 112, 16, 99,
                185, 161, 2, 19, 27, 53, 80, 133, 39, 39, 227, 103, 94, 140, 0, 10, 28, 243, 255,
                131, 163, 170, 206, 229, 116, 99, 153, 52, 99, 205, 177, 133, 247, 63, 43, 78, 154,
                105, 161, 55, 68, 177, 211, 17, 132, 10
            ]
            .into(),
            [
                29, 230, 209, 28, 238, 102, 138, 220, 223, 130, 169, 79, 197, 133, 90, 183, 214,
                204, 190, 100, 0, 89, 221, 164, 60, 131, 235, 21, 68, 150, 204, 174
            ]
            .into(),
        ),
        (
            [
                181, 94, 198, 199, 78, 194, 64, 118, 146, 51, 79, 70, 73, 206, 87, 51, 96, 13, 145,
                198, 78, 245, 179, 182, 169, 147, 95, 137, 77, 33, 247, 4
            ]
            .to_vec(),
            [
                236, 92, 56, 252, 142, 62, 64, 84, 192, 160, 2, 119, 164, 201, 38, 248, 102, 22,
                225, 65, 13, 211, 236, 96, 223, 56, 108, 205, 49, 55, 177, 22, 11, 66, 165, 120,
                114, 57, 144, 81, 170, 50, 119, 41, 138, 151, 109, 248, 211, 141, 208, 110, 241,
                74, 89, 83, 44, 229, 150, 19, 22, 20, 87, 14
            ]
            .into(),
            [
                148, 203, 97, 110, 231, 9, 144, 249, 147, 244, 215, 94, 152, 157, 76, 192, 180, 57,
                176, 250, 88, 62, 162, 242, 207, 194, 43, 31, 224, 131, 21, 60
            ]
            .into(),
        ),
        (
            [
                108, 144, 146, 213, 254, 69, 48, 80, 172, 3, 26, 161, 90, 132, 50, 0, 192, 101, 98,
                65, 167, 199, 175, 147, 29, 177, 217, 226, 72, 233, 93, 118
            ]
            .to_vec(),
            [
                56, 134, 173, 220, 187, 159, 248, 241, 192, 65, 91, 20, 145, 141, 192, 118, 101,
                83, 180, 225, 148, 168, 3, 234, 209, 34, 73, 8, 8, 234, 82, 191, 135, 191, 151, 1,
                205, 158, 224, 35, 144, 77, 131, 209, 80, 26, 118, 189, 39, 76, 246, 20, 37, 26,
                16, 70, 36, 69, 253, 26, 251, 230, 155, 10
            ]
            .into(),
            [
                173, 141, 143, 17, 147, 238, 245, 194, 249, 178, 20, 48, 35, 192, 16, 157, 150, 22,
                130, 105, 149, 14, 242, 169, 63, 116, 193, 121, 41, 196, 140, 50
            ]
            .into(),
        ),
        (
            [
                106, 232, 15, 114, 137, 49, 56, 62, 28, 42, 176, 201, 156, 86, 109, 234, 3, 191,
                44, 231, 22, 110, 103, 24, 204, 71, 207, 80, 255, 34, 150, 208
            ]
            .to_vec(),
            [
                111, 66, 64, 152, 248, 159, 2, 254, 64, 180, 110, 136, 157, 80, 123, 138, 169, 106,
                86, 180, 105, 131, 109, 52, 252, 70, 48, 86, 230, 35, 181, 88, 158, 173, 73, 67,
                200, 96, 243, 192, 24, 34, 242, 54, 113, 199, 214, 118, 39, 244, 142, 254, 143,
                188, 217, 138, 190, 137, 201, 249, 154, 116, 129, 10
            ]
            .into(),
            [
                43, 53, 235, 27, 103, 60, 163, 115, 78, 225, 5, 178, 113, 55, 93, 90, 210, 191, 0,
                249, 175, 80, 72, 140, 22, 173, 81, 237, 10, 176, 48, 20
            ]
            .into(),
        ),
        (
            [
                115, 113, 54, 112, 20, 175, 145, 55, 156, 102, 8, 55, 94, 188, 73, 151, 45, 250,
                55, 84, 176, 22, 100, 238, 102, 177, 44, 154, 49, 69, 93, 64
            ]
            .to_vec(),
            [
                50, 60, 156, 134, 29, 212, 219, 84, 244, 133, 219, 209, 173, 253, 95, 255, 33, 237,
                112, 125, 118, 237, 53, 93, 136, 130, 244, 81, 208, 90, 39, 202, 152, 111, 35, 218,
                255, 167, 173, 5, 160, 80, 97, 104, 109, 174, 255, 108, 171, 192, 91, 209, 108,
                211, 145, 157, 160, 72, 3, 137, 176, 59, 241, 13
            ]
            .into(),
            [
                59, 231, 146, 62, 217, 119, 251, 42, 75, 155, 214, 126, 56, 81, 69, 171, 74, 233,
                46, 25, 1, 115, 192, 232, 135, 133, 182, 16, 6, 255, 185, 143
            ]
            .into(),
        ),
        (
            [
                93, 203, 169, 70, 124, 231, 163, 178, 40, 136, 241, 181, 37, 231, 93, 208, 80, 45,
                166, 216, 63, 169, 147, 117, 109, 223, 26, 76, 4, 125, 65, 115
            ]
            .to_vec(),
            [
                50, 143, 129, 178, 231, 153, 77, 215, 240, 11, 70, 238, 116, 184, 15, 53, 241, 64,
                218, 248, 113, 64, 112, 68, 44, 253, 43, 176, 149, 190, 198, 31, 159, 23, 55, 150,
                205, 228, 255, 194, 176, 93, 150, 100, 254, 82, 5, 105, 47, 204, 171, 137, 210,
                184, 133, 234, 41, 242, 21, 221, 226, 3, 96, 5
            ]
            .into(),
            [
                223, 237, 26, 32, 148, 201, 169, 138, 2, 140, 193, 114, 251, 9, 165, 58, 162, 221,
                21, 86, 227, 81, 70, 110, 73, 26, 60, 174, 201, 202, 155, 8
            ]
            .into(),
        ),
        (
            [
                10, 18, 144, 141, 46, 99, 152, 10, 220, 179, 10, 31, 65, 109, 230, 15, 245, 46,
                235, 2, 149, 188, 163, 203, 110, 98, 125, 129, 101, 155, 205, 189
            ]
            .to_vec(),
            [
                197, 92, 160, 39, 41, 130, 79, 201, 86, 5, 94, 19, 212, 221, 176, 36, 85, 10, 10,
                233, 57, 235, 195, 79, 236, 245, 170, 80, 10, 165, 163, 162, 9, 166, 219, 231, 132,
                233, 78, 252, 87, 160, 104, 37, 36, 66, 252, 8, 39, 130, 227, 91, 215, 255, 167,
                111, 182, 21, 143, 220, 88, 233, 72, 10
            ]
            .into(),
            [
                102, 59, 228, 88, 48, 218, 240, 208, 210, 166, 88, 74, 248, 76, 3, 159, 248, 34,
                163, 166, 251, 51, 150, 127, 247, 151, 82, 124, 167, 104, 89, 186
            ]
            .into(),
        ),
        (
            [
                188, 253, 211, 252, 199, 54, 126, 222, 14, 31, 80, 124, 161, 10, 199, 39, 248, 45,
                154, 68, 162, 227, 125, 107, 172, 41, 41, 151, 234, 171, 164, 136
            ]
            .to_vec(),
            [
                105, 147, 47, 0, 211, 122, 10, 13, 159, 118, 59, 227, 149, 223, 237, 2, 225, 167,
                66, 17, 49, 217, 15, 139, 210, 153, 215, 249, 168, 178, 156, 61, 128, 100, 72, 84,
                108, 84, 245, 17, 123, 78, 64, 220, 253, 63, 114, 157, 176, 1, 3, 87, 36, 121, 246,
                244, 88, 153, 84, 173, 143, 170, 27, 2
            ]
            .into(),
            [
                75, 29, 132, 106, 68, 195, 233, 158, 128, 93, 96, 70, 99, 43, 43, 19, 126, 162,
                176, 32, 72, 79, 194, 8, 59, 203, 16, 219, 85, 226, 17, 198
            ]
            .into(),
        ),
        (
            [
                226, 227, 113, 72, 108, 239, 130, 37, 21, 121, 69, 158, 159, 30, 187, 20, 25, 129,
                18, 211, 152, 237, 217, 225, 103, 173, 225, 84, 130, 14, 98, 152
            ]
            .to_vec(),
            [
                237, 88, 120, 116, 191, 17, 131, 145, 14, 29, 52, 144, 131, 236, 139, 72, 220, 147,
                29, 218, 137, 162, 242, 146, 103, 115, 140, 233, 68, 111, 36, 156, 179, 18, 92,
                188, 154, 211, 159, 152, 76, 56, 48, 11, 53, 67, 249, 47, 236, 72, 54, 90, 145, 13,
                112, 129, 201, 234, 39, 40, 94, 110, 181, 2
            ]
            .into(),
            [
                71, 123, 7, 83, 166, 112, 96, 41, 72, 21, 124, 204, 220, 242, 114, 18, 136, 114,
                188, 3, 109, 89, 207, 142, 211, 220, 75, 170, 138, 159, 91, 65
            ]
            .into(),
        ),
        (
            [
                225, 242, 213, 131, 75, 206, 121, 1, 11, 169, 169, 241, 41, 113, 255, 138, 109,
                216, 145, 166, 98, 81, 85, 105, 178, 76, 8, 213, 3, 25, 15, 164
            ]
            .to_vec(),
            [
                218, 39, 132, 40, 46, 40, 16, 196, 214, 110, 211, 229, 200, 208, 146, 67, 169, 158,
                66, 110, 184, 166, 117, 120, 16, 27, 105, 176, 36, 74, 187, 103, 245, 186, 59, 233,
                171, 172, 120, 25, 182, 219, 84, 55, 219, 242, 39, 138, 89, 74, 1, 193, 184, 194,
                188, 190, 113, 136, 30, 158, 182, 91, 215, 0
            ]
            .into(),
            [
                111, 220, 167, 18, 14, 97, 147, 165, 160, 155, 21, 255, 127, 176, 119, 237, 210,
                112, 99, 193, 66, 166, 167, 180, 108, 198, 153, 227, 94, 85, 232, 13
            ]
            .into(),
        ),
        (
            [
                225, 47, 87, 102, 125, 86, 130, 79, 147, 184, 26, 138, 230, 229, 254, 180, 191,
                252, 9, 107, 29, 126, 101, 54, 98, 134, 228, 105, 109, 143, 118, 42
            ]
            .to_vec(),
            [
                197, 43, 231, 143, 82, 220, 24, 243, 80, 117, 1, 39, 150, 133, 77, 40, 26, 178, 95,
                158, 79, 60, 194, 143, 98, 128, 244, 255, 65, 191, 239, 179, 92, 159, 105, 207,
                138, 161, 38, 24, 201, 232, 230, 204, 200, 157, 93, 195, 178, 71, 88, 242, 173, 77,
                0, 56, 16, 165, 227, 188, 40, 49, 155, 7
            ]
            .into(),
            [
                251, 172, 204, 143, 30, 134, 108, 243, 119, 104, 161, 93, 43, 50, 86, 5, 110, 151,
                249, 48, 103, 221, 89, 79, 235, 116, 245, 103, 94, 170, 172, 227
            ]
            .into(),
        ),
        (
            [
                208, 177, 124, 14, 226, 70, 2, 251, 231, 125, 90, 16, 123, 73, 47, 104, 159, 135,
                54, 131, 77, 244, 234, 106, 178, 9, 156, 169, 107, 233, 234, 187
            ]
            .to_vec(),
            [
                1, 233, 198, 32, 153, 142, 54, 180, 79, 77, 194, 197, 49, 31, 59, 191, 245, 228,
                202, 195, 97, 67, 182, 115, 151, 80, 207, 183, 198, 87, 179, 129, 8, 92, 28, 164,
                209, 203, 46, 58, 160, 26, 157, 8, 149, 163, 214, 113, 127, 37, 212, 125, 179, 177,
                95, 244, 189, 21, 246, 72, 183, 184, 185, 9
            ]
            .into(),
            [
                246, 252, 81, 48, 0, 123, 41, 78, 72, 116, 61, 73, 248, 229, 4, 234, 152, 149, 48,
                198, 137, 14, 231, 240, 242, 205, 159, 99, 9, 183, 104, 31
            ]
            .into(),
        ),
        (
            [
                71, 61, 57, 222, 190, 1, 120, 139, 238, 48, 237, 151, 52, 201, 242, 21, 200, 29,
                54, 2, 193, 153, 8, 73, 78, 186, 156, 183, 224, 126, 164, 52
            ]
            .to_vec(),
            [
                209, 4, 57, 213, 70, 129, 222, 208, 129, 130, 35, 133, 204, 89, 47, 250, 107, 176,
                210, 179, 51, 173, 240, 112, 146, 112, 138, 1, 244, 87, 116, 144, 250, 178, 27,
                125, 61, 125, 186, 206, 249, 173, 60, 142, 188, 89, 96, 102, 246, 43, 184, 96, 157,
                190, 240, 123, 169, 191, 249, 8, 183, 64, 170, 12
            ]
            .into(),
            [
                243, 221, 190, 199, 165, 66, 152, 23, 26, 31, 188, 218, 196, 35, 177, 38, 169, 55,
                212, 132, 148, 172, 124, 129, 232, 238, 162, 154, 224, 80, 154, 109
            ]
            .into(),
        ),
        (
            [
                88, 104, 132, 39, 159, 181, 16, 226, 167, 230, 84, 135, 90, 15, 145, 23, 88, 99,
                83, 19, 224, 144, 136, 116, 8, 33, 201, 84, 20, 31, 189, 203
            ]
            .to_vec(),
            [
                198, 23, 212, 133, 252, 186, 229, 167, 235, 162, 178, 212, 131, 110, 44, 140, 219,
                147, 61, 202, 117, 207, 65, 200, 234, 244, 31, 94, 126, 125, 73, 129, 119, 15, 241,
                221, 212, 70, 245, 188, 178, 112, 3, 245, 89, 109, 193, 44, 0, 195, 97, 51, 167,
                147, 58, 75, 170, 172, 57, 217, 159, 38, 218, 7
            ]
            .into(),
            [
                110, 105, 34, 78, 21, 10, 22, 184, 31, 158, 81, 25, 80, 20, 80, 134, 249, 158, 153,
                79, 235, 183, 66, 227, 204, 37, 48, 98, 90, 16, 176, 177
            ]
            .into(),
        ),
        (
            [
                234, 4, 248, 40, 225, 229, 244, 118, 26, 187, 63, 246, 11, 176, 53, 108, 51, 54, 2,
                132, 191, 16, 8, 85, 111, 150, 228, 94, 80, 192, 144, 159
            ]
            .to_vec(),
            [
                218, 62, 49, 124, 86, 4, 230, 89, 33, 189, 120, 96, 24, 96, 252, 254, 143, 23, 237,
                110, 139, 76, 156, 136, 19, 227, 27, 95, 35, 27, 139, 41, 7, 137, 94, 250, 244,
                182, 21, 18, 179, 83, 149, 0, 217, 26, 254, 153, 10, 124, 163, 238, 174, 97, 247,
                27, 25, 122, 23, 217, 151, 174, 175, 4
            ]
            .into(),
            [
                24, 112, 210, 128, 205, 41, 123, 217, 111, 198, 27, 183, 47, 137, 121, 176, 224,
                199, 232, 150, 60, 2, 61, 77, 173, 222, 150, 210, 111, 117, 114, 41
            ]
            .into(),
        ),
        (
            [
                188, 124, 52, 209, 210, 48, 97, 114, 229, 41, 172, 41, 106, 43, 131, 106, 26, 107,
                32, 24, 108, 47, 188, 99, 255, 193, 23, 43, 140, 103, 188, 88
            ]
            .to_vec(),
            [
                37, 194, 31, 193, 246, 146, 76, 202, 97, 55, 83, 155, 34, 170, 227, 154, 233, 129,
                3, 185, 171, 72, 145, 76, 41, 81, 48, 56, 191, 105, 136, 58, 131, 113, 119, 246,
                112, 224, 61, 98, 22, 194, 40, 197, 166, 146, 250, 12, 181, 35, 169, 216, 80, 209,
                245, 244, 85, 242, 74, 236, 97, 194, 68, 12
            ]
            .into(),
            [
                169, 123, 177, 52, 156, 222, 254, 190, 143, 121, 35, 177, 170, 147, 58, 246, 210,
                32, 213, 223, 102, 205, 179, 158, 7, 55, 113, 112, 36, 65, 202, 110
            ]
            .into(),
        ),
        (
            [
                17, 95, 185, 175, 219, 86, 64, 58, 174, 21, 201, 150, 129, 250, 36, 20, 195, 110,
                0, 113, 182, 24, 206, 73, 68, 16, 116, 151, 217, 0, 183, 164
            ]
            .to_vec(),
            [
                9, 110, 191, 175, 105, 143, 76, 125, 98, 79, 34, 200, 167, 236, 63, 108, 131, 130,
                245, 74, 214, 68, 138, 239, 163, 250, 49, 52, 34, 188, 67, 2, 119, 229, 90, 91,
                153, 236, 151, 216, 105, 48, 164, 148, 165, 209, 106, 20, 20, 22, 237, 200, 103,
                194, 62, 55, 190, 28, 26, 210, 183, 197, 167, 1
            ]
            .into(),
            [
                206, 64, 12, 105, 63, 20, 210, 176, 8, 212, 248, 201, 164, 150, 222, 49, 166, 39,
                119, 217, 70, 248, 8, 110, 185, 11, 69, 69, 36, 71, 244, 51
            ]
            .into(),
        ),
        (
            [
                237, 133, 151, 46, 146, 77, 61, 123, 70, 133, 58, 223, 107, 97, 229, 81, 87, 62,
                207, 195, 97, 170, 15, 82, 211, 5, 20, 89, 163, 217, 128, 35
            ]
            .to_vec(),
            [
                231, 246, 248, 149, 194, 92, 119, 254, 228, 30, 210, 128, 15, 253, 106, 166, 147,
                225, 234, 233, 237, 43, 36, 251, 10, 165, 108, 167, 232, 223, 25, 48, 47, 106, 238,
                79, 44, 147, 10, 185, 132, 95, 41, 145, 246, 136, 218, 229, 196, 167, 150, 183, 19,
                31, 12, 166, 186, 121, 95, 255, 144, 25, 145, 6
            ]
            .into(),
            [
                75, 0, 191, 179, 0, 38, 200, 103, 240, 232, 146, 127, 48, 255, 143, 78, 126, 37,
                225, 254, 192, 232, 53, 173, 108, 253, 215, 87, 105, 175, 132, 65
            ]
            .into(),
        ),
        (
            [
                109, 96, 113, 89, 67, 135, 43, 157, 40, 37, 177, 153, 123, 63, 40, 163, 123, 81,
                136, 113, 150, 209, 88, 176, 208, 58, 208, 160, 86, 26, 225, 66
            ]
            .to_vec(),
            [
                75, 105, 202, 118, 233, 163, 238, 49, 119, 252, 136, 92, 19, 145, 50, 165, 182,
                178, 187, 249, 59, 201, 146, 191, 137, 65, 18, 72, 116, 85, 97, 186, 74, 17, 104,
                200, 141, 102, 38, 80, 13, 68, 231, 138, 76, 14, 32, 167, 62, 37, 156, 126, 23, 81,
                76, 227, 70, 246, 11, 187, 144, 151, 242, 8
            ]
            .into(),
            [
                179, 224, 82, 251, 155, 17, 16, 45, 185, 115, 203, 211, 196, 116, 225, 39, 224, 18,
                222, 193, 92, 208, 170, 176, 232, 53, 34, 151, 235, 167, 237, 33
            ]
            .into(),
        ),
        (
            [
                216, 78, 65, 19, 9, 23, 203, 211, 17, 208, 180, 215, 155, 65, 140, 75, 188, 241,
                162, 82, 190, 215, 90, 217, 220, 160, 31, 168, 221, 172, 182, 129
            ]
            .to_vec(),
            [
                80, 223, 42, 239, 232, 12, 152, 95, 94, 83, 91, 78, 234, 167, 180, 158, 3, 179,
                180, 66, 38, 57, 135, 13, 48, 35, 33, 46, 126, 202, 192, 84, 76, 31, 156, 164, 157,
                134, 187, 206, 16, 241, 165, 166, 99, 70, 150, 62, 233, 42, 181, 123, 68, 253, 219,
                241, 90, 65, 24, 22, 19, 248, 102, 10
            ]
            .into(),
            [
                242, 136, 176, 89, 81, 79, 168, 124, 193, 245, 92, 245, 54, 59, 132, 221, 149, 60,
                59, 239, 141, 207, 146, 80, 131, 218, 22, 9, 26, 44, 78, 217
            ]
            .into(),
        ),
        (
            [
                113, 173, 51, 242, 20, 55, 96, 26, 102, 45, 249, 157, 86, 94, 171, 22, 215, 33,
                156, 121, 103, 97, 95, 98, 87, 226, 153, 54, 60, 83, 140, 122
            ]
            .to_vec(),
            [
                245, 115, 96, 107, 149, 128, 251, 197, 15, 162, 231, 250, 22, 228, 83, 221, 99,
                198, 55, 58, 252, 150, 167, 83, 47, 227, 226, 48, 32, 245, 112, 226, 156, 79, 128,
                147, 171, 72, 104, 101, 111, 241, 233, 80, 37, 163, 13, 64, 72, 124, 68, 78, 67,
                204, 190, 119, 81, 119, 228, 238, 191, 126, 50, 5
            ]
            .into(),
            [
                243, 223, 148, 190, 197, 217, 142, 40, 5, 20, 215, 119, 252, 195, 180, 3, 47, 186,
                57, 72, 136, 28, 143, 75, 213, 11, 238, 183, 105, 249, 0, 238
            ]
            .into(),
        ),
        (
            [
                49, 72, 74, 103, 248, 213, 54, 113, 74, 246, 9, 23, 71, 53, 169, 246, 99, 224, 151,
                195, 252, 116, 201, 4, 137, 105, 42, 92, 70, 108, 150, 227
            ]
            .to_vec(),
            [
                176, 80, 159, 43, 192, 138, 206, 126, 251, 224, 143, 144, 99, 140, 182, 171, 242,
                189, 90, 83, 4, 85, 210, 211, 44, 62, 235, 229, 119, 240, 73, 56, 167, 44, 29, 148,
                110, 104, 58, 89, 132, 146, 38, 163, 44, 21, 148, 15, 248, 91, 85, 100, 23, 71,
                140, 45, 34, 193, 6, 248, 100, 1, 140, 5
            ]
            .into(),
            [
                66, 17, 61, 186, 34, 197, 117, 75, 215, 64, 48, 114, 168, 139, 201, 95, 250, 32,
                242, 102, 65, 198, 97, 157, 250, 228, 211, 122, 140, 107, 22, 73
            ]
            .into(),
        ),
    ];
    static ref LIBRUSTZCASH_BINDING_SIGS: [(
        Vec<u8>,
        Signature<sapling::Binding>,
        VerificationKeyBytes<sapling::Binding>
    ); 32] = [
        (
            [
                16, 28, 190, 75, 156, 66, 96, 79, 4, 199, 3, 195, 150, 247, 136, 198, 203, 45, 109,
                125, 88, 244, 84, 48, 177, 46, 178, 237, 214, 64, 7, 108
            ]
            .to_vec(),
            [
                237, 87, 106, 226, 60, 11, 207, 242, 46, 84, 76, 231, 183, 226, 137, 168, 172, 81,
                133, 166, 208, 118, 55, 133, 102, 155, 236, 16, 30, 208, 68, 84, 22, 83, 106, 161,
                209, 8, 179, 99, 82, 239, 252, 99, 238, 87, 130, 246, 133, 95, 231, 43, 122, 10,
                251, 88, 227, 199, 85, 210, 70, 164, 190, 13
            ]
            .into(),
            [
                231, 192, 142, 217, 209, 34, 78, 213, 129, 213, 83, 32, 218, 125, 131, 62, 236,
                189, 238, 23, 236, 235, 101, 182, 241, 2, 117, 248, 83, 18, 94, 133
            ]
            .into(),
        ),
        (
            [
                108, 56, 40, 255, 104, 11, 219, 166, 98, 154, 153, 67, 75, 194, 79, 17, 86, 115,
                26, 175, 150, 173, 228, 209, 66, 119, 33, 94, 87, 187, 19, 49
            ]
            .to_vec(),
            [
                25, 185, 147, 122, 61, 235, 191, 32, 59, 56, 250, 154, 49, 19, 148, 205, 195, 201,
                60, 131, 133, 238, 221, 50, 237, 81, 7, 211, 116, 217, 205, 115, 236, 103, 208, 32,
                75, 8, 47, 250, 144, 184, 131, 108, 140, 53, 151, 106, 128, 163, 19, 69, 15, 42,
                82, 114, 234, 34, 191, 159, 57, 241, 36, 9
            ]
            .into(),
            [
                21, 9, 29, 239, 240, 242, 41, 16, 105, 198, 200, 212, 207, 17, 76, 88, 75, 79, 223,
                157, 217, 114, 102, 78, 169, 140, 244, 4, 152, 73, 216, 47
            ]
            .into(),
        ),
        (
            [
                145, 63, 229, 40, 221, 46, 129, 128, 68, 42, 148, 149, 87, 96, 109, 94, 188, 197,
                117, 95, 73, 255, 11, 214, 198, 41, 249, 226, 22, 130, 163, 215
            ]
            .to_vec(),
            [
                151, 162, 38, 68, 132, 8, 213, 252, 81, 56, 12, 242, 57, 61, 182, 221, 111, 66,
                218, 143, 184, 210, 235, 125, 18, 226, 157, 128, 213, 96, 89, 84, 208, 167, 64, 60,
                57, 240, 43, 230, 124, 148, 16, 233, 41, 194, 218, 171, 10, 250, 25, 93, 243, 71,
                213, 87, 74, 250, 214, 223, 30, 52, 193, 3
            ]
            .into(),
            [
                208, 32, 161, 164, 158, 193, 116, 72, 31, 83, 149, 118, 131, 138, 140, 54, 21, 92,
                188, 31, 124, 204, 144, 72, 186, 253, 87, 4, 193, 138, 130, 157
            ]
            .into(),
        ),
        (
            [
                208, 243, 205, 249, 49, 107, 35, 33, 206, 165, 99, 248, 160, 186, 39, 50, 71, 156,
                226, 164, 9, 17, 162, 158, 198, 154, 10, 123, 189, 221, 247, 116
            ]
            .to_vec(),
            [
                129, 23, 184, 104, 87, 77, 84, 102, 62, 149, 61, 114, 220, 195, 21, 194, 115, 44,
                26, 30, 72, 30, 86, 61, 179, 74, 70, 16, 192, 0, 220, 195, 129, 209, 112, 61, 226,
                172, 132, 127, 52, 103, 213, 217, 213, 42, 202, 6, 179, 184, 7, 17, 29, 5, 28, 24,
                189, 18, 135, 5, 58, 118, 93, 4
            ]
            .into(),
            [
                15, 202, 69, 167, 230, 248, 5, 36, 71, 19, 94, 186, 96, 22, 6, 232, 59, 99, 131,
                148, 100, 216, 51, 195, 129, 89, 238, 56, 81, 50, 245, 220
            ]
            .into(),
        ),
        (
            [
                73, 53, 27, 188, 30, 74, 75, 136, 216, 22, 115, 179, 41, 12, 101, 135, 241, 208,
                177, 226, 3, 109, 120, 48, 2, 120, 150, 27, 29, 59, 61, 180
            ]
            .to_vec(),
            [
                158, 145, 214, 27, 179, 127, 199, 48, 80, 13, 226, 197, 111, 19, 72, 183, 73, 51,
                34, 147, 241, 236, 21, 160, 99, 218, 202, 73, 231, 123, 62, 243, 148, 119, 42, 185,
                145, 225, 239, 177, 132, 157, 101, 0, 210, 63, 72, 22, 62, 130, 52, 222, 36, 13,
                124, 0, 208, 141, 234, 3, 38, 207, 72, 13
            ]
            .into(),
            [
                13, 167, 148, 29, 141, 128, 10, 130, 105, 36, 188, 29, 169, 176, 172, 169, 162, 21,
                90, 3, 180, 221, 68, 90, 252, 154, 62, 61, 20, 47, 119, 145
            ]
            .into(),
        ),
        (
            [
                158, 55, 154, 163, 138, 43, 56, 169, 209, 184, 225, 86, 39, 131, 0, 194, 62, 122,
                84, 176, 197, 115, 88, 216, 47, 127, 36, 131, 215, 205, 251, 69
            ]
            .to_vec(),
            [
                244, 19, 20, 115, 160, 228, 19, 65, 61, 5, 118, 252, 98, 166, 95, 8, 152, 88, 170,
                220, 129, 233, 169, 252, 99, 157, 119, 63, 243, 158, 82, 170, 248, 192, 197, 111,
                151, 237, 195, 54, 128, 105, 62, 34, 177, 118, 224, 123, 185, 188, 212, 14, 56, 56,
                177, 208, 159, 190, 229, 97, 43, 231, 65, 7
            ]
            .into(),
            [
                89, 97, 162, 229, 121, 16, 170, 148, 38, 103, 34, 212, 200, 198, 164, 51, 143, 191,
                7, 42, 178, 208, 97, 236, 230, 113, 1, 197, 82, 39, 203, 83
            ]
            .into(),
        ),
        (
            [
                217, 101, 76, 11, 40, 202, 129, 190, 187, 84, 92, 68, 181, 190, 66, 89, 233, 122,
                72, 49, 98, 38, 40, 129, 112, 67, 119, 39, 61, 230, 140, 107
            ]
            .to_vec(),
            [
                218, 247, 117, 213, 208, 142, 220, 44, 216, 96, 137, 132, 148, 31, 152, 156, 113,
                69, 28, 193, 149, 80, 30, 133, 55, 153, 151, 101, 55, 180, 57, 222, 123, 37, 88, 7,
                15, 26, 6, 186, 101, 236, 56, 130, 31, 172, 73, 71, 16, 248, 103, 15, 141, 24, 57,
                119, 157, 228, 42, 253, 120, 43, 98, 1
            ]
            .into(),
            [
                122, 178, 202, 205, 106, 145, 155, 121, 178, 142, 171, 204, 133, 94, 219, 185, 58,
                222, 73, 44, 245, 198, 158, 142, 246, 214, 29, 113, 110, 55, 234, 95
            ]
            .into(),
        ),
        (
            [
                10, 212, 76, 139, 47, 234, 73, 233, 131, 223, 100, 101, 161, 23, 26, 161, 12, 107,
                27, 76, 50, 232, 165, 234, 34, 170, 126, 169, 179, 7, 168, 216
            ]
            .to_vec(),
            [
                48, 232, 149, 96, 68, 229, 204, 221, 41, 15, 163, 211, 84, 69, 221, 144, 189, 141,
                144, 1, 47, 243, 97, 195, 247, 246, 120, 43, 220, 114, 88, 95, 94, 191, 240, 214,
                205, 145, 222, 102, 134, 72, 175, 115, 87, 216, 118, 1, 122, 112, 186, 41, 167,
                179, 167, 120, 59, 222, 49, 226, 201, 166, 210, 5
            ]
            .into(),
            [
                176, 34, 144, 56, 153, 248, 153, 225, 109, 38, 93, 184, 90, 94, 254, 213, 242, 15,
                245, 212, 5, 200, 50, 48, 195, 249, 197, 75, 249, 58, 59, 65
            ]
            .into(),
        ),
        (
            [
                7, 121, 157, 250, 188, 252, 108, 217, 56, 87, 183, 45, 98, 129, 233, 210, 121, 143,
                173, 195, 172, 2, 154, 125, 176, 148, 3, 234, 3, 250, 127, 116
            ]
            .to_vec(),
            [
                236, 49, 41, 146, 67, 207, 165, 207, 121, 176, 150, 26, 46, 207, 118, 112, 226, 31,
                19, 98, 57, 97, 183, 215, 227, 144, 101, 255, 163, 15, 186, 104, 231, 212, 237,
                159, 240, 78, 188, 177, 74, 109, 117, 211, 2, 2, 88, 98, 208, 83, 235, 61, 194,
                114, 178, 8, 215, 207, 175, 206, 156, 183, 212, 4
            ]
            .into(),
            [
                114, 45, 120, 112, 212, 36, 97, 130, 84, 73, 186, 171, 73, 208, 112, 166, 193, 161,
                139, 138, 33, 191, 122, 112, 173, 253, 115, 135, 25, 216, 165, 128
            ]
            .into(),
        ),
        (
            [
                161, 98, 15, 93, 184, 31, 193, 208, 84, 234, 38, 229, 208, 207, 27, 172, 152, 170,
                39, 138, 187, 88, 195, 82, 253, 108, 101, 128, 110, 86, 114, 242
            ]
            .to_vec(),
            [
                250, 82, 164, 64, 95, 96, 202, 190, 186, 124, 162, 48, 64, 185, 207, 25, 159, 197,
                148, 50, 180, 55, 10, 11, 1, 34, 219, 194, 215, 160, 187, 204, 161, 176, 232, 43,
                129, 43, 17, 183, 80, 223, 55, 39, 58, 160, 120, 162, 247, 92, 64, 233, 145, 81,
                136, 129, 158, 168, 47, 193, 17, 243, 141, 0
            ]
            .into(),
            [
                51, 144, 219, 246, 218, 85, 243, 42, 222, 208, 32, 16, 237, 244, 131, 139, 69, 139,
                164, 231, 14, 239, 64, 94, 192, 127, 86, 97, 228, 50, 123, 65
            ]
            .into(),
        ),
        (
            [
                123, 7, 11, 119, 243, 237, 229, 160, 242, 163, 173, 216, 117, 94, 8, 29, 112, 178,
                23, 50, 180, 75, 18, 189, 215, 221, 21, 8, 86, 209, 177, 87
            ]
            .to_vec(),
            [
                25, 68, 135, 235, 188, 239, 156, 9, 63, 43, 65, 178, 241, 114, 230, 156, 94, 118,
                114, 97, 121, 71, 206, 185, 144, 55, 241, 233, 254, 111, 209, 39, 206, 8, 217, 109,
                180, 71, 175, 235, 226, 39, 101, 241, 231, 147, 14, 121, 250, 107, 212, 228, 201,
                63, 134, 232, 110, 101, 43, 40, 209, 83, 254, 10
            ]
            .into(),
            [
                196, 239, 106, 25, 44, 155, 136, 240, 60, 108, 136, 236, 150, 16, 37, 71, 89, 205,
                31, 166, 13, 136, 31, 105, 126, 55, 63, 223, 79, 123, 41, 144
            ]
            .into(),
        ),
        (
            [
                181, 94, 198, 199, 78, 194, 64, 118, 146, 51, 79, 70, 73, 206, 87, 51, 96, 13, 145,
                198, 78, 245, 179, 182, 169, 147, 95, 137, 77, 33, 247, 4
            ]
            .to_vec(),
            [
                201, 195, 83, 81, 83, 219, 229, 100, 171, 188, 138, 28, 52, 163, 215, 12, 33, 12,
                163, 52, 146, 64, 74, 212, 148, 195, 255, 242, 96, 87, 106, 212, 174, 204, 31, 138,
                144, 32, 191, 21, 193, 63, 4, 204, 126, 248, 171, 61, 19, 156, 114, 244, 46, 18,
                160, 156, 211, 83, 216, 156, 222, 131, 105, 6
            ]
            .into(),
            [
                170, 211, 15, 217, 62, 99, 68, 140, 87, 122, 65, 150, 24, 91, 62, 183, 205, 43,
                222, 241, 69, 89, 213, 235, 51, 214, 156, 238, 67, 119, 239, 62
            ]
            .into(),
        ),
        (
            [
                108, 144, 146, 213, 254, 69, 48, 80, 172, 3, 26, 161, 90, 132, 50, 0, 192, 101, 98,
                65, 167, 199, 175, 147, 29, 177, 217, 226, 72, 233, 93, 118
            ]
            .to_vec(),
            [
                85, 140, 40, 250, 36, 52, 31, 161, 79, 218, 140, 106, 155, 15, 129, 202, 227, 206,
                72, 110, 176, 134, 218, 71, 19, 139, 164, 230, 148, 55, 34, 6, 245, 17, 95, 19,
                253, 21, 21, 156, 119, 213, 25, 201, 31, 130, 28, 215, 2, 0, 120, 56, 248, 134,
                224, 82, 138, 50, 150, 134, 144, 86, 176, 2
            ]
            .into(),
            [
                5, 132, 40, 151, 157, 13, 233, 182, 145, 5, 148, 185, 157, 60, 26, 187, 1, 142, 87,
                236, 45, 171, 63, 192, 150, 27, 3, 65, 129, 120, 47, 137
            ]
            .into(),
        ),
        (
            [
                106, 232, 15, 114, 137, 49, 56, 62, 28, 42, 176, 201, 156, 86, 109, 234, 3, 191,
                44, 231, 22, 110, 103, 24, 204, 71, 207, 80, 255, 34, 150, 208
            ]
            .to_vec(),
            [
                76, 143, 197, 238, 95, 10, 247, 248, 222, 235, 98, 201, 131, 124, 227, 105, 9, 253,
                233, 16, 51, 122, 62, 96, 253, 17, 146, 247, 90, 90, 226, 45, 0, 148, 219, 178,
                245, 188, 14, 131, 241, 153, 54, 65, 124, 176, 112, 161, 117, 130, 154, 212, 169,
                222, 227, 123, 23, 159, 43, 212, 240, 195, 105, 2
            ]
            .into(),
            [
                129, 237, 183, 32, 252, 37, 62, 9, 191, 183, 110, 221, 208, 147, 195, 155, 170, 90,
                165, 62, 148, 239, 184, 96, 5, 193, 251, 175, 104, 22, 86, 20
            ]
            .into(),
        ),
        (
            [
                115, 113, 54, 112, 20, 175, 145, 55, 156, 102, 8, 55, 94, 188, 73, 151, 45, 250,
                55, 84, 176, 22, 100, 238, 102, 177, 44, 154, 49, 69, 93, 64
            ]
            .to_vec(),
            [
                173, 45, 205, 174, 114, 159, 62, 74, 64, 1, 168, 92, 241, 121, 55, 124, 160, 55,
                250, 71, 230, 228, 135, 71, 28, 232, 153, 135, 249, 105, 179, 85, 169, 85, 150,
                126, 168, 172, 253, 116, 217, 80, 179, 39, 245, 185, 138, 144, 204, 111, 7, 194,
                185, 241, 184, 6, 166, 103, 210, 97, 27, 24, 83, 2
            ]
            .into(),
            [
                102, 255, 40, 224, 138, 207, 8, 238, 77, 169, 100, 66, 36, 118, 240, 162, 156, 55,
                103, 103, 65, 234, 190, 160, 149, 137, 75, 180, 194, 77, 163, 102
            ]
            .into(),
        ),
        (
            [
                93, 203, 169, 70, 124, 231, 163, 178, 40, 136, 241, 181, 37, 231, 93, 208, 80, 45,
                166, 216, 63, 169, 147, 117, 109, 223, 26, 76, 4, 125, 65, 115
            ]
            .to_vec(),
            [
                229, 204, 69, 161, 126, 241, 215, 126, 254, 103, 223, 201, 45, 103, 237, 217, 74,
                150, 218, 24, 154, 200, 249, 94, 241, 88, 231, 134, 242, 211, 12, 78, 99, 31, 12,
                60, 63, 235, 62, 5, 147, 224, 243, 161, 46, 209, 65, 113, 105, 220, 49, 153, 213,
                16, 244, 133, 36, 110, 201, 30, 35, 129, 4, 7
            ]
            .into(),
            [
                206, 114, 177, 72, 56, 89, 233, 157, 190, 23, 239, 84, 26, 103, 247, 220, 31, 117,
                100, 129, 54, 116, 9, 254, 141, 21, 69, 145, 40, 247, 240, 109
            ]
            .into(),
        ),
        (
            [
                10, 18, 144, 141, 46, 99, 152, 10, 220, 179, 10, 31, 65, 109, 230, 15, 245, 46,
                235, 2, 149, 188, 163, 203, 110, 98, 125, 129, 101, 155, 205, 189
            ]
            .to_vec(),
            [
                176, 119, 123, 137, 30, 22, 101, 77, 181, 10, 152, 72, 84, 223, 25, 119, 147, 186,
                213, 209, 251, 176, 185, 110, 74, 111, 62, 51, 91, 85, 209, 158, 201, 200, 44, 28,
                235, 76, 102, 83, 228, 57, 246, 69, 127, 114, 203, 202, 6, 106, 44, 94, 153, 66,
                230, 238, 30, 44, 159, 31, 19, 151, 110, 12
            ]
            .into(),
            [
                64, 131, 183, 133, 193, 39, 56, 61, 212, 66, 113, 5, 204, 197, 56, 160, 202, 240,
                50, 93, 214, 52, 90, 201, 106, 70, 190, 136, 73, 153, 110, 8
            ]
            .into(),
        ),
        (
            [
                188, 253, 211, 252, 199, 54, 126, 222, 14, 31, 80, 124, 161, 10, 199, 39, 248, 45,
                154, 68, 162, 227, 125, 107, 172, 41, 41, 151, 234, 171, 164, 136
            ]
            .to_vec(),
            [
                17, 154, 126, 54, 92, 150, 140, 160, 171, 95, 141, 143, 250, 132, 59, 14, 50, 54,
                116, 195, 157, 200, 83, 83, 53, 167, 141, 93, 125, 64, 197, 95, 95, 14, 230, 27, 2,
                137, 105, 185, 52, 223, 164, 48, 90, 134, 63, 185, 32, 204, 11, 147, 124, 216, 235,
                239, 228, 145, 48, 40, 141, 163, 98, 0
            ]
            .into(),
            [
                68, 208, 194, 182, 6, 214, 24, 148, 56, 42, 161, 152, 138, 217, 59, 172, 193, 119,
                237, 56, 238, 225, 133, 77, 224, 14, 75, 145, 53, 5, 142, 101
            ]
            .into(),
        ),
        (
            [
                226, 227, 113, 72, 108, 239, 130, 37, 21, 121, 69, 158, 159, 30, 187, 20, 25, 129,
                18, 211, 152, 237, 217, 225, 103, 173, 225, 84, 130, 14, 98, 152
            ]
            .to_vec(),
            [
                246, 81, 2, 59, 76, 187, 86, 118, 239, 175, 45, 169, 81, 49, 216, 203, 99, 159,
                145, 134, 248, 26, 79, 247, 17, 49, 133, 39, 73, 106, 60, 173, 37, 129, 155, 148,
                224, 137, 174, 230, 242, 206, 205, 1, 254, 247, 32, 212, 34, 154, 21, 145, 54, 173,
                234, 52, 104, 40, 106, 6, 242, 69, 96, 10
            ]
            .into(),
            [
                49, 83, 150, 31, 12, 102, 183, 80, 9, 184, 33, 207, 190, 17, 184, 63, 229, 152,
                183, 90, 179, 119, 135, 199, 193, 61, 251, 141, 132, 254, 53, 19
            ]
            .into(),
        ),
        (
            [
                225, 242, 213, 131, 75, 206, 121, 1, 11, 169, 169, 241, 41, 113, 255, 138, 109,
                216, 145, 166, 98, 81, 85, 105, 178, 76, 8, 213, 3, 25, 15, 164
            ]
            .to_vec(),
            [
                144, 121, 70, 141, 125, 29, 95, 18, 88, 111, 23, 199, 155, 19, 207, 156, 118, 120,
                148, 46, 38, 254, 235, 48, 228, 123, 158, 166, 63, 244, 228, 4, 202, 228, 160, 81,
                68, 19, 187, 107, 109, 149, 80, 238, 223, 104, 91, 210, 115, 22, 217, 74, 43, 239,
                186, 194, 133, 90, 224, 160, 220, 38, 140, 13
            ]
            .into(),
            [
                138, 225, 61, 235, 12, 42, 193, 198, 100, 212, 166, 123, 121, 117, 207, 170, 164,
                30, 120, 74, 16, 151, 239, 199, 48, 11, 36, 162, 187, 205, 70, 147
            ]
            .into(),
        ),
        (
            [
                225, 47, 87, 102, 125, 86, 130, 79, 147, 184, 26, 138, 230, 229, 254, 180, 191,
                252, 9, 107, 29, 126, 101, 54, 98, 134, 228, 105, 109, 143, 118, 42
            ]
            .to_vec(),
            [
                68, 166, 113, 4, 147, 21, 5, 61, 112, 131, 210, 120, 128, 9, 14, 44, 10, 108, 23,
                196, 44, 185, 172, 243, 180, 73, 220, 122, 240, 52, 94, 86, 65, 85, 132, 140, 74,
                17, 172, 85, 19, 230, 180, 79, 34, 166, 24, 180, 131, 97, 102, 44, 208, 99, 236,
                107, 124, 2, 199, 192, 241, 229, 106, 2
            ]
            .into(),
            [
                125, 43, 204, 57, 195, 59, 117, 225, 126, 221, 27, 33, 79, 38, 139, 176, 186, 47,
                7, 150, 242, 182, 218, 190, 23, 94, 132, 3, 140, 245, 239, 18
            ]
            .into(),
        ),
        (
            [
                208, 177, 124, 14, 226, 70, 2, 251, 231, 125, 90, 16, 123, 73, 47, 104, 159, 135,
                54, 131, 77, 244, 234, 106, 178, 9, 156, 169, 107, 233, 234, 187
            ]
            .to_vec(),
            [
                152, 136, 190, 89, 167, 242, 143, 37, 179, 64, 119, 195, 44, 125, 172, 222, 184,
                49, 95, 18, 64, 121, 3, 211, 213, 191, 252, 187, 60, 81, 12, 137, 228, 95, 216, 32,
                78, 235, 209, 241, 2, 228, 161, 132, 75, 203, 49, 154, 81, 66, 14, 78, 205, 123,
                249, 65, 106, 43, 72, 208, 183, 158, 178, 9
            ]
            .into(),
            [
                114, 113, 231, 198, 235, 57, 190, 236, 51, 245, 171, 233, 225, 34, 227, 33, 144,
                210, 9, 70, 101, 117, 163, 244, 82, 164, 70, 162, 164, 202, 249, 173
            ]
            .into(),
        ),
        (
            [
                71, 61, 57, 222, 190, 1, 120, 139, 238, 48, 237, 151, 52, 201, 242, 21, 200, 29,
                54, 2, 193, 153, 8, 73, 78, 186, 156, 183, 224, 126, 164, 52
            ]
            .to_vec(),
            [
                198, 185, 85, 138, 190, 54, 233, 102, 247, 52, 43, 110, 176, 53, 49, 66, 65, 165,
                166, 130, 43, 157, 250, 149, 135, 102, 253, 98, 31, 205, 62, 88, 70, 37, 179, 241,
                29, 57, 104, 204, 159, 111, 28, 249, 250, 38, 13, 101, 60, 101, 47, 48, 193, 114,
                223, 36, 51, 45, 33, 150, 219, 69, 193, 2
            ]
            .into(),
            [
                250, 27, 172, 162, 248, 64, 155, 57, 38, 153, 71, 119, 202, 180, 187, 215, 160, 63,
                243, 69, 10, 226, 114, 197, 154, 168, 127, 211, 82, 53, 5, 91
            ]
            .into(),
        ),
        (
            [
                88, 104, 132, 39, 159, 181, 16, 226, 167, 230, 84, 135, 90, 15, 145, 23, 88, 99,
                83, 19, 224, 144, 136, 116, 8, 33, 201, 84, 20, 31, 189, 203
            ]
            .to_vec(),
            [
                24, 190, 145, 96, 30, 91, 86, 153, 77, 234, 166, 254, 76, 245, 157, 23, 152, 129,
                141, 84, 0, 105, 92, 100, 108, 196, 197, 126, 156, 15, 209, 230, 50, 216, 235, 177,
                223, 116, 54, 192, 134, 106, 33, 82, 110, 136, 54, 47, 198, 141, 123, 47, 191, 163,
                129, 176, 165, 224, 121, 158, 119, 168, 219, 6
            ]
            .into(),
            [
                205, 92, 57, 103, 120, 65, 0, 136, 125, 30, 25, 141, 176, 72, 191, 198, 200, 70, 9,
                5, 52, 155, 123, 23, 16, 196, 109, 169, 146, 225, 165, 171
            ]
            .into(),
        ),
        (
            [
                234, 4, 248, 40, 225, 229, 244, 118, 26, 187, 63, 246, 11, 176, 53, 108, 51, 54, 2,
                132, 191, 16, 8, 85, 111, 150, 228, 94, 80, 192, 144, 159
            ]
            .to_vec(),
            [
                13, 69, 57, 44, 76, 45, 23, 39, 17, 117, 23, 6, 98, 211, 163, 51, 119, 240, 48, 2,
                88, 73, 205, 190, 4, 203, 4, 250, 75, 134, 75, 82, 5, 89, 255, 171, 109, 22, 83,
                76, 177, 99, 224, 243, 84, 192, 83, 127, 105, 61, 96, 247, 107, 67, 78, 144, 206,
                56, 230, 210, 55, 150, 3, 0
            ]
            .into(),
            [
                7, 46, 165, 146, 105, 160, 80, 138, 114, 236, 217, 138, 9, 100, 41, 231, 215, 92,
                215, 229, 24, 32, 129, 25, 16, 1, 243, 176, 0, 204, 147, 35
            ]
            .into(),
        ),
        (
            [
                188, 124, 52, 209, 210, 48, 97, 114, 229, 41, 172, 41, 106, 43, 131, 106, 26, 107,
                32, 24, 108, 47, 188, 99, 255, 193, 23, 43, 140, 103, 188, 88
            ]
            .to_vec(),
            [
                115, 233, 189, 93, 167, 14, 110, 88, 157, 118, 162, 29, 195, 11, 250, 122, 252,
                224, 68, 12, 243, 10, 159, 127, 50, 65, 235, 115, 91, 120, 47, 211, 89, 0, 180, 99,
                140, 182, 215, 120, 146, 16, 247, 74, 149, 104, 13, 30, 64, 239, 55, 77, 204, 75,
                94, 231, 232, 217, 60, 7, 53, 133, 124, 4
            ]
            .into(),
            [
                251, 201, 115, 163, 5, 40, 10, 63, 211, 146, 80, 78, 1, 16, 107, 199, 146, 72, 180,
                141, 183, 19, 199, 93, 33, 229, 205, 70, 32, 143, 163, 54
            ]
            .into(),
        ),
        (
            [
                17, 95, 185, 175, 219, 86, 64, 58, 174, 21, 201, 150, 129, 250, 36, 20, 195, 110,
                0, 113, 182, 24, 206, 73, 68, 16, 116, 151, 217, 0, 183, 164
            ]
            .to_vec(),
            [
                5, 171, 71, 27, 104, 155, 152, 225, 171, 250, 77, 64, 133, 52, 21, 11, 195, 1, 249,
                69, 239, 136, 150, 180, 164, 139, 50, 132, 81, 146, 130, 39, 115, 69, 122, 24, 113,
                125, 135, 202, 222, 68, 56, 18, 111, 90, 170, 209, 231, 75, 90, 125, 210, 192, 36,
                168, 7, 253, 154, 47, 142, 204, 253, 12
            ]
            .into(),
            [
                228, 135, 125, 188, 76, 87, 101, 6, 105, 175, 105, 96, 87, 202, 53, 118, 51, 202,
                103, 210, 10, 155, 106, 16, 142, 210, 53, 248, 207, 131, 121, 51
            ]
            .into(),
        ),
        (
            [
                237, 133, 151, 46, 146, 77, 61, 123, 70, 133, 58, 223, 107, 97, 229, 81, 87, 62,
                207, 195, 97, 170, 15, 82, 211, 5, 20, 89, 163, 217, 128, 35
            ]
            .to_vec(),
            [
                94, 58, 85, 222, 114, 105, 9, 111, 156, 63, 24, 17, 189, 80, 195, 29, 204, 168, 0,
                170, 46, 12, 220, 2, 27, 31, 178, 205, 217, 146, 221, 242, 32, 34, 62, 194, 157,
                102, 152, 161, 12, 11, 220, 49, 126, 197, 57, 143, 98, 24, 127, 211, 128, 189, 79,
                65, 75, 88, 157, 47, 67, 40, 95, 6
            ]
            .into(),
            [
                204, 43, 68, 82, 129, 227, 172, 253, 204, 243, 143, 138, 221, 73, 236, 38, 143, 9,
                41, 52, 38, 227, 152, 202, 198, 118, 237, 255, 65, 5, 251, 229
            ]
            .into(),
        ),
        (
            [
                109, 96, 113, 89, 67, 135, 43, 157, 40, 37, 177, 153, 123, 63, 40, 163, 123, 81,
                136, 113, 150, 209, 88, 176, 208, 58, 208, 160, 86, 26, 225, 66
            ]
            .to_vec(),
            [
                78, 191, 253, 51, 223, 212, 196, 105, 82, 191, 242, 41, 117, 77, 104, 209, 172,
                202, 40, 253, 38, 157, 54, 13, 235, 157, 47, 31, 88, 249, 2, 196, 88, 126, 211,
                173, 47, 13, 111, 53, 112, 179, 185, 203, 119, 4, 74, 67, 53, 11, 245, 184, 42,
                160, 147, 154, 209, 242, 196, 248, 159, 150, 142, 1
            ]
            .into(),
            [
                120, 105, 195, 25, 208, 187, 21, 168, 238, 79, 44, 94, 96, 0, 112, 179, 221, 190,
                141, 53, 19, 61, 42, 48, 175, 210, 148, 183, 79, 40, 7, 45
            ]
            .into(),
        ),
        (
            [
                216, 78, 65, 19, 9, 23, 203, 211, 17, 208, 180, 215, 155, 65, 140, 75, 188, 241,
                162, 82, 190, 215, 90, 217, 220, 160, 31, 168, 221, 172, 182, 129
            ]
            .to_vec(),
            [
                153, 116, 51, 238, 50, 167, 226, 181, 122, 111, 73, 88, 42, 159, 251, 212, 194, 20,
                98, 43, 48, 125, 80, 116, 7, 243, 13, 203, 169, 71, 111, 165, 16, 236, 122, 203, 4,
                125, 111, 20, 96, 233, 121, 71, 138, 215, 169, 154, 134, 205, 9, 68, 197, 236, 179,
                102, 229, 168, 162, 245, 111, 85, 106, 3
            ]
            .into(),
            [
                48, 161, 121, 172, 29, 185, 169, 8, 50, 189, 168, 130, 140, 254, 197, 6, 188, 93,
                253, 67, 198, 198, 185, 212, 42, 224, 217, 195, 129, 84, 230, 5
            ]
            .into(),
        ),
        (
            [
                113, 173, 51, 242, 20, 55, 96, 26, 102, 45, 249, 157, 86, 94, 171, 22, 215, 33,
                156, 121, 103, 97, 95, 98, 87, 226, 153, 54, 60, 83, 140, 122
            ]
            .to_vec(),
            [
                235, 56, 216, 155, 20, 252, 54, 184, 249, 162, 116, 122, 221, 112, 171, 18, 182,
                233, 210, 116, 84, 216, 69, 75, 219, 2, 138, 87, 172, 233, 50, 200, 135, 188, 136,
                55, 81, 43, 101, 66, 176, 248, 24, 208, 106, 131, 44, 240, 166, 159, 162, 107, 58,
                126, 190, 11, 22, 56, 17, 96, 27, 179, 173, 9
            ]
            .into(),
            [
                87, 69, 133, 181, 55, 135, 65, 98, 105, 71, 175, 166, 169, 105, 11, 82, 66, 250,
                166, 198, 232, 223, 130, 20, 236, 2, 0, 207, 179, 65, 151, 72
            ]
            .into(),
        ),
        (
            [
                49, 72, 74, 103, 248, 213, 54, 113, 74, 246, 9, 23, 71, 53, 169, 246, 99, 224, 151,
                195, 252, 116, 201, 4, 137, 105, 42, 92, 70, 108, 150, 227
            ]
            .to_vec(),
            [
                138, 52, 43, 171, 87, 79, 36, 251, 109, 39, 155, 54, 200, 106, 231, 234, 75, 234,
                140, 168, 179, 68, 180, 227, 220, 212, 43, 44, 175, 177, 75, 108, 47, 216, 185, 32,
                126, 143, 204, 58, 46, 123, 85, 224, 214, 181, 247, 48, 252, 37, 5, 186, 193, 168,
                142, 205, 97, 153, 36, 161, 178, 162, 152, 3
            ]
            .into(),
            [
                75, 64, 164, 20, 239, 96, 148, 68, 235, 208, 11, 77, 136, 192, 177, 116, 193, 84,
                44, 205, 239, 143, 46, 162, 51, 228, 61, 95, 160, 21, 200, 102
            ]
            .into(),
        ),
    ];
}
