//! Test vectors for Orchard key components.

pub(crate) struct TestVector {
    pub(crate) incoming_viewing_key: [u8; 64],
    pub(crate) ovk: [u8; 32],
    pub(crate) default_d: [u8; 11],
    pub(crate) default_pk_d: [u8; 32],
    pub(crate) v: u64,
    pub(crate) rseed: [u8; 32],
    pub(crate) memo: [u8; 512],
    pub(crate) cv_net: [u8; 32],
    pub(crate) rho: [u8; 32],
    pub(crate) cmx: [u8; 32],
    pub(crate) esk: [u8; 32],
    pub(crate) ephemeral_key: [u8; 32],
    pub(crate) shared_secret: [u8; 32],
    pub(crate) k_enc: [u8; 32],
    pub(crate) p_enc: [u8; 564],
    pub(crate) c_enc: [u8; 580],
    pub(crate) ock: [u8; 32],
    pub(crate) op: [u8; 64],
    pub(crate) c_out: [u8; 80],
}

pub(crate) fn test_vectors() -> Vec<TestVector> {
    // From https://github.com/zcash-hackworks/zcash-test-vectors/blob/master/orchard_note_encryption.py
    vec![
        TestVector {
            incoming_viewing_key: [
                0x10, 0x39, 0xd8, 0xe6, 0x4a, 0x80, 0x90, 0x2e, 0x10, 0x59, 0x47, 0x81, 0x7d, 0xf3,
                0xbd, 0xfb, 0x7d, 0xf7, 0x03, 0x0e, 0x68, 0x73, 0x9f, 0x9c, 0x53, 0x3a, 0x36, 0xbf,
                0x5a, 0x6a, 0x80, 0x72, 0x43, 0x10, 0x6d, 0xe9, 0xa7, 0xec, 0x54, 0xdd, 0x36, 0xdf,
                0xa7, 0x0b, 0xdb, 0xd9, 0x07, 0x2d, 0xbd, 0xda, 0xb5, 0xe0, 0x66, 0xaa, 0xef, 0xfc,
                0xf9, 0xbb, 0xa3, 0x20, 0xd4, 0xff, 0xf7, 0x12,
            ],
            ovk: [
                0x5d, 0x7a, 0x8f, 0x73, 0x9a, 0x2d, 0x9e, 0x94, 0x5b, 0x0c, 0xe1, 0x52, 0xa8, 0x04,
                0x9e, 0x29, 0x4c, 0x4d, 0x6e, 0x66, 0xb1, 0x64, 0x93, 0x9d, 0xaf, 0xfa, 0x2e, 0xf6,
                0xee, 0x69, 0x21, 0x48,
            ],
            default_d: [
                0x56, 0xe8, 0x4b, 0x1a, 0xdc, 0x94, 0x23, 0xc3, 0x67, 0x6c, 0x04,
            ],
            default_pk_d: [
                0x63, 0xf7, 0x12, 0x5d, 0xf4, 0x83, 0x6f, 0xd2, 0x81, 0x6b, 0x02, 0x4e, 0xe7, 0x0e,
                0xfe, 0x09, 0xfb, 0x9a, 0x7b, 0x38, 0x63, 0xc6, 0xea, 0xcd, 0xf9, 0x5e, 0x03, 0x89,
                0x49, 0x50, 0x69, 0x2c,
            ],
            v: 8567075990963576717,
            rseed: [
                0xbf, 0x69, 0xb8, 0x25, 0x0c, 0x18, 0xef, 0x41, 0x29, 0x4c, 0xa9, 0x79, 0x93, 0xdb,
                0x54, 0x6c, 0x1f, 0xe0, 0x1f, 0x7e, 0x9c, 0x8e, 0x36, 0xd6, 0xa5, 0xe2, 0x9d, 0x4e,
                0x30, 0xa7, 0x35, 0x94,
            ],
            memo: [
                0xff, 0xbf, 0x50, 0x98, 0x42, 0x1c, 0x69, 0x37, 0x8a, 0xf1, 0xe4, 0x0f, 0x64, 0xe1,
                0x25, 0x94, 0x6f, 0x62, 0xc2, 0xfa, 0x7b, 0x2f, 0xec, 0xbc, 0xb6, 0x4b, 0x69, 0x68,
                0x91, 0x2a, 0x63, 0x81, 0xce, 0x3d, 0xc1, 0x66, 0xd5, 0x6a, 0x1d, 0x62, 0xf5, 0xa8,
                0xd7, 0x55, 0x1d, 0xb5, 0xfd, 0x93, 0x13, 0xe8, 0xc7, 0x20, 0x3d, 0x99, 0x6a, 0xf7,
                0xd4, 0x77, 0x08, 0x37, 0x56, 0xd5, 0x9a, 0xf8, 0x0d, 0x06, 0xa7, 0x45, 0xf4, 0x4a,
                0xb0, 0x23, 0x75, 0x2c, 0xb5, 0xb4, 0x06, 0xed, 0x89, 0x85, 0xe1, 0x81, 0x30, 0xab,
                0x33, 0x36, 0x26, 0x97, 0xb0, 0xe4, 0xe4, 0xc7, 0x63, 0xcc, 0xb8, 0xf6, 0x76, 0x49,
                0x5c, 0x22, 0x2f, 0x7f, 0xba, 0x1e, 0x31, 0xde, 0xfa, 0x3d, 0x5a, 0x57, 0xef, 0xc2,
                0xe1, 0xe9, 0xb0, 0x1a, 0x03, 0x55, 0x87, 0xd5, 0xfb, 0x1a, 0x38, 0xe0, 0x1d, 0x94,
                0x90, 0x3d, 0x3c, 0x3e, 0x0a, 0xd3, 0x36, 0x0c, 0x1d, 0x37, 0x10, 0xac, 0xd2, 0x0b,
                0x18, 0x3e, 0x31, 0xd4, 0x9f, 0x25, 0xc9, 0xa1, 0x38, 0xf4, 0x9b, 0x1a, 0x53, 0x7e,
                0xdc, 0xf0, 0x4b, 0xe3, 0x4a, 0x98, 0x51, 0xa7, 0xaf, 0x9d, 0xb6, 0x99, 0x0e, 0xd8,
                0x3d, 0xd6, 0x4a, 0xf3, 0x59, 0x7c, 0x04, 0x32, 0x3e, 0xa5, 0x1b, 0x00, 0x52, 0xad,
                0x80, 0x84, 0xa8, 0xb9, 0xda, 0x94, 0x8d, 0x32, 0x0d, 0xad, 0xd6, 0x4f, 0x54, 0x31,
                0xe6, 0x1d, 0xdf, 0x65, 0x8d, 0x24, 0xae, 0x67, 0xc2, 0x2c, 0x8d, 0x13, 0x09, 0x13,
                0x1f, 0xc0, 0x0f, 0xe7, 0xf2, 0x35, 0x73, 0x42, 0x76, 0xd3, 0x8d, 0x47, 0xf1, 0xe1,
                0x91, 0xe0, 0x0c, 0x7a, 0x1d, 0x48, 0xaf, 0x04, 0x68, 0x27, 0x59, 0x1e, 0x97, 0x33,
                0xa9, 0x7f, 0xa6, 0xb6, 0x79, 0xf3, 0xdc, 0x60, 0x1d, 0x00, 0x82, 0x85, 0xed, 0xcb,
                0xda, 0xe6, 0x9c, 0xe8, 0xfc, 0x1b, 0xe4, 0xaa, 0xc0, 0x0f, 0xf2, 0x71, 0x1e, 0xbd,
                0x93, 0x1d, 0xe5, 0x18, 0x85, 0x68, 0x78, 0xf7, 0x34, 0x76, 0xf2, 0x1a, 0x48, 0x2e,
                0xc9, 0x37, 0x83, 0x65, 0xc8, 0xf7, 0x39, 0x3c, 0x94, 0xe2, 0x88, 0x53, 0x15, 0xeb,
                0x46, 0x71, 0x09, 0x8b, 0x79, 0x53, 0x5e, 0x79, 0x0f, 0xe5, 0x3e, 0x29, 0xfe, 0xf2,
                0xb3, 0x76, 0x66, 0x97, 0xac, 0x32, 0xb4, 0xf4, 0x73, 0xf4, 0x68, 0xa0, 0x08, 0xe7,
                0x23, 0x89, 0xfc, 0x03, 0x88, 0x0d, 0x78, 0x0c, 0xb0, 0x7f, 0xcf, 0xaa, 0xbe, 0x3f,
                0x1a, 0x84, 0xb2, 0x7d, 0xb5, 0x9a, 0x4a, 0x15, 0x3d, 0x88, 0x2d, 0x2b, 0x21, 0x03,
                0x59, 0x65, 0x55, 0xed, 0x94, 0x94, 0xc6, 0xac, 0x89, 0x3c, 0x49, 0x72, 0x38, 0x33,
                0xec, 0x89, 0x26, 0xc1, 0x03, 0x95, 0x86, 0xa7, 0xaf, 0xcf, 0x4a, 0x0d, 0x9c, 0x73,
                0x1e, 0x98, 0x5d, 0x99, 0x58, 0x9c, 0x8b, 0xb8, 0x38, 0xe8, 0xaa, 0xf7, 0x45, 0x53,
                0x3e, 0xd9, 0xe8, 0xae, 0x3a, 0x1c, 0xd0, 0x74, 0xa5, 0x1a, 0x20, 0xda, 0x8a, 0xba,
                0x18, 0xd1, 0xdb, 0xeb, 0xbc, 0x86, 0x2d, 0xed, 0x42, 0x43, 0x5e, 0x92, 0x47, 0x69,
                0x30, 0xd0, 0x69, 0x89, 0x6c, 0xff, 0x30, 0xeb, 0x41, 0x4f, 0x72, 0x7b, 0x89, 0xe0,
                0x01, 0xaf, 0xa2, 0xfb, 0x8d, 0xc3, 0x43, 0x6d, 0x75, 0xa4, 0xa6, 0xf2, 0x65, 0x72,
                0x50, 0x4b, 0x19, 0x22, 0x32, 0xec, 0xb9, 0xf0, 0xc0, 0x24, 0x11, 0xe5, 0x25, 0x96,
                0xbc, 0x5e, 0x90, 0x45, 0x7e, 0x74, 0x59, 0x39, 0xff, 0xed, 0xbd, 0x12, 0x86, 0x3c,
                0xe7, 0x1a, 0x02, 0xaf, 0x11, 0x7d, 0x41, 0x7a, 0xdb, 0x3d, 0x15, 0xcc, 0x54, 0xdc,
                0xb1, 0xfc, 0xe4, 0x67, 0x50, 0x0c, 0x6b, 0x8f, 0xb8, 0x6b, 0x12, 0xb5, 0x6d, 0xa9,
                0xc3, 0x82, 0x85, 0x7d, 0xee, 0xcc, 0x40, 0xa9,
            ],
            cv_net: [
                0xdd, 0xba, 0x24, 0xf3, 0x9f, 0x70, 0x8e, 0xd7, 0xa7, 0x48, 0x57, 0x13, 0x71, 0x11,
                0x42, 0xc2, 0x38, 0x51, 0x38, 0x15, 0x30, 0x2d, 0xf0, 0xf4, 0x83, 0x04, 0x21, 0xa6,
                0xc1, 0x3e, 0x71, 0x01,
            ],
            rho: [
                0xc5, 0x96, 0xfb, 0xd3, 0x2e, 0xbb, 0xcb, 0xad, 0xae, 0x60, 0xd2, 0x85, 0xc7, 0xd7,
                0x5f, 0xa8, 0x36, 0xf9, 0xd2, 0xfa, 0x86, 0x10, 0x0a, 0xb8, 0x58, 0xea, 0x2d, 0xe1,
                0xf1, 0x1c, 0x83, 0x06,
            ],
            cmx: [
                0xa5, 0x70, 0x6f, 0x3d, 0x1b, 0x68, 0x8e, 0x9d, 0xc6, 0x34, 0xee, 0xe4, 0xe6, 0x5b,
                0x02, 0x8a, 0x43, 0xee, 0xae, 0xd2, 0x43, 0x5b, 0xea, 0x2a, 0xe3, 0xd5, 0x16, 0x05,
                0x75, 0xc1, 0x1a, 0x3b,
            ],
            esk: [
                0x56, 0x66, 0x9d, 0x64, 0x3f, 0x78, 0x0b, 0x6a, 0xd8, 0xb3, 0xd3, 0x5a, 0xd7, 0x46,
                0x8a, 0xaa, 0x73, 0x27, 0x66, 0x57, 0x5f, 0x84, 0xa9, 0x5d, 0x20, 0xa6, 0x25, 0xff,
                0x38, 0x77, 0xea, 0x3f,
            ],
            ephemeral_key: [
                0xad, 0xdb, 0x47, 0xb6, 0xac, 0x5d, 0xfc, 0x16, 0x55, 0x89, 0x23, 0xd3, 0xa8, 0xf3,
                0x76, 0x09, 0x5c, 0x69, 0x5c, 0x04, 0x7c, 0x4e, 0x32, 0x66, 0xae, 0x67, 0x69, 0x87,
                0xf7, 0xe3, 0x13, 0x81,
            ],
            shared_secret: [
                0x30, 0x3c, 0x1c, 0x3f, 0x2b, 0xcb, 0xb9, 0xd8, 0x49, 0x70, 0x15, 0xa6, 0xdf, 0xca,
                0x95, 0x4e, 0xce, 0x0d, 0x3b, 0x6c, 0xf1, 0x0a, 0xb9, 0xf7, 0x19, 0xeb, 0x89, 0x19,
                0x9e, 0xdf, 0xe9, 0x89,
            ],
            k_enc: [
                0xef, 0x9d, 0x62, 0x25, 0x68, 0x7c, 0x2c, 0x91, 0x7b, 0x5c, 0xad, 0xa1, 0x75, 0x23,
                0xe8, 0xfb, 0xdb, 0x08, 0x82, 0x25, 0xaf, 0x2d, 0xb5, 0x72, 0xbb, 0x40, 0x0e, 0x44,
                0x8b, 0x5e, 0xa8, 0x71,
            ],
            p_enc: [
                0x02, 0x56, 0xe8, 0x4b, 0x1a, 0xdc, 0x94, 0x23, 0xc3, 0x67, 0x6c, 0x04, 0x8d, 0x5f,
                0x29, 0x35, 0x39, 0x5e, 0xe4, 0x76, 0xbf, 0x69, 0xb8, 0x25, 0x0c, 0x18, 0xef, 0x41,
                0x29, 0x4c, 0xa9, 0x79, 0x93, 0xdb, 0x54, 0x6c, 0x1f, 0xe0, 0x1f, 0x7e, 0x9c, 0x8e,
                0x36, 0xd6, 0xa5, 0xe2, 0x9d, 0x4e, 0x30, 0xa7, 0x35, 0x94, 0xff, 0xbf, 0x50, 0x98,
                0x42, 0x1c, 0x69, 0x37, 0x8a, 0xf1, 0xe4, 0x0f, 0x64, 0xe1, 0x25, 0x94, 0x6f, 0x62,
                0xc2, 0xfa, 0x7b, 0x2f, 0xec, 0xbc, 0xb6, 0x4b, 0x69, 0x68, 0x91, 0x2a, 0x63, 0x81,
                0xce, 0x3d, 0xc1, 0x66, 0xd5, 0x6a, 0x1d, 0x62, 0xf5, 0xa8, 0xd7, 0x55, 0x1d, 0xb5,
                0xfd, 0x93, 0x13, 0xe8, 0xc7, 0x20, 0x3d, 0x99, 0x6a, 0xf7, 0xd4, 0x77, 0x08, 0x37,
                0x56, 0xd5, 0x9a, 0xf8, 0x0d, 0x06, 0xa7, 0x45, 0xf4, 0x4a, 0xb0, 0x23, 0x75, 0x2c,
                0xb5, 0xb4, 0x06, 0xed, 0x89, 0x85, 0xe1, 0x81, 0x30, 0xab, 0x33, 0x36, 0x26, 0x97,
                0xb0, 0xe4, 0xe4, 0xc7, 0x63, 0xcc, 0xb8, 0xf6, 0x76, 0x49, 0x5c, 0x22, 0x2f, 0x7f,
                0xba, 0x1e, 0x31, 0xde, 0xfa, 0x3d, 0x5a, 0x57, 0xef, 0xc2, 0xe1, 0xe9, 0xb0, 0x1a,
                0x03, 0x55, 0x87, 0xd5, 0xfb, 0x1a, 0x38, 0xe0, 0x1d, 0x94, 0x90, 0x3d, 0x3c, 0x3e,
                0x0a, 0xd3, 0x36, 0x0c, 0x1d, 0x37, 0x10, 0xac, 0xd2, 0x0b, 0x18, 0x3e, 0x31, 0xd4,
                0x9f, 0x25, 0xc9, 0xa1, 0x38, 0xf4, 0x9b, 0x1a, 0x53, 0x7e, 0xdc, 0xf0, 0x4b, 0xe3,
                0x4a, 0x98, 0x51, 0xa7, 0xaf, 0x9d, 0xb6, 0x99, 0x0e, 0xd8, 0x3d, 0xd6, 0x4a, 0xf3,
                0x59, 0x7c, 0x04, 0x32, 0x3e, 0xa5, 0x1b, 0x00, 0x52, 0xad, 0x80, 0x84, 0xa8, 0xb9,
                0xda, 0x94, 0x8d, 0x32, 0x0d, 0xad, 0xd6, 0x4f, 0x54, 0x31, 0xe6, 0x1d, 0xdf, 0x65,
                0x8d, 0x24, 0xae, 0x67, 0xc2, 0x2c, 0x8d, 0x13, 0x09, 0x13, 0x1f, 0xc0, 0x0f, 0xe7,
                0xf2, 0x35, 0x73, 0x42, 0x76, 0xd3, 0x8d, 0x47, 0xf1, 0xe1, 0x91, 0xe0, 0x0c, 0x7a,
                0x1d, 0x48, 0xaf, 0x04, 0x68, 0x27, 0x59, 0x1e, 0x97, 0x33, 0xa9, 0x7f, 0xa6, 0xb6,
                0x79, 0xf3, 0xdc, 0x60, 0x1d, 0x00, 0x82, 0x85, 0xed, 0xcb, 0xda, 0xe6, 0x9c, 0xe8,
                0xfc, 0x1b, 0xe4, 0xaa, 0xc0, 0x0f, 0xf2, 0x71, 0x1e, 0xbd, 0x93, 0x1d, 0xe5, 0x18,
                0x85, 0x68, 0x78, 0xf7, 0x34, 0x76, 0xf2, 0x1a, 0x48, 0x2e, 0xc9, 0x37, 0x83, 0x65,
                0xc8, 0xf7, 0x39, 0x3c, 0x94, 0xe2, 0x88, 0x53, 0x15, 0xeb, 0x46, 0x71, 0x09, 0x8b,
                0x79, 0x53, 0x5e, 0x79, 0x0f, 0xe5, 0x3e, 0x29, 0xfe, 0xf2, 0xb3, 0x76, 0x66, 0x97,
                0xac, 0x32, 0xb4, 0xf4, 0x73, 0xf4, 0x68, 0xa0, 0x08, 0xe7, 0x23, 0x89, 0xfc, 0x03,
                0x88, 0x0d, 0x78, 0x0c, 0xb0, 0x7f, 0xcf, 0xaa, 0xbe, 0x3f, 0x1a, 0x84, 0xb2, 0x7d,
                0xb5, 0x9a, 0x4a, 0x15, 0x3d, 0x88, 0x2d, 0x2b, 0x21, 0x03, 0x59, 0x65, 0x55, 0xed,
                0x94, 0x94, 0xc6, 0xac, 0x89, 0x3c, 0x49, 0x72, 0x38, 0x33, 0xec, 0x89, 0x26, 0xc1,
                0x03, 0x95, 0x86, 0xa7, 0xaf, 0xcf, 0x4a, 0x0d, 0x9c, 0x73, 0x1e, 0x98, 0x5d, 0x99,
                0x58, 0x9c, 0x8b, 0xb8, 0x38, 0xe8, 0xaa, 0xf7, 0x45, 0x53, 0x3e, 0xd9, 0xe8, 0xae,
                0x3a, 0x1c, 0xd0, 0x74, 0xa5, 0x1a, 0x20, 0xda, 0x8a, 0xba, 0x18, 0xd1, 0xdb, 0xeb,
                0xbc, 0x86, 0x2d, 0xed, 0x42, 0x43, 0x5e, 0x92, 0x47, 0x69, 0x30, 0xd0, 0x69, 0x89,
                0x6c, 0xff, 0x30, 0xeb, 0x41, 0x4f, 0x72, 0x7b, 0x89, 0xe0, 0x01, 0xaf, 0xa2, 0xfb,
                0x8d, 0xc3, 0x43, 0x6d, 0x75, 0xa4, 0xa6, 0xf2, 0x65, 0x72, 0x50, 0x4b, 0x19, 0x22,
                0x32, 0xec, 0xb9, 0xf0, 0xc0, 0x24, 0x11, 0xe5, 0x25, 0x96, 0xbc, 0x5e, 0x90, 0x45,
                0x7e, 0x74, 0x59, 0x39, 0xff, 0xed, 0xbd, 0x12, 0x86, 0x3c, 0xe7, 0x1a, 0x02, 0xaf,
                0x11, 0x7d, 0x41, 0x7a, 0xdb, 0x3d, 0x15, 0xcc, 0x54, 0xdc, 0xb1, 0xfc, 0xe4, 0x67,
                0x50, 0x0c, 0x6b, 0x8f, 0xb8, 0x6b, 0x12, 0xb5, 0x6d, 0xa9, 0xc3, 0x82, 0x85, 0x7d,
                0xee, 0xcc, 0x40, 0xa9,
            ],
            c_enc: [
                0x1a, 0x9a, 0xdb, 0x14, 0x24, 0x98, 0xe3, 0xdc, 0xc7, 0x6f, 0xed, 0x77, 0x86, 0x14,
                0xdd, 0x31, 0x6c, 0x02, 0xfb, 0xb8, 0xba, 0x92, 0x44, 0xae, 0x4c, 0x2e, 0x32, 0xa0,
                0x7d, 0xae, 0xec, 0xa4, 0x12, 0x26, 0xb9, 0x8b, 0xfe, 0x74, 0xf9, 0xfc, 0xb2, 0x28,
                0xcf, 0xc1, 0x00, 0xf3, 0x18, 0x0f, 0x57, 0x75, 0xec, 0xe3, 0x8b, 0xe7, 0xed, 0x45,
                0xd9, 0x40, 0x21, 0xf4, 0x40, 0x1b, 0x2a, 0x4d, 0x75, 0x82, 0xb4, 0x28, 0xd4, 0x9e,
                0xc7, 0xf5, 0xb5, 0xa4, 0x98, 0x97, 0x3e, 0x60, 0xe3, 0x8e, 0x74, 0xf5, 0xc3, 0xe5,
                0x77, 0x82, 0x7c, 0x38, 0x28, 0x57, 0xd8, 0x16, 0x6b, 0x54, 0xe6, 0x4f, 0x66, 0xef,
                0x5c, 0x7e, 0x8c, 0x9b, 0xaa, 0x2a, 0x3f, 0xa9, 0xe3, 0x7d, 0x08, 0x77, 0x17, 0xd5,
                0xe9, 0x6b, 0xc2, 0xf7, 0x3d, 0x03, 0x14, 0x50, 0xdc, 0x24, 0x32, 0xba, 0x49, 0xd8,
                0xb7, 0x4d, 0xb2, 0x13, 0x09, 0x9e, 0xa9, 0xba, 0x04, 0xeb, 0x63, 0xb6, 0x57, 0x4d,
                0x46, 0xc0, 0x3c, 0xe7, 0x90, 0x0d, 0x4a, 0xc4, 0xbb, 0x18, 0x8e, 0xe9, 0x03, 0x0d,
                0x7f, 0x69, 0xc8, 0x95, 0xa9, 0x4f, 0xc1, 0x82, 0xf2, 0x25, 0xa9, 0x4f, 0x0c, 0xde,
                0x1b, 0x49, 0x88, 0x68, 0x71, 0xa3, 0x76, 0x34, 0x1e, 0xa9, 0x41, 0x71, 0xbe, 0xfd,
                0x95, 0xa8, 0x30, 0xfa, 0x18, 0x40, 0x70, 0x97, 0xdc, 0xa5, 0x11, 0x02, 0x54, 0x63,
                0xd4, 0x37, 0xe9, 0x69, 0x5c, 0xaa, 0x07, 0x9a, 0x2f, 0x68, 0xcd, 0xc7, 0xf2, 0xc1,
                0x32, 0x67, 0xbf, 0xf4, 0x19, 0x51, 0x37, 0xfa, 0x89, 0x53, 0x25, 0x2a, 0x81, 0xb2,
                0xaf, 0xa1, 0x58, 0x2b, 0x9b, 0xfb, 0x4a, 0xc9, 0x60, 0x37, 0xed, 0x29, 0x91, 0xd3,
                0xcb, 0xc7, 0xd5, 0x4a, 0xff, 0x6e, 0x62, 0x1b, 0x06, 0xa7, 0xb2, 0xb9, 0xca, 0xf2,
                0x95, 0x5e, 0xfa, 0xf4, 0xea, 0x8e, 0xfc, 0xfd, 0x02, 0x3a, 0x3c, 0x17, 0x48, 0xdf,
                0x3c, 0xbd, 0x43, 0xe0, 0xb9, 0xa8, 0xb0, 0x94, 0x56, 0x88, 0xd5, 0x20, 0x56, 0xc1,
                0xd1, 0x6e, 0xea, 0x37, 0xe7, 0x98, 0xba, 0x31, 0xdc, 0x3e, 0x5d, 0x49, 0x52, 0xbd,
                0x51, 0xec, 0x76, 0x9d, 0x57, 0x88, 0xb6, 0xe3, 0x5f, 0xe9, 0x04, 0x2b, 0x95, 0xd4,
                0xd2, 0x17, 0x81, 0x40, 0x0e, 0xaf, 0xf5, 0x86, 0x16, 0xad, 0x56, 0x27, 0x96, 0x63,
                0x6a, 0x50, 0xb8, 0xed, 0x6c, 0x7f, 0x98, 0x1d, 0xc7, 0xba, 0x81, 0x4e, 0xff, 0x15,
                0x2c, 0xb2, 0x28, 0xa2, 0xea, 0xd2, 0xf8, 0x32, 0x66, 0x2f, 0xa4, 0xa4, 0xa5, 0x07,
                0x97, 0xb0, 0xf8, 0x5b, 0x62, 0xd0, 0x8b, 0x1d, 0xd2, 0xd8, 0xe4, 0x3b, 0x4a, 0x5b,
                0xfb, 0xb1, 0x59, 0xed, 0x57, 0x8e, 0xf7, 0x47, 0x5d, 0xe0, 0xad, 0xa1, 0x3e, 0x17,
                0xad, 0x87, 0xcc, 0x23, 0x05, 0x67, 0x2b, 0xcc, 0x55, 0xa8, 0x88, 0x13, 0x17, 0xfd,
                0xc1, 0xbf, 0xc4, 0x59, 0xb6, 0x8b, 0x2d, 0xf7, 0x0c, 0xad, 0x37, 0x70, 0xed, 0x0f,
                0xd0, 0x2d, 0x64, 0xb9, 0x6f, 0x2b, 0xbf, 0x6f, 0x8f, 0x63, 0x2e, 0x86, 0x6c, 0xa5,
                0xd1, 0x96, 0xd2, 0x48, 0xad, 0x05, 0xc3, 0xde, 0x64, 0x41, 0x48, 0xa8, 0x0b, 0x51,
                0xad, 0xa9, 0x5b, 0xd0, 0x8d, 0x73, 0xcd, 0xbb, 0x45, 0x26, 0x4f, 0x3b, 0xd1, 0x13,
                0x83, 0x5b, 0x46, 0xf9, 0xbe, 0x7b, 0x6d, 0x23, 0xa4, 0x3b, 0xdd, 0xfe, 0x1e, 0x74,
                0x08, 0xc9, 0x70, 0x31, 0xe1, 0xa8, 0x21, 0x4b, 0xab, 0x46, 0x39, 0x10, 0x44, 0xb7,
                0x00, 0xd3, 0x8f, 0x51, 0x92, 0xc5, 0x7f, 0xe6, 0xf8, 0x71, 0x59, 0xb5, 0x55, 0x12,
                0x09, 0x4e, 0x29, 0xd2, 0xce, 0xba, 0xb8, 0x68, 0xc8, 0xf1, 0xad, 0xba, 0xd5, 0x70,
                0x77, 0xcb, 0xeb, 0x5e, 0x69, 0x65, 0x85, 0x82, 0xbf, 0x98, 0xd1, 0x9d, 0x64, 0xf4,
                0x4b, 0x0d, 0x50, 0xc7, 0xe2, 0x20, 0x9a, 0xb3, 0xfc, 0x56, 0xb4, 0xf4, 0x09, 0x12,
                0x3a, 0xae, 0xb0, 0x26, 0x3a, 0x22, 0x45, 0x1b, 0xc1, 0x4e, 0xd7, 0x56, 0xd0, 0x48,
                0x38, 0x5a, 0xed, 0xbb, 0x86, 0xa8, 0x46, 0x77, 0xbb, 0x2d, 0x21, 0xc5, 0x2c, 0xc9,
                0x49, 0x41, 0x47, 0xbf, 0x0f, 0xb1, 0x02, 0x74, 0x52, 0x82, 0x99, 0x09, 0x09, 0x72,
                0x62, 0x28, 0x18, 0x6e, 0x02, 0xc8,
            ],
            ock: [
                0x4e, 0x9d, 0x45, 0x94, 0x6b, 0x3e, 0xea, 0xe7, 0xfe, 0x30, 0x5d, 0x5b, 0x90, 0x50,
                0x36, 0x14, 0x1f, 0x9f, 0x40, 0x09, 0xa6, 0x29, 0x4b, 0x96, 0xc7, 0x22, 0xa4, 0xa0,
                0xbe, 0x68, 0x5d, 0xff,
            ],
            op: [
                0x63, 0xf7, 0x12, 0x5d, 0xf4, 0x83, 0x6f, 0xd2, 0x81, 0x6b, 0x02, 0x4e, 0xe7, 0x0e,
                0xfe, 0x09, 0xfb, 0x9a, 0x7b, 0x38, 0x63, 0xc6, 0xea, 0xcd, 0xf9, 0x5e, 0x03, 0x89,
                0x49, 0x50, 0x69, 0x2c, 0x56, 0x66, 0x9d, 0x64, 0x3f, 0x78, 0x0b, 0x6a, 0xd8, 0xb3,
                0xd3, 0x5a, 0xd7, 0x46, 0x8a, 0xaa, 0x73, 0x27, 0x66, 0x57, 0x5f, 0x84, 0xa9, 0x5d,
                0x20, 0xa6, 0x25, 0xff, 0x38, 0x77, 0xea, 0x3f,
            ],
            c_out: [
                0xcb, 0xdf, 0x68, 0xa5, 0x7f, 0xb4, 0xa4, 0x6f, 0x34, 0x60, 0xff, 0x22, 0x7b, 0xc6,
                0x18, 0xda, 0xe1, 0x12, 0x29, 0x45, 0xb3, 0x80, 0xc7, 0xe5, 0x49, 0xcf, 0x4a, 0x6e,
                0x8b, 0xf3, 0x75, 0x49, 0xba, 0xe1, 0x89, 0x1f, 0xd8, 0xd1, 0xa4, 0x94, 0x4f, 0xdf,
                0x41, 0x0f, 0x07, 0x02, 0xed, 0xa5, 0x44, 0x2f, 0x0e, 0xa0, 0x1a, 0x5d, 0xf0, 0x12,
                0xa0, 0xae, 0x4d, 0x84, 0xed, 0x79, 0x80, 0x33, 0x28, 0xbd, 0x1f, 0xd5, 0xfa, 0xc7,
                0x19, 0x21, 0x6a, 0x77, 0x6d, 0xe6, 0x4f, 0xd1, 0x67, 0xdb,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0xfd, 0x9e, 0x9a, 0x1f, 0x38, 0x1c, 0xbe, 0x75, 0xcd, 0x8d, 0x6a, 0xe1, 0x2f, 0xca,
                0x87, 0x2e, 0x94, 0x00, 0xf0, 0x02, 0x72, 0xb0, 0x29, 0x65, 0x2e, 0x65, 0x6c, 0x8f,
                0x3c, 0x4b, 0xf0, 0x37, 0xee, 0xef, 0x96, 0x42, 0x1b, 0x2f, 0xab, 0x2f, 0xb3, 0xad,
                0x1e, 0x0a, 0xd8, 0x50, 0x2d, 0x74, 0xe6, 0xf0, 0x8f, 0x0d, 0xd5, 0x18, 0xf8, 0xfa,
                0x82, 0x2a, 0x65, 0xbe, 0x27, 0x40, 0xc0, 0x21,
            ],
            ovk: [
                0xe7, 0x30, 0x81, 0xef, 0x8d, 0x62, 0xcb, 0x78, 0x0a, 0xb6, 0x88, 0x3a, 0x50, 0xa0,
                0xd4, 0x70, 0x19, 0x0d, 0xfb, 0xa1, 0x0a, 0x85, 0x7f, 0x82, 0x84, 0x2d, 0x38, 0x25,
                0xb3, 0xd6, 0xda, 0x05,
            ],
            default_d: [
                0x55, 0x6e, 0x5e, 0x1b, 0xf5, 0x1b, 0xc6, 0xa6, 0x11, 0x58, 0xf7,
            ],
            default_pk_d: [
                0xb4, 0xca, 0xc5, 0x6f, 0x06, 0x2b, 0xfb, 0x2e, 0x27, 0x15, 0xea, 0xf9, 0xc8, 0xfc,
                0xdb, 0xc2, 0x0c, 0x86, 0x79, 0x3f, 0x23, 0x57, 0xdd, 0xd0, 0x4a, 0xad, 0x39, 0xf9,
                0x4a, 0xd7, 0xc7, 0x84,
            ],
            v: 9072946746592546880,
            rseed: [
                0xae, 0xab, 0x01, 0x6b, 0x6b, 0xc1, 0xec, 0x14, 0x4b, 0x4e, 0x55, 0x3a, 0xcf, 0xd6,
                0x70, 0xf7, 0x7e, 0x75, 0x5f, 0xc8, 0x8e, 0x06, 0x77, 0xe3, 0x1b, 0xa4, 0x59, 0xb4,
                0x4e, 0x30, 0x77, 0x68,
            ],
            memo: [
                0xff, 0x95, 0x8f, 0xe3, 0x78, 0x9d, 0x41, 0xc2, 0xb1, 0xff, 0x43, 0x4c, 0xb3, 0x0e,
                0x15, 0x91, 0x4f, 0x01, 0xbc, 0x6b, 0xc2, 0x30, 0x7b, 0x48, 0x8d, 0x25, 0x56, 0xd7,
                0xb7, 0x38, 0x0e, 0xa4, 0xff, 0xd7, 0x12, 0xf6, 0xb0, 0x2f, 0xe8, 0x06, 0xb9, 0x45,
                0x69, 0xcd, 0x40, 0x59, 0xf3, 0x96, 0xbf, 0x29, 0xb9, 0x9d, 0x0a, 0x40, 0xe5, 0xe1,
                0x71, 0x1c, 0xa9, 0x44, 0xf7, 0x2d, 0x43, 0x6a, 0x10, 0x2f, 0xca, 0x4b, 0x97, 0x69,
                0x3d, 0xa0, 0xb0, 0x86, 0xfe, 0x9d, 0x2e, 0x71, 0x62, 0x47, 0x0d, 0x02, 0xe0, 0xf0,
                0x5d, 0x4b, 0xec, 0x95, 0x12, 0xbf, 0xb3, 0xf3, 0x83, 0x27, 0x29, 0x6e, 0xfa, 0xa7,
                0x43, 0x28, 0xb1, 0x18, 0xc2, 0x74, 0x02, 0xc7, 0x0c, 0x3a, 0x90, 0xb4, 0x9a, 0xd4,
                0xbb, 0xc6, 0x8e, 0x37, 0xc0, 0xaa, 0x7d, 0x9b, 0x3f, 0xe1, 0x77, 0x99, 0xd7, 0x3b,
                0x84, 0x1e, 0x75, 0x17, 0x13, 0xa0, 0x29, 0x43, 0x90, 0x5a, 0xae, 0x08, 0x03, 0xfd,
                0x69, 0x44, 0x2e, 0xb7, 0x68, 0x1e, 0xc2, 0xa0, 0x56, 0x00, 0x05, 0x4e, 0x92, 0xee,
                0xd5, 0x55, 0x02, 0x8f, 0x21, 0xb6, 0xa1, 0x55, 0x26, 0x8a, 0x2d, 0xd6, 0x64, 0x0a,
                0x69, 0x30, 0x1a, 0x52, 0xa3, 0x8d, 0x4d, 0x9f, 0x9f, 0x95, 0x7a, 0xe3, 0x5a, 0xf7,
                0x16, 0x71, 0x18, 0x14, 0x1c, 0xe4, 0xc9, 0xbe, 0x0a, 0x6a, 0x49, 0x2f, 0xe7, 0x9f,
                0x15, 0x81, 0xa1, 0x55, 0xfa, 0x3a, 0x2b, 0x9d, 0xaf, 0xd8, 0x2e, 0x65, 0x0b, 0x38,
                0x6a, 0xd3, 0xa0, 0x8c, 0xb6, 0xb8, 0x31, 0x31, 0xac, 0x30, 0x0b, 0x08, 0x46, 0x35,
                0x4a, 0x7e, 0xef, 0x9c, 0x41, 0x0e, 0x4b, 0x62, 0xc4, 0x7c, 0x54, 0x26, 0x90, 0x7d,
                0xfc, 0x66, 0x85, 0xc5, 0xc9, 0x9b, 0x71, 0x41, 0xac, 0x62, 0x6a, 0xb4, 0x76, 0x1f,
                0xd3, 0xf4, 0x1e, 0x72, 0x8e, 0x1a, 0x28, 0xf8, 0x9d, 0xb8, 0x9f, 0xfd, 0xec, 0xa3,
                0x64, 0xdd, 0x2f, 0x0f, 0x07, 0x39, 0xf0, 0x53, 0x45, 0x56, 0x48, 0x31, 0x99, 0xc7,
                0x1f, 0x18, 0x93, 0x41, 0xac, 0x9b, 0x78, 0xa2, 0x69, 0x16, 0x42, 0x06, 0xa0, 0xea,
                0x1c, 0xe7, 0x3b, 0xfb, 0x2a, 0x94, 0x2e, 0x73, 0x70, 0xb2, 0x47, 0xc0, 0x46, 0xf8,
                0xe7, 0x5e, 0xf8, 0xe3, 0xf8, 0xbd, 0x82, 0x1c, 0xf5, 0x77, 0x49, 0x18, 0x64, 0xe2,
                0x0e, 0x6d, 0x08, 0xfd, 0x2e, 0x32, 0xb5, 0x55, 0xc9, 0x2c, 0x66, 0x1f, 0x19, 0x58,
                0x8b, 0x72, 0xa8, 0x95, 0x99, 0x71, 0x0a, 0x88, 0x06, 0x12, 0x53, 0xca, 0x28, 0x5b,
                0x63, 0x04, 0xb3, 0x7d, 0xa2, 0xb5, 0x29, 0x4f, 0x5c, 0xb3, 0x54, 0xa8, 0x94, 0x32,
                0x28, 0x48, 0xcc, 0xbd, 0xc7, 0xc2, 0x54, 0x5b, 0x7d, 0xa5, 0x68, 0xaf, 0xac, 0x87,
                0xff, 0xa0, 0x05, 0xc3, 0x12, 0x24, 0x1c, 0x2d, 0x57, 0xf4, 0xb4, 0x5d, 0x64, 0x19,
                0xf0, 0xd2, 0xe2, 0xc5, 0xaf, 0x33, 0xae, 0x24, 0x37, 0x85, 0xb3, 0x25, 0xcd, 0xab,
                0x95, 0x40, 0x4f, 0xc7, 0xae, 0xd7, 0x05, 0x25, 0xcd, 0xdb, 0x41, 0x87, 0x2c, 0xfc,
                0xc2, 0x14, 0xb1, 0x32, 0x32, 0xed, 0xc7, 0x86, 0x09, 0x75, 0x3d, 0xbf, 0xf9, 0x30,
                0xeb, 0x0d, 0xc1, 0x56, 0x61, 0x2b, 0x9c, 0xb4, 0x34, 0xbc, 0x4b, 0x69, 0x33, 0x92,
                0xde, 0xb8, 0x7c, 0x53, 0x04, 0x35, 0x31, 0x2e, 0xdc, 0xed, 0xc6, 0xa9, 0x61, 0x13,
                0x33, 0x38, 0xd7, 0x86, 0xc4, 0xa3, 0xe1, 0x03, 0xf6, 0x01, 0x10, 0xa1, 0x6b, 0x13,
                0x37, 0x12, 0x97, 0x04, 0xbf, 0x47, 0x54, 0xff, 0x6b, 0xa9, 0xfb, 0xe6, 0x59, 0x51,
                0xe6, 0x10, 0x62, 0x0f, 0x71, 0xcd, 0xa8, 0xfc, 0x87, 0x76, 0x25, 0xf2, 0xc5, 0xbb,
                0x04, 0xcb, 0xe1, 0x22, 0x8b, 0x1e, 0x88, 0x6f,
            ],
            cv_net: [
                0x15, 0x49, 0x70, 0x7e, 0x1e, 0xd2, 0xb2, 0xeb, 0x66, 0x15, 0x65, 0x0b, 0xec, 0x45,
                0xa2, 0x17, 0x64, 0x10, 0x4a, 0x23, 0xea, 0xf6, 0xba, 0x49, 0x6c, 0xb9, 0xb8, 0xe8,
                0x25, 0x7a, 0xd8, 0xb3,
            ],
            rho: [
                0x33, 0x88, 0xda, 0x05, 0x06, 0xda, 0x9e, 0xa2, 0xd5, 0x16, 0x73, 0x9b, 0x95, 0x1c,
                0x7c, 0xc0, 0x58, 0x53, 0x36, 0xb4, 0x4d, 0xf9, 0xb3, 0xb5, 0x0e, 0x48, 0x93, 0xe4,
                0xb1, 0x84, 0x92, 0x11,
            ],
            cmx: [
                0x9e, 0x04, 0x32, 0xb2, 0xb3, 0x33, 0xcd, 0xe8, 0xce, 0x92, 0x1b, 0x77, 0xca, 0x7e,
                0x9e, 0x41, 0x51, 0xe3, 0x74, 0xd5, 0x16, 0xcd, 0xa1, 0x17, 0x63, 0x83, 0x6a, 0xf3,
                0xb6, 0x6f, 0x5b, 0x15,
            ],
            esk: [
                0x68, 0x65, 0x87, 0xce, 0x53, 0xc3, 0x39, 0xf9, 0xce, 0xcf, 0x4d, 0x80, 0x4a, 0x17,
                0x09, 0x39, 0x2b, 0x6a, 0xb1, 0x08, 0xea, 0x2c, 0x57, 0x79, 0x92, 0x1c, 0xd5, 0xda,
                0x8a, 0x6e, 0x1a, 0x08,
            ],
            ephemeral_key: [
                0x91, 0x92, 0x3e, 0xd8, 0x2b, 0x76, 0xd7, 0x97, 0x30, 0x7c, 0xaa, 0x23, 0x02, 0xc0,
                0xcf, 0x75, 0x56, 0x12, 0x17, 0x24, 0x98, 0x67, 0x53, 0x2a, 0xe5, 0x1c, 0x2e, 0xa0,
                0x05, 0xed, 0xad, 0xb6,
            ],
            shared_secret: [
                0x53, 0xd7, 0xe4, 0x84, 0x3a, 0x36, 0xd5, 0x79, 0xb5, 0xa7, 0xc1, 0x04, 0x11, 0x96,
                0xbd, 0x4c, 0x85, 0x80, 0x5c, 0xcd, 0x0a, 0x3f, 0x95, 0xbc, 0x9e, 0x71, 0x06, 0x90,
                0xd1, 0x1b, 0x7a, 0xaa,
            ],
            k_enc: [
                0xc2, 0xe3, 0x38, 0x9d, 0x6f, 0xc3, 0xcd, 0x06, 0x7f, 0x59, 0x0a, 0x93, 0x73, 0x05,
                0x04, 0xad, 0x9b, 0x63, 0xc4, 0x55, 0x45, 0x69, 0xb9, 0x62, 0x0f, 0x3e, 0xf0, 0xb0,
                0x65, 0x94, 0xde, 0xd6,
            ],
            p_enc: [
                0x02, 0x55, 0x6e, 0x5e, 0x1b, 0xf5, 0x1b, 0xc6, 0xa6, 0x11, 0x58, 0xf7, 0x40, 0x50,
                0xaf, 0xd8, 0xfe, 0x94, 0xe9, 0x7d, 0xae, 0xab, 0x01, 0x6b, 0x6b, 0xc1, 0xec, 0x14,
                0x4b, 0x4e, 0x55, 0x3a, 0xcf, 0xd6, 0x70, 0xf7, 0x7e, 0x75, 0x5f, 0xc8, 0x8e, 0x06,
                0x77, 0xe3, 0x1b, 0xa4, 0x59, 0xb4, 0x4e, 0x30, 0x77, 0x68, 0xff, 0x95, 0x8f, 0xe3,
                0x78, 0x9d, 0x41, 0xc2, 0xb1, 0xff, 0x43, 0x4c, 0xb3, 0x0e, 0x15, 0x91, 0x4f, 0x01,
                0xbc, 0x6b, 0xc2, 0x30, 0x7b, 0x48, 0x8d, 0x25, 0x56, 0xd7, 0xb7, 0x38, 0x0e, 0xa4,
                0xff, 0xd7, 0x12, 0xf6, 0xb0, 0x2f, 0xe8, 0x06, 0xb9, 0x45, 0x69, 0xcd, 0x40, 0x59,
                0xf3, 0x96, 0xbf, 0x29, 0xb9, 0x9d, 0x0a, 0x40, 0xe5, 0xe1, 0x71, 0x1c, 0xa9, 0x44,
                0xf7, 0x2d, 0x43, 0x6a, 0x10, 0x2f, 0xca, 0x4b, 0x97, 0x69, 0x3d, 0xa0, 0xb0, 0x86,
                0xfe, 0x9d, 0x2e, 0x71, 0x62, 0x47, 0x0d, 0x02, 0xe0, 0xf0, 0x5d, 0x4b, 0xec, 0x95,
                0x12, 0xbf, 0xb3, 0xf3, 0x83, 0x27, 0x29, 0x6e, 0xfa, 0xa7, 0x43, 0x28, 0xb1, 0x18,
                0xc2, 0x74, 0x02, 0xc7, 0x0c, 0x3a, 0x90, 0xb4, 0x9a, 0xd4, 0xbb, 0xc6, 0x8e, 0x37,
                0xc0, 0xaa, 0x7d, 0x9b, 0x3f, 0xe1, 0x77, 0x99, 0xd7, 0x3b, 0x84, 0x1e, 0x75, 0x17,
                0x13, 0xa0, 0x29, 0x43, 0x90, 0x5a, 0xae, 0x08, 0x03, 0xfd, 0x69, 0x44, 0x2e, 0xb7,
                0x68, 0x1e, 0xc2, 0xa0, 0x56, 0x00, 0x05, 0x4e, 0x92, 0xee, 0xd5, 0x55, 0x02, 0x8f,
                0x21, 0xb6, 0xa1, 0x55, 0x26, 0x8a, 0x2d, 0xd6, 0x64, 0x0a, 0x69, 0x30, 0x1a, 0x52,
                0xa3, 0x8d, 0x4d, 0x9f, 0x9f, 0x95, 0x7a, 0xe3, 0x5a, 0xf7, 0x16, 0x71, 0x18, 0x14,
                0x1c, 0xe4, 0xc9, 0xbe, 0x0a, 0x6a, 0x49, 0x2f, 0xe7, 0x9f, 0x15, 0x81, 0xa1, 0x55,
                0xfa, 0x3a, 0x2b, 0x9d, 0xaf, 0xd8, 0x2e, 0x65, 0x0b, 0x38, 0x6a, 0xd3, 0xa0, 0x8c,
                0xb6, 0xb8, 0x31, 0x31, 0xac, 0x30, 0x0b, 0x08, 0x46, 0x35, 0x4a, 0x7e, 0xef, 0x9c,
                0x41, 0x0e, 0x4b, 0x62, 0xc4, 0x7c, 0x54, 0x26, 0x90, 0x7d, 0xfc, 0x66, 0x85, 0xc5,
                0xc9, 0x9b, 0x71, 0x41, 0xac, 0x62, 0x6a, 0xb4, 0x76, 0x1f, 0xd3, 0xf4, 0x1e, 0x72,
                0x8e, 0x1a, 0x28, 0xf8, 0x9d, 0xb8, 0x9f, 0xfd, 0xec, 0xa3, 0x64, 0xdd, 0x2f, 0x0f,
                0x07, 0x39, 0xf0, 0x53, 0x45, 0x56, 0x48, 0x31, 0x99, 0xc7, 0x1f, 0x18, 0x93, 0x41,
                0xac, 0x9b, 0x78, 0xa2, 0x69, 0x16, 0x42, 0x06, 0xa0, 0xea, 0x1c, 0xe7, 0x3b, 0xfb,
                0x2a, 0x94, 0x2e, 0x73, 0x70, 0xb2, 0x47, 0xc0, 0x46, 0xf8, 0xe7, 0x5e, 0xf8, 0xe3,
                0xf8, 0xbd, 0x82, 0x1c, 0xf5, 0x77, 0x49, 0x18, 0x64, 0xe2, 0x0e, 0x6d, 0x08, 0xfd,
                0x2e, 0x32, 0xb5, 0x55, 0xc9, 0x2c, 0x66, 0x1f, 0x19, 0x58, 0x8b, 0x72, 0xa8, 0x95,
                0x99, 0x71, 0x0a, 0x88, 0x06, 0x12, 0x53, 0xca, 0x28, 0x5b, 0x63, 0x04, 0xb3, 0x7d,
                0xa2, 0xb5, 0x29, 0x4f, 0x5c, 0xb3, 0x54, 0xa8, 0x94, 0x32, 0x28, 0x48, 0xcc, 0xbd,
                0xc7, 0xc2, 0x54, 0x5b, 0x7d, 0xa5, 0x68, 0xaf, 0xac, 0x87, 0xff, 0xa0, 0x05, 0xc3,
                0x12, 0x24, 0x1c, 0x2d, 0x57, 0xf4, 0xb4, 0x5d, 0x64, 0x19, 0xf0, 0xd2, 0xe2, 0xc5,
                0xaf, 0x33, 0xae, 0x24, 0x37, 0x85, 0xb3, 0x25, 0xcd, 0xab, 0x95, 0x40, 0x4f, 0xc7,
                0xae, 0xd7, 0x05, 0x25, 0xcd, 0xdb, 0x41, 0x87, 0x2c, 0xfc, 0xc2, 0x14, 0xb1, 0x32,
                0x32, 0xed, 0xc7, 0x86, 0x09, 0x75, 0x3d, 0xbf, 0xf9, 0x30, 0xeb, 0x0d, 0xc1, 0x56,
                0x61, 0x2b, 0x9c, 0xb4, 0x34, 0xbc, 0x4b, 0x69, 0x33, 0x92, 0xde, 0xb8, 0x7c, 0x53,
                0x04, 0x35, 0x31, 0x2e, 0xdc, 0xed, 0xc6, 0xa9, 0x61, 0x13, 0x33, 0x38, 0xd7, 0x86,
                0xc4, 0xa3, 0xe1, 0x03, 0xf6, 0x01, 0x10, 0xa1, 0x6b, 0x13, 0x37, 0x12, 0x97, 0x04,
                0xbf, 0x47, 0x54, 0xff, 0x6b, 0xa9, 0xfb, 0xe6, 0x59, 0x51, 0xe6, 0x10, 0x62, 0x0f,
                0x71, 0xcd, 0xa8, 0xfc, 0x87, 0x76, 0x25, 0xf2, 0xc5, 0xbb, 0x04, 0xcb, 0xe1, 0x22,
                0x8b, 0x1e, 0x88, 0x6f,
            ],
            c_enc: [
                0x6d, 0x21, 0x18, 0x0a, 0xc4, 0x74, 0x94, 0x57, 0x8a, 0x90, 0x06, 0xaf, 0x5b, 0xeb,
                0x4e, 0x34, 0x55, 0x59, 0xab, 0xdb, 0x0e, 0x23, 0xed, 0x52, 0x65, 0xaf, 0x79, 0xce,
                0x0c, 0xc5, 0x96, 0x71, 0xea, 0x31, 0x7b, 0x3e, 0xb1, 0x52, 0x93, 0xd1, 0xe0, 0x4a,
                0x9a, 0xd8, 0x39, 0x69, 0xab, 0x9e, 0x17, 0x08, 0xf2, 0x28, 0x85, 0x3a, 0x28, 0x1f,
                0xcf, 0x6c, 0x97, 0x60, 0xae, 0x71, 0x96, 0x5b, 0xb1, 0xd4, 0x45, 0x2e, 0x5e, 0xd0,
                0x06, 0x00, 0xab, 0x58, 0x94, 0x27, 0x75, 0xb7, 0x4e, 0x12, 0x62, 0xb4, 0x22, 0x5d,
                0x3b, 0x61, 0x9c, 0x31, 0x65, 0x96, 0xd7, 0xcf, 0x9c, 0x93, 0xbd, 0xb5, 0x2a, 0xfa,
                0x77, 0x8a, 0xa1, 0x20, 0x8d, 0x56, 0xd0, 0x69, 0xe5, 0x6e, 0x27, 0x79, 0x61, 0x4d,
                0x56, 0xf4, 0x36, 0x10, 0x72, 0xbc, 0x15, 0x16, 0xa9, 0xb4, 0x56, 0x2a, 0x4f, 0x54,
                0x63, 0xa5, 0x13, 0xc4, 0x59, 0x42, 0x15, 0x70, 0xf3, 0x34, 0xef, 0xb6, 0xbc, 0xd2,
                0x08, 0xf8, 0xf8, 0x05, 0x0e, 0x15, 0x3d, 0x4e, 0x61, 0xf3, 0x1e, 0xdd, 0xbb, 0x5a,
                0x98, 0xf1, 0x70, 0xd3, 0xd0, 0x80, 0xe8, 0xec, 0x3f, 0x65, 0x20, 0xb6, 0xa2, 0xd6,
                0x08, 0x83, 0xa5, 0x87, 0xff, 0x0e, 0x98, 0x21, 0x1c, 0x73, 0x45, 0x16, 0xb5, 0xdc,
                0xc7, 0x5e, 0xf2, 0x3c, 0xfb, 0x9f, 0x55, 0xf1, 0xde, 0xed, 0xf1, 0x26, 0xc2, 0xce,
                0x17, 0x27, 0x3f, 0x41, 0xdb, 0xbb, 0xbd, 0x2f, 0x49, 0xe3, 0x55, 0x77, 0x6e, 0xc0,
                0x46, 0x98, 0x35, 0xf7, 0x9d, 0x94, 0x80, 0x42, 0xf8, 0x42, 0x0f, 0x11, 0xe1, 0xab,
                0xd7, 0x45, 0x06, 0xb7, 0x8b, 0x5e, 0x41, 0xcb, 0xe0, 0xc7, 0x07, 0x17, 0xf4, 0x6e,
                0x7e, 0xb9, 0xac, 0xdc, 0x35, 0x1c, 0x94, 0x98, 0x83, 0x3a, 0xfd, 0xed, 0x93, 0x06,
                0xa0, 0x43, 0x5b, 0x10, 0xb8, 0x3a, 0xe3, 0x95, 0xd5, 0x7f, 0x5b, 0x0a, 0x5d, 0x41,
                0xa9, 0x34, 0x2d, 0x02, 0xec, 0x58, 0xb6, 0xee, 0x16, 0x87, 0x77, 0x50, 0x16, 0xb8,
                0x74, 0x9b, 0x28, 0x7a, 0xbd, 0xd3, 0xed, 0x1a, 0x83, 0x5e, 0xa8, 0xf3, 0xb1, 0x4d,
                0x08, 0x18, 0xfe, 0x0d, 0x5d, 0x9a, 0x48, 0xeb, 0x02, 0x13, 0x64, 0x0e, 0xec, 0xc1,
                0x9a, 0x5d, 0x16, 0x61, 0xdb, 0x82, 0x2b, 0x77, 0x9b, 0x08, 0x0c, 0xd8, 0xba, 0x7f,
                0x3a, 0x27, 0x23, 0x21, 0xee, 0x5d, 0xa2, 0x27, 0x8c, 0x53, 0x85, 0x67, 0xd0, 0xd9,
                0xbb, 0x28, 0xce, 0x64, 0x21, 0x31, 0x15, 0x03, 0xa4, 0xa0, 0x17, 0x14, 0xcf, 0x91,
                0x01, 0x55, 0x2b, 0xa5, 0xef, 0xc8, 0x5c, 0x94, 0xd5, 0xe5, 0x09, 0x72, 0x7e, 0x5e,
                0x01, 0x1c, 0x15, 0xb1, 0xb2, 0xec, 0xdf, 0xf3, 0x99, 0xc0, 0xbe, 0x33, 0x42, 0xab,
                0x6e, 0xdd, 0xa8, 0xe3, 0xed, 0x81, 0x1a, 0x7d, 0x9c, 0x9c, 0xa4, 0xbb, 0x71, 0xa5,
                0x63, 0xe1, 0x59, 0x78, 0xbf, 0x8e, 0x64, 0x04, 0xac, 0x79, 0x6e, 0xb7, 0x81, 0xfd,
                0x2c, 0xf2, 0x19, 0x1b, 0x2f, 0x4d, 0x40, 0x76, 0xd9, 0x3c, 0xcc, 0x80, 0xf7, 0xe5,
                0x92, 0xfa, 0x66, 0x9b, 0x72, 0x26, 0x57, 0x82, 0xee, 0x8f, 0x5c, 0xe3, 0x03, 0x12,
                0xd3, 0x51, 0x2b, 0x35, 0x49, 0x0a, 0xfe, 0x00, 0x6b, 0xad, 0xb1, 0x62, 0x0d, 0x1b,
                0x0c, 0x79, 0xfb, 0xc4, 0xbe, 0xc5, 0x65, 0xd7, 0x46, 0x1d, 0x68, 0xef, 0x72, 0x27,
                0x79, 0x11, 0x77, 0x6c, 0xd5, 0xa7, 0xb5, 0xfc, 0x6f, 0xa8, 0xb3, 0xee, 0xfd, 0x7a,
                0x39, 0xe8, 0xe5, 0xb4, 0xf6, 0xa5, 0x0c, 0x7d, 0x58, 0xd9, 0xeb, 0x08, 0x38, 0x0d,
                0x32, 0x0b, 0x36, 0xea, 0x04, 0x37, 0x00, 0xac, 0xa7, 0x64, 0xb4, 0x8c, 0x3d, 0xa4,
                0x93, 0x67, 0xfa, 0x93, 0x35, 0x6a, 0xaa, 0x4f, 0x87, 0x08, 0xea, 0x6e, 0x34, 0x59,
                0x81, 0x84, 0x5b, 0xe7, 0x37, 0x6d, 0xa7, 0x98, 0x40, 0x53, 0xef, 0x7d, 0xd4, 0xb6,
                0xa7, 0x27, 0x92, 0x35, 0x6a, 0x6c, 0x34, 0x62, 0x68, 0x88, 0xcc, 0x70, 0xde, 0x49,
                0x9e, 0xf9, 0x10, 0x26, 0x95, 0xd9, 0xdb, 0x12, 0xaf, 0x29, 0x62, 0xfc, 0x75, 0xd4,
                0x36, 0x56, 0x19, 0xdb, 0x0e, 0x87, 0x6c, 0xdb, 0x82, 0x02, 0xe8, 0x16, 0xfd, 0xc2,
                0xcd, 0xf3, 0x7a, 0xd3, 0xbe, 0x3b,
            ],
            ock: [
                0x91, 0x36, 0x59, 0x30, 0x9e, 0xcf, 0xcd, 0xfd, 0x7e, 0x0c, 0xef, 0x23, 0xf8, 0x80,
                0xae, 0x4c, 0xf4, 0xd8, 0xcf, 0x67, 0x78, 0xb9, 0xc4, 0xe6, 0xf4, 0xc7, 0x71, 0x7b,
                0xf5, 0xca, 0xf0, 0x9e,
            ],
            op: [
                0xb4, 0xca, 0xc5, 0x6f, 0x06, 0x2b, 0xfb, 0x2e, 0x27, 0x15, 0xea, 0xf9, 0xc8, 0xfc,
                0xdb, 0xc2, 0x0c, 0x86, 0x79, 0x3f, 0x23, 0x57, 0xdd, 0xd0, 0x4a, 0xad, 0x39, 0xf9,
                0x4a, 0xd7, 0xc7, 0x84, 0x68, 0x65, 0x87, 0xce, 0x53, 0xc3, 0x39, 0xf9, 0xce, 0xcf,
                0x4d, 0x80, 0x4a, 0x17, 0x09, 0x39, 0x2b, 0x6a, 0xb1, 0x08, 0xea, 0x2c, 0x57, 0x79,
                0x92, 0x1c, 0xd5, 0xda, 0x8a, 0x6e, 0x1a, 0x08,
            ],
            c_out: [
                0x4d, 0xf8, 0xda, 0x22, 0xec, 0x17, 0xf4, 0x16, 0xe0, 0x59, 0x1a, 0xac, 0xc1, 0x6b,
                0x6d, 0xd2, 0xbb, 0xbf, 0x47, 0xbe, 0x04, 0x30, 0x3d, 0xc8, 0x85, 0xd3, 0x5a, 0xc3,
                0xf9, 0x92, 0x3e, 0xea, 0x41, 0xf3, 0x6b, 0x3a, 0x4a, 0x5c, 0x5e, 0x73, 0x3e, 0x32,
                0x6e, 0x96, 0xdb, 0xe5, 0x5e, 0xf9, 0xe7, 0xe8, 0x42, 0x27, 0x0c, 0xbf, 0x46, 0x7c,
                0xdc, 0x16, 0x0e, 0xbf, 0x4f, 0x10, 0x9a, 0xd6, 0x92, 0x0a, 0x6a, 0xed, 0x4a, 0x01,
                0x71, 0xd9, 0x06, 0xe3, 0xe8, 0x13, 0x32, 0xe6, 0xc5, 0x61,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0x91, 0xee, 0x20, 0x54, 0x48, 0xc9, 0x8b, 0x69, 0xa3, 0x3e, 0xbf, 0x29, 0x35, 0x09,
                0x5d, 0x79, 0xc2, 0x53, 0x02, 0x9e, 0x5e, 0x5d, 0xc0, 0x2d, 0xf5, 0x8a, 0x10, 0x03,
                0xd1, 0xd8, 0x5c, 0x27, 0xf2, 0xde, 0xf5, 0xb1, 0x10, 0xfd, 0x43, 0xd7, 0x15, 0xe8,
                0xd5, 0x9e, 0xc4, 0xad, 0x0f, 0x41, 0x02, 0x0e, 0xc6, 0x60, 0xcd, 0x97, 0x33, 0xe7,
                0x79, 0xb5, 0x1a, 0x7a, 0xc2, 0xd5, 0xa6, 0x31,
            ],
            ovk: [
                0x18, 0x2f, 0x20, 0x7b, 0x31, 0x75, 0x96, 0x1f, 0x64, 0x11, 0xa4, 0x93, 0xbf, 0xfd,
                0x04, 0x8e, 0x7d, 0x0d, 0x87, 0xd8, 0x2f, 0xe6, 0xf9, 0x90, 0xa2, 0xb0, 0xa2, 0x5f,
                0x5a, 0xa0, 0x11, 0x1a,
            ],
            default_d: [
                0x08, 0xab, 0x2e, 0xe9, 0x9d, 0x4d, 0x9b, 0x98, 0x3d, 0xdd, 0x22,
            ],
            default_pk_d: [
                0x82, 0xfe, 0xf6, 0x43, 0xdb, 0xf4, 0x2d, 0xca, 0x51, 0x56, 0xfb, 0x51, 0xd4, 0xc4,
                0xee, 0x00, 0x8a, 0x72, 0xf0, 0xdb, 0xc3, 0xf3, 0x1e, 0xfa, 0xb0, 0x75, 0xf2, 0x75,
                0x15, 0x37, 0x14, 0x0d,
            ],
            v: 14400879385556610631,
            rseed: [
                0xd5, 0x07, 0xcd, 0xfe, 0x6f, 0xbd, 0xaa, 0x86, 0x16, 0x3e, 0x9c, 0xf5, 0xde, 0x31,
                0x00, 0xfb, 0xca, 0x7e, 0x8d, 0xa0, 0x47, 0xb0, 0x90, 0xdb, 0x9f, 0x37, 0x95, 0x2f,
                0xbf, 0xee, 0x76, 0xaf,
            ],
            memo: [
                0xff, 0x61, 0x66, 0x81, 0x90, 0xbd, 0x52, 0xed, 0x49, 0x0e, 0x67, 0x7b, 0x51, 0x5d,
                0x01, 0x43, 0x84, 0xaf, 0x07, 0x21, 0x9c, 0x7c, 0x0e, 0xe7, 0xfc, 0x7b, 0xfc, 0x79,
                0xf3, 0x25, 0x64, 0x4e, 0x4d, 0xf4, 0xc0, 0xd7, 0xdb, 0x08, 0xe9, 0xf0, 0xbd, 0x02,
                0x49, 0x43, 0xc7, 0x05, 0xab, 0xff, 0x89, 0x94, 0xbf, 0xa6, 0x05, 0xcf, 0xbc, 0x7e,
                0xd7, 0x46, 0xa7, 0xd3, 0xf7, 0xc3, 0x7d, 0x9e, 0x8b, 0xdc, 0x43, 0x3b, 0x7d, 0x79,
                0xe0, 0x8a, 0x12, 0xf7, 0x38, 0xa8, 0xf0, 0xdb, 0xdd, 0xfe, 0xf2, 0xf2, 0x65, 0x7e,
                0xf3, 0xe4, 0x7d, 0x1b, 0x0f, 0xd1, 0x1e, 0x6a, 0x13, 0x31, 0x1f, 0xb7, 0x99, 0xc7,
                0x9c, 0x64, 0x1d, 0x9d, 0xa4, 0x3b, 0x33, 0xe7, 0xad, 0x01, 0x2e, 0x28, 0x25, 0x53,
                0x98, 0x78, 0x92, 0x62, 0x27, 0x5f, 0x11, 0x75, 0xbe, 0x84, 0x62, 0xc0, 0x14, 0x91,
                0xc4, 0xd8, 0x42, 0x40, 0x6d, 0x0e, 0xc4, 0x28, 0x2c, 0x95, 0x26, 0x17, 0x4a, 0x09,
                0x87, 0x8f, 0xe8, 0xfd, 0xde, 0x33, 0xa2, 0x96, 0x04, 0xe5, 0xe5, 0xe7, 0xb2, 0xa0,
                0x25, 0xd6, 0x65, 0x0b, 0x97, 0xdb, 0xb5, 0x2b, 0xef, 0xb5, 0x9b, 0x1d, 0x30, 0xa5,
                0x74, 0x33, 0xb0, 0xa3, 0x51, 0x47, 0x44, 0x44, 0x09, 0x9d, 0xaa, 0x37, 0x10, 0x46,
                0x61, 0x32, 0x60, 0xcf, 0x33, 0x54, 0xcf, 0xcd, 0xad, 0xa6, 0x63, 0xec, 0xe8, 0x24,
                0xff, 0xd7, 0xe4, 0x43, 0x93, 0x88, 0x6a, 0x86, 0x16, 0x5d, 0xdd, 0xdf, 0x2b, 0x4c,
                0x41, 0x77, 0x35, 0x54, 0xc8, 0x69, 0x95, 0x26, 0x94, 0x08, 0xb1, 0x1e, 0x67, 0x37,
                0xa4, 0xc4, 0x47, 0x58, 0x6f, 0x69, 0x17, 0x34, 0x46, 0xd8, 0xe4, 0x8b, 0xf8, 0x4c,
                0xbc, 0x00, 0x0a, 0x80, 0x78, 0x99, 0x97, 0x3e, 0xb9, 0x3c, 0x5e, 0x81, 0x9a, 0xad,
                0x66, 0x94, 0x13, 0xf8, 0x38, 0x79, 0x33, 0xad, 0x15, 0x84, 0xaa, 0x35, 0xe4, 0x3f,
                0x4e, 0xcd, 0x1e, 0x2d, 0x04, 0x07, 0xc0, 0xb1, 0xb8, 0x99, 0x20, 0xff, 0xdf, 0xdb,
                0x9b, 0xea, 0x51, 0xac, 0x95, 0xb5, 0x57, 0xaf, 0x71, 0xb8, 0x9f, 0x90, 0x3f, 0x5d,
                0x98, 0x48, 0xf1, 0x4f, 0xcb, 0xeb, 0x18, 0x37, 0x57, 0x0f, 0x54, 0x4d, 0x63, 0x59,
                0xeb, 0x23, 0xfa, 0xf3, 0x8a, 0x08, 0x22, 0xda, 0x36, 0xce, 0x42, 0x6c, 0x4a, 0x2f,
                0xbe, 0xff, 0xeb, 0x0a, 0x8a, 0x2e, 0x29, 0x7a, 0x9d, 0x19, 0xba, 0x15, 0x02, 0x45,
                0x90, 0xe3, 0x32, 0x9d, 0x9f, 0xa9, 0x26, 0x1f, 0x99, 0x38, 0xa4, 0x03, 0x2d, 0xd3,
                0x46, 0x06, 0xc9, 0xcf, 0x9f, 0x3d, 0xd3, 0x3e, 0x57, 0x6f, 0x05, 0xcd, 0x1d, 0xd6,
                0x81, 0x1c, 0x62, 0x98, 0x75, 0x7d, 0x77, 0xd9, 0xe8, 0x10, 0xab, 0xdb, 0x22, 0x6a,
                0xfc, 0xaa, 0x43, 0x46, 0xa6, 0x56, 0x0f, 0x89, 0x32, 0xb3, 0x18, 0x1f, 0xd3, 0x55,
                0xd5, 0xd3, 0x91, 0x97, 0x61, 0x83, 0xf8, 0xd9, 0x93, 0x88, 0x83, 0x96, 0x32, 0xd6,
                0x35, 0x4f, 0x66, 0x6d, 0x09, 0xd3, 0xe5, 0x62, 0x9e, 0xa1, 0x97, 0x37, 0x38, 0x86,
                0x13, 0xd3, 0x8a, 0x34, 0xfd, 0x0f, 0x6e, 0x50, 0xee, 0x5a, 0x0c, 0xc9, 0x67, 0x71,
                0x77, 0xf5, 0x00, 0x28, 0xc1, 0x41, 0x37, 0x81, 0x87, 0xbd, 0x28, 0x19, 0x40, 0x3f,
                0xc5, 0x34, 0xf8, 0x00, 0x76, 0xe9, 0x38, 0x0c, 0xb4, 0x96, 0x4d, 0x3b, 0x6b, 0x45,
                0x81, 0x9d, 0x3b, 0x8e, 0x9c, 0xaf, 0x54, 0xf0, 0x51, 0x85, 0x2d, 0x67, 0x1b, 0xf8,
                0xc1, 0xff, 0xde, 0x2d, 0x15, 0x10, 0x75, 0x64, 0x18, 0xcb, 0x48, 0x10, 0x93, 0x6a,
                0xa5, 0x7e, 0x69, 0x65, 0xd6, 0xfb, 0x65, 0x6a, 0x76, 0x0b, 0x7f, 0x19, 0xad, 0xf9,
                0x6c, 0x17, 0x34, 0x88, 0x55, 0x21, 0x93, 0xb1,
            ],
            cv_net: [
                0xc3, 0x94, 0x68, 0x5d, 0x92, 0x95, 0x59, 0x7e, 0x21, 0x55, 0x7f, 0x21, 0x9f, 0x3c,
                0x9d, 0x5e, 0x64, 0x07, 0x19, 0xbc, 0xa5, 0xc8, 0xed, 0x49, 0x99, 0x97, 0x34, 0xe6,
                0xc5, 0xb3, 0x73, 0x3e,
            ],
            rho: [
                0xbe, 0xf8, 0xcf, 0x16, 0x98, 0xe4, 0x78, 0x47, 0xd3, 0x8e, 0x1a, 0xaa, 0x88, 0x86,
                0x10, 0x77, 0xcd, 0xb5, 0xad, 0x4c, 0xf6, 0x6f, 0xe4, 0x2f, 0xd6, 0x52, 0x57, 0x81,
                0xb6, 0xd3, 0x4f, 0x1e,
            ],
            cmx: [
                0xd8, 0x19, 0xa6, 0x37, 0x7a, 0xce, 0x33, 0xf9, 0x21, 0xf2, 0x29, 0xf9, 0x32, 0x86,
                0x6d, 0x9f, 0xcd, 0xb9, 0xd0, 0x42, 0x6a, 0xfa, 0xca, 0x9e, 0x60, 0x50, 0xb4, 0x7a,
                0x83, 0x19, 0xd6, 0x0d,
            ],
            esk: [
                0x59, 0xd1, 0x0a, 0x5b, 0x94, 0x15, 0x8a, 0x3f, 0x3a, 0x78, 0xb3, 0x5d, 0xa9, 0xc6,
                0x27, 0xbe, 0xdf, 0x7c, 0xfb, 0x84, 0x7e, 0x3e, 0x59, 0x86, 0xec, 0x8a, 0xd7, 0xf7,
                0x4c, 0xd9, 0xb2, 0x1b,
            ],
            ephemeral_key: [
                0x5b, 0xcb, 0xf9, 0xf1, 0xd7, 0xdd, 0x68, 0xe7, 0xcc, 0x6d, 0x6c, 0x78, 0x49, 0x50,
                0xd1, 0xc2, 0xe0, 0xbe, 0x6a, 0x84, 0xa7, 0xa8, 0x8d, 0x6f, 0x7a, 0x20, 0x98, 0xc3,
                0xdc, 0xae, 0x3f, 0x2f,
            ],
            shared_secret: [
                0x37, 0x35, 0x1c, 0xe2, 0x57, 0xb2, 0x79, 0x4d, 0x86, 0xa5, 0x3d, 0x26, 0x8d, 0xc9,
                0x00, 0x06, 0x40, 0xc2, 0x76, 0xf3, 0xf4, 0x65, 0xe1, 0xaa, 0x70, 0xbf, 0xde, 0xf4,
                0x99, 0xa3, 0xd7, 0xaa,
            ],
            k_enc: [
                0xea, 0x22, 0x99, 0x65, 0x39, 0xd3, 0x74, 0xda, 0x6a, 0x75, 0x34, 0x39, 0x5a, 0xe9,
                0x23, 0x36, 0xfc, 0xa7, 0x85, 0x11, 0x20, 0xdd, 0x1a, 0xe4, 0x9e, 0x45, 0xb3, 0x3e,
                0x0b, 0xed, 0xe9, 0xac,
            ],
            p_enc: [
                0x02, 0x08, 0xab, 0x2e, 0xe9, 0x9d, 0x4d, 0x9b, 0x98, 0x3d, 0xdd, 0x22, 0x47, 0xee,
                0x58, 0x85, 0x80, 0x33, 0xda, 0xc7, 0xd5, 0x07, 0xcd, 0xfe, 0x6f, 0xbd, 0xaa, 0x86,
                0x16, 0x3e, 0x9c, 0xf5, 0xde, 0x31, 0x00, 0xfb, 0xca, 0x7e, 0x8d, 0xa0, 0x47, 0xb0,
                0x90, 0xdb, 0x9f, 0x37, 0x95, 0x2f, 0xbf, 0xee, 0x76, 0xaf, 0xff, 0x61, 0x66, 0x81,
                0x90, 0xbd, 0x52, 0xed, 0x49, 0x0e, 0x67, 0x7b, 0x51, 0x5d, 0x01, 0x43, 0x84, 0xaf,
                0x07, 0x21, 0x9c, 0x7c, 0x0e, 0xe7, 0xfc, 0x7b, 0xfc, 0x79, 0xf3, 0x25, 0x64, 0x4e,
                0x4d, 0xf4, 0xc0, 0xd7, 0xdb, 0x08, 0xe9, 0xf0, 0xbd, 0x02, 0x49, 0x43, 0xc7, 0x05,
                0xab, 0xff, 0x89, 0x94, 0xbf, 0xa6, 0x05, 0xcf, 0xbc, 0x7e, 0xd7, 0x46, 0xa7, 0xd3,
                0xf7, 0xc3, 0x7d, 0x9e, 0x8b, 0xdc, 0x43, 0x3b, 0x7d, 0x79, 0xe0, 0x8a, 0x12, 0xf7,
                0x38, 0xa8, 0xf0, 0xdb, 0xdd, 0xfe, 0xf2, 0xf2, 0x65, 0x7e, 0xf3, 0xe4, 0x7d, 0x1b,
                0x0f, 0xd1, 0x1e, 0x6a, 0x13, 0x31, 0x1f, 0xb7, 0x99, 0xc7, 0x9c, 0x64, 0x1d, 0x9d,
                0xa4, 0x3b, 0x33, 0xe7, 0xad, 0x01, 0x2e, 0x28, 0x25, 0x53, 0x98, 0x78, 0x92, 0x62,
                0x27, 0x5f, 0x11, 0x75, 0xbe, 0x84, 0x62, 0xc0, 0x14, 0x91, 0xc4, 0xd8, 0x42, 0x40,
                0x6d, 0x0e, 0xc4, 0x28, 0x2c, 0x95, 0x26, 0x17, 0x4a, 0x09, 0x87, 0x8f, 0xe8, 0xfd,
                0xde, 0x33, 0xa2, 0x96, 0x04, 0xe5, 0xe5, 0xe7, 0xb2, 0xa0, 0x25, 0xd6, 0x65, 0x0b,
                0x97, 0xdb, 0xb5, 0x2b, 0xef, 0xb5, 0x9b, 0x1d, 0x30, 0xa5, 0x74, 0x33, 0xb0, 0xa3,
                0x51, 0x47, 0x44, 0x44, 0x09, 0x9d, 0xaa, 0x37, 0x10, 0x46, 0x61, 0x32, 0x60, 0xcf,
                0x33, 0x54, 0xcf, 0xcd, 0xad, 0xa6, 0x63, 0xec, 0xe8, 0x24, 0xff, 0xd7, 0xe4, 0x43,
                0x93, 0x88, 0x6a, 0x86, 0x16, 0x5d, 0xdd, 0xdf, 0x2b, 0x4c, 0x41, 0x77, 0x35, 0x54,
                0xc8, 0x69, 0x95, 0x26, 0x94, 0x08, 0xb1, 0x1e, 0x67, 0x37, 0xa4, 0xc4, 0x47, 0x58,
                0x6f, 0x69, 0x17, 0x34, 0x46, 0xd8, 0xe4, 0x8b, 0xf8, 0x4c, 0xbc, 0x00, 0x0a, 0x80,
                0x78, 0x99, 0x97, 0x3e, 0xb9, 0x3c, 0x5e, 0x81, 0x9a, 0xad, 0x66, 0x94, 0x13, 0xf8,
                0x38, 0x79, 0x33, 0xad, 0x15, 0x84, 0xaa, 0x35, 0xe4, 0x3f, 0x4e, 0xcd, 0x1e, 0x2d,
                0x04, 0x07, 0xc0, 0xb1, 0xb8, 0x99, 0x20, 0xff, 0xdf, 0xdb, 0x9b, 0xea, 0x51, 0xac,
                0x95, 0xb5, 0x57, 0xaf, 0x71, 0xb8, 0x9f, 0x90, 0x3f, 0x5d, 0x98, 0x48, 0xf1, 0x4f,
                0xcb, 0xeb, 0x18, 0x37, 0x57, 0x0f, 0x54, 0x4d, 0x63, 0x59, 0xeb, 0x23, 0xfa, 0xf3,
                0x8a, 0x08, 0x22, 0xda, 0x36, 0xce, 0x42, 0x6c, 0x4a, 0x2f, 0xbe, 0xff, 0xeb, 0x0a,
                0x8a, 0x2e, 0x29, 0x7a, 0x9d, 0x19, 0xba, 0x15, 0x02, 0x45, 0x90, 0xe3, 0x32, 0x9d,
                0x9f, 0xa9, 0x26, 0x1f, 0x99, 0x38, 0xa4, 0x03, 0x2d, 0xd3, 0x46, 0x06, 0xc9, 0xcf,
                0x9f, 0x3d, 0xd3, 0x3e, 0x57, 0x6f, 0x05, 0xcd, 0x1d, 0xd6, 0x81, 0x1c, 0x62, 0x98,
                0x75, 0x7d, 0x77, 0xd9, 0xe8, 0x10, 0xab, 0xdb, 0x22, 0x6a, 0xfc, 0xaa, 0x43, 0x46,
                0xa6, 0x56, 0x0f, 0x89, 0x32, 0xb3, 0x18, 0x1f, 0xd3, 0x55, 0xd5, 0xd3, 0x91, 0x97,
                0x61, 0x83, 0xf8, 0xd9, 0x93, 0x88, 0x83, 0x96, 0x32, 0xd6, 0x35, 0x4f, 0x66, 0x6d,
                0x09, 0xd3, 0xe5, 0x62, 0x9e, 0xa1, 0x97, 0x37, 0x38, 0x86, 0x13, 0xd3, 0x8a, 0x34,
                0xfd, 0x0f, 0x6e, 0x50, 0xee, 0x5a, 0x0c, 0xc9, 0x67, 0x71, 0x77, 0xf5, 0x00, 0x28,
                0xc1, 0x41, 0x37, 0x81, 0x87, 0xbd, 0x28, 0x19, 0x40, 0x3f, 0xc5, 0x34, 0xf8, 0x00,
                0x76, 0xe9, 0x38, 0x0c, 0xb4, 0x96, 0x4d, 0x3b, 0x6b, 0x45, 0x81, 0x9d, 0x3b, 0x8e,
                0x9c, 0xaf, 0x54, 0xf0, 0x51, 0x85, 0x2d, 0x67, 0x1b, 0xf8, 0xc1, 0xff, 0xde, 0x2d,
                0x15, 0x10, 0x75, 0x64, 0x18, 0xcb, 0x48, 0x10, 0x93, 0x6a, 0xa5, 0x7e, 0x69, 0x65,
                0xd6, 0xfb, 0x65, 0x6a, 0x76, 0x0b, 0x7f, 0x19, 0xad, 0xf9, 0x6c, 0x17, 0x34, 0x88,
                0x55, 0x21, 0x93, 0xb1,
            ],
            c_enc: [
                0x12, 0xd6, 0x64, 0xed, 0x05, 0xd6, 0x46, 0x26, 0x89, 0xd4, 0xf2, 0x4a, 0xee, 0x5a,
                0x4f, 0x0f, 0x32, 0x35, 0xff, 0x11, 0x0b, 0x2d, 0xf9, 0x9f, 0x67, 0xd8, 0xc5, 0xb3,
                0x68, 0xdd, 0x47, 0x69, 0xd8, 0x44, 0xd3, 0xdd, 0xa0, 0x3f, 0x58, 0xc5, 0x48, 0x63,
                0x62, 0xe8, 0x90, 0x81, 0xa5, 0xdf, 0xd0, 0xa6, 0x06, 0xa3, 0x91, 0x26, 0x4b, 0x56,
                0xca, 0x3a, 0xfc, 0x4f, 0xe0, 0xe4, 0xc3, 0x05, 0xf3, 0x07, 0x78, 0x09, 0x4a, 0x00,
                0xb7, 0x33, 0x4b, 0xdd, 0x82, 0x45, 0xac, 0x56, 0x0e, 0xf3, 0x29, 0xbc, 0x68, 0x97,
                0xd4, 0xd7, 0xba, 0x31, 0xac, 0x84, 0x54, 0x44, 0x1a, 0x15, 0xc8, 0xd3, 0xce, 0xcc,
                0x71, 0x32, 0xdf, 0x0d, 0x9d, 0x0e, 0xcf, 0x92, 0x84, 0x34, 0xa0, 0xd2, 0x8c, 0x1b,
                0x00, 0x48, 0x52, 0x01, 0xec, 0x33, 0xbe, 0x9a, 0x28, 0x74, 0xb4, 0x29, 0x6c, 0x04,
                0x22, 0xc7, 0xe7, 0xa0, 0xa3, 0xa2, 0x2e, 0xc7, 0xe7, 0x21, 0xa4, 0x79, 0x22, 0x8d,
                0xa2, 0x8b, 0x47, 0x37, 0xaf, 0x52, 0x06, 0xdf, 0x7d, 0x74, 0xe4, 0x84, 0xc4, 0xf7,
                0xa8, 0x56, 0xbe, 0x8c, 0xd0, 0x4b, 0x21, 0x26, 0xb5, 0x27, 0x11, 0xe7, 0xb0, 0xaf,
                0x75, 0xc7, 0x52, 0x84, 0xa1, 0x57, 0x20, 0x40, 0xe8, 0xad, 0xe5, 0x85, 0xe8, 0xa4,
                0x82, 0x80, 0x03, 0x59, 0x67, 0x46, 0xc4, 0x0c, 0x9d, 0x76, 0x0d, 0x92, 0x74, 0xb1,
                0x25, 0x42, 0x2b, 0x63, 0x48, 0x1a, 0x17, 0xff, 0xba, 0xb8, 0xc2, 0xde, 0x13, 0xb2,
                0x19, 0xf5, 0x8a, 0x35, 0x95, 0x2d, 0x88, 0x7a, 0xed, 0xe8, 0xe0, 0x2f, 0x10, 0x33,
                0x8c, 0x23, 0x98, 0x23, 0xfb, 0x43, 0x49, 0x51, 0x84, 0x47, 0x12, 0xf6, 0x8d, 0x6e,
                0x4f, 0xef, 0xae, 0x2b, 0x79, 0x5b, 0xa9, 0x78, 0xe9, 0x81, 0xc1, 0x09, 0x27, 0xab,
                0xbc, 0x16, 0x30, 0x66, 0xa0, 0xe9, 0x60, 0xb3, 0xb8, 0xa3, 0x26, 0xc0, 0x39, 0x85,
                0x81, 0x10, 0x93, 0x99, 0xf6, 0xed, 0x60, 0x44, 0x9a, 0xa8, 0x58, 0xd5, 0xdd, 0x27,
                0xdb, 0xf8, 0x89, 0x9f, 0x9c, 0x9a, 0x50, 0x20, 0x5f, 0x25, 0xd0, 0xcc, 0x50, 0xb2,
                0xde, 0xe3, 0x63, 0x54, 0xc5, 0xe4, 0x48, 0x4d, 0x36, 0xf6, 0x3c, 0x97, 0x63, 0xd8,
                0x41, 0xad, 0x5e, 0x00, 0x21, 0x63, 0x6a, 0x85, 0x7c, 0xfb, 0x79, 0xa5, 0x12, 0x3c,
                0x3d, 0xfb, 0x77, 0x3d, 0x0c, 0x1b, 0xeb, 0x9f, 0x90, 0xa9, 0x72, 0xd0, 0xfc, 0x80,
                0x5f, 0x65, 0x5d, 0x69, 0x40, 0x85, 0x23, 0xb9, 0x9b, 0x62, 0xa8, 0xfa, 0xbe, 0xf0,
                0xc0, 0x24, 0xf2, 0x1f, 0x50, 0xe4, 0xc1, 0x12, 0xe2, 0xfe, 0xdd, 0x58, 0xca, 0xe9,
                0x60, 0x9a, 0xc6, 0xf7, 0xcc, 0x79, 0x83, 0x86, 0xc9, 0xd9, 0x06, 0x42, 0x1c, 0xa5,
                0x7c, 0xf8, 0x1b, 0x09, 0x6b, 0xba, 0xda, 0x64, 0xd0, 0xee, 0x76, 0x95, 0x18, 0x9d,
                0x5f, 0xb1, 0x7a, 0xe2, 0x53, 0x1d, 0xbb, 0x2c, 0x00, 0x58, 0x5a, 0x26, 0xa6, 0x8c,
                0x27, 0xf9, 0x77, 0x77, 0x84, 0x1a, 0x3e, 0x39, 0x30, 0xc7, 0x0f, 0xc3, 0xfa, 0x8e,
                0x2b, 0x7f, 0xc2, 0x1e, 0x87, 0xcf, 0x9f, 0x63, 0xb3, 0x63, 0xb8, 0x8d, 0xaa, 0x1f,
                0xb6, 0x7b, 0xda, 0xe8, 0xe5, 0x5b, 0x68, 0x51, 0x6d, 0x19, 0xdf, 0xef, 0xec, 0x9b,
                0x3d, 0x38, 0xe6, 0xe1, 0xd0, 0xa6, 0xe4, 0x51, 0xd6, 0xd1, 0xf5, 0x2d, 0x1f, 0x96,
                0xdd, 0x0d, 0x53, 0x6d, 0x68, 0xd2, 0x69, 0x86, 0x70, 0x9f, 0x41, 0xe7, 0x60, 0x74,
                0x05, 0x5b, 0xf7, 0x52, 0xbf, 0x38, 0x86, 0x92, 0xc8, 0x2c, 0xfd, 0xa1, 0xeb, 0xb0,
                0x17, 0x8b, 0x8e, 0x0c, 0x85, 0xad, 0x7b, 0x15, 0x99, 0x14, 0x42, 0x8e, 0x30, 0x21,
                0xda, 0xe3, 0x01, 0x0d, 0x65, 0x6c, 0x10, 0x36, 0xf4, 0xa5, 0x7e, 0x7f, 0xad, 0xe0,
                0xfc, 0x32, 0x2a, 0xa6, 0xfd, 0xde, 0x71, 0x4a, 0x8c, 0x53, 0x78, 0x79, 0xe7, 0x04,
                0x41, 0x6f, 0x51, 0x04, 0xdb, 0xbc, 0x8f, 0xf2, 0x42, 0xc1, 0x6d, 0x2d, 0xf4, 0xa8,
                0x41, 0xeb, 0x6b, 0x45, 0x3a, 0x12, 0x83, 0xf6, 0x5f, 0xe1, 0x0d, 0x70, 0xc8, 0x76,
                0x41, 0x8e, 0x44, 0x4a, 0xb3, 0x1b, 0x93, 0x71, 0xa2, 0x7d, 0x36, 0xd8, 0x6e, 0x8f,
                0x1c, 0x32, 0x77, 0xca, 0xfd, 0xf8,
            ],
            ock: [
                0xca, 0x6f, 0x98, 0xe6, 0xcf, 0x9f, 0xc3, 0x6a, 0xf5, 0xf6, 0x98, 0x82, 0x74, 0x2d,
                0x06, 0xec, 0x5d, 0x29, 0xac, 0x2a, 0x02, 0x89, 0x44, 0xa2, 0x01, 0x89, 0xd5, 0x1c,
                0x8a, 0x9b, 0xdc, 0xd6,
            ],
            op: [
                0x82, 0xfe, 0xf6, 0x43, 0xdb, 0xf4, 0x2d, 0xca, 0x51, 0x56, 0xfb, 0x51, 0xd4, 0xc4,
                0xee, 0x00, 0x8a, 0x72, 0xf0, 0xdb, 0xc3, 0xf3, 0x1e, 0xfa, 0xb0, 0x75, 0xf2, 0x75,
                0x15, 0x37, 0x14, 0x0d, 0x59, 0xd1, 0x0a, 0x5b, 0x94, 0x15, 0x8a, 0x3f, 0x3a, 0x78,
                0xb3, 0x5d, 0xa9, 0xc6, 0x27, 0xbe, 0xdf, 0x7c, 0xfb, 0x84, 0x7e, 0x3e, 0x59, 0x86,
                0xec, 0x8a, 0xd7, 0xf7, 0x4c, 0xd9, 0xb2, 0x1b,
            ],
            c_out: [
                0xa9, 0x5e, 0x1a, 0x22, 0x45, 0x65, 0x29, 0x5e, 0x9b, 0x55, 0x3a, 0xdd, 0xe8, 0xb0,
                0x14, 0x47, 0x5c, 0x68, 0x7b, 0x5d, 0x13, 0x49, 0xc1, 0xdf, 0x8a, 0xd4, 0xb7, 0xcf,
                0xd3, 0xdf, 0xc1, 0x00, 0x6c, 0x7c, 0x37, 0xde, 0x67, 0x6d, 0x6f, 0xde, 0x31, 0x8e,
                0x2f, 0xdf, 0xcc, 0x2e, 0x2e, 0x23, 0x2c, 0xc5, 0xf8, 0x85, 0x28, 0x39, 0xe7, 0xef,
                0xab, 0x8b, 0x20, 0x0e, 0xbc, 0x6a, 0x4d, 0xf8, 0x2a, 0x75, 0x21, 0xce, 0x0f, 0x38,
                0x4f, 0xe2, 0x7a, 0x0d, 0xec, 0x59, 0xe9, 0xd2, 0xc0, 0xe3,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0xf1, 0x90, 0x42, 0xb9, 0xd1, 0x0c, 0xc4, 0x80, 0xa0, 0x8c, 0x04, 0x32, 0x2d, 0xb6,
                0xec, 0x4e, 0x41, 0x2e, 0xaa, 0x84, 0xc9, 0x71, 0x82, 0x8c, 0xcc, 0xd7, 0x33, 0xa1,
                0x1f, 0x25, 0x3e, 0xda, 0x8a, 0xc3, 0x0b, 0xa3, 0x1f, 0xbc, 0x89, 0x5d, 0x60, 0xb9,
                0x83, 0x06, 0x2a, 0x5f, 0x45, 0x33, 0x90, 0x79, 0x32, 0x26, 0xff, 0xd9, 0x21, 0xbd,
                0x64, 0xac, 0x39, 0x07, 0x03, 0x85, 0x6a, 0x0b,
            ],
            ovk: [
                0xda, 0xdc, 0x96, 0x6c, 0x8a, 0x54, 0x66, 0xb6, 0x1f, 0xc9, 0x98, 0xc3, 0x1f, 0x10,
                0x70, 0xd9, 0xa5, 0xc9, 0xa6, 0xd2, 0x68, 0xd3, 0x04, 0xfe, 0x6b, 0x8f, 0xd3, 0xb4,
                0x01, 0x03, 0x48, 0x61,
            ],
            default_d: [
                0xaa, 0x14, 0x92, 0x9c, 0x57, 0x89, 0x85, 0x85, 0xce, 0x66, 0x5a,
            ],
            default_pk_d: [
                0x78, 0xa4, 0xe3, 0x39, 0x88, 0xd7, 0x1d, 0x71, 0x8e, 0x59, 0x55, 0x55, 0x28, 0x4c,
                0x24, 0x9a, 0x62, 0xb7, 0x12, 0x88, 0x06, 0xa5, 0x4c, 0x3b, 0x36, 0xa3, 0xaa, 0x57,
                0x14, 0x93, 0x16, 0x36,
            ],
            v: 17936016275122962426,
            rseed: [
                0x49, 0x95, 0x0a, 0xfc, 0xb0, 0xef, 0x46, 0x2a, 0x2a, 0xe0, 0x24, 0xb0, 0xf0, 0x22,
                0x4d, 0xfd, 0x73, 0x68, 0x4b, 0x88, 0xc7, 0xfb, 0xe9, 0x2d, 0x02, 0xb6, 0x8f, 0x75,
                0x9c, 0x47, 0x52, 0x66,
            ],
            memo: [
                0xff, 0x3c, 0xd7, 0xb9, 0x7a, 0x14, 0x94, 0x36, 0x49, 0x30, 0x55, 0x21, 0x32, 0x6b,
                0xde, 0x08, 0x56, 0x30, 0x86, 0x46, 0x29, 0x29, 0x1b, 0xae, 0x25, 0xff, 0x88, 0x22,
                0xa1, 0x4c, 0x4b, 0x66, 0x6a, 0x92, 0x59, 0xad, 0x0d, 0xc4, 0x2a, 0x82, 0x90, 0xac,
                0x7b, 0xc7, 0xf5, 0x3a, 0x16, 0xf3, 0x79, 0xf7, 0x58, 0xe5, 0xde, 0x75, 0x0f, 0x04,
                0xfd, 0x7c, 0xad, 0x47, 0x70, 0x1c, 0x85, 0x97, 0xf9, 0x78, 0x88, 0xbe, 0xa6, 0xfa,
                0x0b, 0xf2, 0x99, 0x99, 0x56, 0xfb, 0xfd, 0x0e, 0xe6, 0x8e, 0xc3, 0x6e, 0x46, 0x88,
                0x80, 0x9a, 0xe2, 0x31, 0xeb, 0x8b, 0xc4, 0x36, 0x9f, 0x5f, 0xe1, 0x57, 0x3f, 0x57,
                0xe0, 0x99, 0xd9, 0xc0, 0x99, 0x01, 0xbf, 0x39, 0xca, 0xac, 0x48, 0xdc, 0x11, 0x95,
                0x6a, 0x8a, 0xe9, 0x05, 0xea, 0xd8, 0x69, 0x54, 0x54, 0x7c, 0x44, 0x8a, 0xe4, 0x3d,
                0x31, 0x5e, 0x66, 0x9c, 0x42, 0x42, 0xda, 0x56, 0x59, 0x38, 0xf4, 0x17, 0xbf, 0x43,
                0xce, 0x7b, 0x2b, 0x30, 0xb1, 0xcd, 0x40, 0x18, 0x38, 0x8e, 0x1a, 0x91, 0x0f, 0x0f,
                0xc4, 0x1f, 0xb0, 0x87, 0x7a, 0x59, 0x25, 0xe4, 0x66, 0x81, 0x9d, 0x37, 0x5b, 0x0a,
                0x91, 0x2d, 0x4f, 0xe8, 0x43, 0xb7, 0x6e, 0xf6, 0xf2, 0x23, 0xf0, 0xf7, 0xc8, 0x94,
                0xf3, 0x8f, 0x7a, 0xb7, 0x80, 0xdf, 0xd7, 0x5f, 0x66, 0x9c, 0x8c, 0x06, 0xcf, 0xfa,
                0x43, 0xeb, 0x47, 0x56, 0x5a, 0x50, 0xe3, 0xb1, 0xfa, 0x45, 0xad, 0x61, 0xce, 0x9a,
                0x1c, 0x47, 0x27, 0xb7, 0xaa, 0xa5, 0x35, 0x62, 0xf5, 0x23, 0xe7, 0x39, 0x52, 0xbb,
                0xf3, 0x3d, 0x8a, 0x41, 0x04, 0x07, 0x8a, 0xde, 0x3e, 0xaa, 0xa4, 0x96, 0x99, 0xa6,
                0x9f, 0xdf, 0x1c, 0x5a, 0xc7, 0x73, 0x21, 0x46, 0xee, 0x5e, 0x1d, 0x6b, 0x6c, 0xa9,
                0xb9, 0x18, 0x0f, 0x96, 0x4c, 0xc9, 0xd0, 0x87, 0x8a, 0xe1, 0x37, 0x35, 0x24, 0xd7,
                0xd5, 0x10, 0xe5, 0x82, 0x27, 0xdf, 0x6d, 0xe9, 0xd3, 0x0d, 0x27, 0x18, 0x67, 0x64,
                0x01, 0x77, 0xb0, 0xf1, 0x85, 0x6e, 0x28, 0xd5, 0xc8, 0xaf, 0xb0, 0x95, 0xef, 0x61,
                0x84, 0xfe, 0xd6, 0x51, 0x58, 0x90, 0x22, 0xee, 0xae, 0xa4, 0xc0, 0xce, 0x1f, 0xa6,
                0xf0, 0x85, 0x09, 0x2b, 0x04, 0x97, 0x94, 0x89, 0x17, 0x2b, 0x3e, 0xf8, 0x19, 0x4a,
                0x79, 0x8d, 0xf5, 0x72, 0x4d, 0x6b, 0x05, 0xf1, 0xae, 0x00, 0x00, 0x13, 0xa0, 0x8d,
                0x61, 0x2b, 0xca, 0x8a, 0x8c, 0x31, 0x44, 0x3c, 0x10, 0x34, 0x6d, 0xbf, 0x61, 0xde,
                0x84, 0x75, 0xc0, 0xbb, 0xec, 0x51, 0x04, 0xb4, 0x75, 0x56, 0xaf, 0x3d, 0x51, 0x44,
                0x58, 0xe2, 0x32, 0x1d, 0x14, 0x60, 0x71, 0x78, 0x9d, 0x23, 0x35, 0x93, 0x4a, 0x68,
                0x06, 0x14, 0xe8, 0x35, 0x62, 0xf8, 0x2d, 0xfd, 0x40, 0x5b, 0x54, 0xa4, 0x5e, 0xb3,
                0x2c, 0x16, 0x54, 0x48, 0xd4, 0xd5, 0xd6, 0x1c, 0xa2, 0x85, 0x95, 0x85, 0x36, 0x9f,
                0x53, 0xf1, 0xa1, 0x37, 0xe9, 0xe8, 0x2b, 0x67, 0xb8, 0xfd, 0xaf, 0x01, 0xbd, 0xa5,
                0x4a, 0x31, 0x73, 0x11, 0x89, 0x6a, 0xe1, 0x02, 0x80, 0xa0, 0x32, 0x44, 0x0c, 0x42,
                0x0a, 0x42, 0x1e, 0x94, 0x4d, 0x1e, 0x95, 0x2b, 0x70, 0xd5, 0x82, 0x6c, 0xd3, 0xb0,
                0x8b, 0x7d, 0xb9, 0x63, 0x0f, 0xe4, 0xfd, 0x5f, 0x22, 0x12, 0x5d, 0xe8, 0x40, 0xfc,
                0xc4, 0x0b, 0x98, 0x03, 0x8a, 0xf1, 0x1d, 0x55, 0xbe, 0x25, 0x43, 0x25, 0x97, 0xb4,
                0xb6, 0x5b, 0x9e, 0xc1, 0xc7, 0xa8, 0xbb, 0xfd, 0x05, 0x2c, 0xbf, 0x7e, 0x1c, 0x17,
                0x85, 0x31, 0x49, 0x34, 0xb2, 0x62, 0xd5, 0x85, 0x37, 0x54, 0xf1, 0xf1, 0x77, 0x71,
                0xcf, 0xb7, 0x50, 0x30, 0x72, 0x65, 0x57, 0x53,
            ],
            cv_net: [
                0xd4, 0x51, 0xb4, 0x62, 0x89, 0xba, 0x99, 0x8c, 0x0c, 0xce, 0xd1, 0xcc, 0x15, 0xb3,
                0xfa, 0xde, 0x94, 0xfa, 0x0b, 0x46, 0xe3, 0xb1, 0xa5, 0x73, 0x34, 0x99, 0x34, 0xe2,
                0x32, 0xb5, 0x0e, 0x96,
            ],
            rho: [
                0x18, 0x89, 0x8e, 0x75, 0x21, 0x7b, 0x32, 0x9b, 0x3a, 0x56, 0x7b, 0x09, 0x37, 0x89,
                0xa4, 0xd8, 0x19, 0xcd, 0xb0, 0x34, 0x88, 0xb8, 0x10, 0xda, 0x22, 0x0c, 0x3f, 0x59,
                0xba, 0x03, 0x39, 0x26,
            ],
            cmx: [
                0x23, 0xad, 0xc3, 0xbf, 0x35, 0x0e, 0xb0, 0x84, 0xbd, 0x28, 0xbc, 0x2d, 0xcf, 0xb7,
                0x98, 0x3f, 0xf6, 0xb3, 0xb6, 0xf2, 0xeb, 0x2d, 0x6f, 0x12, 0x4f, 0xfc, 0x46, 0x85,
                0xab, 0xe8, 0xde, 0x3a,
            ],
            esk: [
                0x34, 0x39, 0x22, 0x52, 0xc9, 0xcc, 0x9f, 0x45, 0x4b, 0x54, 0x2c, 0xf1, 0xb8, 0x88,
                0xb3, 0xab, 0x02, 0xe6, 0x05, 0xa8, 0xda, 0x26, 0x10, 0x7d, 0x98, 0x02, 0xf1, 0x53,
                0x6a, 0x9e, 0x9f, 0x2b,
            ],
            ephemeral_key: [
                0x91, 0xaf, 0x13, 0x8e, 0xd9, 0x95, 0x44, 0x66, 0x95, 0x16, 0x1b, 0x14, 0x2d, 0xc2,
                0xde, 0x59, 0xac, 0x23, 0x05, 0xec, 0xbe, 0xc6, 0x29, 0x33, 0xf5, 0x2f, 0x2a, 0xa1,
                0xf9, 0xb2, 0x71, 0x86,
            ],
            shared_secret: [
                0xe1, 0xd3, 0xaf, 0x73, 0x6a, 0xb9, 0xc6, 0x11, 0x6f, 0x46, 0x8f, 0x91, 0x66, 0x80,
                0x63, 0x20, 0x35, 0x0f, 0x7e, 0x73, 0x51, 0x3b, 0xa6, 0x05, 0x50, 0xeb, 0x2d, 0xf0,
                0x1f, 0xf7, 0x83, 0x2d,
            ],
            k_enc: [
                0xb4, 0x3e, 0x43, 0x3e, 0x16, 0x45, 0xeb, 0x51, 0x5e, 0x4a, 0x4c, 0x7d, 0x10, 0xc9,
                0x3d, 0x08, 0xf2, 0xf1, 0xc3, 0x30, 0x95, 0xbf, 0x8c, 0x80, 0x0a, 0x04, 0x17, 0x62,
                0xab, 0x5c, 0xf8, 0x2c,
            ],
            p_enc: [
                0x02, 0xaa, 0x14, 0x92, 0x9c, 0x57, 0x89, 0x85, 0x85, 0xce, 0x66, 0x5a, 0xfa, 0x3f,
                0x54, 0xec, 0xc5, 0x87, 0xe9, 0xf8, 0x49, 0x95, 0x0a, 0xfc, 0xb0, 0xef, 0x46, 0x2a,
                0x2a, 0xe0, 0x24, 0xb0, 0xf0, 0x22, 0x4d, 0xfd, 0x73, 0x68, 0x4b, 0x88, 0xc7, 0xfb,
                0xe9, 0x2d, 0x02, 0xb6, 0x8f, 0x75, 0x9c, 0x47, 0x52, 0x66, 0xff, 0x3c, 0xd7, 0xb9,
                0x7a, 0x14, 0x94, 0x36, 0x49, 0x30, 0x55, 0x21, 0x32, 0x6b, 0xde, 0x08, 0x56, 0x30,
                0x86, 0x46, 0x29, 0x29, 0x1b, 0xae, 0x25, 0xff, 0x88, 0x22, 0xa1, 0x4c, 0x4b, 0x66,
                0x6a, 0x92, 0x59, 0xad, 0x0d, 0xc4, 0x2a, 0x82, 0x90, 0xac, 0x7b, 0xc7, 0xf5, 0x3a,
                0x16, 0xf3, 0x79, 0xf7, 0x58, 0xe5, 0xde, 0x75, 0x0f, 0x04, 0xfd, 0x7c, 0xad, 0x47,
                0x70, 0x1c, 0x85, 0x97, 0xf9, 0x78, 0x88, 0xbe, 0xa6, 0xfa, 0x0b, 0xf2, 0x99, 0x99,
                0x56, 0xfb, 0xfd, 0x0e, 0xe6, 0x8e, 0xc3, 0x6e, 0x46, 0x88, 0x80, 0x9a, 0xe2, 0x31,
                0xeb, 0x8b, 0xc4, 0x36, 0x9f, 0x5f, 0xe1, 0x57, 0x3f, 0x57, 0xe0, 0x99, 0xd9, 0xc0,
                0x99, 0x01, 0xbf, 0x39, 0xca, 0xac, 0x48, 0xdc, 0x11, 0x95, 0x6a, 0x8a, 0xe9, 0x05,
                0xea, 0xd8, 0x69, 0x54, 0x54, 0x7c, 0x44, 0x8a, 0xe4, 0x3d, 0x31, 0x5e, 0x66, 0x9c,
                0x42, 0x42, 0xda, 0x56, 0x59, 0x38, 0xf4, 0x17, 0xbf, 0x43, 0xce, 0x7b, 0x2b, 0x30,
                0xb1, 0xcd, 0x40, 0x18, 0x38, 0x8e, 0x1a, 0x91, 0x0f, 0x0f, 0xc4, 0x1f, 0xb0, 0x87,
                0x7a, 0x59, 0x25, 0xe4, 0x66, 0x81, 0x9d, 0x37, 0x5b, 0x0a, 0x91, 0x2d, 0x4f, 0xe8,
                0x43, 0xb7, 0x6e, 0xf6, 0xf2, 0x23, 0xf0, 0xf7, 0xc8, 0x94, 0xf3, 0x8f, 0x7a, 0xb7,
                0x80, 0xdf, 0xd7, 0x5f, 0x66, 0x9c, 0x8c, 0x06, 0xcf, 0xfa, 0x43, 0xeb, 0x47, 0x56,
                0x5a, 0x50, 0xe3, 0xb1, 0xfa, 0x45, 0xad, 0x61, 0xce, 0x9a, 0x1c, 0x47, 0x27, 0xb7,
                0xaa, 0xa5, 0x35, 0x62, 0xf5, 0x23, 0xe7, 0x39, 0x52, 0xbb, 0xf3, 0x3d, 0x8a, 0x41,
                0x04, 0x07, 0x8a, 0xde, 0x3e, 0xaa, 0xa4, 0x96, 0x99, 0xa6, 0x9f, 0xdf, 0x1c, 0x5a,
                0xc7, 0x73, 0x21, 0x46, 0xee, 0x5e, 0x1d, 0x6b, 0x6c, 0xa9, 0xb9, 0x18, 0x0f, 0x96,
                0x4c, 0xc9, 0xd0, 0x87, 0x8a, 0xe1, 0x37, 0x35, 0x24, 0xd7, 0xd5, 0x10, 0xe5, 0x82,
                0x27, 0xdf, 0x6d, 0xe9, 0xd3, 0x0d, 0x27, 0x18, 0x67, 0x64, 0x01, 0x77, 0xb0, 0xf1,
                0x85, 0x6e, 0x28, 0xd5, 0xc8, 0xaf, 0xb0, 0x95, 0xef, 0x61, 0x84, 0xfe, 0xd6, 0x51,
                0x58, 0x90, 0x22, 0xee, 0xae, 0xa4, 0xc0, 0xce, 0x1f, 0xa6, 0xf0, 0x85, 0x09, 0x2b,
                0x04, 0x97, 0x94, 0x89, 0x17, 0x2b, 0x3e, 0xf8, 0x19, 0x4a, 0x79, 0x8d, 0xf5, 0x72,
                0x4d, 0x6b, 0x05, 0xf1, 0xae, 0x00, 0x00, 0x13, 0xa0, 0x8d, 0x61, 0x2b, 0xca, 0x8a,
                0x8c, 0x31, 0x44, 0x3c, 0x10, 0x34, 0x6d, 0xbf, 0x61, 0xde, 0x84, 0x75, 0xc0, 0xbb,
                0xec, 0x51, 0x04, 0xb4, 0x75, 0x56, 0xaf, 0x3d, 0x51, 0x44, 0x58, 0xe2, 0x32, 0x1d,
                0x14, 0x60, 0x71, 0x78, 0x9d, 0x23, 0x35, 0x93, 0x4a, 0x68, 0x06, 0x14, 0xe8, 0x35,
                0x62, 0xf8, 0x2d, 0xfd, 0x40, 0x5b, 0x54, 0xa4, 0x5e, 0xb3, 0x2c, 0x16, 0x54, 0x48,
                0xd4, 0xd5, 0xd6, 0x1c, 0xa2, 0x85, 0x95, 0x85, 0x36, 0x9f, 0x53, 0xf1, 0xa1, 0x37,
                0xe9, 0xe8, 0x2b, 0x67, 0xb8, 0xfd, 0xaf, 0x01, 0xbd, 0xa5, 0x4a, 0x31, 0x73, 0x11,
                0x89, 0x6a, 0xe1, 0x02, 0x80, 0xa0, 0x32, 0x44, 0x0c, 0x42, 0x0a, 0x42, 0x1e, 0x94,
                0x4d, 0x1e, 0x95, 0x2b, 0x70, 0xd5, 0x82, 0x6c, 0xd3, 0xb0, 0x8b, 0x7d, 0xb9, 0x63,
                0x0f, 0xe4, 0xfd, 0x5f, 0x22, 0x12, 0x5d, 0xe8, 0x40, 0xfc, 0xc4, 0x0b, 0x98, 0x03,
                0x8a, 0xf1, 0x1d, 0x55, 0xbe, 0x25, 0x43, 0x25, 0x97, 0xb4, 0xb6, 0x5b, 0x9e, 0xc1,
                0xc7, 0xa8, 0xbb, 0xfd, 0x05, 0x2c, 0xbf, 0x7e, 0x1c, 0x17, 0x85, 0x31, 0x49, 0x34,
                0xb2, 0x62, 0xd5, 0x85, 0x37, 0x54, 0xf1, 0xf1, 0x77, 0x71, 0xcf, 0xb7, 0x50, 0x30,
                0x72, 0x65, 0x57, 0x53,
            ],
            c_enc: [
                0x27, 0x62, 0x0d, 0xf5, 0xdb, 0x17, 0x76, 0x6e, 0xbd, 0x9a, 0x99, 0xd6, 0x9a, 0x9a,
                0x49, 0x38, 0x73, 0x21, 0xf0, 0xc9, 0x47, 0x25, 0x23, 0x57, 0xb5, 0xb7, 0x48, 0x49,
                0x38, 0xe3, 0xf5, 0xff, 0x9f, 0x32, 0x0e, 0x80, 0x6c, 0x4b, 0x54, 0xc3, 0x6e, 0x41,
                0x2e, 0x8a, 0x6e, 0xad, 0xe3, 0x5b, 0x83, 0x01, 0xf7, 0x34, 0xf3, 0x71, 0x2e, 0x6a,
                0xac, 0xe6, 0x99, 0xf5, 0x9e, 0x11, 0x2a, 0x7b, 0xf5, 0x44, 0x59, 0xdc, 0x5d, 0x9f,
                0x7b, 0xbf, 0x3b, 0x9f, 0x74, 0x6c, 0x45, 0xbb, 0xed, 0x69, 0xa4, 0x25, 0x1b, 0x29,
                0xf3, 0xd2, 0x15, 0xc2, 0x9f, 0x76, 0x1c, 0x7d, 0x05, 0x88, 0x19, 0xf0, 0xdd, 0x4e,
                0x70, 0x71, 0xb0, 0x77, 0xcb, 0x00, 0xcb, 0x93, 0x62, 0xa4, 0x98, 0x16, 0x88, 0x0d,
                0x49, 0xb7, 0x11, 0xf9, 0x20, 0x65, 0xf1, 0x53, 0x2e, 0x58, 0x18, 0x0b, 0xbd, 0xb4,
                0x91, 0xdb, 0xbb, 0x96, 0x8a, 0x09, 0xb5, 0x63, 0xce, 0x1d, 0x29, 0x87, 0x6f, 0xd0,
                0x52, 0x6d, 0x65, 0xda, 0x67, 0x27, 0xad, 0x40, 0xf9, 0x64, 0x02, 0xf9, 0x9a, 0xa5,
                0xac, 0x06, 0x42, 0x51, 0xc4, 0x65, 0xe3, 0xc7, 0xdb, 0x1f, 0xfe, 0xef, 0xac, 0xd7,
                0xf5, 0x1b, 0xa4, 0xf1, 0x9c, 0x6c, 0x17, 0x87, 0xa0, 0xff, 0xb4, 0x9d, 0xbb, 0x7b,
                0x0a, 0x2b, 0x15, 0x64, 0x03, 0xd6, 0x6c, 0x22, 0x13, 0xe5, 0x1d, 0x58, 0xea, 0xef,
                0xe8, 0x6d, 0x5d, 0xef, 0x0d, 0x24, 0xb7, 0xf2, 0xf8, 0xc1, 0x10, 0x32, 0x9a, 0x3a,
                0x73, 0xcb, 0x33, 0x50, 0x14, 0x0e, 0x6b, 0xf7, 0x2c, 0xb6, 0xaa, 0x22, 0x2d, 0xef,
                0x5a, 0x47, 0xe2, 0x1a, 0xf0, 0xb9, 0xae, 0xeb, 0x74, 0x8c, 0x01, 0xc6, 0x7a, 0xb6,
                0xc9, 0xfd, 0x6e, 0x53, 0x6a, 0x0d, 0x92, 0x76, 0x61, 0x51, 0xb1, 0xac, 0x7c, 0xc9,
                0x85, 0x5c, 0xa9, 0x8d, 0xea, 0x74, 0x85, 0x14, 0xef, 0xee, 0x89, 0xe8, 0x9a, 0x01,
                0x68, 0xf5, 0xdd, 0xf4, 0xac, 0x2b, 0x7c, 0xe1, 0xc9, 0xc2, 0x92, 0xfb, 0xef, 0x2f,
                0x45, 0x51, 0xa8, 0x88, 0xc3, 0x34, 0x5c, 0x65, 0x92, 0x30, 0x39, 0xfc, 0x21, 0xf7,
                0x31, 0x55, 0x9b, 0xd9, 0x24, 0xbc, 0x2c, 0x15, 0x5b, 0xc0, 0xbe, 0x80, 0x38, 0x4a,
                0x9e, 0x49, 0xbd, 0xa6, 0x9a, 0x70, 0x6b, 0x1a, 0xd6, 0xa2, 0x62, 0xab, 0xc6, 0x26,
                0x50, 0x77, 0x2f, 0xd7, 0xea, 0xbc, 0x3f, 0x75, 0xa9, 0xac, 0xca, 0xa2, 0x8b, 0xcd,
                0xea, 0x65, 0xf9, 0x4e, 0x16, 0xcc, 0x3d, 0x05, 0x38, 0xe5, 0x49, 0x86, 0x0a, 0x60,
                0x12, 0x5b, 0xb4, 0xbc, 0x0c, 0x23, 0xe3, 0x22, 0x27, 0x68, 0x2c, 0x09, 0xb5, 0xaa,
                0x30, 0x4a, 0x16, 0x09, 0x2a, 0xd4, 0xa3, 0xe2, 0xf6, 0x28, 0x3c, 0x38, 0x51, 0x80,
                0x6e, 0x72, 0x17, 0x3f, 0x7d, 0x32, 0x97, 0xed, 0x92, 0xe5, 0x32, 0x40, 0x39, 0xa7,
                0x31, 0x4f, 0x5f, 0xb7, 0x38, 0x6e, 0x09, 0x94, 0xf5, 0x2f, 0x8c, 0xcc, 0xf1, 0x87,
                0xd6, 0x20, 0x41, 0x0c, 0xce, 0x9d, 0x0b, 0x91, 0x93, 0xac, 0xec, 0x6d, 0x4c, 0x9b,
                0xd3, 0x4e, 0x08, 0x80, 0x58, 0x0a, 0xbe, 0xae, 0xd9, 0x7c, 0xb7, 0x80, 0x0f, 0x6a,
                0xbc, 0x67, 0xc2, 0x5c, 0x49, 0x19, 0x2e, 0x37, 0xdc, 0xf3, 0x3d, 0x1a, 0x59, 0x16,
                0x47, 0x5a, 0xe9, 0x99, 0x90, 0xd8, 0x29, 0xc1, 0xd5, 0x9e, 0x69, 0x2f, 0x47, 0x36,
                0x93, 0xbc, 0xe3, 0x58, 0x5a, 0xec, 0xd3, 0xc1, 0x3b, 0xae, 0x15, 0xcb, 0xef, 0xf2,
                0x98, 0x52, 0x2a, 0xab, 0xf4, 0x6b, 0xea, 0x3a, 0xbf, 0x63, 0x30, 0xa5, 0x6e, 0x37,
                0x24, 0x51, 0x81, 0x32, 0xce, 0x94, 0x39, 0x41, 0x6a, 0x28, 0xe9, 0x52, 0x0d, 0xdf,
                0x64, 0x17, 0x00, 0xb4, 0x6f, 0x37, 0x49, 0x50, 0xf3, 0x27, 0xaf, 0x3d, 0x0b, 0x3d,
                0x3b, 0x3f, 0x61, 0xa8, 0x84, 0xcf, 0x4f, 0x82, 0x02, 0x56, 0xfb, 0x91, 0x65, 0xdc,
                0xa0, 0xe4, 0x32, 0x60, 0xfc, 0xb5, 0x63, 0xef, 0x1a, 0xb4, 0xe7, 0x12, 0xef, 0x07,
                0x23, 0xd6, 0x75, 0x90, 0xa4, 0xff, 0xc3, 0x66, 0xc4, 0xa7, 0x92, 0x50, 0x29, 0x93,
                0x1b, 0xf0, 0x87, 0x3d, 0xac, 0xaa, 0xe9, 0x38, 0x5d, 0x9a, 0xd9, 0x1a, 0xed, 0x75,
                0x93, 0x9d, 0x8b, 0xd1, 0xaf, 0x5d,
            ],
            ock: [
                0x2a, 0xec, 0x11, 0xd4, 0xae, 0x79, 0x84, 0xe1, 0x69, 0xd1, 0xdf, 0xf1, 0xff, 0x0f,
                0x9a, 0xf5, 0x19, 0x96, 0x34, 0x51, 0xa4, 0x1c, 0x9f, 0x5a, 0xdc, 0x58, 0xe4, 0xf9,
                0x0a, 0xf3, 0x8d, 0x47,
            ],
            op: [
                0x78, 0xa4, 0xe3, 0x39, 0x88, 0xd7, 0x1d, 0x71, 0x8e, 0x59, 0x55, 0x55, 0x28, 0x4c,
                0x24, 0x9a, 0x62, 0xb7, 0x12, 0x88, 0x06, 0xa5, 0x4c, 0x3b, 0x36, 0xa3, 0xaa, 0x57,
                0x14, 0x93, 0x16, 0x36, 0x34, 0x39, 0x22, 0x52, 0xc9, 0xcc, 0x9f, 0x45, 0x4b, 0x54,
                0x2c, 0xf1, 0xb8, 0x88, 0xb3, 0xab, 0x02, 0xe6, 0x05, 0xa8, 0xda, 0x26, 0x10, 0x7d,
                0x98, 0x02, 0xf1, 0x53, 0x6a, 0x9e, 0x9f, 0x2b,
            ],
            c_out: [
                0xe6, 0xeb, 0x22, 0x47, 0xc9, 0x33, 0xe2, 0x4c, 0x9d, 0xf1, 0x28, 0xb1, 0xe0, 0x4e,
                0x8b, 0xc0, 0x5c, 0x65, 0xeb, 0x31, 0x97, 0xdf, 0x9b, 0xa8, 0x70, 0xd8, 0xa0, 0xa1,
                0x8d, 0x9c, 0x24, 0xb7, 0xc9, 0x78, 0xc3, 0x4d, 0x3c, 0x7b, 0xbd, 0x21, 0xe8, 0x7b,
                0x22, 0x39, 0x21, 0x87, 0x54, 0xd9, 0x67, 0x3a, 0x56, 0xa2, 0x73, 0x58, 0x0f, 0x6b,
                0x41, 0xc6, 0x91, 0xe5, 0xfd, 0x30, 0x9c, 0xd5, 0xd1, 0xd2, 0x6d, 0x57, 0x63, 0xa9,
                0xe8, 0xd2, 0x71, 0xc9, 0x77, 0x05, 0x0e, 0x05, 0xdc, 0x96,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0x0b, 0xb5, 0x6c, 0x49, 0xc0, 0x63, 0x2d, 0x4c, 0xc7, 0xe4, 0x85, 0x51, 0xdb, 0x46,
                0x42, 0x8f, 0x1b, 0x1a, 0x52, 0x66, 0x1e, 0x07, 0xe0, 0xc3, 0xbc, 0xc2, 0x31, 0x74,
                0xcc, 0xbb, 0xbd, 0xa1, 0xfa, 0x19, 0x24, 0xf4, 0x16, 0xcd, 0x48, 0x39, 0x0e, 0x2b,
                0x11, 0xc6, 0xe7, 0x82, 0x56, 0xd4, 0xc4, 0xc5, 0x64, 0x1a, 0xca, 0xd9, 0xa2, 0x0c,
                0x24, 0xfb, 0xe6, 0xcb, 0x4e, 0xe7, 0x81, 0x25,
            ],
            ovk: [
                0x21, 0xe9, 0x1a, 0x3c, 0x4a, 0xa3, 0xf2, 0x7f, 0xa1, 0xb6, 0x33, 0x96, 0xe2, 0xb4,
                0x1d, 0xb9, 0x08, 0xfd, 0xab, 0x8b, 0x18, 0xcc, 0x73, 0x04, 0xe9, 0x4e, 0x97, 0x05,
                0x68, 0xf9, 0x42, 0x1c,
            ],
            default_d: [
                0xe0, 0x66, 0xb5, 0xe7, 0x96, 0x86, 0xe9, 0xf3, 0x6e, 0xce, 0xc7,
            ],
            default_pk_d: [
                0x3b, 0x3e, 0x88, 0x3e, 0x95, 0x8c, 0xd6, 0xe0, 0x75, 0x4d, 0x74, 0xca, 0xae, 0x1e,
                0x5a, 0x43, 0x98, 0xab, 0xeb, 0x7d, 0x10, 0xee, 0x5f, 0x75, 0xa4, 0xab, 0x8e, 0xf7,
                0x03, 0x8e, 0x3d, 0xb3,
            ],
            v: 12119135386131850622,
            rseed: [
                0xc3, 0x6d, 0xcf, 0xd3, 0x4a, 0x0c, 0xb6, 0x63, 0x78, 0x76, 0x10, 0x5e, 0x79, 0xbf,
                0x3b, 0xd5, 0x8e, 0xc1, 0x48, 0xcb, 0x64, 0x97, 0x0e, 0x32, 0x23, 0xa9, 0x1f, 0x71,
                0xdf, 0xcf, 0xd5, 0xa0,
            ],
            memo: [
                0xff, 0x4b, 0x66, 0x7f, 0xba, 0xf3, 0xd4, 0xb3, 0xb9, 0x08, 0xb9, 0x82, 0x88, 0x20,
                0xdf, 0xec, 0xdd, 0x75, 0x37, 0x50, 0xb5, 0xf9, 0xd2, 0x21, 0x6e, 0x56, 0xc6, 0x15,
                0x27, 0x2f, 0x85, 0x44, 0x64, 0xc0, 0xca, 0x4b, 0x1e, 0x85, 0xae, 0xdd, 0x03, 0x82,
                0x92, 0xc4, 0xe1, 0xa5, 0x77, 0x44, 0xeb, 0xba, 0x01, 0x0b, 0x9e, 0xbf, 0xbb, 0x01,
                0x1b, 0xd6, 0xf0, 0xb7, 0x88, 0x05, 0x02, 0x5d, 0x27, 0xf3, 0xc1, 0x77, 0x46, 0xba,
                0xe1, 0x16, 0xc1, 0x5d, 0x9f, 0x47, 0x1f, 0x0f, 0x62, 0x88, 0xa1, 0x50, 0x64, 0x7b,
                0x2a, 0xfe, 0x9d, 0xf7, 0xcc, 0xcf, 0x01, 0xf5, 0xcd, 0xe5, 0xf0, 0x46, 0x80, 0xbb,
                0xfe, 0xd8, 0x7f, 0x6c, 0xf4, 0x29, 0xfb, 0x27, 0xad, 0x6b, 0xab, 0xe7, 0x91, 0x76,
                0x66, 0x11, 0xcf, 0x5b, 0xc2, 0x0e, 0x48, 0xbe, 0xf1, 0x19, 0x25, 0x9b, 0x9b, 0x8a,
                0x0e, 0x39, 0xc3, 0xdf, 0x28, 0xcb, 0x95, 0x82, 0xea, 0x33, 0x86, 0x01, 0xcd, 0xc4,
                0x81, 0xb3, 0x2f, 0xb8, 0x2a, 0xde, 0xeb, 0xb3, 0xda, 0xde, 0x25, 0xd1, 0xa3, 0xdf,
                0x20, 0xc3, 0x7e, 0x71, 0x25, 0x06, 0xb5, 0xd9, 0x96, 0xc4, 0x9a, 0x9f, 0x0f, 0x30,
                0xdd, 0xcb, 0x91, 0xfe, 0x90, 0x04, 0xe1, 0xe8, 0x32, 0x94, 0xa6, 0xc9, 0x20, 0x3d,
                0x94, 0xe8, 0xdc, 0x2c, 0xbb, 0x44, 0x9d, 0xe4, 0x15, 0x50, 0x32, 0x60, 0x4e, 0x47,
                0x99, 0x70, 0x16, 0xb3, 0x04, 0xfd, 0x43, 0x7d, 0x82, 0x35, 0x04, 0x5e, 0x25, 0x5a,
                0x19, 0xb7, 0x43, 0xa0, 0xa9, 0xf2, 0xe3, 0x36, 0xb4, 0x4c, 0xae, 0x30, 0x7b, 0xb3,
                0x98, 0x7b, 0xd3, 0xe4, 0xe7, 0x77, 0xfb, 0xb3, 0x4c, 0x0a, 0xb8, 0xcc, 0x3d, 0x67,
                0x46, 0x6c, 0x0a, 0x88, 0xdd, 0x4c, 0xca, 0xd1, 0x8a, 0x07, 0xa8, 0xd1, 0x06, 0x8d,
                0xf5, 0xb6, 0x29, 0xe5, 0x71, 0x8d, 0x0f, 0x6d, 0xf5, 0xc9, 0x57, 0xcf, 0x71, 0xbb,
                0x00, 0xa5, 0x17, 0x8f, 0x17, 0x5c, 0xac, 0xa9, 0x44, 0xe6, 0x35, 0xc5, 0x15, 0x9f,
                0x73, 0x8e, 0x24, 0x02, 0xa2, 0xd2, 0x1a, 0xa0, 0x81, 0xe1, 0x0e, 0x45, 0x6a, 0xfb,
                0x00, 0xb9, 0xf6, 0x24, 0x16, 0xc8, 0xb9, 0xc0, 0xf7, 0x22, 0x8f, 0x51, 0x07, 0x29,
                0xe0, 0xbe, 0x3f, 0x30, 0x53, 0x13, 0xd7, 0x7f, 0x73, 0x79, 0xdc, 0x2a, 0xf2, 0x48,
                0x69, 0xc6, 0xc7, 0x4e, 0xe4, 0x47, 0x14, 0x98, 0x86, 0x1d, 0x19, 0x2f, 0x0f, 0xf0,
                0xf5, 0x08, 0x28, 0x5d, 0xab, 0x6b, 0x6a, 0x36, 0xcc, 0xf7, 0xd1, 0x22, 0x56, 0xcc,
                0x76, 0xb9, 0x55, 0x03, 0x72, 0x0a, 0xc6, 0x72, 0xd0, 0x82, 0x68, 0xd2, 0xcf, 0x77,
                0x73, 0xb6, 0xba, 0x2a, 0x5f, 0x66, 0x48, 0x47, 0xbf, 0x70, 0x7f, 0x2f, 0xc1, 0x0c,
                0x98, 0xf2, 0xf0, 0x06, 0xec, 0x22, 0xcc, 0xb5, 0xa8, 0xc8, 0xb7, 0xc4, 0x0c, 0x7c,
                0x2d, 0x49, 0xa6, 0x63, 0x9b, 0x9f, 0x2c, 0xe3, 0x3c, 0x25, 0xc0, 0x4b, 0xc4, 0x61,
                0xe7, 0x44, 0xdf, 0xa5, 0x36, 0xb0, 0x0d, 0x94, 0xba, 0xdd, 0xf4, 0xf4, 0xd1, 0x40,
                0x44, 0xc6, 0x95, 0xa3, 0x38, 0x81, 0x47, 0x7d, 0xf1, 0x24, 0xf0, 0xfc, 0xf2, 0x06,
                0xa9, 0xfb, 0x2e, 0x65, 0xe3, 0x04, 0xcd, 0xbf, 0x0c, 0x4d, 0x23, 0x90, 0x17, 0x0c,
                0x13, 0x0a, 0xb8, 0x49, 0xc2, 0xf2, 0x2b, 0x5c, 0xdd, 0x39, 0x21, 0x64, 0x0c, 0x8c,
                0xf1, 0x97, 0x6a, 0xe1, 0x01, 0x0b, 0x0d, 0xfd, 0x9c, 0xb2, 0x54, 0x3e, 0x45, 0xf9,
                0x97, 0x49, 0xcc, 0x4d, 0x61, 0xf2, 0xe8, 0xaa, 0xbf, 0xe9, 0x8b, 0xd9, 0x05, 0xfa,
                0x39, 0x95, 0x1b, 0x33, 0xea, 0x76, 0x9c, 0x45, 0xab, 0x95, 0x31, 0xc5, 0x72, 0x09,
                0x86, 0x2a, 0xd1, 0x2f, 0xd7, 0x6b, 0xa4, 0x80,
            ],
            cv_net: [
                0xca, 0xf6, 0x40, 0x8d, 0xef, 0x1f, 0x0f, 0x2b, 0xaa, 0x17, 0xb1, 0x30, 0xc3, 0xae,
                0x72, 0x95, 0x89, 0xbe, 0x69, 0xd8, 0x28, 0xbe, 0x54, 0x30, 0x69, 0x16, 0x41, 0x3c,
                0xd2, 0x50, 0x21, 0x17,
            ],
            rho: [
                0x23, 0x39, 0xa8, 0x95, 0x29, 0xcf, 0x35, 0x7b, 0x06, 0x7d, 0xd2, 0x8b, 0xe4, 0x06,
                0x6e, 0x16, 0x23, 0x6d, 0xc5, 0xd7, 0x87, 0x06, 0x14, 0x9a, 0x72, 0x8c, 0x3e, 0x3d,
                0x9d, 0xc1, 0x08, 0x1c,
            ],
            cmx: [
                0xe3, 0x04, 0xcf, 0x08, 0xc7, 0x86, 0xaf, 0xcb, 0x1d, 0x65, 0x2f, 0x7a, 0x94, 0x17,
                0x29, 0x32, 0x01, 0x51, 0x71, 0x37, 0x93, 0x82, 0x89, 0x42, 0xcb, 0x88, 0xd0, 0x1f,
                0x68, 0x60, 0x4e, 0x1f,
            ],
            esk: [
                0x0f, 0xba, 0x68, 0x76, 0xb3, 0x76, 0x5d, 0xff, 0x4f, 0x4c, 0x40, 0xaf, 0x89, 0x3e,
                0x4b, 0xd1, 0x34, 0x3a, 0x53, 0x4c, 0xdf, 0x50, 0x20, 0x15, 0x6b, 0x5e, 0xd1, 0xf6,
                0xbf, 0x62, 0xbe, 0x3c,
            ],
            ephemeral_key: [
                0xdf, 0xe0, 0xf0, 0x0e, 0xb2, 0xb2, 0xf7, 0x08, 0x04, 0x19, 0x1b, 0x5b, 0x2d, 0xdf,
                0x57, 0x8f, 0x6a, 0xda, 0x21, 0x1b, 0x8b, 0x41, 0x7d, 0x95, 0xd2, 0xb7, 0x34, 0xff,
                0xa2, 0x2d, 0x81, 0x30,
            ],
            shared_secret: [
                0x0e, 0xc6, 0x82, 0xc8, 0xdc, 0x66, 0x41, 0x41, 0x02, 0x4f, 0x92, 0x9d, 0x1a, 0xe5,
                0xd3, 0x9c, 0x39, 0x4d, 0x05, 0xa2, 0x4d, 0xad, 0x0e, 0x4a, 0x6f, 0x94, 0x98, 0xed,
                0x5f, 0x87, 0xd4, 0x88,
            ],
            k_enc: [
                0xb1, 0x24, 0x5f, 0xb6, 0x37, 0x47, 0xc0, 0x13, 0x39, 0xcd, 0xe8, 0xf9, 0x35, 0x98,
                0xdf, 0xb2, 0xd3, 0x99, 0xec, 0x5e, 0x03, 0xbe, 0xa1, 0x3f, 0x3d, 0xbf, 0x01, 0xca,
                0xdb, 0x93, 0xc9, 0xba,
            ],
            p_enc: [
                0x02, 0xe0, 0x66, 0xb5, 0xe7, 0x96, 0x86, 0xe9, 0xf3, 0x6e, 0xce, 0xc7, 0x7e, 0x65,
                0x41, 0x7b, 0x6c, 0xd1, 0x2f, 0xa8, 0xc3, 0x6d, 0xcf, 0xd3, 0x4a, 0x0c, 0xb6, 0x63,
                0x78, 0x76, 0x10, 0x5e, 0x79, 0xbf, 0x3b, 0xd5, 0x8e, 0xc1, 0x48, 0xcb, 0x64, 0x97,
                0x0e, 0x32, 0x23, 0xa9, 0x1f, 0x71, 0xdf, 0xcf, 0xd5, 0xa0, 0xff, 0x4b, 0x66, 0x7f,
                0xba, 0xf3, 0xd4, 0xb3, 0xb9, 0x08, 0xb9, 0x82, 0x88, 0x20, 0xdf, 0xec, 0xdd, 0x75,
                0x37, 0x50, 0xb5, 0xf9, 0xd2, 0x21, 0x6e, 0x56, 0xc6, 0x15, 0x27, 0x2f, 0x85, 0x44,
                0x64, 0xc0, 0xca, 0x4b, 0x1e, 0x85, 0xae, 0xdd, 0x03, 0x82, 0x92, 0xc4, 0xe1, 0xa5,
                0x77, 0x44, 0xeb, 0xba, 0x01, 0x0b, 0x9e, 0xbf, 0xbb, 0x01, 0x1b, 0xd6, 0xf0, 0xb7,
                0x88, 0x05, 0x02, 0x5d, 0x27, 0xf3, 0xc1, 0x77, 0x46, 0xba, 0xe1, 0x16, 0xc1, 0x5d,
                0x9f, 0x47, 0x1f, 0x0f, 0x62, 0x88, 0xa1, 0x50, 0x64, 0x7b, 0x2a, 0xfe, 0x9d, 0xf7,
                0xcc, 0xcf, 0x01, 0xf5, 0xcd, 0xe5, 0xf0, 0x46, 0x80, 0xbb, 0xfe, 0xd8, 0x7f, 0x6c,
                0xf4, 0x29, 0xfb, 0x27, 0xad, 0x6b, 0xab, 0xe7, 0x91, 0x76, 0x66, 0x11, 0xcf, 0x5b,
                0xc2, 0x0e, 0x48, 0xbe, 0xf1, 0x19, 0x25, 0x9b, 0x9b, 0x8a, 0x0e, 0x39, 0xc3, 0xdf,
                0x28, 0xcb, 0x95, 0x82, 0xea, 0x33, 0x86, 0x01, 0xcd, 0xc4, 0x81, 0xb3, 0x2f, 0xb8,
                0x2a, 0xde, 0xeb, 0xb3, 0xda, 0xde, 0x25, 0xd1, 0xa3, 0xdf, 0x20, 0xc3, 0x7e, 0x71,
                0x25, 0x06, 0xb5, 0xd9, 0x96, 0xc4, 0x9a, 0x9f, 0x0f, 0x30, 0xdd, 0xcb, 0x91, 0xfe,
                0x90, 0x04, 0xe1, 0xe8, 0x32, 0x94, 0xa6, 0xc9, 0x20, 0x3d, 0x94, 0xe8, 0xdc, 0x2c,
                0xbb, 0x44, 0x9d, 0xe4, 0x15, 0x50, 0x32, 0x60, 0x4e, 0x47, 0x99, 0x70, 0x16, 0xb3,
                0x04, 0xfd, 0x43, 0x7d, 0x82, 0x35, 0x04, 0x5e, 0x25, 0x5a, 0x19, 0xb7, 0x43, 0xa0,
                0xa9, 0xf2, 0xe3, 0x36, 0xb4, 0x4c, 0xae, 0x30, 0x7b, 0xb3, 0x98, 0x7b, 0xd3, 0xe4,
                0xe7, 0x77, 0xfb, 0xb3, 0x4c, 0x0a, 0xb8, 0xcc, 0x3d, 0x67, 0x46, 0x6c, 0x0a, 0x88,
                0xdd, 0x4c, 0xca, 0xd1, 0x8a, 0x07, 0xa8, 0xd1, 0x06, 0x8d, 0xf5, 0xb6, 0x29, 0xe5,
                0x71, 0x8d, 0x0f, 0x6d, 0xf5, 0xc9, 0x57, 0xcf, 0x71, 0xbb, 0x00, 0xa5, 0x17, 0x8f,
                0x17, 0x5c, 0xac, 0xa9, 0x44, 0xe6, 0x35, 0xc5, 0x15, 0x9f, 0x73, 0x8e, 0x24, 0x02,
                0xa2, 0xd2, 0x1a, 0xa0, 0x81, 0xe1, 0x0e, 0x45, 0x6a, 0xfb, 0x00, 0xb9, 0xf6, 0x24,
                0x16, 0xc8, 0xb9, 0xc0, 0xf7, 0x22, 0x8f, 0x51, 0x07, 0x29, 0xe0, 0xbe, 0x3f, 0x30,
                0x53, 0x13, 0xd7, 0x7f, 0x73, 0x79, 0xdc, 0x2a, 0xf2, 0x48, 0x69, 0xc6, 0xc7, 0x4e,
                0xe4, 0x47, 0x14, 0x98, 0x86, 0x1d, 0x19, 0x2f, 0x0f, 0xf0, 0xf5, 0x08, 0x28, 0x5d,
                0xab, 0x6b, 0x6a, 0x36, 0xcc, 0xf7, 0xd1, 0x22, 0x56, 0xcc, 0x76, 0xb9, 0x55, 0x03,
                0x72, 0x0a, 0xc6, 0x72, 0xd0, 0x82, 0x68, 0xd2, 0xcf, 0x77, 0x73, 0xb6, 0xba, 0x2a,
                0x5f, 0x66, 0x48, 0x47, 0xbf, 0x70, 0x7f, 0x2f, 0xc1, 0x0c, 0x98, 0xf2, 0xf0, 0x06,
                0xec, 0x22, 0xcc, 0xb5, 0xa8, 0xc8, 0xb7, 0xc4, 0x0c, 0x7c, 0x2d, 0x49, 0xa6, 0x63,
                0x9b, 0x9f, 0x2c, 0xe3, 0x3c, 0x25, 0xc0, 0x4b, 0xc4, 0x61, 0xe7, 0x44, 0xdf, 0xa5,
                0x36, 0xb0, 0x0d, 0x94, 0xba, 0xdd, 0xf4, 0xf4, 0xd1, 0x40, 0x44, 0xc6, 0x95, 0xa3,
                0x38, 0x81, 0x47, 0x7d, 0xf1, 0x24, 0xf0, 0xfc, 0xf2, 0x06, 0xa9, 0xfb, 0x2e, 0x65,
                0xe3, 0x04, 0xcd, 0xbf, 0x0c, 0x4d, 0x23, 0x90, 0x17, 0x0c, 0x13, 0x0a, 0xb8, 0x49,
                0xc2, 0xf2, 0x2b, 0x5c, 0xdd, 0x39, 0x21, 0x64, 0x0c, 0x8c, 0xf1, 0x97, 0x6a, 0xe1,
                0x01, 0x0b, 0x0d, 0xfd, 0x9c, 0xb2, 0x54, 0x3e, 0x45, 0xf9, 0x97, 0x49, 0xcc, 0x4d,
                0x61, 0xf2, 0xe8, 0xaa, 0xbf, 0xe9, 0x8b, 0xd9, 0x05, 0xfa, 0x39, 0x95, 0x1b, 0x33,
                0xea, 0x76, 0x9c, 0x45, 0xab, 0x95, 0x31, 0xc5, 0x72, 0x09, 0x86, 0x2a, 0xd1, 0x2f,
                0xd7, 0x6b, 0xa4, 0x80,
            ],
            c_enc: [
                0x31, 0xec, 0x78, 0xd4, 0x02, 0x14, 0xc1, 0xbc, 0x7b, 0x25, 0x94, 0x7b, 0x42, 0xf3,
                0x54, 0x34, 0x60, 0x72, 0x90, 0x27, 0x7c, 0x44, 0xc4, 0x89, 0xe9, 0x38, 0x72, 0x2f,
                0x7c, 0x12, 0xff, 0x55, 0xc4, 0xf1, 0xdf, 0xbe, 0x0f, 0x1a, 0x65, 0xb1, 0x5c, 0x0c,
                0xc4, 0x7a, 0x46, 0xc9, 0xbb, 0x33, 0x5b, 0xd1, 0xa3, 0xe7, 0xe6, 0x62, 0xee, 0x85,
                0x84, 0x57, 0x6b, 0x6f, 0x42, 0x81, 0x95, 0xc9, 0x8c, 0x72, 0xfb, 0x99, 0x75, 0xc7,
                0x17, 0x9c, 0x4c, 0xab, 0x58, 0x66, 0xf5, 0xb4, 0x85, 0x42, 0x8b, 0x50, 0x8e, 0x2e,
                0x05, 0xbe, 0x0c, 0x98, 0x41, 0xa3, 0xa2, 0x91, 0x12, 0x80, 0x2c, 0x52, 0x15, 0xe2,
                0x51, 0x37, 0xcf, 0xdd, 0x13, 0x8c, 0x99, 0x39, 0x0e, 0xfb, 0xce, 0xb7, 0x52, 0x4e,
                0x84, 0x7c, 0x36, 0xb4, 0x3f, 0x3e, 0xde, 0x32, 0xf2, 0x3b, 0x39, 0x81, 0xf6, 0xd3,
                0xc3, 0x2b, 0x17, 0xbb, 0xbe, 0x65, 0xf2, 0xfd, 0x01, 0xab, 0xdd, 0x99, 0x87, 0xf1,
                0xb8, 0x88, 0xeb, 0xef, 0x41, 0xef, 0x45, 0x55, 0xc9, 0xfb, 0x0c, 0x8a, 0xee, 0x9b,
                0x6d, 0xd0, 0x85, 0xb9, 0xfb, 0xc7, 0x4f, 0xe4, 0x6a, 0x1e, 0x36, 0x3d, 0x68, 0x32,
                0xb5, 0x08, 0x20, 0x0e, 0x50, 0x0d, 0xa6, 0x95, 0x13, 0xb7, 0x9e, 0x81, 0xf7, 0x33,
                0xb6, 0x6c, 0x3f, 0x24, 0x3a, 0x28, 0x5b, 0xf8, 0x6f, 0xfe, 0xef, 0x4f, 0xfd, 0x1c,
                0xac, 0xa7, 0x83, 0x07, 0x26, 0x63, 0xbf, 0x9b, 0x48, 0x7f, 0xbc, 0x3e, 0x7f, 0x5a,
                0xd2, 0xf7, 0xac, 0xbf, 0x2b, 0x6e, 0x00, 0x8e, 0x3e, 0x4d, 0x6f, 0x31, 0xe1, 0x14,
                0x75, 0xe1, 0x96, 0x98, 0x9d, 0xb9, 0x62, 0x8a, 0x5d, 0x56, 0x39, 0x7c, 0x9a, 0x04,
                0x2a, 0xab, 0x55, 0xe1, 0xec, 0xc4, 0x92, 0xfe, 0x94, 0x27, 0xd4, 0x90, 0xf2, 0x73,
                0xb6, 0x01, 0xd6, 0x51, 0x05, 0x56, 0x82, 0xd3, 0x5b, 0x30, 0x75, 0xfa, 0xda, 0x85,
                0x25, 0x84, 0x48, 0x14, 0x95, 0x7f, 0xfc, 0x9b, 0xc7, 0xfb, 0x39, 0x39, 0x73, 0x7b,
                0x01, 0x50, 0x2e, 0x0b, 0x6f, 0x1f, 0x9b, 0x88, 0xf2, 0x71, 0x54, 0x80, 0xae, 0x42,
                0x2e, 0x9b, 0xb7, 0xb7, 0x6e, 0x5d, 0x22, 0xde, 0x0d, 0x3d, 0x7a, 0xea, 0x58, 0x10,
                0x01, 0xdc, 0xf4, 0x6a, 0x62, 0x2f, 0x08, 0x03, 0x10, 0xbe, 0x50, 0xdf, 0x07, 0x75,
                0x21, 0x92, 0xd4, 0xe4, 0x1a, 0xc5, 0x18, 0xe4, 0x24, 0x1e, 0x06, 0x67, 0x76, 0xa8,
                0xea, 0xec, 0xc7, 0x42, 0xfd, 0x2c, 0x1b, 0xf0, 0x6f, 0xc5, 0x8b, 0xd2, 0x8d, 0x1d,
                0x6c, 0x60, 0x1f, 0x91, 0x5d, 0x75, 0x2e, 0x7c, 0xc3, 0xd9, 0x76, 0x3c, 0x8b, 0x9e,
                0xec, 0x14, 0x2c, 0x84, 0x81, 0xf9, 0xc5, 0x23, 0x59, 0xbf, 0xbf, 0x51, 0x24, 0x36,
                0x67, 0x84, 0xe1, 0x71, 0xd7, 0xa4, 0xaa, 0x01, 0x5e, 0x85, 0x56, 0x47, 0x4a, 0x6d,
                0x0f, 0xee, 0x69, 0xb0, 0xd5, 0x3e, 0xe7, 0xf4, 0xed, 0xf5, 0xea, 0x59, 0xfa, 0x55,
                0x97, 0xf4, 0x8d, 0xd6, 0xaa, 0x88, 0x51, 0x84, 0x29, 0xac, 0x8c, 0x18, 0xc4, 0x73,
                0x8d, 0xb8, 0x01, 0x09, 0xa5, 0xe4, 0xbc, 0x81, 0xbb, 0x84, 0xb3, 0x7d, 0x9b, 0x2a,
                0xd4, 0xb1, 0xd2, 0x6b, 0x36, 0x3d, 0x16, 0x60, 0x72, 0xf0, 0x8c, 0x56, 0x70, 0x62,
                0x27, 0xa9, 0xf8, 0x32, 0x97, 0x03, 0x70, 0x12, 0xd1, 0xf6, 0xda, 0x24, 0xa8, 0x69,
                0xeb, 0x32, 0x97, 0xbc, 0xbc, 0xcc, 0x8d, 0x7d, 0xed, 0x03, 0x0a, 0x3d, 0x60, 0xdf,
                0x88, 0x82, 0xe0, 0x62, 0x9e, 0x36, 0x14, 0x3c, 0x09, 0xca, 0x05, 0x21, 0x43, 0xcb,
                0x56, 0x62, 0x93, 0x8f, 0xa9, 0xf1, 0x15, 0xdd, 0xc0, 0x96, 0x65, 0x4c, 0x0b, 0x40,
                0x59, 0x4c, 0xba, 0x19, 0xee, 0xd4, 0x99, 0x53, 0x2b, 0x70, 0x8a, 0xde, 0xbe, 0x47,
                0x72, 0x6a, 0x83, 0xaf, 0x46, 0x3c, 0x80, 0xba, 0x2a, 0x1f, 0xe0, 0x71, 0xe1, 0x66,
                0xc4, 0x4d, 0x70, 0xdd, 0xbb, 0xba, 0x67, 0x06, 0xa1, 0xdc, 0x43, 0x27, 0x26, 0xfe,
                0x0f, 0xdb, 0xc6, 0x28, 0x21, 0xb5, 0x04, 0xea, 0x11, 0x66, 0xda, 0x48, 0xa8, 0x1b,
                0x63, 0x5e, 0x37, 0x63, 0x33, 0xd9, 0xbe, 0xfe, 0xc4, 0x93, 0xa0, 0x7d, 0xf4, 0x7b,
                0xba, 0x0a, 0x2e, 0x2f, 0x65, 0xe7,
            ],
            ock: [
                0xec, 0xa3, 0x70, 0x7a, 0x74, 0x5d, 0x5d, 0x58, 0x65, 0x86, 0xb2, 0x35, 0xf2, 0x92,
                0xad, 0x20, 0x48, 0x93, 0x14, 0xc4, 0x58, 0x80, 0xd9, 0x83, 0x2b, 0x7f, 0x47, 0xee,
                0xbb, 0xd4, 0x5c, 0xfe,
            ],
            op: [
                0x3b, 0x3e, 0x88, 0x3e, 0x95, 0x8c, 0xd6, 0xe0, 0x75, 0x4d, 0x74, 0xca, 0xae, 0x1e,
                0x5a, 0x43, 0x98, 0xab, 0xeb, 0x7d, 0x10, 0xee, 0x5f, 0x75, 0xa4, 0xab, 0x8e, 0xf7,
                0x03, 0x8e, 0x3d, 0xb3, 0x0f, 0xba, 0x68, 0x76, 0xb3, 0x76, 0x5d, 0xff, 0x4f, 0x4c,
                0x40, 0xaf, 0x89, 0x3e, 0x4b, 0xd1, 0x34, 0x3a, 0x53, 0x4c, 0xdf, 0x50, 0x20, 0x15,
                0x6b, 0x5e, 0xd1, 0xf6, 0xbf, 0x62, 0xbe, 0x3c,
            ],
            c_out: [
                0xd0, 0x65, 0x9a, 0x03, 0xc5, 0x3e, 0x15, 0xdc, 0x11, 0x01, 0xa6, 0x6e, 0xa8, 0x0d,
                0xf4, 0x14, 0x4d, 0x5e, 0xe0, 0x1c, 0x3d, 0xbf, 0x20, 0x2c, 0xe6, 0x08, 0x7e, 0x96,
                0xa8, 0x5a, 0xe1, 0x39, 0xdd, 0x46, 0xfc, 0x82, 0xa1, 0xc3, 0x7d, 0x06, 0x05, 0xf8,
                0x49, 0x98, 0x53, 0x13, 0x16, 0xf1, 0xf0, 0xcb, 0xc0, 0x4e, 0xa2, 0x0a, 0xa4, 0x04,
                0xb3, 0xd5, 0xb0, 0x42, 0x96, 0x01, 0x49, 0x81, 0x6d, 0x9b, 0x8f, 0x06, 0x2a, 0xfc,
                0xc1, 0x93, 0x89, 0x43, 0x82, 0x7a, 0x1b, 0x43, 0xfc, 0xbd,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0xeb, 0xd4, 0x80, 0x6d, 0x81, 0x25, 0x49, 0x89, 0xfa, 0xdb, 0xa8, 0xcd, 0x58, 0x96,
                0x7d, 0x6f, 0xd8, 0x73, 0x83, 0xbc, 0x09, 0x38, 0x63, 0xd5, 0xab, 0xfc, 0xdd, 0xd3,
                0x8f, 0x15, 0x39, 0xfa, 0xb7, 0xe5, 0xd4, 0xf0, 0x61, 0x91, 0x67, 0xb8, 0xd4, 0x82,
                0xcb, 0x54, 0x8c, 0xb5, 0x59, 0x83, 0x49, 0x6f, 0x77, 0xd3, 0xdc, 0xaf, 0xf5, 0x6e,
                0x32, 0x41, 0x0b, 0xfe, 0xc1, 0xf2, 0x68, 0x11,
            ],
            ovk: [
                0xb2, 0x5f, 0x30, 0x3f, 0x58, 0x15, 0xc4, 0x53, 0x31, 0x24, 0xac, 0xf9, 0xd1, 0x89,
                0x40, 0xe7, 0x75, 0x22, 0xac, 0x5d, 0xc4, 0xb9, 0x57, 0x0a, 0xae, 0x8f, 0x47, 0xb7,
                0xf5, 0x7f, 0xd8, 0x76,
            ],
            default_d: [
                0x1c, 0xa7, 0xb6, 0x49, 0x39, 0x9e, 0x13, 0xe4, 0x39, 0x44, 0x62,
            ],
            default_pk_d: [
                0x3f, 0xeb, 0x34, 0x5a, 0xec, 0xd3, 0x42, 0x9a, 0x16, 0xe1, 0x0f, 0x3d, 0x13, 0x20,
                0xbc, 0x99, 0x71, 0xb5, 0x9e, 0x63, 0x9d, 0x62, 0xb6, 0x96, 0x1a, 0xea, 0x78, 0x15,
                0x67, 0xa8, 0x60, 0x9e,
            ],
            v: 9624581763228770449,
            rseed: [
                0x4a, 0x95, 0xb2, 0x05, 0x52, 0x6c, 0xfc, 0xb4, 0xc4, 0xe1, 0xcc, 0x95, 0x51, 0x75,
                0xb3, 0xe8, 0xde, 0x1f, 0x5d, 0x81, 0xb1, 0x86, 0x69, 0x69, 0x23, 0x50, 0xaa, 0xa1,
                0xa1, 0xd7, 0x97, 0x61,
            ],
            memo: [
                0xff, 0x75, 0x82, 0xe5, 0x4d, 0x7a, 0x5b, 0x57, 0xa6, 0x83, 0xb3, 0x2f, 0xb1, 0x09,
                0x80, 0x62, 0xda, 0xd7, 0xb0, 0xc2, 0xeb, 0x51, 0x8f, 0x68, 0x62, 0xe8, 0x3d, 0xb2,
                0x5e, 0x3d, 0xba, 0xf7, 0xae, 0xd5, 0x04, 0xde, 0x93, 0x2a, 0xcb, 0x99, 0xd7, 0x35,
                0x99, 0x2c, 0xe6, 0x2b, 0xae, 0x9e, 0xf8, 0x93, 0xff, 0x6a, 0xcc, 0x0f, 0xfc, 0xf8,
                0xe3, 0x48, 0x3e, 0x14, 0x6b, 0x9d, 0x49, 0xdd, 0x8c, 0x78, 0x35, 0xf4, 0x3a, 0x37,
                0xdc, 0xa0, 0x78, 0x7e, 0x3e, 0xc9, 0xf6, 0x60, 0x52, 0x23, 0xd5, 0xba, 0x7a, 0xe0,
                0xab, 0x90, 0x25, 0xb7, 0x3b, 0xc0, 0x3f, 0x7f, 0xac, 0x36, 0xc0, 0x09, 0xa5, 0x6d,
                0x4d, 0x95, 0xd1, 0xe8, 0x1d, 0x3b, 0x3e, 0xbc, 0xa7, 0xe5, 0x4c, 0xc1, 0xa1, 0x2d,
                0x12, 0x7b, 0x57, 0xc8, 0x13, 0x89, 0x76, 0xe7, 0x91, 0x01, 0x3b, 0x01, 0x5f, 0x06,
                0xa6, 0x24, 0xf5, 0x21, 0xb6, 0xee, 0x04, 0xec, 0x98, 0x08, 0x93, 0xc7, 0xe5, 0xe0,
                0x1a, 0x33, 0x62, 0x03, 0x59, 0x40, 0x94, 0xf8, 0x28, 0x33, 0xd7, 0x44, 0x5f, 0xe2,
                0xd0, 0x91, 0x30, 0xf6, 0x35, 0x11, 0xda, 0x54, 0x83, 0x2d, 0xe9, 0x13, 0x6b, 0x39,
                0xf4, 0x59, 0x9f, 0x5a, 0xa5, 0xdf, 0xbb, 0x45, 0xda, 0x60, 0xcd, 0xce, 0xab, 0x7e,
                0xef, 0xde, 0x89, 0xbe, 0x63, 0xf3, 0xf7, 0xc0, 0xd2, 0x32, 0x48, 0x47, 0xcc, 0xe1,
                0x40, 0x5d, 0xef, 0x7c, 0x46, 0x9b, 0x0e, 0x27, 0x24, 0x94, 0xe5, 0xdf, 0x54, 0xf5,
                0x68, 0x65, 0x6c, 0xb9, 0xc8, 0x81, 0x8d, 0x92, 0xb7, 0x2b, 0x8b, 0xc3, 0x4d, 0xb7,
                0xbb, 0x31, 0x12, 0x48, 0x7e, 0x74, 0x6e, 0xef, 0xe4, 0xe8, 0x08, 0xbb, 0xb2, 0x87,
                0xd9, 0x9b, 0xf0, 0x7d, 0x00, 0xda, 0xbe, 0xde, 0xdc, 0x5e, 0x5f, 0x07, 0x4f, 0xfe,
                0xae, 0x0c, 0xba, 0x7d, 0xa3, 0xa5, 0x16, 0xc1, 0x73, 0xbe, 0x1c, 0x51, 0x33, 0x23,
                0xe1, 0x19, 0xf6, 0x35, 0xe8, 0x20, 0x9a, 0x07, 0x4b, 0x21, 0x6b, 0x70, 0x23, 0xfa,
                0xdc, 0x2d, 0x25, 0x94, 0x9c, 0x90, 0x03, 0x7e, 0x71, 0xe3, 0xe5, 0x50, 0x72, 0x6d,
                0x21, 0x0a, 0x2c, 0x68, 0x83, 0x42, 0xe5, 0x24, 0x40, 0x63, 0x5e, 0x9c, 0xc1, 0x4a,
                0xfe, 0x10, 0x10, 0x26, 0x21, 0xa9, 0xc9, 0xac, 0xcb, 0x78, 0x2e, 0x9e, 0x4a, 0x5f,
                0xa8, 0x7f, 0x0a, 0x95, 0x6f, 0x5b, 0x85, 0x50, 0x99, 0x60, 0x28, 0x5c, 0x22, 0x62,
                0x7c, 0x59, 0x48, 0x3a, 0x5a, 0x4c, 0x28, 0xcc, 0xe4, 0xb1, 0x56, 0xe5, 0x51, 0x40,
                0x6a, 0x7e, 0xe8, 0x35, 0x56, 0x56, 0xa2, 0x1e, 0x43, 0xe3, 0x8c, 0xe1, 0x29, 0xfd,
                0xad, 0xb7, 0x59, 0xed, 0xdf, 0xa0, 0x8f, 0x00, 0xfc, 0x8e, 0x56, 0x7c, 0xef, 0x93,
                0xc6, 0x79, 0x2d, 0x01, 0xdf, 0x05, 0xe6, 0xd5, 0x80, 0xf4, 0xd5, 0xd4, 0x8d, 0xf0,
                0x42, 0x45, 0x1a, 0x33, 0x59, 0x0d, 0x3e, 0x8c, 0xf4, 0x9b, 0x26, 0x27, 0x21, 0x8f,
                0x0c, 0x29, 0x2f, 0xa6, 0x6a, 0xda, 0x94, 0x5f, 0xa5, 0x5b, 0xb2, 0x35, 0x48, 0xe3,
                0x3a, 0x83, 0xa5, 0x62, 0x95, 0x7a, 0x31, 0x49, 0xa9, 0x93, 0xcc, 0x47, 0x23, 0x62,
                0x29, 0x87, 0x36, 0xa8, 0xb7, 0x78, 0xd9, 0x7c, 0xe4, 0x23, 0x01, 0x3d, 0x64, 0xb3,
                0x2c, 0xd1, 0x72, 0xef, 0xa5, 0x51, 0xbf, 0x7f, 0x36, 0x8f, 0x04, 0xbd, 0xae, 0xc6,
                0x09, 0x1a, 0x30, 0x04, 0xa7, 0x57, 0x59, 0x8b, 0x80, 0x1d, 0xcf, 0x67, 0x5c, 0xb8,
                0x3e, 0x43, 0xa5, 0x3a, 0xe8, 0xb2, 0x54, 0xd3, 0x33, 0xbc, 0xda, 0x20, 0xd4, 0x81,
                0x7d, 0x34, 0x77, 0xab, 0xfb, 0xa2, 0x5b, 0xb8, 0x3d, 0xf5, 0x94, 0x9c, 0x12, 0x6f,
                0x14, 0x9b, 0x1d, 0x99, 0x34, 0x1e, 0x4e, 0x6f,
            ],
            cv_net: [
                0xd2, 0xf9, 0xad, 0xff, 0x53, 0x1b, 0x65, 0x43, 0x2b, 0xa2, 0xd7, 0xda, 0xa6, 0xd8,
                0x6e, 0x62, 0xe4, 0xed, 0xc7, 0x86, 0xd9, 0xe0, 0xb2, 0x7d, 0x26, 0x62, 0x8b, 0x79,
                0xda, 0x6b, 0x15, 0x14,
            ],
            rho: [
                0xe5, 0xd0, 0x8c, 0x40, 0x26, 0x3e, 0x4a, 0x2a, 0x56, 0x96, 0xda, 0x21, 0x0d, 0x8e,
                0x9a, 0x77, 0xf0, 0xaf, 0xc4, 0xc6, 0x8a, 0x6d, 0xda, 0x38, 0xe2, 0x85, 0xf4, 0xe3,
                0xef, 0x13, 0xb8, 0x17,
            ],
            cmx: [
                0xad, 0xc2, 0x44, 0x3a, 0xf9, 0x57, 0x67, 0x47, 0xca, 0x4f, 0x10, 0x4b, 0x1f, 0x5a,
                0x7a, 0xdc, 0x80, 0x5a, 0xc5, 0x5a, 0xcc, 0x56, 0x33, 0xa0, 0xb3, 0xc2, 0xdc, 0x7a,
                0xad, 0xff, 0x21, 0x26,
            ],
            esk: [
                0x24, 0x75, 0x30, 0x5c, 0x35, 0xcf, 0x37, 0xeb, 0xd2, 0x77, 0x82, 0x35, 0x16, 0x58,
                0x5a, 0xfa, 0x06, 0x05, 0x78, 0x52, 0x17, 0x1e, 0x0c, 0xe4, 0xb0, 0x52, 0xd5, 0xb4,
                0x20, 0x41, 0xd8, 0x1c,
            ],
            ephemeral_key: [
                0x00, 0x5d, 0x31, 0x5b, 0xd1, 0x80, 0xa2, 0x94, 0x0b, 0xd7, 0xfb, 0x34, 0xa7, 0x0e,
                0x90, 0xe8, 0x32, 0x6c, 0xb1, 0x34, 0x30, 0x66, 0x05, 0x46, 0x53, 0xd2, 0x64, 0x4d,
                0x20, 0x45, 0xe3, 0x0f,
            ],
            shared_secret: [
                0x8e, 0x59, 0x0d, 0x06, 0x3f, 0x7e, 0xc3, 0x8e, 0xb2, 0x00, 0x84, 0x70, 0xf9, 0xbb,
                0x42, 0x29, 0x04, 0xfc, 0x0b, 0xaf, 0x97, 0x80, 0xf8, 0xfd, 0x1f, 0x54, 0x22, 0xea,
                0x30, 0x8c, 0x47, 0xba,
            ],
            k_enc: [
                0xeb, 0xb2, 0xff, 0x9b, 0x1f, 0x29, 0x0d, 0xa5, 0x63, 0x27, 0xa9, 0x67, 0x71, 0xb2,
                0x5f, 0x71, 0x29, 0xff, 0x10, 0x2e, 0xe6, 0xd2, 0xb2, 0x0d, 0xa0, 0x9a, 0x06, 0x25,
                0xf8, 0xbb, 0x26, 0x07,
            ],
            p_enc: [
                0x02, 0x1c, 0xa7, 0xb6, 0x49, 0x39, 0x9e, 0x13, 0xe4, 0x39, 0x44, 0x62, 0x91, 0x20,
                0xf4, 0xd4, 0x1e, 0x62, 0x91, 0x85, 0x4a, 0x95, 0xb2, 0x05, 0x52, 0x6c, 0xfc, 0xb4,
                0xc4, 0xe1, 0xcc, 0x95, 0x51, 0x75, 0xb3, 0xe8, 0xde, 0x1f, 0x5d, 0x81, 0xb1, 0x86,
                0x69, 0x69, 0x23, 0x50, 0xaa, 0xa1, 0xa1, 0xd7, 0x97, 0x61, 0xff, 0x75, 0x82, 0xe5,
                0x4d, 0x7a, 0x5b, 0x57, 0xa6, 0x83, 0xb3, 0x2f, 0xb1, 0x09, 0x80, 0x62, 0xda, 0xd7,
                0xb0, 0xc2, 0xeb, 0x51, 0x8f, 0x68, 0x62, 0xe8, 0x3d, 0xb2, 0x5e, 0x3d, 0xba, 0xf7,
                0xae, 0xd5, 0x04, 0xde, 0x93, 0x2a, 0xcb, 0x99, 0xd7, 0x35, 0x99, 0x2c, 0xe6, 0x2b,
                0xae, 0x9e, 0xf8, 0x93, 0xff, 0x6a, 0xcc, 0x0f, 0xfc, 0xf8, 0xe3, 0x48, 0x3e, 0x14,
                0x6b, 0x9d, 0x49, 0xdd, 0x8c, 0x78, 0x35, 0xf4, 0x3a, 0x37, 0xdc, 0xa0, 0x78, 0x7e,
                0x3e, 0xc9, 0xf6, 0x60, 0x52, 0x23, 0xd5, 0xba, 0x7a, 0xe0, 0xab, 0x90, 0x25, 0xb7,
                0x3b, 0xc0, 0x3f, 0x7f, 0xac, 0x36, 0xc0, 0x09, 0xa5, 0x6d, 0x4d, 0x95, 0xd1, 0xe8,
                0x1d, 0x3b, 0x3e, 0xbc, 0xa7, 0xe5, 0x4c, 0xc1, 0xa1, 0x2d, 0x12, 0x7b, 0x57, 0xc8,
                0x13, 0x89, 0x76, 0xe7, 0x91, 0x01, 0x3b, 0x01, 0x5f, 0x06, 0xa6, 0x24, 0xf5, 0x21,
                0xb6, 0xee, 0x04, 0xec, 0x98, 0x08, 0x93, 0xc7, 0xe5, 0xe0, 0x1a, 0x33, 0x62, 0x03,
                0x59, 0x40, 0x94, 0xf8, 0x28, 0x33, 0xd7, 0x44, 0x5f, 0xe2, 0xd0, 0x91, 0x30, 0xf6,
                0x35, 0x11, 0xda, 0x54, 0x83, 0x2d, 0xe9, 0x13, 0x6b, 0x39, 0xf4, 0x59, 0x9f, 0x5a,
                0xa5, 0xdf, 0xbb, 0x45, 0xda, 0x60, 0xcd, 0xce, 0xab, 0x7e, 0xef, 0xde, 0x89, 0xbe,
                0x63, 0xf3, 0xf7, 0xc0, 0xd2, 0x32, 0x48, 0x47, 0xcc, 0xe1, 0x40, 0x5d, 0xef, 0x7c,
                0x46, 0x9b, 0x0e, 0x27, 0x24, 0x94, 0xe5, 0xdf, 0x54, 0xf5, 0x68, 0x65, 0x6c, 0xb9,
                0xc8, 0x81, 0x8d, 0x92, 0xb7, 0x2b, 0x8b, 0xc3, 0x4d, 0xb7, 0xbb, 0x31, 0x12, 0x48,
                0x7e, 0x74, 0x6e, 0xef, 0xe4, 0xe8, 0x08, 0xbb, 0xb2, 0x87, 0xd9, 0x9b, 0xf0, 0x7d,
                0x00, 0xda, 0xbe, 0xde, 0xdc, 0x5e, 0x5f, 0x07, 0x4f, 0xfe, 0xae, 0x0c, 0xba, 0x7d,
                0xa3, 0xa5, 0x16, 0xc1, 0x73, 0xbe, 0x1c, 0x51, 0x33, 0x23, 0xe1, 0x19, 0xf6, 0x35,
                0xe8, 0x20, 0x9a, 0x07, 0x4b, 0x21, 0x6b, 0x70, 0x23, 0xfa, 0xdc, 0x2d, 0x25, 0x94,
                0x9c, 0x90, 0x03, 0x7e, 0x71, 0xe3, 0xe5, 0x50, 0x72, 0x6d, 0x21, 0x0a, 0x2c, 0x68,
                0x83, 0x42, 0xe5, 0x24, 0x40, 0x63, 0x5e, 0x9c, 0xc1, 0x4a, 0xfe, 0x10, 0x10, 0x26,
                0x21, 0xa9, 0xc9, 0xac, 0xcb, 0x78, 0x2e, 0x9e, 0x4a, 0x5f, 0xa8, 0x7f, 0x0a, 0x95,
                0x6f, 0x5b, 0x85, 0x50, 0x99, 0x60, 0x28, 0x5c, 0x22, 0x62, 0x7c, 0x59, 0x48, 0x3a,
                0x5a, 0x4c, 0x28, 0xcc, 0xe4, 0xb1, 0x56, 0xe5, 0x51, 0x40, 0x6a, 0x7e, 0xe8, 0x35,
                0x56, 0x56, 0xa2, 0x1e, 0x43, 0xe3, 0x8c, 0xe1, 0x29, 0xfd, 0xad, 0xb7, 0x59, 0xed,
                0xdf, 0xa0, 0x8f, 0x00, 0xfc, 0x8e, 0x56, 0x7c, 0xef, 0x93, 0xc6, 0x79, 0x2d, 0x01,
                0xdf, 0x05, 0xe6, 0xd5, 0x80, 0xf4, 0xd5, 0xd4, 0x8d, 0xf0, 0x42, 0x45, 0x1a, 0x33,
                0x59, 0x0d, 0x3e, 0x8c, 0xf4, 0x9b, 0x26, 0x27, 0x21, 0x8f, 0x0c, 0x29, 0x2f, 0xa6,
                0x6a, 0xda, 0x94, 0x5f, 0xa5, 0x5b, 0xb2, 0x35, 0x48, 0xe3, 0x3a, 0x83, 0xa5, 0x62,
                0x95, 0x7a, 0x31, 0x49, 0xa9, 0x93, 0xcc, 0x47, 0x23, 0x62, 0x29, 0x87, 0x36, 0xa8,
                0xb7, 0x78, 0xd9, 0x7c, 0xe4, 0x23, 0x01, 0x3d, 0x64, 0xb3, 0x2c, 0xd1, 0x72, 0xef,
                0xa5, 0x51, 0xbf, 0x7f, 0x36, 0x8f, 0x04, 0xbd, 0xae, 0xc6, 0x09, 0x1a, 0x30, 0x04,
                0xa7, 0x57, 0x59, 0x8b, 0x80, 0x1d, 0xcf, 0x67, 0x5c, 0xb8, 0x3e, 0x43, 0xa5, 0x3a,
                0xe8, 0xb2, 0x54, 0xd3, 0x33, 0xbc, 0xda, 0x20, 0xd4, 0x81, 0x7d, 0x34, 0x77, 0xab,
                0xfb, 0xa2, 0x5b, 0xb8, 0x3d, 0xf5, 0x94, 0x9c, 0x12, 0x6f, 0x14, 0x9b, 0x1d, 0x99,
                0x34, 0x1e, 0x4e, 0x6f,
            ],
            c_enc: [
                0x96, 0x3b, 0xe3, 0x6e, 0xdc, 0xa4, 0x8b, 0x7c, 0x4a, 0x1c, 0xd6, 0xa1, 0x37, 0x2e,
                0x84, 0xc0, 0x54, 0x54, 0x83, 0x4c, 0x5e, 0xe1, 0x92, 0x5e, 0xed, 0xeb, 0x48, 0xda,
                0x16, 0x77, 0x79, 0xe3, 0x99, 0x83, 0x58, 0xcb, 0x92, 0x7a, 0x0a, 0x0b, 0x36, 0x9c,
                0x85, 0xee, 0xb3, 0x30, 0xf7, 0x00, 0x5c, 0x26, 0x9d, 0x0b, 0xe4, 0xf2, 0x38, 0x09,
                0x57, 0xb1, 0x62, 0x97, 0x7c, 0xae, 0x7a, 0x7b, 0x1d, 0x0b, 0xe5, 0xd0, 0xb7, 0x03,
                0xd3, 0x92, 0xac, 0xae, 0xd2, 0x4f, 0x52, 0x5b, 0x83, 0xcf, 0xdd, 0x7f, 0x35, 0xc1,
                0xf1, 0x74, 0xbf, 0x06, 0x6b, 0x7a, 0xfd, 0xb7, 0x45, 0xed, 0xbe, 0xdc, 0x0b, 0x46,
                0xa5, 0x91, 0x36, 0x21, 0x47, 0x61, 0x9d, 0x6b, 0xd3, 0xf3, 0xce, 0x6b, 0x45, 0x23,
                0xf0, 0x64, 0x9a, 0x7c, 0x65, 0x32, 0x2d, 0x05, 0xfa, 0xe2, 0x22, 0x70, 0x1a, 0x8c,
                0xd6, 0xca, 0xbf, 0x8d, 0xc3, 0x2f, 0x05, 0x5f, 0xeb, 0x14, 0x1e, 0x55, 0x6d, 0xdf,
                0xb1, 0x98, 0x30, 0xb7, 0x20, 0x4c, 0x30, 0x98, 0x4e, 0xdd, 0xf4, 0x34, 0xec, 0xc5,
                0xf0, 0xea, 0x82, 0x5c, 0xf8, 0xd9, 0xd5, 0x03, 0x8f, 0x28, 0xe2, 0x3e, 0xf3, 0x6b,
                0xa9, 0x38, 0x52, 0xe5, 0x8e, 0x85, 0xf8, 0x90, 0xb1, 0x77, 0x5c, 0x6d, 0x4c, 0x13,
                0x5b, 0xef, 0x0e, 0x2a, 0x19, 0x33, 0x02, 0xd9, 0x0a, 0x80, 0x4c, 0x85, 0x31, 0x25,
                0xaa, 0x5b, 0x11, 0x6f, 0x8c, 0x58, 0x0e, 0xb6, 0x54, 0xfe, 0x35, 0xe6, 0x73, 0x79,
                0x9e, 0x93, 0xcf, 0x58, 0xfe, 0x1e, 0x70, 0xcd, 0xe1, 0x19, 0xab, 0x58, 0x6c, 0x12,
                0xc4, 0x95, 0x75, 0xe6, 0x1a, 0xc4, 0xb7, 0x71, 0xfa, 0x8e, 0xbf, 0x76, 0xca, 0x95,
                0xd6, 0x51, 0xa4, 0xba, 0x87, 0x3b, 0x24, 0xcf, 0x97, 0xff, 0x75, 0x5b, 0xc7, 0x49,
                0xf4, 0x09, 0x6d, 0x2d, 0xa1, 0x5c, 0xf8, 0x30, 0x36, 0xcc, 0x22, 0x0f, 0x0a, 0x68,
                0x93, 0x43, 0x21, 0xc9, 0xae, 0x33, 0x4e, 0x2d, 0x99, 0xa9, 0x95, 0xe9, 0x29, 0x04,
                0xc1, 0x45, 0x23, 0x33, 0x19, 0x00, 0xcb, 0xca, 0x20, 0x4a, 0xdc, 0xb6, 0x93, 0x9d,
                0xc1, 0x71, 0x87, 0x53, 0x53, 0xa1, 0x1e, 0x12, 0xba, 0xcb, 0x2a, 0xab, 0x0f, 0x57,
                0x17, 0x9f, 0x1b, 0x67, 0xea, 0xcc, 0x7e, 0x7b, 0x6c, 0x5c, 0xc8, 0xa3, 0x78, 0x64,
                0x9b, 0x62, 0xb8, 0x52, 0xfa, 0x47, 0x6f, 0x93, 0x37, 0x88, 0x59, 0xd0, 0xd9, 0xa3,
                0x03, 0x3d, 0xa1, 0x31, 0x28, 0x17, 0x57, 0xad, 0x98, 0x5c, 0x86, 0xfe, 0x03, 0x7a,
                0x4e, 0xdd, 0x0a, 0x4a, 0x49, 0xee, 0x5c, 0x4f, 0x48, 0x15, 0xf0, 0x14, 0x73, 0xb9,
                0x60, 0xa1, 0x8a, 0x3b, 0x7f, 0x75, 0x07, 0xee, 0x3d, 0x8b, 0x39, 0x5a, 0x76, 0x68,
                0x40, 0x95, 0x5b, 0xaa, 0xdd, 0x0a, 0xbb, 0xf4, 0xf5, 0x91, 0x4e, 0x92, 0x10, 0xf5,
                0x9e, 0x4c, 0xd1, 0x3c, 0x82, 0xe0, 0xac, 0xf8, 0x79, 0x51, 0x78, 0x02, 0x91, 0xd0,
                0xa9, 0xa7, 0x0f, 0x6b, 0x4f, 0xf5, 0x0d, 0x81, 0x0d, 0x50, 0x90, 0x20, 0x45, 0x30,
                0xad, 0x69, 0x44, 0xb1, 0x3d, 0x23, 0x83, 0xab, 0xca, 0x42, 0x0c, 0xb6, 0xd9, 0xfc,
                0xd8, 0xfd, 0xcb, 0xe9, 0xb3, 0xb5, 0xf4, 0x2f, 0x0a, 0xf5, 0xdf, 0x36, 0x37, 0x8c,
                0xe7, 0x3d, 0xc9, 0x06, 0x16, 0x7d, 0xd4, 0xb3, 0x80, 0x41, 0x8f, 0x17, 0xe4, 0x50,
                0x61, 0x18, 0x57, 0x23, 0xda, 0xbb, 0x6f, 0xfd, 0xbb, 0xd5, 0xa4, 0x18, 0xc3, 0xc6,
                0xfd, 0x33, 0x70, 0x34, 0x09, 0x94, 0x27, 0x7b, 0x88, 0x8e, 0xe4, 0x9b, 0x08, 0x91,
                0xbb, 0xb9, 0x4b, 0x6b, 0x8b, 0x06, 0x5f, 0xd1, 0x92, 0x00, 0xdc, 0x3a, 0x68, 0x1c,
                0xa5, 0xff, 0x2a, 0x98, 0x8f, 0xa6, 0x6d, 0x02, 0x9d, 0xdd, 0xf9, 0xb0, 0x3a, 0x5e,
                0x6c, 0x6e, 0x99, 0xb6, 0x35, 0x41, 0x15, 0x6a, 0x09, 0x51, 0xd9, 0x08, 0x7d, 0xd8,
                0x83, 0xd5, 0x13, 0x9e, 0xe6, 0x7a, 0x8c, 0xe5, 0xdd, 0x6e, 0xf1, 0x4c, 0x9a, 0x10,
                0xdb, 0x81, 0x8b, 0x93, 0x69, 0x02, 0xc7, 0x99, 0x93, 0x90, 0xf9, 0x72, 0xba, 0xfa,
                0x5a, 0x57, 0x2f, 0xc8, 0x73, 0x42, 0xe2, 0xc5, 0x1c, 0xeb, 0x80, 0x30, 0xe0, 0x6b,
                0x72, 0x40, 0x95, 0xf9, 0x8d, 0x7b,
            ],
            ock: [
                0x5b, 0x5b, 0x20, 0xfc, 0x46, 0xba, 0x14, 0xbd, 0x18, 0x43, 0xb6, 0xc7, 0x7e, 0xc3,
                0xf4, 0x55, 0xb7, 0x65, 0xb4, 0xd1, 0x2d, 0xb6, 0x52, 0xeb, 0x34, 0x20, 0x0c, 0x41,
                0x48, 0x88, 0x1f, 0xfc,
            ],
            op: [
                0x3f, 0xeb, 0x34, 0x5a, 0xec, 0xd3, 0x42, 0x9a, 0x16, 0xe1, 0x0f, 0x3d, 0x13, 0x20,
                0xbc, 0x99, 0x71, 0xb5, 0x9e, 0x63, 0x9d, 0x62, 0xb6, 0x96, 0x1a, 0xea, 0x78, 0x15,
                0x67, 0xa8, 0x60, 0x9e, 0x24, 0x75, 0x30, 0x5c, 0x35, 0xcf, 0x37, 0xeb, 0xd2, 0x77,
                0x82, 0x35, 0x16, 0x58, 0x5a, 0xfa, 0x06, 0x05, 0x78, 0x52, 0x17, 0x1e, 0x0c, 0xe4,
                0xb0, 0x52, 0xd5, 0xb4, 0x20, 0x41, 0xd8, 0x1c,
            ],
            c_out: [
                0xe0, 0x1b, 0x5c, 0x1e, 0x01, 0x90, 0x0f, 0xf7, 0x9b, 0xf7, 0x27, 0xf5, 0x37, 0xc4,
                0xac, 0x36, 0xf8, 0x06, 0x7a, 0x0c, 0xc5, 0xc8, 0xe9, 0xc8, 0x20, 0x44, 0x28, 0x43,
                0x69, 0x35, 0x30, 0x91, 0x8a, 0xea, 0x67, 0x2c, 0x4c, 0xd8, 0xfa, 0x3f, 0x6f, 0x2c,
                0xb6, 0x8b, 0x87, 0x17, 0x22, 0xe7, 0x78, 0xf6, 0xe2, 0x5e, 0x78, 0xae, 0x00, 0xa8,
                0x43, 0x28, 0xc4, 0xc2, 0xbf, 0x1c, 0x79, 0xb2, 0xc9, 0xd8, 0x69, 0x72, 0x60, 0xcd,
                0x44, 0x7e, 0x2e, 0xff, 0x31, 0x5d, 0x74, 0xb5, 0xb1, 0xfd,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0xc3, 0x7c, 0x7d, 0xbb, 0xe5, 0x51, 0xd9, 0xd3, 0xb1, 0xa4, 0x96, 0x88, 0x7d, 0xb2,
                0xe8, 0x42, 0xdc, 0x94, 0x52, 0x01, 0xf4, 0x08, 0x10, 0xdf, 0x4d, 0x76, 0x39, 0x32,
                0xed, 0x5c, 0x76, 0x39, 0x8b, 0x35, 0x73, 0xfe, 0x23, 0xf1, 0xe8, 0xb7, 0xe7, 0x9f,
                0x1c, 0x16, 0x95, 0xc0, 0x97, 0xc1, 0x24, 0xff, 0x1f, 0x7d, 0x6e, 0x61, 0xf2, 0xc5,
                0x8f, 0x14, 0x39, 0xa7, 0x56, 0x96, 0x9d, 0x19,
            ],
            ovk: [
                0xa6, 0x68, 0xa0, 0xae, 0x2b, 0xb9, 0x34, 0xc8, 0x2c, 0x41, 0x42, 0xda, 0x69, 0xd1,
                0x2c, 0xa7, 0xde, 0x9a, 0x7d, 0xf7, 0x06, 0x40, 0x0e, 0xc7, 0x98, 0x78, 0xd8, 0x68,
                0xe1, 0x7e, 0x8f, 0x71,
            ],
            default_d: [
                0x56, 0x4f, 0xc3, 0x81, 0xfc, 0x4d, 0xc8, 0x11, 0x8d, 0xe4, 0x7c,
            ],
            default_pk_d: [
                0xae, 0xee, 0xa5, 0x0c, 0x6b, 0xb0, 0x2e, 0x5e, 0x22, 0x4d, 0xc2, 0x95, 0x9c, 0x22,
                0x9d, 0x0e, 0x3b, 0xb8, 0x79, 0xc4, 0xab, 0x00, 0xaa, 0x0a, 0xb2, 0x5a, 0x40, 0x10,
                0x6b, 0x80, 0xbb, 0xb7,
            ],
            v: 11137853725062838288,
            rseed: [
                0x25, 0x37, 0xb8, 0x71, 0xb4, 0x29, 0x4a, 0x65, 0xd3, 0xe0, 0x55, 0xff, 0x71, 0x8d,
                0xd9, 0xdc, 0x8c, 0x75, 0xe7, 0xe5, 0xb2, 0xef, 0xe4, 0x42, 0x63, 0x73, 0x71, 0xb7,
                0xc4, 0x8f, 0x6e, 0xe9,
            ],
            memo: [
                0xff, 0x9e, 0x3e, 0xa3, 0x8a, 0x4b, 0x0f, 0x2f, 0x67, 0xfc, 0x2b, 0x90, 0x8c, 0xda,
                0x65, 0x7e, 0xae, 0x75, 0x4e, 0x03, 0x7e, 0x26, 0x2e, 0x9a, 0x9f, 0x9b, 0xd7, 0xec,
                0x42, 0x67, 0xed, 0x8e, 0x96, 0x93, 0x0e, 0x10, 0x84, 0x78, 0x3c, 0x37, 0xd6, 0xf9,
                0xdd, 0x15, 0xfd, 0x29, 0xf4, 0xcc, 0x47, 0x7e, 0x66, 0xf1, 0x30, 0xd6, 0x30, 0x43,
                0x0d, 0xcc, 0x01, 0x04, 0x89, 0x9b, 0x4f, 0x9f, 0x46, 0xeb, 0x09, 0x0e, 0xf7, 0xfc,
                0x90, 0xb4, 0x79, 0xab, 0xf6, 0x1f, 0x93, 0x95, 0x5e, 0xe0, 0x0e, 0x6a, 0x18, 0x48,
                0xf1, 0xab, 0x14, 0xad, 0x33, 0x4f, 0x2b, 0x68, 0x03, 0x58, 0x08, 0xcd, 0xf1, 0xbb,
                0x9e, 0x9d, 0x9a, 0x81, 0x6b, 0xaf, 0x72, 0x8a, 0x95, 0x5b, 0x96, 0x0b, 0x77, 0x01,
                0xfa, 0x62, 0x66, 0x87, 0xdc, 0x3c, 0x9c, 0xba, 0x64, 0x63, 0x37, 0xb5, 0x3e, 0x29,
                0x81, 0x6e, 0x94, 0x82, 0xdd, 0xf5, 0x57, 0x8a, 0x87, 0x68, 0xaa, 0xe4, 0x77, 0xfc,
                0xe4, 0x10, 0xac, 0x2d, 0x5d, 0xe6, 0x09, 0x58, 0x61, 0xc1, 0x11, 0xd7, 0xfe, 0xb3,
                0xe6, 0xbb, 0x4f, 0xbb, 0x5a, 0x54, 0x95, 0x54, 0x95, 0x97, 0x27, 0x98, 0x35, 0x0a,
                0x25, 0x3f, 0x05, 0xf6, 0x6c, 0x2e, 0xcf, 0xcb, 0xc0, 0xed, 0x43, 0xf5, 0xec, 0x2e,
                0x6d, 0x8d, 0xba, 0x15, 0xa5, 0x12, 0x54, 0xd9, 0x7b, 0x18, 0x21, 0x10, 0x7c, 0x07,
                0xdd, 0x9a, 0x16, 0xef, 0x84, 0x06, 0xf9, 0x43, 0xe2, 0x82, 0xb9, 0x5d, 0x4b, 0x36,
                0x25, 0x30, 0xc9, 0x13, 0xd6, 0xba, 0x42, 0x1d, 0xf6, 0x02, 0x7d, 0xe5, 0xaf, 0x1e,
                0x47, 0x45, 0xd5, 0x86, 0x81, 0x06, 0x95, 0x4b, 0xe6, 0xc1, 0x96, 0x27, 0x80, 0xa2,
                0x94, 0x10, 0x72, 0xe9, 0x51, 0x31, 0xb1, 0x67, 0x9d, 0xf0, 0x63, 0x76, 0x25, 0x04,
                0x2c, 0x37, 0xd4, 0x8f, 0xfb, 0x15, 0x2e, 0x5e, 0xbc, 0x18, 0x5c, 0x8a, 0x2b, 0x7d,
                0x43, 0x85, 0xf1, 0xc9, 0x5a, 0xf9, 0x37, 0xdf, 0x78, 0xdf, 0xd8, 0x75, 0x7f, 0xab,
                0x43, 0x49, 0x68, 0xb0, 0xb5, 0x7c, 0x66, 0x57, 0x44, 0x68, 0xf1, 0x60, 0xb4, 0x47,
                0xac, 0x82, 0x21, 0xe5, 0x06, 0x06, 0x76, 0xa8, 0x42, 0xa1, 0xc6, 0xb7, 0x17, 0x2d,
                0xd3, 0x34, 0x0f, 0x76, 0x40, 0x70, 0xab, 0x1f, 0xe0, 0x91, 0xc5, 0xc7, 0x4c, 0x95,
                0xa5, 0xdc, 0x04, 0x33, 0x90, 0x72, 0x3a, 0x4c, 0x12, 0x7d, 0xa1, 0x4c, 0xdd, 0xe1,
                0xdc, 0x26, 0x75, 0xa6, 0x23, 0x40, 0xb3, 0xe6, 0xaf, 0xd0, 0x52, 0x2a, 0x31, 0xde,
                0x26, 0xe7, 0xd1, 0xec, 0x3a, 0x9c, 0x8a, 0x09, 0x1f, 0xfd, 0xc7, 0x5b, 0x7e, 0xcf,
                0xdc, 0x7c, 0x12, 0x99, 0x5a, 0x5e, 0x37, 0xce, 0x34, 0x88, 0xbd, 0x29, 0xf8, 0x62,
                0x9d, 0x68, 0xf6, 0x96, 0x49, 0x24, 0x48, 0xdd, 0x52, 0x66, 0x97, 0x47, 0x6d, 0xc0,
                0x61, 0x34, 0x6e, 0xbe, 0x3f, 0x67, 0x72, 0x17, 0xff, 0x9c, 0x60, 0xef, 0xce, 0x94,
                0x3a, 0xf2, 0x8d, 0xfd, 0x3f, 0x9e, 0x59, 0x69, 0x25, 0x98, 0xa6, 0x04, 0x7c, 0x23,
                0xc4, 0xc0, 0x14, 0x00, 0xf1, 0xab, 0x57, 0x30, 0xea, 0xc0, 0xae, 0x8d, 0x58, 0x43,
                0xd5, 0x05, 0x1c, 0x37, 0x62, 0x40, 0x17, 0x2a, 0xf2, 0x18, 0xd7, 0xa1, 0xec, 0xfe,
                0x65, 0xb4, 0xf7, 0x51, 0x00, 0x63, 0x89, 0x83, 0xc1, 0x4d, 0xe4, 0x97, 0x47, 0x55,
                0xda, 0xde, 0x80, 0x18, 0xc9, 0xb8, 0xf4, 0x54, 0x3f, 0xb0, 0x95, 0x96, 0x15, 0x13,
                0xe6, 0x7c, 0x61, 0xdb, 0xc5, 0x9c, 0x60, 0x7f, 0x9b, 0x51, 0xf8, 0xd0, 0x9b, 0xdc,
                0xad, 0x28, 0xbc, 0xfb, 0x9e, 0x5d, 0x27, 0x44, 0xea, 0x88, 0x48, 0xb2, 0x62, 0x3a,
                0xc0, 0x7f, 0x8e, 0xf6, 0x1a, 0x81, 0xa3, 0x59,
            ],
            cv_net: [
                0xb2, 0x7f, 0x48, 0x59, 0x15, 0x0d, 0x48, 0x45, 0xab, 0x57, 0x78, 0x82, 0x61, 0x50,
                0x0a, 0x12, 0x01, 0x2d, 0x63, 0xc0, 0x09, 0xc6, 0x77, 0x44, 0xba, 0xe0, 0xd5, 0x83,
                0x88, 0xff, 0xee, 0x2f,
            ],
            rho: [
                0xe9, 0x16, 0x93, 0xc3, 0x7d, 0x04, 0x37, 0x5e, 0x67, 0xc5, 0x71, 0x5a, 0x39, 0xc9,
                0x79, 0x4a, 0x4e, 0xcd, 0x08, 0x38, 0xe2, 0x35, 0x1f, 0xd7, 0xcd, 0x93, 0xa1, 0x55,
                0x7f, 0x01, 0x02, 0x3e,
            ],
            cmx: [
                0xcc, 0x29, 0x41, 0xe0, 0xf6, 0x38, 0x25, 0x7f, 0xb6, 0x51, 0x6a, 0x27, 0xc3, 0x0e,
                0xaa, 0xe0, 0xb4, 0x6d, 0x2f, 0xf0, 0x6a, 0x73, 0x50, 0x70, 0x41, 0x74, 0x6c, 0xdc,
                0x23, 0x07, 0x61, 0x25,
            ],
            esk: [
                0xec, 0x99, 0xeb, 0x6f, 0x67, 0x1d, 0x0a, 0x9e, 0x84, 0x5a, 0xbf, 0xb7, 0x8a, 0xec,
                0xe9, 0x32, 0xeb, 0x13, 0xf5, 0xae, 0xd6, 0x4f, 0x20, 0xa7, 0x77, 0xc9, 0x46, 0xe9,
                0xcd, 0x7a, 0x24, 0x23,
            ],
            ephemeral_key: [
                0xad, 0xa7, 0x60, 0x27, 0xdc, 0xae, 0xb5, 0xf1, 0x58, 0xac, 0x6c, 0x5c, 0x6d, 0x7f,
                0x05, 0x31, 0x8f, 0xec, 0x26, 0xd5, 0xc7, 0x94, 0xf2, 0xe9, 0x00, 0xcb, 0x96, 0x29,
                0x92, 0x49, 0xe0, 0xb5,
            ],
            shared_secret: [
                0xc2, 0x5c, 0xf1, 0xd6, 0x53, 0x2e, 0x06, 0xcb, 0x8b, 0x15, 0x22, 0x14, 0x76, 0x6e,
                0xee, 0xd7, 0x5a, 0x17, 0x8c, 0x82, 0x2a, 0x11, 0x51, 0xf6, 0x9e, 0x92, 0xe9, 0xcf,
                0xfa, 0x47, 0xcb, 0x19,
            ],
            k_enc: [
                0xd4, 0xef, 0x89, 0xc4, 0x64, 0xbe, 0x8f, 0x42, 0xf6, 0xb7, 0xf2, 0x6d, 0x89, 0x37,
                0x8b, 0x73, 0xa3, 0x55, 0xa4, 0xfe, 0x24, 0x6c, 0x2e, 0xc8, 0xe8, 0x14, 0x6b, 0xef,
                0x7d, 0xdd, 0x14, 0x19,
            ],
            p_enc: [
                0x02, 0x56, 0x4f, 0xc3, 0x81, 0xfc, 0x4d, 0xc8, 0x11, 0x8d, 0xe4, 0x7c, 0x10, 0xb8,
                0xa1, 0xba, 0xf3, 0x9a, 0x91, 0x9a, 0x25, 0x37, 0xb8, 0x71, 0xb4, 0x29, 0x4a, 0x65,
                0xd3, 0xe0, 0x55, 0xff, 0x71, 0x8d, 0xd9, 0xdc, 0x8c, 0x75, 0xe7, 0xe5, 0xb2, 0xef,
                0xe4, 0x42, 0x63, 0x73, 0x71, 0xb7, 0xc4, 0x8f, 0x6e, 0xe9, 0xff, 0x9e, 0x3e, 0xa3,
                0x8a, 0x4b, 0x0f, 0x2f, 0x67, 0xfc, 0x2b, 0x90, 0x8c, 0xda, 0x65, 0x7e, 0xae, 0x75,
                0x4e, 0x03, 0x7e, 0x26, 0x2e, 0x9a, 0x9f, 0x9b, 0xd7, 0xec, 0x42, 0x67, 0xed, 0x8e,
                0x96, 0x93, 0x0e, 0x10, 0x84, 0x78, 0x3c, 0x37, 0xd6, 0xf9, 0xdd, 0x15, 0xfd, 0x29,
                0xf4, 0xcc, 0x47, 0x7e, 0x66, 0xf1, 0x30, 0xd6, 0x30, 0x43, 0x0d, 0xcc, 0x01, 0x04,
                0x89, 0x9b, 0x4f, 0x9f, 0x46, 0xeb, 0x09, 0x0e, 0xf7, 0xfc, 0x90, 0xb4, 0x79, 0xab,
                0xf6, 0x1f, 0x93, 0x95, 0x5e, 0xe0, 0x0e, 0x6a, 0x18, 0x48, 0xf1, 0xab, 0x14, 0xad,
                0x33, 0x4f, 0x2b, 0x68, 0x03, 0x58, 0x08, 0xcd, 0xf1, 0xbb, 0x9e, 0x9d, 0x9a, 0x81,
                0x6b, 0xaf, 0x72, 0x8a, 0x95, 0x5b, 0x96, 0x0b, 0x77, 0x01, 0xfa, 0x62, 0x66, 0x87,
                0xdc, 0x3c, 0x9c, 0xba, 0x64, 0x63, 0x37, 0xb5, 0x3e, 0x29, 0x81, 0x6e, 0x94, 0x82,
                0xdd, 0xf5, 0x57, 0x8a, 0x87, 0x68, 0xaa, 0xe4, 0x77, 0xfc, 0xe4, 0x10, 0xac, 0x2d,
                0x5d, 0xe6, 0x09, 0x58, 0x61, 0xc1, 0x11, 0xd7, 0xfe, 0xb3, 0xe6, 0xbb, 0x4f, 0xbb,
                0x5a, 0x54, 0x95, 0x54, 0x95, 0x97, 0x27, 0x98, 0x35, 0x0a, 0x25, 0x3f, 0x05, 0xf6,
                0x6c, 0x2e, 0xcf, 0xcb, 0xc0, 0xed, 0x43, 0xf5, 0xec, 0x2e, 0x6d, 0x8d, 0xba, 0x15,
                0xa5, 0x12, 0x54, 0xd9, 0x7b, 0x18, 0x21, 0x10, 0x7c, 0x07, 0xdd, 0x9a, 0x16, 0xef,
                0x84, 0x06, 0xf9, 0x43, 0xe2, 0x82, 0xb9, 0x5d, 0x4b, 0x36, 0x25, 0x30, 0xc9, 0x13,
                0xd6, 0xba, 0x42, 0x1d, 0xf6, 0x02, 0x7d, 0xe5, 0xaf, 0x1e, 0x47, 0x45, 0xd5, 0x86,
                0x81, 0x06, 0x95, 0x4b, 0xe6, 0xc1, 0x96, 0x27, 0x80, 0xa2, 0x94, 0x10, 0x72, 0xe9,
                0x51, 0x31, 0xb1, 0x67, 0x9d, 0xf0, 0x63, 0x76, 0x25, 0x04, 0x2c, 0x37, 0xd4, 0x8f,
                0xfb, 0x15, 0x2e, 0x5e, 0xbc, 0x18, 0x5c, 0x8a, 0x2b, 0x7d, 0x43, 0x85, 0xf1, 0xc9,
                0x5a, 0xf9, 0x37, 0xdf, 0x78, 0xdf, 0xd8, 0x75, 0x7f, 0xab, 0x43, 0x49, 0x68, 0xb0,
                0xb5, 0x7c, 0x66, 0x57, 0x44, 0x68, 0xf1, 0x60, 0xb4, 0x47, 0xac, 0x82, 0x21, 0xe5,
                0x06, 0x06, 0x76, 0xa8, 0x42, 0xa1, 0xc6, 0xb7, 0x17, 0x2d, 0xd3, 0x34, 0x0f, 0x76,
                0x40, 0x70, 0xab, 0x1f, 0xe0, 0x91, 0xc5, 0xc7, 0x4c, 0x95, 0xa5, 0xdc, 0x04, 0x33,
                0x90, 0x72, 0x3a, 0x4c, 0x12, 0x7d, 0xa1, 0x4c, 0xdd, 0xe1, 0xdc, 0x26, 0x75, 0xa6,
                0x23, 0x40, 0xb3, 0xe6, 0xaf, 0xd0, 0x52, 0x2a, 0x31, 0xde, 0x26, 0xe7, 0xd1, 0xec,
                0x3a, 0x9c, 0x8a, 0x09, 0x1f, 0xfd, 0xc7, 0x5b, 0x7e, 0xcf, 0xdc, 0x7c, 0x12, 0x99,
                0x5a, 0x5e, 0x37, 0xce, 0x34, 0x88, 0xbd, 0x29, 0xf8, 0x62, 0x9d, 0x68, 0xf6, 0x96,
                0x49, 0x24, 0x48, 0xdd, 0x52, 0x66, 0x97, 0x47, 0x6d, 0xc0, 0x61, 0x34, 0x6e, 0xbe,
                0x3f, 0x67, 0x72, 0x17, 0xff, 0x9c, 0x60, 0xef, 0xce, 0x94, 0x3a, 0xf2, 0x8d, 0xfd,
                0x3f, 0x9e, 0x59, 0x69, 0x25, 0x98, 0xa6, 0x04, 0x7c, 0x23, 0xc4, 0xc0, 0x14, 0x00,
                0xf1, 0xab, 0x57, 0x30, 0xea, 0xc0, 0xae, 0x8d, 0x58, 0x43, 0xd5, 0x05, 0x1c, 0x37,
                0x62, 0x40, 0x17, 0x2a, 0xf2, 0x18, 0xd7, 0xa1, 0xec, 0xfe, 0x65, 0xb4, 0xf7, 0x51,
                0x00, 0x63, 0x89, 0x83, 0xc1, 0x4d, 0xe4, 0x97, 0x47, 0x55, 0xda, 0xde, 0x80, 0x18,
                0xc9, 0xb8, 0xf4, 0x54, 0x3f, 0xb0, 0x95, 0x96, 0x15, 0x13, 0xe6, 0x7c, 0x61, 0xdb,
                0xc5, 0x9c, 0x60, 0x7f, 0x9b, 0x51, 0xf8, 0xd0, 0x9b, 0xdc, 0xad, 0x28, 0xbc, 0xfb,
                0x9e, 0x5d, 0x27, 0x44, 0xea, 0x88, 0x48, 0xb2, 0x62, 0x3a, 0xc0, 0x7f, 0x8e, 0xf6,
                0x1a, 0x81, 0xa3, 0x59,
            ],
            c_enc: [
                0xea, 0x86, 0x7e, 0x07, 0xac, 0xb7, 0xfa, 0x03, 0xe7, 0x36, 0xce, 0xdd, 0xed, 0x82,
                0x3d, 0x19, 0x91, 0x30, 0xfb, 0x3f, 0xef, 0x62, 0xcc, 0x3e, 0xa9, 0xa0, 0xc9, 0x35,
                0xdf, 0x05, 0xc5, 0xad, 0x5e, 0x67, 0xf7, 0x4f, 0xcb, 0xa6, 0x67, 0xe0, 0x81, 0x1c,
                0xdc, 0x0c, 0xa9, 0xf3, 0xd7, 0x81, 0x76, 0x43, 0x75, 0x56, 0x47, 0xbe, 0x59, 0xdf,
                0x09, 0x6e, 0x46, 0x5e, 0xc1, 0x11, 0x24, 0x36, 0x3c, 0x98, 0x1c, 0x55, 0xd5, 0x5f,
                0x8d, 0x8a, 0xc8, 0xb6, 0x54, 0x0a, 0x98, 0xcf, 0xcc, 0x29, 0x4a, 0xa7, 0xcc, 0x62,
                0x95, 0x50, 0x85, 0x3a, 0x25, 0xbf, 0x6b, 0x32, 0x35, 0x16, 0xe1, 0x58, 0x2b, 0x4e,
                0x8b, 0x82, 0x95, 0xe2, 0x7f, 0x6b, 0x99, 0x94, 0x80, 0x2e, 0xfe, 0xf5, 0x43, 0x1f,
                0x03, 0x7d, 0x22, 0x09, 0xa0, 0x88, 0x07, 0x34, 0x6f, 0xb3, 0x9c, 0xad, 0xd8, 0x93,
                0xd6, 0x5f, 0x64, 0xc7, 0xfe, 0x92, 0xec, 0xbc, 0xdf, 0xe5, 0x55, 0x54, 0xca, 0xff,
                0xb0, 0xcf, 0x41, 0xeb, 0x5c, 0x55, 0x72, 0xba, 0x44, 0x39, 0x90, 0x00, 0xe9, 0xfa,
                0x9a, 0xaf, 0x6e, 0xb7, 0x6c, 0x2a, 0x8a, 0x5d, 0x7b, 0x6b, 0x52, 0xb9, 0xa9, 0xef,
                0x6e, 0xc3, 0xed, 0xa6, 0x65, 0xfe, 0x8a, 0xa8, 0x07, 0x48, 0xb8, 0x1f, 0x7d, 0x55,
                0x96, 0xf1, 0x94, 0x2f, 0xf7, 0x1d, 0x29, 0xc6, 0x78, 0xa3, 0xb6, 0xc6, 0x6b, 0xa7,
                0x0f, 0x45, 0xb1, 0xfc, 0xf2, 0x22, 0x38, 0x84, 0x50, 0x68, 0xf3, 0x60, 0xb0, 0x99,
                0xae, 0xe9, 0xf2, 0xf0, 0xef, 0x22, 0x33, 0x3f, 0xd6, 0x4f, 0xf1, 0x1e, 0x48, 0x15,
                0x43, 0xa5, 0x2b, 0xb3, 0x3f, 0x52, 0x3d, 0xe2, 0xec, 0x92, 0x3a, 0xe9, 0x86, 0x58,
                0x57, 0x71, 0x7b, 0x65, 0xd4, 0x4c, 0x3d, 0x9d, 0xb7, 0xb3, 0xec, 0xb2, 0xff, 0x02,
                0x25, 0x29, 0x7d, 0xeb, 0x83, 0xdd, 0x1b, 0x9a, 0x39, 0x4d, 0x69, 0x7c, 0x09, 0xd7,
                0xfe, 0xc6, 0x1e, 0xac, 0xee, 0x39, 0xa2, 0xf1, 0xce, 0xd9, 0xe6, 0xfd, 0xa5, 0xc0,
                0xf3, 0x27, 0x71, 0xc6, 0xce, 0x3f, 0x17, 0xbe, 0x0e, 0xef, 0x81, 0x10, 0x99, 0xc1,
                0x09, 0xe4, 0xfb, 0x6a, 0xec, 0x30, 0xdf, 0x04, 0x5b, 0x43, 0xda, 0x89, 0x30, 0x8b,
                0xc8, 0x37, 0x68, 0x8b, 0xb0, 0xf4, 0xa2, 0xc6, 0x04, 0xa1, 0x54, 0xa8, 0x47, 0xc5,
                0xd7, 0x87, 0x27, 0xf0, 0xe1, 0xab, 0x11, 0xe3, 0x40, 0xcf, 0xd4, 0xc7, 0xe8, 0xc6,
                0xba, 0xff, 0xfe, 0xfc, 0x74, 0xef, 0x55, 0x3d, 0x4b, 0x29, 0x26, 0x9f, 0x38, 0xb4,
                0xb7, 0xfe, 0x6e, 0x07, 0x3a, 0x70, 0xae, 0xf2, 0x0d, 0x6c, 0x23, 0x8b, 0x9f, 0xfd,
                0x24, 0x1f, 0xf6, 0x99, 0x99, 0x92, 0x30, 0xf0, 0xb3, 0x9c, 0x96, 0x3e, 0xfa, 0x00,
                0xef, 0x7f, 0x09, 0x2d, 0x76, 0x0c, 0x40, 0x9a, 0x2b, 0x6c, 0x26, 0xb4, 0x97, 0x82,
                0x9b, 0x81, 0xcb, 0xa4, 0xc8, 0x3b, 0x11, 0x44, 0xc6, 0x5a, 0x2d, 0x31, 0x78, 0xbb,
                0x28, 0xc1, 0xd8, 0xe3, 0x30, 0xd3, 0xf4, 0xaf, 0x90, 0x10, 0xad, 0x49, 0xa5, 0xdb,
                0x70, 0xd6, 0x6a, 0xf5, 0x70, 0x19, 0x21, 0x49, 0x80, 0x5f, 0xe1, 0xca, 0x61, 0xc6,
                0xd4, 0xec, 0xaf, 0xa7, 0x97, 0x51, 0x7f, 0x33, 0x06, 0xaf, 0x2a, 0x32, 0x27, 0x3b,
                0xf3, 0xd6, 0x98, 0xaa, 0x55, 0xa5, 0x72, 0xb2, 0xdf, 0x80, 0x36, 0xd9, 0x7c, 0xf5,
                0x8c, 0x12, 0x9f, 0x82, 0x85, 0xd0, 0xd7, 0xea, 0x04, 0xc1, 0x88, 0xa8, 0x39, 0x6e,
                0x73, 0x8b, 0xd4, 0x48, 0x46, 0x5e, 0x7b, 0x9a, 0x64, 0xb7, 0x96, 0x7b, 0xcb, 0x25,
                0xf8, 0xaa, 0x85, 0x9e, 0xa5, 0xca, 0x06, 0xc3, 0xdf, 0x39, 0x0f, 0xac, 0x8a, 0xc6,
                0x06, 0x05, 0x03, 0x1a, 0x02, 0x09, 0xbb, 0x80, 0x24, 0x0d, 0x05, 0x45, 0xf4, 0x11,
                0x77, 0xea, 0xb4, 0x40, 0x2e, 0x8a, 0x42, 0x3a, 0x40, 0xcc, 0xff, 0x58, 0x6b, 0x2e,
                0xdd, 0x6a, 0xcf, 0xb8, 0xf1, 0xd8, 0x0d, 0x8c, 0x60, 0x5f, 0x49, 0x05, 0x2e, 0x7a,
                0x30, 0x82, 0x6e, 0x1f, 0x03, 0x09, 0x9c, 0x71, 0x43, 0xad, 0x0b, 0x51, 0xfc, 0x63,
                0x6f, 0x1d, 0x87, 0x05, 0x83, 0xd6, 0xed, 0xce, 0xcd, 0xe9, 0x42, 0x38, 0x9e, 0x6c,
                0x01, 0x00, 0x66, 0x78, 0xda, 0xad,
            ],
            ock: [
                0xf9, 0x59, 0x87, 0xcc, 0x0c, 0x73, 0x57, 0xc7, 0x20, 0x26, 0x27, 0xa5, 0xa5, 0x50,
                0x09, 0xef, 0xd7, 0x17, 0x55, 0x3e, 0x9d, 0x34, 0xc1, 0x5f, 0xad, 0xf5, 0xc9, 0x46,
                0xa9, 0xa0, 0x72, 0x17,
            ],
            op: [
                0xae, 0xee, 0xa5, 0x0c, 0x6b, 0xb0, 0x2e, 0x5e, 0x22, 0x4d, 0xc2, 0x95, 0x9c, 0x22,
                0x9d, 0x0e, 0x3b, 0xb8, 0x79, 0xc4, 0xab, 0x00, 0xaa, 0x0a, 0xb2, 0x5a, 0x40, 0x10,
                0x6b, 0x80, 0xbb, 0xb7, 0xec, 0x99, 0xeb, 0x6f, 0x67, 0x1d, 0x0a, 0x9e, 0x84, 0x5a,
                0xbf, 0xb7, 0x8a, 0xec, 0xe9, 0x32, 0xeb, 0x13, 0xf5, 0xae, 0xd6, 0x4f, 0x20, 0xa7,
                0x77, 0xc9, 0x46, 0xe9, 0xcd, 0x7a, 0x24, 0x23,
            ],
            c_out: [
                0x28, 0x9d, 0x00, 0xd4, 0x62, 0x78, 0x0b, 0xdb, 0xb6, 0xcd, 0x6c, 0xec, 0x93, 0x6b,
                0xe6, 0x5e, 0x63, 0x0d, 0x4a, 0x3c, 0x2a, 0xf7, 0x6e, 0x10, 0x4b, 0x17, 0x4e, 0x3d,
                0x5b, 0x71, 0x66, 0x39, 0xb0, 0x68, 0xb3, 0x85, 0x00, 0xd8, 0x04, 0x00, 0xee, 0x64,
                0x2f, 0x8e, 0x08, 0xc8, 0xdb, 0x29, 0xff, 0xd5, 0xa9, 0xda, 0xde, 0x84, 0xa5, 0x49,
                0x5d, 0x57, 0x7b, 0x2f, 0x56, 0xd3, 0x2b, 0x07, 0x96, 0xca, 0x23, 0xf0, 0x0f, 0x0f,
                0x8e, 0xf9, 0x93, 0x07, 0x4f, 0x88, 0x0d, 0x6a, 0xeb, 0xcb,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0x74, 0xa8, 0x41, 0x1a, 0x20, 0xbc, 0x3c, 0x53, 0xf7, 0xe7, 0xab, 0xb9, 0x31, 0x6c,
                0x44, 0x2b, 0x4b, 0x09, 0xcf, 0x88, 0xbb, 0xed, 0x4a, 0x90, 0xb9, 0x2f, 0x5a, 0x1c,
                0xed, 0x93, 0x16, 0x2b, 0xc3, 0x37, 0x34, 0x67, 0x20, 0xec, 0x0c, 0xd0, 0xea, 0x73,
                0x5d, 0x9e, 0x32, 0x3f, 0x20, 0xdb, 0x77, 0x8a, 0xd1, 0x8a, 0x84, 0xc7, 0x9e, 0xe6,
                0x28, 0x77, 0x99, 0xef, 0x02, 0x76, 0x41, 0x07,
            ],
            ovk: [
                0x0c, 0x81, 0x1e, 0x4c, 0x31, 0xfb, 0xb4, 0x9f, 0x3a, 0x90, 0xbb, 0xd0, 0x5d, 0xce,
                0x62, 0xf3, 0x44, 0xe7, 0x07, 0x75, 0x93, 0x15, 0x9a, 0xe3, 0x50, 0x50, 0xb0, 0x4c,
                0x9e, 0x6b, 0x86, 0xbc,
            ],
            default_d: [
                0xc6, 0xe8, 0xf0, 0xd5, 0x0a, 0xe8, 0x05, 0x87, 0x91, 0xdc, 0x0e,
            ],
            default_pk_d: [
                0x8e, 0x66, 0xb7, 0x92, 0xec, 0xb1, 0x56, 0xef, 0x68, 0x5e, 0xe8, 0xea, 0x35, 0xd3,
                0x82, 0x75, 0x8b, 0xa4, 0x15, 0x97, 0xa3, 0x3a, 0x93, 0xba, 0xf3, 0x81, 0xd6, 0x3c,
                0x17, 0x5b, 0xa9, 0x8b,
            ],
            v: 7387862906040043846,
            rseed: [
                0x25, 0x01, 0xe5, 0x1b, 0x01, 0x2a, 0xea, 0x94, 0x46, 0xa2, 0x10, 0x4e, 0x93, 0xf8,
                0x15, 0xa0, 0xb3, 0xa2, 0x9b, 0x45, 0x83, 0x14, 0xf3, 0xd8, 0xbe, 0x2b, 0x98, 0x23,
                0xd3, 0x42, 0xf4, 0x62,
            ],
            memo: [
                0xff, 0x13, 0xe9, 0x42, 0xa7, 0xe1, 0x9a, 0x46, 0xe9, 0x70, 0xb5, 0xc5, 0x06, 0x70,
                0x84, 0x30, 0x31, 0x7b, 0x1b, 0xb3, 0xb3, 0x5d, 0xf6, 0x8a, 0xe3, 0x3a, 0x49, 0x26,
                0xa0, 0x3e, 0x6b, 0xfe, 0xb5, 0x51, 0x04, 0x16, 0xfc, 0xbb, 0x05, 0x24, 0xc9, 0xca,
                0x50, 0x74, 0x15, 0x6c, 0xc5, 0xa5, 0xd6, 0xfe, 0x1c, 0x99, 0x5e, 0xdc, 0x60, 0xa2,
                0xf5, 0x50, 0x41, 0x1a, 0xa4, 0x1e, 0x3d, 0xa3, 0xbd, 0xcf, 0x64, 0xbc, 0xf0, 0x4a,
                0x05, 0x10, 0x57, 0x1b, 0x93, 0x6d, 0x47, 0xe5, 0x5c, 0xec, 0x03, 0x30, 0xee, 0x8d,
                0xfe, 0x73, 0x56, 0x34, 0x04, 0xf0, 0x47, 0xd7, 0xf3, 0xa8, 0xa3, 0xd7, 0x74, 0x3b,
                0xc5, 0x54, 0x95, 0x52, 0x10, 0xf1, 0xeb, 0x0d, 0x08, 0x59, 0x9e, 0xa7, 0x7d, 0x5f,
                0x97, 0x4d, 0x87, 0x17, 0x6d, 0x37, 0xd9, 0x8b, 0x9c, 0x0a, 0xd4, 0x40, 0x40, 0x72,
                0x09, 0xed, 0x6a, 0x9f, 0x08, 0x46, 0x4d, 0x56, 0x55, 0x93, 0xe1, 0xa6, 0x3b, 0x93,
                0x85, 0x36, 0xb4, 0x92, 0x44, 0xe9, 0x7d, 0x88, 0x01, 0x73, 0xb6, 0x40, 0xf2, 0xdd,
                0xb7, 0x4d, 0x06, 0x8e, 0xcb, 0x46, 0xcf, 0x28, 0x9b, 0x7d, 0x89, 0x13, 0x07, 0xbb,
                0xa3, 0x70, 0x54, 0xcf, 0x91, 0xb3, 0x1f, 0xc8, 0x2f, 0x74, 0xd5, 0xfc, 0xc0, 0x00,
                0x94, 0x2e, 0xde, 0x91, 0x18, 0x25, 0xf5, 0x3f, 0xe6, 0x09, 0x68, 0x6f, 0x46, 0x32,
                0x23, 0xb1, 0xe9, 0xbc, 0x03, 0xbd, 0xe8, 0x95, 0xd1, 0x23, 0x8f, 0xad, 0x04, 0xa3,
                0xbf, 0xce, 0x68, 0xa0, 0x75, 0xe8, 0xa3, 0x7c, 0x0e, 0x87, 0xbf, 0x46, 0xdd, 0x01,
                0x55, 0x45, 0xf9, 0xb4, 0xfb, 0x0e, 0xec, 0x64, 0x5f, 0xfc, 0xbb, 0xe0, 0xca, 0x5f,
                0x8c, 0x56, 0x1b, 0x25, 0x7d, 0x52, 0xd6, 0x02, 0xd8, 0xc9, 0x4c, 0x50, 0x28, 0x73,
                0xa0, 0x1d, 0x92, 0x51, 0xd8, 0xc8, 0x60, 0xc0, 0x41, 0x52, 0x5b, 0x3b, 0xf4, 0xe3,
                0xa2, 0xeb, 0x92, 0x72, 0x81, 0x5c, 0x75, 0x86, 0x76, 0x84, 0x28, 0xb4, 0xc2, 0xb2,
                0x5e, 0x37, 0x45, 0xf0, 0x09, 0xc5, 0xdc, 0xe2, 0x0b, 0x69, 0xd5, 0xd7, 0xc4, 0x3c,
                0xeb, 0x73, 0x6b, 0x68, 0x31, 0xe8, 0xc1, 0x10, 0xf1, 0x6c, 0xfd, 0xb3, 0xa4, 0x67,
                0xe9, 0x41, 0x4c, 0x00, 0xec, 0xf1, 0x37, 0x31, 0x50, 0x08, 0x94, 0x55, 0x56, 0x78,
                0xc4, 0x97, 0xfa, 0xba, 0x9a, 0x95, 0xd0, 0x1c, 0xc4, 0x64, 0x39, 0x0f, 0xc4, 0xa7,
                0x6b, 0xfa, 0x8b, 0x0e, 0x1c, 0x68, 0xa5, 0x25, 0xd7, 0x06, 0xd6, 0x60, 0x4b, 0x23,
                0x30, 0xb6, 0xb3, 0x48, 0x52, 0x15, 0xf6, 0x06, 0xf1, 0x88, 0x3a, 0x75, 0x15, 0x88,
                0xc7, 0xef, 0xa5, 0x06, 0xc3, 0xe8, 0xd0, 0xc6, 0x01, 0x92, 0xe8, 0x47, 0x6b, 0xd1,
                0x17, 0x5d, 0x95, 0x62, 0x08, 0x7b, 0xdb, 0x81, 0x8e, 0x66, 0x21, 0x62, 0x86, 0xba,
                0xfe, 0x47, 0xff, 0x4d, 0xbc, 0xce, 0xd5, 0x14, 0x44, 0x48, 0x0a, 0x9a, 0x56, 0x73,
                0xec, 0xe7, 0xfa, 0xc7, 0x3a, 0x0e, 0xd4, 0x1a, 0xb0, 0x05, 0x17, 0x53, 0xa7, 0xca,
                0xa8, 0x9b, 0xe3, 0x13, 0x9a, 0xfd, 0x97, 0x93, 0xb3, 0xe0, 0x2f, 0x27, 0xf0, 0x40,
                0x04, 0x65, 0x95, 0xac, 0xd4, 0x7b, 0xf1, 0x3f, 0xd0, 0xda, 0x27, 0xf0, 0x9e, 0xda,
                0x48, 0x03, 0x6d, 0x3e, 0xe4, 0x37, 0xf2, 0xee, 0x8f, 0x86, 0x06, 0xea, 0x97, 0x34,
                0x3c, 0x33, 0x58, 0x46, 0x57, 0xf4, 0x6d, 0xba, 0x99, 0xdb, 0x5c, 0xfe, 0x6c, 0xa1,
                0x76, 0xfa, 0xb7, 0xb0, 0xf3, 0xbf, 0xa0, 0xab, 0x61, 0xe3, 0x40, 0xc3, 0x4e, 0xb9,
                0xf1, 0x7c, 0x7e, 0xc2, 0xbe, 0x03, 0xb1, 0x80, 0xf0, 0xbb, 0x6f, 0x43, 0x4c, 0x2a,
                0x65, 0x42, 0xe0, 0x0e, 0x84, 0x37, 0x3f, 0x4f,
            ],
            cv_net: [
                0x47, 0x35, 0xa6, 0xfd, 0x21, 0x5c, 0x7b, 0x95, 0x03, 0x3d, 0xab, 0x62, 0xcc, 0xf9,
                0xcd, 0x51, 0x00, 0x89, 0x08, 0xa6, 0xcd, 0xd0, 0xaa, 0x02, 0x1b, 0x88, 0x8b, 0x98,
                0xe2, 0x3c, 0x39, 0x11,
            ],
            rho: [
                0x35, 0x6f, 0xc7, 0x2e, 0x1b, 0xf1, 0xe3, 0xa2, 0xa5, 0x9a, 0xa9, 0xe4, 0x75, 0x15,
                0x5c, 0xf7, 0x43, 0xf8, 0xfd, 0xf0, 0xd1, 0x5b, 0x4c, 0xc4, 0x02, 0x60, 0xd0, 0xd0,
                0x9a, 0xda, 0x04, 0x08,
            ],
            cmx: [
                0xca, 0xa2, 0x8a, 0x69, 0x04, 0x54, 0x66, 0x37, 0xa7, 0xd4, 0xe5, 0xfb, 0xc2, 0x65,
                0x4c, 0xbf, 0x24, 0x4d, 0x18, 0x77, 0x9d, 0x35, 0x62, 0x25, 0x6c, 0x14, 0xd5, 0xb1,
                0x00, 0x5d, 0xc6, 0x0f,
            ],
            esk: [
                0x01, 0xa3, 0x66, 0x1e, 0xa9, 0xaa, 0xb8, 0xf4, 0x32, 0x53, 0x42, 0x0e, 0xff, 0xd7,
                0xa4, 0x83, 0xc2, 0x79, 0xd4, 0x18, 0x18, 0xbc, 0xb3, 0xee, 0x91, 0x90, 0x01, 0xf8,
                0x66, 0xa8, 0xe9, 0x2c,
            ],
            ephemeral_key: [
                0x00, 0x62, 0x7e, 0x29, 0xc1, 0x83, 0x3e, 0x4e, 0x8f, 0xad, 0xe0, 0x82, 0x52, 0xf3,
                0x83, 0x67, 0x78, 0xb9, 0x39, 0x4c, 0x1f, 0xfe, 0xab, 0x70, 0xbf, 0x35, 0x93, 0xd5,
                0x9a, 0x81, 0xa0, 0xa5,
            ],
            shared_secret: [
                0xa9, 0x93, 0x34, 0x1f, 0x99, 0xeb, 0xa8, 0x2d, 0xb6, 0xec, 0x5e, 0x71, 0x3e, 0xe2,
                0x9c, 0x01, 0xf4, 0xce, 0x2a, 0x8b, 0xb6, 0xb5, 0xeb, 0xb6, 0x0b, 0xa6, 0xeb, 0xa9,
                0x6b, 0xa4, 0x17, 0x9b,
            ],
            k_enc: [
                0x35, 0x81, 0x19, 0x19, 0x3c, 0x9d, 0x2a, 0xb6, 0xd4, 0x95, 0xe0, 0x19, 0xf9, 0x7a,
                0x1c, 0x41, 0x30, 0xae, 0xe6, 0x3d, 0xae, 0xc6, 0xbb, 0xa2, 0xf2, 0x74, 0x40, 0x0f,
                0xd3, 0x4f, 0xad, 0x28,
            ],
            p_enc: [
                0x02, 0xc6, 0xe8, 0xf0, 0xd5, 0x0a, 0xe8, 0x05, 0x87, 0x91, 0xdc, 0x0e, 0x46, 0x49,
                0xcd, 0xa3, 0x2b, 0xf6, 0x86, 0x66, 0x25, 0x01, 0xe5, 0x1b, 0x01, 0x2a, 0xea, 0x94,
                0x46, 0xa2, 0x10, 0x4e, 0x93, 0xf8, 0x15, 0xa0, 0xb3, 0xa2, 0x9b, 0x45, 0x83, 0x14,
                0xf3, 0xd8, 0xbe, 0x2b, 0x98, 0x23, 0xd3, 0x42, 0xf4, 0x62, 0xff, 0x13, 0xe9, 0x42,
                0xa7, 0xe1, 0x9a, 0x46, 0xe9, 0x70, 0xb5, 0xc5, 0x06, 0x70, 0x84, 0x30, 0x31, 0x7b,
                0x1b, 0xb3, 0xb3, 0x5d, 0xf6, 0x8a, 0xe3, 0x3a, 0x49, 0x26, 0xa0, 0x3e, 0x6b, 0xfe,
                0xb5, 0x51, 0x04, 0x16, 0xfc, 0xbb, 0x05, 0x24, 0xc9, 0xca, 0x50, 0x74, 0x15, 0x6c,
                0xc5, 0xa5, 0xd6, 0xfe, 0x1c, 0x99, 0x5e, 0xdc, 0x60, 0xa2, 0xf5, 0x50, 0x41, 0x1a,
                0xa4, 0x1e, 0x3d, 0xa3, 0xbd, 0xcf, 0x64, 0xbc, 0xf0, 0x4a, 0x05, 0x10, 0x57, 0x1b,
                0x93, 0x6d, 0x47, 0xe5, 0x5c, 0xec, 0x03, 0x30, 0xee, 0x8d, 0xfe, 0x73, 0x56, 0x34,
                0x04, 0xf0, 0x47, 0xd7, 0xf3, 0xa8, 0xa3, 0xd7, 0x74, 0x3b, 0xc5, 0x54, 0x95, 0x52,
                0x10, 0xf1, 0xeb, 0x0d, 0x08, 0x59, 0x9e, 0xa7, 0x7d, 0x5f, 0x97, 0x4d, 0x87, 0x17,
                0x6d, 0x37, 0xd9, 0x8b, 0x9c, 0x0a, 0xd4, 0x40, 0x40, 0x72, 0x09, 0xed, 0x6a, 0x9f,
                0x08, 0x46, 0x4d, 0x56, 0x55, 0x93, 0xe1, 0xa6, 0x3b, 0x93, 0x85, 0x36, 0xb4, 0x92,
                0x44, 0xe9, 0x7d, 0x88, 0x01, 0x73, 0xb6, 0x40, 0xf2, 0xdd, 0xb7, 0x4d, 0x06, 0x8e,
                0xcb, 0x46, 0xcf, 0x28, 0x9b, 0x7d, 0x89, 0x13, 0x07, 0xbb, 0xa3, 0x70, 0x54, 0xcf,
                0x91, 0xb3, 0x1f, 0xc8, 0x2f, 0x74, 0xd5, 0xfc, 0xc0, 0x00, 0x94, 0x2e, 0xde, 0x91,
                0x18, 0x25, 0xf5, 0x3f, 0xe6, 0x09, 0x68, 0x6f, 0x46, 0x32, 0x23, 0xb1, 0xe9, 0xbc,
                0x03, 0xbd, 0xe8, 0x95, 0xd1, 0x23, 0x8f, 0xad, 0x04, 0xa3, 0xbf, 0xce, 0x68, 0xa0,
                0x75, 0xe8, 0xa3, 0x7c, 0x0e, 0x87, 0xbf, 0x46, 0xdd, 0x01, 0x55, 0x45, 0xf9, 0xb4,
                0xfb, 0x0e, 0xec, 0x64, 0x5f, 0xfc, 0xbb, 0xe0, 0xca, 0x5f, 0x8c, 0x56, 0x1b, 0x25,
                0x7d, 0x52, 0xd6, 0x02, 0xd8, 0xc9, 0x4c, 0x50, 0x28, 0x73, 0xa0, 0x1d, 0x92, 0x51,
                0xd8, 0xc8, 0x60, 0xc0, 0x41, 0x52, 0x5b, 0x3b, 0xf4, 0xe3, 0xa2, 0xeb, 0x92, 0x72,
                0x81, 0x5c, 0x75, 0x86, 0x76, 0x84, 0x28, 0xb4, 0xc2, 0xb2, 0x5e, 0x37, 0x45, 0xf0,
                0x09, 0xc5, 0xdc, 0xe2, 0x0b, 0x69, 0xd5, 0xd7, 0xc4, 0x3c, 0xeb, 0x73, 0x6b, 0x68,
                0x31, 0xe8, 0xc1, 0x10, 0xf1, 0x6c, 0xfd, 0xb3, 0xa4, 0x67, 0xe9, 0x41, 0x4c, 0x00,
                0xec, 0xf1, 0x37, 0x31, 0x50, 0x08, 0x94, 0x55, 0x56, 0x78, 0xc4, 0x97, 0xfa, 0xba,
                0x9a, 0x95, 0xd0, 0x1c, 0xc4, 0x64, 0x39, 0x0f, 0xc4, 0xa7, 0x6b, 0xfa, 0x8b, 0x0e,
                0x1c, 0x68, 0xa5, 0x25, 0xd7, 0x06, 0xd6, 0x60, 0x4b, 0x23, 0x30, 0xb6, 0xb3, 0x48,
                0x52, 0x15, 0xf6, 0x06, 0xf1, 0x88, 0x3a, 0x75, 0x15, 0x88, 0xc7, 0xef, 0xa5, 0x06,
                0xc3, 0xe8, 0xd0, 0xc6, 0x01, 0x92, 0xe8, 0x47, 0x6b, 0xd1, 0x17, 0x5d, 0x95, 0x62,
                0x08, 0x7b, 0xdb, 0x81, 0x8e, 0x66, 0x21, 0x62, 0x86, 0xba, 0xfe, 0x47, 0xff, 0x4d,
                0xbc, 0xce, 0xd5, 0x14, 0x44, 0x48, 0x0a, 0x9a, 0x56, 0x73, 0xec, 0xe7, 0xfa, 0xc7,
                0x3a, 0x0e, 0xd4, 0x1a, 0xb0, 0x05, 0x17, 0x53, 0xa7, 0xca, 0xa8, 0x9b, 0xe3, 0x13,
                0x9a, 0xfd, 0x97, 0x93, 0xb3, 0xe0, 0x2f, 0x27, 0xf0, 0x40, 0x04, 0x65, 0x95, 0xac,
                0xd4, 0x7b, 0xf1, 0x3f, 0xd0, 0xda, 0x27, 0xf0, 0x9e, 0xda, 0x48, 0x03, 0x6d, 0x3e,
                0xe4, 0x37, 0xf2, 0xee, 0x8f, 0x86, 0x06, 0xea, 0x97, 0x34, 0x3c, 0x33, 0x58, 0x46,
                0x57, 0xf4, 0x6d, 0xba, 0x99, 0xdb, 0x5c, 0xfe, 0x6c, 0xa1, 0x76, 0xfa, 0xb7, 0xb0,
                0xf3, 0xbf, 0xa0, 0xab, 0x61, 0xe3, 0x40, 0xc3, 0x4e, 0xb9, 0xf1, 0x7c, 0x7e, 0xc2,
                0xbe, 0x03, 0xb1, 0x80, 0xf0, 0xbb, 0x6f, 0x43, 0x4c, 0x2a, 0x65, 0x42, 0xe0, 0x0e,
                0x84, 0x37, 0x3f, 0x4f,
            ],
            c_enc: [
                0x31, 0xac, 0xda, 0xa6, 0xc1, 0x76, 0xbb, 0x7a, 0x2c, 0x7d, 0x66, 0x09, 0xdc, 0x2c,
                0x5e, 0x7b, 0x2e, 0xe7, 0x1d, 0xa9, 0x3c, 0x73, 0x87, 0x52, 0x74, 0xfa, 0x6c, 0x2a,
                0xd6, 0x26, 0x13, 0xc7, 0x18, 0x9b, 0x35, 0x33, 0xec, 0xf0, 0x34, 0xd8, 0x76, 0xc7,
                0x26, 0xf1, 0xed, 0x99, 0x43, 0xd4, 0x45, 0x07, 0x87, 0x52, 0x75, 0xa8, 0xe3, 0x71,
                0x0a, 0x11, 0x8d, 0x91, 0x64, 0x72, 0x91, 0x28, 0x6c, 0xf8, 0x80, 0xa7, 0x82, 0xab,
                0xea, 0xa8, 0xa6, 0xc3, 0x2f, 0xdf, 0x6f, 0x30, 0x4b, 0x0e, 0xe5, 0xbc, 0xb1, 0x4b,
                0x82, 0x79, 0x2a, 0xa3, 0xaf, 0xd7, 0x24, 0x3f, 0x57, 0xb7, 0xdc, 0xa7, 0x93, 0x52,
                0x19, 0xdf, 0x98, 0x2c, 0xe1, 0x28, 0xae, 0xa6, 0xf6, 0xbd, 0x18, 0xe1, 0x30, 0x7e,
                0xba, 0x0e, 0x3d, 0xb0, 0x06, 0x14, 0xc2, 0x65, 0xc6, 0xf2, 0x8b, 0xfe, 0x58, 0xc1,
                0x1d, 0x4a, 0xc9, 0x6d, 0x49, 0x02, 0x96, 0x7b, 0x54, 0xbc, 0x5d, 0xd1, 0x5c, 0x14,
                0x3f, 0xf4, 0x2b, 0xbb, 0x62, 0xb9, 0x34, 0xb0, 0x9e, 0x79, 0xb6, 0x1e, 0xaf, 0xe7,
                0x9a, 0xbc, 0x86, 0x94, 0x47, 0x5b, 0x6c, 0x8e, 0x19, 0x94, 0xba, 0x05, 0x5e, 0xa3,
                0xc1, 0x82, 0x93, 0xb0, 0x3c, 0x42, 0x49, 0x50, 0x1d, 0xfd, 0xc0, 0x14, 0x60, 0xcf,
                0x78, 0xcd, 0x97, 0x51, 0x30, 0xae, 0x34, 0x05, 0xba, 0x7d, 0xdc, 0x71, 0x30, 0xcb,
                0xdb, 0xb9, 0x8c, 0x7a, 0xaf, 0x6b, 0x1d, 0x0b, 0x44, 0xa5, 0x16, 0x79, 0xaa, 0x63,
                0x0a, 0x43, 0xae, 0x23, 0xb3, 0xd2, 0x2f, 0x73, 0x4c, 0xe1, 0xdb, 0xed, 0xea, 0x17,
                0x5a, 0x00, 0x62, 0xb0, 0x6e, 0x23, 0xf4, 0xd0, 0x6d, 0x2a, 0xd0, 0x45, 0xae, 0x98,
                0x2d, 0xb4, 0x34, 0x8f, 0x20, 0xc8, 0x9b, 0xf9, 0x67, 0x0e, 0x2f, 0xda, 0x47, 0x2e,
                0x55, 0xce, 0x4c, 0x35, 0x82, 0xb7, 0x64, 0x43, 0xe0, 0xab, 0xbb, 0x77, 0x8a, 0xec,
                0xa0, 0xf3, 0x9c, 0x55, 0xb6, 0xab, 0xbe, 0xd8, 0x1f, 0xde, 0x89, 0xad, 0x2c, 0x56,
                0x7d, 0xfe, 0x27, 0x7b, 0xb2, 0x69, 0xac, 0x6a, 0xe0, 0xe1, 0x88, 0x39, 0x8d, 0xea,
                0x24, 0xad, 0xcc, 0xe1, 0x82, 0xe7, 0xfd, 0xdc, 0x80, 0xeb, 0xd7, 0x69, 0xd0, 0xf0,
                0x76, 0xf9, 0xaf, 0x2d, 0xd9, 0x83, 0x07, 0xa5, 0x27, 0xc6, 0x99, 0x42, 0xdf, 0xa3,
                0xe7, 0xf4, 0x86, 0x76, 0x10, 0x1a, 0x47, 0xeb, 0x07, 0x80, 0xb6, 0x90, 0xb1, 0xaf,
                0x10, 0xfc, 0xfb, 0x5e, 0xe2, 0xbd, 0x40, 0xd2, 0x7d, 0x10, 0x9b, 0xa1, 0x5a, 0xb0,
                0xb1, 0xe9, 0x55, 0x4f, 0xdd, 0xfa, 0x81, 0x6b, 0x99, 0xcd, 0x8f, 0xdd, 0xe6, 0x81,
                0xae, 0x6b, 0x6c, 0xbb, 0xfb, 0xf0, 0x2c, 0x36, 0x32, 0x68, 0xd0, 0xf3, 0xc6, 0xa7,
                0x26, 0x1b, 0x6d, 0x00, 0x87, 0xbc, 0xad, 0xb6, 0xfb, 0x9b, 0xf3, 0x93, 0x04, 0xfc,
                0x08, 0x41, 0x5d, 0x83, 0x6f, 0xe4, 0x09, 0xa4, 0x3f, 0xaf, 0x9e, 0x28, 0xfb, 0x48,
                0x3f, 0x4a, 0x47, 0xaa, 0xd7, 0xe1, 0xf7, 0x97, 0x30, 0xb3, 0x21, 0x53, 0x60, 0x80,
                0xdb, 0x35, 0x12, 0x48, 0xb2, 0x66, 0x9b, 0x6e, 0x74, 0x48, 0x90, 0x87, 0xae, 0x72,
                0xba, 0x15, 0xd2, 0xae, 0xdd, 0x0c, 0x1e, 0x7e, 0xb1, 0x5a, 0x2f, 0x5a, 0x77, 0x31,
                0xeb, 0x45, 0xa6, 0x17, 0x8a, 0x44, 0x87, 0x09, 0x31, 0xec, 0x8e, 0x34, 0x8c, 0x19,
                0x2b, 0xc8, 0x87, 0xc8, 0x63, 0x60, 0x56, 0x67, 0x6f, 0x58, 0xd0, 0xc6, 0x34, 0xfc,
                0x99, 0xea, 0x7b, 0x07, 0xfa, 0x1b, 0x62, 0x99, 0xae, 0x5d, 0xbf, 0xe0, 0x84, 0x45,
                0xad, 0x99, 0x9f, 0x45, 0xdf, 0x00, 0xf1, 0xa4, 0x7a, 0xa5, 0xef, 0x6f, 0x88, 0xcd,
                0xba, 0x80, 0xc8, 0x8f, 0x94, 0x01, 0xe6, 0xe9, 0x09, 0xca, 0x2c, 0x5d, 0xe2, 0xcf,
                0x8f, 0x6a, 0x98, 0x44, 0xca, 0x32, 0xfe, 0x91, 0xf7, 0x13, 0xfe, 0x10, 0xa0, 0x69,
                0x8a, 0x1b, 0x3b, 0xfd, 0xf4, 0x47, 0x43, 0x75, 0xb4, 0x79, 0x1d, 0xc8, 0x50, 0x50,
                0xc9, 0x28, 0x90, 0x0e, 0x73, 0x1f, 0x7c, 0x4a, 0x12, 0x9d, 0x8e, 0x21, 0xfc, 0xf4,
                0x17, 0x62, 0x7c, 0x47, 0xdd, 0xc9, 0xf5, 0x88, 0x40, 0x38, 0x41, 0x31, 0x7a, 0x9a,
                0xc2, 0x6e, 0xef, 0x6c, 0xda, 0x23,
            ],
            ock: [
                0x9c, 0x10, 0x3d, 0xd5, 0xd6, 0x38, 0x6c, 0xdd, 0x67, 0x69, 0x51, 0xe6, 0x56, 0x4b,
                0x16, 0x6b, 0xc5, 0xc5, 0x72, 0x32, 0xcf, 0xc3, 0x1e, 0x0e, 0x69, 0xce, 0x84, 0xda,
                0xe8, 0x32, 0x76, 0xbe,
            ],
            op: [
                0x8e, 0x66, 0xb7, 0x92, 0xec, 0xb1, 0x56, 0xef, 0x68, 0x5e, 0xe8, 0xea, 0x35, 0xd3,
                0x82, 0x75, 0x8b, 0xa4, 0x15, 0x97, 0xa3, 0x3a, 0x93, 0xba, 0xf3, 0x81, 0xd6, 0x3c,
                0x17, 0x5b, 0xa9, 0x8b, 0x01, 0xa3, 0x66, 0x1e, 0xa9, 0xaa, 0xb8, 0xf4, 0x32, 0x53,
                0x42, 0x0e, 0xff, 0xd7, 0xa4, 0x83, 0xc2, 0x79, 0xd4, 0x18, 0x18, 0xbc, 0xb3, 0xee,
                0x91, 0x90, 0x01, 0xf8, 0x66, 0xa8, 0xe9, 0x2c,
            ],
            c_out: [
                0xdf, 0x6b, 0xb6, 0x2e, 0x3b, 0x64, 0xf7, 0xe9, 0x37, 0xc1, 0xde, 0x38, 0xaa, 0xc5,
                0xe6, 0xb1, 0x61, 0x35, 0xba, 0x2f, 0x9d, 0xbe, 0xed, 0xb0, 0x7a, 0x45, 0xb6, 0xf0,
                0x9d, 0xf3, 0xeb, 0xec, 0xac, 0x04, 0x6c, 0x61, 0x92, 0xfb, 0xe8, 0x95, 0xd2, 0x31,
                0x02, 0x77, 0xb3, 0xe0, 0x3a, 0x90, 0xfd, 0xc8, 0x53, 0x48, 0x4a, 0x01, 0x5c, 0x88,
                0xd6, 0x63, 0x2e, 0x97, 0xba, 0x98, 0xad, 0xca, 0x9e, 0x49, 0xf5, 0x38, 0xc3, 0xa7,
                0xcb, 0x6d, 0x77, 0x23, 0xbd, 0xcc, 0x9c, 0x03, 0xae, 0x88,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0x73, 0xa2, 0x5e, 0xba, 0x9b, 0xd7, 0xa8, 0xed, 0x2b, 0x5b, 0x1b, 0x8d, 0x5a, 0x05,
                0x6b, 0xde, 0x8d, 0x05, 0xe6, 0xa2, 0x80, 0x67, 0xb3, 0x84, 0x57, 0x91, 0xbe, 0xbf,
                0xa7, 0xae, 0x2a, 0xcd, 0x36, 0x32, 0x6f, 0xe6, 0x27, 0xbe, 0xe8, 0x0e, 0x32, 0x92,
                0xe0, 0xe5, 0x13, 0x2d, 0xe1, 0x6c, 0xa4, 0xf8, 0x1e, 0x5a, 0x6f, 0xc0, 0x9c, 0x95,
                0xff, 0x13, 0xb5, 0x2e, 0x96, 0xb7, 0x89, 0x0f,
            ],
            ovk: [
                0xf5, 0xe8, 0xde, 0xd8, 0x18, 0x92, 0x51, 0x1c, 0xc2, 0x85, 0x1b, 0x00, 0xb8, 0x32,
                0x71, 0x2a, 0x6d, 0x3b, 0xa5, 0x66, 0x65, 0x17, 0xbc, 0xd3, 0x56, 0x76, 0x21, 0xa7,
                0xcf, 0x84, 0x45, 0x58,
            ],
            default_d: [
                0x81, 0xf2, 0x75, 0x7c, 0x53, 0x2e, 0xd3, 0xb6, 0x2e, 0x89, 0x01,
            ],
            default_pk_d: [
                0x55, 0xdb, 0x72, 0x90, 0x07, 0x3b, 0xa0, 0x06, 0x66, 0xe8, 0x7d, 0x25, 0x61, 0xb8,
                0x88, 0x3c, 0x66, 0x2c, 0x56, 0x78, 0xff, 0x27, 0x30, 0x2a, 0x82, 0xe2, 0x0a, 0x72,
                0x01, 0x70, 0x89, 0x1a,
            ],
            v: 17209482587585417762,
            rseed: [
                0xfc, 0x54, 0x88, 0x62, 0xf5, 0xa0, 0x70, 0x94, 0xfd, 0x42, 0x8a, 0x7b, 0xbc, 0x15,
                0xd7, 0xb3, 0x8d, 0x05, 0x36, 0x2c, 0x9c, 0xa9, 0x85, 0xf5, 0x8a, 0x76, 0x64, 0x7d,
                0x2b, 0xe4, 0xc2, 0xcd,
            ],
            memo: [
                0xff, 0x6b, 0x3d, 0x17, 0xd6, 0x87, 0x09, 0x71, 0xd7, 0xa0, 0x98, 0xba, 0xf7, 0x2c,
                0x6f, 0x6f, 0x12, 0x14, 0xcf, 0x1f, 0xaa, 0xe4, 0x88, 0xbd, 0x7d, 0xe2, 0x59, 0xd3,
                0x41, 0x5c, 0x2f, 0x0d, 0xde, 0xc7, 0x45, 0x70, 0x04, 0xf3, 0x57, 0x08, 0xd1, 0xec,
                0xcc, 0xcc, 0x0d, 0xf6, 0x5a, 0x04, 0x94, 0x3a, 0xd5, 0xcb, 0xc1, 0x3f, 0x29, 0x5f,
                0x00, 0x0f, 0xe0, 0x56, 0xc4, 0x0b, 0x2d, 0x88, 0xf2, 0x7d, 0xc3, 0x4c, 0xfe, 0xb8,
                0x03, 0xbe, 0x34, 0x83, 0xa9, 0xeb, 0xf9, 0xb5, 0xa9, 0x02, 0x60, 0x57, 0x72, 0x5d,
                0x63, 0xea, 0xd2, 0xc0, 0xc0, 0xff, 0x1f, 0xe2, 0x6a, 0xc1, 0xe7, 0xbd, 0xfc, 0xd6,
                0xfa, 0xd8, 0x75, 0x84, 0x2d, 0x19, 0x4f, 0x33, 0x17, 0x50, 0x46, 0x2c, 0x06, 0xb8,
                0xd7, 0x98, 0x2d, 0x67, 0x99, 0x5e, 0xd5, 0xd3, 0xae, 0x96, 0xa0, 0x5a, 0xe0, 0x06,
                0x7f, 0x4e, 0xb1, 0xc7, 0xc9, 0x32, 0x31, 0xbd, 0x39, 0x77, 0x3c, 0xbe, 0x0a, 0x9d,
                0x66, 0xb0, 0xc9, 0xaa, 0x8c, 0xff, 0x6a, 0x37, 0x6e, 0x1f, 0x37, 0x2e, 0xac, 0x6a,
                0xc4, 0xe4, 0x6c, 0xc0, 0x94, 0x22, 0x45, 0xd4, 0xc2, 0xdc, 0xf0, 0x2d, 0x76, 0x40,
                0xff, 0xcc, 0x5a, 0x6a, 0xc3, 0xa8, 0x7f, 0x5c, 0x41, 0x15, 0x51, 0xbc, 0xc2, 0xf2,
                0x6c, 0xb9, 0x49, 0x61, 0xd5, 0x3f, 0x95, 0xdd, 0xb1, 0x9a, 0xe9, 0x30, 0xc8, 0xd7,
                0x0f, 0x03, 0x1b, 0x29, 0xa5, 0xdf, 0x99, 0xff, 0x36, 0x69, 0x5e, 0x80, 0x2c, 0xbc,
                0xb6, 0xb5, 0x8c, 0x1b, 0xa7, 0xed, 0x5e, 0xac, 0xfa, 0x76, 0x41, 0x4a, 0x41, 0xad,
                0x4a, 0x44, 0xf7, 0x1f, 0x1b, 0x58, 0x0d, 0x34, 0xc3, 0xa9, 0x52, 0x92, 0x0b, 0x25,
                0x4a, 0x14, 0x5f, 0xea, 0x51, 0x7f, 0x5b, 0x42, 0xb2, 0xf6, 0x5e, 0xcd, 0x0f, 0x82,
                0x59, 0x54, 0x78, 0xd8, 0x0a, 0xe5, 0xc8, 0xce, 0xea, 0x12, 0xa1, 0x61, 0xcc, 0xbb,
                0x5e, 0xac, 0x09, 0x99, 0x0f, 0xc6, 0x19, 0xa4, 0x60, 0x80, 0x43, 0x6d, 0xbd, 0x08,
                0xd7, 0x47, 0x84, 0xaf, 0x00, 0x2d, 0x58, 0xe0, 0x6f, 0xaf, 0x7f, 0x3c, 0xea, 0xe7,
                0xd3, 0x41, 0x9b, 0x1f, 0xca, 0x26, 0x5a, 0x55, 0x59, 0xcf, 0x9e, 0x2d, 0x3b, 0x60,
                0x97, 0x8d, 0x81, 0xa6, 0x78, 0xb9, 0xed, 0x8e, 0x44, 0x86, 0xb4, 0xd1, 0x46, 0x09,
                0xd6, 0xc1, 0x27, 0xc0, 0xc2, 0xfb, 0xff, 0xe3, 0x0a, 0x60, 0xf7, 0xbf, 0xf1, 0xd9,
                0xfb, 0x83, 0x00, 0xed, 0x00, 0x92, 0x53, 0xba, 0x9b, 0x99, 0x6f, 0xa0, 0x52, 0x41,
                0xb1, 0x0f, 0x5a, 0xc9, 0xa8, 0x40, 0x8e, 0x92, 0x5b, 0x62, 0x6b, 0xb2, 0x1a, 0x47,
                0x1f, 0xe3, 0xbe, 0xde, 0x52, 0xbb, 0xa0, 0x97, 0xb2, 0xa9, 0x9a, 0x9b, 0xa5, 0xa8,
                0x66, 0x58, 0xc3, 0xfd, 0x9e, 0xc5, 0x5b, 0xfa, 0x9b, 0x32, 0x85, 0x67, 0x25, 0x4a,
                0xb3, 0x6d, 0x2c, 0x7f, 0x44, 0xd2, 0xc7, 0xe1, 0x3e, 0xb5, 0x4b, 0xeb, 0x70, 0xea,
                0x8f, 0xa9, 0x4b, 0x6c, 0x6e, 0x01, 0x2d, 0x79, 0xe3, 0xf5, 0x36, 0x89, 0xc2, 0xb1,
                0xa1, 0x8e, 0xaf, 0x2d, 0x47, 0x1d, 0x13, 0xc1, 0xab, 0x39, 0xd9, 0x19, 0x4a, 0xe8,
                0x43, 0xab, 0x1d, 0x28, 0xff, 0xa8, 0xf6, 0x9d, 0xc7, 0xe1, 0x5c, 0xc3, 0x8b, 0x12,
                0xe8, 0xfc, 0xd7, 0x92, 0x55, 0xb7, 0x21, 0x60, 0x56, 0xd9, 0xed, 0xb7, 0x48, 0x2f,
                0xb9, 0x8a, 0xa0, 0x33, 0xb6, 0x5e, 0x51, 0xc1, 0xa0, 0x8b, 0x8a, 0x11, 0xd8, 0x4d,
                0x04, 0x09, 0xb7, 0x34, 0xf4, 0x52, 0xaa, 0xf0, 0xd6, 0xb1, 0x8f, 0x50, 0x25, 0x86,
                0x83, 0xd3, 0xf9, 0xa7, 0x6d, 0x39, 0x9f, 0xd0, 0x47, 0xee, 0xe2, 0x88, 0xbb, 0x45,
                0x85, 0x85, 0x1d, 0xc9, 0x3e, 0xcc, 0xc6, 0x23,
            ],
            cv_net: [
                0xe8, 0x06, 0x5c, 0x40, 0x96, 0xd3, 0x54, 0x33, 0x40, 0x01, 0x1f, 0x58, 0x90, 0xb1,
                0x7e, 0xed, 0xd2, 0xa7, 0x06, 0x44, 0x07, 0x34, 0x78, 0x41, 0x01, 0xae, 0x2d, 0x8e,
                0x87, 0xe5, 0x05, 0xad,
            ],
            rho: [
                0x32, 0x91, 0x87, 0x35, 0x66, 0x3f, 0x34, 0xad, 0xa0, 0x22, 0x8a, 0xea, 0x4a, 0xcc,
                0x19, 0x2a, 0x12, 0x3f, 0xcf, 0xa0, 0x60, 0x46, 0x89, 0xf9, 0x1a, 0xcb, 0xe9, 0x38,
                0x31, 0xe4, 0x8c, 0x0c,
            ],
            cmx: [
                0xde, 0x7b, 0xf1, 0x55, 0x08, 0x29, 0x51, 0x96, 0x0a, 0x10, 0xbb, 0x8b, 0x75, 0x41,
                0x02, 0x43, 0x0f, 0x89, 0xf5, 0x32, 0x67, 0x24, 0x04, 0x36, 0x81, 0xf5, 0x06, 0xf7,
                0x48, 0xe2, 0x6f, 0x07,
            ],
            esk: [
                0x1d, 0xb6, 0x79, 0x07, 0x9a, 0xcd, 0xef, 0xe9, 0xfc, 0x1e, 0x59, 0xa1, 0x33, 0xf3,
                0x7b, 0x6d, 0x1f, 0xfb, 0xed, 0x78, 0x8c, 0xce, 0x3b, 0x0c, 0xdd, 0x63, 0xe0, 0x62,
                0x83, 0x05, 0x47, 0x08,
            ],
            ephemeral_key: [
                0x18, 0x20, 0x84, 0x5b, 0x2d, 0x90, 0xe5, 0x45, 0x76, 0x0f, 0xca, 0x4d, 0xab, 0x30,
                0xa6, 0x78, 0x3e, 0x03, 0x1c, 0x0e, 0x54, 0x28, 0xcc, 0x22, 0x6f, 0x3f, 0x40, 0x1d,
                0xec, 0x20, 0x0b, 0x11,
            ],
            shared_secret: [
                0x79, 0x38, 0x77, 0xa1, 0xae, 0xda, 0xe9, 0xac, 0x6d, 0xa3, 0xd7, 0xe8, 0x02, 0xb5,
                0xbc, 0x77, 0x3c, 0x0d, 0x93, 0x1c, 0x79, 0x6d, 0x17, 0x0c, 0x59, 0x7f, 0x22, 0xba,
                0x6f, 0xcc, 0xa2, 0x06,
            ],
            k_enc: [
                0x28, 0x20, 0xb9, 0x38, 0xd6, 0xfc, 0xea, 0x99, 0xa7, 0x23, 0x37, 0x50, 0xa0, 0xf0,
                0x9a, 0x32, 0x10, 0xad, 0x91, 0x73, 0x46, 0x16, 0x6a, 0xea, 0xdc, 0x89, 0xbb, 0x50,
                0xf4, 0x54, 0x50, 0xa3,
            ],
            p_enc: [
                0x02, 0x81, 0xf2, 0x75, 0x7c, 0x53, 0x2e, 0xd3, 0xb6, 0x2e, 0x89, 0x01, 0x22, 0x92,
                0x4c, 0xd1, 0x3b, 0x5d, 0xd4, 0xee, 0xfc, 0x54, 0x88, 0x62, 0xf5, 0xa0, 0x70, 0x94,
                0xfd, 0x42, 0x8a, 0x7b, 0xbc, 0x15, 0xd7, 0xb3, 0x8d, 0x05, 0x36, 0x2c, 0x9c, 0xa9,
                0x85, 0xf5, 0x8a, 0x76, 0x64, 0x7d, 0x2b, 0xe4, 0xc2, 0xcd, 0xff, 0x6b, 0x3d, 0x17,
                0xd6, 0x87, 0x09, 0x71, 0xd7, 0xa0, 0x98, 0xba, 0xf7, 0x2c, 0x6f, 0x6f, 0x12, 0x14,
                0xcf, 0x1f, 0xaa, 0xe4, 0x88, 0xbd, 0x7d, 0xe2, 0x59, 0xd3, 0x41, 0x5c, 0x2f, 0x0d,
                0xde, 0xc7, 0x45, 0x70, 0x04, 0xf3, 0x57, 0x08, 0xd1, 0xec, 0xcc, 0xcc, 0x0d, 0xf6,
                0x5a, 0x04, 0x94, 0x3a, 0xd5, 0xcb, 0xc1, 0x3f, 0x29, 0x5f, 0x00, 0x0f, 0xe0, 0x56,
                0xc4, 0x0b, 0x2d, 0x88, 0xf2, 0x7d, 0xc3, 0x4c, 0xfe, 0xb8, 0x03, 0xbe, 0x34, 0x83,
                0xa9, 0xeb, 0xf9, 0xb5, 0xa9, 0x02, 0x60, 0x57, 0x72, 0x5d, 0x63, 0xea, 0xd2, 0xc0,
                0xc0, 0xff, 0x1f, 0xe2, 0x6a, 0xc1, 0xe7, 0xbd, 0xfc, 0xd6, 0xfa, 0xd8, 0x75, 0x84,
                0x2d, 0x19, 0x4f, 0x33, 0x17, 0x50, 0x46, 0x2c, 0x06, 0xb8, 0xd7, 0x98, 0x2d, 0x67,
                0x99, 0x5e, 0xd5, 0xd3, 0xae, 0x96, 0xa0, 0x5a, 0xe0, 0x06, 0x7f, 0x4e, 0xb1, 0xc7,
                0xc9, 0x32, 0x31, 0xbd, 0x39, 0x77, 0x3c, 0xbe, 0x0a, 0x9d, 0x66, 0xb0, 0xc9, 0xaa,
                0x8c, 0xff, 0x6a, 0x37, 0x6e, 0x1f, 0x37, 0x2e, 0xac, 0x6a, 0xc4, 0xe4, 0x6c, 0xc0,
                0x94, 0x22, 0x45, 0xd4, 0xc2, 0xdc, 0xf0, 0x2d, 0x76, 0x40, 0xff, 0xcc, 0x5a, 0x6a,
                0xc3, 0xa8, 0x7f, 0x5c, 0x41, 0x15, 0x51, 0xbc, 0xc2, 0xf2, 0x6c, 0xb9, 0x49, 0x61,
                0xd5, 0x3f, 0x95, 0xdd, 0xb1, 0x9a, 0xe9, 0x30, 0xc8, 0xd7, 0x0f, 0x03, 0x1b, 0x29,
                0xa5, 0xdf, 0x99, 0xff, 0x36, 0x69, 0x5e, 0x80, 0x2c, 0xbc, 0xb6, 0xb5, 0x8c, 0x1b,
                0xa7, 0xed, 0x5e, 0xac, 0xfa, 0x76, 0x41, 0x4a, 0x41, 0xad, 0x4a, 0x44, 0xf7, 0x1f,
                0x1b, 0x58, 0x0d, 0x34, 0xc3, 0xa9, 0x52, 0x92, 0x0b, 0x25, 0x4a, 0x14, 0x5f, 0xea,
                0x51, 0x7f, 0x5b, 0x42, 0xb2, 0xf6, 0x5e, 0xcd, 0x0f, 0x82, 0x59, 0x54, 0x78, 0xd8,
                0x0a, 0xe5, 0xc8, 0xce, 0xea, 0x12, 0xa1, 0x61, 0xcc, 0xbb, 0x5e, 0xac, 0x09, 0x99,
                0x0f, 0xc6, 0x19, 0xa4, 0x60, 0x80, 0x43, 0x6d, 0xbd, 0x08, 0xd7, 0x47, 0x84, 0xaf,
                0x00, 0x2d, 0x58, 0xe0, 0x6f, 0xaf, 0x7f, 0x3c, 0xea, 0xe7, 0xd3, 0x41, 0x9b, 0x1f,
                0xca, 0x26, 0x5a, 0x55, 0x59, 0xcf, 0x9e, 0x2d, 0x3b, 0x60, 0x97, 0x8d, 0x81, 0xa6,
                0x78, 0xb9, 0xed, 0x8e, 0x44, 0x86, 0xb4, 0xd1, 0x46, 0x09, 0xd6, 0xc1, 0x27, 0xc0,
                0xc2, 0xfb, 0xff, 0xe3, 0x0a, 0x60, 0xf7, 0xbf, 0xf1, 0xd9, 0xfb, 0x83, 0x00, 0xed,
                0x00, 0x92, 0x53, 0xba, 0x9b, 0x99, 0x6f, 0xa0, 0x52, 0x41, 0xb1, 0x0f, 0x5a, 0xc9,
                0xa8, 0x40, 0x8e, 0x92, 0x5b, 0x62, 0x6b, 0xb2, 0x1a, 0x47, 0x1f, 0xe3, 0xbe, 0xde,
                0x52, 0xbb, 0xa0, 0x97, 0xb2, 0xa9, 0x9a, 0x9b, 0xa5, 0xa8, 0x66, 0x58, 0xc3, 0xfd,
                0x9e, 0xc5, 0x5b, 0xfa, 0x9b, 0x32, 0x85, 0x67, 0x25, 0x4a, 0xb3, 0x6d, 0x2c, 0x7f,
                0x44, 0xd2, 0xc7, 0xe1, 0x3e, 0xb5, 0x4b, 0xeb, 0x70, 0xea, 0x8f, 0xa9, 0x4b, 0x6c,
                0x6e, 0x01, 0x2d, 0x79, 0xe3, 0xf5, 0x36, 0x89, 0xc2, 0xb1, 0xa1, 0x8e, 0xaf, 0x2d,
                0x47, 0x1d, 0x13, 0xc1, 0xab, 0x39, 0xd9, 0x19, 0x4a, 0xe8, 0x43, 0xab, 0x1d, 0x28,
                0xff, 0xa8, 0xf6, 0x9d, 0xc7, 0xe1, 0x5c, 0xc3, 0x8b, 0x12, 0xe8, 0xfc, 0xd7, 0x92,
                0x55, 0xb7, 0x21, 0x60, 0x56, 0xd9, 0xed, 0xb7, 0x48, 0x2f, 0xb9, 0x8a, 0xa0, 0x33,
                0xb6, 0x5e, 0x51, 0xc1, 0xa0, 0x8b, 0x8a, 0x11, 0xd8, 0x4d, 0x04, 0x09, 0xb7, 0x34,
                0xf4, 0x52, 0xaa, 0xf0, 0xd6, 0xb1, 0x8f, 0x50, 0x25, 0x86, 0x83, 0xd3, 0xf9, 0xa7,
                0x6d, 0x39, 0x9f, 0xd0, 0x47, 0xee, 0xe2, 0x88, 0xbb, 0x45, 0x85, 0x85, 0x1d, 0xc9,
                0x3e, 0xcc, 0xc6, 0x23,
            ],
            c_enc: [
                0xd9, 0x6a, 0xe8, 0x2f, 0xba, 0xff, 0xb9, 0xe4, 0xbd, 0x36, 0x47, 0x57, 0x96, 0x33,
                0xbc, 0x8a, 0x89, 0x66, 0xae, 0x4e, 0x18, 0x85, 0x99, 0xdc, 0x3c, 0xf0, 0x30, 0x41,
                0xd2, 0x64, 0x4f, 0x60, 0x3c, 0xe5, 0x56, 0x2e, 0x7f, 0xa1, 0xd3, 0x83, 0x12, 0x79,
                0xb6, 0x32, 0x60, 0x4d, 0x06, 0x5e, 0xd2, 0x46, 0xad, 0x2d, 0x4f, 0x73, 0xf3, 0xa4,
                0x1b, 0x2a, 0x27, 0x99, 0xe5, 0xba, 0xf8, 0x6e, 0x7d, 0x9f, 0xcc, 0x64, 0xd5, 0x6b,
                0xde, 0x56, 0xcb, 0xb0, 0x01, 0xeb, 0x7c, 0x7c, 0x0c, 0xf1, 0xe2, 0xae, 0xc3, 0xce,
                0xb1, 0x49, 0x2d, 0xdf, 0x4e, 0x35, 0x20, 0x76, 0x1f, 0x70, 0xf6, 0xa9, 0x5c, 0x9e,
                0xde, 0xed, 0x51, 0x43, 0x6d, 0xca, 0xcf, 0x71, 0x1e, 0xb5, 0x53, 0x24, 0xd3, 0xf0,
                0x1d, 0xcc, 0xa0, 0x1b, 0xca, 0x15, 0xba, 0xf2, 0x8f, 0xed, 0x81, 0x4c, 0xc3, 0x3a,
                0x43, 0x6b, 0xbc, 0x08, 0x60, 0x56, 0x78, 0x55, 0xa4, 0x9f, 0x5e, 0xfd, 0x49, 0xea,
                0x78, 0x30, 0xc2, 0xf6, 0x00, 0x61, 0xd8, 0x13, 0xa8, 0x49, 0xb4, 0x40, 0xb1, 0x2e,
                0x8f, 0x31, 0xe2, 0xdc, 0x0c, 0x39, 0x4c, 0xeb, 0x92, 0x4f, 0x0d, 0xc6, 0xd5, 0x7b,
                0xcc, 0x39, 0x1d, 0x4a, 0x5f, 0x56, 0x9f, 0x34, 0x74, 0x6b, 0x9d, 0x92, 0x08, 0x82,
                0x25, 0xb1, 0xaa, 0x3e, 0x90, 0x6f, 0x6d, 0xe4, 0xa7, 0x92, 0x2a, 0xc1, 0x16, 0xac,
                0xb2, 0x78, 0xe8, 0xef, 0xbb, 0xf6, 0xf1, 0xe3, 0xe0, 0x8d, 0x66, 0xb3, 0x4d, 0x6e,
                0xde, 0x34, 0xae, 0x78, 0x40, 0xa8, 0x80, 0x2b, 0x7c, 0x10, 0x0f, 0xa7, 0x98, 0x46,
                0x4d, 0xb5, 0x29, 0x45, 0xdb, 0xbe, 0x35, 0xe7, 0x7a, 0x77, 0x21, 0xb2, 0xe0, 0xf8,
                0xb9, 0xa1, 0x0f, 0x6b, 0xf9, 0xa2, 0x80, 0x8f, 0xa5, 0x85, 0xff, 0x21, 0xb0, 0xa3,
                0xaf, 0xfd, 0x5f, 0x3c, 0xb2, 0x30, 0x63, 0x4e, 0x2e, 0x43, 0xff, 0xde, 0x6d, 0x09,
                0x0a, 0xfc, 0xc6, 0x70, 0x4d, 0x7b, 0x3f, 0xc6, 0x15, 0x44, 0x85, 0x3b, 0xcd, 0xa7,
                0xa1, 0x06, 0xd8, 0x5b, 0xd8, 0xd8, 0x6c, 0x6f, 0x8c, 0xe0, 0x34, 0x01, 0x6e, 0xd9,
                0xa3, 0x69, 0x46, 0xf8, 0x52, 0x8b, 0x6f, 0x1e, 0x1a, 0x19, 0x82, 0xd6, 0x8d, 0x38,
                0x86, 0xe5, 0xea, 0xe6, 0xc1, 0xe3, 0x88, 0xf3, 0xde, 0xad, 0x0d, 0x35, 0x3b, 0x6c,
                0x0c, 0xbf, 0x57, 0xc2, 0xe4, 0x7d, 0x30, 0x72, 0x3e, 0xac, 0x95, 0x7b, 0x4a, 0xec,
                0x82, 0xc8, 0xa1, 0x00, 0x9e, 0x3d, 0x71, 0x96, 0x92, 0xb2, 0xfc, 0xbd, 0xda, 0xae,
                0x62, 0x5a, 0x89, 0x6a, 0x47, 0x29, 0x85, 0xb7, 0x9e, 0xb6, 0x2b, 0x1f, 0xe3, 0x3d,
                0x6e, 0x27, 0xbc, 0x1f, 0x10, 0xe4, 0xfe, 0x5c, 0x06, 0xb2, 0x4c, 0x59, 0x7f, 0x72,
                0x3c, 0x67, 0x13, 0x36, 0x13, 0xae, 0x8e, 0x15, 0x4d, 0x81, 0x69, 0x78, 0xb8, 0xfc,
                0xa6, 0x50, 0xc0, 0x1d, 0x77, 0x1a, 0x62, 0x69, 0x2a, 0x84, 0x82, 0x94, 0x2e, 0x28,
                0xfe, 0xf0, 0x45, 0x19, 0x9e, 0xd1, 0xa6, 0x64, 0x99, 0xeb, 0xa1, 0xee, 0xc1, 0x9a,
                0xc8, 0x4a, 0x12, 0xe4, 0x10, 0x29, 0xd2, 0x2f, 0x21, 0x87, 0x6e, 0xd7, 0x4a, 0x76,
                0xef, 0x39, 0xa0, 0x57, 0xce, 0x0a, 0x15, 0x8e, 0x68, 0x51, 0xec, 0x35, 0x6d, 0x97,
                0x7b, 0x1a, 0xa6, 0x8d, 0xcf, 0x70, 0x88, 0xa9, 0xf0, 0xf9, 0xe4, 0x75, 0xa2, 0xbb,
                0xc1, 0xc4, 0x49, 0x5b, 0x54, 0x6f, 0xff, 0xed, 0xaa, 0x66, 0xc4, 0xf9, 0x51, 0x74,
                0xc6, 0x2f, 0x56, 0x5a, 0x3c, 0xc0, 0xac, 0xaf, 0x85, 0x4a, 0xde, 0xd4, 0xb4, 0x25,
                0xa0, 0xc7, 0xdb, 0xcd, 0x37, 0x42, 0xa7, 0xe0, 0x59, 0x2e, 0x83, 0x73, 0x41, 0xf8,
                0x95, 0x32, 0x90, 0x99, 0xe1, 0x70, 0xb6, 0xff, 0xb1, 0x05, 0xfd, 0xbc, 0x77, 0x29,
                0x8e, 0x8c, 0x0f, 0x5e, 0xeb, 0x9e, 0x99, 0xc6, 0x58, 0x4b, 0xcf, 0xf1, 0x20, 0x20,
                0x9d, 0x69, 0x22, 0xb5, 0x34, 0xbe, 0xc9, 0xfa, 0xc1, 0xd7, 0xd2, 0x74, 0xdb, 0xcb,
                0x4a, 0x12, 0xea, 0x5a, 0x99, 0x21, 0x39, 0x2f, 0x00, 0x96, 0x60, 0x29, 0xff, 0x26,
                0x0e, 0xd2, 0x4c, 0x32, 0x78, 0x58, 0xe0, 0x34, 0x64, 0x0a, 0x8c, 0xfb, 0x28, 0xad,
                0x97, 0xe5, 0x3d, 0x80, 0xe1, 0xc9,
            ],
            ock: [
                0x28, 0xcf, 0x3b, 0xea, 0xc3, 0xbd, 0xe2, 0xe9, 0x63, 0xaa, 0x60, 0x91, 0x3f, 0x10,
                0x5d, 0x25, 0x67, 0xcd, 0xaf, 0xbb, 0x66, 0x09, 0x08, 0x5a, 0x84, 0x3f, 0x75, 0x68,
                0xe4, 0x92, 0xd4, 0x4e,
            ],
            op: [
                0x55, 0xdb, 0x72, 0x90, 0x07, 0x3b, 0xa0, 0x06, 0x66, 0xe8, 0x7d, 0x25, 0x61, 0xb8,
                0x88, 0x3c, 0x66, 0x2c, 0x56, 0x78, 0xff, 0x27, 0x30, 0x2a, 0x82, 0xe2, 0x0a, 0x72,
                0x01, 0x70, 0x89, 0x1a, 0x1d, 0xb6, 0x79, 0x07, 0x9a, 0xcd, 0xef, 0xe9, 0xfc, 0x1e,
                0x59, 0xa1, 0x33, 0xf3, 0x7b, 0x6d, 0x1f, 0xfb, 0xed, 0x78, 0x8c, 0xce, 0x3b, 0x0c,
                0xdd, 0x63, 0xe0, 0x62, 0x83, 0x05, 0x47, 0x08,
            ],
            c_out: [
                0x0b, 0x2c, 0xc0, 0xa2, 0x2d, 0x06, 0xfc, 0x36, 0xa0, 0x8a, 0x7d, 0x82, 0x33, 0x8d,
                0x4a, 0xd0, 0x95, 0xa3, 0x93, 0xa1, 0xc2, 0x4a, 0x78, 0x8d, 0x45, 0x24, 0x35, 0x94,
                0x4a, 0xcc, 0xe6, 0x38, 0x1e, 0xcc, 0x69, 0x37, 0xf2, 0xc3, 0x8c, 0x89, 0xa5, 0xf5,
                0x1a, 0xa6, 0x0c, 0xa6, 0x58, 0xfe, 0x71, 0x37, 0x1c, 0x2a, 0x83, 0xf4, 0x96, 0xca,
                0x2e, 0x62, 0x49, 0x79, 0x2e, 0x09, 0xeb, 0x79, 0xea, 0x3a, 0x13, 0x80, 0x32, 0x18,
                0xff, 0x20, 0x88, 0x9d, 0x8c, 0x59, 0xc8, 0x5e, 0x90, 0x99,
            ],
        },
        TestVector {
            incoming_viewing_key: [
                0xa4, 0xd7, 0x9c, 0x81, 0x9a, 0x6c, 0x5e, 0x01, 0x67, 0xfc, 0xa9, 0x8c, 0xe2, 0x62,
                0x98, 0x15, 0xf9, 0xba, 0xc9, 0x26, 0xb6, 0x27, 0x18, 0xcf, 0xbe, 0x50, 0x45, 0xd9,
                0x2d, 0xd7, 0x1c, 0xd3, 0x36, 0x75, 0xd5, 0x56, 0xe0, 0x77, 0x1e, 0x40, 0xcc, 0x3d,
                0x61, 0x8d, 0x9b, 0xda, 0x13, 0x2f, 0x13, 0x95, 0x3d, 0x82, 0x43, 0x2e, 0x81, 0x59,
                0x4a, 0x97, 0x1e, 0x98, 0xb0, 0x71, 0x40, 0x39,
            ],
            ovk: [
                0x67, 0x79, 0x9a, 0x90, 0x01, 0xa2, 0xed, 0x36, 0x76, 0xa8, 0xb4, 0x03, 0xae, 0x25,
                0xff, 0xd7, 0x72, 0xf7, 0x08, 0x1e, 0x9a, 0x32, 0xbc, 0xc1, 0xc5, 0xe2, 0xed, 0xd4,
                0xe2, 0xa6, 0x57, 0x6b,
            ],
            default_d: [
                0xdd, 0xb7, 0xc5, 0xbc, 0x4d, 0xe9, 0xdf, 0x52, 0x1b, 0xb0, 0x4b,
            ],
            default_pk_d: [
                0x65, 0x3d, 0x07, 0xc9, 0x07, 0x94, 0x6a, 0xc3, 0x02, 0x0e, 0xbd, 0xe1, 0xb4, 0xf6,
                0x10, 0x21, 0x0c, 0x30, 0xc4, 0x50, 0xe4, 0x27, 0x12, 0x65, 0xa0, 0x5d, 0x6e, 0xce,
                0x44, 0x6d, 0xf4, 0x39,
            ],
            v: 7122345086698755501,
            rseed: [
                0x2d, 0xd4, 0x17, 0xdf, 0x26, 0xdc, 0xd2, 0x20, 0xf2, 0xb7, 0x31, 0x77, 0x2b, 0x43,
                0x9e, 0x96, 0xd6, 0x14, 0xe1, 0xfa, 0xcb, 0x48, 0x6c, 0x7a, 0x7d, 0x51, 0x71, 0xb1,
                0xde, 0x35, 0x9f, 0x6a,
            ],
            memo: [
                0xff, 0xd3, 0xa9, 0x6f, 0x64, 0x9c, 0x96, 0x91, 0x02, 0xa1, 0x96, 0x4f, 0xb4, 0xb4,
                0xa1, 0xa4, 0x27, 0x9c, 0x68, 0xe6, 0xc3, 0x72, 0xe4, 0x21, 0x87, 0xd7, 0x54, 0xe8,
                0x04, 0xa6, 0x16, 0x53, 0x09, 0x20, 0x69, 0xfb, 0x9b, 0x6d, 0x25, 0x26, 0x68, 0x90,
                0x80, 0x8b, 0x01, 0x5d, 0xf2, 0x8c, 0x80, 0x10, 0x65, 0xda, 0x6f, 0xeb, 0xdc, 0x1a,
                0x56, 0xbf, 0xd0, 0x02, 0x62, 0x5a, 0xcf, 0xaa, 0x53, 0x73, 0xfd, 0xe1, 0x49, 0xc1,
                0xcf, 0xc3, 0x64, 0x9b, 0x48, 0x69, 0x69, 0x6d, 0x44, 0xec, 0xb1, 0x24, 0x79, 0xc5,
                0xeb, 0xef, 0x99, 0x5f, 0x10, 0x02, 0x9f, 0x8b, 0x53, 0x0e, 0xeb, 0x3f, 0xdc, 0x2e,
                0x50, 0xe8, 0x75, 0x7f, 0xc0, 0xbb, 0x9e, 0x26, 0x30, 0x23, 0xdb, 0x82, 0xf8, 0x78,
                0xd9, 0xac, 0x7f, 0xfb, 0x0b, 0xd4, 0x39, 0x1d, 0xf1, 0xd8, 0x79, 0x89, 0x9a, 0x3e,
                0xf5, 0x7b, 0xfd, 0x0d, 0x1f, 0x77, 0x55, 0x64, 0x8e, 0xdd, 0x85, 0xbb, 0x05, 0x2a,
                0x6e, 0xdf, 0x71, 0xcd, 0x26, 0x28, 0xc9, 0x87, 0x42, 0x9f, 0x36, 0xdc, 0x50, 0x5c,
                0xcc, 0x43, 0xf3, 0x0e, 0x7a, 0x86, 0x9c, 0x9e, 0x25, 0x5e, 0x2a, 0xf9, 0xfc, 0xf3,
                0x0c, 0x12, 0x17, 0x96, 0xd1, 0x90, 0x00, 0x09, 0x60, 0xcb, 0x6f, 0xe2, 0xf1, 0xbf,
                0x24, 0x61, 0x18, 0xb4, 0x98, 0xf3, 0x24, 0x7f, 0x9d, 0x48, 0x4c, 0x73, 0xcf, 0x09,
                0x39, 0x30, 0x39, 0xe4, 0x53, 0x26, 0xb8, 0xff, 0xff, 0xb3, 0xe7, 0xe6, 0x15, 0x9c,
                0x46, 0x69, 0x9f, 0x10, 0x07, 0x92, 0xd4, 0x67, 0x29, 0x50, 0x34, 0x8a, 0x90, 0x55,
                0x2e, 0x45, 0x94, 0x3b, 0xee, 0xac, 0xf0, 0x3f, 0x32, 0x16, 0xf9, 0x4e, 0x27, 0x4d,
                0x63, 0xd6, 0x37, 0xd9, 0xf1, 0x90, 0xe8, 0xa2, 0x66, 0xcd, 0xee, 0xf1, 0x53, 0x53,
                0x0b, 0xee, 0x5c, 0xb8, 0x35, 0x52, 0x60, 0x50, 0x5c, 0x2c, 0x2e, 0x5d, 0x99, 0x0f,
                0xff, 0xdc, 0x34, 0xec, 0x0f, 0xf7, 0xf1, 0xaf, 0x81, 0xb2, 0x4c, 0xed, 0x0e, 0xfa,
                0x62, 0x13, 0xda, 0x6c, 0x7c, 0x60, 0xc4, 0x87, 0xf5, 0xf7, 0xb0, 0x3f, 0x81, 0x60,
                0xa0, 0x57, 0xf4, 0x6d, 0x05, 0xbf, 0x82, 0x18, 0xb3, 0xad, 0xd9, 0xc0, 0x68, 0x93,
                0xbd, 0x02, 0xdb, 0x9b, 0x61, 0x19, 0x1d, 0xfb, 0x13, 0x3b, 0xfa, 0xbe, 0x48, 0x58,
                0xe4, 0x7a, 0x4c, 0xc3, 0x2e, 0x41, 0x6e, 0xc0, 0x8b, 0x8a, 0xc7, 0x91, 0x5a, 0x43,
                0x73, 0x3f, 0x44, 0x06, 0xe9, 0xd9, 0x67, 0xc5, 0x60, 0xf3, 0x44, 0xd7, 0xe9, 0x04,
                0xa2, 0x80, 0x45, 0xd9, 0x9f, 0x3a, 0xf8, 0xc8, 0x2e, 0x97, 0xe1, 0xb9, 0xc1, 0xb2,
                0x05, 0xe5, 0x85, 0xfb, 0xeb, 0xb4, 0x8f, 0xaf, 0x58, 0xf1, 0xb6, 0x5d, 0xca, 0x24,
                0x97, 0xe0, 0x9a, 0x70, 0xaa, 0xd4, 0x86, 0x5f, 0x85, 0x71, 0x5a, 0x28, 0x0e, 0x18,
                0x6f, 0x3f, 0xc1, 0x74, 0x0d, 0x81, 0x84, 0xd3, 0x3e, 0x83, 0x22, 0x16, 0x95, 0x21,
                0xcd, 0xc1, 0x32, 0x21, 0x29, 0x39, 0xc8, 0x4a, 0x10, 0x89, 0x64, 0xe2, 0xde, 0x74,
                0xb6, 0xea, 0x55, 0xb4, 0xcb, 0x8f, 0x6f, 0x9b, 0xee, 0x98, 0xb1, 0x0d, 0x41, 0x51,
                0x09, 0x45, 0x5f, 0x48, 0xb7, 0x76, 0x08, 0x2d, 0xc3, 0x0b, 0x4b, 0xc7, 0x34, 0x77,
                0x07, 0x55, 0x11, 0x70, 0x03, 0x08, 0x15, 0x8c, 0xe2, 0xf2, 0xf9, 0xbf, 0x0f, 0x69,
                0x1b, 0x2c, 0xe5, 0x3e, 0x61, 0x14, 0x2c, 0xb7, 0x40, 0xc1, 0x5b, 0x7b, 0x62, 0x3c,
                0xf4, 0x8b, 0x3f, 0x7b, 0xfe, 0xfa, 0x31, 0xbc, 0xdc, 0x66, 0x5c, 0x6d, 0x71, 0x23,
                0xe9, 0x53, 0x50, 0x81, 0x13, 0x75, 0x94, 0x7b, 0x05, 0x5a, 0x43, 0xdb, 0x07, 0xe0,
                0x3f, 0x33, 0x62, 0x7d, 0xf5, 0xc6, 0x38, 0xbf,
            ],
            cv_net: [
                0x00, 0x55, 0xf3, 0x5c, 0x6c, 0x82, 0x62, 0xac, 0x74, 0xfe, 0x27, 0xd7, 0x2a, 0x33,
                0xbd, 0xb9, 0x6f, 0x1c, 0xe0, 0x57, 0xc3, 0x30, 0xd1, 0xcc, 0xba, 0x2f, 0x7d, 0xa8,
                0x71, 0x55, 0x00, 0xb5,
            ],
            rho: [
                0x3b, 0x37, 0x96, 0x78, 0x0c, 0x0a, 0xec, 0x14, 0xed, 0x28, 0x74, 0xb5, 0x23, 0x06,
                0xe1, 0xc3, 0xd5, 0xde, 0x45, 0x93, 0xc6, 0x69, 0xaf, 0x1c, 0xaf, 0x11, 0xbc, 0xb4,
                0xd3, 0x5c, 0x60, 0x12,
            ],
            cmx: [
                0x4d, 0xa9, 0xdf, 0xdc, 0x70, 0x8c, 0xe8, 0xa0, 0x77, 0xa0, 0x6e, 0xc0, 0x67, 0x79,
                0x24, 0xcf, 0x37, 0x70, 0xed, 0xc2, 0x07, 0xfd, 0x5e, 0x7d, 0x69, 0x5f, 0x71, 0xb0,
                0x15, 0xbb, 0x03, 0x05,
            ],
            esk: [
                0xae, 0xec, 0x0d, 0xb9, 0x0e, 0x55, 0x83, 0x3c, 0x8b, 0xaf, 0x52, 0x6d, 0x66, 0x54,
                0xa1, 0x74, 0x5b, 0xa4, 0x4c, 0xad, 0x3c, 0xf9, 0xa6, 0x2c, 0xfb, 0xee, 0x5d, 0xe3,
                0x99, 0xca, 0x31, 0x26,
            ],
            ephemeral_key: [
                0x2d, 0xa0, 0x59, 0x4c, 0xd8, 0x74, 0x91, 0x46, 0x52, 0x67, 0xe7, 0x2c, 0x61, 0x89,
                0x07, 0x91, 0xfe, 0xb4, 0x25, 0xa2, 0xbb, 0xcd, 0xda, 0xcf, 0xe4, 0x5a, 0x66, 0x62,
                0x2f, 0x49, 0xef, 0x35,
            ],
            shared_secret: [
                0xe0, 0xb1, 0x52, 0x67, 0xfd, 0x21, 0x08, 0xeb, 0xbd, 0xd4, 0x16, 0x3f, 0x83, 0xad,
                0xef, 0xb6, 0x1c, 0x3e, 0xdf, 0x56, 0x6d, 0x94, 0x6f, 0xa1, 0xc1, 0x5e, 0x96, 0x46,
                0x43, 0xb1, 0x9c, 0x8e,
            ],
            k_enc: [
                0x28, 0x48, 0xae, 0x53, 0xaa, 0xce, 0xbe, 0x7c, 0xab, 0x58, 0x73, 0x1d, 0xc2, 0x21,
                0x88, 0x1b, 0x60, 0x2c, 0xc5, 0xa5, 0x3b, 0xcc, 0x1f, 0x76, 0xc8, 0x20, 0xb0, 0xea,
                0x13, 0x55, 0x68, 0x8e,
            ],
            p_enc: [
                0x02, 0xdd, 0xb7, 0xc5, 0xbc, 0x4d, 0xe9, 0xdf, 0x52, 0x1b, 0xb0, 0x4b, 0xad, 0x95,
                0x6d, 0xdc, 0x1e, 0xa7, 0xd7, 0x62, 0x2d, 0xd4, 0x17, 0xdf, 0x26, 0xdc, 0xd2, 0x20,
                0xf2, 0xb7, 0x31, 0x77, 0x2b, 0x43, 0x9e, 0x96, 0xd6, 0x14, 0xe1, 0xfa, 0xcb, 0x48,
                0x6c, 0x7a, 0x7d, 0x51, 0x71, 0xb1, 0xde, 0x35, 0x9f, 0x6a, 0xff, 0xd3, 0xa9, 0x6f,
                0x64, 0x9c, 0x96, 0x91, 0x02, 0xa1, 0x96, 0x4f, 0xb4, 0xb4, 0xa1, 0xa4, 0x27, 0x9c,
                0x68, 0xe6, 0xc3, 0x72, 0xe4, 0x21, 0x87, 0xd7, 0x54, 0xe8, 0x04, 0xa6, 0x16, 0x53,
                0x09, 0x20, 0x69, 0xfb, 0x9b, 0x6d, 0x25, 0x26, 0x68, 0x90, 0x80, 0x8b, 0x01, 0x5d,
                0xf2, 0x8c, 0x80, 0x10, 0x65, 0xda, 0x6f, 0xeb, 0xdc, 0x1a, 0x56, 0xbf, 0xd0, 0x02,
                0x62, 0x5a, 0xcf, 0xaa, 0x53, 0x73, 0xfd, 0xe1, 0x49, 0xc1, 0xcf, 0xc3, 0x64, 0x9b,
                0x48, 0x69, 0x69, 0x6d, 0x44, 0xec, 0xb1, 0x24, 0x79, 0xc5, 0xeb, 0xef, 0x99, 0x5f,
                0x10, 0x02, 0x9f, 0x8b, 0x53, 0x0e, 0xeb, 0x3f, 0xdc, 0x2e, 0x50, 0xe8, 0x75, 0x7f,
                0xc0, 0xbb, 0x9e, 0x26, 0x30, 0x23, 0xdb, 0x82, 0xf8, 0x78, 0xd9, 0xac, 0x7f, 0xfb,
                0x0b, 0xd4, 0x39, 0x1d, 0xf1, 0xd8, 0x79, 0x89, 0x9a, 0x3e, 0xf5, 0x7b, 0xfd, 0x0d,
                0x1f, 0x77, 0x55, 0x64, 0x8e, 0xdd, 0x85, 0xbb, 0x05, 0x2a, 0x6e, 0xdf, 0x71, 0xcd,
                0x26, 0x28, 0xc9, 0x87, 0x42, 0x9f, 0x36, 0xdc, 0x50, 0x5c, 0xcc, 0x43, 0xf3, 0x0e,
                0x7a, 0x86, 0x9c, 0x9e, 0x25, 0x5e, 0x2a, 0xf9, 0xfc, 0xf3, 0x0c, 0x12, 0x17, 0x96,
                0xd1, 0x90, 0x00, 0x09, 0x60, 0xcb, 0x6f, 0xe2, 0xf1, 0xbf, 0x24, 0x61, 0x18, 0xb4,
                0x98, 0xf3, 0x24, 0x7f, 0x9d, 0x48, 0x4c, 0x73, 0xcf, 0x09, 0x39, 0x30, 0x39, 0xe4,
                0x53, 0x26, 0xb8, 0xff, 0xff, 0xb3, 0xe7, 0xe6, 0x15, 0x9c, 0x46, 0x69, 0x9f, 0x10,
                0x07, 0x92, 0xd4, 0x67, 0x29, 0x50, 0x34, 0x8a, 0x90, 0x55, 0x2e, 0x45, 0x94, 0x3b,
                0xee, 0xac, 0xf0, 0x3f, 0x32, 0x16, 0xf9, 0x4e, 0x27, 0x4d, 0x63, 0xd6, 0x37, 0xd9,
                0xf1, 0x90, 0xe8, 0xa2, 0x66, 0xcd, 0xee, 0xf1, 0x53, 0x53, 0x0b, 0xee, 0x5c, 0xb8,
                0x35, 0x52, 0x60, 0x50, 0x5c, 0x2c, 0x2e, 0x5d, 0x99, 0x0f, 0xff, 0xdc, 0x34, 0xec,
                0x0f, 0xf7, 0xf1, 0xaf, 0x81, 0xb2, 0x4c, 0xed, 0x0e, 0xfa, 0x62, 0x13, 0xda, 0x6c,
                0x7c, 0x60, 0xc4, 0x87, 0xf5, 0xf7, 0xb0, 0x3f, 0x81, 0x60, 0xa0, 0x57, 0xf4, 0x6d,
                0x05, 0xbf, 0x82, 0x18, 0xb3, 0xad, 0xd9, 0xc0, 0x68, 0x93, 0xbd, 0x02, 0xdb, 0x9b,
                0x61, 0x19, 0x1d, 0xfb, 0x13, 0x3b, 0xfa, 0xbe, 0x48, 0x58, 0xe4, 0x7a, 0x4c, 0xc3,
                0x2e, 0x41, 0x6e, 0xc0, 0x8b, 0x8a, 0xc7, 0x91, 0x5a, 0x43, 0x73, 0x3f, 0x44, 0x06,
                0xe9, 0xd9, 0x67, 0xc5, 0x60, 0xf3, 0x44, 0xd7, 0xe9, 0x04, 0xa2, 0x80, 0x45, 0xd9,
                0x9f, 0x3a, 0xf8, 0xc8, 0x2e, 0x97, 0xe1, 0xb9, 0xc1, 0xb2, 0x05, 0xe5, 0x85, 0xfb,
                0xeb, 0xb4, 0x8f, 0xaf, 0x58, 0xf1, 0xb6, 0x5d, 0xca, 0x24, 0x97, 0xe0, 0x9a, 0x70,
                0xaa, 0xd4, 0x86, 0x5f, 0x85, 0x71, 0x5a, 0x28, 0x0e, 0x18, 0x6f, 0x3f, 0xc1, 0x74,
                0x0d, 0x81, 0x84, 0xd3, 0x3e, 0x83, 0x22, 0x16, 0x95, 0x21, 0xcd, 0xc1, 0x32, 0x21,
                0x29, 0x39, 0xc8, 0x4a, 0x10, 0x89, 0x64, 0xe2, 0xde, 0x74, 0xb6, 0xea, 0x55, 0xb4,
                0xcb, 0x8f, 0x6f, 0x9b, 0xee, 0x98, 0xb1, 0x0d, 0x41, 0x51, 0x09, 0x45, 0x5f, 0x48,
                0xb7, 0x76, 0x08, 0x2d, 0xc3, 0x0b, 0x4b, 0xc7, 0x34, 0x77, 0x07, 0x55, 0x11, 0x70,
                0x03, 0x08, 0x15, 0x8c, 0xe2, 0xf2, 0xf9, 0xbf, 0x0f, 0x69, 0x1b, 0x2c, 0xe5, 0x3e,
                0x61, 0x14, 0x2c, 0xb7, 0x40, 0xc1, 0x5b, 0x7b, 0x62, 0x3c, 0xf4, 0x8b, 0x3f, 0x7b,
                0xfe, 0xfa, 0x31, 0xbc, 0xdc, 0x66, 0x5c, 0x6d, 0x71, 0x23, 0xe9, 0x53, 0x50, 0x81,
                0x13, 0x75, 0x94, 0x7b, 0x05, 0x5a, 0x43, 0xdb, 0x07, 0xe0, 0x3f, 0x33, 0x62, 0x7d,
                0xf5, 0xc6, 0x38, 0xbf,
            ],
            c_enc: [
                0xf1, 0x42, 0xc4, 0xe5, 0x0e, 0xef, 0x2c, 0x64, 0x44, 0xcc, 0xd4, 0x0b, 0x8c, 0x99,
                0xe5, 0x6a, 0x72, 0xec, 0x4e, 0x30, 0xcf, 0x73, 0x68, 0x75, 0xf0, 0xaa, 0x6b, 0x8e,
                0x6d, 0x9c, 0xc3, 0x67, 0x73, 0x2d, 0xbc, 0x9a, 0xf0, 0xb3, 0x7d, 0x1f, 0xca, 0x6d,
                0x5b, 0xdd, 0x1d, 0xc1, 0x23, 0x79, 0x29, 0x66, 0x99, 0x9d, 0x62, 0xea, 0xf8, 0xc5,
                0xfe, 0x8d, 0x99, 0x91, 0x84, 0xf2, 0x8b, 0x99, 0xec, 0xef, 0xcc, 0x8f, 0x14, 0x8d,
                0xbd, 0x52, 0x02, 0x6f, 0xdb, 0x48, 0xdf, 0xda, 0x7b, 0xad, 0xb6, 0xd1, 0xfc, 0xd2,
                0x9a, 0xd2, 0x19, 0xea, 0xbf, 0xbb, 0x44, 0x3f, 0x5d, 0x0f, 0x98, 0xb6, 0x6a, 0x3a,
                0x25, 0x73, 0x1b, 0x52, 0xd7, 0xf2, 0xdf, 0x70, 0x01, 0x4b, 0x4a, 0xc6, 0x99, 0x34,
                0x32, 0xf8, 0x98, 0x1e, 0x9c, 0xbf, 0xe1, 0x69, 0x13, 0xf6, 0x8a, 0x93, 0x5d, 0x2d,
                0xd0, 0x06, 0xd4, 0x28, 0xf1, 0x45, 0x53, 0xe7, 0x29, 0x86, 0xc7, 0x0f, 0xb4, 0x43,
                0x18, 0xd2, 0x7c, 0x4d, 0x6f, 0x6f, 0xd3, 0x92, 0x3a, 0xb0, 0xf6, 0x28, 0x9a, 0x02,
                0x48, 0x5e, 0x87, 0x17, 0xe5, 0x7d, 0xa0, 0x24, 0xeb, 0xe1, 0x6e, 0x9c, 0xf8, 0x35,
                0xcf, 0x53, 0xd2, 0x19, 0x4f, 0xd4, 0x25, 0x50, 0x06, 0xb8, 0x1b, 0xfd, 0x51, 0xd1,
                0xef, 0x5a, 0xe9, 0xf9, 0xef, 0x6a, 0xf6, 0x57, 0x41, 0x81, 0xe2, 0xe2, 0x26, 0x50,
                0xcb, 0x91, 0x2a, 0x6b, 0x89, 0x88, 0xa0, 0x28, 0x86, 0x32, 0xbc, 0x73, 0x62, 0xbb,
                0xe1, 0x87, 0x23, 0xd8, 0x27, 0xf7, 0x94, 0x58, 0x62, 0x26, 0x0d, 0xf7, 0x8e, 0x95,
                0xd7, 0xd5, 0xe3, 0x31, 0x3f, 0x5a, 0xff, 0x72, 0xe2, 0x1c, 0xe2, 0xdf, 0x00, 0xee,
                0x7e, 0x81, 0x5a, 0xba, 0x17, 0xcc, 0xde, 0x15, 0xc2, 0x7e, 0xee, 0x08, 0x5f, 0x52,
                0xf3, 0x6c, 0x02, 0xec, 0xd6, 0x6c, 0xe1, 0x8b, 0x40, 0x15, 0xbe, 0xb6, 0x09, 0x23,
                0x3f, 0x6e, 0xb2, 0x8e, 0x4a, 0xd5, 0xcd, 0xbe, 0x6f, 0xdc, 0xab, 0x68, 0xbf, 0xbb,
                0x6f, 0xfd, 0x87, 0xd3, 0x86, 0xd8, 0x7e, 0xb1, 0xfe, 0x00, 0x34, 0x27, 0x0f, 0x41,
                0x27, 0x1e, 0xa0, 0x1f, 0x9e, 0xae, 0xa9, 0xe8, 0x9f, 0x78, 0x35, 0x9e, 0x41, 0x73,
                0x94, 0xbb, 0x9d, 0xf5, 0xb6, 0x1c, 0x36, 0xe3, 0x0b, 0xc1, 0xce, 0x4a, 0xb1, 0xbd,
                0xd7, 0x9f, 0xa4, 0x08, 0x3e, 0x82, 0x8d, 0xd1, 0x04, 0xe4, 0x73, 0x80, 0xcd, 0x83,
                0xcd, 0x65, 0x9d, 0xf4, 0x4d, 0xb9, 0x43, 0xdc, 0x07, 0xbc, 0xc8, 0x07, 0x05, 0x04,
                0xa1, 0xc6, 0x55, 0x23, 0x02, 0xe7, 0x4b, 0xe4, 0xb9, 0xc4, 0x32, 0x75, 0xec, 0xc2,
                0x88, 0xce, 0xda, 0x41, 0x59, 0xa9, 0xcc, 0x55, 0x7c, 0x18, 0x19, 0x5c, 0xec, 0x92,
                0x62, 0x24, 0xd8, 0xd6, 0x9e, 0x98, 0xe1, 0x83, 0x5a, 0x2e, 0x29, 0x05, 0x63, 0xef,
                0x20, 0xd0, 0x83, 0xd1, 0x4e, 0x93, 0xcc, 0x1f, 0x3f, 0x76, 0x3e, 0xf5, 0x58, 0x0e,
                0x13, 0x5f, 0xae, 0x1b, 0xb8, 0x54, 0x4a, 0x0c, 0x5c, 0x6d, 0x88, 0x17, 0x41, 0xe4,
                0x51, 0x34, 0x47, 0xac, 0xeb, 0x09, 0x33, 0xe6, 0xeb, 0xaf, 0x0c, 0xe3, 0x13, 0xc1,
                0x8c, 0x9a, 0xf9, 0x5b, 0xa5, 0x61, 0x31, 0xf7, 0x8f, 0x42, 0x72, 0x41, 0x22, 0x65,
                0xbc, 0xf4, 0xc5, 0xf6, 0x80, 0x89, 0x3c, 0xcd, 0xa5, 0x73, 0x7d, 0xa8, 0x23, 0xb7,
                0x63, 0x6e, 0x98, 0xdb, 0xa5, 0x62, 0x44, 0xf2, 0xb9, 0x6a, 0x10, 0x90, 0xa6, 0x60,
                0x38, 0x15, 0xc0, 0xef, 0x54, 0x97, 0x50, 0xf2, 0x47, 0x06, 0x19, 0x0b, 0x55, 0x76,
                0x6e, 0x8a, 0x62, 0x09, 0xa1, 0xc2, 0x2f, 0x67, 0xe8, 0x77, 0x62, 0x66, 0xb6, 0xfa,
                0xe4, 0x5b, 0xf7, 0x94, 0x90, 0x7f, 0x64, 0x71, 0x4f, 0xbe, 0x26, 0xc3, 0x0a, 0xc4,
                0x04, 0x11, 0xf5, 0xe6, 0x4f, 0xc1, 0x66, 0xc8, 0x4f, 0x28, 0xb8, 0x23, 0xfd, 0xaa,
                0x68, 0x32, 0xa3, 0x25, 0x63, 0x31, 0x7d, 0x25, 0x4c, 0x53, 0x16, 0x9b, 0x9f, 0xfb,
                0x24, 0x53, 0xa1, 0x12, 0x2f, 0xa4, 0x4c, 0x7f, 0x17, 0xc1, 0x36, 0xb5, 0x7e, 0x20,
                0xad, 0x17, 0x7a, 0x7e, 0xee, 0xbf, 0x9f, 0x56, 0xfb, 0x0b, 0x55, 0xcc, 0xcf, 0x68,
                0x2b, 0x8a, 0x5b, 0xd8, 0xa7, 0x45,
            ],
            ock: [
                0x06, 0x3c, 0x83, 0xa4, 0x95, 0x74, 0xe7, 0x80, 0x35, 0x89, 0xcc, 0x3d, 0x34, 0xb4,
                0x38, 0x90, 0xf3, 0xd7, 0x63, 0x87, 0x35, 0xe7, 0xbd, 0x5e, 0xbd, 0xd1, 0xa5, 0xea,
                0xb9, 0xd9, 0xc5, 0xd6,
            ],
            op: [
                0x65, 0x3d, 0x07, 0xc9, 0x07, 0x94, 0x6a, 0xc3, 0x02, 0x0e, 0xbd, 0xe1, 0xb4, 0xf6,
                0x10, 0x21, 0x0c, 0x30, 0xc4, 0x50, 0xe4, 0x27, 0x12, 0x65, 0xa0, 0x5d, 0x6e, 0xce,
                0x44, 0x6d, 0xf4, 0x39, 0xae, 0xec, 0x0d, 0xb9, 0x0e, 0x55, 0x83, 0x3c, 0x8b, 0xaf,
                0x52, 0x6d, 0x66, 0x54, 0xa1, 0x74, 0x5b, 0xa4, 0x4c, 0xad, 0x3c, 0xf9, 0xa6, 0x2c,
                0xfb, 0xee, 0x5d, 0xe3, 0x99, 0xca, 0x31, 0x26,
            ],
            c_out: [
                0xca, 0xb7, 0x8d, 0xd1, 0x83, 0xf3, 0xe1, 0x6a, 0xf7, 0x6e, 0x9a, 0x90, 0x3d, 0xd4,
                0x2a, 0x96, 0x67, 0xbf, 0xe7, 0x08, 0xce, 0x88, 0x79, 0xb8, 0x38, 0x6e, 0x62, 0xac,
                0x9f, 0x66, 0x74, 0xf4, 0x93, 0x59, 0x5d, 0xa5, 0x06, 0xe9, 0xef, 0x06, 0xbb, 0xa2,
                0x4a, 0x93, 0x80, 0x60, 0xe5, 0xd3, 0x82, 0xcc, 0x75, 0xdf, 0xab, 0x97, 0xe0, 0xf8,
                0x49, 0x4b, 0x47, 0x6a, 0xdf, 0x4f, 0xfd, 0x96, 0xff, 0x7f, 0x1b, 0x4f, 0x16, 0xf8,
                0x59, 0x4b, 0x7c, 0x5a, 0x21, 0x9c, 0x7a, 0x00, 0xad, 0x15,
            ],
        },
    ]
}
