/*
 * Season7 -- Videocrypt smard card emulator for DOS.
 *
 * (C) 1996 Markus Kuhn
 *
 * Compiles e.g. with Borland C++ 2.0, compact model (SYS_DOS)
 * or on Linux (SYS_LINUX).
 *
 * Warning: This code hasn't been cleaned up for a long time and
 * because of long experimental usage, it has become an excellent example
 * of bad software engineering.
 *
 */

#define VERSION_MAJOR 3                        /* last change: 1996-05-05 */
#define VERSION_MINOR 2

#undef   SYS_LINUX           /* define when compiling for a Linux system */
#define  SYS_DOS             /* define when compiling a 16-bit DOS version */

#define ASCII_LOG      /* define if you want an ASCII column in debug log */
#undef  TEST_DECODE
#undef  SECRETS            /* define to compile with secret hash function */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <limits.h>

#ifdef SYS_DOS
#include <conio.h>              /* for kbhit() & clrscr() */
#include <dos.h>                /* for int86() */
#include <alloc.h>              /* for farmalloc(), etc. */
#define malloc  farmalloc
#define realloc farrealloc
#define free    farfree
#define memcpy  _fmemcpy
#define memcmp  _fmemcmp
#define memset  _fmemset
#endif

#ifdef SYS_LINUX
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#define huge
/* Linux POSIX real-time extensions */
#if (_POSIX_VERSION >= 199309L)
#include <sched.h>
#include <sys/mman.h>
#else
/* preliminary support for Linux real-time functions */
#include <linux/unistd.h>     /* for _syscallX macros/related stuff */
/* quick 'n dirty replacement of <sys/mman.h> */
#if !defined(_POSIX_MEMLOCK) && defined(__NR_mlockall)
#define MCL_CURRENT    1               /* lock all current mappings */
#define MCL_FUTURE     2               /* lock all future mappings */
_syscall1(int, mlockall, int, flags)
_syscall0(int, munlockall)
int mlockall(int flags);
int munlockall(void);
#define _POSIX_MEMLOCK
#endif
#if !defined(_POSIX_MEMLOCK_RANGE) && defined(__NR_mlock)
_syscall2(int, mlock, const void *, addr, size_t, len)
_syscall2(int, munlock, const void *, addr, size_t, len)
int mlock(const void *addr, size_t len);
int munlock(const void *addr, size_t len);
#define _POSIX_MEMLOCK_RANGE
#endif
/* quick 'n dirty replacement of <sched.h> */
#if !defined(_POSIX_PRIORITY_SCHEDULING) && defined(__NR_sched_setscheduler)
struct sched_param {
  int sched_priority;
};
#define SCHED_OTHER    0
#define SCHED_FIFO     1
#define SCHED_RR       2
_syscall2(int, sched_setparam, pid_t, pid, const struct sched_param *, param)
_syscall2(int, sched_getparam, pid_t, pid, struct sched_param *, param)
_syscall3(int, sched_setscheduler, pid_t, pid, int, policy,
          const struct sched_param *, param)
_syscall1(int, sched_getscheduler, pid_t, pid)
_syscall0(int, sched_yield)
_syscall1(int, sched_get_priority_max, int, policy)
_syscall1(int, sched_get_priority_min, int, policy)
_syscall2(int, sched_rr_get_interval, pid_t, pid, struct timespec *, interval)
int sched_setparam(pid_t pid, const struct sched_param *param);
int sched_getparam(pid_t pid, struct sched_param *param);
int sched_setscheduler(pid_t pid, int policy, const struct sched_param *param);
int sched_getscheduler(pid_t pid);
int sched_yield(void);
int sched_get_priority_max(int policy);
int sched_get_priority_min(int policy);
int sched_rr_get_interval(pid_t pid, struct timespec *interval);
#define _POSIX_PRIORITY_SCHEDULING
#endif
/* quick 'n dirty replacement for <time.h> extensions */
#if 1
struct timespec {
  time_t  tv_sec;         /* seconds */
  long    tv_nsec;        /* nanoseconds */
};
#endif
#ifdef __NR_nanosleep
_syscall2(int, nanosleep, const struct timespec *, rqtp,
          struct timespec *, rmtp)
int nanosleep(const struct timespec *rqtp, struct timespec *rmtp);
#endif
#endif /* (_POSIX_VERSION >= 199309L) */
#endif /* SYS_LINUX */

#ifdef SECRETS
#include "decrypt.h"
#endif

#ifdef SYS_DOS
#include "async.h"
#endif

/*
 * information in answer to reset: (as defined in ISO 7816-3)
 * inverse convention (3f), protocol type T=0, F=372, D=1, I=50mA, P=5V, N=5.
 * historic characters (final 10 bytes): ???
 */
int reset_answer_len = 16;
unsigned char reset_answer[33] = {
  0x3f, 0xfa, 0x11, 0x25, 0x05, 0x00, 0x01, 0xb0,
  0x02, 0x3b, 0x36, 0x4d, 0x59, 0x02, 0x81, 0x80
};

/* an arbitrary card number */
unsigned char serial[6] = {
  0x27, 0x00, 0xa1, 0x2e, 0x4a, 0x01
};

/* a suitable Fiat-Shamir answer for the above card number */
unsigned char answer_82[64] = {
  0x51, 0xa0, 0x43, 0xd6, 0x9f, 0xee, 0x4f, 0xf3,
  0xa4, 0x04, 0xaa, 0x38, 0xae, 0x0f, 0x66, 0x1a,
  0xe9, 0x18, 0x9e, 0x00, 0x3c, 0xad, 0x47, 0xcb,
  0x68, 0x20, 0x33, 0x61, 0x90, 0xa0, 0xba, 0xc3,
  0x22, 0x08, 0xf2, 0xc5, 0x9a, 0xed, 0xe2, 0x8f,
  0x53, 0x5a, 0x10, 0x7f, 0xe9, 0x85, 0x60, 0x68,
  0x19, 0x00, 0x87, 0x22, 0x12, 0x9d, 0xff, 0x14,
  0xaf, 0xca, 0x3a, 0x7b, 0xbc, 0x56, 0xf1, 0x00
};

/* we show this shortly after a reset on the screen */
unsigned char display[25] ="\x80"
  "SEASON7 V . "
  "   --:--    ";

/* BSkyB channel id (based on byte 7 in 74 messages) for TV screen display */
char *channel_id[16] = {
  "    ?00?    ", " SKY MOVIES ", "MOVIECHANNEL", "MOVIES  GOLD",
  "    ?04?    ", "    ?05?    ", " SKY SPORTS ", "    ?07?    ",
  "  TV ASIA   ", "    TVX     ", "    ?0A?    ", "    ?0B?    ",
  "MULTICHANNEL", "    ?0D?    ", "    ?0E?    ", "    ?0F?    "
};

/* BSkyB channel name for text screen display */
char *channel_name[16] = {
  "???", "Sky Movies", "The Moviechannel", "Sky Movies Gold",
  "???", "???", "Sky Sports", "???",
  "TV Asia", "TVX", "???", "???",
  "Sky Multichannels", "???", "???", "???"
};

/* Command bytes sent in BSkyB 09 74h messages in msg[3] to card */
struct cmd_s{
  int code;
  char *text;
} command[] = {
  0x00, "Deactivate Card",
  0x01, "Disable Sky Movies",
  0x02, "Disable The Moviechannel",
  0x03, "Disable Sky Movies Gold",
  0x06, "Disable Sky Sports",
  0x08, "Disable TV Asia",
  0x09, "Disable TVX",
  0x0c, "Disable Sky Multichannels",
  0x20, "Activate Card",
  0x21, "Enable Sky Movies",
  0x22, "Enable The Moviechannel",
  0x23, "Enable Sky Movies Gold",
  0x26, "Enable Sky Sports",
  0x28, "Enable TV Asia",
  0x29, "Enable TVX",
  0x2c, "Enable Sky Multichannels",
  -1,   "???"
};

/* some global variables */
int debug = 0;
int economy = 0;

/* message directions */
#define IN   0   /* to card */
#define OUT  1   /* to decoder */

/* ISO 7816-3 T=0 response codes */
#define EVERYTHING_OK              ((unsigned char * ) "\x90\x00")
#define UNSUPPORTED_CLASS          ((unsigned char * ) "\x6e\x00")
#define UNSUPPORTED_INSTRUCTION    ((unsigned char * ) "\x6d\x00")
#define INCORRECT_REFERENCE        ((unsigned char * ) "\x6b\x00")
#define INCORRECT_LENGTH           ((unsigned char * ) "\x67\x00")

/*
 * These are some delay values (microseconds) used in the program.
 * Use command line option w to modify them.
 */
#define DELAYS 5
unsigned long delus[DELAYS] = {
  30000,      /* wait between reset and answer to reset */
  500,        /* wait after each outgoing byte */
  0,          /* wait before sending procedure byte */
  0,          /* wait before sending first data byte */
  0           /* wait before sending SW1 */
};


/*
 * Reverse and invert all bits in each byte of a string.
 * E.g. 10010111b (0x97) becomes 00010110b (0x16).
 */
void inverse(unsigned char *data, int len) {
  int i, j;
  unsigned char c;

  for (i = 0; i < len; i++) {
    c = 0;
    for (j = 0; j < 8; j++)
      c |= ((data[i] & (1 << j)) != 0) << (7 - j);
    data[i] = ~c;
  }

  return;
}


/*
 * wait for the specified number of microseconds
 */
void wait_us(unsigned long usec)
{
#ifdef SYS_DOS

  if (economy) {

#define CNT_FREQ 1193180L      /* IBM PC timer clock frequency in Hz */
#define CNT_CNT0 0x40
#define CNT_MODE 0x43
#define CNT_US_CYCLE 54925L    /* time between counter overflows in s
                                * = (1000000 / CNT_FREQ) * 65536       */
#define CNT_READ() (outp(CNT_MODE, 0x00),          \
                    counter = inp(CNT_CNT0),       \
                    counter |= inp(CNT_CNT0) << 8, \
                    counter = start - counter)

    unsigned long cycles, jiffies;
    unsigned short start = 0, counter, old_counter = 0;

    cycles = usec / CNT_US_CYCLE;
    jiffies = usec % CNT_US_CYCLE;
    jiffies *= (CNT_FREQ + 50) / 100;
    jiffies /= 1000000L / 100;

    outp(CNT_MODE, 0x00);      /* latch counter */
    start = inp(CNT_CNT0);
    start |= inp(CNT_CNT0) << 8;
    while (cycles) {
      while ((CNT_READ() & 0x8000) == 0);
      while ((CNT_READ() & 0x8000) != 0);
      cycles--;
    }
    while (CNT_READ() < jiffies && counter >= old_counter)
      old_counter = counter;

  } else {

    /*
     * Uses a BIOS function in order to wait a specified number of
     * microseconds. This BIOS functions is only implemented in
     * IBM AT compatible BIOSes (i.e., not in old PC and XT systems)!
     * The timing resolution of this function isn't really down
     * to 1 s on most systems, but it's fine for our purposes.
     */
    union REGS regs;

    if (usec == 0) return;
    regs.h.ah = 0x86;
    regs.x.cx = (int) (usec >> 16);
    regs.x.dx = (int) (usec & 0xffff);
    int86(0x15, &regs, &regs);

  }
#endif

#if defined(SYS_LINUX) && defined(__NR_nanosleep)
  struct timespec t;

  t.tv_sec = usec / 1000000L;
  t.tv_nsec = (usec % 1000000L) * 1000;
  nanosleep(&t, NULL);
#endif

  return;
}


#ifdef SYS_DOS

void activate_com(int com)
{
  int port;

  switch (com) {
  case 1: port = COM1; break;
  case 2: port = COM2; break;
  case 3: port = COM3; break;
  case 4: port = COM4; break;
  default:
    fprintf(stderr, "Port COM%d not available!\n", com);
    exit(1);
  }
  if (AsyncInit(port)) {
    fprintf(stderr, "Can't initialize port COM%d!\n", com);
    exit(1);
  }
  AsyncHand(DTR | RTS);
  AsyncSet(9600, BITS_8 | STOP_2 | ODD_PARITY);
  AsyncStat();   /* clear D_DCD */

  return;
}


void deactivate_com(void)
{
  AsyncStop();
}

void prepare_console(void)
{
#ifdef _NOCURSOR
  _setcursortype(_NOCURSOR);
#endif

  return;
}

void restore_console(void)
{
#ifdef _NORMALCURSOR
  _setcursortype(_NORMALCURSOR);
#endif

  return;
}

#endif /* SYS_DOS */

#ifdef SYS_LINUX

int serial_port;    /* file descriptor of serial port */

void activate_com(int com)
{
  char *fn = NULL;
  struct termios t;

  switch (com) {
  case 1: fn = "/dev/cua0"; break;
  case 2: fn = "/dev/cua1"; break;
  case 3: fn = "/dev/cua2"; break;
  case 4: fn = "/dev/cua3"; break;
  default:
    fprintf(stderr, "Port COM%d not available!\n", com);
    exit(1);
  }
  serial_port = open(fn, O_RDWR | O_NONBLOCK);
  if (serial_port < 0) {
    fprintf(stderr, "Can't open '%s", fn);
    perror("'");
    exit(1);
  }
  if (tcgetattr(serial_port, &t)) {
    perror("tcgetattr() failed");
    exit(1);
  }
  t.c_iflag &= ~(BRKINT | IGNBRK | INPCK | ISTRIP | INLCR | IGNCR | ICRNL |
                IXON | IXOFF);
  t.c_iflag |= IGNBRK;
  t.c_oflag &= ~OPOST;
  t.c_cflag &= ~CSIZE;
  t.c_cflag |= CLOCAL | CREAD | CS8 | CSTOPB | PARENB | PARODD;
  t.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL | ICANON | ISIG | IEXTEN);
  t.c_lflag |= NOFLSH;
  t.c_cc[VMIN] = 0;
  t.c_cc[VTIME] = 1;
  if (cfsetispeed(&t, B9600)) {
    perror("cfsetispeed() failed");
    exit(1);
  }
  if (cfsetospeed(&t, B9600)) {
    perror("cfsetospeed() failed");
    exit(1);
  }
  if (tcsetattr(serial_port, TCSANOW, &t)) {
    perror("tcsetattr() failed");
    exit(1);
  }
  if (ioctl(serial_port, TIOCMBIS, TIOCM_DTR | TIOCM_RTS)) {
    perror("ioctl() failed");
    exit(1);
  }

  return;
}


void deactivate_com(void)
{
  if (close(serial_port)) {
    perror("close(serial_port) failed");
    exit(1);
  }
}

int keypressed = 0;
char keybuf;
struct termios old_termios;

void prepare_console(void)
{
  struct termios t;

  if (tcgetattr(STDIN_FILENO, &t)) {
    perror("tcgetattr(STDIN_FILENO) failed");
    exit(1);
  }
  old_termios = t;
  t.c_lflag &= ~(ECHO | ICANON);
  t.c_cc[VMIN] = 0;
  t.c_cc[VTIME] = 0;
  if (tcsetattr(STDIN_FILENO, TCSANOW, &t)) {
    perror("tcsetattr() failed");
    exit(1);
  }

  return;
}

void restore_console(void)
{
  if (tcsetattr(STDIN_FILENO, TCSANOW, &old_termios)) {
    perror("tcsetattr() failed");
    exit(1);
  }

  return;
}

int kbhit(void)
{
  int r;

  if (!keypressed) {
    r = read(STDIN_FILENO, &keybuf, 1);
    if (r > 0) keypressed = 1;
  }

  return keypressed;
}

int getch(void)
{
  if (keypressed) {
    keypressed = 0;
    return keybuf;
  }

  return 0;
}

#endif /* SYS_LINUX */


/*
 * test for a reset signal via DCD
 */
int test_reset(void) {
#ifdef SYS_DOS
  return (AsyncStat() & DCD);
#endif
#ifdef SYS_LINUX
  unsigned int status;

  if (ioctl(serial_port, TIOCMGET, &status) < 0) {
    perror("ioctl(serial_port, TIOCMGET) failed");
    exit(1);
  }
  return (status & TIOCM_CAR);
#endif
}


void send_byte(unsigned char data)
{
#ifdef SYS_DOS
    AsyncOut(data);
#endif
#ifdef SYS_LINUX
    int result;

    do {
      result = write(serial_port, &data, 1);
      if (result < 0 && errno != EAGAIN) {
	perror("write(serial_port) failed");
	exit(1);
      }
    } while (result < 0 &&  errno == EAGAIN);
#endif

    return;
}


int receive_byte(void)
{
#ifdef SYS_DOS
  return AsyncIn();
#endif
#ifdef SYS_LINUX
  char c;
  int result;

  /* another ugly busy wait */
  do {
    result = read(serial_port, &c, 1);
    if (result > 0) break;
    if (result < 0 && errno != EAGAIN) {
      perror("read(serial_port) failed");
      exit(1);
    }
    if (test_reset()) return -1;
  } while (1);
  
  return c;
#endif
}


void purge_fifos(void)
{
#ifdef SYS_DOS
  AsyncClear();
#endif
#ifdef SYS_LINUX
  if (tcflush(serial_port, TCIOFLUSH) < 0) {
    perror("tcflush(serial_port, TCIOFLUSH) failed");
    exit(1);
  }
#endif
  
  return;
}



/*
 * Send a string of bytes to serial port and wait for each single echo.
 * (More efficient send methods are too fast for decoder, it seems to
 * need more than 2 stop bits, so we wait between individual bytes.)
 */
int send(unsigned char *data, int len)
{
  int i;
  unsigned char c, r;

  purge_fifos();
  for (i = 0; i < len; i++) {
    c = data[i];
    inverse(&c, 1);
    send_byte(c);
#ifdef SYS_DOS
    while (AsyncInStat() == 0)
      if ((AsyncStat() & DCD) || kbhit()) return 1;
#endif
    r = receive_byte();
    if (r != c)
      fprintf(stderr, "Warning: wrong signal echo (sent %02x, rec %02x)!\n",
	      c, r);
    wait_us(delus[1]);
  }

  return 0;
}


/*
 * Get all bytes available from serial port FIFO and return how many
 * bytes were available.
 */
int receive(unsigned char *data, int max)
{
  int i = 0;

#ifdef SYS_DOS
  while (AsyncInStat() > 0 && i < max)
    data[i++] = AsyncIn();
#endif

#ifdef SYS_LINUX
  i = read(serial_port, data, max);
  if (i < 0 && errno == EAGAIN) i = 0;
#endif

  inverse(data, i);

  return i;
}


/*
 * Get n bytes from serial port FIFO. Don't wait more than timeout
 * microseconds and return how many bytes were available.
 */
int receive_timeout(unsigned char *data, int n, long timeout)
{
  int i = 0;

  while (i < n && timeout > 0) {
    i += receive(data + i, n - i);
    if (i < n) {
      wait_us(15000);
      timeout -= 15000;
    }
  }
  inverse(data, i);

  return i;
}


/*
 * In debugging mode, print the contents of all packets.
 */
void log_packet(const unsigned char *header, const unsigned char *data,
                int direction)
{
  int i, j;
  static time_t last_time = 0;
  time_t now;
  char tmp[40];
#ifdef SECRETS
  unsigned char xmsg[32];
#endif

  /* print time from time to time */
  time(&now);
  if (difftime(now, last_time) >= 10) {
    strftime(tmp, 40, "%Y-%m-%d %H:%M:%SZ", gmtime(&now));
    printf("%s\n", tmp);
    last_time = now;
  }

  if (debug != 3 && (debug == 2 || header[1] == 0x74 || header[1] == 0x78)) {
    if (header[1] == 0x60)    /* illegal command code 0x60: reset */
      printf("RESET:   ");
    else
      printf("%s %02x: ", (direction == IN) ? "dcdr" : "card", header[1]);
    for (i = 0; i < header[4]; i += 16) {
      if (i > 0) printf("         ");
      for (j = 0; j < 16; j++)
        if (i + j < header[4])
          printf("%02x ", data[i + j]);
#ifdef ASCII_LOG
        else
          printf("   ");
      printf("  ");
      for (j = 0; j < 16 && i + j < header[4]; j++)
        putchar((data[i + j] > 31 && data[i + j] < 127) ? data[i + j] : '.');
#endif
      putchar('\n');
    }
  }
#ifdef SECRETS
  if (header[1] == 0x74) {
    memcpy(xmsg, data, 32);
    de_xor(xmsg);
    if (xmsg[3] == 0x80) {
      /* output the BSkyB 09 series nanocommands */
      if (debug != 3) printf("         tokens:  ");
      i = 0;
      for (j = 12; j < 27; j++) {
        printf("%02x ", xmsg[j]);
        if (i == 0) {
          /* here starts new nano command */
          if (xmsg[j] == 0x03 || xmsg[j] == 0x46) break;
          switch (xmsg[j]) {
          case 0x09: i = 3; break;
          case 0x0c: i = 2; break;
          case 0x11: i = 4; break;
          case 0x14: i = 5; break;
          case 0x19: i = 1; break;
          case 0x1d: i = 2; break;
          case 0x24: i = 3; break;
          case 0x28: i = 5; break;
          case 0x2c: i = 1; break;
          case 0x30: i = 2; break;
          case 0x34: i = 4; break;
          case 0x39: i = 2; break;
          case 0x3c: i = 1; break;
          case 0x41: i = 1; break;
          case 0x46: i = 1; break;
          case 0x49: i = 1; break;
          default:   i = 1;
          }
        }
        i--;
      }
      putchar('\n');
    }
  }
#endif

  return;
}



/*
 * Parse an answer to reset packet.
 *
 * Return value:  >0  correct answer to reset received, length returned
 *                -1  no correct answer to reset received
 */
int parse_reset(void)
{
  int y, p, i, tck_expected = 0;

  reset_answer_len = 0;

  /* Skip reset impulse noise characters and read TS */
  reset_answer[0] = 0;
  for (i = 0; i < 5 && reset_answer[0] != 0x3f; i++)
    if (receive_timeout(reset_answer, 1, 500000L) == 0) return -1;
  if (reset_answer[0] != 0x3f) return -1;

  /* Read T0 */
  if (receive_timeout(reset_answer + 1, 1, 500000L) == 0) return -1;

  /* Read TA_i, TB_i, TC_i, TD_i */
  p = 2;
  y = reset_answer[1] >> 4;
  while (y && p < 33 - 5 - (reset_answer[1] & 15)) {
    for (i = 0; i < 4; i++) {
      if ((y >> i) & 1) {
        if (receive_timeout(reset_answer + p, 1, 500000L) != 1) return -1;
        if (i == 3) {
          tck_expected |= (reset_answer[p] & 15) != 0;
          y = reset_answer[p] >> 4;
        }
        p++;
      } else
        if (i == 3) y = 0;
    }
  }  /* while () */

  /* historic characters */
  if (receive_timeout(reset_answer + p, reset_answer[1] & 15, 500000L) !=
      (reset_answer[1] & 15))
    return -1;
  p += (reset_answer[1] & 15);

  /* TCK */
  if (tck_expected) {
    if (receive_timeout(reset_answer + p, 1, 500000L) != 1) return -1;
    p++;
    y = 0;
    for (i = 1; i < p; i++)
      y ^= reset_answer[i];
    if (y != 0) return -1;
  }

  /* everything was fine */
  reset_answer_len = p;

  return p;
}


/*
 * Calculate week number and day of week according to ISO 8601.
 * Implementation derived from code provided by L. Kirby.
 */
void get_week_number(const struct tm *t, int *week, int *dayofweek)
{
  int tmp1, tmp2, fourthdaynum, year, isleap;

  tmp1 = 1 - t->tm_wday;
  tmp2 = t->tm_yday + ((tmp1 > 0) ? 3 : 10) + tmp1;
  fourthdaynum = tmp2 % 7;
    
  *week = tmp2 / 7;

  if (*week == 0) {
    year = t->tm_year + 1900 - 1;
    isleap = !(year % 4) && ((year % 100) || !(year % 400));
    *week = (fourthdaynum + isleap >= 6) ? 53 : 52;
  } else if (*week == 53) {
    year = t->tm_year + 1900;
    isleap = !(year % 4) && ((year % 100) || !(year % 400));
    if (fourthdaynum > isleap)
      *week = 1;
  }

  if (t->tm_wday == 0)
    *dayofweek = 7;
  else
    *dayofweek = t->tm_wday;

  return;
}


/*
 * Convert a string like hh:mm into a time value
 */
time_t parse_time(char *s)
{
  time_t now;
  struct tm t;
  int error = 0;

  time(&now);
  memcpy(&t, localtime(&now), sizeof(struct tm));

  if (isdigit(s[0]) && isdigit(s[1]) && !s[2]) {
    /* hh */
    t.tm_hour = (s[0]-'0')*10 + s[1]-'0';
    t.tm_min = 0;
    t.tm_sec = 0;
  } else if (isdigit(s[0]) && isdigit(s[1]) && s[2] == ':' &&
             isdigit(s[3]) && isdigit(s[4])) {
    /* hh:mm */
    t.tm_hour = (s[0]-'0')*10 + s[1]-'0';
    t.tm_min =  (s[3]-'0')*10 + s[4]-'0';
    if (s[5] == ':' && isdigit(s[6]) && isdigit(s[7]) && !s[8])
      /* hh:mm:ss */
      t.tm_sec =  (s[6]-'0')*10 + s[7]-'0';
    else if (!s[5])
      t.tm_sec = 0;
    else
      error = 1;
  } else if (isdigit(s[0]) && isdigit(s[1]) &&
             isdigit(s[2]) && isdigit(s[3])) {
    /* hhmm */
    t.tm_hour = (s[0]-'0')*10 + s[1]-'0';
    t.tm_min =  (s[2]-'0')*10 + s[3]-'0';
    if (isdigit(s[4]) && isdigit(s[5]) && !s[6])
      /* hhmmss */
      t.tm_sec =  (s[4]-'0')*10 + s[5]-'0';
    else if (!s[4])
      t.tm_sec = 0;
    else
      error = 1;
  }

  if (error) {
    fprintf(stderr, "Unknown time string format '%s'!\n"
            "Please write 'hh', 'hh:mm', or 'hh:mm:ss'.\n", s);
    exit(1);
  }

  return mktime(&t);
}


/*
 * Help message
 */
void usage(void)
{
  fprintf(stderr, "Command line options:\n\n");
  fprintf(stderr,
	  "  -<number>\tselect COM port specified by number\n"
	  "  -m\t\tsuppress Season7 messages on TV screen\n"
	  "  -e\t\teconomy mode, suppress unnecessary activity to avoid\n"
	  "\t\ttiming problems on slow machines\n"
          "  -s\t\tsuppress screen saver (do not clear screen after 5 min)\n"
	  "  -o\t\tswitch to passive mode and record a VCL file with an\n"
	  "\t\tautomatically selected filename\n"
	  "  -O <filename>\tlike -o, but with a specified VCL filename\n"
          "  -b\t\trecord VBL file (also -B <filename>)\n"
	  "  -n <string>\tname of recorded show for VCL header information\n"
	  "  -r <string>\tname of transponder for VCL header information\n"
	  "  -i <filename>\tload a VCL file and use it to answer the decoder "
	  "requests\n"
          "  -c <hhmm>\tdelay start until specified time\n"
          "  -l <number>\tterminate after specified number of minutes\n"
	  "  -v\t\tdisplay VCL file loaded with -i as hexdump and exit\n"
	  "  -p\t\tenter passive mode and listen only to data traffic\n"
	  "  -d\t\tdebug level 1: show only instructions 0x74/0x78\n"
	  "  -D\t\tdebug level 2: show all ISO 7816 instructions\n"
	  "  -t\t\tdebug level 3: show only nanocommands (only for BSkyB 09)\n"
          "  -wa <number>\tmodify delay between reset and answer-to-reset "
	  "[s]\n"
	  "  -wb <number>\tmodify delay between all bytes sent [s]\n");
  exit(1);
}


int main(int argc, char **argv)
{
  
#if (INT_MAX == 2147483647L)
  typedef int int32;
#elif (LONG_MAX == 2147483647L)
  typedef int int32;
#else
#error "Sorry, no 32-bit integer type available!"
#endif

  unsigned char header[5];
  int hc = 0, received = -1, length = -1, dir = -1;
  unsigned char buffer[65];
  unsigned char msg[32];
  unsigned char seed[8];
  unsigned char *data = NULL;
  unsigned char fs_query = 0;  /* Fiat-Shamir query bit Q (instr. 0x80) */
  unsigned char proc_byte;     /* procedure byte in passive mode */
  int sw2_next = 0;  /* second procedure byte SW2 will follow */
  int sw2_ok = 0;    /* second procedure byte SW2 has been received */
  int sync_lost = 1;
  time_t start, now, sched = 0;
  struct tm *date;
#define LOGMAX 200
  time_t logtime[LOGMAX];
  unsigned char huge *logmsg;
  int loglength = 0, lognext = 0;
  char *logfn = "VCLOG";
  char *current_name = "???";
  int station = 0;
  int com = 2;
  int quit = 0, message = 1;
  int vcl_print = 0;
  int screen_saver = 1, dark = 0;
  int passive = 0;   /* passive==1: Season7 listens only to card traffic */
  int suspended = 0, vbl = 0;
  int paranoid = 1;         /* paranoid = 1: be carefull about VCL leaks */
  int week, dayofweek;           /* ISO 8601 week and day-of-week number */
  int monthformat = 0;  /* 1: use oldstyle vcmmddhh.stt file name format */
  long duration = -1;       /* maximum VCL/VBL recording time in minutes */
  int i, j, k, c;
  int check = 0;
  char satellite[81] = "", transponder[81] = "", program[81] = "";
  char tmp[129], fn[16];
  FILE *f;
  FILE *vcl_out = NULL, *vcl_in = NULL;
  char *vcl_out_fn = NULL, *vcl_in_fn = NULL;
  unsigned vcl_count = 0, vcl_length, l;
  unsigned vcl_current = 0, u;
  int32 huge *vcl_msg_hash = NULL;
  char (huge *vcl_seed)[8] = NULL;
#ifdef _POSIX_PRIORITY_SCHEDULING
  struct sched_param my_priority;
#endif
#ifdef TEST_DECODE
  extern int rom_try[], rom_count;
  unsigned char decmsg[32], testseed[8];
#endif

#if defined(SYS_DOS) && !defined(__MSC)
  clrscr();
#endif

  fprintf(stderr, "Season7 %d.%d", VERSION_MAJOR, VERSION_MINOR);
  fprintf(stderr, " -- VideoCrypt Smart Card Emulator -- ");
  fprintf(stderr, "(C) 1996 Markus G. Kuhn\n\n");
  display[10] = VERSION_MAJOR + '0';
  display[12] = VERSION_MINOR + '0';

#if 0
  fprintf(stderr, "COPYRIGHTED SOFTWARE: REDISTRIBUTION ILLEGAL!\n\n");
  fprintf(stderr, "Licensed for: ");
  fprintf(stderr, "tv-crypt list members only");
  fprintf(stderr, "\n\n");
#endif

  fprintf(stderr,
  "Important: This software must not be used in order to view channels\n"
  "           for which a regular subscription would be available in your\n"
  "           country, unless you have already such a subscription and\n"
  "           want to use this software only for educational purposes.\n\n");

  tzset();  /* get local time zone from environment variable TZ */
  time(&start);

  if (LONG_MAX != 2147483647L) {
    fprintf(stderr, "Porting error!\n");
    exit(1);
  }

#ifdef SECRETS
  for (i = 0; i < ADR_SPACE_SIZE; i++)
    modified[i] = no_mod;
#endif

  for (i = 1; i < argc; i++) {
    for (j = 0; j < 999 && argv[i][j]; j++)
      switch(argv[i][j]) {
      case '-':
	/* ignore a hyphen */
	break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	com = atoi(argv[i] + j);
	while (isdigit(argv[i][j + 1])) j++;
	break;
      case 'd':
	/* switch on screen log mode */
	debug = 1;
	break;
      case 'D':
	/* switch on full screen log mode */
	debug = 2;
	break;
      case 't':
	/* switch on nanocommand screen log mode */
	debug = 3;
	break;
      case 'p':
	/* activate passive mode: Season7 is now only listening */
	passive = 1;
	break;
      case 'P':
	/* deactivate paranoia mode, in case we were wrong about
	 * soft scrambling */
	paranoid = 0;
	break;
      case 'm':
	/* suppress TV display messages */
	message = 0;
	break;
      case 'M':
	/* use oldstyle VCL filenames with month and day of month
	 * instead of new week number format */
	monthformat = 1;
	break;
      case 'e':
	/* economy mode: avoid time consuming tasks like printf()s
	 * in order to allow stable operation on very slow systems */
	economy = 1;
	break;
      case 's':
	/* supress screen saver */
	screen_saver = 0;
	break;
      case 'b':
      case 'B':
	/* produce a VBL (videocrypt broadcast log) file from the data */
        vbl = 1;
        /* fall through */
      case 'o':
      case 'O':
	/* produce a VCL (videocrypt card log) file from the data
	 * recorded in passive mode.                              */
	if (vcl_out_fn) {
          fprintf(stderr, "More than one VCL/VBL output file specified!\n");
          exit(1);
        };
        if (vcl_in_fn) {
          fprintf(stderr, "Both VCL input and output file specified!\n");
          exit(1);
        }
	if (program[0] == 0) {
	  fprintf(stderr, "Which satellite do you receive (Astra): ");
	  gets(satellite);
	}
	if (strlen(satellite) == 0) strcpy(satellite, "Astra");
	if (program[0] == 0 && transponder[0] == 0) {
	  fprintf(stderr, "Which transponder on '%s' do you record (08): ",
		  satellite);
	  gets(transponder);
	}
	if (strlen(transponder) == 0) strcpy(transponder, "08");
	if (program[0] == 0) {
	  fprintf(stderr, "What are you recording: ");
	  gets(program);
	  fprintf(stderr, "\n");
	}
	satellite[31] = 0;
	transponder[7] = 0;
	program[63] = 0;
	for (k = strlen(satellite); k < 32; k++) satellite[k] = 0;
	for (k = strlen(transponder); k < 8; k++) transponder[k] = 0;
	for (k = strlen(program); k < 64; k++) program[k] = 0;
	if (argv[i][j] >= 'a') {
          now = start;
          if (sched) now = sched;
	  now += 60 * 5; /* round up the hour by 5 minutes */
	  date = gmtime(&now);
	  if (monthformat) {
	    /* old filename convention */
	    strftime(fn, 9, "vc%m%d%H", date);
	    if (vbl) fn[1] = 'b';
	    sprintf(fn + 8, ".%c%.2s", tolower(satellite[0]), transponder);
	  } else {
	    /* modern file name convention */
	    get_week_number(date, &week, &dayofweek);
	    sprintf(fn, "%02d%1d%02d%c%.2s.v%cl", week, dayofweek,
		    date->tm_hour, tolower(satellite[0]), transponder,
		    vbl ? 'b' : 'c');
	  }
	  vcl_out_fn = fn;
	} else if (argv[i][j+1] != '\0')
	  vcl_out_fn = argv[i] + j + 1;
	else if (++i < argc)
	  vcl_out_fn = argv[i];
	else {
	  fprintf(stderr, "Specify filename after option -O/-B "
		  "(e.g. 'season7 -O vc061817.a08')!\n");
	  exit(1);
	}
	j = 1000;
	if (!vbl) passive = 1;
	break;
      case 'i':
	/* read in a VCL file and use it in order to answer the
	 * hash function queries of the decoder.                */
        if (vcl_out_fn) {
          fprintf(stderr, "Both VCL input and output file specified!\n");
          exit(1);
        }
	if (argv[i][j+1] != '\0')
	  vcl_in_fn = argv[i] + j + 1;
	else if (++i < argc)
	  vcl_in_fn = argv[i];
	else {
	  fprintf(stderr, "Specify filename after option i "
		  "(e.g. 'season7 -i vc061817.a08')!\n");
	  exit(1);
	}
	j = 1000;
	do {
	  vcl_in = fopen(vcl_in_fn, "rb");
	  if (vcl_in == NULL) {
	    fprintf(stderr, "Sorry, can't open VCL file '%s", vcl_in_fn);
	    perror("'");
	    exit(1);
	  }
	  printf("VCL file '%s': ", vcl_in_fn);
	  fflush(stdout);
	  fread(tmp, 1, 128, vcl_in);
	  if (feof(vcl_in)) {
	    fprintf(stderr, "Unexpected end of VCL input file!\n");
	    exit(1);
	  }
	  if (strncmp(tmp, "VCL1", 4)) {
	    fprintf(stderr, "Sorry, '%s' is not a VCL file!\n", vcl_in_fn);
	    exit(1);
	  }
	  tmp[55] = tmp[63] = tmp[127] = 0;
	  printf("%s, transp. %s, %.4s-%.2s-%.2s "
		 "%.2s:%.2s:%.2s%.1s\n%s\n\n",
		 tmp + 24, tmp + 56, tmp + 8, tmp + 12, tmp + 14,
		 tmp + 17, tmp + 19, tmp + 21, tmp + 23, tmp + 64);
	  if (tmp[4] || tmp[5]) {
	    fprintf(stderr, "Sorry, VCL file too long.\n");
	    exit(1);
	  }
	  vcl_length = ((unsigned)(unsigned char) tmp[6] << 8) |
	    (unsigned)(unsigned char) tmp[7];
	  if (!vcl_length) break;
	  if (vcl_count + (unsigned long) vcl_length > UINT_MAX / 8) {
	    fprintf(stderr, "Too much VCL data for this system.\n");
	    exit(1);
	  }
	  if (vcl_count) {
	    vcl_msg_hash = (int32 huge *)
	      realloc(vcl_msg_hash, (vcl_count + vcl_length) * 4L);
	    vcl_seed = (char (huge *)[8])
	      realloc(vcl_seed, (vcl_count + vcl_length) * 8L);
	  } else {
	    vcl_msg_hash = (int32 huge *)  malloc(vcl_length * 4L);
	    vcl_seed = (char (huge *)[8]) malloc(vcl_length * 8L);
	  }
	  if (!vcl_msg_hash || !vcl_seed) {
	    fprintf(stderr, "Not enough memory for %lu bytes (%u entries).\n",
		    (vcl_count + vcl_length) * 12L, vcl_count + vcl_length);
	    exit(1);
	  }
	  for (l = vcl_count; l < vcl_count + vcl_length; l++)
	    if (fread(vcl_msg_hash + l, 1, 4, vcl_in) != 4 ||
		fread(vcl_seed     + l, 1, 8, vcl_in) != 8) {
	      fprintf(stderr, "Unexpected end of VCL input file!\n");
	      exit(1);
	    }
	  if (vcl_print) {
	    printf("VCL start (%s)\n", vcl_in_fn);
	    for (l = vcl_count; l < vcl_count + vcl_length; l++) {
	      for (k = 0; k < 4; k++)
		printf("%02x ", *((unsigned char *)&vcl_msg_hash[l] + k));
	      printf("  ");
	      for (k = 0; k < 8; k++)
		printf("%02x ", (unsigned char) vcl_seed[l][k]);
	      putchar('\n');
	    }
	    printf("VCL end (%s)\n\n", vcl_in_fn);
	  }
	  vcl_count += vcl_length;
          fclose(vcl_in);
	} while (++i < argc && *(vcl_in_fn = argv[i]) != '-');
	--i;
	if (vcl_print)
	  exit(0);
	break;
      case 'n':
	if (argv[i][j+1] != '\0')
	  strncpy(program, argv[i] + j + 1, 80);
	else if (++i < argc) {
	  strncpy(program, argv[i], 80);
          while (i+1 < argc && argv[i+1][0] != '-') {
            strncat(program, " ", 80-strlen(program));
            strncat(program, argv[++i], 80-strlen(program));
          }
	} else {
	  fprintf(stderr, "Give a string after option -n "
		  "(e.g. 'season7 -n StarTrek-Episode-178')!\n");
	  exit(1);
	}
	program[79] = 0;
	j = 1000;
	break;
      case 'r':
	if (argv[i][j+1] != '\0')
	  strncpy(transponder, argv[i] + j + 1, 80);
	else if (++i < argc)
	  strncpy(transponder, argv[i], 80);
	else {
	  fprintf(stderr, "Give a string after option -r "
		  "(e.g. 'season7 -r 08')!\n");
	  exit(1);
	}
	transponder[79] = 0;
	j = 1000;
	break;
      case 'v':
	/* print contents of VCL file to stdout */
	vcl_print = 1;
        break;
      case 'l':
	/* length of VCL recording in minutes for unsupervised VCL recording */
	j++;
	if (!isdigit(argv[i][j]))
	  i++, j = 0;
	if (!isdigit(argv[i][j]))
	  usage();
	duration = atol(argv[i] + j);
	while (isdigit(argv[i][j + 1])) j++;
	break;
      case 'c':
	if (argv[i][j+1] != '\0')
	  sched = parse_time(argv[i] + j + 1);
	else if (++i < argc)
	  sched = parse_time(argv[i]);
	else {
	  fprintf(stderr, "Give a time after option -t "
		  "(e.g. 'season7 -t 15:00')!\n");
	  exit(1);
	}
	j = 1000;
	break;
      case 'w':
	/*
	 * modify delay table, e.g. option wb200 waits 200 us after each
	 * byte because this sets delus['b' - 'a'] = 200.
	 */
	j++;
	k = tolower(argv[i][j]) - 'a';
	if (k < 0 || k >= DELAYS) {
	  fprintf(stderr, "Only wait options between -wa <number> and "
		  "-w%c <number> possible!\n", 'a' + DELAYS - 1);
	  exit(1);
	}
	j++;
	if (!isdigit(argv[i][j]))
	  i++, j = 0;
	if (!isdigit(argv[i][j]))
	  usage();
	delus[k] = atol(argv[i] + j);
	while (isdigit(argv[i][j + 1])) j++;
	break;
      case 'h':
      case 'H':
      case '?':
	usage();
	break;
      default:
	fprintf(stderr, "Error in command line argument '%s'!\n", argv[i]);
	fprintf(stderr, "                                %*s^\n", j, "");
	usage();
	break;
      }
  }

  fprintf(stderr, "Press q for exit.\n\n");
  fprintf(stderr, "Using serial port COM%d, byte delay %lu s.\n",
          com, delus[1]);
  if (passive) {
    fprintf(stderr, "Passive mode: only listening to data traffic.\n");
    memset(serial, 0, 6);
  }

  /* wait on programmed recording start */
  time(&now);
  strftime(tmp, 80, "%Y-%m-%d %H:%M:%S", localtime(&sched));
  prepare_console();
  while (!quit && now < sched) {
    if (!dark)
      fprintf(stderr, "\rWaiting until %s, %02ld:%02ld:%02ld remaining.",
              tmp, (long) difftime(sched, now) / 3600,
              ((long) difftime(sched, now) % 3600) / 60,
              (long) difftime(sched, now) % 60);
    wait_us(200000L);
    if (kbhit()) {
      c = getch();
      switch (c) {
      case 0:
        /* ignore function keys */
        getch();
        break;
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 27:
        /* abort program */
        restore_console();
        exit(0);
      default:
        dark = 0;
        time(&start);
        break;
      }
    }
    /* should the screen be blanked? */
    if (screen_saver && !dark && difftime(now, start) > 5 * 60) {
      dark = 1;
#if defined(SYS_DOS) && !defined(__MSC)
     clrscr();
#else
     for (i = 1; i < 60; i++)
       fputc('\n',stderr);
#endif
    }
    time(&now);
  }
  restore_console();
  time(&start);
  dark = 0;

  /* Print time and determine week number */
  date = gmtime(&start);
  strftime(tmp, 80, "%Y-%m-%d", date);
  get_week_number(date, &week, &dayofweek);
  fprintf(stderr, "\rUniversal Time (UTC): %s (-W%02d-%d) ",
	  tmp, week, dayofweek);
  strftime(tmp, 80, "%H:%M:%S", date);
  fprintf(stderr, "%sZ\t\n", tmp);

  if (!passive) {
    /* switch to real-time mode */
#ifdef _POSIX_MEMLOCK
    if (mlockall(MCL_CURRENT | MCL_FUTURE)) {
      if (errno == EPERM) {
	fprintf(stderr, "This real-time software must run as root!\n");
	exit(1);
      }
#ifdef SYS_LINUX
      if (errno == ENOSYS) {
	fprintf(stderr, "Season7 requires Linux kernel 1.3.67 or later!\n");
	exit(1);
      }
#endif
      perror("mlockall(MCL_CURRENT | MCL_FUTURE)");
      exit(-1);
    }
#endif
#ifdef _POSIX_PRIORITY_SCHEDULING
    if (!passive) {
      my_priority.sched_priority = sched_get_priority_max(SCHED_FIFO);
      if (sched_get_priority_min(SCHED_FIFO) < my_priority.sched_priority)
	my_priority.sched_priority--;
      if (sched_setscheduler(0, SCHED_FIFO, &my_priority)) {
	perror("sched_setscheduler(..., SCHED_FIFO, ...)");
	exit(-1);
      }
    }
#endif
  }

  logmsg = (unsigned char *) malloc(32U * LOGMAX);
  if (!logmsg) {
    fprintf(stderr, "Not enough memory for logmsg, LOGMAX too large!\n");
    exit(1);
  }

  if (vcl_out_fn) {
    /* open VCL output file */
    vcl_out = fopen(vcl_out_fn, "wb");
    if (vcl_out == NULL) {
      fprintf(stderr, "Sorry, can't open VCL/VBL file '%s", vcl_out_fn);
      perror("'");
      exit(1);
    }
    fprintf(stderr, "Writing %s file '%s'.\n", vbl ? "VBL" : "VCL",
            vcl_out_fn);
    /* Write VCL/VBL header (128 bytes) */
    fprintf(vcl_out, vbl ? "VBL1" : "VCL1");  /* Magic code, version 1 */
    putc(0, vcl_out);          /* space for number of seeds (bigendian) */
    putc(0, vcl_out);
    putc(0, vcl_out);
    putc(0, vcl_out);
    strftime(tmp, 80, "%Y%m%dT%H%M%SZ", gmtime(&start));
    fwrite(tmp, 1, 16, vcl_out);
    fwrite(satellite, 1, 32, vcl_out);
    fwrite(transponder, 1, 8, vcl_out);
    fwrite(program, 1, 64, vcl_out);
  }

  activate_com(com);
  prepare_console();

  if (economy)
    fprintf(stderr, "\nOperating ...");
  else
    fprintf(stderr, "\nWaiting for reset ...");
  fflush(stderr);

  while (!quit) {
    /* keyboard commands */
    if (kbhit()) {
      c = getch();
      switch (c) {
      case 0:
        /* ignore function keys */
        getch();
        break;
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 27:
        /* abort program */
        quit = 1;
        continue;
      case 'd':
      case 'D':
      case 't':
        /* switch on/off screen log */
        switch (c) {
        case 'd': debug = debug ? 0 : 1;  break;
        case 'D': debug = (debug == 2) ? 0 : 2;  break;
        case 't': debug = (debug == 3) ? 0 : 3;  break;
        }
        fprintf(stderr, "\rDebug mode %s.                               \n",
          debug ? ((debug == 2) ? "full" :
                  ((debug == 3) ? "tokens" :
                  "on")) : "off");
        break;
#ifdef SECRETS
      case 'w':
        /* write a list of recently modified address space positions */
        fprintf(stderr, "\r\t\t\t\t\t\t\n");
        for (i = 0; i < ADR_SPACE_SIZE; i++)
          if (modified[i] != no_mod) {
            fprintf(stderr, "%04x: %3d %02x %c ", i, adr_space[i],
                    adr_space[i], (adr_space[i] > 31 && adr_space[i] < 127)
                    ? adr_space[i] : ' ');
            switch (modified[i]) {
              case consistent_mod:   putc('c', stderr); break;
              case inconsistent_mod: putc('i', stderr); break;
              case identical_mod:    putc('0', stderr); break;
              case illegal_mod:      putc('X', stderr); break;
              default:               putc('?', stderr); break;
            }
            putc('\n', stderr);
          }
        break;
#endif
      case 'l':
#ifdef SECRETS
      case 'L':
#endif
        /* write last crypto messages (command 0x74) to a file */
        if (economy) break;
        i = lognext - loglength;
        while (i < 0) i += LOGMAX;
        fprintf(stderr, "\rWriting last %d messages to logfile '%s'"
                "...\t\t\t\n", loglength, logfn);
        f = fopen(logfn, "w");
        if (f == NULL)
          perror("Sorry, can't open logfile");
        else {
          strftime(tmp, 80, "%Y-%m-%d", gmtime(&logtime[i]));
          fprintf(f, "%s\n", tmp);
          for (j = 0; j < loglength; j++) {
            strftime(tmp, 80, "%H:%M:%SZ", gmtime(&logtime[i]));
            fprintf(f, "%s", tmp);
            memcpy(tmp, logmsg + i*32U, 32);
#ifdef SECRETS
            if (c == 'L') {
              de_xor((unsigned char *) tmp);
              fputc('.', f);
            } else
#endif
	      fputc(' ', f);
            for (k = 0; k < 32; k++) {
              fprintf(f, "%02x", (unsigned char) tmp[k]);
              if (k == 2 || k == 6 || k == 10 || k == 26)
                fputc(' ', f);
            }
            fprintf(f, "\n");
            if (++i >= LOGMAX) i = 0;
          }
          fclose(f);
        }
        break;
      case '+':
        /* increase byte delay by 0.5 ms */
        if (delus[1] < 2000) delus[1] += 500;
        fprintf(stderr, "\rByte delay set to %lu \xe6s.\t\t\t\t\t\n",
                delus[1]);
        break;
      case '-':
        /* decrease byte delay by 0.5 ms */
        if (delus[1] >= 500) delus[1] -= 500;
        fprintf(stderr, "\rByte delay set to %lu \xe6s.\t\t\t\t\t\n",
                delus[1]);
        break;
      case 's':
      case 'S':
        if (!vcl_out || suspended) break;
        fprintf(stderr, "\rVCL/VBL file recording suspended.\t\t\t\t\t\n");
        suspended = 1;
        break;
      case 'c':
      case 'C':
        if (!vcl_out || !suspended) break;
        fprintf(stderr, "\rVCL/VBL file recording continues.\t\t\t\t\t\n");
        suspended = 0;
        break;
      default:
        break;
      }
    }
    if (test_reset()) {
      /*
       *  we have a reset and the decoder now expects an
       *  answer-to-reset packet from the card as described in
       *  ISO 7816.
       */
      if (!passive || test_reset())
        purge_fifos();
      sync_lost = 0;
      if (!economy) fprintf(stderr, "\r\t\t\t\t\t\r");
      if (!economy && !debug && !dark)
        printf("RESET");
      while (test_reset() && !kbhit()) {
#ifdef SYS_LINUX
	wait_us(5000);
#endif
      }
      if (kbhit()) continue;
      if (passive) {
        if (parse_reset() < 0) sync_lost = 1;
      } else {
        wait_us(delus[0]);
        if (send(reset_answer, reset_answer_len)) continue;
        if (!economy && message) {
          now = time(NULL);
          strftime((char *) display + 13, 13, "   %H:%M    ",
                   localtime(&now));
          display[0] = '\xd8';
        }
      }
      if (debug && !sync_lost) {
        header[1] = 0x60; /* special value indicates log_packet a reset */
        header[4] = reset_answer_len;
        log_packet(header, (unsigned char *) reset_answer, 0);
      }
      station = 0;
      hc = 0;
      received = -1;
      sw2_next = 0;
      if (passive) memset(serial, 0, 6);
      test_reset();  /* clear D_DCD (just historical and now obsolete) */
    }

    if (passive && sync_lost) {
      /*
       * In passive mode, we can't just wait for a reset if
       * anything went wrong. So we try to resynchronize here
       * by waiting for characteristic byte sequences like
       * 90 00, 53 7x, or 53 8x. Let's hope this makes the passive mode a
       * little bit more robust.
       */
       if (receive(buffer + sync_lost - 1, 1)) {
         if (sync_lost == 1 && (buffer[0] == 0x90 || buffer[0] == 0x53)) {
           sync_lost++;
         } else {
           if (sync_lost == 2) {
             if (buffer[0] == 0x90 && buffer[1] == 0x00) {
               /* same state as after a reset */
               hc = 0;
               received = -1;
               sw2_next = 0;
               sync_lost = 0;
               if (!economy) fprintf(stderr, "\r\t\t\t\t\t\r");
             } else if (buffer[0] == 0x53 &&
                        ((buffer[1] & 0xf0) == 0x70 ||
                         (buffer[1] & 0xf0) == 0x80)) {
               header[0] = buffer[0];
               header[1] = buffer[1];
               hc = 2;
               received = -1;
               sw2_next = 0;
               sync_lost = 0;
               if (!economy) fprintf(stderr, "\r\t\t\t\t\t\r");
             } else {
               /* resynchronization attempt failed, because of 2nd byte */
               sync_lost = 1;
             }
           }  /* if (sync_lost == 2) */
         }
       } /* if (receive()) */
    } else if (hc < 5) {
      /* waiting for more header bytes */
      hc += receive(header + hc, 5 - hc);
    } else {
      if (received < 0) {
        /* new 5 byte command header has arrived */
	if (!passive) wait_us(delus[2]);
        if (header[0] != 0x53) {
          if (!passive) {
            send(UNSUPPORTED_CLASS, 2);
            hc = 0;
          } else sync_lost = 1;
          printf("\rUnsupported class code %02x!\t\t\n", header[0]);
        } else {
          data = buffer;
          memset(buffer, 0, 64); /* default answer: 00 00 00 ... */
          switch (header[1]) { /* the different instruction types */
            case 0x70: length =  6; dir = OUT; data = serial; break;
            case 0x72: length = 16; dir = IN;  break;
            case 0x74: length = 32; dir = IN;  data = msg; break;
            case 0x76: length =  1; dir = IN;  break;
            case 0x78: length =  8; dir = OUT; data = seed; break;
            case 0x7a: length = 25; dir = OUT; data = display; break;
            case 0x7c: length = 16; dir = OUT; break;
            case 0x7e: length = 64; dir = OUT;
              /* the card developers considered the following to be
               * a suitable Fiat-Shamir random number R^2 mod N ... ;-) */
              data[0] = 0x01;
              break;
            case 0x80: length =  1; dir = IN;  data = &fs_query; break;
            case 0x82: length = 64; dir = OUT;
              if (fs_query)
                data = answer_82;
              else
                data[0] = 0x01;
              break;
            default:
              if (!passive) {
                send(UNSUPPORTED_INSTRUCTION, 2);
                hc = 0;
              } else sync_lost = 1;
              printf("\rUnsupported instruction code %02x!\t\t\n", header[1]);
          }
          if (header[2] != 0x00 || header[3] != 0x00) {
            if (!passive) {
              send(INCORRECT_REFERENCE, 2);
              hc = 0;
            } else sync_lost = 1;
            printf("\rUnexpected P1 P2 values: %02x %02x!\t\t\t\n",
                   header[2], header[3]);
          }
          if (length != header[4]) {
            if (!passive) {
              send(INCORRECT_LENGTH, 2);
              hc = 0;
            } else sync_lost = 1;
            printf("\rIncorrect length %d in instruction %02x!\t\t\n",
                   header[4], header[1]);
          }
          if (hc == 5)
            if (!passive) {
              /* procedure byte: no Vpp, send all data in one piece */
              if (send(header + 1, 1)) continue;
              if (dir == IN)
                received = 0;
              else {
                /* here comes the answer */
		wait_us(delus[3]);
                if (send(data, length)) continue;
		wait_us(delus[4]);
                if (send(EVERYTHING_OK, 2)) continue;
                if (debug) log_packet(header, data, OUT);
                hc = 0;
                received = -1;
              }
            } else {
              if (receive(&proc_byte, 1)) {
                if (sw2_next) {
                  hc = 0;
                  received = -1;
                  sw2_next = 0;
                } else if (proc_byte != 0x60) {
                  if (((proc_byte & 0xf0) == 0x60 ||
                       (proc_byte & 0xf0) == 0x90))
                    sw2_next = 1;
                  else if (((proc_byte ^ header[1]) & 0xfe) == 0xfe) {
                    printf("\rSingle byte handshaking not supported!\t\t\n");
                    sync_lost = 1;
                  }
                  else if ((proc_byte & 0xfe) != header[1]) {
                    printf("\rUnexpected procedure byte %02x, INS = %02x!\t\n",
                           (unsigned char) proc_byte, header[1]);
                    sync_lost = 1;
                  } else {
                    received = 0;
                    sw2_ok = 0;
                  }
                }
              }
            }
        }
      } else if (received < header[4]) {
        /* waiting for more data to receive */
        received += receive(data + received, header[4] - received);
      } else if (!passive || sw2_ok) {
        /* here, all data has been received */
        if (!passive) {
	  wait_us(delus[4]);
          if (send(EVERYTHING_OK, 2)) continue;
	}
        if (debug) log_packet(header, data, dir);
        /* reaction on received data */
        switch (header[1]) {
          case 0x70:
            if (debug || !passive) break;
            printf("\rCard number: %02d %08lu\t\t\t\t\n\n",
                   serial[0] & 0x0f, ((unsigned long) serial[1] << 24) |
                   ((unsigned long) serial[2] << 16) |
                   ((unsigned long) serial[3] << 8) |
                   (unsigned long) serial[4]);
            break;
          case 0x72:
            if (debug) break;
            printf("\nMessage from previous card:\n");
            for (i = 0; i < 16; i++)
              printf("%02x ", buffer[i]);
            printf("  ");
            for (i = 0; i < 16; i++)
              putchar((buffer[i] > 31 && buffer[i] < 127) ? buffer[i] : '.');
            putchar('\n');
            break;
          case 0x74:
            if (vcl_in) {
              if (msg[0] & 8) {
                /* lookup random number seed in VCL file */
                u = vcl_current + 1;
                if (u >= vcl_count) u = 0;
                while (*(int32 *)(msg + 28) != vcl_msg_hash[u] &&
                       u != vcl_current)
                  if (++u >= vcl_count) u = 0;
                if (u != vcl_current) {
                  vcl_current = u;
                  memcpy(seed, vcl_seed + u, 8);
                  display[0] = 0x80;
                  if (!debug && !economy && !dark)
                    printf("\r\t\t\t\t\t\rMatched position %u.", u);
                } else {
                  memset(seed, 0, 8);
                  if (!debug && !economy && !dark)
                    printf("\r\t\t\t\t\t\rNo match currently in VCL file.");
                  if (!economy && message) {
                    display[0] = '\xd8';
                    memcpy(display + 13, "NO MATCH :-(", 12);
                  }
                }
              }
            } else {
	      if (vcl_out && vbl && !suspended) {
		fwrite(msg, 1, 32, vcl_out); /* 74h INS data */
                vcl_count++;
              }
#ifdef SECRETS
              /* calculate random number seed from crypto message */
              check = decode(msg, seed);
#else
	      /* dummy answer for tests */
	      for (i = 0; i < 8; i++) seed[i] = 0;
#endif
              if (!economy) {
                /* keep history of last LOGMAX crypto messages */
                time(&logtime[lognext]);
                memcpy(logmsg + (lognext++) * 32U, msg, 32);
                if (loglength < LOGMAX) loglength++;
                if (lognext >= LOGMAX) lognext = 0;
#ifdef SECRETS
                /* in passive mode, check for card commands */
                if (passive && serial[0] && !debug) {
                  memcpy(tmp, msg, 32);
                  de_xor((unsigned char *) tmp);
                  if (memcmp(serial + 1, tmp + 8, 3) == 0 &&
                      memchr(tmp + 11, serial[4], 16)) {
                    /* report that a command has been received */
                    for (i = 0; command[i].code != (unsigned char) tmp[3] &&
                                command[i].code >= 0; i++);
                    printf("\rCard received command %02xh ('%s') at ",
                           (unsigned char) tmp[3], command[i].text);
                    now = time(NULL);
                    strftime(tmp, 80, "%H:%M:%S", localtime(&now));
                    printf("%s.\a      \n", tmp);
                  }
                }
#endif
                /* manage on screen display */
                if (vcl_out) {
                  if (!debug && !dark && ((msg[0] & 8) || vbl))
                    printf("\r\t\t\t\t\t\rRecording position %d.",
                           vcl_count);
                  display[0] = 0x80;
                } else if (msg[0] & 8) {
                  if (station == 0) {
                      if ((check >> 2) == 1) {
                        /* it's a Sky channel */
                        if (data[6] != 0xd1) {
                          station = 1;
                          strcpy((char *) display + 13,
                                 channel_id[data[6] & 0x0f]);
                        }
                      } else if ((check >> 2) == 2) {
                        station = 1;
                        strcpy((char *) display + 13, "ADULTCHANNEL");
                      } else if ((check >> 3) == 3) {
                        station = 1;
                        strcpy((char *) display + 13, "    JSTV    ");
                      } else {
                        station = 1;
                        if ((data[0] & 0xe0) != 0xc0)
                          strcpy((char *) display + 13, "NO KEY FITS!");
                      }
                  } else if (station == 1) {
                    /* now remove on-screen display */
                    display[0] = 0x80;
                    station = 2;
                  }
                  /* PC display message */
                  if ((check >> 2) == 1) {
                    /* it's a Sky channel */
                    if (data[6] != 0xd1)
                      current_name = channel_name[data[6] & 0x0f];
                  } else if ((check >> 2) == 2)
                    current_name = "Adult Channel";
                  else if ((check >> 2) == 3)
                    current_name = "JSTV";
                  else if ((data[0] & 0xe0) != 0xc0)
                    current_name = "???, no key fits";
                  if (!debug && !dark)
                    printf("\r\t\t\t\t\t\rDecrypting %s (%02x %02x %02x)",
                           current_name, data[0], data[1], data[6]);
                } /* if (msg[0] & 8) */
                if (check & 2)
                  printf("\nChecksum wrong !!!\n");
              }  /* if (!economy) */
            }  /* if (vcl_in) else */
            break;
          case 0x76:
            printf("\rAuthorize button pressed.\t\t\t\t\n");
            break;
          case 0x78:
            if (vcl_out && !suspended) {
              if (!vbl) {
                fwrite(msg + 28, 1, 4, vcl_out); /* final 4 bytes of 74h */
	        if ((msg[0] & 0xef) == 0xe8 || !paranoid) {
		  fwrite(seed, 1, 7, vcl_out);
		  putc(seed[7] & 0x0f, vcl_out); /* only 60-bit are needed */
	        } else {
		  /* avoid soft scrambling identification attack */
		  for (i = 0; i < 8; i++)
		    putc(0, vcl_out);
	        }
                vcl_count++;
              }
            }
#if 1
	    if (passive && msg[0] != 0xf8)
	      printf("\rVCL WARNING: msg[0] = %02x!\n", msg[0]);
#endif
#ifdef TEST_DECODER
            if (passive && !economy) {
              /* check whether decrypt would have been correct */
              memcpy(decmsg, msg, 32);
              decode(decmsg, testseed);
              if (memcmp(seed, testseed, 8)) {
                printf("\rdecode() failed!\t\t\t\n");
#if 0
                for (i = 0; i < 32; i++)
                  printf("%02x%c", decmsg[i],
                          (i & 15) == 15 ? '\n' : ' ');
                printf("\n");
#endif
#define TEST_LOOPS 1
                rom_count = TEST_LOOPS;
                for (rom_try[0] = 0; rom_try[0] < 256; rom_try[0]++)
#if (TEST_LOOPS==2)
                  for (rom_try[1] = 0; rom_try[1] < 256; rom_try[1]++)
#endif
                  {
                    memcpy(decmsg, msg, 32);
                    decode(decmsg, testseed);
                    if (!memcmp(seed, testseed, 8))
                      printf("rom_try[0..1] = %02x %02x "
                             "succeeded!\n", rom_try[0], rom_try[1]);
                  }
                rom_count = 0;
              }
            }
#endif
            break;
          default: break;
        }
        hc = 0;
        received = -1;
      } else {
        /* wait in passive mode for SW1 and SW2 */
        if (receive(&proc_byte, 1)) {
          if (sw2_next) {
            sw2_ok = 1;
            sw2_next = 0;
          } else if (proc_byte != 0x60) {
            if (((proc_byte & 0xf0) == 0x60 ||
                 (proc_byte & 0xf0) == 0x90))
              sw2_next = 1;
            else {
              printf("\rUnexpected SW1 byte %02x, INS = %02x!\t\n",
                     (unsigned char) proc_byte, header[1]);
              sync_lost = 1;
            }
          }
        }
      }  /* if (!passive || sw2_ok) else */
    }  /* if (hc < 5) else */

    /* is our recording time over? */
    time(&now);
    if (duration > 0 && difftime(now, start) > duration * 60)
      quit = 1;

    /* should the screen be blanked? */
    if (screen_saver && !dark && difftime(now, start) > 5 * 60) {
      dark = 1;
#if defined(SYS_DOS) && !defined(__MSC)
     clrscr();
#else
     for (i = 1; i < 60; i++)
       fputc('\n',stderr);
#endif
    }

  }  /* while (!quit) */

  restore_console();
  deactivate_com();
  fprintf(stderr, "\n");

  if (vcl_out) {
    /* close VCL/VBL file */
    buffer[0] = 0;
    /* fill up file to a multiple of 1024, so that there are
     * no undeleted bytes in final DOS disk clusters. */
    for (l = 128 + vcl_count * (vbl ? 32 : 12); (l & 0x3ff) != 0; l++)
      fwrite(buffer, 1, 1, vcl_out);
    /* write number of seeds into header */
    if (fseek(vcl_out, 4, SEEK_SET)) {
      perror("Sorry, can't fseek() in VCL/VBL file");
      exit(1);
    }
    putc((vcl_count >> 24) & 0xff, vcl_out);
    putc((vcl_count >> 16) & 0xff, vcl_out);
    putc((vcl_count >>  8) & 0xff, vcl_out);
    putc(vcl_count & 0xff, vcl_out);
    if (fclose(vcl_out)) {
      perror("Sorry, can't close VCL/VBL file!");
      exit(1);
    }
  }

  return 0;
}
