;******************************************************************
; 8051 implementation of the Serpent cryptosystem
;
; Author: Vincent Journot            Version: 1.3
;******************************************************************

; Bi is stored from 60H to 6FH
; The wi used in the Key schedule are stored from 40H to 5FH
; Part of the Ki is temporarily stored from 30H to 33H
; 35H is used to store the round number 

; It requires 56 internal RAM bytes
;             70 339 machine cycles, ie 0.070 s for a 12 MHz processor      
;                                       0.235 s for a 3.57 Mhz one

      LJMP     TEST
      
;******************************************************************
; Apply the right S_BOX
; 
; It uses the improvement of the SBOXES' implementation by Dag Arne
; Osvik : R0 is 30H, R1 is 31H, R2 is 32H, R3 is 33H, R4 is 34H
;******************************************************************

SBOX0:
      MOV      A,30H
      XRL      33H,A                  ; R3 ^= R0      
      MOV      34H,31H                ; R4 = R1
      MOV      A,33H
      ANL      31H,A                  ; R1 &= R3
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,30H
      XRL      31H,A                  ; R1 ^= R0
      MOV      A,33H                  
      ORL      30H,A                  ; R0 |= R3
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      A,33H
      XRL      34H,A                  ; R4 ^= R3
      MOV      A,32H            
      XRL      33H,A                  ; R3 ^= R2
      MOV      A,31H
      ORL      32H,A                  ; R2 |= R1
      MOV      A,34H
      XRL      32H,A                  ; R2 ^= R4
      XRL      34H,#0FFH              ; R4 =~ R4
      MOV      A,31H
      ORL      34H,A                  ; R4 |= R1
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,34H
      XRL      31H,A                  ; R1 ^= R4
      MOV      A,30H
      ORL      33H,A                  ; R3 |= R0
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,33H
      XRL      34H,A                  ; R4 ^= R3
      MOV      33H,30H
      MOV      30H,31H
      MOV      31H,34H
      RET

SBOX1:
      XRL      30H,#0FFH              ; R0 =~ R0
      XRL      32H,#0FFH              ; R2 =~ R2
      MOV      34H,30H                ; R4 = R0
      MOV      A,31H
      ANL      30H,A                  ; R0 &= R1
      MOV      A,30H
      XRL      32H,A                  ; R2 ^= R0
      MOV      A,33H                  
      ORL      30H,A                  ; R0 |= R3
      MOV      A,32H            
      XRL      33H,A                  ; R3 ^= R2
      MOV      A,30H
      XRL      31H,A                  ; R1 ^= R0
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      A,31H
      ORL      34H,A                  ; R4 |= R1
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,30H
      ORL      32H,A                  ; R2 |= R0
      MOV      A,34H
      ANL      32H,A                  ; R2 &= R4
      MOV      A,31H
      XRL      30H,A                  ; R0 ^= R1
      MOV      A,32H
      ANL      31H,A                  ; R1 &= R2
      MOV      A,30H
      XRL      31H,A                  ; R1 ^= R0
      MOV      A,32H
      ANL      30H,A                  ; R0 &= R2
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      34H,30H
      MOV      30H,32H
      MOV      32H,33H
      MOV      33H,31H
      MOV      31H,34H
      RET
      
SBOX2:
      MOV      34H,30H                ; R4 = R0
      MOV      A,32H
      ANL      30H,A                  ; R0 &= R2
      MOV      A,33H
      XRL      30H,A                  ; R0 ^= R3
      MOV      A,31H
      XRL      32H,A                  ; R2 ^= R1
      MOV      A,30H
      XRL      32H,A                  ; R2 ^= R0
      MOV      A,34H
      ORL      33H,A                  ; R3 |= R4
      MOV      A,31H            
      XRL      33H,A                  ; R3 ^= R1
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      31H,33H
      MOV      A,34H
      ORL      33H,A                  ; R3 |= R4
      MOV      A,30H            
      XRL      33H,A                  ; R3 ^= R0
      MOV      A,31H
      ANL      30H,A                  ; R0 &= R1
      MOV      A,30H            
      XRL      34H,A                  ; R4 ^= R0
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,34H
      XRL      31H,A                  ; R1 &= R4
      XRL      34H,#0FFH              ; R4 =~ R4
      MOV      30H,32H
      MOV      32H,31H
      MOV      31H,33H
      MOV      33H,34H
      RET

SBOX3: 
      MOV      34H,30H                ; R4 = R0  
      MOV      A,33H                  
      ORL      30H,A                  ; R0 |= R3
      MOV      A,31H            
      XRL      33H,A                  ; R3 ^= R1
      MOV      A,34H
      ANL      31H,A                  ; R1 &= R4
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,33H
      XRL      32H,A                  ; R2 ^= R3
      MOV      A,30H
      ANL      33H,A                  ; R3 &= R0
      MOV      A,31H
      ORL      34H,A                  ; R4 |= R1
      MOV      A,34H
      XRL      33H,A                  ; R3 ^= R4
      MOV      A,31H
      XRL      30H,A                  ; R0 ^= R1
      MOV      A,30H
      ANL      34H,A                  ; R4 &= R0
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,30H
      ORL      31H,A                  ; R1 |= R0
      MOV      A,32H
      XRL      31H,A                  ; R1 ^= R2
      MOV      A,33H
      XRL      30H,A                  ; R0 ^= R3
      MOV      32H,31H                ; R2 = R1
      MOV      A,33H
      ORL      31H,A                  ; R1 |= R3
      MOV      A,31H
      XRL      30H,A                  ; R0 ^= R1
      MOV      31H,32H
      MOV      32H,33H
      MOV      33H,34H
      RET

SBOX4:
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      XRL      33H,#0FFH              ; R3 =~ R3
      MOV      A,33H
      XRL      32H,A                  ; R2 ^= R3
      MOV      A,30H            
      XRL      33H,A                  ; R3 ^= R0
      MOV      34H,31H                ; R4 = R1
      MOV      A,33H
      ANL      31H,A                  ; R1 &= R3
      MOV      A,32H
      XRL      31H,A                  ; R1 ^= R2
      MOV      A,33H
      XRL      34H,A                  ; R4 ^= R3
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      A,34H
      ANL      32H,A                  ; R2 &= R4
      MOV      A,30H
      XRL      32H,A                  ; R2 ^= R0
      MOV      A,31H
      ANL      30H,A                  ; R0 &= R1
      MOV      A,30H            
      XRL      33H,A                  ; R3 ^= R0
      MOV      A,31H
      ORL      34H,A                  ; R4 |= R1
      MOV      A,30H
      XRL      34H,A                  ; R4 ^= R0
      MOV      A,33H
      ORL      30H,A                  ; R0 |= R3
      MOV      A,32H
      XRL      30H,A                  ; R0 ^= R2
      MOV      A,33H
      ANL      32H,A                  ; R2 &= R3
      XRL      30H,#0FFH              ; R0 =~ R0
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      32H,30H
      MOV      30H,31H
      MOV      31H,34H
      RET

SBOX5:
      MOV      A,31H
      XRL      30H,A                  ; R0 ^= R1
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      XRL      33H,#0FFH              ; R3 =~ R3
      MOV      34H,31H                ; R4 = R1
      MOV      A,30H
      ANL      31H,A                  ; R1 &= R0
      MOV      A,33H
      XRL      32H,A                  ; R2 ^= R3
      MOV      A,32H
      XRL      31H,A                  ; R1 ^= R2
      MOV      A,34H
      ORL      32H,A                  ; R2 |= R4
      MOV      A,33H
      XRL      34H,A                  ; R4 ^= R3
      MOV      A,31H
      ANL      33H,A                  ; R3 &= R1
      MOV      A,30H
      XRL      33H,A                  ; R3 ^= R0
      MOV      A,31H
      XRL      34H,A                  ; R4 ^= R1
      MOV      A,32H
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,30H            
      XRL      32H,A                  ; R2 ^= R0
      MOV      A,33H
      ANL      30H,A                  ; R0 &= R3
      XRL      32H,#0FFH              ; R2 =~ R2
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      A,33H
      ORL      34H,A                  ; R4 |= R3
      MOV      A,34H
      XRL      32H,A                  ; R2 ^= R4
      MOV      34H,30H
      MOV      30H,31H
      MOV      31H,33H
      MOV      33H,32H
      MOV      32H,34H
      RET

SBOX6:
      XRL      32H,#0FFH              ; R2 =~ R2
      MOV      34H,33H                ; R4 = R3
      MOV      A,30H
      ANL      33H,A                  ; R3 &= R0
      MOV      A,34H
      XRL      30H,A                  ; R0 ^= R4
      MOV      A,32H
      XRL      33H,A                  ; R3 ^= R2
      MOV      A,34H
      ORL      32H,A                  ; R2 |= R4
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,30H            
      XRL      32H,A                  ; R2 ^= R0
      MOV      A,31H
      ORL      30H,A                  ; R0 |= R1
      MOV      A,31H            
      XRL      32H,A                  ; R2 ^= R1
      MOV      A,30H            
      XRL      34H,A                  ; R4 ^= R0
      MOV      A,33H
      ORL      30H,A                  ; R0 |= R3
      MOV      A,32H
      XRL      30H,A                  ; R0 ^= R2
      MOV      A,33H            
      XRL      34H,A                  ; R4 ^= R3
      MOV      A,30H            
      XRL      34H,A                  ; R4 ^= R0
      XRL      33H,#0FFH              ; R3 =~ R3
      MOV      A,34H
      ANL      32H,A                  ; R2 &= R4
      MOV      A,33H            
      XRL      32H,A                  ; R2 ^= R3
      MOV      33H,32H
      MOV      32H,34H
      RET

SBOX7:
      MOV      34H,31H                ; R4 = R1
      MOV      A,32H
      ORL      31H,A                  ; R1 |= R2
      MOV      A,33H
      XRL      31H,A                  ; R1 ^= R3
      MOV      A,32H            
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,31H            
      XRL      32H,A                  ; R2 ^= R1
      MOV      A,34H
      ORL      33H,A                  ; R3 |= R4
      MOV      A,30H
      ANL      33H,A                  ; R3 &= R0      
      MOV      A,32H            
      XRL      34H,A                  ; R4 ^= R2
      MOV      A,31H            
      XRL      33H,A                  ; R3 ^= R1
      MOV      A,34H
      ORL      31H,A                  ; R1 |= R4
      MOV      A,30H
      XRL      31H,A                  ; R1 ^= R0
      MOV      A,34H
      ORL      30H,A                  ; R0 |= R4
      MOV      A,32H
      XRL      30H,A                  ; R0 ^= R2
      MOV      A,34H
      XRL      31H,A                  ; R1 ^= R4
      MOV      A,31H            
      XRL      32H,A                  ; R2 ^= R1
      MOV      A,30H
      ANL      31H,A                  ; R1 &= R0      
      MOV      A,34H
      XRL      31H,A                  ; R1 ^= R4
      XRL      32H,#0FFH              ; R2 =~ R2
      MOV      A,30H
      ORL      32H,A                  ; R2 |= R0
      MOV      A,32H            
      XRL      34H,A                  ; R4 ^= R2
      MOV      32H,31H
      MOV      31H,33H
      MOV      33H,30H
      MOV      30H,34H
      RET

;******************************************************************
; Store 
;
; Function used in order to store a 32-bit word in 30H 31H 32H 33H
;******************************************************************

STORE:
      MOV      36H,#04H
      MOV      R1,#30H
LBL:  MOV      A,@R0
      MOV      @R1,A
      MOV      A,R0
      ADD      A,#04H
      MOV      R0,A
      INC      R1
      DJNZ     36H,LBL
      RET

STORE_INV:      
      MOV      36H,#04H
      MOV      R1,#30H
LBL0: MOV      A,@R1
      MOV      @R0,A
      MOV      A,R0
      ADD      A,#04H
      MOV      R0,A
      INC      R1
      DJNZ     36H,LBL0
      RET

;****************************************************************** 
; XOR
;
; XOR between two 32-bit words ; result in the first one
; XORB is used to do a XOR between Bi and Ki
;******************************************************************

XOR4  MACRO    left,right
      MOV      R1,#right
      MOV      R0,#left
      MOV      A,@R1
      XRL      left,A
      INC      R0
      INC      R1
      MOV      A,@R0
      XRL      A,@R1
      MOV      @R0,A
      INC      R0
      INC      R1
      MOV      A,@R0
      XRL      A,@R1
      MOV      @R0,A
      INC      R0
      INC      R1
      MOV      A,@R0
      XRL      A,@R1
      MOV      @R0,A      
ENDM

XORB: MOV      36H,#04H
      MOV      R1,#30H
LBL1: MOV      A,@R1
      XRL      A,@R0
      MOV      @R1,A
      MOV      A,R0
      ADD      A,#04H
      MOV      R0,A
      INC      R1
      DJNZ      36H,LBL1
      RET

;******************************************************************
; The functions which make a rotation of a 32-bit word are used 
; during the linear transformation
;******************************************************************

;******************************************************************
; Rotate left a 32-bit word by 3 bits
;******************************************************************

ROTL3 MACRO    one
      MOV      A,one
      MOV      0F0H,#8
      MUL      AB      
      MOV      34H,0F0H
      MOV      one,A
      MOV      A,one+1
      MOV      0F0H,#8
      MUL      AB      
      MOV      one+1,A
      MOV      A,0F0H
      ORL      one,A
      MOV      A,one+2
      MOV      0F0H,#8
      MUL      AB      
      MOV      one+2,A
      MOV      A,0F0H
      ORL      one+1,A      
      MOV      A,one+3
      MOV      0F0H,#8
      MUL      AB      
      MOV      one+3,A
      MOV      A,0F0H
      ORL      one+2,A
      MOV      A,34H      
      ORL      one+3,A      
ENDM      

;******************************************************************
; Rotate right a 32-bit word by 3 bits
;******************************************************************

ROTR3 MACRO    one
      MOV      A,one+3
      MOV      0F0H,#32
      MUL      AB      
      MOV      34H,A
      MOV      one+3,0F0H
      MOV      A,one+2
      MOV      0F0H,#32
      MUL      AB      
      MOV      one+2,0F0H
      ORL      one+3,A
      MOV      A,one+1
      MOV      0F0H,#32
      MUL      AB      
      MOV      one+1,0F0H
      ORL      one+2,A      
      MOV      A,one
      MOV      0F0H,#32
      MUL      AB      
      MOV      one,0F0H
      ORL      one+1,A
      MOV      A,34H      
      ORL      one,A      
ENDM      

;******************************************************************
; Rotate left a 32-bit word by 1 bit
;******************************************************************

ROTL1 MACRO    one
      MOV      A,one
      RLC      A
      MOV      A,one+3
      RLC      A
      MOV      one+3,A
      MOV      A,one+2
      RLC      A
      MOV      one+2,A
      MOV      A,one+1
      RLC      A
      MOV      one+1,A
      MOV      A,one
      RLC      A
      MOV      one,A
ENDM

;******************************************************************
; Rotate right a 32-bit word by 1 bit
;******************************************************************

ROTR1 MACRO    one
      MOV      A,one+3
      RRC      A
      MOV      A,one
      RRC      A
      MOV      one,A
      MOV      A,one+1
      RRC      A
      MOV      one+1,A
      MOV      A,one+2
      RRC      A
      MOV      one+2,A
      MOV      A,one+3
      RRC      A
      MOV      one+3,A
ENDM

;******************************************************************
; Order
; 
; Rotate left a 256-bit bloc by 32 bits
; This function is used in order to reorder the wi in the Key 
; schedule
;******************************************************************

ORDER:      
      MOV      A,40H
      MOV      40H,44H
      MOV      44H,48H
      MOV      48H,4CH
      MOV      4CH,50H
      MOV      50H,54H
      MOV      54H,58H
      MOV      58H,5CH
      MOV      5CH,A
      MOV      A,41H
      MOV      41H,45H
      MOV      45H,49H
      MOV      49H,4DH
      MOV      4DH,51H
      MOV      51H,55H
      MOV      55H,59H
      MOV      59H,5DH
      MOV      5DH,A
      MOV      A,42H
      MOV      42H,46H
      MOV      46H,4AH
      MOV      4AH,4EH
      MOV      4EH,52H
      MOV      52H,56H
      MOV      56H,5AH
      MOV      5AH,5EH
      MOV      5EH,A
      MOV      A,43H
      MOV      43H,47H
      MOV      47H,4BH
      MOV      4BH,4FH
      MOV      4FH,53H
      MOV      53H,57H
      MOV      57H,5BH
      MOV      5BH,5FH
      MOV      5FH,A
      RET      

;******************************************************************
; Order2
; 
; Replace a 32-bit word ABCD by DCBA 
; This function is used in little vs big endian
;******************************************************************

ORDER2:
      MOV      36H,#02
LOOP5:
      MOV      0F0H,@R1
      MOV      A,@R0
      MOV      @R1,A
      MOV      @R0,0F0H
      INC      R0
      DEC      R1
      DJNZ     36H,LOOP5
      INC      R0
      INC      R0
      INC      R1
      INC      R1
      INC      R1
      INC      R1
      INC      R1
      INC      R1
      DJNZ     37H,ORDER2
      RET

;******************************************************************
; Linear transformation to obtain the next Bi
;
; X0,X1,X2,X3 := Si(Bi XOR Ki)
; X0, X1, X2, X3 are four 32-bit words which are divided into four
; 8-bit words 
;******************************************************************

LINEAR:
      MOV      30H,62H
      MOV      31H,63H
      MOV      32H,60H
      MOV      33H,61H
      ANL      33H,#0F8H
      MOV      A,60H
      XCH      A,62H
      MOV      60H,A
      MOV      A,61H
      XCH      A,63H
      MOV      61H,A
ROTR3 60H                        ; X0 := X0 <<< 13 (rotate) 
ROTL3 68H                        ; X2 := X2 <<< 3 (rotate)
XOR4  64H,60H                    ; X1 := X1 XOR X0
XOR4  64H,68H                    ; X1 := X1 XOR X2
XOR4  6CH,30H                    ; X3 := X3 XOR (X0 << 3) (shift)
XOR4  6CH,68H                    ; X3 := X3 XOR X2
      MOV      30H,65H
      MOV      31H,66H
      MOV      32H,67H
      MOV      33H,64H
      ANL      33H,#80H
ROTL1 64H                        ; X1 := X1 <<< 1
      MOV      A,6CH
      XCH      A,6FH
      XCH      A,6EH
      XCH      A,6DH
      MOV      6CH,A
ROTR1 6CH                        ; X3 := X3 <<< 7 
XOR4  60H,64H                    ; X0 := X0 XOR X1
XOR4  60H,6CH                    ; X0 := X0 XOR X3
XOR4  68H,30H                    ; X2 := X2 XOR (X1 << 7) 
XOR4  68H,6CH                    ; X2 := X3 XOR X2
      MOV      A,60H
      XCH      A,63H
      XCH      A,62H
      XCH      A,61H
      MOV      60H,A
ROTR3 60H                        ; X0 := X0 <<< 5 
ROTR1 68H
      MOV      A,68H
      XCH      A,69H
      XCH      A,6AH
      XCH      A,6BH
      MOV      68H,A
ROTR1 68H                        ; X2 := X2 <<< 22 
      RET

;******************************************************************
; Key schedule
;
; This function provides the Subkeys Ki
; It uses 8 32-bit words for each round 
;******************************************************************
 
SKEY:      
XOR4  40H,4CH                       ; wi-8 := wi-8 XOR wi-5
XOR4  40H,54H                       ; wi-8 := wi-8 XOR wi-3
XOR4  40H,5CH                       ; wi-8 := wi-8 XOR wi-1
      XRL      40H,#9EH        
      XRL      41H,#37H
      XRL      42H,#79H
      XRL      43H,#0B9H            ; wi-8 := wi-8 XOR 0x9e3779b9
      MOV      A,35H
      MOV      0f0h,#04H
      MUL      AB
      ADD      A,32H                ; A stores i
      XRL      43H,A                ; wi-8 := wi-8 XOR i
      MOV      A,40H
      XCH      A,43H
      XCH      A,42H
      XCH      A,41H
      MOV      40H,A
ROTL3 40H                           ; wi-8 := wi-8 <<< 11, ie wi-8 := wi 
      ACALL    ORDER                ; puts the wi in the right order 
      INC      32H
      RET

ENCRYPT:
      MOV      35H,#0               ; starts at Round 0

ROUNDS:
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND0:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX3
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX0
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND0
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND1:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX2
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX1
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND1
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND2:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX1
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX2
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND2
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND3:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX0
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX3
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND3
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND4:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX7
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX4
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND4
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND5:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX6
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX5
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND5
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND6:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX5
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX6
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND6
      ACALL    LINEAR      
      INC      35H             
      MOV      37H,#0
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
ROUND7:
      MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX4
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      ACALL    SBOX7
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,ROUND7
      MOV      A,35H
      XRL      A,#1FH
      JZ       ROUND32      ; at round 31, avoids the linear transformation
      ACALL    LINEAR      
      INC      35H             
      MOV      A,35H
      LJMP     ROUNDS 

ROUND32:
      MOV      37H,#0
      INC      35H
      MOV      32H,#0H
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
      ACALL    SKEY
S3:   MOV      A,#50H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE
      ACALL    SBOX3
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    XORB
      MOV      A,#60H
      ADD      A,37H
      MOV      R0,A
      ACALL    STORE_INV      
      INC      37H
      MOV      A,37H
      CJNE     A,#4,S3
      RET                        

TEST: MOV      R1,#43H
      MOV      R0,#40H
      MOV      37H,#0CH
      ACALL    ORDER2
      ACALL    ENCRYPT
      MOV      R1,#63H
      MOV      R0,#60H
      MOV      37H,#04H
      ACALL    ORDER2
      END