; ********************************************************************
; 8051 implementation of the RC6 cryptosystem
;
; Author : Gael Hachez (C) UCL Crypto Group 1998
; ********************************************************************


$MOD51
        SKEY    IDATA   255
        L       IDATA   SKEY-176    ; Used up to SETKEY
        AKEY    IDATA   L-16        ; Used only in SETKEY
        IKEY    DATA    30H         ; Used only in SETKEY
        JKEY    DATA    31H         ; Used only in SETKEY
	BADDR	DATA	32H	    ; Used only in SETKEY
	AADDR	DATA	33H	    ; Used only in SETKEY

        ATEXT   IDATA   L           ; Used in ECRYPT and DCRYPT
        BTEXT   IDATA   L-4         ; Used in ECRYPT and DCRYPT
        CTEXT   IDATA   L-8         ; Used in ECRYPT and DCRYPT
        DTEXT   IDATA   L-12        ; Used in ECRYPT and DCRYPT
        TTEXT   IDATA   L-16        ; Used in ECRYPT and DCRYPT
        UTEXT   IDATA   L-20        ; Used in ECRYPT and DCRYPT
	ADDR	DATA	33H	    ; Used in ECRYPT and DCRYPT
	SADDR	DATA	34H	    ; Used in ECRYPT and DCRYPT

        CSEG
        ORG     0H                                      
        JMP     TEST

; ********************************************************************
; Load        
; Load @R0 in the registers (32 bits)
; Result in R2|R3|R4|R5
; A and all flags are destroyed 
; ********************************************************************
         
LOAD:   MOV     A,@R0       ; Put @R0 in the registers
        MOV     R2,A
        INC     R0
        MOV     A,@R0
        MOV     R3,A
        INC     R0
        MOV     A,@R0
        MOV     R4,A
        INC     R0
        MOV     A,@R0
        MOV     R5,A        
        
        DEC     R0
        DEC     R0
        DEC     R0
         
        RET

; ********************************************************************
; Store        
; Store the registers (R2|R3|R4|R5) in @R0 (32 bits)
; A and all flags are destroyed 
; ********************************************************************
                
STORE:  MOV     A,R2        ; Save registers in @R0
        MOV     @R0,A
        INC     R0
        MOV     A,R3
        MOV     @R0,A
        INC     R0
        MOV     A,R4
        MOV     @R0,A
        INC     R0
        MOV     A,R5
        MOV     @R0,A
      
        DEC     R0
        DEC     R0
        DEC     R0
           
        RET
        
; ********************************************************************
; Rotate left        
; Rotate word R2|R3|R4|R5
; Rotate R6                          
; A,B,R6 and all flags are destroyed 
; ********************************************************************
 
ROTL:   MOV     A,R6            ; A = R6
        MOV     B,#8H           ; B = 8
        DIV     AB              ; A = A div B, B = A mod B
        MOV     R6,A            ; Store A in R6
        JNB     B.2,IROTL       ; Jump if B < 4 to IROTL (Inside Rotate Left)
        INC     R6              ; Increment R6 to make one more swap

; ***** Inside Rotate Right ******************************************
 
	MOV     A,#8H           ; A = 8
        SUBB    A,B             ; A = 8 - B
        MOV     B,A             ; B = A
        MOV     A,R5            ; A = R5
        
IROTR1: MOV     C,ACC.0         ; Lowest bit stored in the Carry
	MOV     A,R2            ; Rotate R2
        RRC     A
        MOV     R2,A
        MOV     A,R3            ; Rotate R3
        RRC     A
        MOV     R3,A
        MOV     A,R4            ; Rotate R4
        RRC     A
        MOV     R4,A
        MOV     A,R5            ; Rotate R5
        RRC     A
        MOV     R5,A                
        
        DJNZ    B,IROTR1        ; IF B <> 0 then go back

; ****** We continue to process the block switch *********************

ROTL1:  MOV     B,R6            ; Restore R6                
	JB      B.1,ROTRB     	; Jump if A > 2 to ROTRB (Rotate Right Block)
	JNB	B.0,ROTE	; If no rotate then end              

	; make one rotate left

	MOV     A,R5            ; A = R5
        XCH     A,R4            ; A = R4, R4=R5
        XCH     A,R3            ; A = R3, R3=R4
        XCH     A,R2            ; A = R2, R2=R3
        MOV     R5,A            ; R5 = A = R2
	
ROTE:   RET

; ********************************************************************
; Inside Rotate Left        
; Rotate word R2|R3|R4|R5
; Rotate B                           
; A and all flags are destroyed 
; ********************************************************************

IROTL:  MOV	A,B
	JZ	ROTL1		; If = 0 nothing to do
	MOV     A,R2            ; A = R2    
        
IROTL1: MOV     C,ACC.7         ; Highest bit stored in the Carry
	MOV     A,R5            ; Rotate R5
        RLC     A
        MOV     R5,A
        MOV     A,R4            ; Rotate R4
        RLC     A
        MOV     R4,A
        MOV     A,R3            ; Rotate R3
        RLC     A
        MOV     R3,A
        MOV     A,R2            ; Rotate R2
        RLC     A
        MOV     R2,A                
        
        DJNZ    B,IROTL1
        JMP     ROTL1
        
; ********************************************************************
; Rotate Right Block       
; Rotate word R2|R3|R4|R5
; Rotate B        
; A and all flags are destroyed  
; ********************************************************************
                 
ROTRB:	JB	B.0,ROTRBE

	MOV     A,R2            ; A = R2
        XCH     A,R4            ; A = R4, R4 = R2
	MOV	R2,A
	MOV	A,R3		; A = R3
        XCH     A,R5            ; A = R5, R5 = R3
	MOV	R3,A
	RET

ROTRBE:	MOV     A,R2            ; A = R2
        XCH     A,R3            ; A = R3, R3 = R2
        XCH     A,R4            ; A = R4, R4 = R3
        XCH     A,R5            ; A = R5, R5 = R4
	MOV     R2,A            ; R2 = A = R5
	RET

; ********************************************************************
; XOR        
; XOR word 1 R2|R3|R4|R5
; Address of Second Operand  = R1                          
; Result is stored in R2|R3|R4|R5                          
; A,R1 and all flags are destroyed 
; ********************************************************************
 
XOR4:   MOV     A,R2            ; XOR R2           
        XRL     A,@R1
        MOV     R2,A
        INC     R1
        MOV     A,R3            ; XOR R3
        XRL     A,@R1
        MOV     R3,A     
        INC     R1
        MOV     A,R4            ; XOR R4
        XRL     A,@R1
        MOV     R4,A     
        INC     R1
        MOV     A,R5            ; XOR R5
        XRL     A,@R1
        MOV     R5,A 
        
        RET
 
; ********************************************************************
; ADD        
; ADD word 1 R2|R3|R4|R5
; Address of Second Operand  = R1
; Result is stored in R2|R3|R4|R5                          
; A and all flags are destroyed 
; ********************************************************************
            
ADD4:   INC     R1
        INC     R1
        INC     R1
        MOV     A,R5            ; ADD R5
        ADD     A,@R1
        MOV     R5,A 
        DEC     R1
        MOV     A,R4            ; ADDC R4
        ADDC    A,@R1
        MOV     R4,A     
        DEC     R1
        MOV     A,R3            ; ADDC R3
        ADDC    A,@R1
        MOV     R3,A     
        DEC     R1
        MOV     A,R2            ; ADDC R2           
        ADDC    A,@R1
        MOV     R2,A
        RET 

; ********************************************************************
; Set Key        
; SKey in SKEY
; User key in L (max size 128 bits)
; Size of the user key in R7
; Number of rounds fixed to 20 rounds
; A,B,R0,R1,R6 and all flags are destroyed 
; ********************************************************************
 
SETKEY: MOV     R0,#SKEY-176+4
        MOV     @R0,#63H         ; Initialize S[0] with P_32 = B7E15163                             
        MOV     R5,#63H          ; Initialize registers with S[0]
        DEC     R0
        MOV     @R0,#51H
        MOV     R4,#51H
        DEC     R0
        MOV     @R0,#0E1H
        MOV     R3,#0E1H
        DEC     R0
        MOV     @R0,#0B7H       
        MOV     R2,#0B7H
        
        MOV     R1,#AKEY        ; Initialize A with Q_32 = 9E3779B9
        MOV     @R1,#0B9H       ; A used as temporary space
        DEC     R1
        MOV     @R1,#79H
        DEC     R1
        MOV     @R1,#37H
        DEC     R1
        MOV     @R1,#9EH

        MOV     R6,#1H           
        
SKB1:   CALL    ADD4            ; Initialize S[R6]

        MOV     A,R0
        ADD     A,#4H
        MOV     R0,A
        CALL    STORE
                
        INC     R6
        CJNE    R6,#44D,SKB1
        
	MOV	R2,#0H		; Initialize B = 0 in the registers
	MOV	R3,#0H
	MOV	R4,#0H
	MOV	R5,#0H

	MOV	R0,#AKEY-3
	CALL	STORE		; Initialize AKEY

	MOV	AADDR,#AKEY-3	; Set A address for the first round
	MOV	BADDR,#AKEY-3	; Set B address for the first round

	MOV	R6,#0H
                
SKB3:   MOV     A,R6            ; Compute I = R6 mod 44
        MOV     B,#44D
        DIV     AB
        MOV     IKEY,B
        
        MOV     A,R6            ; Compute J = R6 mod R7 (Key length)
        MOV     B,R7
        DIV     AB
        MOV     JKEY,B
        
	MOV	A,AADDR
        MOV     R1,A		; Put A address in R1
        
        CALL    ADD4            ; Rx = Rx + A
        
        MOV     R1,#SKEY-175    ; Put address of S[i] in R1 (need some computations)
        MOV     A,IKEY          ; A = I
        RL      A               ; A = 2 * A
        RL      A               ; A = 2 * A
        ADD     A,R1            ; A = A + R1
        MOV     R1,A            ; R1 = A  (= 4 * IKEY + R1)
        
        CALL    ADD4            ; Rx = Rx + S[I]
        
        PUSH    06H             ; Save R6
        
        MOV     R6,#3H          ; Rotate value = 3
        CALL    ROTL            ; A = A <<< 3
        
        POP     06H
        
        MOV     A,R1		; Save S[i] and address of A = @S[i]	
	MOV	AADDR,A
        MOV     R0,A
        CALL    STORE
        
	MOV	A,BADDR
        MOV     R1,A		; Put B address in R1
        
        CALL    ADD4            ; Rx = Rx + B
        
        MOV     A,R5
        ANL     A,#00011111B    ; Acc = Acc mod 32
        PUSH    ACC             ; Save Acc
        
        MOV     R1,#L-15        ; Put address of L[j] in R1 (need some computations)
        MOV     A,JKEY          ; A = J
        RL      A               ; A = 2 * A
        RL      A               ; A = 2 * A
        ADD     A,R1            ; A = A + R1
        MOV     R1,A            ; R1 = A  (= 4 * JKEY + R1)
        
        CALL    ADD4            ; Rx = Rx + L[j]
        
        POP     ACC
        
        PUSH    06H             ; Save R6
        
        MOV     R6,A            ; Rotate value = Acc
        CALL    ROTL            ; B = B <<< Acc
        
        POP     06H
        
        MOV     A,R1		; Save L[j] and address of B = @L[j]
	MOV	BADDR,A	
        MOV     R0,A
        CALL    STORE
        
        INC     R6
        CJNE    R6,#132D,SKB3

	RET
       
; ********************************************************************
; First Encryt sequence        
; @R0 = (2 * @R1^2 + @R1) <<< 5 in 32 bits
; A,B,R2,R3,R4,R5,R6 and all flags are destroyed 
; ********************************************************************
 
ENC1:   MOV     R2,#0H
	MOV	R3,#0H

; ********************************************************************
; MUL        
; MUL word in @R1
; result =  R2|R3|R4|R5
; ********************************************************************
        
	INC	R1
	INC	R1
	INC	R1
        MOV     A,@R1
        MOV     B,@R1
        MUL     AB              ; B0 * B0
	MOV	R5,A		; store B|A in R5|R4
	MOV	R4,B

; *** B0 * B1
        
        MOV     A,@R1
	DEC	R1
        MOV     B,@R1
        MUL     AB              ; B0 * B1

        MOV     R6,A            ; Save A in R6
        
        ADD     A,R4            ; R4 = R4 + LO(B01) 
        MOV     R4,A
        
        MOV     A,B             ; R3 = R3 + HI(B01)
        ADDC    A,R3
        MOV     R3,A

        MOV     A,R6            ; R4 = R4 + LO(B01)
        ADD     A,R4
        MOV     R4,A
        
        MOV     A,B		; R3 = R3 + HI(B01)
        ADDC    A,R3
        MOV     R3,A
        
        MOV     A,R2		; R2 = carry
        ADDC    A,#0H
        MOV     R2,A

; *** B1 * B1

        MOV     A,@R1
        MOV     B,@R1
        MUL     AB              ; B1 * B1
                        
        ADD     A,R3            ; R3 = R3 + LO(B11)
        MOV     R3,A
        
        MOV     A,B		; R2 = R2 + HI(B11)
        ADDC    A,R2
        MOV     R2,A

; *** B0 * B2

	DEC	R1        
        MOV     A,@R1
	INC	R1
	INC	R1
        MOV     B,@R1
        MUL     AB              ; B2 * B0
        
        MOV     R6,A            ; Save A in R6
        
        ADD     A,R3            ; R3 = R3 + LO(B02)
        MOV     R3,A
        
        MOV     A,B		; R2 = R2 + HI(B02)
        ADDC    A,R2
        MOV     R2,A

        MOV     A,R6            ; R3 = R3 + LO(B02)
        ADD     A,R3            
        MOV     R3,A
        
        MOV     A,B		; R2 = R2 + HI(B02)
        ADDC    A,R2
        MOV     R2,A
       
; *** B0 * B3

        MOV     A,@R1
	DEC	R1
	DEC	R1
	DEC	R1
        MOV     B,@R1
        MUL     AB              ; B0 * B3
        
        RLC     A               ; mult by 2
        
        ADD     A,R2            ; R2 = R2 + LO(B03)
        MOV     R2,A                   

;*** B1 * B2

	INC	R1
	INC	R1
        MOV     A,@R1
	DEC	R1
        MOV     B,@R1
	DEC	R1
        MUL     AB              ; B1 * B2
        
        RLC     A               ; mult by 2
        
        ADD     A,R2            ; R2 = R2 + LO(B03)
        MOV     R2,A                   
        
; ***** END of multipliation ******

; ***** Multiply the result by 2 **

	CLR     C	        ; Clear Carry
        MOV     A,R5            ; Rotate R5
        RLC     A
        MOV     R5,A
        MOV     A,R4            ; Rotate R4
        RLC     A
        MOV     R4,A
        MOV     A,R3            ; Rotate R3
        RLC     A
        MOV     R3,A
        MOV     A,R2            ; Rotate R2
        RLC     A
        MOV     R2,A                

        CALL    ADD4            ; (R2|R3|R4|R5) + @R1 
        
; ***** Rotate left of 5 *********************************************

        MOV     A,R5            ; A = R5
        
        MOV     C,ACC.0         ; Lowest bit stored in the Carry
	MOV     A,R2            ; Rotate R2
        RRC     A
        MOV     R2,A
        MOV     A,R3            ; Rotate R3
        RRC     A
        MOV     R3,A
        MOV     A,R4            ; Rotate R4
        RRC     A
        MOV     R4,A
        MOV     A,R5            ; Rotate R5
        RRC     A
        MOV     R5,A                

        MOV     C,ACC.0         ; Lowest bit stored in the Carry
	MOV     A,R2            ; Rotate R2
        RRC     A
        MOV     R2,A
        MOV     A,R3            ; Rotate R3
        RRC     A
        MOV     R3,A
        MOV     A,R4            ; Rotate R4
        RRC     A
        MOV     R4,A
        MOV     A,R5            ; Rotate R5
        RRC     A
        MOV     R5,A                

        MOV     C,ACC.0         ; Lowest bit stored in the Carry
	MOV     A,R2            ; Rotate R2
        RRC     A
        MOV     R2,A
        MOV     A,R3            ; Rotate R3
        RRC     A
        MOV     R3,A
        MOV     A,R4            ; Rotate R4
        RRC     A
        MOV     R4,A
        MOV     A,R5            ; Rotate R5
        RRC     A

        XCH     A,R4            ; A = R4, R4=R5
        XCH     A,R3            ; A = R3, R3=R4
        XCH     A,R2            ; A = R2, R2=R3
        MOV     R5,A            ; R5 = A = R2

        CALL    STORE           ; Store registers in @R0 
      
        RET
                   
; ********************************************************************
; Encrypt        
; SKey in SKEY
; Plaintext in A|B|C|D
; Ciphertext output in A|B|C|D
; Number of rounds fixed to 20 rounds
; A,B,R0,R1,R2,R3,R4,R5,R6,R7 and all flags are destroyed 
; ********************************************************************

ECRYPT: MOV     R0,#BTEXT-3     ; @R0 = BTEXT
        MOV     R1,#SKEY-175    ; @R1 = S[0]
        CALL    LOAD            ; Registers = @R0
        CALL    ADD4            ; @R0 = @R0 + @R1
        CALL    STORE           ; @R0 = Registers
        
        MOV     R0,#DTEXT-3     ; @R0 = DTEXT
        MOV     R1,#SKEY-175+4  ; @R1 = S[1]
        CALL    LOAD            ; Registers = @R0
        CALL    ADD4            ; @R0 = @R0 + @R1
        CALL    STORE           ; @R0 = Registers

	MOV	A,R1		; Save S[i] address
	MOV	SADDR,A        
        MOV     R7,#1H                 

	MOV	R0,#BTEXT-3	; Initialize B address
				; next iteration B = C no need
				; to recompute B address

	MOV	ADDR,#ATEXT-3	; Init address vector
        MOV	ADDR-1,#BTEXT-3
	MOV	ADDR-2,#CTEXT-3
	MOV	ADDR-3,#DTEXT-3

ECT1:   MOV	A,R0
        MOV     R1,A
        MOV     R0,#TTEXT-3

        CALL    ENC1            ; T = ENC1(B)
                
        MOV     R1,ADDR-3	; Load DTEXT address
        MOV     R0,#UTEXT-3

        CALL    ENC1            ; U = ENC1(D)
        
	MOV	R0,ADDR		; Load ATEXT address

	CALL    LOAD            ; LOAD @R0 in the registers       

        MOV     R1,#TTEXT-3     ; @R1 = T

        CALL    XOR4            ; Rx = Rx XOR @R1
             
        MOV     R1,#UTEXT        ; Compute the rotate value (last byte needed)
        MOV     A,@R1
        ANL     A,#00011111B
        MOV     R6,A            ; R6 = u

        CALL    ROTL            ; Rx <<< R6
        
        MOV     A,SADDR         ; Compute S[2i] address
        ADD     A,#4
	MOV	SADDR,A   
        MOV     R1,A            ; @R1 = S[2i]

        CALL    ADD4            ; @Rx = @Rx + @R1
        
        CALL    STORE           ; Store registers in @R0

        MOV     R0,ADDR-2       ; Load CTEXT address 

	CALL    LOAD            ; LOAD @R0 in the registers       

        MOV     R1,#UTEXT-3     ; @R1 = U

        CALL    XOR4            ; Rx = Rx XOR @R1

        MOV     R1,#TTEXT        ; Compute the rotate value (last byte needed)
        MOV     A,@R1
        ANL     A,#00011111B
        MOV     R6,A            ; R6 = t

        CALL    ROTL            ; Rx <<< R6
                                   
        MOV     A,SADDR		
        ADD     A,#4H
	MOV	SADDR,A
        MOV     R1,A            ; Computes S[2i+1]
        
        CALL    ADD4            ; @Rx = @Rx + @R1
        
        CALL    STORE           ; Store registers in @R0

	MOV	A,ADDR-3	; Swap the value of the address
	XCH	A,ADDR-2
	XCH	A,ADDR-1
	XCH	A,ADDR
	MOV	ADDR-3,A

        INC     R7
        CJNE    R7,#21D,ECT1
        
        MOV     R0,#ATEXT-3     ; @R0 = BTEXT
        MOV     R1,#SKEY-3-4    ; @R1 = S[43]
        CALL    LOAD            ; Registers = @R0
        CALL    ADD4            ; @R0 = @R0 + @R1
        CALL    STORE           ; @R0 = Registers
        
        MOV     R0,#CTEXT-3     ; @R0 = CTEXT
        MOV     R1,#SKEY-3      ; @R1 = S[1]
        CALL    LOAD            ; Registers = @R0
        CALL    ADD4            ; @R0 = @R0 + @R1
        CALL    STORE           ; @R0 = Registers
                 
        RET

; ********************************************************************

TEST:   MOV     R1,#L-15
        MOV     @R1,#067H
        INC     R1
        MOV     @R1,#045H
        INC     R1
        MOV     @R1,#023H
        INC     R1
        MOV     @R1,#001H
        INC     R1
        MOV     @R1,#0EFH
        INC     R1
        MOV     @R1,#0CDH
        INC     R1
        MOV     @R1,#0ABH
        INC     R1
        MOV     @R1,#089H
        INC     R1
        MOV     @R1,#034H
        INC     R1
        MOV     @R1,#023H
        INC     R1
        MOV     @R1,#012H
        INC     R1
        MOV     @R1,#001H
        INC     R1
        MOV     @R1,#078H
        INC     R1
        MOV     @R1,#067H
        INC     R1
        MOV     @R1,#056H
        INC     R1
        MOV     @R1,#045H
        
;TE1:    MOV     @R1,#0H
;        INC     R1
;        CJNE    R1,#L+1,TE1
                                                             
        MOV     R7,#4H 
        CALL    SETKEY
        
        MOV     R1,#ATEXT-15
        MOV     @R1,#0F1H
        INC     R1
        MOV     @R1,#0E0H
        INC     R1
        MOV     @R1,#0DFH
        INC     R1
        MOV     @R1,#0CEH
        INC     R1
        MOV     @R1,#0BDH
        INC     R1
        MOV     @R1,#0ACH
        INC     R1
        MOV     @R1,#09BH
        INC     R1
        MOV     @R1,#08AH
        INC     R1
        MOV     @R1,#079H
        INC     R1
        MOV     @R1,#068H
        INC     R1
        MOV     @R1,#057H
        INC     R1
        MOV     @R1,#046H
        INC     R1
        MOV     @R1,#035H
        INC     R1
        MOV     @R1,#024H
        INC     R1
        MOV     @R1,#013H
        INC     R1
        MOV     @R1,#002H

;TE2:    MOV     @R1,#0H
;        INC     R1
;        CJNE    R1,#ATEXT+1,TE2
        
        CALL    ECRYPT
                
        END                                                  


