#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <visa.h>
#include <windows.h>
#include <winsock.h>

#define BUFLEN 1024*1024+65536
#define IO_INPUT_BUFLEN 65536

#define CMD_INVALID	0
#define CMD_WRITE	1
#define CMD_READ	2
#define CMD_INSTRUMENT	3

char	vi_buf[BUFLEN];

struct tcp_connection {
	int	tc_sock;
	int	tc_buf_len;
	char	tc_buf[IO_INPUT_BUFLEN];
};

void
buf_print(char *buf, int len)
{
	int	i;

	printf("buffer: [");
	for (i = 0; i < len; i++) {
		printf("%c", buf[i]);
	}
	printf("]\n");
}

char
*socket_getline(struct tcp_connection *tc)
{
	static char	outbuf[IO_INPUT_BUFLEN], tmpbuf[IO_INPUT_BUFLEN];
	int	i;

	while (1) {
		/* check to see if we already have one in our buffer*/
		for (i = 0; i < tc->tc_buf_len; i++) {
			if (tc->tc_buf[i] == '\n') {
				/* found a line, return it and slide the buffer back */
				/* printf("Processing data:\n");
				printf("Old ");
				buf_print(tc->tc_buf, tc->tc_buf_len); 
				if (tc->tc_buf_len) {
					printf("First character is %d\n", tc->tc_buf[0]);
				}
				*/
				memcpy(outbuf, &tc->tc_buf, i + 1);
				outbuf[i] = '\0';
				if (i && outbuf[i-1] == '\r') {
					/* printf("outbuf[0] = '\\0'\n"); */
					outbuf[i-1] = '\0';
				}
				/* printf("Line retrieved ");
				buf_print(outbuf, strlen(outbuf)); */
				tc->tc_buf_len -= i + 1;
				memcpy(tmpbuf, &tc->tc_buf[i+1], tc->tc_buf_len);
				memcpy(&tc->tc_buf, tmpbuf, tc->tc_buf_len);
				/* printf("Line remaning ");
				buf_print(tc->tc_buf, tc->tc_buf_len); */
				return(outbuf);
			}
		}

		// printf("socket_getline reading from socket\n");
		/* wasn't in existing buffer, read some more in */
		if (tc->tc_buf_len == IO_INPUT_BUFLEN) {
			fprintf(stderr, "socket_getline: buffer full\n");
			return(NULL);
		}
		i = recv(tc->tc_sock, tc->tc_buf + tc->tc_buf_len, IO_INPUT_BUFLEN - tc->tc_buf_len, 0);
		if (i == -1 || i == 0) {
			perror("socket_getline.recv");
			return(NULL);
		}
		tc->tc_buf_len += i;

		// printf("socket_getline read %d bytes\n", i);
	}
}


int
socket_senddata(struct tcp_connection *tc, char *data, int len)
{
	int	i, sofar=0;

	while (sofar < len) {
		i = send(tc->tc_sock, data + sofar, len - sofar, 0);
		if (i < 0) {
			i = WSAGetLastError();
			fprintf(stderr, "socket_sendline.send returned %d\n", i);
			return(-1);
		}
		sofar += i;
	}

	return(0);
}


int
socket_sendline(struct tcp_connection *tc, char *line)
{
	return socket_senddata(tc, line, strlen(line));
}


int
line_parse_command(char *line, char **data)
{
	char	*tmp;

	tmp = strchr(line, ' ');

	if (tmp) {
		if (data)
			*data = tmp + 1;
		*tmp = '\0';
	}
	
	if (!strcmp(line, "write"))
		return(CMD_WRITE);
	else if (!strcmp(line, "read"))
		return(CMD_READ);
	else if (!strcmp(line, "instrument"))
		return(CMD_INSTRUMENT);
	else
		return(CMD_INVALID);
}


void main(void)
{
	struct tcp_connection	tc;
	struct sockaddr_in	addr, conaddr;
	int	sock, con, i, len;
	char	buf[IO_INPUT_BUFLEN], errbuf[1024], *from_socket, *data;
	ViSession	defaultRM, vi;
	ViStatus	rc;
	int	viopened=0;
	WORD	wVersionRequested;
	WSADATA	wsdata;

	wVersionRequested = MAKEWORD(1,1);
	i = WSAStartup(wVersionRequested, &wsdata);
	if (i) {
		perror("WSAStartup");
		exit(-1);
	}

	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock == INVALID_SOCKET) {
		i = WSAGetLastError();
		printf("socket: i=%d\n", i);
		perror("socket");
		exit(-1);
	}

	addr.sin_family = AF_INET;
	addr.sin_port = htons(4500);
	addr.sin_addr.s_addr = INADDR_ANY;

	i = bind(sock, (struct sockaddr *)&addr, sizeof(addr));
	if (i == -1) {
		perror("bind");
		exit(-1);
	}

	i = listen(sock, 0);
	if (i == -1) {
		perror("listen");
		exit(-1);
	}

	
	while (1) {
		len = sizeof(conaddr);		
		con = accept(sock, (struct sockaddr *) &conaddr, &len);
		if (i == -1) {
			perror("accept");
			continue;
		}
		printf("connection from %s\n", inet_ntoa(conaddr.sin_addr));

		tc.tc_sock = con;
		tc.tc_buf_len = 0;

		viOpenDefaultRM(&defaultRM);
		
		while (from_socket = socket_getline(&tc)) {
			/* printf("Got line: [%s]\n", from_socket); */
			switch(line_parse_command(from_socket, &data)) {
			case CMD_INVALID:
				printf("invalid request: [%s]\n", from_socket);
				socket_sendline(&tc, "parser: error: invalid request\n");
				break;
			case CMD_WRITE:
				/* printf("write [%s]\n", data); */
				if (!viopened) {
					sprintf(buf, "write: error: instrument not open\n");
					printf(buf);
					socket_sendline(&tc, buf);
				} else {
					sprintf(vi_buf, "%s\n", data); /* reterminate, no snprintf? */
					if (((rc = viWrite(vi, vi_buf, strlen(vi_buf), &len))) != VI_SUCCESS) {
						viStatusDesc(vi, rc, errbuf);
						sprintf(buf, "write: error: viWrite returned %d (%s)\n", rc, errbuf);
						printf(buf);
						socket_sendline(&tc, buf);
			
					} else {
						if (len != strlen(vi_buf)) {
							sprintf(buf, "write: viWrite only wrote %d/%d bytes\n", len, strlen(vi_buf));
							printf(buf);
							socket_sendline(&tc, buf);
						} else {
							sprintf(buf, "write: ok\n");
							/* printf(buf); */
							socket_sendline(&tc, buf);
						}
					}
				}
				break;
			case CMD_READ:
				/* printf("read\n"); */
				if (!viopened) {
					sprintf(buf, "read: error: instrument not open\n");
					printf(buf);
					socket_sendline(&tc, buf);
				} else {
					if (((rc = viRead(vi, vi_buf, BUFLEN, &len))) != VI_SUCCESS) {
						viStatusDesc(vi, rc, errbuf);
						sprintf(buf, "read: error: viRead returned %d (%s)\n", rc, errbuf);
						printf(buf);
						socket_sendline(&tc, buf);
					} else {
						sprintf(buf, "read: ok: returned %d bytes\n", len);
						/* printf(buf); */
						socket_sendline(&tc, buf);
						socket_senddata(&tc, vi_buf, len);
					}
				}
				break;
			case CMD_INSTRUMENT:
				/* printf("instrument\n");*/
				if (viopened) {
					/* close existing instrument */
					viClose(vi);
				}
				if (((rc = viOpen(defaultRM, data, VI_NULL, VI_NULL, &vi))) != VI_SUCCESS) {
					viStatusDesc(vi, rc, errbuf);
					sprintf(buf, "instrument: error: viOpen returned %d (%s)\n", rc, errbuf);
					printf(buf);
					socket_sendline(&tc, buf);
					viopened = 0;
				} else {
					viSetAttribute(vi, VI_ATTR_TMO_VALUE, 20000); /* 20 second timeout on transfers */
					sprintf(buf, "instrument: ok\n");
					socket_sendline(&tc, buf);
					viopened = 1;
				}
				break;
			default:
				fprintf(stderr, "line_parse_command returned invalid value\n");
				exit(-1);
			}
		}
		if (viopened) {
			viClose(vi);
		}
		viClose(defaultRM);
		i = closesocket(con);
		if (i == -1) {
			perror("closesocket");
			/* exit(-1); */
		}
	}
}