#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "osc.h"


/*
 * libosc -- provide support for communications with oscilloscope via windows
 * gateway software.
 *
 * Build and maintain a TCP connection to the windows gateway, and provide
 * utility functions for sending requests, retrieving data, and processing
 * that data.
 */

#define OSC_LINE_MAX	1024

#define BUFLEN 65536

struct tcp_connection {
	int	tc_sock;
	int	tc_buf_len;
	char	tc_buf[BUFLEN];
};

static struct {
	struct tcp_connection	osc_tc;
} osc_data = { { -1 } };

/*
static void
buf_print(char *buf, int len)
{
	int	i;

	printf("buffer: [");
	for (i = 0; i < len; i++) {
		printf("%c", buf[i]);
	}
	printf("]\n");
}
*/

static char
*socket_getline(struct tcp_connection *tc)
{
	static char	outbuf[BUFLEN], tmpbuf[BUFLEN];
	int	i;

	while (1) {
		/* check to see if we already have one in our buffer*/
		for (i = 0; i < tc->tc_buf_len; i++) {
			if (tc->tc_buf[i] == '\n') {
				/* found a line, return it and slide the buffer back */
				/* printf("Processing data:\n");
				printf("Old ");
				buf_print(tc->tc_buf, tc->tc_buf_len); 
				if (tc->tc_buf_len) {
					printf("First character is %d\n", tc->tc_buf[0]);
				}
				*/
				memcpy(outbuf, &tc->tc_buf, i + 1);
				outbuf[i] = '\0';
				if (i && outbuf[i-1] == '\r') {
					/* printf("outbuf[0] = '\\0'\n"); */
					outbuf[i-1] = '\0';
				}
				/* printf("Line retrieved ");
				buf_print(outbuf, strlen(outbuf)); */
				tc->tc_buf_len -= i + 1;
				memcpy(tmpbuf, &tc->tc_buf[i+1], tc->tc_buf_len);
				memcpy(&tc->tc_buf, tmpbuf, tc->tc_buf_len);
				/* printf("Line remaning ");
				buf_print(tc->tc_buf, tc->tc_buf_len); */
				return(outbuf);
			}
		}

		// printf("socket_getline reading from socket\n");
		/* wasn't in existing buffer, read some more in */
		if (tc->tc_buf_len == BUFLEN) {
			fprintf(stderr, "socket_getline: buffer full\n");
			return(NULL);
		}
		i = recv(tc->tc_sock, tc->tc_buf + tc->tc_buf_len, BUFLEN - tc->tc_buf_len, 0);
		if (i == -1 || i == 0) {
			perror("socket_getline.recv");
			return(NULL);
		}
		tc->tc_buf_len += i;

		// printf("socket_getline read %d bytes\n", i);
	}
}


static int
socket_senddata(struct tcp_connection *tc, char *data, int len)
{
	int	i, sofar=0;

	while (sofar < len) {
		i = send(tc->tc_sock, data + sofar, len - sofar, 0);
		if (i < 0) {
			return(-1);
		}
		sofar += i;
	}

	return(0);
}

static int
socket_getdata(struct tcp_connection *tc, char *data, int len)
{
	int	i, sofar=0;

	while (sofar < len) {
		i = recv(tc->tc_sock, data+sofar, len-sofar, 0);
		if (i < 0) {
			return(-1);
		}
		sofar += i;
	}
	return(0);
}

static int
socket_sendline(struct tcp_connection *tc, char *line)
{
	return socket_senddata(tc, line, strlen(line));
}


/*
 * Connect to the specified hostname/port (host byte order) and return on
 * success, -1 on fail + errno with error.
 */
int
osc_connect(struct in_addr in_addr, short port, char *instr)
{
	struct sockaddr_in	addr;
	int	i;

	if (osc_data.osc_tc.tc_sock != -1) {
		/* already connected */
		/* XXX find appropriate errno */
		errno = EISCONN;
		return(-1);
	}

	i = socket(PF_INET, SOCK_STREAM, 0);
	if (i == -1) {
		return(-1);
	}

	osc_data.osc_tc.tc_sock = i;
	addr.sin_addr = in_addr;
	addr.sin_port = htons(port);
	addr.sin_family = AF_INET;
	/* XXX BSD would require this: addr.sin_len = sizeof(addr); */

	i = connect(osc_data.osc_tc.tc_sock, (struct sockaddr *) &addr,
	    sizeof(addr));
	if (i == -1) {
		osc_data.osc_tc.tc_sock = -1;
		return(-1);
	}

	i = osc_instrument(instr);
	if (i == -1) {
		osc_disconnect();
		return(-1);
	}

	return(0);
}


/*
 * Disconnect active connection.
 * 0 on success, -1 + errno on fail
 */
int
osc_disconnect(void)
{

	if (osc_data.osc_tc.tc_sock != -1) {
		/* already connected */
		/* XXX find appropriate errno */
		errno = ENOTCONN;
		return(-1);
	}

	close(osc_data.osc_tc.tc_sock);
	osc_data.osc_tc.tc_sock = -1;

	return(0);
}


static int
osc_read_results(char *msgbuf)
{
	char	line[OSC_LINE_MAX];
	char	*tmp, *tmp2;

	/* read in a line from the socket */
	tmp = socket_getline(&osc_data.osc_tc);
	if (!tmp) {
		return(-1);
	}
	strcpy(line, tmp);

	/* tmp -> result, tmp2 -> details (optional) */
	tmp = line;
	tmp2 = strsep(&tmp, ":");
	strcpy(msgbuf, tmp);

	while (tmp[0] == ' ')
		tmp++;

	if (strlen(tmp) < strlen("ok"))
		return(-1);

	if (tmp[0] == 'o' && tmp[1] == 'k')
		return(0);

	return(-1);
}


/*
 * Write out a null-terminated ascii string to the port
 * return 0 on success, -1 + errno on fail
 */
int
osc_write(char *line)
{
	char	retline[OSC_LINE_MAX];
	int	i;

	if (osc_data.osc_tc.tc_sock == -1) {
		errno = ENOTCONN;
		return(-1);
	}

	i = socket_sendline(&osc_data.osc_tc, "write ");
	if (i == -1)
		return(i);
	i = socket_sendline(&osc_data.osc_tc, line);
	if (i == -1) 
		return(i);
	i = socket_sendline(&osc_data.osc_tc, "\n");
	if (i == -1)
		return(i);

	/* read in results to make sure it went out ok */
	i = osc_read_results(retline);
	if (i)
		return(i);

	/* results string is in retline, but currently we don't care */

	return(0);
}


/*
 * Specify the instrument to use; return 0 on success, errno and -1 on
 * failure.
 */
int
osc_instrument(char *instrument)
{
	char	retline[OSC_LINE_MAX];
	int	i;

	if (osc_data.osc_tc.tc_sock == -1) {
		errno = ENOTCONN;
		return(-1);
	}

	i = socket_sendline(&osc_data.osc_tc, "instrument ");
	if (i == -1)
		return(i);
	i = socket_sendline(&osc_data.osc_tc, instrument);
	if (i == -1)
		return(i);
	i = socket_sendline(&osc_data.osc_tc, "\n");
	if (i == -1)
		return(i);

	/* read in results to make sure it went out ok */
	i = osc_read_results(retline);
	if (i)	
		return(i);

	/* results string is in retline, but currently we don't care */

	return(0);
}


/*
 * Read in to the provide buffer from the port; return positive int
 * on success, -1 + errno on fail.
 */
int
osc_read(char *buf, int buflen)
{
	char	line[OSC_LINE_MAX];
	char	*tmp, *tmp2;
	int	i, datalen;

	if (osc_data.osc_tc.tc_sock == -1) {
		errno = ENOTCONN;
		return(-1);
	}

	i = socket_sendline(&osc_data.osc_tc, "read\n");
	if (i)
		return(i);

	i = osc_read_results(line);
	if (i)
		return(i);

	
	/* parse out number of bytes from first line of response */
	tmp = line;
	tmp2 = strsep(&tmp, " ");
	if (!tmp) {
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	tmp2 = strsep(&tmp, " ");
	if (!tmp2) {
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	tmp2 = strsep(&tmp, " ");
	if (!tmp2) {
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	tmp2 = strsep(&tmp, " ");
	if (!tmp2) {
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	/* tmp2 now value from within spaces */
	datalen = atoi(tmp2);
	if (!datalen)	{
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	if (datalen > buflen) {
		/* XXX need an errno here */
		errno = EBADMSG;
		return(-1);
	}

	i = socket_getdata(&osc_data.osc_tc, buf, datalen);
	if (i == -1) {
		return(-1);
	}

	datalen--; /* remove trailing \n */

	return(datalen);
}


int
osc_waveform_parse(char *buf, int buflen, char **data)
{

	if (buflen < 2) 
		return(-1);

	if (buf[0] != '#')
		return(-1);

	if (buf[1] < '0' || buf[1] > '9')
		return(-1);

	if (buflen < 1 + buf[1] - '0')
		return(-1);

	*data = buf + 2 + buf[1] - '0';

	return(0);
}


int
osc_parse_preamble(char *buf, long *format,
		   long *type, long *points, long *count,
		   double *xincrement, double *xorigin, double *xreference,
		   double *yincrement, double *yorigin, double *yreference)
{
	int	i;

	i = sscanf(buf, "%ld, %ld, %ld, %ld, %lf, %lf, %lf, %lf, %lf, %lf", 
	    format, type, points, count, xincrement, xorigin, xreference,
	    yincrement, yorigin, yreference);

	if (i != 10)
		return(-1);

	return(0);
}


void
osc_data_to_point(double x, double xinc, double xorig, double xref,
		  double y, double yinc, double yorig, double yref,
		  double *newx, double *newy)
{
	if (newx)
		*newx = ((x - xref) * xinc) + xorig;
	if (newy)
		*newy = ((y - yref) * yinc) + yorig;
}



int
osc_waveform_to_text(char *buf, int buflen, unsigned char *data, int datalen,
		     double xinc, double xorig, double xref, double yinc,
		     double yorig, double yref)
{
	char	*tmp;
	int	i, len, tmplen;
	double	x, y;

	buf[0] = '\0';
	tmp = buf;
	len = 0;
	for (i = 0; i < datalen; i++) {

		osc_data_to_point(i, xinc, xorig, xref,
				  data[i], yinc, yorig, yref,
				  &x, &y);
		/* printf("%f, %f; ", 
		    x, y); */
		tmplen = snprintf(tmp, buflen - len, "%f, %f\n", x, y);
		if (tmp[0] == ',') {
			printf("bad line at %d\n", i);
		}
		if (tmplen >= buflen - len) {
			return(-1);
		}
		len += tmplen;
		tmp += tmplen;
	}

	return(0);
}



