(*  Title: 	Pure/SYNTAX/extension
    Author: 	Tobias Nipkow
    Copyright   1991  University of Cambridge

Syntax Definition
*)

signature EXTENSION0 =
sig
  val typeT: typ
  val constrainC: string
  val max_pri: int
end;

signature EXTENSION =
sig
  include EXTENSION0
  structure XGram : XGRAM
  datatype mfix = Mfix of string * typ * string * int list * int
  datatype ext = Ext of
	{roots: string list, mfix: mfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term list -> term)) list,
	 tmixfixs: string list}
  val ANY: string
  val ARG_LIST: string
  val appl_const': string
  val empty : string XGram.XGram
  val extend: string XGram.XGram -> ext -> string XGram.XGram
  val ID: string
  val idT: typ
  val TFREE: string
  val TVAR: string
  val tfreeT: typ
  val tvarT: typ
  val typ_to_nt: typ -> string
  val VAR: string
  val varT: typ
end;

functor ExtensionFun(XGram:XGRAM) : EXTENSION =
struct

structure XGram = XGram;
structure Symtab = XGram.Symtab;

local open XGram in

val max_pri = 1000; (* maximum legal priority *)

val ANY = "ANY";
val anyT = Type(ANY,[]);

val ANY1 = "ANY1";
val any1T = Type(ANY1,[]);

val funT = Type("fun",[]);

fun typ_to_nt (Type(c,_)) = c
  | typ_to_nt _ = ANY;

(* Terminal types *)

val ID = "ID";
val VAR = "VAR";
val TFREE = "TFREE";
val TVAR = "TVAR";

val idT    = Type(ID,[]);
val varT   = Type(VAR,[]);
val tfreeT = Type(TFREE,[]);
val tvarT = Type(TVAR,[]);
val terminalTs = [idT,varT,tfreeT,tvarT];

fun nonts syn = foldl (fn (i,"_") => i+1 | (i,_) => i) (0,explode syn);

val meta_chs = ["(",")","/","_"];

fun mk_term(pref,[]) = (pref,[])
  | mk_term(pref,"'"::c::cl) = mk_term(pref^c,cl)
  | mk_term(pref,l as c::cl) = if is_blank(c) orelse c mem meta_chs
	then (pref,l) else mk_term(pref^c,cl);

fun mk_space(sp,[]) = (sp,[]) |
    mk_space(sp,cl as c::cl') =
      if is_blank(c) then mk_space(sp^c,cl') else (sp,cl);

exception ARG_EXN;
exception BLOCK_EXN;

fun mk_syntax([],ar,_,b,sy) = if b=0 then (sy,ar) else raise BLOCK_EXN
  | mk_syntax("_"::cs,Type("fun",[ar,ar']),[],b,sy) =
	mk_syntax(cs,ar',[],b,sy@[Nonterminal(typ_to_nt ar,0)])
  | mk_syntax("_"::cs,Type("fun",[ar,ar']),p::pl,b,sy) =
	mk_syntax(cs,ar',pl,b,sy@[Nonterminal(typ_to_nt ar,p)])
  | mk_syntax("_"::cs,_,_,_,_) = raise ARG_EXN
  | mk_syntax("("::cs,ar,pl,b,sy) = let val (i,cs') = scan_int cs
	in mk_syntax(cs',ar,pl,b+1,sy@[Bg(i)]) end
  | mk_syntax(")"::cs,ar,pl,b,sy) =
	if b>0 then mk_syntax(cs,ar,pl,b-1,sy@[En]) else raise BLOCK_EXN
  | mk_syntax("/"::cs,ar,pl,b,sy) = let val (sp,cs') = take_prefix is_blank cs
	in mk_syntax(cs',ar,pl,b,sy@[Brk(length sp)]) end
  | mk_syntax(c::cs,ar,pl,b,sy) =
	let val (term,rest) =
	   if is_blank(c)
	   then let val (sp,cs') = mk_space(c,cs) in (Space(sp),cs') end
	   else let val (tk,cs') = mk_term("",c::cs) in(Terminal(tk),cs') end
	in mk_syntax(rest,ar,pl,b,sy@[term]) end;

fun pri_test1 p = if 0 <= p andalso p <= max_pri then ()
	else error("Priority out of range: " ^ string_of_int p)
fun pri_test(pl,p) = (pri_test1 p; seq pri_test1 pl);

fun mk_prod2(sy,T,opn,pl,p) =
let val (syn,T') = mk_syntax(explode sy, T, pl, 0, []) handle
	ARG_EXN =>
	error("More arguments in "^sy^" than in corresponding type") |
	BLOCK_EXN => error("Unbalanced block parantheses in "^sy);
    val nt = case T' of Type(c,_) => c | _ => ANY1;
in Prod(nt,syn,opn,p) end;

fun mk_prod1(sy,T,opn,pl,p) = (pri_test(pl,p); mk_prod2(sy,T,opn,pl,p));

datatype mfix = Mfix of string * typ * string * int list * int;

fun terminal1(T as Type("fun",_)) = hd(binder_types T) mem terminalTs
  | terminal1 _ = false;

fun mk_prod(Mfix(sy,T,"",pl,p)) = if nonts sy <> 1
	then error"Copy op must have exactly one argument" else
	if filter_out is_blank (explode sy) = ["_"] andalso
	   not(terminal1 T)
	then mk_prod2(sy,T,"",[copy_pri],copy_pri)
	else mk_prod1(sy,T,"",pl,p)
  | mk_prod(Mfix(sy,T,const,pl,p)) = mk_prod1(sy,T,const,pl,p)

(* The pure syntax *)

val appl_const' = "_F(...)";

val ARG_LIST = "ARG_LIST";
val argsT = Type(ARG_LIST,[]);

fun descend(from,to) = Mfix("_",to-->from,"",[0],0);

fun parents(T) = Mfix("(1'(_'))",T-->T,"",[0],max_pri);

fun mkappl(T) =
    Mfix("_(1'(_'))", [funT,argsT]--->T, appl_const', [max_pri,0],max_pri);

fun mkid(T) = Mfix("_",idT-->T,"",[],max_pri);
fun mkvar(T) = Mfix("_",varT-->T,"",[],max_pri);

val constrainC = "_constrain";

val typeT = Type("TYPE",[]);
fun constrain T =
	Mfix("_::_", [T,typeT]--->T, constrainC, [max_pri,0], max_pri-1);

val empty = XGram{Roots=[], TrTab1=Symtab.null, TrTab2=Symtab.null, Prods=[],
		  TMixFixs=[]};

datatype ext = Ext of
	{roots: string list, mfix: mfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term list -> term)) list,
	 tmixfixs: string list}

fun extend (XGram{Roots,TrTab1,TrTab2,Prods,TMixFixs})
           (Ext{roots,mfix,parse_translation,print_translation,tmixfixs}) =
let val Troots = map (apr(Type,[])) (roots \\ Roots);
    val Troots' = Troots \\ [typeT, propT, anyT];
    val mfix' = mfix @ map parents (Troots \ anyT) @ map mkappl Troots' @
                map mkid Troots' @ map mkvar Troots' @ map constrain Troots' @
                map (apl(anyT,descend)) (Troots \\ [typeT,anyT]) @
                map (apr(descend,any1T)) Troots'

in XGram{Roots = roots union Roots,
         TrTab1 = Symtab.balance(Symtab.st_of_alist(parse_translation,TrTab1))
	    handle Symtab.DUPLICATE(s) =>
	    error("More than one parse translation for "^s),
         TrTab2 = Symtab.balance(Symtab.st_of_alist(print_translation,TrTab2))
	    handle Symtab.DUPLICATE(s) =>
	    error("More than one print translation for "^s),
         Prods = Prods @ map mk_prod mfix',
	 TMixFixs = tmixfixs @ TMixFixs}
end;

end;

end;
