(*  Title: 	HOL/nat
    Author: 	Tobias Nipkow, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Definition of types ind and nat.

Type nat is defined as a set Nat over type ind.
*)

structure Nat =
struct

val sext = 
  Sext{mixfix=[Delimfix("0", "nat", "0"),
	       Infixl("<","[nat,nat] => bool",50)],
       print_translation=[], 
       parse_translation=[]}

val thy = extend_theory WF.thy "Nat"
([], [], [],
 [
  (["ind"], ([],"term")),
  (["nat"], ([],"term"))
 ],
 [
  (["Zero_Rep"],	"ind"),
  (["Suc_Rep"],		"ind => ind"),
  (["Nat"],		"ind set"),
  (["Rep_Nat"],		"nat => ind"),
  (["Abs_Nat"],		"ind => nat"),
  (["Suc"],		"nat => nat"),
  (["nat_case"],    	"[nat, 'a, nat=>'a] =>'a"),
  (["pred_nat"],	"(nat*nat) set"),
  (["nat_rec"],		"[nat, 'a, [nat, 'a]=>'a] => 'a")
 ],
 Some(sext))
 [  (*the axiom of infinity in 2 parts*)
  ("inj_Suc_Rep",  		"inj(Suc_Rep)"),
  ("Suc_Rep_not_Zero_Rep",	"~(Suc_Rep(x) = Zero_Rep)"),
  ("Nat_def",		"Nat == lfp(%X. {Zero_Rep} Un (Suc_Rep``X))"),
    (*faking a type definition...*)
  ("Rep_Nat", 		"Rep_Nat(n): Nat"),
  ("Rep_Nat_inverse", 	"Abs_Nat(Rep_Nat(n)) = n"),
  ("Abs_Nat_inverse", 	"i: Nat ==> Rep_Nat(Abs_Nat(i)) = i"),
    (*defining the abstract constants*)
  ("Zero_def",  	"0 == Abs_Nat(Zero_Rep)"),
  ("Suc_def",  		"Suc == (%n. Abs_Nat(Suc_Rep(Rep_Nat(n))))"),
     (*nat operations and recursion*)
  ("nat_case_def",	"nat_case == (%n a f. @z.  (n=0 --> z=a)  \
\                                          & (!x. n=Suc(x) --> z=f(x)))"),
  ("pred_nat_def", 	"pred_nat == {p. ? n. p = <n, Suc(n)>}" ),

  ("less_def", "m<n == <m,n>:trancl(pred_nat)"),

  ("nat_rec_def",	
   "nat_rec(n,c,d) == wfrec(pred_nat, n,   \
\                        %l g. nat_case(l, c, %m. d(m,g(m))))" )
 ]
end;

local val ax = get_axiom Nat.thy
in
val inj_Suc_Rep= ax"inj_Suc_Rep";
val Suc_Rep_not_Zero_Rep= ax"Suc_Rep_not_Zero_Rep";
val Nat_def  = ax"Nat_def";
val Rep_Nat= ax"Rep_Nat";
val Rep_Nat_inverse = ax"Rep_Nat_inverse";
val Abs_Nat_inverse = ax"Abs_Nat_inverse";
val Zero_def    = ax"Zero_def";
val Suc_def     = ax"Suc_def";
val nat_case_def = ax"nat_case_def";
val pred_nat_def = ax"pred_nat_def";
val less_def 	= ax "less_def";
val nat_rec_def = ax"nat_rec_def";
end;


goal Nat.thy "mono(%X. {Zero_Rep} Un (Suc_Rep``X))";
by (REPEAT (ares_tac [monoI, subset_refl, image_mono, Un_mono] 1));
val Nat_fun_mono = result();

val Nat_unfold = Nat_fun_mono RS (Nat_def RS Tarski_def_theorem);

(* Zero is a natural number -- this also justifies the type definition*)
goal Nat.thy "Zero_Rep: Nat";
by (rtac (Nat_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val Zero_RepI = result();

val prems = goal Nat.thy "i: Nat ==> Suc_Rep(i) : Nat";
by (rtac (Nat_unfold RS ssubst) 1);
by (rtac (imageI RS UnI2) 1);
by (resolve_tac prems 1);
val Suc_RepI = result();

(*** Induction ***)

val major::prems = goalw Nat.thy [Nat_def]
    "[| i: Nat;  P(Zero_Rep);   \
\       !!j. [| j: Nat; P(j) |] ==> P(Suc_Rep(j)) |]  ==> P(i)";
by (rtac (Nat_fun_mono RS (major RS general_induction)) 1);
by (fast_tac (set_cs addIs prems) 1);
val Nat_induct = result();

val prems = goalw Nat.thy [Zero_def,Suc_def]
    "[| P(0);   \
\       !!k. [| P(k) |] ==> P(Suc(k)) |]  ==> P(n)";
by (rtac (Rep_Nat_inverse RS subst) 1);   (*types force good instantiation*)
by (rtac (Rep_Nat RS Nat_induct) 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [Abs_Nat_inverse RS subst] 1));
val nat_induct = result();

(*Toby's preferred style of induction rule*)
val prems = goal Nat.thy "[| P(0);  (!n. P(n) --> P(Suc(n))) |] ==> (!n.P(n))";
by (cut_facts_tac prems 1);
by (rtac (nat_induct RS allI) 1);
by (assume_tac 1);
by (fast_tac set_cs 1);
val nat_ind = result();

(*Perform induction on n. *)
fun nat_ind_tac a i = 
    EVERY [res_inst_tac [("n",a)] nat_induct i,
	   rename_last_tac a ["1"] (i+1)];

(*A special form of induction for reasoning about m<n and m-n*)
val prems = goal Nat.thy
    "[| !!x. P(x,0);  \
\       !!y. P(0,Suc(y));  \
\       !!x y. [| P(x,y) |] ==> P(Suc(x),Suc(y))  \
\    |] ==> P(m,n)";
by (res_inst_tac [("x","m")] spec 1);
by (nat_ind_tac "n" 1);
by (rtac allI 2);
by (nat_ind_tac "x" 2);
by (REPEAT (ares_tac (prems@[allI]) 1 ORELSE etac spec 1));
val diff_induct = result();

(*** Isomorphisms: Abs_Nat and Rep_Nat ***)

(*We can't take these properties as axioms, or take Abs_Nat==Inv(Rep_Nat),
  since we assume the isomorphism equations will one day be given by Isabelle*)

goal Nat.thy "inj(Rep_Nat)";
by (rtac inj_inverseI 1);
by (rtac Rep_Nat_inverse 1);
val inj_Rep_Nat = result();

goal Nat.thy "inj_onto(Abs_Nat,Nat)";
by (rtac inj_onto_inverseI 1);
by (etac Abs_Nat_inverse 1);
val inj_onto_Abs_Nat = result();

(*** Distinctness of constructors ***)

goalw Nat.thy [Zero_def,Suc_def] "~ Suc(m)=0";
by (rtac (inj_onto_Abs_Nat RS inj_onto_contraD) 1);
by (rtac Suc_Rep_not_Zero_Rep 1);
by (REPEAT (resolve_tac [Rep_Nat, Suc_RepI, Zero_RepI] 1));
val Suc_not_Zero = result();

val Zero_not_Suc = standard (Suc_not_Zero RS neg_sym);

val Suc_neq_Zero = standard (Suc_not_Zero RS notE);
val Zero_neq_Suc = sym RS Suc_neq_Zero;

(** Injectiveness of Suc **)

goalw Nat.thy [Suc_def] "inj(Suc)";
by (rtac injI 1);
by (dtac (inj_onto_Abs_Nat RS inj_ontoD) 1);
by (REPEAT (resolve_tac [Rep_Nat, Suc_RepI] 1));
by (dtac (inj_Suc_Rep RS injD) 1);
by (etac (inj_Rep_Nat RS injD) 1);
val inj_Suc = result();

val Suc_inject = inj_Suc RS injD;;

goal Nat.thy "(Suc(m)=Suc(n)) = (m=n)";
by (EVERY1 [rtac iffI, etac Suc_inject, etac arg_cong]); 
val Suc_Suc_eq = result();

goal Nat.thy "~ n=Suc(n)";
by (nat_ind_tac "n" 1);
by (SIMP_TAC (HOL_ss addrews [Zero_not_Suc]) 1);
by (ASM_SIMP_TAC (HOL_ss addrews [Suc_Suc_eq]) 1);
val n_not_Suc_n = result();

(*** nat_case -- the selection operator for nat ***)

goalw Nat.thy [nat_case_def] "nat_case(0, a, f) = a";
by (fast_tac (set_cs addIs [select_equality] addEs [Zero_neq_Suc]) 1);
val nat_case_0_conv = result();

goalw Nat.thy [nat_case_def] "nat_case(Suc(k), a, f) = f(k)";
by (fast_tac (set_cs addIs [select_equality] 
	               addEs [make_elim Suc_inject, Suc_neq_Zero]) 1);
val nat_case_Suc_conv = result();

(** Introduction rules for 'pred_nat' **)

goalw Nat.thy [pred_nat_def] "<n, Suc(n)> : pred_nat";
by (fast_tac set_cs 1);
val pred_natI = result();

val major::prems = goalw Nat.thy [pred_nat_def]
    "[| p : pred_nat;  !!x n. [| p = <n, Suc(n)> |] ==> R \
\    |] ==> R";
by (rtac (major RS CollectE) 1);
by (REPEAT (eresolve_tac ([asm_rl,exE]@prems) 1));
val pred_natE = result();

goalw Nat.thy [wf_def] "wf(pred_nat)";
by (strip_tac 1);
by (nat_ind_tac "x" 1);
by (fast_tac (HOL_cs addSEs [mp, pred_natE, Pair_inject, 
			     make_elim Suc_inject]) 2);
by (fast_tac (HOL_cs addSEs [mp, pred_natE, Pair_inject, Zero_neq_Suc]) 1);
val wf_pred_nat = result();


(*** nat_rec -- by wf recursion on pred_nat ***)

val nat_rec_unfold = standard (wf_pred_nat RS (nat_rec_def RS wfrec_def_conv));

(** conversion rules **)

goal Nat.thy "nat_rec(0,c,h) = c";
by (rtac (nat_rec_unfold RS trans) 1);
by (rtac nat_case_0_conv 1);
val nat_rec_0_conv = result();

val prems = goal Nat.thy "nat_rec(Suc(n), c, h) = h(n, nat_rec(n,c,h))";
by (rtac (nat_rec_unfold RS trans) 1);
by (rtac (nat_case_Suc_conv RS trans) 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Nat.thy [("h", "[nat, ?'a]=> ?'a")])
      addrews [pred_natI, cut_apply]) 1);
val nat_rec_Suc_conv = result();

(*These 2 rules ease the use of primitive recursion.  NOTE USE OF == *)
val rew::prems = goal Nat.thy
    "[| !!n. f(n) == nat_rec(n,c,h) |] ==> f(0) = c";
by (rewtac rew);
by (rtac nat_rec_0_conv 1);
val nat_rec_def_0_conv = result();

val rew::prems = goal Nat.thy
    "[| !!n. f(n) == nat_rec(n,c,h) |] ==> f(Suc(n)) = h(n,f(n))";
by (rewtac rew);
by (rtac nat_rec_Suc_conv 1);
val nat_rec_def_Suc_conv = result();


(*** Basic properties of "less than" ***)

(** Introduction properties **)

val prems = goalw Nat.thy [less_def] "[| i<j;  j<k |] ==> i<k";
by (rtac (trans_trancl RS transD) 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
val less_trans = result();

goalw Nat.thy [less_def] "n < Suc(n)";
by (rtac (pred_natI RS r_into_trancl) 1);
val lessI = result();

(* i<j ==> i<Suc(j) *)
val less_SucI = lessI RSN (2, less_trans);

goal Nat.thy "0 < Suc(n)";
by (nat_ind_tac "n" 1);
by (rtac lessI 1);
by (etac less_trans 1);
by (rtac lessI 1);
val zero_less_Suc = result();

(** Elimination properties **)

goalw Nat.thy [less_def] "n<m --> ~ m<n";
by (rtac (wf_pred_nat RS wf_trancl RS wf_anti_sym RS notI RS impI) 1);
by (assume_tac 1);
by (assume_tac 1);
val less_not_sym = result();

(* [| n<m; m<n |] ==> R *)
val less_anti_sym = standard (less_not_sym RS mp RS notE);


goalw Nat.thy [less_def] "~ n<n";
by (rtac notI 1);
by (etac (wf_pred_nat RS wf_trancl RS wf_anti_refl) 1);
val less_not_refl = result();

(* n<n ==> R *)
val less_anti_refl = standard (less_not_refl RS notE);


val major::prems = goalw Nat.thy [less_def]
    "[| i<k;  k=Suc(i) ==> P;  !!j. [| i<j;  k=Suc(j) |] ==> P \
\    |] ==> P";
by (rtac (major RS tranclE) 1);
by (fast_tac (HOL_cs addSEs (prems@[pred_natE, Pair_inject])) 1);
by (fast_tac (HOL_cs addSEs (prems@[pred_natE, Pair_inject])) 1);
val lessE = result();

goal Nat.thy "~ n<0";
by (rtac notI 1);
by (etac lessE 1);
by (etac Zero_neq_Suc 1);
by (etac Zero_neq_Suc 1);
val not_less0 = result();

(* ~ n<0 *)
val less_zeroE = standard (not_less0 RS notE);

val [major,less,eq] = goal Nat.thy
    "[| m < Suc(n);  m<n ==> P;  m=n ==> P |] ==> P";
by (rtac (major RS lessE) 1);
by (rtac eq 1);
by (fast_tac (HOL_cs addSEs [make_elim Suc_inject]) 1);
by (rtac less 1);
by (fast_tac (HOL_cs addSEs [make_elim Suc_inject]) 1);
val less_SucE = result();

goal Nat.thy "(m < Suc(n)) = (m < n | m = n)";
by (fast_tac (HOL_cs addSIs [lessI]
		     addEs  [less_trans, less_SucE]) 1);
val less_Suc_eq = result();


(** Inductive properties **)

val prems = goal Nat.thy "Suc(m) < n ==> m<n";
by (subgoal_tac "Suc(m) < n --> m < n" 1);
by (fast_tac (HOL_cs addIs prems) 1);
by (nat_ind_tac "n" 1);
by (rtac impI 1);
by (etac less_zeroE 1);
by (fast_tac (HOL_cs addSIs [lessI RS less_SucI]
	 	     addSEs [make_elim Suc_inject]
		     addEs  [less_trans, lessE]) 1);
val Suc_lessD = result();

val [major] = goal Nat.thy "Suc(m) < Suc(n) ==> m<n";
by (rtac (major RS lessE) 1);
by (REPEAT (rtac lessI 1
     ORELSE eresolve_tac [make_elim Suc_inject, ssubst, Suc_lessD] 1));
val Suc_less_SucD = result();

val prems = goal Nat.thy "m<n ==> Suc(m) < Suc(n)";
by (subgoal_tac "m<n --> Suc(m) < Suc(n)" 1);
by (fast_tac (HOL_cs addIs prems) 1);
by (nat_ind_tac "n" 1);
by (rtac impI 1);
by (etac less_zeroE 1);
by (fast_tac (HOL_cs addSIs [lessI]
	 	     addSEs [make_elim Suc_inject]
		     addEs  [less_trans, lessE]) 1);
val Suc_mono = result();

goal Nat.thy "(Suc(m) < Suc(n)) = (m<n)";
by (EVERY1 [rtac iffI, etac Suc_less_SucD, etac Suc_mono]);
val Suc_less_eq = result();

val [major] = goal Nat.thy "Suc(n)<n ==> P";
by (rtac (major RS Suc_lessD RS less_anti_refl) 1);
val not_Suc_n_less_n = result();

(*"Less than" is a linear ordering*)
goal Nat.thy "m<n | m=n | n<m";
by (nat_ind_tac "m" 1);
by (nat_ind_tac "n" 1);
by (rtac (refl RS disjI1 RS disjI2) 1);
by (rtac (zero_less_Suc RS disjI1) 1);
by (fast_tac (HOL_cs addIs [lessI, Suc_mono, less_SucI] addEs [lessE]) 1);
val less_linear = result();

(*Can be used with less_Suc_eq to get n=m | n<m *)
goal Nat.thy "(~ m < n) = (n < Suc(m))";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC (HOL_ss addrews [not_less0,zero_less_Suc]) 1);
by (SIMP_TAC (HOL_ss addrews [not_less0,zero_less_Suc,Suc_less_eq]) 1);
by (ASM_SIMP_TAC (HOL_ss addrews [Suc_less_eq]) 1);
val not_less_eq = result();

val prems = goalw Nat.thy [less_def]
    "[| !!n. [| ! m. m<n --> P(m) |] ==> P(n) |]  ==>  P(n)";
by (wf_ind_tac "n" [wf_pred_nat RS wf_trancl] 1);
by (eresolve_tac prems 1);
val less_induct = result();

