(*  Title: 	ZF/trancl
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Transitive closure of a relation
*)

structure Trancl =
struct
val const_decs = 
 [ 
  (["rtrancl","trancl"],	"i=>i")
 ];

val thy = extend_theory Fixedpt.thy "trancl"
     ([], [], [], const_decs, None)
 [ 
  ("rtrancl_def",  
 "rtrancl(r) == lfp(lam s: Pow(field(r)*field(r)). id(field(r)) Un (r O s))"),

  ("trancl_def", "trancl(r) == r O rtrancl(r)")
 ];

end;

local val ax = get_axiom Trancl.thy
in
val trancl_def = ax"trancl_def";
val rtrancl_def = ax"rtrancl_def";
end;

goal Trancl.thy 
    "(lam s: Pow(field(r)*field(r)). id(field(r)) Un (r O s)) \
\      : mono(Pow(field(r)*field(r)), Pow(field(r)*field(r)))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, comp_mono, Un_mono, SigmaI, subsetI] 1 
     ORELSE eresolve_tac [fieldI2, Pair_inject, UnE, compE, idE,
			  ssubst, subsetD RS SigmaE] 1));
val rtrancl_fun_mono = result();


goalw Trancl.thy [rtrancl_def]
    "rtrancl(r) = id(field(r)) Un (r O rtrancl(r))";
by (rtac (rtrancl_fun_mono RS lam_Tarski_theorem) 1);
val rtrancl_unfold = result();

(** The relation rtrancl **)

goalw Trancl.thy [rtrancl_def] "rtrancl(r) <= field(r)*field(r)";
by (rtac (rtrancl_fun_mono RS mono_type RS lfix_type RS PowD) 1);
val rtrancl_type = result();

(*Reflexivity of rtrancl*)
val [prem] = goal Trancl.thy "[| a: field(r) |] ==> <a,a> : rtrancl(r)";
by (resolve_tac [rtrancl_unfold RS ssubst] 1);
by (rtac (prem RS idI RS UnI1) 1);
val rtrancl_refl = result();

(*Closure under composition with r*)
val prems = goal Trancl.thy
    "[| <a,b> : rtrancl(r);  <b,c> : r |] ==> <a,c> : rtrancl(r)";
by (resolve_tac [rtrancl_unfold RS ssubst] 1);
by (rtac (compI RS UnI2) 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
val rtrancl_I2 = result();

(*rtrancl of r contains r*)
val prems = goal Trancl.thy "[| <a,b> : r |] ==> <a,b> : rtrancl(r)";
by (resolve_tac [rtrancl_refl RS rtrancl_I2] 1);
by (REPEAT (resolve_tac (prems@[fieldI1]) 1));
val r_sub_rtrancl = result();


(** standard induction rule **)

val major::prems = goalw Trancl.thy [rtrancl_def]
  "[| <a,b> : rtrancl(r); \
\     !!x. x: field(r) ==> P(<x,x>); \
\     !!x y z.[| P(<x,y>); <x,y>: rtrancl(r); <y,z>: r |]  ==>  P(<x,z>) |] \
\  ==>  P(<a,b>)";
by (rtac (major RS lam_general_induction) 1);
(*try fast_tac!!*)
by (REPEAT (ares_tac [rtrancl_fun_mono] 1
     ORELSE eresolve_tac (prems@[CollectE,idE,UnE,compE,ssubst]) 1));
val rtrancl_full_induct = result();

(*nice induction rule*)
val major::prems = goal Trancl.thy
    "[| <a,b> : rtrancl(r);    \
\       P(a); \
\	!!y z.[| <a,y> : rtrancl(r);  <y,z> : r;  P(y) |] ==> P(z) |]  \
\     ==> P(b)";
(*by induction on this formula*)
by (subgoal_tac "ALL y. <a,b> = <a,y> --> P(y)" 1);
(*now solve first subgoal: this formula is sufficient*)
by (EVERY1 [etac (spec RS mp), rtac refl]);
(*now do the induction*)
by (resolve_tac [major RS rtrancl_full_induct] 1);
by (ALLGOALS (fast_tac (ZF_cs addIs prems addSEs [Pair_inject])));
val rtrancl_induct = result();

(*transitivity of transitive closure!! -- by induction.*)
goalw Trancl.thy [trans_def] "trans(rtrancl(r))";
by (REPEAT (resolve_tac [allI,impI] 1 ORELSE etac conjE 1));
by (res_inst_tac [("b","z")] rtrancl_induct 1);
by (DEPTH_SOLVE (eresolve_tac [asm_rl, rtrancl_I2] 1));
val trans_rtrancl = result();


(*elimination of rtrancl -- by induction on a special formula*)
val major::prems = goal Trancl.thy
    "[| <a,b> : rtrancl(r);  (a = b) ==> P; \
\	!!y.[| <a,y> : rtrancl(r); <y,b> : r |] ==> P |] \
\    ==> P";
by (subgoal_tac "a = b  | (EX y. <a,y> : rtrancl(r) & <y,b> : r)" 1);
(*see HOL/trancl*)
by (res_inst_tac [("b","b")] rtrancl_induct 2);
by (REPEAT (eresolve_tac ([exE,disjE,conjE]@prems) 1 
     ORELSE ares_tac [major, refl RS disjI1] 1));
by (REPEAT (ares_tac [conjI,exI,disjI2] 1));
val rtranclE = result();


(**** The relation trancl ****)

(** Conversions between trancl and rtrancl **)

val [major] = goalw Trancl.thy [trancl_def]
    "[| <a,b> : trancl(r) |] ==> <a,b> : rtrancl(r)";
by (resolve_tac [major RS compEpair] 1);
by (REPEAT (ares_tac [rtrancl_I2] 1));
val trancl_sub_rtrancl = result();

(*trancl(r) contains r*)
val [prem] = goalw Trancl.thy [trancl_def]
   "[| <a,b> : r |] ==> <a,b> : trancl(r)";
by (REPEAT (ares_tac [prem,compI,rtrancl_refl,fieldI1] 1));
val r_sub_trancl = result();

(*intro rule by definition: from rtrancl and r*)
val prems = goalw Trancl.thy [trancl_def]
    "[| <a,b> : rtrancl(r);  <b,c> : r |]   ==>  <a,c> : trancl(r)";
by (REPEAT (resolve_tac ([compI]@prems) 1));
val trancl_I1 = result();

(*intro rule from r and rtrancl*)
val prems = goal Trancl.thy
    "[| <a,b> : r;  <b,c> : rtrancl(r) |]   ==>  <a,c> : trancl(r)";
by (resolve_tac (prems RL [rtranclE]) 1);
by (etac subst 1);
by (resolve_tac (prems RL [r_sub_trancl]) 1);
by (rtac (trans_rtrancl RS transD RS trancl_I1) 1);
by (REPEAT (ares_tac (prems@[r_sub_rtrancl]) 1));
val trancl_I12 = result();

(*elimination of trancl(r) -- NOT an induction rule*)
val major::prems = goal Trancl.thy
    "[| <a,b> : trancl(r);  \
\       <a,b> : r ==> P; \
\	!!y.[| <a,y> : trancl(r); <y,b> : r |] ==> P  \
\    |] ==> P";
by (subgoal_tac "<a,b> : r | (EX y. <a,y> : trancl(r)  &  <y,b> : r)" 1);
by (REPEAT (eresolve_tac ([disjE,exE,conjE]@prems) 1 
	ORELSE  ares_tac [major] 1));  (**DELETE??*)
by (rtac (rewrite_rule [trancl_def] major RS compEpair) 1);
by (etac rtranclE 1);
by (REPEAT (eresolve_tac [trancl_I1,ssubst,disjI1] 1
       ORELSE ares_tac [conjI RS exI RS disjI2] 1));
val tranclE = result();

(*Transitivity of trancl(r).
  Proved by unfolding since it uses transitivity of rtrancl. *)
goalw Trancl.thy [trans_def,trancl_def] "trans(trancl(r))";
by (REPEAT (resolve_tac [allI,impI] 1
     ORELSE eresolve_tac [conjE,compEpair] 1));
by (rtac (rtrancl_I2 RS (trans_rtrancl RS transD RS compI)) 1);
by (REPEAT (assume_tac 1));
val trans_trancl = result();

goalw Trancl.thy [trancl_def] "trancl(r) <= field(r)*field(r)";
by (fast_tac (ZF_cs addEs [compE, rtrancl_type RS subsetD RS SigmaE2]) 1);
val trancl_type = result();

(*transitive closure of a WF relation is WF!*)
val [prem] = goal Trancl.thy "wf(r) ==> wf(trancl(r))";
by (rtac (trancl_type RS field_rel_subset RS wfI2) 1);
by (rtac subsetI 1);
(*must retain the universal formula for later use!*)
by (rtac ballE 1 THEN assume_tac 1);
by (etac mp 1);
by (contr_tac 2);
by (eres_inst_tac [("a","x")] (prem RS wf_induct2) 1);
by (rtac subset_refl 1);
by (rtac (impI RS allI) 1);
by (etac tranclE 1);
by (etac (bspec RS mp) 1);
by (etac fieldI1 1);
by (fast_tac ZF_cs 1);
by (fast_tac ZF_cs 1);
val wf_trancl = result();
