(*  Title: 	ZF/ex/list
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Lists in Zermelo-Fraenkel Set Theory 

"length" is defined as the fixed point of a large expression that has to be
copied out twice more, and is doubly expanded in certain proofs.  The
alternative is to give this expression a name, such as lengthfun, but
tricky folding and unfolding would then be required.  *)

writeln"File ZF/ex/list";

structure List =
struct
val const_decs = 
 [ 
   (["list"],		"i=>i"),
   (["listcase"],	"[i, i, [i,i]=>i] => i"),
   (["length"],		"i => i"),
   (["listrec"],	"[i, i, [i,i,i]=>i] => i")
 ];

val thy = extend_theory Univ.thy "list"
     ([], [], [], const_decs, None)
 [ 
   ("list_def", "list(A) == lfp(lam r: Pow(univ(A)). {0} Un A*r)" ),

   ("listcase_def",	
    "listcase(l,c,d) == THE z. l=0 & z=c | (EX x y. l = <x,y> & z=d(x,y))" ),

   ("length_def",	
    "length(A) == lfp(lam f: Pow(list(A) * nat). \
\        {<0,0>} Un  \
\        (UNION p:f. {z : list(A)*nat. \
\                     EX x:A. EX xs:list(A). EX k:nat. \
\                        p=<xs,k> & z = <<x,xs>, succ(k)>}))" ),

   ("listrec_def",	
    "listrec(A,c,d) == ofrec(length(A), list(A), nat, \
\                           %l g. listcase(l, c, %x xs. d(x,xs,g`xs)))" )
 ];
end;

local val ax = get_axiom List.thy
in 
val list_def = ax"list_def";
val listcase_def = ax"listcase_def";
val length_def = ax"length_def";
val listrec_def = ax"listrec_def";
end;


(** Monotonicity and unfolding of the function **)

goal List.thy
    "(lam r: Pow(univ(A)). {0} Un A*r) : mono(Pow(univ(A)), Pow(univ(A)))";
br lam_mono_Powtype 1;
by (REPEAT (ares_tac [subset_refl, zero_in_univ, 
		      singleton_subsetI, univ_A_subset, 
		      Un_mono, product_mono, Un_least, univ_product_mono] 1));
val listfun_mono = result();

goalw List.thy [list_def] "list(A) = {0} Un (A * list(A))";
by (rtac (listfun_mono RS lam_Tarski_theorem) 1);
val list_unfold = result();

(** Type checking of 0 (as nil) and <-,-> (as cons) **)

goal List.thy "0 : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val list_nilI = result();

val prems = goal List.thy "[| a: A;  l: list(A) |] ==> <a,l> : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI RS UnI2]) 1));
val list_PairI = result();

(** Injectivity properties and induction **)

val [major] = goal List.thy "0 = <a,b> ==> P";
by (rtac (PairI1 RS (major RS sym RS equals0D)) 1);
val list_distinct = result();

(* There is no need for 'cons inject' since we have Pair_inject *)

(*Structural induction on lists*)
val prems = goalw List.thy [list_def]
    "[| l: list(A);  \
\       P(0);        \
\       !!x y. [| x: A;  y: list(A);  P(y) |] ==> P(<x,y>) \
\    |] ==> P(l)";
by (rtac lam_general_induction 1);
by (REPEAT (ares_tac (listfun_mono::prems) 1
     ORELSE eresolve_tac [UnE, singletonE, SigmaE, CollectE, ssubst] 1));
val list_induct = result();

val prems = goal List.thy
    "l: list(A) ==> l=0 | (EX x:A. EX xs: list(A). l=<x,xs>)";
brs (reslist(prems,1,list_induct)) 1;
by (DEPTH_SOLVE (ares_tac [refl, disjI1, disjI2, bexI] 1));
val list_cases = result();

(** listcase **)

goalw List.thy [listcase_def] "listcase(0,c,d) = c";
by (REPEAT (eresolve_tac [exE, conjE, disjE, list_distinct, ssubst] 1 
     ORELSE ares_tac [refl,the_equality,exI,disjI1,conjI] 1));
val listcase_nil_conv = result();

goalw List.thy [listcase_def] "listcase(<a,l>, c, d) = d(a,l)";
by (REPEAT (eresolve_tac [exE, conjE, disjE, Pair_inject,
			  sym RS list_distinct, ssubst] 1 
     ORELSE ares_tac [refl,the_equality,exI,disjI2,conjI] 1));
val listcase_Pair_conv = result();

val prems = goal List.thy
    "[| l: list(A);    \
\       c: C(0);       \
\       !!x y. [| x: A;  y: list(A) |] ==> d(x,y): C(<x,y>)  \
\    |] ==> listcase(l,c,d) : C(l)";
by (resolve_tac (reslist(prems,1,list_induct)) 1);
br (listcase_Pair_conv RS ssubst) 2;
br (listcase_nil_conv RS ssubst) 1;
by (REPEAT (ares_tac prems 1));
val listcase_type = result();


(**** The length function for lists, as an inductively defined predicate ****)

(*Collect is hard to prove monotonic -- needs "logical" reasoning:
        {z : list(A)*nat. EX x xs k. <xs,k>:f & z = <<x,xs>, succ(k)>}
Use UNION over all "recursive" cases (for monotonicity) and
finally Collect (for ease of type checking) 
*)

goal List.thy 
    "(lam f: Pow(list(A) * nat). {<0,0>} Un  \
\        (UNION p:f. {z : list(A)*nat. \
\                     EX x:A. EX xs:list(A). EX k:nat. \
\                        p=<xs,k> & z = <<x,xs>, succ(k)>})) \
\    : mono(Pow(list(A) * nat),  Pow(list(A) * nat))";
br lam_mono_Powtype 1;
by (REPEAT (ares_tac [subset_refl,list_nilI,nat_0_I,SigmaI,
		      UNION_mono,Un_mono,Un_least,UNION_least,
		      Collect_subset,singleton_subsetI] 1));
val lengthfun_mono = result();

goalw List.thy [length_def] "length(A) = {<0,0>} Un  \
\        (UNION p:length(A). {z : list(A)*nat. \
\                     EX x:A. EX xs:list(A). EX k:nat. \
\                        p=<xs,k> & z = <<x,xs>, succ(k)>})";
by (rtac (lengthfun_mono RS lam_Tarski_theorem) 1);
val length_unfold = result();

(*In fact, it is a function!*)
goalw List.thy [length_def] "length(A) : Pow(list(A) * nat)";
by (rtac (lengthfun_mono RS mono_type RS lfix_type) 1);
val length_reltype = result();

(** Introduction rules for 'length' **)

goal List.thy "<0,0> : length(A)";
by (rtac (length_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val length_nilI = result();

val prems = goal List.thy
    "[| a: A;  <l,m>: length(A) |] ==> <<a,l>, succ(m)> : length(A)";
by (cut_facts_tac prems 1);
by (rtac (length_unfold RS ssubst) 1);
by (rtac (length_reltype RS PowD RS subsetD RS SigmaE2) 1);
by (REPEAT (ares_tac [refl,bexI,conjI,SigmaI,UnI2,UNION_I,CollectI,
		      list_PairI,nat_succ_I] 1));
val length_PairI = result();

val prems = goal List.thy
    "l: list(A) ==> EX n: nat. <l,n> : length(A)";
by (resolve_tac (reslist(prems,1,list_induct)) 1);
by (REPEAT (etac bexE 1
     ORELSE ares_tac [bexI,length_nilI,nat_0_I,length_PairI,nat_succ_I] 1));
val length_total = result();

(** Elimination -- rule induction on length **)
val prems = goalw List.thy [length_def]
    "[| z: length(A);  \
\       P(<0,0>);        \
\       !!x xs k. [| x: A;  <xs,k>: length(A);  P(<xs,k>) \
\                |] ==> P(<<x,xs>, succ(k)>) \
\    |] ==> P(z)";
by (rtac lam_general_induction 1);
by (rtac lengthfun_mono 3);
by (resolve_tac prems 1);
by (Set.fast_tac prems 1);
val length_induct = result();


(** Proof that length is a function **)

val prems = goal List.thy
    "<0,j>: length(A) ==> j = 0";
by (resolve_tac (reslist(prems,1,setup_induction)) 1);
by (etac length_induct 1);
by (REPEAT (ares_tac [impI] 1
     ORELSE eresolve_tac [Pair_inject, list_distinct] 1));
val length_fun_nil = result();

(*If length(A) maps l to a unique m then it maps <a,l> uniquely to succ(m).*)
val prems = goal List.thy
    "[| <<a,l>,n>: length(A);           \
\       ALL j. <l,j>: length(A) --> j=m |] ==>  n = succ(m)";
by (cut_facts_tac prems 1);
by (etac setup_induction 1);
by (etac length_induct 1);
(*NOTE: the induction hypotheses are useless!*)
by (REPEAT (ares_tac [impI] 1
     ORELSE eresolve_tac [Pair_inject, sym RS list_distinct] 1));
by (REPEAT (ares_tac [refl] 1
     ORELSE eresolve_tac [spec RS impE, ssubst] 1));
val length_fun_cons = result();

val prems = goal List.thy
    "<l,i>: length(A) ==> (ALL j. <l,j>: length(A) --> j=i)";
by (resolve_tac (reslist(prems,1,setup_induction)) 1);
by (res_inst_tac [("x","i")] spec 1);
by (res_inst_tac [("x","l")] spec 1);
by (etac length_induct 1);
(*prepare the induction step*)
by (etac (spec RS spec RS impE) 2);
by (rtac refl 2);
by (REPEAT (eresolve_tac [Pair_inject,length_fun_nil,length_fun_cons,ssubst] 1
     ORELSE ares_tac [impI,allI] 1));
val length_fun_lemma = result();

goal List.thy
    "length(A) : list(A) -> nat";
by (REPEAT (eresolve_tac [length_total RS bexE, 
			  length_fun_lemma RS spec RS mp] 1
     ORELSE ares_tac [PiI, length_reltype RS PowD, ex1I] 1));
val length_funtype = result();

(** Conversion rules for the function "length" **)

goal List.thy
    "length(A)`0 = 0";
by (rtac (length_nilI RS apply_equality) 1);
by (rtac length_funtype 1);
val length_nil_conv = result();

val prems = goal List.thy
    "[| a: A;  l: list(A) |]  ==>  length(A) ` <a,l> = succ(length(A)`l)";
by (rtac (length_PairI RS apply_equality) 1);
by (REPEAT (resolve_tac (prems@[length_funtype,apply_Pair]) 1));
val length_Pair_conv = result();

(*justifies primitive recursion for lists*)
val prems = goal List.thy
    "[| a: A;  l: list(A) |]  ==>  length(A)`l : length(A) ` <a,l>";
br (length_Pair_conv RS ssubst) 1;
by (REPEAT (resolve_tac (prems@[succI1]) 1));
val length_Pair_mem = result();


(*** listrec -- by ordinal recursion on length ***)

(*Setting up the simplifier --
  was "d" instead of "h", but new treatment of var names caused a 
  clash with the "d" in subst_context3... *)
val h_cong = read_instantiate List.thy [("t","h")] subst_context3;

val list_ss = FOL_ss addcongs ([h_cong] @ ZF_congs)
      addrews ([ofrec_conv,listcase_nil_conv,listcase_Pair_conv,restrict_conv]@
	       [Ord_nat, length_funtype, list_PairI, CollectI, 
		length_Pair_mem]);

(** conversion rules **)

goalw List.thy [listrec_def] "listrec(A,c,h)`0 = c";
by (SIMP_TAC (list_ss addrews [list_nilI]) 1);
val listrec_nil_conv = result();

val prems = goalw List.thy [listrec_def]
    "[| a: A;  l: list(A) |] ==> \
\   listrec(A,c,h) ` <a,l> = h(a, l, listrec(A,c,h)`l)";
by (SIMP_TAC (list_ss addrews prems) 1);
val listrec_Pair_conv = result();

(*type checking*)
val prems = goalw List.thy [listrec_def]
     "[| l: list(A);    \
\        c: C(0);       \
\        !!x y r. [| x: A;  y: list(A);  r: C(y) |] ==> h(x,y,r): C(<x,y>)  \
\     |] ==> listrec(A,c,h)`l : C(l)";
by (EVERY1 [rtac ofrec_apply_type,
	    rtac Ord_nat,
	    rtac length_funtype,
	    resolve_tac prems,
	    etac (list_cases RS disjE)]);
by Set.safe_tac;
by (all_simp_tac list_ss (prems@[apply_type]));
val listrec_apply_type = result();
