(*  Title: 	ZF/perm
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

The theory underlying permutation groups
  -- Composition of relations, the identity relation, converse of a relation
  -- Injections, surjections, bijections
  -- Lemmas for the Schroeder-Bernstein Theorem
*)

writeln"File ZF/ex/perm";

val perm_mixfix =                    (*composition*)
    [ Infixr("O","[i,i]=>i",60) ];

val perm_sext = Sext{mixfix		= perm_mixfix,
                     parse_translation	= [],
                     print_translation	= []};

val perm_const_decs =
    [ (["id","converse"],	"i=>i"),
      (["inj","surj","bij"],	"[i,i]=>i") ];

val perm_thy = extend_theory set_thy "perm"
     ([], [], [], perm_const_decs, Some perm_sext)
  [ 
    ("comp_def",	(*composition of relations/functions*)
	"r O s == {xz : domain(s)*range(r) . \
\                  EX x y z. xz=<x,z> & <x,y>:s & <y,z>:r}"),
    ("id_def",		(*the identity function for A*)
	"id(A) == (lam x:A. x)"),
    ("converse_def",	(*the converse of relation r (inverse of function)*)
	"converse(r) == {yx. xy:r, EX x y. xy=<x,y> & yx=<y,x>}"),
    ("inj_def",		(*one-to-one functions from A to B*)
	"inj(A,B) == \
\            { f: A->B. ALL w:A. ALL x:A. f`w=f`x --> w=x}"),
    ("surj_def",	(*onto functions from A to B*)
	"surj(A,B) == { f: A->B . ALL y:B. EX x:A. f`x=y}"),
    ("bij_def",		(*one-to-one and onto functions*)
	"bij(A,B) == inj(A,B) Int surj(A,B)")
  ];

local val ax = get_axiom perm_thy in 
val comp_def = ax"comp_def";
val id_def = ax"id_def";
val converse_def = ax"converse_def";
val inj_def = ax"inj_def";
val surj_def = ax"surj_def";
val bij_def = ax"bij_def";
end;


(** Surjective function space **)

val prems = goalw perm_thy [surj_def] "f: surj(A,B) ==> f: A->B";
by (cut_facts_tac prems 1);
by (fast_tac ZF_cs 1);
val surj_is_fun = result();

val prems = goalw perm_thy [surj_def] "f : Pi(A,B) ==> f: surj(A,range(f))";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addIs [apply_equality] 
		    addEs [range_of_fun,domain_type]) 1);
val fun_is_surj = result();

val prems = goalw perm_thy [surj_def] "f: surj(A,B) ==> range(f)=B";
by (cut_facts_tac prems 1);
by (best_tac (ZF_cs addIs [equalityI,apply_Pair] addEs [range_type]) 1);
val surj_range = result();


(** Injective function space **)

val prems = goalw perm_thy [inj_def] "f: inj(A,B) ==> f: A->B";
by (cut_facts_tac prems 1);
by (fast_tac ZF_cs 1);
val inj_is_fun = result();

val prems = goalw perm_thy [inj_def]
    "[| <a,b>:f;  <c,b>:f;  f: inj(A,B) |] ==> a=c";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, Pair_mem_PiE, CollectE] 1));
by (fast_tac ZF_cs 1);
val inj_equality = result();

(** Bijections -- simple lemmas but no intro/elim rules -- use unfolding **)

val prems = goalw perm_thy [inj_def,bij_def] "f: bij(A,B) ==> f: A->B";
by (cut_facts_tac prems 1);
by (fast_tac ZF_cs 1);
val bij_is_fun = result();

(** Identity function **)

val [prem] = goalw perm_thy [id_def] "a:A ==> <a,a> : id(A)";  
by (rtac (prem RS lamI) 1);
val idI = result();

val major::prems = goalw perm_thy [id_def]
    "[| p: id(A);  !!x.[| x:A; p=<x,x> |] ==> P  \
\    |] ==>  P";  
by (rtac (major RS lamE) 1);
by (REPEAT (ares_tac prems 1));
val idE = result();

goalw perm_thy [id_def] "id(A) : A->A";  
by (rtac lam_type 1);
by (assume_tac 1);
val id_type = result();

goalw perm_thy [inj_def,id_def] "id(A): inj(A,A)";
by (REPEAT (ares_tac [CollectI,lam_type] 1));
by (SIMP_TAC ZF_ss 1);
val id_inj = result();

goalw perm_thy [id_def,surj_def] "id(A): surj(A,A)";
by (fast_tac (ZF_cs addIs [lam_type,beta_conv]) 1);
val id_surj = result();

goalw perm_thy [bij_def] "id(A): bij(A,A)";
by (fast_tac (ZF_cs addIs [id_inj,id_surj]) 1);
val id_bij = result();


(** Converse of a relation **)

val prems = goalw perm_thy [converse_def] "<a,b>:r ==> <b,a>:converse(r)";
by (fast_tac (ZF_cs addIs prems) 1);
val converseI = result();

val prems = goalw perm_thy [converse_def]
    "<a,b> : converse(r) ==> <b,a> : r";
by (cut_facts_tac prems 1);
by (fast_tac ZF_cs 1);
val converseD = result();

val [major,minor] = goalw perm_thy [converse_def]
    "[| yx : converse(r);  \
\       !!x y. [| yx=<y,x>;  <x,y>:r |] ==> P \
\    |] ==> P";
by (rtac (major RS ReplaceE) 1);
by (REPEAT (eresolve_tac [exE, conjE, minor] 1));
by (hyp_subst_tac 1);
by (assume_tac 1);
val converseE = result();

val converse_cs = ZF_cs addIs [converseI] addEs [converseE];

val prems = goal perm_thy "r<=s ==> converse(r) <= converse(s)";
by (cut_facts_tac prems 1);
by (fast_tac converse_cs 1);
val converse_mono = result();

val prems = goal perm_thy "r<=A*B ==> converse(converse(r)) = r";
by (cut_facts_tac prems 1);
by (fast_tac (converse_cs addIs [equalityI]) 1);
val converse_of_converse = result();

val prems = goal perm_thy "r<=A*B ==> converse(r)<=B*A";
by (cut_facts_tac prems 1);
by (fast_tac converse_cs 1);
val converse_type = result();

val [prem] = goal perm_thy "f: inj(A,B) ==> converse(f) : range(f)->A";
by (rtac (prem RS inj_is_fun RS PiE) 1);
by (rtac (converse_type RS PiI) 1);
by (fast_tac ZF_cs 1);
by (fast_tac (converse_cs addEs [prem RSN (3,inj_equality)]) 1);
by flexflex_tac;
val inj_converse_fun = result();

val prems = goalw perm_thy [surj_def]
    "f: inj(A,B) ==> converse(f): surj(range(f), A)";
by (fast_tac (ZF_cs addIs (prems@[inj_converse_fun,apply_Pair,apply_equality,
			 converseI,inj_is_fun])) 1);
val inj_converse_surj = result();

(*The premises are equivalent to saying that f is injective...*) 
val prems = goal perm_thy
    "[| f: A->B;  converse(f): C->A;  a: A |] ==> converse(f)`(f`a) = a";
by (fast_tac (ZF_cs addIs (prems@[apply_Pair,apply_equality,converseI])) 1);
val left_inverse_lemma = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  a: A |] ==> converse(f)`(f`a) = a";
by (fast_tac (ZF_cs addIs (prems@
       [left_inverse_lemma,inj_converse_fun,inj_is_fun])) 1);
val left_inverse = result();

val prems = goal perm_thy
    "[| f: A->B;  converse(f): C->A;  b: C |] ==> f`(converse(f)`b) = b";
by (rtac (apply_Pair RS (converseD RS apply_equality)) 1);
by (REPEAT (resolve_tac prems 1));
val right_inverse_lemma = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  b: range(f) |] ==> f`(converse(f)`b) = b";
by (rtac right_inverse_lemma 1);
by (REPEAT (resolve_tac (prems@ [inj_converse_fun,inj_is_fun]) 1));
val right_inverse = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> converse(f): inj(range(f), A)";
bw inj_def;  (*rewrite subgoal but not prems!!*)
by (cut_facts_tac prems 1);
by (safe_tac ZF_cs);
(*apply f to both sides and simplify using right_inverse
  -- could also use  etac[subst_context RS box_equals]  in this proof *)
by (rtac simp_equals 2);
by (REPEAT (eresolve_tac [inj_converse_fun, right_inverse RS sym, ssubst] 1
     ORELSE ares_tac [refl,rangeI] 1));
val inj_converse_inj = result();

val prems = goalw perm_thy [bij_def] "f: bij(A,B) ==> converse(f): bij(B,A)";
by (cut_facts_tac prems 1);
by (etac IntE 1);
by (eresolve_tac [(surj_range RS subst)] 1);
by (rtac IntI 1);
by (etac inj_converse_inj 1);
by (etac inj_converse_surj 1);
val bij_converse_bij = result();


(** Composition of two relations **)

val prems = goalw perm_thy [comp_def]
    "[| <a,b>:s; <b,c>:r |] ==> <a,c> : r O s";
by (fast_tac (ZF_cs addIs prems) 1);
val compI = result();

(*proof requires higher-level assumptions or a delaying of hyp_subst_tac*)
val prems = goalw perm_thy [comp_def]
    "[| xz : r O s;  \
\       !!x y z. [| xz=<x,z>;  <x,y>:s;  <y,z>:r |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [CollectE, exE, conjE] 1 ORELSE ares_tac prems 1));
val compE = result();

(*proof requires higher-level assumptions or a delaying of hyp_subst_tac*)
val prems = goal perm_thy
    "[| <a,c> : r O s;  \
\       !!y. [| <a,y>:s;  <y,c>:r |] ==> P \
\    |] ==> P";
by (rtac compE 1);
by (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [Pair_inject,ssubst] 1));
val compEpair = result();

val comp_cs = converse_cs addIs [compI,idI] addEs [compE,idE];

val prems = goal perm_thy
    "[| r'<=r; s'<=s |] ==> (r' O s') <= (r O s)";
by (cut_facts_tac prems 1);
by (fast_tac comp_cs 1);
val comp_mono = result();

(*a typing theorem: composition preserves relations*)
val prems = goal perm_thy "[| s<=A*B;  r<=B*C |] ==> (r O s) <= A*C";
by (cut_facts_tac prems 1);
by (fast_tac comp_cs 1);
val comp_type = result();

(*associative law for composition*)
goal perm_thy "(r O s) O t = r O (s O t)";
by (fast_tac (comp_cs addIs [equalityI]) 1);
val comp_assoc = result();

(*left identity of composition; provable inclusions are
        id(A) O r <= r       
  and   [| r<=A*B; B<=C |] ==> r <= id(C) O r *)
val prems = goal perm_thy
    "r<=A*B ==> id(B) O r = r";
by (cut_facts_tac prems 1);
by (fast_tac (comp_cs addIs [equalityI]) 1);
val left_comp_id = result();

(*right identity of composition; provable inclusions are
        r O id(A) <= r
  and   [| r<=A*B; A<=C |] ==> r <= r O id(C) *)
val prems = goal perm_thy
    "r<=A*B ==> r O id(A) = r";
by (cut_facts_tac prems 1);
by (fast_tac (comp_cs addIs [equalityI]) 1);
val right_comp_id = result();


(** Composition preserves functions, injections, and surjections **)

val prems = goal perm_thy
    "[| g: A->B;  f: B->C |] ==> (f O g) : A->C";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [PiI,comp_type,ex1I,compI] 1
     ORELSE (dtac apply_equality 1 THEN assume_tac 1)
     ORELSE hyp_subst_tac 1
     ORELSE eresolve_tac [fun_is_rel,apply_Pair,apply_type,compE,
			  Pair_inject] 1));
val comp_func = result();

val prems = goal perm_thy
    "[| g: A->B;  f: B->C;  a:A |] ==> (f O g)`a = f`(g`a)";
by (REPEAT (resolve_tac (prems@[comp_func,apply_equality,compI,
	                        apply_Pair,apply_type]) 1));
val comp_func_apply = result();

val prems = goalw perm_thy [inj_def]
    "[| g: inj(A,B);  f: inj(B,C) |] ==> (f O g) : inj(A,C)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [bspec RS bspec RS mp, box_equals] 1
     ORELSE step_tac (ZF_cs addSIs [comp_func,apply_type,comp_func_apply]) 1));
val comp_inj = result();

val prems = goalw perm_thy [surj_def]
    "[| g: surj(A,B);  f: surj(B,C) |] ==> (f O g) : surj(A,C)";
by (cut_facts_tac prems 1);
by (best_tac (ZF_cs addIs [comp_func,comp_func_apply]) 1);
val comp_surj = result();

val prems = goalw perm_thy [bij_def]
    "[| g: bij(A,B);  f: bij(B,C) |] ==> (f O g) : bij(A,C)";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addIs [comp_inj,comp_surj]) 1);
val comp_bij = result();


(** inverses of composition **)

(*left inverse of composition; one inclusion is
        f: A->B ==> id(A) <= converse(f) O f *)
val [prem] = goal perm_thy
    "f: inj(A,B) ==> converse(f) O f = id(A)";
val injfD = prem RSN (3,inj_equality);
by (cut_facts_tac [prem RS inj_is_fun] 1);
by (fast_tac (comp_cs addIs [equalityI,apply_Pair] 
		      addEs [domain_type, make_elim injfD]) 1);
val left_comp_inverse = result();

(*right inverse of composition; one inclusion is
        f: A->B ==> f O converse(f) <= id(B) *)
val [prem] = goalw perm_thy [surj_def]
    "f: surj(A,B) ==> f O converse(f) = id(B)";
val appfD = (prem RS CollectD1) RSN (3,apply_equality2);
by (cut_facts_tac [prem] 1);
by (best_tac (comp_cs addIs [equalityI,apply_Pair] 
		      addEs [domain_type, range_type, make_elim appfD]) 1);
val right_comp_inverse = result();

(*Injective case applies converse(f) to both sides then simplifies
    using left_inverse_lemma*)
val prems = goalw perm_thy [bij_def,inj_def,surj_def]
    "[| converse(f): B->A;  f: A->B |] ==> f : bij(A,B)";
val cf_cong = read_instantiate_sg (sign_of perm_thy)
                [("t","%x.converse(f)`x")]   subst_context;
by (fast_tac (ZF_cs addIs (prems@[left_inverse_lemma,right_inverse_lemma,
				 apply_type])
		    addEs [cf_cong RS box_equals]) 1);
val invertible_imp_bijective = result();

(** Unions of functions -- cf similar theorems on func.ML **)

goal perm_thy "converse(r Un s) = converse(r) Un converse(s)";
by (rtac equalityI 1);
by (DEPTH_SOLVE_1 (resolve_tac [Un_least,converse_mono,
				Un_upper1,Un_upper2] 2));
by (fast_tac converse_cs 1);
val converse_of_Un = result();

val prems = goalw perm_thy [surj_def]
    "[| f: surj(A,B);  g: surj(C,D);  A Int C = 0 |] ==> \
\    (f Un g) : surj(A Un C, B Un D)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [fun_disjoint_apply1, fun_disjoint_apply2] 1
     ORELSE ball_tac 1
     ORELSE (rtac trans 1 THEN atac 2)
     ORELSE step_tac (ZF_cs addIs [fun_disjoint_Un]) 1));
val surj_disjoint_Un = result();

(*A simple, high-level proof; the version for injections follows from it,
  using  f:inj(A,B)<->f:bij(A,range(f))  *)
val prems = goal perm_thy
    "[| f: bij(A,B);  g: bij(C,D);  A Int C = 0;  B Int D = 0 |] ==> \
\    (f Un g) : bij(A Un C, B Un D)";
by (rtac invertible_imp_bijective 1);
by (rtac (converse_of_Un RS ssubst) 1);
by (REPEAT (resolve_tac (prems@[fun_disjoint_Un, bij_is_fun, 
				bij_converse_bij]) 1));
val bij_disjoint_Un = result();


(** Restrictions as injections, surjections, and bijections *)

val prems = goalw perm_thy [surj_def]
    "[| f: Pi(A,B);  C<=A |] ==> restrict(f,C): surj(C, f``C)";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addIs [restrict_conv RS trans, restrict_type,
			   apply_Pair,apply_equality]) 1);
val restrict_surj = result();

val prems = goalw perm_thy [inj_def,bij_def]
    "[| f: inj(A,B);  C<=A |] ==> restrict(f,C): bij(C, f``C)";
by (cut_facts_tac prems 1);
by (safe_tac ZF_cs);
by (REPEAT (eresolve_tac [bspec RS bspec RS mp, subsetD,
                          box_equals, restrict_conv] 1
     ORELSE ares_tac [surj_is_fun,restrict_surj] 1));
val restrict_bij = result();
