(*  Title: 	ZF/ordinal
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Ordinals in Zermelo-Fraenkel Set Theory 
*)

writeln"File ZF/ex/ordinal";

structure Ord =
struct
val const_decs = 
    [ (["Memrel"],            	"i=>i"),
      (["Transset","Ord"],	"i=>o"),
      (["transrec"],	"[i, i, [i,i]=>i] =>i") 
    ];

val thy = extend_theory WF.thy "ord"
     ([], [], [], const_decs, None)
  [
    ("Memrel_def",  "Memrel(A) == {z: A*A . EX x y. z=<x,y> & x:y }"),

    ("Transset_def","Transset(i)  == ALL x:i. x<=i"),

    ("Ord_def",	    "Ord(i) == Transset(i) & (ALL x:i. Transset(x))"),

    ("transrec_def", "transrec(i,j,H) == wfrec(Memrel(i), j, H)")

  ];
end;

local val ax = get_axiom Ord.thy
in 
val Transset_def = ax"Transset_def";
val Memrel_def = ax"Memrel_def";
val Ord_def = ax"Ord_def";
val transrec_def = ax"transrec_def";
end;


(*** Natural Deduction rules for Transset ***)

val prems = goalw Ord.thy [Transset_def]
    "(!!x. x:i ==> x<=i) ==> Transset(i)";
by (REPEAT (ares_tac (prems@[ballI]) 1));
val TranssetI = result();

val [major,minor] = goalw Ord.thy [Transset_def]
    "[| Transset(i);  a:i |] ==> a<=i";
by (rtac (minor RS (major RS bspec)) 1);
val TranssetD = result();

(*** Closure properties for Transset ***)

val prems = goal Ord.thy
    "[| Transset(i);  Transset(j) |] ==> Transset(i Un j)";
by (cut_facts_tac prems 1);
by (DEPTH_SOLVE (eresolve_tac [UnE, TranssetD] 1
          ORELSE ares_tac [TranssetI, Un_upperbound1, Un_upperbound2] 1));
val Transset_Un = result();

val prems = goal Ord.thy
    "[| Transset(i);  Transset(j) |] ==> Transset(i Int j)";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [TranssetI,Int_greatest] 1
     ORELSE eresolve_tac [IntE,TranssetD] 1));
val Transset_Int = result();

val [prem] = goal Ord.thy "Transset(i) ==> Transset(succ(i))";
by (rtac TranssetI 1);
by (etac succE 1);
by (etac ssubst 1);
by (etac (prem RS TranssetD RS subset_trans) 2);
by (REPEAT (ares_tac [succI2 RS subsetI] 1));
val Transset_succ = result();

val [prem] = goal Ord.thy
    "[| !!i. i:A ==> Transset(i) |] ==> Transset(Union(A))";
by (rtac TranssetI 1);
by (etac UnionE 1);
by (rtac Union_upperbound 1);
by (REPEAT (eresolve_tac [asm_rl, prem RS TranssetD] 1));
val Transset_Union = result();

val prems = goal Ord.thy
    "[| j:A;  !!i. i:A ==> Transset(i) |] ==> Transset(Inter(A))";
by (rtac (Inter_greatest RS TranssetI) 1);
by (REPEAT (ares_tac (prems@[TranssetD]) 1 ORELSE etac InterD 1));
val Transset_Inter = result();

(*** Natural Deduction rules for Ord ***)

val prems = goalw Ord.thy [Ord_def]
    "[| Transset(i);  !!x. x:i ==> Transset(x) |]  ==>  Ord(i) ";
by (REPEAT (ares_tac (prems@[ballI,conjI]) 1));
val OrdI = result();

val [major] = goalw Ord.thy [Ord_def]
    "Ord(i) ==> Transset(i) ";
by (rtac (major RS conjunct1) 1);
val OrdD1 = result();

val [major,minor] = goalw Ord.thy [Ord_def]
    "[| Ord(i);  j:i |] ==> Transset(j) ";
by (rtac (minor RS (major RS conjunct2 RS bspec)) 1);
val OrdD2 = result();

(*** Lemmas for ordinals ***)

val prems = goalw Ord.thy [Ord_def,Transset_def]
    "[| Ord(i);  j:i |] ==> Ord(j) ";
by (cut_facts_tac prems 1);
by (DEPTH_SOLVE (ares_tac [ballI,impI,conjI] 1
     ORELSE set_mp_tac 1 ORELSE ball_tac 1 
     ORELSE eresolve_tac [conjE,impE] 1));
val Ord_in_Ord = result();

val prems = goal Ord.thy
    "[| Ord(i);  Transset(j);  j<=i |] ==> Ord(j)";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [OrdI] 1
     ORELSE eresolve_tac [OrdD2, subsetD] 1));
val Ord_subset_Ord = result();

val prems = goalw Ord.thy [Ord_def,Transset_def]
    "[| j:i;  Ord(i) |] ==> j<=i";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl,conjE,bspec] 1));
val OrdmemD = result();

val prems = goal Ord.thy
    "[| i:j;  j:k;  Ord(k) |] ==> i:k";
by (REPEAT (resolve_tac (prems@[OrdmemD,subsetD]) 1));
val Ord_trans = result();


(*** The construction of ordinals: 0, succ, Union ***)

goal Ord.thy
    "Ord(0) ";
by (REPEAT (ares_tac [OrdI,TranssetI] 1 ORELSE etac emptyE 1));
val Ord_0 = result();

val prems = goal Ord.thy
    "Ord(i) ==> Ord(succ(i)) ";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [OrdI,Transset_succ] 1
     ORELSE eresolve_tac [succE,ssubst,OrdD1,OrdD2] 1));
val Ord_succ = result();

val prems = goal Ord.thy
    "[| !!i. i:A ==> Ord(i) |] ==> Ord(Union(A))";
by (rtac (OrdD1 RS Transset_Union RS OrdI) 1);
by (REPEAT (etac UnionE 1 ORELSE ares_tac ([OrdD2]@prems) 1));
val Ord_Union = result();

val nonempty::prems = goal Ord.thy
    "[| j:A;  !!i. i:A ==> Ord(i) |] ==> Ord(Inter(A))";
by (rtac (nonempty RS Transset_Inter RS OrdI) 1);
by (rtac OrdD1 1);
by (REPEAT (ares_tac ([OrdD2,nonempty]@prems) 1 ORELSE etac InterD 1));
val Ord_Inter = result();


(*** Natural Deduction rules for Memrel ***)

goalw Ord.thy [Memrel_def] "<a,b> : Memrel(C) <-> a:b & a:C & b:C";
by (fast_tac ZF_cs 1);
val Memrel_iff = result();

val prems = goal Ord.thy "[| a: b;  a: C;  b: C |]  ==>  <a,b> : Memrel(C)";
by (REPEAT (resolve_tac (prems@[conjI, Memrel_iff RS iffD1]) 1));
val MemrelI = result();

val [major,minor] = goal Ord.thy
    "[| <a,b> : Memrel(A);  \
\       [| a: A;  b: A;  a:b |]  ==> P \
\    |]  ==> P";
by (rtac (major RS (Memrel_iff RS iffD2) RS conjE) 1);
by (etac conjE 1);
by (rtac minor 1);
by (REPEAT (assume_tac 1));
val MemrelE = result();


(*The membership relation (as a set) is well-founded.
  Proof idea: show A<=B by applying the foundation axiom to A-B *)
goalw Ord.thy [wf_def] "wf(Memrel(X))";
by (rtac (foundation RS disjE RS allI) 1);
by (REPEAT (eresolve_tac [disjI1, bexE, bspec, MemrelE] 1
     ORELSE ares_tac [impI RS allI RS bexI RS disjI2] 1));
val wf_memrel = result();

val [major] = goalw Ord.thy [trans_def] "Ord(i) ==> trans(Memrel(i))";
val itrans = (major RS Ord_in_Ord) RSN (3, Ord_trans);
by (DEPTH_SOLVE (ares_tac [allI,impI,MemrelI] 1 
     ORELSE eresolve_tac [conjE,MemrelE,itrans] 1));
val trans_Memrel = result();


(*** Transfinite induction ***)

val [major,indhyp] = goal Ord.thy
    "[| Ord(i); \
\       !!x.[| Ord(x);  ALL y:x. P(y) |] ==> P(x) \
\    |]  ==>  P(i)";
val Ord_succ_succ = major RS Ord_succ RS Ord_succ;
by (res_inst_tac [ ("X1","succ(i)"), ("a","i") ]
    (succI1 RS (wf_memrel RS wf_induct2)) 1);
by (fast_tac (ZF_cs addEs [MemrelE]) 1);
by (rtac indhyp 1);
by (REPEAT (eresolve_tac [asm_rl, spec RS mp, Ord_trans] 1
     ORELSE ares_tac [succI1,ballI,MemrelI,Ord_succ_succ,Ord_in_Ord] 1));
val trans_induct = result();

(*Perform induction on i, then prove the Ord(i) subgoal using prems. *)
fun trans_ind_tac a prems i = 
    EVERY [res_inst_tac [("i",a)] trans_induct i,
	   rename_last_tac a ["1"] (i+1),
	   ares_tac prems i];

(*** Fundamental properties of the ordering ***)

(*Finds contradictions for the following proof*)
val Ord_trans_tac = EVERY' [etac notE, etac Ord_trans, REPEAT o atac];

val prems = goal Ord.thy
    "Ord(i) ==> (ALL j. Ord(j) --> i:j | i=j | j:i)";
by (trans_ind_tac "i" prems 1);
by (rtac (impI RS allI) 1);
by (trans_ind_tac "j" [] 1);
by (DEPTH_SOLVE (swap_res_tac [disjCI,equalityI,subsetI] 1
     ORELSE ball_tac 1
     ORELSE eresolve_tac [impE,disjE,allE] 1 
     ORELSE hyp_subst_tac 1
     ORELSE Ord_trans_tac 1));
val Ord_trichotomy_lemma = result();

val prems = goal Ord.thy
    "[| Ord(i);  Ord(j);  i:j ==> P;  i=j ==> P;  j:i ==> P |] \
\    ==> P";
by (cut_facts_tac prems 1);
by (rtac allE 1);
by (etac Ord_trichotomy_lemma 1);
(*search in order to pick up j rather than i*)
by (DEPTH_SOLVE (eresolve_tac (prems@[asm_rl,impE,disjE]) 1)); 
val Ord_trichotomy = result();

val prems = goal Ord.thy
    "[| Ord(i);  Ord(j);  i<=j ==> P;  j<=i ==> P |] \
\    ==> P";
by (res_inst_tac [("i","i"),("j","j")] Ord_trichotomy 1);
by (DEPTH_SOLVE (ares_tac (prems@[subset_refl]) 1
          ORELSE eresolve_tac [asm_rl,OrdmemD,ssubst] 1));
val Ord_subset = result();

val prems = goal Ord.thy "[| j<=i;  ~j=i;  Ord(i);  Ord(j) |] ==> j:i";
by (cut_facts_tac prems 1);
by (etac Ord_trichotomy 1);
by (REPEAT ((etac notE 1 THEN etac sym 1) 
     ORELSE eresolve_tac ([asm_rl,OrdmemD,sym]@swapify [equalityI]) 1));
val Ord_member = result();

val prems = goal Ord.thy "[| Ord(i) |] ==> 0: succ(i)";
by (rtac (empty_subsetI RS Ord_member) 1);
by (fast_tac (ZF_cs addEs [equalityE]) 1);
by (REPEAT (resolve_tac (prems@[Ord_0,Ord_succ]) 1));
val Ord_0_mem_succ = result();

val prems = goal Ord.thy "[| Ord(i);  Ord(j) |] ==> i:j <-> i<=j & ~(i=j)";
by (cut_facts_tac prems 1);
by (REPEAT ((etac ssubst 1 THEN assume_tac 1)
     ORELSE ares_tac [iffI,conjI, mem_anti_refl RS notI] 1
     ORELSE eresolve_tac [conjE,Ord_member,OrdmemD] 1));
val Ord_member_iff = result();

val prems = goal Ord.thy "[| i<=j;  Ord(i);  Ord(j) |] ==> i : succ(j)";
by (cut_facts_tac prems 1);
by (rtac Ord_member 1);
by (REPEAT (ares_tac [subsetI,notI,succI2] 1   
     ORELSE eresolve_tac [subsetD,equalityE] 1));
by (rtac mem_anti_refl 1);
by (REPEAT (ares_tac [succI1,Ord_succ] 1   
     ORELSE etac subsetD 1));
val member_succI = result();

val prems = goal Ord.thy "[| i : succ(j);  Ord(j) |] ==> i<=j";
by (cut_facts_tac prems 1);
by (etac succE 1);
by (etac ssubst 1);
by (rtac subset_refl 1);
by (etac (OrdD1 RS TranssetD) 1);
by (assume_tac 1);
val member_succD = result();

val prems = goal Ord.thy
    "[| i:j;  Ord(i);  Ord(j) |] ==> succ(i) : succ(j)";
by (rtac member_succI 1);
by (REPEAT (ares_tac (prems@[subsetI,Ord_succ]) 1   
     ORELSE eresolve_tac [succE,Ord_trans,ssubst] 1));
val succ_mem_succI = result();

val prems = goal Ord.thy "[| succ(i) : succ(j);  Ord(j) |] ==> i:j";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, make_elim member_succD, succ_subsetE] 1));
val succ_mem_succE = result();

val prems = goal Ord.thy
    "[| Ord(i);  Ord(j) |] ==> succ(i) : succ(j) <-> i:j";
by (REPEAT (ares_tac (prems@[iffI,succ_mem_succI,succ_mem_succE]) 1));
val succ_mem_succ_iff = result();

(*** Transfinite induction over the ordinal i ***)

val major::prems = goal Ord.thy
    "[| j: i;  Ord(i);                          \
\       !!x.[| x:i;  ALL y:x. P(y) |] ==> P(x)  \
\    |]  ==>  P(j)";
by (rtac (major RS rev_mp) 1);
by (trans_ind_tac "j" [Ord_in_Ord] 1);
by (REPEAT (ares_tac (prems@[major,impI]) 1));
by (rtac ballI 1);
by (ball_tac 1);
by (etac mp 1);
by (REPEAT (ares_tac (prems@[Ord_trans]) 1));
val Ord_induct = result();

(*Perform induction on j:i, then prove j:i and Ord(i) using prems. *)
fun Ord_ind_tac aj ai prems i = 
    EVERY [res_inst_tac [("j",aj), ("i",ai)] Ord_induct i,
	   rename_last_tac aj ["1"] (i+2),
	   ares_tac prems i,
	   ares_tac prems i];

(*** Transfinite recursion: transrec ***)

val prems = goal Ord.thy
    "[| j:i;  Ord(i) |] ==> under(Memrel(i), j) = j";
by (REPEAT (ares_tac (prems@[equalityI, subsetI, underI, MemrelI]) 1
     ORELSE eresolve_tac [make_elim underD, MemrelE, Ord_trans] 1));
val under_Memrel = result();

val prems = goalw Ord.thy [transrec_def]
    "[| j:i;  Ord(i) |]  ==>  \
\    transrec(i,j,H) = H(j, lam x:j. transrec(i,x,H))";
by (rtac (wfrec_conv RS ssubst) 1);
by (ASM_SIMP_TAC (wf_ss addrews (prems@[under_Memrel])) 3);
by (REPEAT (resolve_tac (prems@[wf_memrel,trans_Memrel]) 1));
val transrec_conv = result();

(*Proof via wfrec_type is tiresome... *)
val prems = goal Ord.thy
    "[| j:i;  Ord(i);  \
\       !!x u. [| x:i;  u: Pi(x,B) |] ==> H(x,u) : B(x)   \
\    |]  ==> transrec(i,j,H) : B(j)";
by (Ord_ind_tac "j" "i" prems 1);
by (rtac (transrec_conv RS ssubst) 1);
by (REPEAT (ares_tac (prems @ [lam_type]) 1 ORELSE etac bspec 1));
val transrec_type = result();

val prems = goalw Ord.thy [transrec_def,Memrel_def]
    "[| i=i';  !!x u. H(x,u)=H'(x,u);  j=j' |]  ==> \
\    transrec(i,j,H)=transrec(i',j',H')";
by (SIMP_TAC (ZF_ss addcongs [wfrec_cong] addrews (prems RL [sym])) 1);
val transrec_cong = result();
