(*  Title: 	tactic
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Tactics 
*)

signature TACTIC =
sig
  structure Tactical: TACTICAL and Stringtree: STRINGTREE
  local open Tactical Tactical.Thm
  in
  val ares_tac: thm list -> int -> tactic   
  val asm_rl: thm
  val assume_tac: int -> tactic   
  val atac: int ->tactic
  val biresolve_tac: (bool*thm)list -> int -> tactic
  val compat_resolve_tac: thm list -> int -> int -> tactic   
  val compat_thms: 'a Stringtree.tree * term -> 'a list   
  val compose_inst_tac: (string*string)list -> (bool*thm*int) -> int -> tactic
  val compose_tac: (bool * thm * int) -> int -> tactic 
  val cut_facts_tac: thm list -> int -> tactic
  val cut_rl: thm
  val delete_thm: thm * thm Stringtree.tree -> thm Stringtree.tree
  val dresolve_tac: thm list -> int -> tactic
  val dres_inst_tac: (string*string)list -> thm -> int -> tactic   
  val dtac: thm -> int ->tactic
  val etac: thm -> int ->tactic
  val eq_assume_tac: int -> tactic   
  val eresolve_tac: thm list -> int -> tactic
  val eres_inst_tac: (string*string)list -> thm -> int -> tactic   
  val filter_thms: (term*term->bool) -> int*term*thm list -> thm list
  val filt_resolve_tac: thm list -> int -> int -> tactic
  val flexflex_tac: tactic
  val fold_goals_tac: thm list -> tactic
  val fold_tac: thm list -> tactic
  val forward_tac: thm list -> int -> tactic   
  val head_string: term -> string list
  val insert_thm: thm * thm Stringtree.tree -> thm Stringtree.tree
  val lessb: (bool * thm) * (bool * thm) -> bool
  val lift_inst_rule: thm * int * (string*string)list * thm -> thm
  val make_elim: thm -> thm
  val metacut_tac: thm -> int -> tactic   
  val PRIMITIVE: (thm -> thm) -> tactic  
  val rename_tac: string -> int -> tactic
  val rename_last_tac: string -> string list -> int -> tactic
  val resolve_tac: thm list -> int -> tactic
  val res_inst_tac: (string*string)list -> thm -> int -> tactic   
  val revcut_rl: thm
  val rewrite_goals_tac: thm list -> tactic
  val rewrite_goals_rule: thm list -> thm -> thm
  val rewrite_rule: thm list -> thm -> thm
  val rewrite_tac: thm list -> tactic
  val rewtac: thm -> tactic
  val rtac: thm -> int ->tactic
  val subgoals_of_brl: bool * thm -> int
  val subgoal_tac: string -> int -> tactic
  val trace_goalno_tac: (int -> tactic) -> int -> tactic
  end
end;


functor TacticFun (structure Logic: LOGIC and Tactical: TACTICAL and
	           Stringtree: STRINGTREE) : TACTIC = 
struct
structure Tactical = Tactical;
structure Thm = Tactical.Thm;
structure Stringtree = Stringtree;
structure Sequence = Thm.Sequence;
structure Sign = Thm.Sign;
local open Tactical Tactical.Thm
in

(*Discover what goal is chosen:  SOMEGOAL(trace_goalno_tac tac) *)
fun trace_goalno_tac tf i = Tactic (fn state => 
    case Sequence.pull(tapply(tf i, state)) of
	None    => Sequence.null
      | seqcell => (prs("Subgoal " ^ string_of_int i ^ " selected\n"); 
    			 Sequence.seqof(fn()=> seqcell)));

(*** Some useful meta-theorems ***)

(*The rule V/V, obtains assumption solving for eresolve_tac*)
val asm_rl = trivial(Sign.read_cterm Sign.pure ("PROP ?psi",propT));

(*Meta-level cut rule: [| V==>W; V |] ==> W *)
val cut_rl = trivial(Sign.read_cterm Sign.pure 
	("PROP ?psi ==> PROP ?theta", propT));

(*Generalized elimination rule for one conclusion --
  like cut_rl with reversed premises: [| V; V==>W |] ==> W *)
val revcut_rl =
  let val V = Sign.read_cterm Sign.pure ("PROP V", propT)
      and VW = Sign.read_cterm Sign.pure ("PROP V ==> PROP W", propT);
  in  standard (implies_intr V 
		(implies_intr VW
		 (implies_elim (assume VW) (assume V))))
  end;


(*** Basic tactics ***)

(*Makes a tactic whose effect on a state is given by thmfun: thm->thm seq.*)
fun PRIMSEQ thmfun = Tactic (fn state => thmfun state
			                 handle THM _ => Sequence.null);

(*Makes a tactic whose effect on a state is given by thmfun: thm->thm.*)
fun PRIMITIVE thmfun = PRIMSEQ (Sequence.single o thmfun);


(*** The following fail if the goal number is out of range:
     thus (REPEAT (resolve_tac rules i)) stops once subgoal i disappears. *)


(*Solve subgoal i by assumption*)
fun assume_tac i = PRIMSEQ (assumption i);

(*Solve subgoal i by assumption, using no unification*)
fun eq_assume_tac i = PRIMITIVE (eq_assumption i);

(*The composition rule/state: no lifting or var renaming.
  The arg = (bires_flg, orule, m) ;  see bicompose for explanation.*)
fun compose_tac arg i = Tactic (fn state =>
    bicompose(state, i)arg  handle THM _ => Sequence.null);

(*Resolution: the simple case, works for introduction rules*)
fun resolve_tac rules i = Tactic (fn state => resolution(state,i,rules)
	handle THM _ => Sequence.null);

(*Attack subgoal i by resolution, using flags to indicate elimination rules*)
fun biresolve_tac brules i = Tactic (fn state=> biresolution(state,i,brules)
	handle THM _ => Sequence.null);

(*Resolution with elimination rules only*)
fun eresolve_tac rules = biresolve_tac (map (pair true) rules);

(*Converts a "destruct" rule like P&Q==>P to an "elimination" rule
  like [| P&Q; P==>R |] ==> R *)
fun make_elim rl = zero_var_indexes (rl RS revcut_rl);

(*Creates and applies elimination rules.  Like forward_tac, but DELETES
  the assumption after use.*)
fun dresolve_tac rls = eresolve_tac (map make_elim rls);

(*Shorthand versions: for resolution with a single theorem*)
fun rtac rl = resolve_tac [rl];
fun etac rl = eresolve_tac [rl];
fun dtac rl = dresolve_tac [rl];
val atac = assume_tac;

(*Use an assumption or some rules ... A popular combination!*)
fun ares_tac rules = assume_tac  ORELSE'  resolve_tac rules;

(*Smash all flex-flex disagreement pairs in the proof state.*)
val flexflex_tac = PRIMSEQ flexflex_rule;

(*Lift and instantiate a rule wrt the given state and subgoal number *)
fun lift_inst_rule (state, i, sinsts, rule) =
let val {maxidx,sign,...} = rep_thm state
    val (_, _, Bi, _) = dest_state(state,i)
    val params = Logic.strip_params Bi	        (*params of subgoal i*)
    val params = rev(rename_wrt_term Bi params) (*as they are printed*)
    val paramTs = map #2 params
    and inc = maxidx+1
    fun liftvar (Var ((a,j), T)) = Var((a, j+inc), paramTs---> incr_tvar inc T)
      | liftvar t = raise TERM("Variable expected", [t]);
    fun liftterm t = list_abs_free (params, 
				    Logic.incr_indexes(paramTs,inc) t)
    (*Lifts instantiation pair over params*)
    fun liftpair (cv,ct) = (Sign.cfun liftvar cv, Sign.cfun liftterm ct)
    fun lifttvar((a,i),ctyp) =
	let val {T,sign} = Sign.rep_ctyp ctyp
	in  ((a,i+inc), Sign.ctyp_of sign (incr_tvar inc T)) end
    val def1 = defaultsv rule and (defT,defS) = defaults state
    val defT = (fn a => case assoc(params,a) of None => defT(a) | sm => sm )
    val (Tinsts,insts) = Sign.read_insts sign def1 (defT,defS) sinsts
in instantiate (map lifttvar Tinsts, map liftpair insts)
		(lift_rule (state,i) rule)
end;


(*** Resolve after lifting and instantation; may refer to parameters of the
     subgoal.  Fails if "i" is out of range.  ***)

(*compose version: arguments are as for bicompose.*)
fun compose_inst_tac sinsts (bires_flg, rule, nsubgoal) i =
  STATE ( fn state => 
	   compose_tac (bires_flg, lift_inst_rule (state, i, sinsts, rule),
			nsubgoal) i
	   handle TERM (msg,_) => (writeln msg;  no_tac)
		| THM _ => no_tac );

(*Resolve version*)
fun res_inst_tac sinsts rule i =
    compose_inst_tac sinsts (false, rule, length(prems_of rule)) i;

(*eresolve (elimination) version*)
fun eres_inst_tac sinsts rule i =
    compose_inst_tac sinsts (true, rule, length(prems_of rule)) i;

(*dresolve version*)
fun dres_inst_tac sinsts rule = eres_inst_tac sinsts (make_elim rule);

(*** Applications of cut_rl -- forwards reasoning ***)

(*Used by forward_tac and metacut_tac*)
fun bires_cut_tac arg i =
    resolve_tac [cut_rl] i  THEN  biresolve_tac arg (i+1) ;

(*Forwards reasoning. 
  Rule has the form A==>B, infers forwards from assumption A.*)
fun forward_tac rules = bires_cut_tac (map (pair true) rules);

(*The conclusion of the rule gets assumed in subgoal i,
  while subgoal i+1,... are the premises of the rule.*)
fun metacut_tac rule = bires_cut_tac [(false,rule)];

(*Recognizes theorems that are not rules, but simple propositions*)
fun is_fact rl =
    case prems_of rl of
	[] => true  |  _::_ => false;

(*"Cut" all facts from theorem list into the goal as assumptions. *)
fun cut_facts_tac ths i =
    EVERY (map (fn th => metacut_tac th i) (filter is_fact ths));

(*Introduce the given proposition as a lemma and subgoal*)
fun subgoal_tac sprop = res_inst_tac [("psi", sprop)] cut_rl;


(*** Indexing and filtering of theorems ***)

(*string of symbols following first argument of combinations
  symbols are constants, parameters, bound vars
  example:  rec(succ(x),...,...)  has   [ "rec", "succ", "*param*" ]  *)
fun head_string t : string list =
  let val (head,args) = strip_comb t;
      fun tail_string [] = []
	| tail_string(arg::_) = head_string arg
  in  case head of
      Const(name,_) =>     name :: tail_string args
    | Free (name,_) => "*"^name :: tail_string args
    | Bound bno => "B."^chr(bno+ord"0") :: tail_string args
    | _ => []
  end;

(*insert a thm in a thm tree using its head_string*)
fun insert_thm (th,rtr) =
  Stringtree.insert((head_string (concl_of th), th), rtr, eq_thm);

(*delete a thm from a thm tree using its head_string*)
fun delete_thm (th,rtr) =
  Stringtree.delete((head_string (concl_of th), th), rtr, eq_thm);

(*returns the list of theorems in the tree that are compatible with prem*)
fun compat_thms (rtr,prem) =
   Stringtree.compat(rtr, head_string (Logic.strip_assums_concl prem));


(*Resolve subgoal i using the tree of rules, unless too flexible --
   which means: >maxr are compatible.      *)
fun rtr_resolve_tac rtr maxr = SUBGOAL(fn (prem,i) =>
    let val rls = compat_thms (rtr,prem)
    in  if length rls > maxr  then no_tac  else resolve_tac rls i
    end);

(*For efficiency, bind its result to a tactic.  It builds a stringtree.*)
fun compat_resolve_tac rules = 
  rtr_resolve_tac (foldr insert_thm (rules, Stringtree.empty));
 

(*Returns the list of potentially resolvable theorems for the goal "prem",
	using the predicate  could(subgoal,concl).
  Resulting list is no longer than "limit"*)
fun filter_thms could (limit, prem, ths) =
  let val pb = Logic.strip_assums_concl prem;   (*delete assumptions*)
      fun filtr (limit, []) = []
	| filtr (limit, th::ths) =
	    if limit=0 then  []
	    else if could(pb, concl_of th)  then th :: filtr(limit-1, ths)
	    else filtr(limit,ths)
  in  filtr(limit,ths)  end;


(*Resolve subgoal i using the rules, unless >maxr are compatible. *)
fun filt_resolve_tac rules maxr = SUBGOAL(fn (prem,i) =>
  let val rls = filter_thms could_unify (maxr+1, prem, rules)
  in  if length rls > maxr  then  no_tac  else resolve_tac rls i
  end);


(*** For Natural Deduction using (bires_flg, rule) pairs ***)

(*The number of new subgoals produced by the brule*)
fun subgoals_of_brl (true,rule) = length (prems_of rule) - 1
  | subgoals_of_brl (false,rule) = length (prems_of rule);

(*Less-than test: for sorting to minimize number of new subgoals*)
fun lessb (brl1,brl2) = subgoals_of_brl brl1 < subgoals_of_brl brl2;


(*** Meta-Rewriting: Rules and Tactics ***)

(** Below, a "conversion" has type sign->term->thm **)

(*In [A1,...,An]==>B, rewrite the selected A's only -- for rewrite_goals_tac*)
fun goals_conv pred cv sign = 
  let val triv = reflexive o Sign.cterm_of sign
      fun gconv i t =
        let val (A,B) = Logic.dest_implies t
	    val thA = if (pred i) then (cv sign A) else (triv A)
	in  combination (combination (triv implies) thA)
                        (gconv (i+1) B)
        end
        handle TERM _ => triv t
  in gconv 1 end;

(*Use a conversion to transform a theorem*)
fun fconv_rule cv th =
    let val {sign,prop,...} = rep_thm th
    in  equal_elim (cv sign prop) th  end;

(*rewriting conversion*)
fun rew_conv rths sign t = rewrite_cterm rths (Sign.cterm_of sign t);

(*Rewrite a theorem*)
fun rewrite_rule rths = fconv_rule (rew_conv rths);

(*Rewrite the subgoals of a proof state (represented by a theorem) *)
fun rewrite_goals_rule rths =
    fconv_rule (goals_conv (K true) (rew_conv rths));

(*Rewrite subgoal i of a proof state*)
fun rewrite_goal_rule rths i =
    fconv_rule (goals_conv (curry op= i) (rew_conv rths));


(*Rewrite or fold throughout proof state. *)
fun rewrite_tac rths = PRIMITIVE (rewrite_rule rths);
fun fold_tac rths = rewrite_tac (map symmetric rths);

(*Rewrite subgoals only, not main goal. *)
fun rewrite_goals_tac rths = PRIMITIVE (rewrite_goals_rule rths);
fun fold_goals_tac rths = rewrite_goals_tac (map symmetric rths);

fun rewtac rth = rewrite_goals_tac [rth];


(** Renaming of parameters in a subgoal
    Names may contain letters, digits or primes and must be
    separated by blanks **)

val is_letdig = is_quasi_letter orf is_digit;

(*Calling this will generate the warning "Same as previous level" since
  it affects nothing but the names of bound variables!*)
fun rename_tac str i = 
  let val cs = explode str 
  in  
  if !Logic.auto_rename 
  then (writeln"Note: setting Logic.auto_rename := false"; 
	Logic.auto_rename := false)
  else ();
  case (strip_prefix (is_letdig orf is_blank) cs) of
      [] => PRIMITIVE (rename_params_rule (scanwords is_letdig cs, i))
    | c::_ => error ("Illegal character: " ^ c)
  end;

(*Rename recent parameters using names generated from (a) and the suffixes,
  provided the string (a), which represents a term, is an identifier. *)
fun rename_last_tac a sufs i = 
  let val names = map (curry op^ a) sufs
  in  if forall is_letdig (explode a)
      then PRIMITIVE (rename_params_rule (names,i))
      else all_tac
  end;

end;
end;

