(*  Title: 	Provers/classical
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Theorem prover for classical reasoning, including predicate calculus, set
theory, etc.

Rules must be classified as intr, elim, safe, hazardous:
Hazardous (unsafe) rules require delay or could cause infinite regress.  A rule
is hazardous if any of the following hold:
  (1) There are variables in the premises that are not in the conclusion.
  (2) There are repeated variables in the conclusion.
  (3) The conclusion contains terms of some type of "individuals"

comp_tac uses the XIs but not the Is (should this change?)
*)

signature CLASSICAL_DATA =
  sig
  val imp_elim: thm	(* "[| P-->Q;  P;  Q ==> R |] ==> R" *)
  val not_elim: thm	(* "[| ~P;  P |] ==> R" *)
  val swap: thm		(* "~P ==> (~Q ==> P) ==> Q" *)
  val size_of_thm : thm -> int
  val hyp_subst_tacs: (int -> tactic) list
  end;

infix 4 addIs addEs addSIs addSEs addXIs;


signature CLASSICAL =
  sig
  type claset
  val empty_cs: claset
  val addXIs: claset * thm list -> claset
  val addEs: claset * thm list -> claset
  val addIs: claset * thm list -> claset
  val addSEs: claset * thm list -> claset
  val addSIs: claset * thm list -> claset
  val print_cs: claset -> unit
  val rep_claset: claset -> 
      {safeIs: thm list, safeEs: thm list, hazIs: thm list, hazEs: thm list, 
       hazXIs: thm list, safe0_brls:(bool*thm)list, safep_brls: (bool*thm)list,
       haz_brls: (bool*thm)list}
  val best_tac : claset -> int -> tactic
  val chain_tac : int -> tactic
  val comp_step_tac : claset -> tactic
  val comp_tac : claset -> int -> tactic
  val contr_tac : int -> tactic
  val determ_prop_tac : claset -> int -> tactic
  val eq_mp_tac: int -> tactic
  val expand_tac : claset -> int -> tactic
  val fast_tac : claset -> int -> tactic
  val joinrules : thm list * thm list -> (bool * thm) list
  val mp_tac: int -> tactic
  val onestep_tac : claset -> int -> tactic
  val safe_tac : claset -> tactic
  val step_tac : claset -> int -> tactic
  val swapify : thm list -> thm list
  val swap_res_tac : thm list -> int -> tactic
  end;


functor ClassicalFun(Data: CLASSICAL_DATA): CLASSICAL = 
struct

local open Data in

(** Useful tactics for classical reasoning **)

(*Solve goal that assumes both P and ~P 
  Does not use imp_elim -- would be redundant in complete prover *)
val contr_tac = eresolve_tac [not_elim]  THEN'  assume_tac;

(*Like contr_tac but instantiates no variables*)
val eq_contr_tac = eresolve_tac [not_elim]  THEN'  eq_assume_tac;

(*Finds P-->Q and P in the assumptions, replaces implication by Q *)
fun mp_tac i = eresolve_tac [imp_elim,not_elim] i  THEN  assume_tac i;

(*Like mp_tac but instantiates no variables*)
fun eq_mp_tac i = eresolve_tac [imp_elim,not_elim] i  THEN  eq_assume_tac i;

(*Creates rules to eliminate ~A, from rules to introduce A*)
fun swapify intrs = intrs RLN (2, [swap]);

(*Uses introduction rules in the normal way, or on negated assumptions;
  this complicated definition ensures that rules are tried in order!*)
fun swap_res_tac rls i = 
    let fun tacf rl = resolve_tac [rl] i INTLEAVE 
	              eresolve_tac (swapify [rl]) i
        val basic_tac = assume_tac i ORELSE contr_tac i
    in  foldl (op INTLEAVE) (basic_tac, map tacf rls)
    end;

(*Given assumption P-->Q, reduces subgoal Q to P [deletes the implication!] *)
fun chain_tac i =
    eresolve_tac [imp_elim] i  THEN
    (assume_tac (i+1)  ORELSE  contr_tac (i+1));

(*** Classical rule sets ***)

datatype claset =
 CS of {safeIs: thm list,
	safeEs: thm list,
	hazIs: thm list,
	hazEs: thm list,
	hazXIs: thm list,
	(*the following are computed from the above*)
	safe0_brls: (bool*thm)list,
	safep_brls: (bool*thm)list,
	haz_brls: (bool*thm)list};
  
fun rep_claset (CS x) = x;

(*For use with biresolve_tac.  Combines intrs with swap to catch negated
  assumptions.  Also pairs elims with true. *)
fun joinrules (intrs,elims) =  
  (map (pair true) (elims @ (intrs RLN (2, [swap])))) @
  (map (pair false) intrs);

(*Note that allE precedes exI in haz_brls*)
fun make_cs {safeIs,safeEs,hazIs,hazEs,hazXIs} =
  let val (safe0_brls, safep_brls) = (*0 subgoals vs 1 or more*)
          partition (apl(0,op=) o subgoals_of_brl) 
             (sort lessb (joinrules(safeIs, safeEs)))
  in CS{safeIs=safeIs, safeEs=safeEs, hazIs=hazIs, hazEs=hazEs, hazXIs=hazXIs,
	safe0_brls=safe0_brls, safep_brls=safep_brls,
	haz_brls = sort lessb (joinrules(hazIs, hazEs))}
  end;


val empty_cs = make_cs{safeIs=[], safeEs=[], hazIs=[], hazEs=[], hazXIs=[]};

fun print_cs (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) =
 (writeln"Introduction rules";  prths hazIs;
  writeln"Safe introduction rules";  prths safeIs;
  writeln"X introduction rules";  prths hazXIs;
  writeln"Elimination rules";  prths hazEs;
  writeln"Safe elimination rules";  prths safeEs;
  ());

fun (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) addSIs ths =
  make_cs {safeIs=ths@safeIs, safeEs=safeEs, 
	   hazIs=hazIs, hazEs=hazEs, hazXIs=hazXIs};

fun (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) addSEs ths =
  make_cs {safeIs=safeIs, safeEs=ths@safeEs, 
	   hazIs=hazIs, hazEs=hazEs, hazXIs=hazXIs};

fun (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) addIs ths =
  make_cs {safeIs=safeIs, safeEs=safeEs, 
	   hazIs=ths@hazIs, hazEs=hazEs, hazXIs=hazXIs};

fun (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) addEs ths =
  make_cs {safeIs=safeIs, safeEs=safeEs, 
	   hazIs=hazIs, hazEs=ths@hazEs, hazXIs=hazXIs};

fun (CS{safeIs,safeEs,hazIs,hazEs,hazXIs,...}) addXIs ths =
  make_cs {safeIs=safeIs, safeEs=safeEs, 
	   hazIs=hazIs, hazEs=hazEs, hazXIs=ths@hazXIs};


(** Complete (?) but slow prover *)

(*Either simple step (which might solve goal completely!) or a hazardous rule. 
  Should expand all quantifiers, not just pick one!  Redundant alternatives! *)
fun expand_tac (CS{hazEs,hazXIs,safe0_brls,...}) = 
    biresolve_tac safe0_brls  INTLEAVE'  eq_contr_tac  INTLEAVE'  
    (resolve_tac hazEs  THEN'  assume_tac)  INTLEAVE'  
    biresolve_tac (joinrules(hazXIs,[]));

(*Attack subgoal i using propositional reasoning*)
fun determ_prop_tac (CS{safep_brls,...}) = 
  DETERM o REPEAT1 o
	FIRST' [eq_assume_tac, 
		eq_mp_tac, 
		FIRST' hyp_subst_tacs,
		biresolve_tac safep_brls];

(*Single step for the "complete" prover.*)
fun comp_step_tac cs =
    REPEAT (FIRSTGOAL (determ_prop_tac cs))   THEN
    COND (has_fewer_prems 1) all_tac
	 (FIRSTGOAL (assume_tac  INTLEAVE'  
		     contr_tac   INTLEAVE'  
		     expand_tac cs));

(*Resolution with the safe rules is presumed deterministic,
  except for the axioms (like reflexivity and assumption!!) *)
fun onestep_tac (CS{safe0_brls,safep_brls,...}) =
    FIRST' [eq_assume_tac,
	    eq_mp_tac, 
	    biresolve_tac safe0_brls,  
	    FIRST' hyp_subst_tacs,
	    assume_tac INTLEAVE' contr_tac,
	    DETERM o biresolve_tac safep_brls,
	    mp_tac] ;

(*Single step for the incomplete prover. 
  FAILS unless it makes reasonable progress. *)
fun step_tac (cs as (CS{haz_brls,...})) = 
    REPEAT1 o (onestep_tac cs) ORELSE' biresolve_tac haz_brls;

fun safe_tac cs = REPEAT_FIRST (onestep_tac cs);

(*Incomplete but fast.  Fails unless it solves one goal! *)
fun fast_tac cs = SELECT_GOAL (DEPTH_SOLVE (step_tac cs 1));

(*Incomplete but fast.  Fails unless it solves one goal! *)
fun best_tac cs = 
  SELECT_GOAL (BEST_FIRST (has_fewer_prems 1, size_of_thm) 
		   (safe_tac cs  ORELSE  step_tac cs 1));

(*"Less incomplete" than the above.  Fails unless it solves one goal!*)
fun comp_tac cs = 
  SELECT_GOAL (BEST_FIRST (has_fewer_prems 1, size_of_thm) 
	           (comp_step_tac cs));
end; 
end;
