(*  Title: 	HOL/sexp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

S-expressions, general binary trees for defining recursive data structures

Type ('a)sexp is defined as a class Sexp over (nat=>nat) * 'a => bool

Each s-expression is represented by a class {[<l0,x0>,...,<lk,xk>]} where
   each lk is a list of integers -- 1 for left and 2 for right --
   giving the access path to each Atom.

The lists above are represented by functions of type nat=>'a.  The "empty
list" is simply coded as a constant 0 function.
*)

structure Sexp =
struct
val sext = 
  Sext{mixfix = [ Infixr(".",	"['a sexp, 'a sexp]=> 'a sexp",	60) ],
       parse_translation = [],
       print_translation = []};

val thy = extend_theory Nat.thy "Sexp"
 ([],[],
  [ (["sexp"], (["term"],"term"))],
  [ 
   (["apfst"],		"['a=>'c, 'a*'b] => 'c*'b"),
   (["Push"],		"['a, nat=>'a] => (nat=>'a)"),
   (["Atom_Rep"],	"'a => ((nat=>nat) * 'a)class"),
   (["Scons_Rep"], 
    "[((nat=>nat)*'a)class, ((nat=>nat)*'a)class] => ((nat=>nat)*'a)class"),
   (["Sexp"],		"(((nat=>nat) * 'a)class)class"),
   (["Rep_Sexp"],	"'a sexp => ((nat=>nat) * 'a)class"),
   (["Abs_Sexp"],	"((nat=>nat) * 'a)class => 'a sexp"),
   (["Atom"],		"'a => 'a sexp"),
   (["sexp_case"],	"['a sexp, 'a=>'b, ['a sexp,'a sexp]=>'b] => 'b") ,
   (["pred_sexp"],	"('a sexp * 'a sexp)class"),
   (["sexp_rec"],   "['a sexp, 'a=>'b, ['a sexp,'a sexp,'b,'b]=>'b] => 'b") 
 ],
  Some sext)
 [
  ("apfst_def", 	"apfst == (%f p.<f(fst(p)),snd(p)>)"),
  ("Push_def", 		"Push == (%b h n. nat_case(n,b,h))"),
  ("Atom_Rep_def", 	"Atom_Rep(a) == {<%k.0, a>}"),
  ("Scons_Rep_def",	
   "Scons_Rep(i,j) == (apfst(Push(Suc(0))) `` i) \
\                  Un (apfst(Push(Suc(Suc(0)))) `` j)"),
  ("Sexp_def",	
   "Sexp == lfp(%Z. range(Atom_Rep) Un (UN i:Z. UN j:Z. {Scons_Rep(i,j)}))"),
    (*faking a type definition...*)
  ("Rep_Sexp", 		"Rep_Sexp(xs) : Sexp"),
  ("Rep_Sexp_inverse", 	"Abs_Sexp(Rep_Sexp(xs)) = xs"),
  ("Abs_Sexp_inverse", 	"i: Sexp ==> Rep_Sexp(Abs_Sexp(i)) = i"),
     (*defining the abstract constants*)
  ("Atom_def",  	"Atom == (%a. Abs_Sexp(Atom_Rep(a)))"),
  ("Scons_def", 	
   "M . N == Abs_Sexp(Scons_Rep(Rep_Sexp(M), Rep_Sexp(N)))"),

  ("sexp_case_def",	
   "sexp_case(M,c,d) == @ z. (? x.   M=Atom(x) & z=c(x))  \
\                          | (? N1 N2. M = N1 . N2  & z=d(N1,N2))" ),

  ("pred_sexp_def", "pred_sexp == {p. ? M N. p=<M, M . N> | p=<N, M . N>}" ),

  ("sexp_rec_def",	
   "sexp_rec(M,c,d) == wfrec(trancl(pred_sexp), \
\             %rec M. sexp_case(M, c, %N1 N2. d(N1, N2, rec(N1), rec(N2))), \
\             M)" )

 ];
end;



local val ax = get_axiom Sexp.thy
in
val apfst_def   	= ax"apfst_def";
val Push_def    	= ax"Push_def";
val Atom_Rep_def        = ax"Atom_Rep_def";
val Scons_Rep_def       = ax"Scons_Rep_def";
val Sexp_def    	= ax"Sexp_def";
val Rep_Sexp    	= ax"Rep_Sexp";
val Rep_Sexp_inverse    = ax"Rep_Sexp_inverse";
val Abs_Sexp_inverse    = ax"Abs_Sexp_inverse";
val Atom_def    	= ax"Atom_def";
val Scons_def   	= ax"Scons_def";
val sexp_case_def 	= ax"sexp_case_def";
val pred_sexp_def 	= ax"pred_sexp_def";
val sexp_rec_def 	= ax"sexp_rec_def";
end;


(** apfst -- can be used in similar type definitions **)

goalw Sexp.thy [apfst_def] "apfst(f,<a,b>) = <f(a),b>";
by (stac fst_conv 1);
by (stac snd_conv 1);
br refl 1;
val apfstI = result();

val [major,minor] = goalw Sexp.thy [apfst_def] 
    "[| q = apfst(f,p);  !!x y. [| p=<x,y>;  q=<f(x),y> |] ==> R \
\    |] ==> R";
br (surjective_pairing RS minor) 1;
br major 1;
val apfstE = result();

(** Push -- an injection, analogous to Cons on lists... *)

val [major] = goalw Sexp.thy [Push_def] "Push(x,f)=Push(y,g) ==> x=y";
br (major RS ap_thm RS box_equals) 1;
br nat_case_0_conv 1;
br nat_case_0_conv 1;
val Push_inject1 = result();

val [major] = goalw Sexp.thy [Push_def] "Push(x,f)=Push(y,g) ==> f=g";
br (major RS ap_thm RS abs RS box_equals) 1;
br (nat_case_Suc_conv RS abs) 1;
br (nat_case_Suc_conv RS abs) 1;
val Push_inject2 = result();

val major::prems = goal Sexp.thy
    "[| Push(x,f)=Push(y,g);  [| x=y;  f=g |] ==> P \
\    |] ==> P";
brs prems 1;
br (major RS Push_inject1) 1;
br (major RS Push_inject2) 1;
val Push_inject = result();

val [major] = goalw Sexp.thy [Push_def] "Push(Suc(k),f)=(%z.0) ==> P";
br (major RS ap_thm RS box_equals RS Suc_neq_Zero) 1;
br nat_case_0_conv 1;
br refl 1;
val PushSuc_neq_K0 = result();

(** the sexp functional **)

goal Sexp.thy
    "mono(%Z. range(Atom_Rep) Un (UN i:Z. UN j:Z. {Scons_Rep(i,j)}))";
by (REPEAT (ares_tac [monoI, subset_refl, UN_mono, Un_mono] 1));
val Sexp_fun_mono = result();

val Sexp_unfold = Sexp_fun_mono RS (Sexp_def RS Tarski_def_theorem);

(** Atom_Rep and Scons_Rep: Representations of the constructors **)

goal Sexp.thy "Atom_Rep(a) : Sexp";
by (rtac (Sexp_unfold RS ssubst) 1);
br (rangeI RS UnI1) 1;
val Atom_RepI = result();

val prems = goal Sexp.thy
    "[| i: Sexp;  j: Sexp |] ==> Scons_Rep(i,j) : Sexp";
by (rtac (Sexp_unfold RS ssubst) 1);
br UnI2 1;
by (DEPTH_SOLVE (resolve_tac (prems@[refl,UN_I,singletonI]) 1));
val Scons_RepI = result();

(** Induction **)

val major::prems = goalw Sexp.thy [Sexp_def]
    "[| ii: Sexp;  !!a. P(Atom_Rep(a));   \
\       !!i j. [| i: Sexp; j: Sexp; P(i); P(j) |] ==> \
\              P(Scons_Rep(i,j)) \
\    |]  ==> P(ii)";
br (Sexp_fun_mono RS (major RS general_induction)) 1;
by (fast_tac (class_cs addIs prems) 1);
val Sexp_induct = result();

val prems = goalw Sexp.thy [Atom_def,Scons_def]
    "[| !!a. P(Atom(a));   \
\       !!M N. [| P(M); P(N) |] ==> P(M.N) |]  ==> P(MM)";
by (rtac (Rep_Sexp_inverse RS subst) 1);   (*types force good instantiation*)
br (Rep_Sexp RS Sexp_induct) 1;
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [Abs_Sexp_inverse RS subst] 1));
val sexp_induct = result();

(*Perform induction on N. *)
fun sexp_ind_tac a i = 
    EVERY [res_inst_tac [("MM",a)] sexp_induct i,
	   rename_last_tac a ["1","2"] (i+1)];


(*** Isomorphisms ***)

goal Sexp.thy "One_One(Rep_Sexp)";
br One_One_inverseI 1;
br Rep_Sexp_inverse 1;
val One_One_Rep_Sexp = result();

goal Sexp.thy "One_One_on(Abs_Sexp,Sexp)";
br One_One_on_inverseI 1;
be Abs_Sexp_inverse 1;
val One_One_on_Abs_Sexp = result();

(*** Distinctness of constructors ***)

goalw Sexp.thy [Atom_Rep_def,Scons_Rep_def] "~ (Scons_Rep(i,j) = Atom_Rep(a))";
br notI 1;
be (equalityD2 RS subsetD RS UnE) 1;
br singletonI 1;
by (REPEAT (eresolve_tac [imageE, apfstE, Pair_inject, 
			  sym RS PushSuc_neq_K0] 1));
val Scons_Rep_not_Atom_Rep = result();

goalw Sexp.thy [Atom_def,Scons_def] "~ ((M.N) = Atom(a))";
br (One_One_on_Abs_Sexp RS One_One_on_contraD) 1;
br Scons_Rep_not_Atom_Rep 1;
by (REPEAT (resolve_tac [Atom_RepI, Rep_Sexp, Scons_RepI] 1));
val Scons_not_Atom = result();
val Atom_not_Scons = standard (Scons_not_Atom RS neg_sym);

val Scons_neq_Atom = standard (Scons_not_Atom RS notE);
val Atom_neq_Scons = sym RS Scons_neq_Atom;


(** Injectiveness of Atom **)

val [major] = goalw Sexp.thy [Atom_Rep_def]
    "Atom_Rep(a) = Atom_Rep(b) ==> a=b";
br (major RS equalityD1 RS subsetD RS singletonD RS Pair_inject) 1;
br singletonI 1;
ba 1;
val Atom_Rep_inject = result();

goalw Sexp.thy [Atom_def] "One_One(Atom)";
br One_OneI 1;
be (One_One_on_Abs_Sexp RS One_One_onD RS Atom_Rep_inject) 1;
by (REPEAT (resolve_tac [Rep_Sexp, Atom_RepI] 1));
val One_One_Atom = result();
val Atom_inject = One_One_Atom RS One_OneD;


(** Injectiveness of Scons **)

val [major,minor] = goalw Sexp.thy [Scons_Rep_def]
    "[| Scons_Rep(i,j)<=Scons_Rep(i',j');  [| i<=i';  j<=j' |] ==> P \
\    |] ==> P";
br minor 1;
by (ALLGOALS (EVERY'
     [rtac subsetI,
      rtac (major RS subsetD RS UnE),
      resolve_tac [UnI1,UnI2],      etac imageI]));
by (REPEAT (hyp_subst_tac 1
     ORELSE eresolve_tac [asm_rl, sym RS PushSuc_neq_K0, make_elim Suc_inject,
			  Suc_neq_Zero, Zero_neq_Suc, 
			  imageE, apfstE, sym RS apfstE, 
			  Pair_inject, Push_inject] 1));
val Scons_Rep_inj_lemma = result();

val major::prems = goal Sexp.thy
    "[| Scons_Rep(i,j)=Scons_Rep(i',j');  [| i=i';  j=j' |] ==> P \
\    |] ==> P";
br (major RS equalityE) 1;
brs prems 1;
by (REPEAT (ares_tac [equalityI] 1
     ORELSE eresolve_tac [Scons_Rep_inj_lemma] 1));
val Scons_Rep_inject = result();

val [major,minor] = goalw Sexp.thy [Scons_def]
    "[| (M.N)=(M'.N');  [| M=M';  N=N' |] ==> P \
\    |] ==> P";
br (One_One_on_Abs_Sexp RS One_One_onD RS Scons_Rep_inject) 1;
br major 1;
by (REPEAT (resolve_tac [Rep_Sexp, Scons_RepI] 1));
br minor 1;
by (REPEAT (etac (One_One_Rep_Sexp RS One_OneD) 1));
val Scons_inject = result();

(** Some rewrite rules **)

goal Sexp.thy "(M.N = M'.N') = (M=M' & N=N')";
by (REPEAT (ares_tac [refl,iffI,conjI] 1
     ORELSE eresolve_tac [conjE, Scons_inject, ssubst] 1));
val Scons_Scons_eq = result();

val sexp_ss = 
    HOL_ss addcongs (mk_congs Sexp.thy ["op ."])
           addrews [Scons_not_Atom, Atom_not_Scons, Scons_Scons_eq];

goal Sexp.thy "!M. ~(N=(M.N))";
by (sexp_ind_tac "N" 1);
br (Atom_not_Scons RS allI) 1;
by (ASM_SIMP_TAC sexp_ss 1);
val n_not_Scons_N = result();

(** sexp_case **)

goalw Sexp.thy [sexp_case_def] "sexp_case(Atom(a),c,d) = c(a)";
by (fast_tac (HOL_cs addIs [select_equality] 
	             addEs [make_elim Atom_inject, Atom_neq_Scons]) 1);
val sexp_case_Atom_conv = result();

goalw Sexp.thy [sexp_case_def] "sexp_case(M.N, c, d) = d(M,N)";
by (fast_tac (HOL_cs addIs [select_equality] 
	             addEs [Scons_inject, Scons_neq_Atom]) 1);
val sexp_case_Scons_conv = result();


(** Introduction rules for 'pred_sexp' **)

goalw Sexp.thy [pred_sexp_def] "<M, M.N> : pred_sexp";
by (fast_tac class_cs 1);
val pred_sexpI1 = result();

goalw Sexp.thy [pred_sexp_def] "<N, M.N> : pred_sexp";
by (fast_tac class_cs 1);
val pred_sexpI2 = result();

val major::prems = goalw Sexp.thy [pred_sexp_def]
    "[| p : pred_sexp;  \
\       !!M N. [| p = <M, M.N> |] ==> R; \
\       !!M N. [| p = <N, M.N> |] ==> R  \
\    |] ==> R";
br (major RS CollectE) 1;
by (REPEAT (eresolve_tac ([asm_rl,exE,disjE]@prems) 1));
val pred_sexpE = result();

goalw Sexp.thy [wf_def] "wf(pred_sexp)";
by (strip_tac 1);
by (sexp_ind_tac "x" 1);
by (fast_tac (HOL_cs addSEs [mp, pred_sexpE, Pair_inject, Scons_inject]) 2);
by (fast_tac (HOL_cs addSEs [mp, pred_sexpE, Pair_inject, Atom_neq_Scons]) 1);
val wf_pred_sexp = result();


(*** sexp_rec -- by wf recursion on pred_sexp ***)

(** conversion rules **)

(*can be used to unfold list recursion, term recursion etc.*)
val wfrec_pred_sexp_unfold = standard 
  (trans_trancl RS (wf_pred_sexp RS wf_trancl RS wfrec_def_conv));

val sexp_rec_unfold = standard (sexp_rec_def RS wfrec_pred_sexp_unfold);

goal Sexp.thy "sexp_rec(Atom(a), c, h) = c(a)";
by (stac sexp_rec_unfold 1);
br sexp_case_Atom_conv 1;
val sexp_rec_Atom_conv = result();

goal Sexp.thy
    "sexp_rec(M.N, c, h) = h(M, N, sexp_rec(M,c,h), sexp_rec(N,c,h))";
by (rtac (sexp_rec_unfold RS trans) 1);
by (rtac (sexp_case_Scons_conv RS trans) 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Sexp.thy [("h", "[?'a, ?'a, ?'b, ?'b]=> ?'b")])
      addrews [trancl_I1,pred_sexpI1,pred_sexpI2]) 1);
val sexp_rec_Scons_conv = result();

val sexp_ss = HOL_ss addrews
  [sexp_case_Atom_conv,sexp_case_Scons_conv,
   sexp_rec_Atom_conv,sexp_rec_Scons_conv];
