(*  Title: 	HOL/lemmas
    Author: 	Tobias Nipkow
    Copyright   1991  University of Cambridge

Derived rules from Appendix of Mike Gordons HOL Report, Cambridge TR 68 
*)

signature HOL_LEMMAS =
  sig
  val allE: thm
  val allI: thm
  val ap_term: thm
  val ap_thm: thm
  val box_equals: thm
  val ccontr: thm
  val classical: thm
  val cong: thm
  val conjunct1: thm
  val conjunct2: thm
  val conjE: thm
  val conjI: thm
  val contrapos: thm
  val disj_cases: thm
  val disjCI: thm
  val disjE: thm
  val disjI1: thm
  val disjI2: thm
  val eq_imp_lr: thm
  val eq_imp_rl: thm
  val eq_mp: thm
  val eqTrueI: thm
  val eqTrueE: thm
  val ex1E: thm
  val ex1I: thm
  val exCI: thm
  val exI: thm
  val exE: thm
  val excluded_middle: thm
  val ex_selectI: thm
  val FalseE: thm
  val False_neq_True: thm
  val gen: thm
  val iffCE : thm
  val iffE: thm
  val iffI: thm
  val impCE: thm
  val impE: thm
  val impI: thm
  val neg_sym: thm
  val notE: thm
  val notI: thm
  val notnotE : thm
  val rev_mp: thm
  val selectI: thm
  val selectE: thm
  val select_equality: thm
  val spec: thm
  val sstac: thm list -> int -> tactic
  val ssubst: thm
  val stac: thm -> int -> tactic
  val strip_tac: int -> tactic
  val swap: thm
  val sym: thm
  val trans: thm
  val TrueI: thm
  val truth: thm
  end;

functor HOL_LemmasFun(HOL_Rule:HOL_RULE):HOL_LEMMAS =

struct
local open HOL_Rule in

(** Equality **)

val sym = prove_goal HOL_Rule.thy "s=t ==> t=s"  (fn prems =>
	[cut_facts_tac prems 1, etac subst 1, rtac refl 1]);

(*calling "standard" reduces maxidx to 0*)
val ssubst = standard (sym RS subst);

val trans = prove_goal HOL_Rule.thy "[| r=s; s=t |] ==> r=t"  (fn prems =>
	[rtac subst 1, resolve_tac prems 1, resolve_tac prems 1]);

(*Useful with eresolve_tac for proving equalties from known equalities.
	a = b
	|   |
	c = d	*)
val box_equals = prove_goal HOL_Rule.thy
    "[| a=b;  a=c;  b=d |] ==> c=d"  
 (fn prems=>
  [ (resolve_tac [trans] 1),
    (resolve_tac [trans] 1),
    (resolve_tac [sym] 1),
    (REPEAT (resolve_tac prems 1)) ]);

val ap_term = prove_goal HOL_Rule.thy "s=t ==> f(s)=f(t)"  (fn prems =>
	[rtac subst 1, resolve_tac prems 1, rtac refl 1]);

val ap_thm = prove_goal HOL_Rule.thy "s::'a=>'b = t ==> s(x)=t(x)"  (fn prems =>
	[res_inst_tac[("'a","'a=>'b")]subst 1,
	 resolve_tac prems 1, rtac refl 1]);

(** Equality of booleans -- iff **)

val eq_mp = prove_goal HOL_Rule.thy "[| P=Q; P |] ==> Q"  (fn prems =>
	[rtac subst 1, resolve_tac prems 1, resolve_tac prems 1]);

val eq_imp_lr = prove_goal HOL_Rule.thy "P=Q ==> P-->Q"  (fn prems =>
	[REPEAT(ares_tac ([disch]@prems@[eq_mp]) 1)]);

val eq_imp_rl = prove_goal HOL_Rule.thy "P=Q ==> Q-->P"  (fn prems =>
	[REPEAT(ares_tac ([disch]@prems@[sym,eq_mp]) 1)]);

(** True **)

val truth = refl RS (True_def RS (sym RS eq_mp));
val TrueI = truth;

val eqTrueI  = prove_goal HOL_Rule.thy "P ==> P=True" 
 (fn prems => [REPEAT(resolve_tac ([iff RS mp RS mp,disch,truth]@prems) 1)]);

val eqTrueE = prove_goal HOL_Rule.thy "P=True ==> P" 
 (fn prems => [REPEAT(resolve_tac (prems@[truth,sym,eq_mp]) 1)]);

(** Universal quantifier **)

val gen = prove_goal HOL_Rule.thy "(!!x::'a. P(x)) ==> !x. P(x)"
 (fn [asm] => [rtac (All_def RS ssubst) 1, rtac (asm RS (eqTrueI RS abs)) 1]);
val allI = gen;

val spec = prove_goal HOL_Rule.thy "!x::'a.P(x) ==> P(x)"
 (fn prems => map (fn rls => resolve_tac rls 1)
                  [[eqTrueE], [ap_thm], [All_def RS subst], prems]);


(** Select: Hilbert's Epsilon-operator **)

val selectI = standard (select RS mp);

val selectE = prove_goal HOL_Rule.thy "[| P(Eps(P)); !!x. P(x) ==> Q |] ==> Q"
	(fn [a1,a2] => [rtac (a1 RS a2) 1]);

(** Existential quantifier **)

val exI = prove_goal HOL_Rule.thy "P(x) ==> ? x::'a.P(x)"  (fn prems =>
	[rtac (selectI RS (Ex_def RS ssubst)) 1,
	 resolve_tac prems 1]);

val exE = prove_goal HOL_Rule.thy "[| ? x::'a.P(x); !!x. P(x) ==> Q |] ==> Q"
	(fn prems =>
	[resolve_tac prems 1, res_inst_tac [("P","%C.C(P)")] subst 1,
	 rtac Ex_def 1, resolve_tac prems 1]);

(*This form can be much easier to apply than selectI*)
val ex_selectI = prove_goal HOL_Rule.thy "Ex(P) ==> P(Eps(P))"
 (fn [prem] => [ (rtac (prem RS exE) 1), (rtac selectI 1), (atac 1) ]);


(* RIGHT_BETA_AP: covered by substitution *)

(** Conjunction **)

val conjI = prove_goal HOL_Rule.thy "[| P; Q |] ==> P&Q"  (fn [P,Q] =>
	[rtac (and_def RS ssubst) 1, rtac gen 1,
	 rtac (Q RSN (2,mp RS disch)) 1, etac (P RSN (2,mp)) 1]);

val conjunct1 = prove_goal HOL_Rule.thy "[| P & Q |] ==> P"
 (fn prems => [rtac mp 1, rtac spec 1, rtac (and_def RS subst) 1, 
	       resolve_tac prems 1, REPEAT(ares_tac [disch] 1)]);

val conjunct2 = prove_goal HOL_Rule.thy "[| P & Q |] ==> Q" 
 (fn prems => [rtac mp 1, rtac spec 1, rtac (and_def RS subst) 1, 
	       resolve_tac prems 1, REPEAT(ares_tac [disch] 1)]);

(** Disjunction *)

val disjI1 = prove_goal HOL_Rule.thy "P ==> P|Q"  (fn [prem] =>
	[rtac (or_def RS ssubst) 1,
	 REPEAT(ares_tac [gen,disch, prem RSN (2,mp)] 1)]);

val disjI2 = prove_goal HOL_Rule.thy "Q ==> P|Q"  (fn [prem] =>
	[rtac (or_def RS ssubst) 1,
	 REPEAT(ares_tac [gen,disch, prem RSN (2,mp)] 1)]);

val disj_cases = prove_goal HOL_Rule.thy "[| P | Q; P ==> R; Q ==> R |] ==> R"
	(fn [a1,a2,a3] =>
	[rtac (mp RS mp) 1, rtac spec 1, rtac (or_def RS subst) 1, rtac a1 1,
	 rtac (a2 RS disch) 1, atac 1, rtac (a3 RS disch) 1, atac 1]);
val disjE = disj_cases;

(** Negation **)

val notI = prove_goal HOL_Rule.thy  "(P ==> False) ==> ~P"
 (fn prems=> [rtac (not_def RS ssubst) 1, rtac disch 1, eresolve_tac prems 1]);

val not_mp = prove_goal HOL_Rule.thy "[| ~P;  P |] ==> False"
 (fn prems => [rtac mp 1, resolve_tac prems 2, rtac (not_def RS ssubst) 1,
	       rtac disch 1, etac mp 1, resolve_tac prems 1]);

val FalseE = prove_goal HOL_Rule.thy "False ==> P"
 (fn prems => [rtac spec 1, rtac (False_def RS subst) 1, resolve_tac prems 1]);

val notE = prove_goal HOL_Rule.thy "[| ~P;  P |] ==> R"
  (fn prems=> [ (REPEAT (resolve_tac (prems@[not_mp,FalseE]) 1)) ]);

val False_neq_True = prove_goal HOL_Rule.thy "False=True ==> P"
 (fn [prem] => [rtac (prem RS eqTrueE RS FalseE) 1]);

(** CCONTR -- classical logic **)

val ccontr = prove_goal HOL_Rule.thy "(~P ==> False) ==> P"
 (fn prems =>
   [rtac (True_or_False RS (disj_cases RS eqTrueE)) 1, atac 1,
    rtac spec 1, rtac (False_def RS subst) 1, resolve_tac prems 1,
    rtac ssubst 1, atac 1, rtac (not_def RS ssubst) 1,
    REPEAT (ares_tac [disch] 1) ]);

val classical = prove_goal HOL_Rule.thy "(~P ==> P) ==> P"
 (fn prems =>
   [rtac ccontr 1,
    REPEAT (ares_tac (prems@[notE]) 1)]);


(*Double negation law*)
val notnotE = prove_goal HOL_Rule.thy "[| ~~P;  P ==> Q |] ==> Q"
 (fn prems=>
  [ (resolve_tac prems 1),
    (resolve_tac [classical] 1),
    (resolve_tac [notE] 1),
    (resolve_tac prems 1),
    (assume_tac 1) ]);

(** Implication **)

val impI = disch;

val impE = prove_goal HOL_Rule.thy "[| P-->Q;  P;  Q ==> R |] ==> R"
 (fn prems=> [ (REPEAT (resolve_tac (prems@[mp]) 1)) ]);

(* Reduces Q to P-->Q, allowing substitution in P. *)
val rev_mp = prove_goal HOL_Rule.thy "[| P;  P --> Q |] ==> Q"
 (fn prems=>  [ (REPEAT (resolve_tac (prems@[mp]) 1)) ]);

val contrapos = prove_goal HOL_Rule.thy "[| ~Q;  P==>Q |] ==> ~P"
 (fn [major,minor]=> 
  [ (rtac (major RS notE RS notI) 1), 
    (etac minor 1) ]);

(* ~(?t = ?s) ==> ~(?s = ?t) *)
val [neg_sym] = compose(sym,2,contrapos);

val allE = prove_goal HOL_Rule.thy "!x.P(x) ==> (P(x) ==> R) ==> R"
 (fn major::prems=>
  [ (REPEAT (resolve_tac (prems @ [major RS spec]) 1)) ]);

val iffI = disch RSN (2,disch RS (iff RS mp RS mp));

val iffE = prove_goal HOL_Rule.thy
    "[| P=Q; [| P --> Q; Q --> P |] ==> R |] ==> R"
 (fn [p1,p2] => [REPEAT(ares_tac([p2, p1 RS eq_imp_lr,
				  p1 RS sym RS eq_imp_lr])1)]);

val conjE = prove_goal HOL_Rule.thy "P&Q ==> ([| P;  Q |] ==> R) ==> R"
	(fn prems =>
	 [cut_facts_tac prems 1, resolve_tac prems 1,
	  etac conjunct1 1, etac conjunct2 1]);


val ex1I = prove_goalw HOL_Rule.thy [Ex1_def]
    "[| P(a);  !!x. P(x) ==> x=a |] ==> ?! x. P(x)"
 (fn prems => [ (REPEAT (ares_tac (prems@[exI,conjI,allI,impI]) 1)) ]);


val ex1E = prove_goalw HOL_Rule.thy [Ex1_def]
    "[| ?! x.P(x);  !!x. [| P(x);  ! y. P(y) --> y=x |] ==> R |] ==> R"
 (fn major::prems =>
  [ (rtac (major RS exE) 1),
    (REPEAT (eresolve_tac [conjE] 1 ORELSE ares_tac prems 1)) ]);

(*Should be used as swap since ~P becomes redundant*)
val swap = prove_goal HOL_Rule.thy "~P ==> (~Q ==> P) ==> Q"
 (fn major::prems=>
  [ rtac ccontr 1, rtac (major RS notE) 1, REPEAT (ares_tac prems 1)]);


val cong = prove_goal HOL_Rule.thy
   "[| f = g; x::'a = y |] ==> f(x) = g(y)"
 (fn prems => [res_inst_tac [("'a","'a"),("P","%y.f(x)=g(y)")] subst 1,
	       resolve_tac prems 1, rtac ap_thm 1, resolve_tac prems 1]);

(** Additional rules for the Epsilon-operator **)

val select_lemma = prove_goal HOL_Rule.thy
    "[| !!x. P(x) = (x=a) |] ==> Eps(P) = a"
 (fn [prem] => [ rtac (prem RS abs RS ssubst) 1, 
		 rtac selectI 1,    rtac refl 1 ]);

val select_equality = prove_goal HOL_Rule.thy
    "[| P(a);  !!x. P(x) ==> x=a |] ==> (@x.P(x)) = a"
 (fn prems => [ rtac (iffI RS select_lemma) 1, 
                eresolve_tac prems 1, 
	        etac ssubst 1, 
		resolve_tac prems 1 ]);

(** Classical intro rules for | and EX *)

val disjCI = prove_goal HOL_Rule.thy "(~Q ==> P) ==> P|Q"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (prems@[disjI1,notI]) 1)),
    (REPEAT (ares_tac (prems@[disjI2,notE]) 1)) ]);

val excluded_middle = prove_goal HOL_Rule.thy "~P | P"
 (fn _ => [ (REPEAT (ares_tac [disjCI] 1)) ]);

(*Classical implies (-->) elimination. *)
val impCE = prove_goal HOL_Rule.thy "[| P-->Q; ~P ==> R; Q ==> R |] ==> R" 
 (fn major::prems=>
  [ rtac (excluded_middle RS disjE) 1,
    REPEAT (DEPTH_SOLVE_1 (ares_tac (prems @ [major RS mp]) 1))]);

(*Classical <-> elimination. *)
val iffCE = prove_goal HOL_Rule.thy
    "[| P=Q;  [| P; Q |] ==> R;  [| ~P; ~Q |] ==> R |] ==> R"
 (fn major::prems =>
  [ (rtac (major RS iffE) 1),
    (REPEAT (DEPTH_SOLVE_1 
	(eresolve_tac ([asm_rl,impCE,notE]@prems) 1))) ]);

val exCI = prove_goal HOL_Rule.thy "(! x. ~P(x) ==> P(a)) ==> ? x.P(x)"
 (fn prems=>
  [ (rtac ccontr 1),
    (REPEAT (ares_tac (prems@[exI,allI,notI,notE]) 1))  ]);

(** Standard abbreviations **)

fun stac th = rtac(th RS ssubst);
fun sstac ths = EVERY' (map stac ths);
fun strip_tac i = REPEAT(resolve_tac [impI,allI] i); 

end;
end;

