(*  Title: 	HOL/ex/term
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Terms over a given alphabet -- function applications; illustrates List functor
  (essentially the same type as in Trees & Forests)

There is no constructor APP because it is simply cons (.) 
*)

structure Term =
struct
val thy = extend_theory List.thy "term"
([], [], 
 [(["term"], (["term"],"term"))],
 [ 
  (["Term"],	   "(('a+nat)sexp)class => (('a+nat)sexp)class"),
  (["Rep_Term"], 	"'a term => ('a+nat)sexp"),
  (["Abs_Term"], 	"('a+nat)sexp => 'a term"),
  (["Rep_TList"], 	"'a term list => ('a+nat)sexp"),
  (["Abs_TList"], 	"('a+nat)sexp => 'a term list"),
  (["App"],		"['a, ('a term)list] => 'a term"),
  (["Term_rec"],	
    "[('a+nat)sexp, [('a+nat)sexp , ('a+nat)sexp, 'b list]=>'b] => 'b"),
  (["term_rec"],        "['a term, ['a ,'a term list, 'b list]=>'b] => 'b")
 ],
 None)
 [ 
  ("Term_def",		"Term(A) == lfp(%Z.  A <*> List(Z))" ),
    (*faking a type definition for term...*)
  ("Rep_Term", 		"Rep_Term(n): Term(range(ATOM))"),
  ("Rep_Term_inverse", 	"Abs_Term(Rep_Term(t)) = t"),
  ("Abs_Term_inverse", 	"M: Term(range(ATOM)) ==> Rep_Term(Abs_Term(M)) = M"),
    (*defining abstraction/representation functions for term list...*)
  ("Rep_TList_def",	"Rep_TList == Rep_map(Rep_Term)" ),
  ("Abs_TList_def",	"Abs_TList == Abs_map(Abs_Term)" ),
    (*defining the abstract constants*)
  ("App_def", 	"App(a,ts) == Abs_Term(ATOM(a) . Rep_TList(ts))" ),
     (*list recursion*)
  ("Term_rec_def",	
   "Term_rec(M,d) == wfrec(trancl(pred_sexp), \
\            %rec M. @u. (? x y. M=x.y & u = d(x,y, Abs_map(rec,y))), \
\                        M)" ),

  ("term_rec_def",
   "term_rec(t,d) == \
\   Term_rec(Rep_Term(t), %x y r. d(Inv(ATOM,x), Abs_TList(y), r))" ) 
 ];
end;

local val ax = get_axiom Term.thy
in 
val Term_def    = ax"Term_def";
val Rep_Term    = ax"Rep_Term";
val Rep_Term_inverse    = ax"Rep_Term_inverse";
val Abs_Term_inverse    = ax"Abs_Term_inverse";
val Rep_TList_def       = ax"Rep_TList_def";
val Abs_TList_def       = ax"Abs_TList_def";
val App_def     = ax"App_def";
val Term_rec_def        = ax"Term_rec_def";
val term_rec_def        = ax"term_rec_def";
end;



(*** Monotonicity and unfolding of the function ***)

goal Term.thy "mono(%Z.  A <*> List(Z))";
by (REPEAT (ares_tac [monoI, subset_refl, List_mono, uprod_mono] 1));
val Term_fun_mono = result();

val Term_unfold = Term_fun_mono RS (Term_def RS Tarski_def_theorem);

(*This justifies using Term in other recursive type definitions*)
val prems = goalw Term.thy [Term_def] "[| A<=B |] ==> Term(A) <= Term(B)";
by (REPEAT (ares_tac (prems@[monoI, subset_refl, lfp_mono, 
			     List_mono, uprod_mono]) 1));
val Term_mono = result();


(** Elimination -- structural induction on the class Term(A) **)

(*Induction for the class Term(A) *)
val [major,minor] = goalw Term.thy [Term_def]
    "[| M: Term(A);  \
\       !!x zs. [| x: A;  zs: List(Term(A));  zs: List({x.R(x)}) \
\               |] ==> R(x.zs)  \
\    |] ==> R(M)";
br (Term_fun_mono RS (major RS general_induction)) 1;
by (REPEAT (eresolve_tac ([uprodE, ssubst, minor] @
 		([Int_lower1,Int_lower2] RL [List_mono RS subsetD])) 1));
val Term_induct = result();

(*Induction on Term(A) followed by induction on List *)
val major::prems = goal Term.thy
    "[| M: Term(A);  \
\       !!x.      [| x: A |] ==> R(x.NIL);  \
\       !!x z zs. [| x: A;  z: Term(A);  zs: List(Term(A));  R(x.zs)  \
\                 |] ==> R(x . CONS(z,zs))  \
\    |] ==> R(M)";
br (major RS Term_induct) 1;
by (etac List_induct 1);
by (REPEAT (ares_tac prems 1));
val Term_induct2 = result();

(*** Structural Induction on the abstract type 'a term ***)

(*Induction for the abstract type 'a term*)
val prems = goalw Term.thy [App_def,Rep_TList_def,Abs_TList_def]
    "[| !!x ts. [| list_all(R,ts) |] ==> R(App(x,ts))  \
\    |] ==> R(t)";
by (rtac (Rep_Term_inverse RS subst) 1);   (*types force good instantiation*)
br (Rep_Term RS Term_induct) 1;
by (eres_inst_tac [("A1","Term(?u)"), ("f1","Rep_Term"), ("g1","Abs_Term")]
    	(Abs_map_inverse RS subst) 1);
be Abs_Term_inverse 1;
be rangeE 1;
by (hyp_subst_tac 1);
brs prems 1;
be List_induct 1;
be CollectE 2;
by (ALLGOALS (ASM_SIMP_TAC list_ss));
val term_induct = result();


(*Induction for the abstract type 'a term*)
val prems = goal Term.thy 
    "[| !!x. R(App(x,Nil));  \
\       !!x t ts. R(App(x,ts)) ==> R(App(x, Cons(t,ts)))  \
\    |] ==> R(t)";
br term_induct 1;  (*types force good instantiation*)
be rev_mp 1;
br list_induct 1;  (*types force good instantiation*)
by (ALLGOALS (ASM_SIMP_TAC (list_ss addrews prems)));
val term_induct2 = result();

(*Perform induction on xs. *)
fun term_ind2_tac a i = 
    EVERY [res_inst_tac [("t",a)] term_induct2 i,
	   rename_last_tac a ["1","s"] (i+1)];


(** Introduction rule for Term **)

(* ?c : ?Ga5 <*> List(Term(?Ga5)) ==> ?c : Term(?Ga5) *)
val TermI = Term_unfold RS equalityD2 RS subsetD;

(*The constant APP is not declared; it is simply . *)
val prems = goal Term.thy "[| M: A;  N : List(Term(A)) |] ==> M.N : Term(A)";
by (REPEAT (resolve_tac (prems@[TermI, ListI, uprodI]) 1));
val APP_I = result();


(*** Term_rec -- by wf recursion on pred_sexp ***)

val Term_rec_unfold = standard (Term_rec_def RS wfrec_pred_sexp_unfold);

(** conversion rules **)

val [prem] = goal Term.thy
    "N: List(Term(A)) ==>  \
\    !M. <N,M>: trancl(pred_sexp) --> \
\        Abs_map(%Z. <Z,M> : trancl(pred_sexp) => h(Z) | a, N) = Abs_map(h,N)";
br (prem RS List_induct) 1;
by (SIMP_TAC list_ss 1);
by (strip_tac 1);
by (forward_tac [pred_sexp_CONS1 RS (trans_trancl RS transD)] 1);
bd (pred_sexp_CONS2 RS (trans_trancl RS transD)) 1;
by (ASM_SIMP_TAC list_ss 1);
val Abs_map_lemma = result();

val [prem] = goal Term.thy
    "N: List(Term(A)) ==> \
\    Term_rec(M.N, d) = d(M, N, Abs_map(%Z. Term_rec(Z,d), N))";
by (rtac (Term_rec_unfold RS trans) 1);
br (select_equality RS trans) 1;
by (fast_tac HOL_cs 1);
by (fast_tac (HOL_cs addEs [Scons_inject]) 1);
by (REPEAT (resolve_tac (refl :: 
		mk_typed_congs Term.thy [("d", "[?'a,?'b,?'c]=>?'d")]) 1));
br (prem RS Abs_map_lemma RS spec RS mp) 1;
br (pred_sexpI2 RS trancl_I1) 1;
val Term_rec_conv = result();


(*** term_rec -- by Term_rec ***)

val Rep_TList = Rep_Term RS Rep_map_type;

goalw Term.thy [term_rec_def, App_def, Rep_TList_def, Abs_TList_def]
    "term_rec(App(f,ts), d) = d(f, ts, map (%t. term_rec(t,d), ts))";
by (SIMP_TAC (HOL_ss 
	addcongs (mk_congs Term.thy ["map","Term_rec"] @ 
		  mk_typed_congs Term.thy [("d", "[?'a, ?'b, ?'c]=>?'d")])
	addrews [rangeI RS APP_I RS Abs_Term_inverse, Term_rec_conv, 
		 Rep_Term_inverse, Rep_TList, One_One_ATOM, Inv_f_f,
		 Abs_Rep_map, map_ident]) 1);
val term_rec_conv = result();

