(*  Title: 	FOL/cla-lemmas
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Classical First-Order Logic
*)

signature CLA_LEMMAS = 
  sig
  val cla_thy : theory
  val classical : thm
  val disjCI : thm
  val excluded_middle : thm
  val exCI : thm
  val iffCE : thm
  val impCE : thm
  val notnotE : thm
  val swap : thm
  end;


functor Cla_LemmasFun(structure Int_Rule: INT_RULE
                      and       Int_Prover: INT_PROVER) : CLA_LEMMAS = 
struct

local open  Int_Prover Int_Rule   
in
val cla_thy = extend_theory Int_Rule.thy  "Classical" ([],[],[],[],None)
  [ ("classical",  "(~P ==> P) ==> P") ];

val classical = get_axiom cla_thy "classical";


(** Classical intro rules for | and EX *)

val disjCI = prove_goal cla_thy 
   "(~Q ==> P) ==> P|Q"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (prems@[disjI1,notI]) 1)),
    (REPEAT (ares_tac (prems@[disjI2,notE]) 1)) ]);


val exCI = prove_goal cla_thy 
   "(ALL x. ~P(x) ==> P(a)) ==> EX x.P(x)"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (prems@[exI,allI,notI,notE]) 1))  ]);


val excluded_middle = prove_goal cla_thy "~P | P"
 (fn _=> [ (REPEAT (ares_tac [disjCI] 1)) ]);


(*** Special elimination rules *)


(*Classical implies (-->) elimination. *)
val impCE = prove_goal cla_thy 
    "[| P-->Q; ~P ==> R; Q ==> R |] ==> R"
 (fn major::prems=>
  [ (resolve_tac [excluded_middle RS disjE] 1),
    (REPEAT (DEPTH_SOLVE_1 (ares_tac (prems@[major RS mp]) 1))) ]);

(*Double negation law*)
val notnotE = prove_goal cla_thy 
   "[| ~~P;  P ==> Q |] ==> Q"
 (fn prems=>
  [ (resolve_tac prems 1),
    (resolve_tac [classical] 1),
    (resolve_tac [notE] 1),
    (resolve_tac prems 1),
    (assume_tac 1) ]);


(*** Tactics for implication and contradiction ***)

(*Classical <-> elimination.  Proof substitutes P=Q in 
    ~P ==> ~Q    and    P ==> Q  *)
val iffCE = prove_goalw cla_thy [iff_def]
    "[| P<->Q;  [| P; Q |] ==> R;  [| ~P; ~Q |] ==> R |] ==> R"
 (fn prems =>
  [ (resolve_tac [conjE] 1),
    (REPEAT (DEPTH_SOLVE_1 
	(etac impCE 1  ORELSE  mp_tac 1  ORELSE  ares_tac prems 1))) ]);


(*Should be used as swap since ~P becomes redundant*)
val swap = prove_goal cla_thy 
   "~P ==> (~Q ==> P) ==> Q"
 (fn major::prems=>
  [ (resolve_tac [classical] 1),
    (rtac (major RS notE) 1),
    (REPEAT (ares_tac prems 1)) ]);

end;
end;

