(*  Title: 	CTT/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Constructive Type Theory: parsing and printing

Should automatically fold/unfold * and --> to Prod and Sum??
*)


signature CTT_SYNTAX =
sig
  structure Syntax : SYNTAX
  val Aexp: typ
  val Atype: typ
  val const_decs: (string list * typ) list
  val parser: (string*Syntax.lexsy) list -> term * (string*Syntax.lexsy) list
  val printer: Syntax.lexsy Syntax.Symtab.table -> term -> unit   
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;


functor CTT_SyntaxFun
    (structure Syntax: SYNTAX and Pretty: PRETTY) : CTT_SYNTAX = 
struct
structure Syntax = Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax
in

(*Types of expressions, types, judgements*)
val Aexp = Ground "exp";
val Atype = Ground "type";
 

(*Constants and their types*)
val const_decs = 
  [ (*constants for the judgements*)
    (["Type"],                 Atype -->Aprop ),
    (["Eqtype"],       [Atype,Atype]--->Aprop ),
    (["Elem"],         [Aexp, Atype]--->Aprop ),
    (["Eqelem"], [Aexp, Aexp, Atype]--->Aprop ),
    (["reduce"], 	[Aexp, Aexp]--->Aprop ),

    (["N"],		Atype ),
    (["0"],		Aexp ),
    (["succ"],		Aexp-->Aexp ),
    (["rec"],		[Aexp, Aexp, [Aexp,Aexp]--->Aexp] ---> Aexp ),

    (["F","T"],		Atype ),
    (["contr"],		Aexp-->Aexp ),
    (["tt"],		Aexp),

    (["Prod"],		[Atype, Aexp-->Atype] ---> Atype ),
    (["-->"],		[Atype,Atype]--->Atype ),
    (["lambda"],	(Aexp-->Aexp) --> Aexp ),
    (["`"],		[Aexp,Aexp]--->Aexp ),

    (["Sum"],		[Atype, Aexp-->Atype] ---> Atype ),
    (["*"],		[Atype,Atype]--->Atype ),
    (["pair"],		[Aexp,Aexp]--->Aexp ),
    (["split"],		[Aexp, [Aexp,Aexp]--->Aexp] ---> Aexp ),
    (["fst","snd"],	Aexp-->Aexp ),

    (["+"],		[Atype,Atype]--->Atype ),
    (["inl","inr"],	Aexp-->Aexp ),
    (["when"],		[Aexp, Aexp-->Aexp, Aexp-->Aexp] ---> Aexp ),

    (["Eq"],		[Atype,Aexp,Aexp]--->Atype ),
    (["eq"],		Aexp ) ];


(*Get constants*)
val get = Sign.get_const const_decs;


val type_con	= get "Type"
and eqtype_con	= get "Eqtype"
and elem_con	= get "Elem"
and eqelem_con	= get "Eqelem";

val prod   = get "Prod"
and lambda = get "lambda"
and sum    = get "Sum"
and pair   = get "pair";

 
(*Type theory syntax has type operators  PROD x:A. B     SUM x:A. B 
  canonical functions lam x.e,   and pairs <a,b>  *)     
fun parser lxs = parse_functl parsetm 0 lxs
and parse_tyop tyop lxs =
      apfst (fn ((s,T), (A,B)) => tyop $ A $ absfree(s,T,B))
	(((parse_ident thenkey":") thenp 
	 (parser  thenkey"." thenp parser)) lxs)
and parsetm(("[|",DelimSy) :: lxs) = (parse_thm1  thenkey"|]") lxs
  | parsetm(("PROD",DelimSy) :: lxs) = parse_tyop prod lxs
  | parsetm(("SUM",DelimSy)  :: lxs) = parse_tyop sum lxs
  | parsetm(("lam",DelimSy)  :: lxs) = 
      apfst (fn ((s,T), b) => lambda $ absfree(s,T,b))
	((parse_ident thenkey"." thenp parser) lxs)
  | parsetm(("<",DelimSy) :: lxs) =
      apfst (fn (a,b) => pair$a$b)
	((parser  thenkey","  thenp  parser  thenkey">") lxs)
  | parsetm lxs = parserr("Illegal expression", lxs)
and parse_thm1 lxs = 
  let fun parse_judge lxs =
	case  parser lxs  of
	    (A, ("type",DelimSy) :: ls) => (type_con$A, ls)
	  | (a, (":",DelimSy) :: ls) =>
		apfst (fn A => elem_con$a$A) (parser ls)
	  | (a, ("=",DelimSy) :: ls) =>
	      (case  parser ls  of
		   (b, (":",DelimSy) :: ls2) =>
			apfst (fn A => eqelem_con$a$b$A) (parser ls2)
		 | (B, ls2) =>  (eqtype_con$a$B, ls2))
	  | all => all  (*an arbitrary term, not standard syntax*)
  in  parse_judge lxs   end;



(*Printing of Type Theory terms and judgements*)

(*Special treatment of Sum pair Prod lambda 
    pairs are a special case: they are always enclosed in < > *)
fun print1 lextab (outleft,outright) =
  let fun printm t = print1 lextab (0,0) t;  (*abstraction prevents loop!*)
      fun pr_tyop (pre, A, (s,B)) =
	 (Pretty.bg 3;  Pretty.st(pre ^ " " ^ s ^ ":");
	  print1 lextab (5,5)  A;  Pretty.brks "." 1; 
	  print1 lextab (0,0) B;  Pretty.en());
      fun pr_lam (s,b) =
	 (Pretty.bg 3;  Pretty.st("lam " ^ s ^ ". ");
	  print1 lextab (0,0) b;  Pretty.en())
  in fn
      (Const("Type",_)   $ A) => 
        (Pretty.bgs"[| "; printm A;  Pretty.ens" type |]")
    | (Const("Eqtype",_) $ A $ B) =>
	(Pretty.bgs"[| "; printm A;  Pretty.brks " =" 1; 
	 printm B;  Pretty.ens" |]")
    | (Const("Elem",_)   $ a $ A) =>
	(Pretty.bgs"[| "; printm a;  Pretty.brks " :" 1;
	 printm A;  Pretty.ens" |]")
    | (Const("Eqelem",_) $ a $ b $ A) => 
	(Pretty.bgs"[| "; printm a;  Pretty.brks " =" 1;  printm b;  
	 Pretty.brks " :" 1;  printm A;  Pretty.ens" |]")
    | Const("Sum",_)  $ A $ Abs(s,T,B) => 
	  enclose_if (outright>0) pr_tyop ("SUM", A, variant_abs(s,T,B))
    | Const("Prod",_) $ A $ Abs(s,T,B) => 
	  enclose_if (outright>0) pr_tyop ("PROD", A, variant_abs(s,T,B))
    | Const("lambda",_) $ Abs(s,T,b) => 
	  enclose_if (outright>0) pr_lam (variant_abs (s,T,b))
    | Const("pair",_) $ a $ b =>  
	Pretty.list ("<", ">", printm) [a,b]
    | tm => print_functl (print1 lextab, lextab) (outleft,outright) tm
  end;

fun printer tab = print1 tab (0,0);
 

(*Infixes and their precedence*)
val infix_decs = 
    [ (["`"],	"left",  7),
      (["*"],	"right", 5),
      (["+"],	"right", 4),
      (["-->"],	"right", 3) ];


val syn = Syntax.extend pure
	(["type", ":", "=", "<", ">", ".", "PROD", "SUM", "lam"],
	infix_decs, parser, printer); 

(*read/print functions*)
fun read a = Syntax.read syn "?" a;
fun prin t = (Pretty.init();  Syntax.prin syn t;  Pretty.nl());


end;
end;
