(*  Title: 	pcset-syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge

Predicate Calculus + ZF Set Theory
*)



val Aexp  = Ground "exp";
val Ahobj = Ground "hobj";
val Ahyps = Ahobj-->Ahobj;
val Aform = Ground "form";


val pcset_lextab : lexsymbol symbol_table = symtab_of_declist
  [ (*Variables*)
    (["A",  "B",  "C",  "D"],		IdentSy(0, Aexp)),
    (["A'", "B'", "C'", "D'"],		IdentSy(0, Aexp-->Aexp)),
    (["A''", "B''", "C''", "D''"],  	IdentSy(0, [Aexp,Aexp]--->Aexp)),

    (explode"abcdefghpquvwxyz",		IdentSy(0, Aexp)),
    (["a'","b'","c'","d'","e'","f'"],	IdentSy(0, Aexp-->Aexp)),
    (["a''","b''","c''","d''","e''","f''"],
	 				IdentSy(0, [Aexp,Aexp]--->Aexp)),
    (["P",  "Q",  "R",  "S"],		IdentSy(0, Aform)),
    (["P'", "Q'", "R'", "S'"],		IdentSy(0, Aexp-->Aform)),
    (["P''","Q''","R''","S''"], 	IdentSy(0, [Aexp,Aexp]--->Aform)),

    (["G", "H", "HA", "HB"],	IdentSy(0, Ahyps)),

    (*Representation of sequents*)
    (["True"],		ConstSy([Aform,Ahyps]--->Agoal)),
    (["Hypof"],		ConstSy(Aform-->Ahyps)),

    (*Keywords*)
    ([ "{", "}", "<", ">", ".", "~", "ALL", "EXISTS"],	DelimSy),

    (*Infixes*)
    (["`"],	InfixSy([Aexp,Aexp]--->Aexp,	69,	70)) ,	(*left assoc*)

    (["::"],	InfixSy([Aexp,Aexp]--->Aexp,	45,	44))  ,	(*right*)

    (["="],	InfixSy([Aexp,Aexp]--->Aform,	39,	40)) ,	(*left*)
    ([":"],	InfixSy([Aexp,Aexp]--->Aform,	39,	40)) ,	(*left*)

    (["&"],	InfixSy([Aform,Aform]--->Aform,	30,	29)) ,	(*right*)
    (["|"],	InfixSy([Aform,Aform]--->Aform,	20,	19)) ,	(*right*)
    (["==>"],	InfixSy([Aform,Aform]--->Aform,	10,	9))  ,	(*right*)

    (["All","Exists"],	ConstSy((Aexp-->Aform) --> Aform)),

    (["Collect"], 	ConstSy([Aexp, Aexp-->Aform] ---> Aexp)),

    (["Pow","Union"],	ConstSy(Aexp-->Aexp)),

    (["0","N"],		ConstSy(Aexp)),

    (["F"],	ConstSy(Aform)),


    (["empty"],	ConstSy(Aexp)) ];


(*Get constants from table*)

val True	= get_const pcset_lextab "True"
and Hypof	= get_const pcset_lextab "Hypof";

val All		= get_const pcset_lextab "All"
and Exists	= get_const pcset_lextab "Exists"
and imp   	= get_const pcset_lextab "==>"
and F   	= get_const pcset_lextab "F"
and zero   	= get_const pcset_lextab "0"
and setcons   	= get_const pcset_lextab "::"
and Collect   	= get_const pcset_lextab "Collect";


(*PAIR WILL BE A DEFINED INFIX, IS NOT YET IN SYMBOL TABLE!*)
val Pair = Const("Pair", [Aexp,Aexp]--->Aexp);


(*Set notation:  {a,b,c,...}  means   a::b::c::...::0 *)

fun mk_setcons(x,y) = setcons$x$y;

(*Exception unless terminated by 0 *)
fun set_elems (Const("::",_) $ a $ b) = a :: set_elems b
  | set_elems (Const("0",_)) = []
  | set_elems _ = raise match;


(*Make an abstraction over hobj.  For handling the representation of hyps*)
fun abs_hobj t = Abs("hobj", Ahobj, t);

(*Representation of empty hypothesis list*)
val Hempty =  abs_hobj (Bound 0);


(*Allow any bound variable name in PROD, SUM, lam, but not %(...).
  It gets expression arity (Aexp) 
  COPIED FROM tt-syntax *)
fun parse_boundvar (lxs0 as ((s, IdentSy(idx,ary)) :: lxs)) =
      if ary=Aexp then (string_of_xname(s,idx), lxs)
      else  raise parserr with 
          ("Bound variable should have expression arity", lxs0)
  | parse_boundvar ((s, OtherSy) :: lxs) = (s,lxs)
  | parse_boundvar lxs = 
      raise parserr with ("Variable expected", lxs);

 
fun parse_pcset_term1 boundids =
  let fun parse_quant (quant,boundids) lxs =
	let val (id, lxs') = (parse_boundvar thenkey ".") lxs;
	    val (P, lxs'') = parse_pcset_term1 (id::boundids) lxs'
	in (quant $ Abs(id,Aexp,P), lxs'') end;
      fun parsetm (boundids: string list) = fn
	 (("ALL",DelimSy)  :: lxs) => parse_quant (All, boundids) lxs
       | (("EXISTS",DelimSy) :: lxs) => parse_quant (Exists,boundids) lxs
       | (("~",DelimSy) :: lxs) => 
	    let val (P,lxs') = genparse parsetm (35,boundids) lxs
	    in (imp$P$F, lxs') end
       | ("{",DelimSy) :: lxs =>
	    let val (elems, lxs') =
		(parepin (",", parse_pcset_term1 boundids) thenkey"}") lxs
	    in (itlist_right mk_setcons (elems,zero), lxs') end
       | ("<",DelimSy) :: lxs =>
	    let val ((a,b), lxs') =
		(parse_pcset_term1 boundids thenkey","  thenp
		 parse_pcset_term1 boundids  thenkey">") lxs
	    in (Pair$a$b, lxs') end
       | (s,OtherSy) :: lxs => (check_bound(s,0,Aexp,boundids), lxs)
       | lxs => raise parserr with ("Illegal expression", lxs)
  in  genparse parsetm (0,boundids)  end;


val parse_pcset_term = parse_pcset_term1 [];


(*Parse a non-empty hypothesis, either P{,...} or H{,...}  *)
fun parse_hyp lxs = 
  let fun add_assum (P,H) =
	if (arity_of P) = Aform  then  Hypof$P$H  else  P$H;
      val (hyps, lxs') = parepin (",", parse_pcset_term) lxs;
      val hbody = itlist_right add_assum (hyps, Bound 0)
  in  (abs_hobj hbody, lxs')  end;


(* hypothesis ::= "[" hypothesis "]"  |  empty  *)
fun parse_hyp_opt (("[", DelimSy)::lxs) = (parse_hyp thenkey "]") lxs
  | parse_hyp_opt lxs = (Hempty, lxs);


(* theorem ::= judgement hypothesis_opt *)
fun parse_thm lxs : term * lexeme list = 
  let val ((tm,hyp), lxs') = (parse_pcset_term thenp parse_hyp_opt) lxs
  in  (check_arity_term Agoal (True$tm$hyp), lxs')    end;



(*Printing*)

(*Special treatment of Exists Pair All 
    pairs are a special case: they are always enclosed in < > *)
fun print_pcset_term1 lextab (All as (outleft,outright,boundids)) tm =
  let val printm = print_pcset_term1 lextab;
      val default_print = print_functl (printm, lextab) All;
      val variant = bound_variant boundids;
      fun sround pfn = 
	    if outright>0  then  (prs"(";  pfn();  prs")")  else pfn(); 
      fun print_quant (pre,id,P) () =
	 (prs (pre ^ " " ^ id ^ ".");  printm (0, 0, id::boundids) P)
  in case tm of
      Const("All",_) $ Abs(id,_,P) => 
  	sround(print_quant("ALL", variant id, P))
    | Const("Exists",_)  $ Abs(id,_,P) =>
	sround(print_quant("EXISTS",  variant id, P))
    | Const("==>",_)  $ P $ Const("F",_) =>
	(prs"~ ";  printm (35, outright, boundids) P)
    | Const("::",_) $ _ $ _ =>  
	print_list("{", "}", printm (0, 0, boundids)) (set_elems tm)
	handle match => default_print tm
    | Const("Pair",_) $ a $ b =>  
	(prs"<";  
	 printm (0, 0, boundids) a;   prs",";
	 printm (0, 0, boundids) b;   prs">") 
    | _ => default_print tm
  end;

fun print_pcset_term lextab : term->unit = print_pcset_term1 lextab (0, 0, []);


(*Convert term of arity hobj into a list of assumptions,
  stripping outer cAlls to Hypof *)
fun hyps_of_hobj (Const("Hypof",_) $ P $ rand) = P :: hyps_of_hobj rand
  | hyps_of_hobj (H $ rand) = H :: hyps_of_hobj rand
  | hyps_of_hobj (Bound 0) = []
  | hyps_of_hobj tm    = raise term_error with ("Ill-formed hypothesis", [tm]);


fun hyps_of_term (Abs(_,_,body)) = hyps_of_hobj body
  | hyps_of_term H = [H];    (*this case should not occur*)


(*Print hypothesis, empty or not.
  WARNING: the bound variable hobj is printed as B.0 since it is not
	print_pcset_term1 is called with the empty list of bound vars!*)
fun print_hyp lextab [] = ()
  | print_hyp lextab (P::PS) =
        (prs" [ ";  print_pcset_term lextab P;  
	 seq (fn Q => (prs ", ";  print_pcset_term lextab Q)) PS;
	 prs" ] ");


fun print_goal lextab (Const("True",_) $ P $ H) =
      (print_pcset_term lextab P;  print_hyp lextab (hyps_of_term H))
  | print_goal lextab tm = raise term_error with ("print_goal", [tm]);
