(* =========================================================================== *)
(*         Arithmetic in the (Ordered) Field of Real Algebraic Numbers         *)
(*                                                                             *)
(* by G.O.Passmore, Aesthetic Integration Ltd & Clare Hall, Univ. of Cambridge *)
(* Contact:   (e) grant.passmore@cl.cam.ac.uk    (w) www.cl.cam.ac.uk/~gp351/. *)
(* =========================================================================== *)

signature RealAlg =
sig

(* An opaque type of real algebraic numbers *)

type real_alg;

(* A purported real algebraic number is not unique and thus can't be
   represented. *)

exception Non_unique of string;

(* Given a polynomial p and a pair of rationals l <= u, construct a
   representation for a real algebraic number r s.t. p(r) = 0 and r is
   the unique root of p in the real interval [l, u]. *)

val mk_real_alg : Algebra.poly -> Rat.rat -> Rat.rat -> real_alg;

(* Given a rational, construct a real_alg representing it *)

val real_alg_of_rat : Rat.rat -> real_alg;

(* Sign of a real algebraic number *)

val sign : real_alg -> Sturm.sign;

(* Sign of a polynomial at a given real algebraic number *)

val sign_at : Algebra.poly -> real_alg -> Sturm.sign;

(* Negate a real algebraic number (additive inverse) *)

val neg : real_alg -> real_alg;

(* Add two real algebraic numbers *)

val add : real_alg * real_alg -> real_alg;

(* Multiply two real algebraic numbers *)

(* val mult : real_alg * real_alg -> real_alg; *)

(* Compute an interval representation for a real algebraic number *)

val intvl_rep : real_alg -> Algebra.poly * Rat.rat * Rat.rat;

(* Given a list of polynomials ps, compute a complete set of sample
   points from the CAD of R^1 induced by ps. *)

val univ_cad_sample : Algebra.poly list -> real_alg list;

(* String representation of a real algebraic number *)

val toString : real_alg -> string;

end
