\name{typeIIErrorProb}
\alias{typeIIErrorProb}
\alias{typeIIErrorProb.chisqtest}
\alias{typeIIErrorProb.ttest}
\alias{getDoseOutcomes}
\alias{ttest}
\alias{chisqtest}
\alias{getDoseOutcomesControlled}
\alias{getDoseOutcomesUncontrolled}
\alias{print.ttest}
\alias{print.chisqtest}
\alias{typeIIErrorProb.default}
\title{Power calculations}
\description{
  Find the power for various sorts of test.}
\usage{
ttest(n0,n1,alpha=.05)
chisqtest(n0,n1,alpha=.05)
typeIIErrorProb(t,...)
\method{typeIIErrorProb}{chisqtest}(t,effectSize,odds0,odds1=odds0*effectSize,simulate=FALSE,nsamp=1000,...)
\method{typeIIErrorProb}{ttest}(t,effectSize,sd,mean0=0,mean1=effectSize,...)
getDoseOutcomes(ns,effects,sd,alpha=.05,nsamp=1000,...)
}
\arguments{
  \item{n0}{the number of patients in the control group}
  \item{n1}{the number of patients in the treated group}
  \item{alpha}{the size of the test, i.e. the Type I error probability
    that will be permitted}
  \item{t}{a test, either a \code{chisqtest} or a \code{ttest}}
  \item{effectSize}{the effect size (this may be omitted if the
    responses in the two groups are specified by \code{odds1} or \code{mean1})}
  \item{sd}{the standard deviation of patient response, for the t test}
  \item{odds0}{the odds ratio of success in the control group}
  \item{odds1}{the odds ratio of success in the treated group}
  \item{mean0}{the mean outcome in the control group (this may be
    omitted if \code{effectSize} is specified)}
  \item{mean1}{the mean outcome in the treated group (this may be
    omitted if \code{effectSize} is specified)}
  \item{ns}{the number of patients in each arm,
    e.g. \code{ns=c(ctrl=20,dose1=20,dose2=20)}. An uncontrolled test
    (i.e. comparison to historical controls) is indicated by setting the
  number in the control arm to be \code{Inf}.}
  \item{effects}{the effect size in each treated arm,
    e.g. \code{effects=c(dose1=5,dose2=7)}; the arm for which no effect
    size is given is taken to be the control group}
  \item{simulate}{whether to simulate or compute, to find the
    answer. Simulation is more accurate but slower.}
  \item{nsamp}{number of samples taken to produce an estimate by simulation}
  \item{...}{other arguments, not used}
}
\value{
  \code{ttest} and \code{chisqtest} return "test objects", which may be
  printed, but which have no other use. These objects are used as input
  to the power calculations; the function \code{typeIIErrorProb}
  is what actually returns the Type II error probability (i.e. the
  probability of failing to detect an effect when the true effect is non-zero).

  The \code{ttest} is used when there are two
  arms with a numerical response, and we wish to test if the two arms
  have the same mean response; effectSize is the actual difference in response.
  
  The \code{chisqtest} is used when there are two arms with binary
  response, and we wish to test if the odds ratio is the same in each
  arm (the odds ratio is Prob(response=1)/Prob(response=0)).
  
  For trials with multiple arms, use \code{getDoseOutcomes}. This
  function assumes that there is a control arm and several treated arms,
  and we wish to find the lowest effective dose (assuming that doses are
  ordered as in the argument \code{ns}, and correcting for multiple comparisons). If for example
  \code{ns=c(ctrl=20,dose1=20,dose2=20)}, the trial could return the
  answer "no dose is effective" or "dose1 is the lowest effective dose"
  or "dose2 is the lowest effective dose". The function
  \code{getDoseOutcomes} returns a vector showing the probability of
  each of these outcomes.
  }
\author{DJW}
\keyword{htest}
\examples{
# A two-arm trial (n0=150 patients on placebo, n1=150 patients on active)
# Categorize patients according to whether they survive or die.
# Assume that on placebo, Prob(survive)=0.61, and that on treatment
# Prob(survive)=0.78. We find that this test has Prob(typeIIError)=0.82.

odds0 <- 0.61/(1-0.61)
odds1 <- 0.78/(1-0.78)
typeIIErrorProb(chisqtest(n0=150,n1=150),odds0=odds0,effectSize=odds1/odds0)

# A two-arm trial (n0=20 patients on placebo, n1=20 patients on active)
# Score patients in each arm. Assume that the standard deviation of
# response is 6 units, and that the effect size is 4.5 units.
# Then Prob(typeIIError)=0.36.
typeIIErrorProb(ttest(n0=20,n1=20),effectSize=4.5,sd=6)

# Normally we want to recruit enough patients to get
# Prob(typeIIError)<0.2. We find we'd need 29 patients in each arm.
leastN(function(n) typeIIErrorProb(ttest(n,n),effectSize=4.5,sd=6), target=.2)

# Maybe, to get safety data, we want twice as many in the treated arm.
# Then we need 22 control, 44 treated.
leastN(function(n) typeIIErrorProb(ttest(n,2*n),effectSize=4.5,sd=6), target=.2)

# Maybe we've decided to do a trial with 40 patients in each arm, and
# we know the standard deviation of response is 6 units. For different
# effect sizes, what is the probability of Type II error?
library(lattice)
effs <- seq(-1,-5,by=-.1)
t2es <- sapply(effs,
               function(e) typeIIErrorProb(ttest(40,40),e,6))
xyplot(t2es~effs, type="l", ylim=c(0,1),
       xlab='Effect size', ylab='Type II error prob')

# Suppose we've decided to do a trial with three arms, placebo plus
# two treated arms, and we want to know the lowest effective dose.
#
# If both doses are effective, how likely are we to conclude (correctly)
# that "low" is the lowest effective dose? Answer: approx 0.87.
getDoseOutcomes(ns=c(placebo=45,low=30,high=45),
                effects=c(low=-4.5,high=-4.5),
                sd=6,
                alpha=.05,nsamp=200)

# If only the "high" dose is effective, how likely are we to conclude
# (correctly) that "high" is the lowest effective dose? Answer: approx 0.89.
getDoseOutcomes(ns=c(placebo=45,low=30,high=45),
                effects=c(low=0,high=-4.5),
                sd=6,
                alpha=.05,nsamp=200)
}

