# NUMERICAL OPTIMIZATION ROUTINES
# Finds the least n such that fn(n)<=z
# Assumes fn is decreasing
leastN <- function(fn,target=0,init=2,maxiter=12) {
  n0 <- init; f0 <- fn(init)
  n1 <- init; f1 <- f0
  z <- target
  # Range-finding
  while (f0<z && n0>1) { n0 <- floor(n0/2); f0<-fn(n0) }
  iter <- 0
  while (f1>=z && iter<maxiter) { n1<-2*n1; f1<-fn(n1); iter<-iter+1 }
  if (iter==maxiter) {
    warning("Cannot bound solution",call.=TRUE)
    return(n1)
    }
  # Searching
  while (n1-n0>1) {
    nm <- floor((n0+n1)/2); fm <- fn(nm)
    #print(c(n0=n0,nm=nm,n1=n1))
    if (fm>z) { n0<-nm; f0<-fm } else {n1<-nm; f1<-fm}
    }
  if (f0<z) n0 else n1
  }

# ROUTINES FOR SIZE/POWER CALCULATIONS

typeIIErrorProb.default <- function(t,...) cat(paste("Unknown test",class(t),"\n"))
typeIIErrorProb <- function(t,...) UseMethod('typeIIErrorProb',t)

# t-test for equality of means

ttest <- function(n0,n1,alpha=.05) {
  t <- list(n0=n0,n1=n1,alpha=alpha);
  class(t) <- "ttest";
  t
  }
print.ttest <- function(x,...)
  cat(paste("t-test for equality of means","\n",
            "  Number in group0=",x$n0," group1=",x$n1,"; ",
            "Prob(TypeIError)=",x$alpha,"\n",sep=""))


chisqtest <- function(n0,n1,alpha=.05) {
  x <- list(n0=n0,n1=n1,alpha=alpha)
  class(x) <- 'chisqtest'
  x }
print.chisqtest <- function(x,...)
  cat(paste("chisq test for equal proportions, for two groups with binary outcomes","\n",
            "  Number in group0=",x$n0," group1=",x$n1,"; ",
            "Prob(TypeIError)=",x$alpha,"\n",sep=""))

typeIIErrorProb.chisqtest <- function(t,effectSize,odds0,odds1=odds0*effectSize,simulate=FALSE,nsamp=1000,...) {
  x <- t
  if (!simulate) {
    library(pwr)
    p <- x$n0/(x$n0+x$n1)
    h0probs <- c(p*odds0/(1+odds0),p/(1+odds0), (1-p)*odds0/(1+odds0), (1-p)/(1+odds0))
    h1probs <- c(p*odds0/(1+odds0),p/(1+odds0), (1-p)*odds1/(1+odds1), (1-p)/(1+odds1))
    w <- sum((h0probs-h1probs)^2/h0probs)
    res <- pwr.chisq.test(w=w, N=x$n0+x$n1, df=1, sig.level=x$alpha)
    return(1-res$power)
  }
  n0impr <- rbinom(nsamp,x$n0,odds0/(1+odds0))
  n1impr <- rbinom(nsamp,x$n1,odds1/(1+odds1))
  n0wors <- x$n0-n0impr
  n1wors <- x$n1-n1impr
  samp <- rbind(n0impr,n0wors,n1impr,n1wors)
  samp <- array(samp,dim=c(2,2,5000))
  p <- sapply(1:nsamp, function(i) chisq.test(samp[,,i])$p.value)
  length(which(p>x$alpha))/nsamp
  }



typeIIErrorProb.ttest <- function(t,effectSize,sd,mean0=0,mean1=effectSize,...) {
  df <- t$n0+t$n1-2 # degrees of freedom for t-distribution
  ncp <- (mean0-mean1) / sqrt(sd^2*(1/t$n0+1/t$n1))
                                        # non-centrality par, for H1=true
  crit <- -qt(t$alpha/2,df) # reject H0 if |T|>crit
  p <- pt(crit,df,ncp)-pt(-crit,df,ncp)
  p
  }





# Given the number in each wing of the test, and the drug effects,
# estimate the probability of each of the three outcomes
# ns = vector giving number of patients in each wing
# effs = vector of effect size for each wing
# sd = standard deviation of effect, assumed the same in each wing
# alpha = TypeIErrorProb
# nsamp = number of samples to take, 2*sd(answer)=1/sqrt(nsamp)

getDoseOutcomes <- function(ns, effects, sd,alpha=.05, nsamp=1000,...) {
  require(multcomp)
  ctrlname <- names(ns)[!(names(ns) %in% names(effects))]
  dosenames <- names(ns)[names(ns) %in% names(effects)]
  if (length(ctrlname)!=1) stop("What is the control group?")
  if (length(dosenames)!=length(effects)) stop("What are the effects?")
  if (ns[ctrlname]==Inf)
    getDoseOutcomesUncontrolled(ns=ns[is.finite(ns)],effects=effects,sd=sd,alpha=alpha,nsamp=nsamp,...)
  else
    getDoseOutcomesControlled(ns=ns,effects=effects,sd=sd,alpha=alpha,nsamp=nsamp,...)
  }

getDoseOutcomesControlled <- function(ns, effects, sd, alpha=.05, nsamp=1000) {
  # Given a sample of responses for three treatments,
  # decide which is the lowest effective dose (and also return p-values)
  # Assumes df is a data frame with r=response, treat=ordered factor,
  # lowest factor level is the control.
  findEffDose <- function(df, alpha=.05) {
    fit <- lm(r~treat, data=df)
    res <- summary(glht(fit, linfct=mcp(treat='Dunnett')), test=adjusted(type='Westfall'))
    pvals <- res$test$pvalues
    names(pvals) <- names(coef(res))
    levnames <- levels(df$treat)
    compnames <- paste(levnames[-1],'-',levnames[1])
    pvals <- pvals[compnames]
    succ <- c(pvals<alpha,neither=TRUE)
    list(pvals=pvals,outcome=which.max(succ))
  }
ctrlname <- names(ns)[!(names(ns) %in% names(effects))]
  dosenames <- names(ns)[names(ns) %in% names(effects)]
  df <- data.frame(r=rep(0,sum(ns)),
                   treat=factor(rep(names(ns),ns),
                     levels=c(ctrlname,dosenames),ordered=TRUE))
  meaneff <- c(rep(0,ns[ctrlname]),rep(effects[dosenames],ns[dosenames]))
  outcomes <- rep(0,length(ns))
  ntot <- sum(ns)
  names(outcomes) <- c(dosenames,"nothing")
  for (i in 1:nsamp) {
    if (i %% 25 == 0) { cat(i); cat(".") }
    z <- rnorm(ntot,sd=sd)
    df$r <- z+meaneff
    o <- findEffDose(df,alpha=alpha)$outcome
    outcomes[o] <- outcomes[o]+1
    }
  cat("\n")
  outcomes/sum(outcomes)
  }

getDoseOutcomesUncontrolled <- function(ns, effects, sd, alpha=.05, nsamp=1000) {
  findEffDoseUncontrolled <- function(df, alpha=.05) {
    contrmat <- diag(length(levels(df$treat)))
    dimnames(contrmat) <- list(NULL,NULL)
    dimnames(contrmat)[[2]] <- paste('treat',levels(df$treat),sep='')
    dimnames(contrmat)[[1]] <- paste('eff',levels(df$treat),sep='')
    fit <- lm(r~treat, data=df)
    res <- summary(glht(fit, linfct=contrmat), test=adjusted(type='Westfall'))
    pvals <- res$test$pvalues
    names(pvals) <- names(coef(res))
    levnames <- levels(df$treat)
    compnames <- paste('eff',levels(df$treat),sep='')
    pvals <- pvals[compnames]
    succ <- c(pvals<alpha,neither=TRUE)
    list(pvals=pvals,outcome=which.max(succ))
  }
  dosenames <- names(ns)
  df <- data.frame(r=rep(0,sum(ns)),
                   treat=factor(rep(dosenames,ns[dosenames]),levels=dosenames))
  meaneff <- rep(effects[dosenames],ns[dosenames])
  outcomes <- rep(0,length(dosenames)+1)
  ntot <- sum(ns)
  names(outcomes) <- c(dosenames,"nothing")
  for (i in 1:nsamp) {
    if (i %% 25 == 0) { cat(i); cat(".") }
    z <- rnorm(ntot,sd=sd)
    df$r <- z+meaneff
    o <- findEffDoseUncontrolled(df,alpha=alpha)$outcome
    outcomes[o] <- outcomes[o]+1
    }
  cat("\n")
  outcomes/sum(outcomes)
  }



