//
// Rent_eval_2d - (C) 2023 DJ Greaves
//
// This program, coded in FSharp, helps answer an end-of-chapter exercise in Modern System-on-Chip Design on Arm by DJ Greaves.
// https://www.cl.cam.ac.uk/~djg11/pubs/modern-soc-design-djg
//
// Note that a naive fractal model, with homogeneous wiring and regular structure at each level with
// give a Rent exponent of unity, since it is self-similar between levels. The key thing to include,
// to achieve the lower exponents encountered in real-world designs (that have some design/engineering
// behind them) is the ratio of terminal contacts to gates needs to get smaller as the number of gates
// gets larger (see The Interpretation and Application of Rent’s Rule Christie& Stroobandt 2000). This
// can be programmed and the exponent will emerge, but in the example code, the Rent formula is used
// explicitly with parameters rent_alpha and rent_exponent.  Hence the average number of contacts to a
// component and the number of nets are emergent (given an average fan out figure).  The model
// averages over 20 random runs for each test and the normalised standard deviation (ie. divided by
// the mean) experienced across runs is reported as one confidence indication.  These figures are all
// in the range 10 to 20 percent, which is find for this exercise.  The program then perturbs each of
// its numeric assumptions by 5 percent to explore the partial derivatives of the assumptions.


let map = List.map
let fold = List.fold
let app = List.iter

let qprintln (ss:string) = System.Console.Write(ss + "\n")

// The form of a component in a component tree.  
type rent_module_t =
    {
       area:         double          // Summative of all area below, of course. 
       n_gates:          int         // Ditto.
       n_contacts:       int
       n_local_nets:     int
       children:         rent_module_t list
       total_net_length: double         
    }

type parameters_t =
    {
       average_no_children:     double   // Average number of child components to a non-leaf level.
       leaf_area_variation:     double   // Variation in area of a leaf component.
       rent_alpha:              double
       rent_exponent:           double
       average_net_fanout:      double   // Each gate output feeds this number of inputs.
       area_swell:              double   // Relative additional local area
       netbeta:                 double   // Average relative distance a net makes inside a hierarchic level
       relative_deviation_amplitude: double // Amount of variation between designs. Randomness/non-uniformness in component synthesis.
    }

let plot_fd = None // Some(System.IO.File.CreateText "/tmp/plotdata.dat")
                  
let prg_state = ref(uint64 0)

let prg_reseed() = prg_state := (uint64 123455678)
prg_reseed()

// Linear congruential pseudo random generator (mmix by D Knuth).
let int64_prg_core () =
    let a = 6364136223846793005UL
    let c = 1442695040888963407UL
    let r = !prg_state * a + c
    prg_state := r
    r/32UL

// Integer and real user random functions.
let int64_prg (range:uint64) = (int64_prg_core() % range)
let real_prg (range:double) =
    let digitrange = 1000000UL
    double(int64_prg digitrange) * range / (double digitrange)

let int_prg (range:int32) = int32(int64_prg_core()) % range

// Add a random relative offset to a value
let perturb relative_deviation_amplitude arg =
    let deviation_amplitude = relative_deviation_amplitude * arg
    let offset = real_prg (deviation_amplitude *2.0) - deviation_amplitude
    let ans = arg + offset
    if ans < 0.0 then
        qprintln(sprintf "Negative peturb result: arg=%A  pram=%A offset=%A" arg  relative_deviation_amplitude offset) 
        arg
    else ans
    

// Generate a random heirarcic design: all branches have same depth which is somewhat unrealistic.
let rec gen_tree prams level id_ =
    let r = perturb prams.relative_deviation_amplitude 
    //qprintln(sprintf "Start %i-%i" level id_)
    if level <= 0 then
        let area = perturb prams.leaf_area_variation 1.0
        { area=area; n_contacts= int_prg(3)+2; children= []; n_local_nets=0; total_net_length=0.0; n_gates=1 }
    else
        let n_children = int(r prams.average_no_children)
        let children = map (gen_tree prams (level-1)) [0..n_children-1]
        let n_gates = List.fold (fun c son -> c+son.n_gates) 0 children // Total number of gates below (none at this level)
        let child_contacts_NCC = List.fold (fun c son -> c+son.n_contacts) 0 children // The number of child contacts.
        let child_area =  List.fold (fun c son -> c+son.area) 0.0 children  // Add up area of children
        let area = child_area + r (prams.area_swell * child_area) // Increase by a local swell if wiring-density-limited.
        let n_contacts_NFC = int (r(prams.rent_alpha * exp(prams.rent_exponent * log(double n_gates) + 0.5))) // Direct form Rent Rule.


        let vb = level > 8
        // Start simultaneous construction
        // Roughly speaking, if each net goes to fanout(=1.3) destinations and sources. This is local fanout and does not included fanout above or below.
        // Eq1: The total number of terminal contacts involved NTC=child_contacts_NCC + formal_NFC exported to the level above.
        // Eq2: The total net count at this level is NNC=(total_contact_count=NTC)/(1+fanout)
        // Ignoring a second-order effect where an external net is also a local net, the local nets is local_net_ratio*NNET. The bonded_net_ratio_bnr = 1-local_net_ratio
        // Eq3: The exported count, formal_NFC, is one per non-local net, hence NFC = NNET*bnr
        
        // Combining Eq1 and Eq3 removes NFC giving: NTC=NNET*bnr+NCC
        // Plugging into Eq2: NNETS =  (NNET*bnr+NCC) + NCC. NNET = (NNET*bnr+NCC)/(1+fanout). NNET*(1-bnr/(1+fanout)) = NCC/(1+fanout).
        //let (one_plus_fanout, bnr) = (1.0 + r prams.average_net_fanout, 1.0 - prams.local_net_ratio)
        //let netcount_NNET = (double child_contacts_NCC/one_plus_fanout) / (1.0-bnr/one_plus_fanout)
        //let n_contacts_NFC_l = int(netcount_NNET * bnr + 0.5)
        // End simultaneous construction

        let n_power_nets = 4 // Clock and power and reset etc.
        let netcount_NNET = int((double(n_contacts_NFC + child_contacts_NCC)/(1.0+prams.average_net_fanout))+0.5) + n_power_nets
        if (double netcount_NNET *1.1 < double n_contacts_NFC) then // All ten percent overspill before complaining!
            qprintln(sprintf "Too few nets for the contacts: level=%i n_gates=%i NNETs=%i < n_formals=%i. (child_contacts_NCC=%i)" level n_gates netcount_NNET n_contacts_NFC child_contacts_NCC)
        
        //if (id_ = 1 && plot_fd <>None) then (valOf plot_fd).Write(sprintf "%i %f %f\n" n_gates area (double n_contacts_NFC))

        if (vb) then qprintln(sprintf "Level %i  n_children=%i n_formal_contacts=%i n_local_nets=%i area=%A" level n_children n_contacts_NFC (int netcount_NNET) area)
        // We assume nets from our contacts to a child's contact have no length at this level, so the wiring
        // length contribution is just from local interconnect.

        let local_net_length = r (double netcount_NNET * prams.netbeta * sqrt area) // Average net length at this level is netbeta (about half) sqrt(area)
        let child_net_length = List.fold (fun c son -> c+son.total_net_length) 0.0 children
        if (vb) then qprintln(sprintf "Level %i-%i: local_net_length=%A child_net_length=%A  total=%A" level id_ local_net_length child_net_length (local_net_length+child_net_length))
        let tnl = local_net_length + child_net_length // Totals
        let ans = { area=area; n_contacts=n_contacts_NFC; children=children; n_local_nets=netcount_NNET; total_net_length=tnl; n_gates=n_gates }
        ans




let g_assumption_names = [ "-none-"; "rent_alpha"; "rent_exponent"; "average_net_fanout"; "average_no_children"; "area_swell"; "netbeta"; "leaf_area_variation"; "relative_deviation_amplitude"]

let rez_prams offset_name offset_rel_amount =
    let dd name nominal =
        if not (List.exists (fun x->x=name) g_assumption_names) then qprintln(sprintf "!!! Bad parameter name %s" name)
        if name=offset_name then nominal*offset_rel_amount else nominal
    let prams =
        {
           rent_exponent=          dd "rent_exponent" 0.65
           rent_alpha=             dd "rent_alpha"   0.2
           average_net_fanout=     dd "average_net_fanout"  1.3    // Each gate output feeds this number of inputs.
           area_swell=             dd "area_swell"  0.15           // Relative additional local area (eg for density dominated design)
           netbeta=                dd "netbeta"  0.2               // Average relative distance a net makes inside a hierarchic level Assume well-placed (random placement gives higher).
           leaf_area_variation=    dd "leaf_area_variation"  (sqrt 3.0)/2.0     // Leaf cell average variation is about 3 to 1
           average_no_children=    dd "average_no_children"  6.0                // Count of children to a non-leaf level
           relative_deviation_amplitude= dd "relative_deviation_amplitude" 0.25 // Amount of variation between components for all other aspects
        }
    prams
    


let n_hierarchy_levels = 8 // N=8.  The number of levels in our design. All branches have the same depth, but this could be randomised if you like.

let run_a_trial prams trial_no_  =
    let circuit = gen_tree prams n_hierarchy_levels 0
    let analyse_circuit circuit =
        qprintln(sprintf "Totals:  area=%.2f    net_length=%.2f"  circuit.area circuit.total_net_length)
        (circuit.area, circuit.total_net_length)
    let results = analyse_circuit circuit
    results

let run_trials offset_amount offset_name  =
    let prams = rez_prams offset_name offset_amount
    prg_reseed()
    let n_trials = 20 // Average the result over this number of trials.
    let trials = [0..n_trials-1]
    let (areas, netlengths) = List.unzip(map (run_a_trial prams) trials)
    (double (List.length areas), areas, netlengths)


let run_groups offset_amount offset_name =
    let (n_trials, areas, netlengths) = run_trials offset_amount offset_name
    let anal title values =
        let m1 = List.fold (fun c x -> x+c) 0.0 values
        let m2 = List.fold (fun c x -> x*x+c) 0.0 values
        let mean = m1/n_trials
        let deviation = sqrt(m2/n_trials - mean*mean)
        qprintln(sprintf "   %s '%s'  Mean %.3g:  Normalised standard_deviation %.3g" title offset_name mean (deviation/mean))
        mean
    let mean_area = anal "Area" areas
    let mean_netlength = anal "Wiring length" netlengths
    (mean_area, mean_netlength)


let explore_derivative baseline offset_amount offset_name =
    let (mean_area, mean_netlength) = run_groups offset_amount offset_name
    let report_derivative title base_value peturbed_value =
        let sensitivity = (peturbed_value - base_value)/base_value / (offset_amount - 1.0)
        qprintln(sprintf "   %s  %s  derivative=%.3g" offset_name title sensitivity)
        sensitivity
    let s_area = report_derivative "Area" (fst baseline) mean_area
    let s_wl = report_derivative "Wiring length" (snd baseline) mean_netlength
    qprintln (sprintf "Latex: %s & %0.2g & %0.2g \\\\" offset_name s_area s_wl)
    
let explore_derivatives () =
    qprintln "Commence baseline model generation."
    let baseline = run_groups 1.0 "-none-"
    qprintln("Baseline design profiled.")
    let offset_amount = 1.05  // A five percent perturbation to measure partial derivative.
    app (explore_derivative baseline offset_amount) g_assumption_names
    ()


//==============================
// Entry point
explore_derivatives()

// eof
