theory Wasm imports Main "$ISABELLE_HOME/src/HOL/Library/Sublist" "Wasm_Type_Abs/Wasm_Type_Abs" begin

datatype 'a ne_list =
  Base 'a
  | Conz 'a "'a ne_list"

definition option_projl :: "('a \<times> 'b) option \<Rightarrow> 'a option" where
  "option_projl x = (case x of Some (a, b) \<Rightarrow> Some a | None \<Rightarrow> None)"

definition option_projr :: "('a \<times> 'b) option \<Rightarrow> 'b option" where
  "option_projr x = (case x of Some (a, b) \<Rightarrow> Some b | None \<Rightarrow> None)"
      
primrec to_list :: "'a ne_list \<Rightarrow> 'a list" where
  "to_list (Base a) = [a]" | "to_list (Conz a l) = (a # (to_list l))"

lemma ne_list_last_length:
  assumes "lis = to_list isne"
  shows "lis \<noteq> []"
  using assms
proof (induction arbitrary: lis)
  case (Base x)
  thus ?case
    unfolding to_list.simps
    by fastforce
next
  case (Conz x1a x2)
  thus ?case
    unfolding to_list.simps
    by fastforce
qed

lemma ne_list_last_exists:
  assumes "lis = to_list isne"
  shows "last lis  = (lis!(length lis - 1))"
  using assms
proof (induction arbitrary: lis)
  case (Base x)
  thus ?case
    unfolding to_list.simps
    by fastforce
next
  case (Conz x1a x2)
  thus ?case
    unfolding to_list.simps
    by fastforce
qed
  
type_synonym (* immediate *)
  i = nat
type_synonym (* static offset *)
  off = nat
type_synonym (* alignment exponent *)
  a = nat
  
(* type_synonym b (* byte *) = "8 word" *)

datatype --"value types"
  t = T_i32 | T_i64 | T_f32 | T_f64

datatype --"packed types"
  tp = Tp_i8 | Tp_i16 | Tp_i32

definition t_length :: "t \<Rightarrow> nat" where
 "t_length t = (case t of
                   T_i32 \<Rightarrow> 4
                 | T_i64 \<Rightarrow> 8
                 | T_f32 \<Rightarrow> 4
                 | T_f64 \<Rightarrow> 8)"

definition tp_length :: "tp \<Rightarrow> nat" where
 "tp_length t = (case t of
                 Tp_i8 \<Rightarrow> 1
               | Tp_i16 \<Rightarrow> 2
               | Tp_i32 \<Rightarrow> 4)"

definition is_int_t :: "t \<Rightarrow> bool" where
 "is_int_t t = (case t of
                   T_i32 \<Rightarrow> True
                 | T_i64 \<Rightarrow> True
                 | T_f32 \<Rightarrow> False
                 | T_f64 \<Rightarrow> False)"  

definition is_float_t :: "t \<Rightarrow> bool" where
 "is_float_t t = (case t of
                   T_i32 \<Rightarrow> False
                 | T_i64 \<Rightarrow> False
                 | T_f32 \<Rightarrow> True
                 | T_f64 \<Rightarrow> True)"
 
lemma int_float_disjoint: "is_int_t t = -(is_float_t t)"
  unfolding is_int_t_def is_float_t_def
  by (cases t, auto)

datatype (* mutability *)
  mut = T_immut | T_mut

record tg = (* global types *)
  tg_mut :: mut
  tg_t :: t

definition is_mut :: "tg \<Rightarrow> bool" where
  "is_mut tg = (tg_mut tg = T_mut)"

datatype --"function types"
  tf = Tf "t list" "t list" ("_ '_> _" 60)

(* TYPING *)
record t_context =
  types_t :: "tf list"
  func_t :: "tf list"
  global :: "tg list"
  table :: "nat option"
  memory :: "nat option"
  local :: "t list"
  label :: "(t list) list"
  return :: "(t list) option"

record s_context =
  s_inst :: "t_context list"
  s_funcs :: "tf list"
  s_tab  :: "nat list"
  s_mem  :: "nat list"
  s_globs :: "tg list"

datatype
  sx = S | U

datatype
  unop_i = Clz | Ctz | Popcnt

datatype
  unop_f = Neg | Abs | Ceil | Floor | Trunc | Nearest | Sqrt
  
datatype
  binop_i = Add | Sub | Mul | Div sx | Rem sx | And | Or | Xor | Shl | Shr sx | Rotl | Rotr

datatype
  binop_f = Addf | Subf | Mulf | Divf | Min | Max | Copysign
  
datatype
  testop = Eqz
  
datatype
  relop_i = Eq | Ne | Lt sx | Gt sx | Le sx | Ge sx
  
datatype
  relop_f = Eqf | Nef | Ltf | Gtf | Lef | Gef
  
datatype
  cvtop = Convert | Reinterpret

definition app_unop_i :: "unop_i \<Rightarrow> 'i::wasm_int \<Rightarrow> 'i::wasm_int" where
  "app_unop_i iop c =
     (case iop of
     Ctz \<Rightarrow> int_ctz c
   | Clz \<Rightarrow> int_clz c
   | Popcnt \<Rightarrow> int_popcnt c)"

definition app_unop_f :: "unop_f \<Rightarrow> 'f::wasm_float \<Rightarrow> 'f::wasm_float" where
  "app_unop_f fop c =
                 (case fop of
                    Neg \<Rightarrow> float_neg c
                  | Abs \<Rightarrow> float_abs c
                  | Ceil \<Rightarrow> float_ceil c
                  | Floor \<Rightarrow> float_floor c
                  | Trunc \<Rightarrow> float_trunc c
                  | Nearest \<Rightarrow> float_nearest c
                  | Sqrt \<Rightarrow> float_sqrt c)"

definition app_binop_i :: "binop_i \<Rightarrow> 'i::wasm_int \<Rightarrow> 'i::wasm_int \<Rightarrow> ('i::wasm_int) option" where
  "app_binop_i iop c1 c2 = (case iop of
                              Add \<Rightarrow> int_add c1 c2
                            | Sub \<Rightarrow> int_sub c1 c2
                            | Mul \<Rightarrow> int_mul c1 c2
                            | Div U \<Rightarrow> int_div_u c1 c2
                            | Div S \<Rightarrow> int_div_s c1 c2
                            | Rem U \<Rightarrow> int_rem_u c1 c2
                            | Rem S \<Rightarrow> int_rem_s c1 c2
                            | And \<Rightarrow> int_and c1 c2
                            | Or \<Rightarrow> int_or c1 c2
                            | Xor \<Rightarrow> int_xor c1 c2
                            | Shl \<Rightarrow> int_shl c1 c2
                            | Shr U \<Rightarrow> int_shr_u c1 c2
                            | Shr S \<Rightarrow> int_shr_s c1 c2
                            | Rotl \<Rightarrow> int_rotl c1 c2
                            | Rotr \<Rightarrow> int_rotr c1 c2)"

definition app_binop_f :: "binop_f \<Rightarrow> 'f::wasm_float \<Rightarrow> 'f::wasm_float \<Rightarrow> ('f::wasm_float) option" where
  "app_binop_f fop c1 c2 = (case fop of
                              Addf \<Rightarrow> Some (float_add c1 c2)
                            | Subf \<Rightarrow> Some (float_sub c1 c2)
                            | Mulf \<Rightarrow> Some (float_mul c1 c2)
                            | Divf \<Rightarrow> Some (float_div c1 c2)
                            | Min \<Rightarrow> Some (float_min c1 c2)
                            | Max \<Rightarrow> Some (float_max c1 c2)
                            | Copysign \<Rightarrow> Some (float_copysign c1 c2))"

definition app_testop_i :: "testop \<Rightarrow> 'i::wasm_int \<Rightarrow> bool" where
  "app_testop_i testop c = (case testop of Eqz \<Rightarrow> int_eqz c)"

definition app_relop_i :: "relop_i \<Rightarrow> 'i::wasm_int \<Rightarrow> 'i::wasm_int \<Rightarrow> bool" where
  "app_relop_i rop c1 c2 = (case rop of
                              Eq \<Rightarrow> int_eq c1 c2
                            | Ne \<Rightarrow> int_ne c1 c2
                            | Lt U \<Rightarrow> int_lt_u c1 c2
                            | Lt S \<Rightarrow> int_lt_s c1 c2
                            | Gt U \<Rightarrow> int_gt_u c1 c2
                            | Gt S \<Rightarrow> int_gt_s c1 c2
                            | Le U \<Rightarrow> int_le_u c1 c2
                            | Le S \<Rightarrow> int_le_s c1 c2
                            | Ge U \<Rightarrow> int_ge_u c1 c2
                            | Ge S \<Rightarrow> int_ge_s c1 c2)"

definition app_relop_f :: "relop_f \<Rightarrow> 'f::wasm_float \<Rightarrow> 'f::wasm_float \<Rightarrow> bool" where
  "app_relop_f rop c1 c2 = (case rop of
                              Eqf \<Rightarrow> float_eq c1 c2
                            | Nef \<Rightarrow> float_ne c1 c2
                            | Ltf \<Rightarrow> float_lt c1 c2
                            | Gtf \<Rightarrow> float_gt c1 c2
                            | Lef \<Rightarrow> float_le c1 c2
                            | Gef \<Rightarrow> float_ge c1 c2)" 

datatype ('a,'b,'c,'d) (* values *)
  v =
    ConstInt32 'a
    | ConstInt64 'b
    | ConstFloat32 'c
    | ConstFloat64 'd

definition typeof :: "('a, 'b, 'c, 'd) v \<Rightarrow> t" where
  "typeof v = (case v of
                 ConstInt32 _ \<Rightarrow> T_i32
               | ConstInt64 _ \<Rightarrow> T_i64
               | ConstFloat32 _ \<Rightarrow> T_f32
               | ConstFloat64 _ \<Rightarrow> T_f64)"

definition types_agree :: "t \<Rightarrow> ('a, 'b, 'c, 'd) v \<Rightarrow> bool" where
  "types_agree t v = (typeof v = t)"
  
datatype ('a,'b,'c,'d) (* basic instructions *)
  b_e =
    Unreachable
    | Nop
    | Drop
    | Select
    | Block tf "(('a,'b,'c,'d) b_e) list"
    | Loop tf "(('a,'b,'c,'d) b_e) list"
    | If tf "(('a,'b,'c,'d) b_e) list" "(('a,'b,'c,'d) b_e) list"
    | Br i
    | Br_if i
    | Br_table "i ne_list"
    | Return
    | Call i
    | Call_indirect i
    | Get_local i
    | Set_local i
    | Tee_local i
    | Get_global i
    | Set_global i
    | Load t "(tp \<times> sx) option" a off
    | Store t "tp option" a off
    | Current_memory
    | Grow_memory
    | EConst "('a,'b,'c,'d) v" ("C _" 60)
    | Unop_i t unop_i
    | Unop_f t unop_f
    | Binop_i t binop_i
    | Binop_f t binop_f
    | Testop t testop
    | Relop_i t relop_i
    | Relop_f t relop_f
    | Cvtop t cvtop t "sx option"

datatype ('a,'b,'c,'d,'host) cl = --"function closures"
  Func_native i tf "t list" "(('a,'b,'c,'d) b_e) list"
| Func_host tf 'host

definition cl_type :: "('a,'b,'c,'d,'host) cl \<Rightarrow> tf" where
  "cl_type cl = (case cl of Func_native _ tf _ _ \<Rightarrow> tf | Func_host tf _ \<Rightarrow> tf)"
  
(* begin instances *)
record  ('a,'b,'c,'d,'host) inst =
  types :: "tf list"
  funcs :: "i list"
  tab :: "i option"
  mem :: "i option"
  globs :: "i list"

type_synonym ('a,'b,'c,'d,'host) tabinst = "((('a,'b,'c,'d,'host) cl) option) list"
  
(* type_synonym meminst = "b list" *)
(* end instances *)

record ('a,'b,'c,'d) global =
  g_mut :: mut
  g_val :: "('a,'b,'c,'d) v"

definition rglob_is_mut :: "('a,'b,'c,'d) global \<Rightarrow> bool" where
  "rglob_is_mut g = (g_mut g = T_mut)"

record  ('a,'b,'c,'d,'meminst,'host) s =(* store *)
  inst :: "(('a,'b,'c,'d,'host) inst) list"
  funcs :: "(('a,'b,'c,'d,'host) cl) list"
  tab :: "(('a,'b,'c,'d,'host) tabinst) list"
  mem :: "'meminst list"
  globs :: "(('a,'b,'c,'d) global) list"

definition stypes :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> tf" where
  "stypes s i j = ((types ((inst s)!i))!j)"
  
definition sfunc_ind :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat" where
  "sfunc_ind s i j = ((inst.funcs ((inst s)!i))!j)"

definition sfunc :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> ('a,'b,'c,'d,'host) cl" where
  "sfunc s i j = (funcs s)!(sfunc_ind s i j)"

definition sglob_ind :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat" where
  "sglob_ind s i j = ((inst.globs ((inst s)!i))!j)"
  
definition sglob :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> ('a,'b,'c,'d) global" where
  "sglob s i j = (globs s)!(sglob_ind s i j)"

definition sglob_val :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> ('a,'b,'c,'d) v" where
  "sglob_val s i j = g_val (sglob s i j)"

definition stab_s :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> (('a,'b,'c,'d,'host) cl) option" where
  "stab_s s i j = (let stabinst = ((tab s)!i) in  (if (length (stabinst) > j) then (stabinst!j) else None))"

definition stab :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> (('a,'b,'c,'d,'host) cl) option" where
  "stab s i j = (case (inst.tab ((inst s)!i)) of Some k => stab_s s k j | None => None)"

lemma stab_unfold:
  assumes "stab s i j = Some cl"
  shows "\<exists>k. inst.tab ((inst s)!i) = Some k \<and> length ((tab s)!k) > j \<and>((tab s)!k)!j = Some cl"
proof -
  obtain k where have_k:"(inst.tab ((inst s)!i)) = Some k"
    using assms
    unfolding stab_def
    by fastforce
  hence s_o:"stab s i j = stab_s s k j"
    using assms
    unfolding stab_def
    by simp
  then obtain stabinst where stabinst_def:"stabinst = ((tab s)!k)"
    by blast
  hence "stab_s s k j = (stabinst!j) \<and> (length stabinst > j)"
    using assms s_o
    unfolding stab_s_def
    by (cases "(length stabinst > j)", auto)
  thus ?thesis
    using have_k stabinst_def assms s_o
    by auto
qed

definition supdate_glob_s :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> ('a,'b,'c,'d) v \<Rightarrow> ('a,'b,'c,'d,'e,'host) s" where
  "supdate_glob_s s k v = s\<lparr>globs := (globs s)[k:=((globs s)!k)\<lparr>g_val := v\<rparr>]\<rparr>"

definition supdate_glob :: "('a,'b,'c,'d,'e,'host) s \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> ('a,'b,'c,'d) v \<Rightarrow> ('a,'b,'c,'d,'e,'host) s" where
  "supdate_glob s i j v = (let k = sglob_ind s i j in supdate_glob_s s k v)"

datatype ('a,'b,'c,'d,'host) e (* administrative operators *) =
  Basic "('a,'b,'c,'d) b_e" ("$_" 60)
  | Trap
  | Callcl "('a,'b,'c,'d,'host) cl"
  | Label nat "(('a,'b,'c,'d,'host) e) list" "(('a,'b,'c,'d,'host) e) list"
  | Local nat i "(('a,'b,'c,'d) v) list" "(('a,'b,'c,'d,'host) e) list"

lemma inj_basic: "inj Basic"
  by (meson e.inject(1) injI)

lemma inj_basic_econst: "inj (\<lambda>v. $C v)"
  by (meson b_e.inject(16) e.inject(1) injI)
    
definition is_const :: "(('a,'b,'c,'d,'host) e) \<Rightarrow> bool" where
  "is_const e = (case e of Basic (C _) \<Rightarrow> True | _ \<Rightarrow> False)"
    
definition const_list :: "(('a,'b,'c,'d,'host) e) list \<Rightarrow> bool" where
  "const_list xs = list_all is_const xs"
  
datatype ('a,'b,'c,'d,'host)
  Lholed =
    --"L0 = v* [<hole>] e*"
    LBase "(('a,'b,'c,'d,'host) e) list" "(('a,'b,'c,'d,'host) e) list"
    --"L(i+1) = v* (label n {e* } Li) e*"
  | LRec "(('a,'b,'c,'d,'host) e) list" nat "(('a,'b,'c,'d,'host) e) list" "('a,'b,'c,'d,'host) Lholed" "(('a,'b,'c,'d,'host) e) list"

locale wasm_memory_base =
  fixes mem_size :: "'mem \<Rightarrow> nat"
begin
  inductive store_extension :: "('a,'b,'c,'d,'mem,'host) s \<Rightarrow> ('a,'b,'c,'d,'mem,'host) s \<Rightarrow> bool" where
  "\<lbrakk>insts = insts'; fs = fs'; tclss = tclss'; list_all2 (\<lambda>bs bs'. mem_size bs \<le> mem_size bs') bss bss'; gs = gs'\<rbrakk> \<Longrightarrow>
    store_extension \<lparr>s.inst = insts, s.funcs = fs, s.tab = tclss, s.mem = bss, s.globs = gs\<rparr>
                      \<lparr>s.inst = insts', s.funcs = fs', s.tab = tclss', s.mem = bss', s.globs = gs'\<rparr>"
end

locale wasm_host_base =
  fixes host_itself :: "'host itself"

locale wasm_typing =
  wasm_memory_base mem_size +
  wasm_host_base host_itself
  for mem_size :: "'mem \<Rightarrow> nat"
  and host_itself :: "'host itself" +
  (* basic types *)
  fixes a_itself :: "'a itself"
    and b_itself :: "'b itself"
    and c_itself :: "'c itself"
    and d_itself :: "'d itself"
begin

abbreviation to_e_list :: "(('a,'b,'c,'d) b_e) list \<Rightarrow> (('a,'b,'c,'d,'host) e) list" ("$* _" 60) where
  "to_e_list b_es \<equiv> map Basic b_es"

lemma to_e_list_1:"[$ a] = $* [a]"
  by simp

lemma to_e_list_2:"[$ a, $ b] = $* [a, b]"
  by simp

lemma to_e_list_3:"[$ a, $ b, $ c] = $* [a, b, c]"
  by simp

abbreviation v_to_e_list :: "(('a,'b,'c,'d) v) list \<Rightarrow> (('a,'b,'c,'d,'host) e) list" ("$$* _" 60) where
  "v_to_e_list ves \<equiv> map (\<lambda>v. $C v) ves"

lemma v_exists_b_e:"\<exists>ves. ($$*vs) = ($*ves)"
proof (induction vs)
  case (Cons a vs)
  thus ?case
  by (metis list.simps(9))
qed auto

  (* Lfilled depth thing-to-fill fill-with result *)
inductive Lfilled :: "nat \<Rightarrow>('a,'b,'c,'d,'host) Lholed \<Rightarrow> (('a,'b,'c,'d,'host) e) list \<Rightarrow> (('a,'b,'c,'d,'host) e) list \<Rightarrow> bool" where
  (* "Lfill (LBase vs es') es = vs @ es @ es'" *)
  L0:"\<lbrakk>const_list vs; lholed = (LBase vs es')\<rbrakk> \<Longrightarrow> Lfilled 0 lholed es (vs @ es @ es')"
  (* "Lfill (LRec vs ts es' l es'') es = vs @ [Label ts es' (Lfill l es)] @ es''" *)
| LN:"\<lbrakk>const_list vs; lholed = (LRec vs n es' l es''); Lfilled k l es lfilledk\<rbrakk> \<Longrightarrow> Lfilled (k+1) lholed es (vs @ [Label n es' lfilledk] @ es'')"

  (* Lfilled depth thing-to-fill fill-with result *)
inductive Lfilled_exact :: "nat \<Rightarrow>('a,'b,'c,'d,'host) Lholed \<Rightarrow> (('a,'b,'c,'d,'host) e) list \<Rightarrow> (('a,'b,'c,'d,'host) e) list \<Rightarrow> bool" where
  (* "Lfill (LBase vs es') es = vs @ es @ es'" *)
  L0:"\<lbrakk>lholed = (LBase [] [])\<rbrakk> \<Longrightarrow> Lfilled_exact 0 lholed es es"
  (* "Lfill (LRec vs ts es' l es'') es = vs @ [Label ts es' (Lfill l es)] @ es''" *)
| LN:"\<lbrakk>const_list vs; lholed = (LRec vs n es' l es''); Lfilled_exact k l es lfilledk\<rbrakk> \<Longrightarrow> Lfilled_exact (k+1) lholed es (vs @ [Label n es' lfilledk] @ es'')"

lemma Lfilled_exact_imp_Lfilled:
  assumes "Lfilled_exact n lholed es LI"
  shows "Lfilled n lholed es LI"
  using assms
proof (induction rule: Lfilled_exact.induct)
  case (L0 lholed es)
  thus ?case
    using const_list_def Lfilled.intros(1)
    by fastforce
next
  case (LN vs lholed n es' l es'' k es lfilledk)
  thus ?case
    using Lfilled.intros(2)
    by fastforce
qed

lemma Lfilled_exact_app_imp_exists_Lfilled:
  assumes "const_list ves"
          "Lfilled_exact n lholed (ves@es) LI"
  shows "\<exists>lholed'. Lfilled n lholed' es LI"
  using assms(2,1)
proof (induction "(ves@es)" LI rule: Lfilled_exact.induct)
  case (L0 lholed)
  show ?case
    using Lfilled.intros(1)[OF L0(2), of _ "[]"]
    by fastforce
next
  case (LN vs lholed n es' l es'' k lfilledk)
  thus ?case
    using Lfilled.intros(2)
    by fastforce
qed

lemma Lfilled_imp_exists_Lfilled_exact:
  assumes "Lfilled n lholed es LI"
  shows "\<exists>lholed' ves es_c. const_list ves \<and> Lfilled_exact n lholed' (ves@es@es_c) LI"
  using assms Lfilled_exact.intros
  by (induction rule: Lfilled.induct) fastforce+

(* TYPING RELATION *)
definition load_store_t_bounds :: "a \<Rightarrow> tp option \<Rightarrow> t \<Rightarrow> bool" where
  "load_store_t_bounds a tp t = (case tp of
                                   None \<Rightarrow> 2^a \<le> t_length t
                                 | Some tp \<Rightarrow> 2^a \<le> tp_length tp \<and> tp_length tp < t_length t \<and>  is_int_t t)"

inductive b_e_typing :: "[t_context, (('a,'b,'c,'d) b_e) list, tf] \<Rightarrow> bool" ("_ \<turnstile> _ : _" 60) where
  --"num ops"
  "\<C> \<turnstile> [C v]         : ([]    _> [(typeof v)])"
| "is_int_t t   \<Longrightarrow> \<C> \<turnstile> [Unop_i t _]  : ([t]   _> [t])"
| "is_float_t t \<Longrightarrow> \<C> \<turnstile> [Unop_f t _]  : ([t]   _> [t])"
| "is_int_t t   \<Longrightarrow> \<C> \<turnstile> [Binop_i t _] : ([t,t] _> [t])"
| "is_float_t t \<Longrightarrow> \<C> \<turnstile> [Binop_f t _] : ([t,t] _> [t])"
| "is_int_t t   \<Longrightarrow> \<C> \<turnstile> [Testop t _]  : ([t]   _> [T_i32])"
| "is_int_t t   \<Longrightarrow> \<C> \<turnstile> [Relop_i t _] : ([t,t] _> [T_i32])"
| "is_float_t t \<Longrightarrow> \<C> \<turnstile> [Relop_f t _] : ([t,t] _> [T_i32])"
  --"convert"
| "\<lbrakk>(t1 \<noteq> t2); (sx = None) = ((is_float_t t1 \<and> is_float_t t2) \<or> (is_int_t t1 \<and> is_int_t t2 \<and> (t_length t1 < t_length t2)))\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Cvtop t1 Convert t2 sx] : ([t2] _> [t1])"
  --"reinterpret"
| "\<lbrakk>(t1 \<noteq> t2); t_length t1 = t_length t2\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Cvtop t1 Reinterpret t2 None] : ([t2] _> [t1])"
  --"unreachable, nop, drop, select"
| "\<C> \<turnstile> [Unreachable] : (ts _> ts')"
| "\<C> \<turnstile> [Nop] : ([] _> [])"
| "\<C> \<turnstile> [Drop] : ([t] _> [])"
| "\<C> \<turnstile> [Select] : ([t,t,T_i32] _> [t])"
  --"block"
| "\<lbrakk>tf = (tn _> tm); \<C>\<lparr>label := ([tm] @ (label \<C>))\<rparr> \<turnstile> es : (tn _> tm)\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Block tf es] : (tn _> tm)"
  --"loop"
| "\<lbrakk>tf = (tn _> tm); \<C>\<lparr>label := ([tn] @ (label \<C>))\<rparr> \<turnstile> es : (tn _> tm)\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Loop tf es] : (tn _> tm)"
  --"if then else"
| "\<lbrakk>tf = (tn _> tm); \<C>\<lparr>label := ([tm] @ (label \<C>))\<rparr> \<turnstile> es1 : (tn _> tm); \<C>\<lparr>label := ([tm] @ (label \<C>))\<rparr> \<turnstile> es2 : (tn _> tm)\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [If tf es1 es2] : (tn @ [T_i32] _> tm)"
  --"br"
| "\<lbrakk>i < length(label \<C>); (label \<C>)!i = ts\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Br i] : (t1s @ ts _> t2s)"
  --"br_if"
| "\<lbrakk>i < length(label \<C>); (label \<C>)!i = ts\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Br_if i] : (ts @ [T_i32] _> ts)"
  --"br_table"
| "\<lbrakk>to_list is = ils; list_all (\<lambda>i. i < length(label \<C>) \<and> (label \<C>)!i = ts) ils\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Br_table is] : (t1s @ ts @ [T_i32] _> t2s)"
  --"return"
| "\<lbrakk>(return \<C>) = Some ts\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Return] : (t1s @ ts _> t2s)"
  --"call"
| "\<lbrakk>i < length(func_t \<C>); (func_t \<C>)!i = tf\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Call i] : tf"
  --"call_indirect"
| "\<lbrakk>i < length(types_t \<C>); (types_t \<C>)!i = (t1s _> t2s); (table \<C>) \<noteq> None\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Call_indirect i] : (t1s @ [T_i32] _> t2s)"
  --"get_local"
| "\<lbrakk>i < length(local \<C>); (local \<C>)!i = t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Get_local i] : ([] _> [t])"
  --"set_local"
| "\<lbrakk>i < length(local \<C>); (local \<C>)!i = t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Set_local i] : ([t] _> [])"
  --"tee_local"
| "\<lbrakk>i < length(local \<C>); (local \<C>)!i = t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Tee_local i] : ([t] _> [t])"
  --"get_global"
| "\<lbrakk>i < length(global \<C>); tg_t ((global \<C>)!i) = t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Get_global i] : ([] _> [t])"
  --"set_global"
| "\<lbrakk>i < length(global \<C>); tg_t ((global \<C>)!i) = t; is_mut ((global \<C>)!i)\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Set_global i] : ([t] _> [])"
  --"load"
| "\<lbrakk>(memory \<C>) \<noteq> None; load_store_t_bounds a (option_projl tp_sx) t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Load t tp_sx a off] : ([T_i32] _> [t])"
  --"store"
| "\<lbrakk>(memory \<C>) \<noteq> None; load_store_t_bounds a tp t\<rbrakk> \<Longrightarrow> \<C> \<turnstile> [Store t tp a off] : ([T_i32,t] _> [])"
  --"current_memory"
| "(memory \<C>) \<noteq> None \<Longrightarrow> \<C> \<turnstile> [Current_memory] : ([] _> [T_i32])"
  --"Grow_memory"
| "(memory \<C>) \<noteq> None \<Longrightarrow> \<C> \<turnstile> [Grow_memory] : ([T_i32] _> [T_i32])"
  --"empty program"
| "\<C> \<turnstile> [] : ([] _> [])"
  --"composition"
| "\<lbrakk>\<C> \<turnstile> es : (t1s _> t2s); \<C> \<turnstile> [e] : (t2s _> t3s)\<rbrakk> \<Longrightarrow> \<C> \<turnstile> es @ [e] : (t1s _> t3s)"
  --"weakening"
| "\<C> \<turnstile> es : (t1s _> t2s) \<Longrightarrow> \<C> \<turnstile> es : (ts @ t1s _> ts @ t2s)"

inductive cl_typing :: "[s_context, ('a,'b,'c,'d,'host) cl, tf] \<Rightarrow> bool" where
   "\<lbrakk>i < length (s_inst \<S>); ((s_inst \<S>)!i) = \<C>; tf = (t1s _> t2s); \<C>\<lparr>local := (local \<C>) @ t1s @ ts, label := ([t2s] @ (label \<C>)), return := Some t2s\<rparr> \<turnstile> es : ([] _> t2s)\<rbrakk> \<Longrightarrow> cl_typing \<S> (Func_native i tf ts es) (t1s _> t2s)"
|  "cl_typing \<S> (Func_host tf h) tf"

(* lifting the b_e_typing relation to the administrative operators *)
inductive e_typing :: "[s_context, t_context, (('a,'b,'c,'d,'host) e) list, tf] \<Rightarrow> bool" ("_\<bullet>_ \<turnstile> _ : _" 60)
and       s_typing :: "[s_context, (t list) option, nat, (('a,'b,'c,'d) v) list, (('a,'b,'c,'d,'host) e) list, t list] \<Rightarrow> bool" ("_\<bullet>_ \<tturnstile>'_ _ _;_ : _" 60) where
(* section: e_typing *)
  (* lifting *)
  "\<C> \<turnstile> b_es : tf \<Longrightarrow> \<S>\<bullet>\<C> \<turnstile> $*b_es : tf"
  (* composition *)
| "\<lbrakk>\<S>\<bullet>\<C> \<turnstile> es : (t1s _> t2s); \<S>\<bullet>\<C> \<turnstile> [e] : (t2s _> t3s)\<rbrakk> \<Longrightarrow> \<S>\<bullet>\<C> \<turnstile> es @ [e] : (t1s _> t3s)"
  (* weakening *)
| "\<S>\<bullet>\<C> \<turnstile> es : (t1s _> t2s) \<Longrightarrow>\<S>\<bullet>\<C> \<turnstile> es : (ts @ t1s _> ts @ t2s)"
  (* trap *)
| "\<S>\<bullet>\<C> \<turnstile> [Trap] : tf"
  (* local *)
| "\<lbrakk>\<S>\<bullet>Some ts \<tturnstile>_i vs;es : ts; length ts = n\<rbrakk> \<Longrightarrow> \<S>\<bullet>\<C> \<turnstile> [Local n i vs es] : ([] _> ts)"
  (* callcl *)
| "cl_typing \<S> cl tf \<Longrightarrow> \<S>\<bullet>\<C>  \<turnstile> [Callcl cl] : tf"
  (* label *)
| "\<lbrakk>\<S>\<bullet>\<C> \<turnstile> e0s : (ts _> t2s); \<S>\<bullet>\<C>\<lparr>label := ([ts] @ (label \<C>))\<rparr> \<turnstile> es : ([] _> t2s); length ts = n\<rbrakk> \<Longrightarrow> \<S>\<bullet>\<C> \<turnstile> [Label n e0s es] : ([] _> t2s)"
(* section: s_typing *)
| "\<lbrakk>i < (length (s_inst \<S>)); tvs = map typeof vs; \<C> =((s_inst \<S>)!i)\<lparr>local := (local ((s_inst \<S>)!i) @ tvs), return := rs\<rparr>; \<S>\<bullet>\<C> \<turnstile> es : ([] _> ts); (rs = Some ts) \<or> rs = None\<rbrakk> \<Longrightarrow> \<S>\<bullet>rs \<tturnstile>_i vs;es : ts"

definition "globi_agree gs n g = (n < length gs \<and> gs!n = g)"

definition "funci_agree fs n f = (n < length fs \<and> fs!n = f)"

inductive inst_typing :: "[s_context, (('a,'b,'c,'d,'host) inst), t_context] \<Rightarrow> bool" where
  "\<lbrakk>list_all2 (funci_agree (s_funcs \<S>)) fs tfs(*; list_all2 (cl_typing \<S>) cls tfs *); list_all2 (globi_agree (s_globs \<S>)) gs tgs; (i = Some i' \<and> i' < length (s_tab \<S>) \<and> (s_tab \<S>)!i' = (the n)) \<or> (i = None \<and> n = None); (j = Some j' \<and> m = Some m' \<and> (s_mem \<S>)!j' = m') \<or> j = None \<and> m = None\<rbrakk> \<Longrightarrow> inst_typing \<S> \<lparr>types = ts, funcs = fs, tab = i, mem = j, globs = gs\<rparr> \<lparr>types_t = ts, func_t = tfs, global = tgs, table = n, memory = m, local = [], label = [], return = None\<rparr>"

definition "glob_agree g tg = (tg_mut tg = g_mut g \<and> tg_t tg = typeof (g_val g))"

definition "tab_agree \<S> tcl = (case tcl of None \<Rightarrow> True | Some cl \<Rightarrow> \<exists>tf. cl_typing \<S> cl tf)"

inductive store_typing :: "[(('a,'b,'c,'d,'mem,'host) s), s_context] \<Rightarrow> bool" where
  "\<lbrakk>\<S> = \<lparr>s_inst = \<C>s, s_funcs = tfs, s_tab = ns, s_mem = ms, s_globs = tgs\<rparr>; list_all2 (inst_typing \<S>) insts \<C>s; list_all2 (cl_typing \<S>) fs tfs; list_all (tab_agree \<S>) (concat tclss); list_all2 (\<lambda> tcls n. n \<le> length tcls) tclss ns; list_all2 (\<lambda> bs m. m \<le> mem_size bs) bss ms; list_all2 glob_agree gs tgs\<rbrakk> \<Longrightarrow> store_typing \<lparr>s.inst = insts, s.funcs = fs, s.tab = tclss, s.mem = bss, s.globs = gs\<rparr> \<S>"

inductive config_typing :: "[nat, (('a,'b,'c,'d,'mem,'host) s), (('a,'b,'c,'d) v) list, (('a,'b,'c,'d,'host) e) list, t list] \<Rightarrow> bool" ("\<turnstile>'_ _ _;_;_ : _" 60) where
  "\<lbrakk>store_typing s \<S>; \<S>\<bullet>None \<tturnstile>_i vs;es : ts\<rbrakk> \<Longrightarrow> \<turnstile>_i s;vs;es : ts "
end

locale wasm_memory =
  wasm_memory_base mem_size
  for mem_size :: "'mem \<Rightarrow> nat" +
  fixes mem_grow :: "'mem \<Rightarrow> nat \<Rightarrow> 'mem option"
    and load :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> nat \<Rightarrow> 'bits option"
    and store :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> 'bits \<Rightarrow> nat \<Rightarrow> 'mem option"
    and load_packed :: "sx \<Rightarrow> 'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> nat \<Rightarrow> 'bits option"
    and store_packed :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> 'bits \<Rightarrow> nat \<Rightarrow> 'mem option"
  assumes mem_grow_mono_size:"\<And>m n m'. (mem_grow m n = Some m') \<Longrightarrow> mem_size m \<le> mem_size m'"
      and store_size:"\<And>m n off v l m'. (store m n off v l = Some m') \<Longrightarrow> mem_size m = mem_size m'"
      and store_packed_size:"\<And>m n off v l m'. (store_packed m n off v l = Some m') \<Longrightarrow> mem_size m = mem_size m'"
begin
  definition smem_ind :: "('a, 'b, 'c, 'd, 'mem, 'host) s \<Rightarrow> nat \<Rightarrow> nat option" where
  "smem_ind s i = (inst.mem ((inst s)!i))"
end

locale wasm_host =
  wasm_memory_base mem_size +
  wasm_host_base host_itself
  for mem_size :: "'mem \<Rightarrow> nat"
  and host_itself :: "'h itself" +
  fixes host_apply :: "('a,'b,'c,'d,'mem,'h) s \<Rightarrow> tf \<Rightarrow> 'h \<Rightarrow> ('a,'b,'c,'d) v list \<Rightarrow>((('a,'b,'c,'d,'mem,'h) s) \<times> (('a,'b,'c,'d) v list)) option"
  assumes host_apply_preserve_store:" list_all2 types_agree t1s vs \<Longrightarrow> host_apply s tf f vs = Some (s', vs') \<Longrightarrow> store_extension s s'"
  assumes host_apply_respect_type:"list_all2 types_agree t1s vs \<Longrightarrow> host_apply s (t1s _> t2s) f vs = Some (s', vs') \<Longrightarrow> list_all2 types_agree t2s vs'"

locale wasm_execution =
   (* memory implementation *)
  wasm_memory mem_size mem_grow load store load_packed store_packed +
   (* basic types *)
  wasm_typing mem_size host_function_type int32 int64 float32 float64 +
    (* host functions *)
  wasm_host mem_size host_function_type host_apply
  for int32 :: "'i32::wasm_int itself"
  and int64 :: "'i64::wasm_int itself"
  and float32 :: "'f32::wasm_float itself"
  and float64 :: "'f64::wasm_float itself"
  and mem_size :: "'mem \<Rightarrow> nat"
  and mem_grow :: "'mem \<Rightarrow> nat \<Rightarrow> 'mem option"
  and load :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> nat \<Rightarrow> 'bits option"
  and store :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> 'bits \<Rightarrow> nat \<Rightarrow> 'mem option"
  and load_packed :: "sx \<Rightarrow> 'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> nat \<Rightarrow> 'bits option"
  and store_packed :: "'mem \<Rightarrow> nat \<Rightarrow> off \<Rightarrow> 'bits \<Rightarrow> nat \<Rightarrow> 'mem option"
  and host_function_type :: "'host itself"
  and host_apply :: "('i32,'i64,'f32,'f64,'mem,'host) s \<Rightarrow> tf \<Rightarrow> 'host \<Rightarrow> ('i32,'i64,'f32,'f64) v list \<Rightarrow> ((('i32,'i64,'f32,'f64,'mem,'host) s) \<times> ('i32,'i64,'f32,'f64) v list) option"
 +
  (* inter-type conversions *)
    (* float to i32 *)
  fixes ui32_trunc_f32 :: "'f32 \<Rightarrow> 'i32 option"
    and si32_trunc_f32 :: "'f32 \<Rightarrow> 'i32 option"
    and ui32_trunc_f64 :: "'f64 \<Rightarrow> 'i32 option"
    and si32_trunc_f64 :: "'f64 \<Rightarrow> 'i32 option"
    (* float to i64 *)
    and ui64_trunc_f32 :: "'f32 \<Rightarrow> 'i64 option"
    and si64_trunc_f32 :: "'f32 \<Rightarrow> 'i64 option"
    and ui64_trunc_f64 :: "'f64 \<Rightarrow> 'i64 option"
    and si64_trunc_f64 :: "'f64 \<Rightarrow> 'i64 option"
    (* int to f32 *)
    and f32_convert_ui32 :: "'i32 \<Rightarrow> 'f32"
    and f32_convert_si32 :: "'i32 \<Rightarrow> 'f32"
    and f32_convert_ui64 :: "'i64 \<Rightarrow> 'f32"
    and f32_convert_si64 :: "'i64 \<Rightarrow> 'f32"
    (* int to f64 *)
    and f64_convert_ui32 :: "'i32 \<Rightarrow> 'f64"
    and f64_convert_si32 :: "'i32 \<Rightarrow> 'f64"
    and f64_convert_ui64 :: "'i64 \<Rightarrow> 'f64"
    and f64_convert_si64 :: "'i64 \<Rightarrow> 'f64"
    (* intra-{int/float} conversions *)
    and wasm_wrap :: "'i64 \<Rightarrow> 'i32"
    and wasm_extend_u :: "'i32 \<Rightarrow> 'i64"
    and wasm_extend_s :: "'i32 \<Rightarrow> 'i64"
    and wasm_demote :: "'f64 \<Rightarrow> 'f32"
    and wasm_promote :: "'f32 \<Rightarrow> 'f64"
    (* boolean encoding *)
    and serialise_i32 :: "'i32 \<Rightarrow> 'bits"
    and serialise_i64 :: "'i64 \<Rightarrow> 'bits"
    and serialise_f32 :: "'f32 \<Rightarrow> 'bits"
    and serialise_f64 :: "'f64 \<Rightarrow> 'bits"
    and wasm_deserialise :: "'bits \<Rightarrow> t \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v"
    and wasm_bool :: "bool \<Rightarrow> 'i32"
    and int32_minus_one :: 'i32    
  assumes wasm_deserialise_type:"\<And>bs t. typeof (wasm_deserialise bs t) = t"
begin

definition cvt_i32 :: "sx option \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v \<Rightarrow> 'i32 option" where
  "cvt_i32 sx v = (case v of
                   ConstInt32 c \<Rightarrow> None
                 | ConstInt64 c \<Rightarrow> Some (wasm_wrap c)
                 | ConstFloat32 c \<Rightarrow> (case sx of
                                      Some U \<Rightarrow> ui32_trunc_f32 c
                                    | Some S \<Rightarrow> si32_trunc_f32 c
                                    | None \<Rightarrow> None)
                 | ConstFloat64 c \<Rightarrow> (case sx of
                                      Some U \<Rightarrow> ui32_trunc_f64 c
                                    | Some S \<Rightarrow> si32_trunc_f64 c
                                    | None \<Rightarrow> None))"

definition cvt_i64 :: "sx option \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v \<Rightarrow> 'i64 option" where
  "cvt_i64 sx v = (case v of
                   ConstInt32 c \<Rightarrow> (case sx of
                                      Some U \<Rightarrow> Some (wasm_extend_u c)
                                    | Some S \<Rightarrow> Some (wasm_extend_s c)
                                    | None \<Rightarrow> None)
                 | ConstInt64 c \<Rightarrow> None
                 | ConstFloat32 c \<Rightarrow> (case sx of
                                      Some U \<Rightarrow> ui64_trunc_f32 c
                                    | Some S \<Rightarrow> si64_trunc_f32 c
                                    | None \<Rightarrow> None)
                 | ConstFloat64 c \<Rightarrow> (case sx of
                                      Some U \<Rightarrow> ui64_trunc_f64 c
                                    | Some S \<Rightarrow> si64_trunc_f64 c
                                    | None \<Rightarrow> None))"

definition cvt_f32 :: "sx option \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v \<Rightarrow> 'f32 option" where
  "cvt_f32 sx v = (case v of
                   ConstInt32 c \<Rightarrow> (case sx of
                                    Some U \<Rightarrow> Some (f32_convert_ui32 c)
                                  | Some S \<Rightarrow> Some (f32_convert_si32 c)
                                  | _ \<Rightarrow> None)
                 | ConstInt64 c \<Rightarrow> (case sx of
                                    Some U \<Rightarrow> Some (f32_convert_ui64 c)
                                  | Some S \<Rightarrow> Some (f32_convert_si64 c)
                                  | _ \<Rightarrow> None)
                 | ConstFloat32 c \<Rightarrow> None
                 | ConstFloat64 c \<Rightarrow> Some (wasm_demote c))"

definition cvt_f64 :: "sx option \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v \<Rightarrow> 'f64 option" where
  "cvt_f64 sx v = (case v of
                   ConstInt32 c \<Rightarrow> (case sx of
                                    Some U \<Rightarrow> Some (f64_convert_ui32 c)
                                  | Some S \<Rightarrow> Some (f64_convert_si32 c)
                                  | _ \<Rightarrow> None)
                 | ConstInt64 c \<Rightarrow> (case sx of
                                    Some U \<Rightarrow> Some (f64_convert_ui64 c)
                                  | Some S \<Rightarrow> Some (f64_convert_si64 c)
                                  | _ \<Rightarrow> None)
                 | ConstFloat32 c \<Rightarrow> Some (wasm_promote c)
                 | ConstFloat64 c \<Rightarrow> None)"

definition cvt :: "t \<Rightarrow> sx option \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v \<Rightarrow> (('i32, 'i64, 'f32, 'f64) v) option" where
  "cvt t sx v = (case t of
                 T_i32 \<Rightarrow> (case (cvt_i32 sx v) of Some c \<Rightarrow> Some (ConstInt32 c) | None \<Rightarrow> None)
               | T_i64 \<Rightarrow> (case (cvt_i64 sx v) of Some c \<Rightarrow> Some (ConstInt64 c) | None \<Rightarrow> None) 
               | T_f32 \<Rightarrow> (case (cvt_f32 sx v) of Some c \<Rightarrow> Some (ConstFloat32 c) | None \<Rightarrow> None)
               | T_f64 \<Rightarrow> (case (cvt_f64 sx v) of Some c \<Rightarrow> Some (ConstFloat64 c) | None \<Rightarrow> None))"

definition bits :: "('i32, 'i64, 'f32, 'f64) v \<Rightarrow> 'bits" where
  "bits v = (case v of
               ConstInt32 c \<Rightarrow> (serialise_i32 c)
             | ConstInt64 c \<Rightarrow> (serialise_i64 c)
             | ConstFloat32 c \<Rightarrow> (serialise_f32 c)
             | ConstFloat64 c \<Rightarrow> (serialise_f64 c))"

  (*
fun bits_as_bytes :: "bool list \<Rightarrow> b list" where
  "bits_as_bytes [] = []" | "bits_as_bytes bl = (bits_as_bytes (drop 8 bl)) @ [of_bl (take 8 bl)]"
 *)

definition bitzero :: "t \<Rightarrow> ('i32, 'i64, 'f32, 'f64) v" where
  "bitzero t = (case t of
                T_i32 \<Rightarrow> ConstInt32 0
              | T_i64 \<Rightarrow> ConstInt64 0
              | T_f32 \<Rightarrow> ConstFloat32 0
              | T_f64 \<Rightarrow> ConstFloat64 0)"

definition n_zeros :: "t list \<Rightarrow> (('i32, 'i64, 'f32, 'f64) v) list" where
  "n_zeros ts = (map (\<lambda>t. bitzero t) ts)"

lemma n_zeros_typeof:
  "n_zeros ts = vs \<Longrightarrow> (ts = map typeof vs)"
proof (induction ts arbitrary: vs)
  case Nil
  thus ?case
    unfolding n_zeros_def
    by simp
next
  case (Cons a ts)
  obtain vs' where "n_zeros ts = vs'"
    using n_zeros_def
    by blast
  moreover
  have "typeof (bitzero a) = a"
    unfolding typeof_def bitzero_def
    by (cases a, simp_all)
  ultimately
  show ?case
    using Cons
    unfolding n_zeros_def
    by auto
qed

inductive reduce_simple :: "[((('i32, 'i64, 'f32, 'f64,'host) e) list), ((('i32, 'i64, 'f32, 'f64,'host) e) list)] \<Rightarrow> bool" ("\<lparr>_\<rparr> \<leadsto> \<lparr>_\<rparr>" 60) where
  --"integer unary ops"
  unop_i32:"\<lparr>[$C (ConstInt32 c), $(Unop_i T_i32 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 (app_unop_i iop c))]\<rparr>"
| unop_i64:"\<lparr>[$C (ConstInt64 c), $(Unop_i T_i64 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt64 (app_unop_i iop c))]\<rparr>"
  --"float unary ops"
| unop_f32:"\<lparr>[$C (ConstFloat32 c), $(Unop_f T_f32 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstFloat32 (app_unop_f fop c))]\<rparr>"
| unop_f64:"\<lparr>[$C (ConstFloat64 c), $(Unop_f T_f64 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstFloat64 (app_unop_f fop c))]\<rparr>"
  --"int32 binary ops"
| binop_i32_Some:"\<lbrakk>app_binop_i iop c1 c2 = (Some c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt32 c1), $C (ConstInt32 c2), $(Binop_i T_i32 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 c)]\<rparr>"
| binop_i32_None:"\<lbrakk>app_binop_i iop c1 c2 = None\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt32 c1), $C (ConstInt32 c2), $(Binop_i T_i32 iop)]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"int64 binary ops"
| binop_i64_Some:"\<lbrakk>app_binop_i iop c1 c2 = (Some c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt64 c1), $C (ConstInt64 c2), $(Binop_i T_i64 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt64 c)]\<rparr>"
| binop_i64_None:"\<lbrakk>app_binop_i iop c1 c2 = None\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt64 c1), $C (ConstInt64 c2), $(Binop_i T_i64 iop)]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"float32 binary ops"
| binop_f32_Some:"\<lbrakk>app_binop_f fop c1 c2 = (Some c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstFloat32 c1), $C (ConstFloat32 c2), $(Binop_f T_f32 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstFloat32 c)]\<rparr>"
| binop_f32_None:"\<lbrakk>app_binop_f fop c1 c2 = None\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstFloat32 c1), $C (ConstFloat32 c2), $(Binop_f T_f32 fop)]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"float64 binary ops"
| binop_f64_Some:"\<lbrakk>app_binop_f fop c1 c2 = (Some c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstFloat64 c1), $C (ConstFloat64 c2), $(Binop_f T_f64 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstFloat64 c)]\<rparr>"
| binop_f64_None:"\<lbrakk>app_binop_f fop c1 c2 = None\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstFloat64 c1), $C (ConstFloat64 c2), $(Binop_f T_f64 fop)]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"testops"
| testop_i32:"\<lparr>[$C (ConstInt32 c), $(Testop T_i32 testop)]\<rparr> \<leadsto> \<lparr>[$C ConstInt32 (wasm_bool (app_testop_i testop c))]\<rparr>"
| testop_i64:"\<lparr>[$C (ConstInt64 c), $(Testop T_i64 testop)]\<rparr> \<leadsto> \<lparr>[$C ConstInt32 (wasm_bool (app_testop_i testop c))]\<rparr>"
  --"int relops"
| relop_i32:"\<lparr>[$C (ConstInt32 c1), $C (ConstInt32 c2), $(Relop_i T_i32 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 (wasm_bool (app_relop_i iop c1 c2)))]\<rparr>"
| relop_i64:"\<lparr>[$C (ConstInt64 c1), $C (ConstInt64 c2), $(Relop_i T_i64 iop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 (wasm_bool (app_relop_i iop c1 c2)))]\<rparr>"
  --"float relops"
| relop_f32:"\<lparr>[$C (ConstFloat32 c1), $C (ConstFloat32 c2), $(Relop_f T_f32 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 (wasm_bool (app_relop_f fop c1 c2)))]\<rparr>"
| relop_f64:"\<lparr>[$C (ConstFloat64 c1), $C (ConstFloat64 c2), $(Relop_f T_f64 fop)]\<rparr> \<leadsto> \<lparr>[$C (ConstInt32 (wasm_bool (app_relop_f fop c1 c2)))]\<rparr>"
  --"convert"
| convert_Some:"\<lbrakk>types_agree t1 v; cvt t2 sx v = (Some v')\<rbrakk> \<Longrightarrow> \<lparr>[$(C v), $(Cvtop t2 Convert t1 sx)]\<rparr> \<leadsto> \<lparr>[$(C v')]\<rparr>"
| convert_None:"\<lbrakk>types_agree t1 v; cvt t2 sx v = None\<rbrakk> \<Longrightarrow> \<lparr>[$(C v), $(Cvtop t2 Convert t1 sx)]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"reinterpret"
| reinterpret:"types_agree t1 v \<Longrightarrow> \<lparr>[$(C v), $(Cvtop t2 Reinterpret t1 None)]\<rparr> \<leadsto> \<lparr>[$(C (wasm_deserialise (bits v) t2))]\<rparr>"
  --"unreachable"
| unreachable:"\<lparr>[$ Unreachable]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"nop"
| nop:"\<lparr>[$ Nop]\<rparr> \<leadsto> \<lparr>[]\<rparr>"
  --"drop"
| drop:"\<lparr>[$(C v), ($ Drop)]\<rparr> \<leadsto> \<lparr>[]\<rparr>"
  --"select"
| select_false:"int_eq n 0 \<Longrightarrow> \<lparr>[$(C v1), $(C v2), $C (ConstInt32 n), ($ Select)]\<rparr> \<leadsto> \<lparr>[$(C v2)]\<rparr>"
| select_true:"int_ne n 0 \<Longrightarrow> \<lparr>[$(C v1), $(C v2), $C (ConstInt32 n), ($ Select)]\<rparr> \<leadsto> \<lparr>[$(C v1)]\<rparr>"
  --"block"
| block:"\<lbrakk>const_list vs; length vs = n; length t1s = n; length t2s = m\<rbrakk> \<Longrightarrow> \<lparr>vs @ [$(Block (t1s _> t2s) es)]\<rparr> \<leadsto> \<lparr>[Label m [] (vs @ ($* es))]\<rparr>"
  --"loop"
| loop:"\<lbrakk>const_list vs; length vs = n; length t1s = n; length t2s = m\<rbrakk> \<Longrightarrow> \<lparr>vs @ [$(Loop (t1s _> t2s) es)]\<rparr> \<leadsto> \<lparr>[Label n [$(Loop (t1s _> t2s) es)] (vs @ ($* es))]\<rparr>"
  --"if"
| if_false:"int_eq n 0 \<Longrightarrow> \<lparr>[$C (ConstInt32 n), $(If tf e1s e2s)]\<rparr> \<leadsto> \<lparr>[$(Block tf e2s)]\<rparr>"
| if_true:"int_ne n 0 \<Longrightarrow> \<lparr>[$C (ConstInt32 n), $(If tf e1s e2s)]\<rparr> \<leadsto> \<lparr>[$(Block tf e1s)]\<rparr>"
  --"label"
| label_const:"const_list vs \<Longrightarrow> \<lparr>[Label n es vs]\<rparr> \<leadsto> \<lparr>vs\<rparr>"
| label_trap:"\<lparr>[Label n es [Trap]]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"br"
| br:"\<lbrakk>const_list vs; length vs = n; Lfilled i lholed (vs @ [$(Br i)]) LI\<rbrakk> \<Longrightarrow> \<lparr>[Label n es LI]\<rparr> \<leadsto> \<lparr>vs @ es\<rparr>"
  --"br_if"
| br_if_false:"int_eq n 0 \<Longrightarrow> \<lparr>[$C (ConstInt32 n), $(Br_if i)]\<rparr> \<leadsto> \<lparr>[]\<rparr>"
| br_if_true:"int_ne n 0 \<Longrightarrow> \<lparr>[$C (ConstInt32 n), $(Br_if i)]\<rparr> \<leadsto> \<lparr>[$(Br i)]\<rparr>"
  --"br_table"
| br_table:"\<lbrakk>to_list isne = is; length is > (nat_of_int c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt32 c), $(Br_table isne)]\<rparr> \<leadsto> \<lparr>[$(Br (is!(nat_of_int c)))]\<rparr>"
| br_table_length:"\<lbrakk>to_list isne = is; length is \<le> (nat_of_int c)\<rbrakk> \<Longrightarrow> \<lparr>[$C (ConstInt32 c), $(Br_table isne)]\<rparr> \<leadsto> \<lparr>[$(Br (last is))]\<rparr>"
  --"local"
| local_const:"\<lbrakk>const_list es; length es = n\<rbrakk> \<Longrightarrow> \<lparr>[Local n i vs es]\<rparr> \<leadsto> \<lparr>es\<rparr>"
| local_trap:"\<lparr>[Local n i vs [Trap]]\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"
  --"return"
| return:"\<lbrakk>const_list vs; length vs = n; Lfilled j lholed (vs @ [$Return]) es\<rbrakk>  \<Longrightarrow> \<lparr>[Local n i vls es]\<rparr> \<leadsto> \<lparr>vs\<rparr>"
  --"tee_local"
| tee_local:"is_const v \<Longrightarrow> \<lparr>[v, $(Tee_local i)]\<rparr> \<leadsto> \<lparr>[v, v, $(Set_local i)]\<rparr>"
| trap:"\<lbrakk>es \<noteq> [Trap]; Lfilled 0 lholed [Trap] es\<rbrakk> \<Longrightarrow> \<lparr>es\<rparr> \<leadsto> \<lparr>[Trap]\<rparr>"

(* full reduction rule *)
inductive reduce :: "[(('i32, 'i64, 'f32, 'f64, 'mem, 'host) s), ((('i32, 'i64, 'f32, 'f64) v) list), ((('i32, 'i64, 'f32, 'f64, 'host) e) list), nat, (('i32, 'i64, 'f32, 'f64, 'mem, 'host) s), ((('i32, 'i64, 'f32, 'f64) v) list), ((('i32, 'i64, 'f32, 'f64, 'host) e) list)] \<Rightarrow> bool" ("\<lparr>_;_;_\<rparr> \<leadsto>'_ _ \<lparr>_;_;_\<rparr>" 60) where
  --"lifting basic reduction"
  basic:"\<lparr>e\<rparr> \<leadsto> \<lparr>e'\<rparr> \<Longrightarrow> \<lparr>s;vs;e\<rparr> \<leadsto>_i \<lparr>s;vs;e'\<rparr>"
  --"call"
| call:"\<lparr>s;vs;[$(Call j)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Callcl (sfunc s i j)]\<rparr>"
  --"call_indirect"
| call_indirect_Some:"\<lbrakk>stab s i (nat_of_int c) = Some cl; stypes s i j = tf; cl_type cl = tf\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 c), $(Call_indirect j)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Callcl cl]\<rparr>"
| call_indirect_None:"\<lbrakk>(stab s i (nat_of_int c) = Some cl \<and> stypes s i j \<noteq> cl_type cl) \<or> stab s i (nat_of_int c) = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 c), $(Call_indirect j)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"call"
| callcl_native:"\<lbrakk>cl = Func_native j (t1s _> t2s) ts es; ves = ($$* vcs); length vcs = n; length ts = k; length t1s = n; length t2s = m; (n_zeros ts = zs) \<rbrakk> \<Longrightarrow> \<lparr>s;vs;ves @ [Callcl cl]\<rparr> \<leadsto>_i \<lparr>s;vs;[Local m j (vcs@zs) [$(Block ([] _> t2s) es)]]\<rparr>"
| callcl_host_Some:"\<lbrakk>cl = Func_host (t1s _> t2s) f; ves = ($$* vcs); length vcs = n; length t1s = n; length t2s = m; host_apply s (t1s _> t2s) f vcs = Some (s', vcs'); list_all2 types_agree t2s vcs'\<rbrakk> \<Longrightarrow> \<lparr>s;vs;ves @ [Callcl cl]\<rparr> \<leadsto>_i \<lparr>s';vs;($$* vcs')\<rparr>"
| callcl_host_None:"\<lbrakk>cl = Func_host (t1s _> t2s) f; ves = ($$* vcs); length vcs = n; length t1s = n; length t2s = m; host_apply s (t1s _> t2s) f vcs = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;ves @ [Callcl cl]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"get_local"
| get_local:"\<lbrakk>length vi = j\<rbrakk> \<Longrightarrow> \<lparr>s;(vi @ [v] @ vs);[$(Get_local j)]\<rparr> \<leadsto>_i \<lparr>s;(vi @ [v] @ vs);[$(C v)]\<rparr>"
  --"set_local"
| set_local:"\<lbrakk>length vi = j\<rbrakk> \<Longrightarrow> \<lparr>s;(vi @ [v] @ vs);[$(C v'), $(Set_local j)]\<rparr> \<leadsto>_i \<lparr>s;(vi @ [v'] @ vs);[]\<rparr>"
  --"get_global"
| get_global:"\<lparr>s;vs;[$(Get_global j)]\<rparr> \<leadsto>_i \<lparr>s;vs;[$ C(sglob_val s i j)]\<rparr>"
  --"set_global"
| set_global:"supdate_glob s i j v = s' \<Longrightarrow> \<lparr>s;vs;[$(C v), $(Set_global j)]\<rparr> \<leadsto>_i \<lparr>s';vs;[]\<rparr>"
  --"load"
| load_Some:"\<lbrakk>smem_ind s i = Some j; load ((mem s)!j) (nat_of_int k) off (t_length t) = Some bs\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $(Load t None a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[$C (wasm_deserialise bs t)]\<rparr>"
| load_None:"\<lbrakk>smem_ind s i = Some j; load ((mem s)!j) (nat_of_int k) off (t_length t) = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $(Load t None a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"load packed"
| load_packed_Some:"\<lbrakk>smem_ind s i = Some j; load_packed sx ((mem s)!j) (nat_of_int k) off (tp_length tp) = Some bs\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $(Load t (Some (tp, sx)) a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[$C (wasm_deserialise bs t)]\<rparr>"
| load_packed_None:"\<lbrakk>smem_ind s i = Some j; load_packed sx ((mem s)!j) (nat_of_int k) off (tp_length tp) = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $(Load t (Some (tp, sx)) a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"store"
| store_Some:"\<lbrakk>types_agree t v; smem_ind s i = Some j; store ((mem s)!j) (nat_of_int k) off (bits v) (t_length t) = Some mem'\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $C v, $(Store t None a off)]\<rparr> \<leadsto>_i \<lparr>s\<lparr>mem:= ((mem s)[j := mem'])\<rparr>;vs;[]\<rparr>"
| store_None:"\<lbrakk>types_agree t v; smem_ind s i = Some j; store ((mem s)!j) (nat_of_int k) off (bits v) (t_length t) = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $C v, $(Store t None a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"store packed" (* take only (tp_length tp) lower order bytes *)
| store_packed_Some:"\<lbrakk>types_agree t v; smem_ind s i = Some j; store_packed ((mem s)!j) (nat_of_int k) off (bits v) (tp_length tp) = Some mem'\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $C v, $(Store t (Some tp) a off)]\<rparr> \<leadsto>_i \<lparr>s\<lparr>mem:= ((mem s)[j := mem'])\<rparr>;vs;[]\<rparr>"
| store_packed_None:"\<lbrakk>types_agree t v; smem_ind s i = Some j; store_packed ((mem s)!j) (nat_of_int k) off (bits v) (tp_length tp) = None\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 k), $C v, $(Store t (Some tp) a off)]\<rparr> \<leadsto>_i \<lparr>s;vs;[Trap]\<rparr>"
  --"current_memory"
| current_memory:"\<lbrakk>smem_ind s i = Some j; mem_size ((mem s)!j) = n\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[ $(Current_memory)]\<rparr> \<leadsto>_i \<lparr>s;vs;[$C (ConstInt32 (int_of_nat n))]\<rparr>"
  --"grow_memory"
| grow_memory_Some:"\<lbrakk>smem_ind s i = Some j; mem_size ((mem s)!j) = n; mem_grow ((mem s)!j) (nat_of_int c) = Some mem'\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 c), $(Grow_memory)]\<rparr> \<leadsto>_i \<lparr>s\<lparr>mem:= ((mem s)[j := mem'])\<rparr>;vs;[$C (ConstInt32 (int_of_nat n))]\<rparr>"
  --"grow_memory fail"
| grow_memory_None:"\<lbrakk>smem_ind s i = Some j\<rbrakk> \<Longrightarrow> \<lparr>s;vs;[$C (ConstInt32 c),$(Grow_memory)]\<rparr> \<leadsto>_i \<lparr>s;vs;[$C (ConstInt32 int32_minus_one)]\<rparr>"
  (* The bad ones. *)
  --"inductive label reduction"
| label:"\<lbrakk>\<lparr>s;vs;es\<rparr> \<leadsto>_i \<lparr>s';vs';es'\<rparr>; Lfilled k lholed es les; Lfilled k lholed es' les'\<rbrakk> \<Longrightarrow> \<lparr>s;vs;les\<rparr> \<leadsto>_i \<lparr>s';vs';les'\<rparr>"
  --"inductive local reduction"
| local:"\<lbrakk>\<lparr>s;vs;es\<rparr> \<leadsto>_i \<lparr>s';vs';es'\<rparr>\<rbrakk> \<Longrightarrow> \<lparr>s;v0s;[Local n i vs es]\<rparr> \<leadsto>_j \<lparr>s';v0s;[Local n i vs' es']\<rparr>"
end
end

(* C \(ConstInt32 ([^()]*|(\(([^()]*|\(([^()]*|\([^()]*\))*\))*\)))\) *)