(*
 * Simple collection of functions useful for writing test cases.
 *)

(* MODIFIED by Conrad Watt *)

open Types
open Values
open Instance


let global (GlobalType (t, _)) =
  match t with
  | I32Type -> I32 666l
  | I64Type -> I64 666L
  | F32Type -> F32 (F32.of_float 666.6)
  | F64Type -> F64 (F64.of_float 666.6)

let table = Table.create AnyFuncType {min = 10l; max = Some 20l}
let memory = Memory.create {min = 1l; max = Some 2l}

let print_value v =
  Printf.printf "%s : %s\n"
    (Values.string_of_value v) (Types.string_of_value_type (Values.type_of v))

let print (FuncType (_, out)) s vs =
  List.iter print_value (List.map Ast_convert.unconvert_value vs);
  flush_all ();
  (s, List.map Ast_convert.default_value out)

let lookup name t =
  match Utf8.encode name, t with
  | "print", ExternalFuncType t -> ExternalFunc (HostFunc (t, ImplWrapperTypes.Hf (print t)))
  | "print", _ ->
    let t = FuncType ([], []) in ExternalFunc (HostFunc (t, ImplWrapperTypes.Hf (print t)))
  | "global", ExternalGlobalType t -> ExternalGlobal (global t)
  | "global", _ -> ExternalGlobal (global (GlobalType (I32Type, Immutable)))
  | "table", _ -> ExternalTable table
  | "memory", _ -> ExternalMemory memory
  | _ -> raise Not_found
