/*
 *  byterun/freshness.c
 *
 *  Low-level support for freshness functionality.
 *
 *  (c) Copyright 2003-2005, Mark R. Shinwell.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fail.h"
#include "alloc.h"
#include "memory.h"
#include "custom.h"
#include "mlvalues.h"
#include "callback.h"
#include "minor_gc.h"
#include "major_gc.h"
#include "hashtable_cwc22.h"
#include "hashtable_itr_cwc22.h"

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <limits.h>

/* Prototypes for external functions. */
extern value compare(value, value);

/***************************************************************************
 * Functions local to this module.
 **************************************************************************/

static int atom_id = 0; /* FIXME needs to go to 64 bits */

static inline unsigned int value_hasher(void* v)
{
    /* Hash function for things of type "value". */

#ifdef ARCH_SIXTYFOUR
    return ((unsigned int) v) % UINT_MAX;
#else
    return (unsigned int) v;
#endif
}

static int value_equality(void* v1, void* v2)
{
    /* Hashtable equality function for things of type "value". */

    return v1 == v2;
}

static void dump_rec(struct hashtable* ht, value v)
{
    CAMLparam1(v);
        
    if (Is_block(v) && Tag_val(v) < No_scan_tag && Tag_val(v) != Atom_tag) {
        void* data = hashtable_search(ht, (void*) v);

        printf("%p=", (void*) v);

        if (data) {
            /* seen this block already */
            printf("<shared>");
        }
        else {
            int x;

            switch (Tag_val(v)) {
                case 0: printf("tuple=["); break;
                case Closure_tag: printf("closure=["); break;
                case Double_array_tag: printf("<double array>"); break;
                case Abstract_tag: printf("<abstract>"); break;
                case Custom_tag: printf("<custom>"); break;
                case String_tag: printf("<string>"); break;
                case Abst_tag: printf("abst=["); break;
                default: printf("tag %d=[", (int) Tag_val(v)); break;
            }

            fflush(stdout);
           
            for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
                 x < Wosize_val(v);
                 x++) {
                dump_rec(ht, Field(v, x));
                if (x < Wosize_val(v) - 1) {
                    printf(", ");
                }
            }
            printf("]");
        }
    }
    else if (Is_block(v) && Tag_val(v) == Atom_tag) {
        printf("%p=atom#%d", (void*) v, Int_val(Field(v, 0)));
    }
    else {
        if (!Is_block(v)) {
            if (v == 0) {
                printf("<null>");
            }
            else {
                printf("<unboxed>");
            }
        }
        else {
            printf("<unstructured>");
        }
    }

    fflush(stdout);

    CAMLreturn0;
}

void dump(value v)
{
    CAMLparam1(v);

    struct hashtable* ht =
        create_hashtable(64, 0.75, value_hasher, value_equality);
    
    dump_rec(ht, v);

    hashtable_destroy(ht, 0);

    printf("\n");
    fflush(stdout);

    CAMLreturn0;
}

/* Allocate a new atom. */
static value alloc_atom(void)
{
    CAMLparam0();
    CAMLlocal1(atom);

    /* an atom is a block with tag Atom_tag, whose first field
     * is an "atom identifier".
     */
    atom = caml_alloc(1, Atom_tag);
    Store_field(atom, 0, Val_int(atom_id++));

    CAMLreturn(atom);
}

/* Allocate a number of fresh atoms and store them in a tuple. */
static value alloc_atoms(int num)
{
    int i;

    CAMLparam0();
    CAMLlocal1(array);

    assert(num > 0);
    
    array = caml_alloc_tuple(num);
    for (i = 0; i < num; i++) {
        Store_field(array, i, alloc_atom());
    }

    CAMLreturn(array);
}

/* Calculate the support of a value, given hashtables as described
 * in calculate_support() below.
 */
static void calculate_support_rec(value v,
                                  struct hashtable* ht_exclude,
                                  struct hashtable* ht_visited,
                                  struct hashtable* ht_support)
{
    CAMLparam1(v);

    if (Is_block(v) && Tag_val(v) < No_scan_tag &&
        Tag_val(v) != Object_tag &&
        (Tag_val(v) == Atom_tag || !hashtable_search(ht_visited, (void*) v))) {
        /* record having visited this value. */
        hashtable_insert(ht_visited, (void*) v, (void*) 1);
        
        switch (Tag_val(v)) {
            case Custom_swap_tag:
                calculate_support_rec(Field(v, 0),
                                      ht_exclude, ht_visited, ht_support);
                break;
                
            case Atom_tag:
            {
                /* determine if this atom is to be excluded */
                if (!hashtable_search(ht_exclude, (void*) v)) {
                    /* the atom is not to be excluded, so see if it's
                     * in the support set already...
                     */
                    if (!hashtable_search(ht_support, (void*) v)) {
                        /* ...no it's not, so add it. */
			hashtable_insert(ht_support, (void*) v,
                                         (void*) Val_int(1));
                    }
                }
                break;
            }

            case Abst_tag:
            {
                /* here we must calculate the support of the body v1 and
                 * then subtract the support of the value v2 in binding
                 * position: this is achieved by adding the support of v1
                 * to ht_exclude and then calculating the support of v2.
                 * We must take care with the ht_exclude hashtable, as we
                 * might be dealing with something like (<<a>>a, a) and it
                 * would be incorrect to have "a" remaining in ht_exclude
                 * when treating the second component of the tuple.
                 */

                /* create a hashtable which will act as a copy of
                 * ht_exclude, augmented with everything in the support of
                 * the value in binding position
                 */
                struct hashtable* ht_exclude_new;
                ht_exclude_new =
                  create_hashtable(64, 0.75, value_hasher, value_equality);

                /* calculate the support of the value
                 * in binding position */
                calculate_support_rec(Field(v, 0),
                                      ht_exclude, ht_visited,
                                      ht_exclude_new);

                /* add everything in ht_exclude to ht_exclude_new */
                if (hashtable_count((struct hashtable*) ht_exclude) > 0) {
                    struct hashtable_itr* iter;
                    iter = hashtable_iterator((struct hashtable*) ht_exclude);
                    assert(iter);

                    do {
                        void* key;
                        void* data;
                        
                        key = hashtable_iterator_key(iter);
                        data = hashtable_iterator_value(iter);

                        hashtable_insert(ht_exclude_new, key, data);
                    } while (hashtable_iterator_advance(iter));
                    free(iter);
                }

                /* calculate the support of the body */
                calculate_support_rec(Field(v, 1),
                                      ht_exclude_new, ht_visited,
                                      ht_support);
              
                /* delete the new hashtable */
                hashtable_destroy(ht_exclude_new, 0);

                break;
            }
	    
            default: /* tuple-like things and closures */
            {
                int i;
                /* take the union of the supports of each field, being
                 * careful to skip over the code pointer for closures */
                for (i = (Tag_val(v) == Closure_tag ? 1 : 0);
                     i < Wosize_val(v); i++) {
                    calculate_support_rec(Field(v, i), ht_exclude,
                                          ht_visited, ht_support);
                }
                
                break;
            }
        }
    }

    CAMLreturn0;
}

/* Calculate the support of a value, returning the support as a tuple.
 *
 * Val_unit is returned if the support is empty.
 * Wosize_val on a tuple result will give the cardinality of the support.
 */
static value calculate_support(value v)
{
    CAMLparam1(v);
    CAMLlocal1(supp);

    /* three hashtables are used for efficiency:
     * - one which holds atoms which should not be added to the
     *   support as we go along (used for treating abstraction values);
     * - one which holds pointers to blocks which we've visited already
     *   (each block only needs to be visited once);
     * - one which acts as a set for collecting the support.
     */
    struct hashtable* ht_exclude;
    struct hashtable* ht_visited;
    struct hashtable* ht_support;

    int cardinality;

    /* return right away if we can: unboxed values have empty support;
     * objects and non-scannable blocks are not inspected */
    if (!Is_block(v) ||
        (Tag_val(v) >= No_scan_tag || Tag_val(v) == Object_tag)) {
        CAMLreturn(Val_unit);
    }

    /* create hashtables */
    ht_exclude = create_hashtable(64, 0.75, value_hasher, value_equality);
    ht_visited = create_hashtable(64, 0.75, value_hasher, value_equality);
    ht_support = create_hashtable(64, 0.75, value_hasher, value_equality);

    /* calculate the support */
    calculate_support_rec(v, ht_exclude, ht_visited, ht_support);
    cardinality = hashtable_count(ht_support);

    /* allocate a tuple for the result and fill it, if the support
     * is non-empty */
    if (cardinality == 0) {
        supp = Val_unit;
    }
    else {
        /* transfer the contents of ht_support into the result tuple */
        struct hashtable_itr* iter;
        int pos;
	value* supp_tmp = (value*) malloc(sizeof(value) * cardinality);
	CAMLxparamN(supp_tmp, cardinality);

        iter = hashtable_iterator((struct hashtable*) ht_support);
        assert(iter);

	pos = 0;

        do {
	    supp_tmp[pos++] = (value) hashtable_iterator_key(iter);
        } while (hashtable_iterator_advance(iter));
	assert(pos == cardinality);
	free(iter);

        supp = caml_alloc_tuple(cardinality);
	for (pos = 0; pos < cardinality; pos++) {
            Store_field(supp, pos, supp_tmp[pos]);
	}

        free(supp_tmp);
    }
    
    /* delete the hashtables */
    hashtable_destroy(ht_exclude, 0);
    hashtable_destroy(ht_visited, 0);
    hashtable_destroy(ht_support, 0);

    CAMLreturn(supp);
}

#if 0
static int calculate_cardinality_of_support(value v)
{
    CAMLparam1(v);
    CAMLlocal1(supp);

    /* three hashtables are used for efficiency:
     * - one which holds atoms which should not be added to the
     *   support as we go along (used for treating abstraction values);
     * - one which holds pointers to blocks which we've visited already
     *   (each block only needs to be visited once);
     * - one which acts as a set for collecting the support.
     */
    struct hashtable* ht_exclude;
    struct hashtable* ht_visited;
    struct hashtable* ht_support;

    int cardinality;

    /* return right away if we can: unboxed values have empty support;
     * objects and non-scannable blocks are not inspected */
    if (!Is_block(v) ||
        (Tag_val(v) >= No_scan_tag || Tag_val(v) == Object_tag)) {
        CAMLreturn(0);
    }

    /* create hashtables */
    ht_exclude = create_hashtable(64, 0.75, value_hasher, value_equality);
    ht_visited = create_hashtable(64, 0.75, value_hasher, value_equality);
    ht_support = create_hashtable(64, 0.75, value_hasher, value_equality);

    /* calculate the support */
    calculate_support_rec(v, ht_exclude, ht_visited, ht_support);
    cardinality = hashtable_count(ht_support);
    
    /* delete the hashtables */
    hashtable_destroy(ht_exclude, 0);
    hashtable_destroy(ht_visited, 0);
    hashtable_destroy(ht_support, 0);

    CAMLreturn(cardinality);
}
#endif

/* Worker function for contains_abstractions(), below. */
static int contains_abstractions_rec(struct hashtable* ht, value v)
{
    CAMLparam1(v);
    int ret = 0;
        
    if (Is_block(v) && Tag_val(v) < No_scan_tag && Tag_val(v) != Object_tag) {
        void* data = hashtable_search(ht, (void*) v);

        /* data is NULL if we haven't seen the block already;
         *         (void*) 1 if we have seen the block already and it doesn't
         *                   contain abstractions;
         *         (void*) 2 if we have seen the block already and it does
         *                   contain abstractions.
         */

        if (data) {
            /* seen this block already */
            ret = ((int) data) - 1;
        }
        else {
            /* see if we have an abstraction block */
            if (Tag_val(v) == Abst_tag) {
                /* yes we do */
                ret = 1;
            }
            else {
                /* here we either have a closure, or another
                 * structured block which isn't one of the special cases.
                 */

                int x;

                /* add the current block to the hashtable.  Just in case
                 * the value is recursive, we first assume that we _do_
                 * contain abstractions to be on the safe side.
                 */
                hashtable_insert(ht, (void*) v, (void*) 2);

                for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
                     x < Wosize_val(v) && !ret;
                     x++) {
                    ret = contains_abstractions_rec(ht, Field(v, x));
                }
                
                /* replace our entry in the hashtable now we know the result */
                hashtable_remove(ht, (void*) v);
                /* insert is done below */
            }
            
            hashtable_insert(ht, (void*) v, (void*) (ret + 1));
        }
    }

    CAMLreturn(ret);
}

/* Approximate whether a value contains any abstraction blocks. */
static int contains_abstractions(value v)
{
    CAMLparam1(v);

    int ret;
    struct hashtable* ht =
        create_hashtable(64, 0.75, value_hasher, value_equality);
  
    /*
    printf("contains_abstractions:\n");
    dump(v);
    printf("-end-\n");
    */
    ret = contains_abstractions_rec(ht, v);

    hashtable_destroy(ht, 0);

    CAMLreturn(ret);
}

/* Worker function for valid_in_binding_pos(), below. */
static int valid_in_binding_pos_rec(struct hashtable* ht, value v)
{
    CAMLparam1(v);
    int ret = 0;

    if (!Is_block(v) || Tag_val(v) >= No_scan_tag) {
	/* do nothing */
    }
    else if (Tag_val(v) == Object_tag ||
	     Tag_val(v) == Closure_tag) {
	/* not allowed! */
	ret = 1;
    }
    else if (Tag_val(v) != Atom_tag /* atoms are OK */) {
        void* data = hashtable_search(ht, (void*) v);

        /* data is NULL if we haven't seen the block already;
         *         (void*) 1 if we have seen the block already and it doesn't
         *                   contain anything illegal in binding position;
         *         (void*) 2 if we have seen the block already and it does
         *                   contain something illegal in binding position.
         */

        if (data) {
            /* seen this block already */
            ret = ((int) data) - 1;
        }
        else {
	    int x;

	    hashtable_insert(ht, (void*) v, (void*) 1);

	    for (x = 0; x < Wosize_val(v) && !ret; x++) {
		ret = valid_in_binding_pos_rec(ht, Field(v, x));
	    }
		    
	    /* replace our entry in the hashtable now we know the result */
	    hashtable_remove(ht, (void*) v);
            hashtable_insert(ht, (void*) v, (void*) (ret + 1));
        }
    }

    CAMLreturn(ret);
}

/* Determine whether a value is acceptable in binding position.
 * Returns 0 if the value is OK, 1 if not.
 */
static int valid_in_binding_pos(value v)
{
    CAMLparam1(v);

    int ret;
    struct hashtable* ht =
        create_hashtable(64, 0.75, value_hasher, value_equality);
   
    ret = valid_in_binding_pos_rec(ht, v);

    hashtable_destroy(ht, 0);

    CAMLreturn(ret);
}

static value tuple_to_list(value v)
{
    /* Turn (x0, ..., xn) into [x0; ...; xn]. */

    CAMLparam1(v);
    CAMLlocal1(ret);
    CAMLlocal1(cur);
    CAMLlocal1(prev);
    int i = 0;
    
    assert(Is_block(v) && Tag_val(v) == 0 && Wosize_val(v) >= 1);

    prev = (value) 0;
    ret = (value) 0;

    do {
        cur = caml_alloc_tuple(2);
        Store_field(cur, 0, Field(v, i));
        Store_field(cur, 1, Val_int(0));

        if (!ret) {
            ret = cur;
        }

        if (prev) {
            Store_field(prev, 1, cur);
        }

        prev = cur;
    } while (++i < Wosize_val(v));

    CAMLreturn(ret);
}

typedef value (*alloc_fn)(int size, int tag);

typedef enum {
    entry_ALLOCATE,
    entry_SHARED
} dynarray_entry_type;

typedef struct {
    dynarray_entry_type type;
    union {
        struct {
            alloc_fn fn;
            int size;
            int tag;
        } allocate;
        struct {
            int index;
        } shared;
    } data;
} dynarray_entry;

typedef struct {
    dynarray_entry* data;
    int allocated;
    int used;
} dynarray;

static dynarray dynarray_alloc(void)
{
    dynarray da;
    da.allocated = 16;
    da.used = 0;
    da.data = (dynarray_entry*) malloc(da.allocated * sizeof(dynarray_entry));
    return da;
}

static void dynarray_free(dynarray da)
{
    free((void*) da.data);
}

static void dynarray_check_size(dynarray* da)
{
    if (da->used == da->allocated) {
        da->allocated *= 2;
        da->data = (dynarray_entry*)
          realloc((void*) da->data, da->allocated * sizeof(dynarray_entry));
    }
}

static int dynarray_add_allocation(dynarray* da,
                                   alloc_fn fn, int size, int tag)
{
    dynarray_check_size(da);
    
    da->data[da->used].type = entry_ALLOCATE;
    da->data[da->used].data.allocate.fn = fn;
    da->data[da->used].data.allocate.size = size;
    da->data[da->used].data.allocate.tag = tag;
    da->used++;

    return da->used - 1;
}

static void dynarray_add_shared(dynarray* da, int index)
{
    assert(index < da->used);
    
    dynarray_check_size(da);

    da->data[da->used].type = entry_SHARED;
    da->data[da->used].data.shared.index = index;
    da->used++;
}

static value dynarray_to_blocks(dynarray* da)
{
    CAMLparam0();
    CAMLlocal1(v);

    v = Val_unit;

    if (da->used > 0) {
        int i;
        v = caml_alloc_tuple(da->used);
        for (i = 0; i < da->used; i++) {
            if (da->data[i].type == entry_ALLOCATE) {
                Store_field(v, i,
                            (*(da->data[i].data.allocate.fn))
                            (da->data[i].data.allocate.size,
                             da->data[i].data.allocate.tag));
            }
            else {
                assert(i > 0);
                assert(da->data[i].data.shared.index < i);

                Store_field(v, i, Field(v, da->data[i].data.shared.index));
            }
        }
    }

    CAMLreturn(v);
}

static int dynarray_is_block_shared(dynarray* da, int index)
{
    /* Returns non-zero if the block at "index" is shared. */

    assert(da);
    assert(index >= 0 && index < da->used);

    return (da->data[index].type == entry_SHARED);
}

static value alloc_fn_gen(int size, int tag)
{
    return caml_alloc(size, tag);
}

/*static value alloc_fn_atom(int unused1, int unused2)
{
    return alloc_atom();
}*/

static void deep_copy_collect_sizes(dynarray* da, struct hashtable* ht,
			            value v, int under_nobind)
{
    /* No allocation on the ML heap may happen within this function, or
     * "ht" will potentially be invalidated.
     */

    /* The hashtable maps from values which we have seen and will need to
     * be copied, to integers specifying the *1-based* index in the array
     * "da" corresponding to that value.  We have to use a 1-based index
     * to distinguish the "not found" return value from hashtable_search.
     */

    if (Is_block(v) && (Tag_val(v) < No_scan_tag &&
			Tag_val(v) != Object_tag &&
			Tag_val(v) != Atom_tag &&
			Tag_val(v) != No_swap_tag)) {
        void* data = hashtable_search(ht, (void*) v);

	if (data) {
	    /* seen this value already: mark it as a shared block in
             * the array */
            dynarray_add_shared(da, ((int) data) - 1);
  	    return;
        }
	else {
            if (Tag_val(v) == Custom_swap_tag) {
                hashtable_insert(ht, (void*) v,
                  (void*) (1 + dynarray_add_allocation(da, alloc_fn_gen,
                                                       2, Custom_swap_tag)));
            }
            else if (under_nobind || (Tag_val(v) != No_bind_tag)) {
                int x;
    
                hashtable_insert(ht, (void*) v,
                  (void*) (1 + dynarray_add_allocation(da, alloc_fn_gen,
                                 Wosize_val(v), Tag_val(v))));
                for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
                     x < Wosize_val(v);
                     x++) {
                    deep_copy_collect_sizes(da, ht, Field(v, x), under_nobind);
                }
            }
	}
    }
}

static value deep_copy(value blocks, int* next_block,
                       dynarray* da, value v,
                       value atoms_1, value atoms_2,
                       int freshening_abstractions, int under_nobind)
{
    CAMLparam4(blocks, v, atoms_1, atoms_2);
    CAMLlocal1(ret);

    assert(Is_block(atoms_1) && Tag_val(atoms_1) == 0);
    assert(Is_block(atoms_2) && Tag_val(atoms_2) == 0);
    assert(Wosize_val(atoms_1) == Wosize_val(atoms_2));

    ret = v;

    if (Is_block(v) && (Tag_val(v) < No_scan_tag &&
                        Tag_val(v) != Object_tag &&
                        Tag_val(v) != Atom_tag &&
			Tag_val(v) != No_swap_tag &&
			(under_nobind || (Tag_val(v) != No_bind_tag)))) {
        CAMLlocal1(data);

        assert(Is_block(blocks));
        assert(Wosize_val(blocks) == da->used);
        assert(blocks != Val_unit);
        assert(*next_block < da->used);

        ret = Field(blocks, *next_block);
        *next_block = *next_block + 1;

        if (!dynarray_is_block_shared(da, *next_block - 1)) {
            if (Tag_val(v) == Custom_swap_tag) {
                CAMLlocal1(atoms_1_list);
                CAMLlocal1(atoms_2_list);
                CAMLlocal1(tuple);
                
                /* call user's function to do the swapping */
                atoms_1_list = tuple_to_list(atoms_1);
                atoms_2_list = tuple_to_list(atoms_2);
                tuple = caml_alloc_tuple(3);
                Store_field(tuple, 0, atoms_1_list);
                Store_field(tuple, 1, atoms_2_list);
                Store_field(tuple, 2, Field(v, 0));
                Store_field(ret, 0, caml_callback(Field(v, 1),  tuple));
                Store_field(ret, 1, Field(v, 1));
            }
            else if (Tag_val(v) == Abst_tag && freshening_abstractions) {
                /* determine the support of the value in binding position */
                CAMLlocal3(new_atoms_1, new_atoms_2, supp);
                /*int supp_card;*/
            
                supp = calculate_support(Field(v, 0));

                if (supp == Val_unit) {
                    /* value in binding position has empty support */
                    new_atoms_1 = atoms_1;
                    new_atoms_2 = atoms_2;
                }
                else {
                    /* value in binding position has non-empty support */
                    int supp_card = Wosize_val(supp);
                    int i;
                    assert(supp_card > 0);
            
                    /* allocate as many fresh atoms as the cardinality
                     * of "supp", enlarging "atoms_1" and "atoms_2" along
                     * the way.  "atoms_1" receives atoms from "supp",
                     * whilst "atoms_2" receives new atoms.
                     */
                    new_atoms_1 = caml_alloc_tuple(Wosize_val(atoms_1) + supp_card);
                    new_atoms_2 = caml_alloc_tuple(Wosize_val(atoms_2) + supp_card);
                    
                    /* copy across stuff from atoms_1 and atoms_2 */
                    for (i = 0; i < Wosize_val(atoms_1); i++) {
                        Store_field(new_atoms_1, i, Field(atoms_1, i));
                        Store_field(new_atoms_2, i, Field(atoms_2, i));
                    }
    
                    /* fill the remainder of new_atoms_1 with "supp" and
                     * the remainder of new_atoms_2 with fresh atoms
                     */
                    for (i = 0; i < supp_card; i++) {
                        Store_field(new_atoms_1, i + Wosize_val(atoms_1),
                                    Field(supp, i));
                        Store_field(new_atoms_2, i + Wosize_val(atoms_2),
                                    alloc_atom());
                    }
                }
    
                /* allocate space for the copy and propagate the
                 * new permutations.  We still need to do this even if
                 * the bound value has no support, as it may still contain
                 * abstraction values which require freshening.
                 */
                Store_field(ret, 0,
                            deep_copy(blocks, next_block, da, Field(v, 0),
                                      new_atoms_1, new_atoms_2,
                                      freshening_abstractions, under_nobind));
                Store_field(ret, 1,
                            deep_copy(blocks, next_block, da, Field(v, 1),
                                      new_atoms_1, new_atoms_2,
                                      freshening_abstractions, under_nobind));
            }
            else {
                int x;
    
                /* copy code pointer for closures */
                if (Tag_val(v) == Closure_tag) {
                    /*Store_field(ret, 0, Field(v, 0));*/
		    Code_val(ret) = Code_val(v);
                }
    
                /* copy the rest */
                for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
                     x < Wosize_val(v);
                     x++) {
                    Store_field(ret, x,
                      deep_copy(blocks, next_block, da, Field(v, x),
                                atoms_1, atoms_2, freshening_abstractions,
				under_nobind));
                }
            }
        }
    }
    else if (Is_block(v) && Tag_val(v) == Atom_tag) {
        /* Apply any swap which might apply to this atom. */
        int x;
        ret = v;
        for (x = 0; x < Wosize_val(atoms_1); x++) {
            if (Field(ret, 0) == Field(Field(atoms_1, x), 0)) {
                ret = Field(atoms_2, x);
            }
            else if (Field(ret, 0) == Field(Field(atoms_2, x), 0)) {
                ret = Field(atoms_1, x);
            }
        }
    }

    CAMLreturn(ret);
}

/* Given two equally-sized tuples of atoms, swap the Nth element
 * of the first tuple and the Nth element of the second tuple
 * throughout the value v, copying the value along the way.
 * At most one swap must apply to any particular atom in v.
 */
static value swap_atoms(value atoms_1, value atoms_2, value v,
                        int freshening_abstractions, int under_nobind)
{
    CAMLparam3(atoms_1, atoms_2, v);
    CAMLlocal2(ret, blocks);
    dynarray da;
    struct hashtable* ht;
    int next_block = 0;

    if (!Is_block(v)) {
	CAMLreturn(v);
    }

    /* This needs to be done in multiple stages to avoid the necessity
     * of having hashtables keyed on "value"s which are stable under GC.
     */

    /* Stage 1: collect sizes of blocks needed for a copy of "v"
     * (no ML allocation is allowed in this stage as we require a hashtable
     *  to detect cyclic structures)
     */
    da = dynarray_alloc();
    ht = create_hashtable(64, 0.75, value_hasher, value_equality);
    deep_copy_collect_sizes(&da, ht, v, under_nobind);
    hashtable_destroy(ht, 0);
    
    /* Stage 2: allocate blocks */
    blocks = dynarray_to_blocks(&da);

    /* Stage 3: perform the copy */
    ret = deep_copy(blocks, &next_block, &da, v, atoms_1, atoms_2,
                    freshening_abstractions, under_nobind);
    dynarray_free(da);

    CAMLreturn(ret);
}

/***************************************************************************
 * Exported functions which aren't primitives.
 **************************************************************************/

/* Test two abstraction values for equality.
 * Returns 0 if equal and -1 if unequal.
 */
long fresh_compare_abstractions(value v1, value v2)
{
    int v1_support_size;
    int v2_support_size;
    long result = -1; /* "not equal" */
    
    CAMLparam2(v1, v2);
    CAMLlocal3(v1_support, v2_support, new_atoms);

    assert(Is_block(v1) && Tag_val(v1) == Abst_tag);
    assert(Is_block(v2) && Tag_val(v2) == Abst_tag);

    /* calculate the support of the values in binding position
     * in v1 and v2...
     */
    v1_support = calculate_support(Field(v1, 0));
    v2_support = calculate_support(Field(v2, 0));

    /* ...determine the size of these supports... */
    v1_support_size = (v1_support == Val_unit) ? 0 : Wosize_val(v1_support);
    v2_support_size = (v2_support == Val_unit) ? 0 : Wosize_val(v2_support);

    /* ...and if the sizes differ then the abstractions cannot be equal. */
    if (v1_support_size == v2_support_size) {
        CAMLlocal2(v1_new, v2_new);

        if (v1_support_size > 0) {
            /* allocate sufficiently many fresh atoms */
            new_atoms = alloc_atoms(v1_support_size);

            /* freshen throughout v1 and v2 with the same set
             * of new atoms.
             */
            v1_new = swap_atoms(v1_support, new_atoms, v1, 0, 0);
            v2_new = swap_atoms(v2_support, new_atoms, v2, 0, 0);
        }
        else {
            v1_new = v1;
            v2_new = v2;
        }
    
        /* now compare v1 and v2, as if they were tuple blocks
         * rather than abstraction blocks.
         */
        if (caml_compare(Field(v1_new, 0), Field(v2_new, 0)) == Val_int(0) &&
            caml_compare(Field(v1_new, 1), Field(v2_new, 1)) == Val_int(0)) {
            result = 0;
        }
    }
    
    CAMLreturn(result);
}

/***************************************************************************
 * Exported primitives.
 **************************************************************************/

/* Inspect a value to see if it might contain any abstractions.
 * If it definitely does not, return the value unchanged.
 * Otherwise, copy the value (preserving sharing to a reasonable
 * extent) and freshen up any atoms in binding position in abstractions therein,
 * as required for the pattern-matching hack.
 */
CAMLprim value fresh_throughout(value v)
{
    CAMLparam1(v);
    CAMLlocal1(ret);

    /*dump(v);printf("--\n");fflush(stdout);*/

    ret = v;

    if (contains_abstractions(v)) {
        /* initialise dummy swap (requirement for swap_atoms()) */
        /* FIXME should adjust deep_copy_rec() so this isn't needed */
        CAMLlocal2(atoms_1, atoms_2);
        atoms_1 = caml_alloc_tuple(1);
        atoms_2 = caml_alloc_tuple(1);
        Store_field(atoms_1, 0, alloc_atom());
        Store_field(atoms_2, 0, alloc_atom());

        /* perform the freshening */
        ret = swap_atoms(atoms_1, atoms_2, v, 1, 0);
    }

    CAMLreturn(ret);
}

/* Test if "n" occurs in the support of "v": return Val_false if
 * so and Val_true otherwise. */
CAMLprim value fresh_freshfor(value n, value v)
{
    CAMLparam1(v);
    CAMLlocal2(ret, supp);
    int supp_size;
    int i;

    assert(Is_block(n) && Tag_val(n) == Atom_tag);

    ret = Val_true;

    if (valid_in_binding_pos(v)) {
        caml_invalid_argument("invalid value for fresh-for test");
    }
   
    supp = calculate_support(v);
    supp_size = (supp == Val_unit ? 0 : Wosize_val(supp));

    for (i = 0; ret == Val_true && i < supp_size; i++) {
        if (Field(n, 0) == Field(Field(supp, i), 0)) {
            ret = Val_false;
        }
    }

    CAMLreturn(ret);
}

/* Allocate a new atom. */
CAMLprim value fresh_new_atom(void)
{
    CAMLparam0();
    CAMLreturn(alloc_atom());
}

/* Swap two atoms throughout a value. */
CAMLprim value fresh_swap_atoms(value atom1, value atom2, value v)
{
    CAMLparam3(atom1, atom2, v);
    CAMLlocal2(blk1, blk2);
    CAMLlocal1(ret);

    blk1 = caml_alloc_tuple(1);
    blk2 = caml_alloc_tuple(1);
    Store_field(blk1, 0, atom1);
    Store_field(blk2, 0, atom2);
    
    ret = swap_atoms(blk1, blk2, v, 0, 1);

    CAMLreturn(ret);
}

/* Swap multiple atoms pairwise throughout a value. */
CAMLprim value fresh_swap_multiple_atoms(value atoms1, value atoms2, value v)
{
    int i;
    int total;
    CAMLparam3(atoms1, atoms2, v);
    CAMLlocal4(blk1, blk2, atoms1_copy, atoms2_copy);
    CAMLlocal1(ret);

    assert(Is_block(atoms1) && Is_block(atoms2));

    total = 0;
    atoms1_copy = atoms1;
    atoms2_copy = atoms2;

    while (Is_block(atoms1)) {
        assert(Is_block(atoms2));

        total++;

        atoms1 = Field(atoms1, 1);
        atoms2 = Field(atoms2, 1);
    }

    assert(total > 0);

    atoms1 = atoms1_copy;
    atoms2 = atoms2_copy;

    blk1 = caml_alloc_tuple(total);
    blk2 = caml_alloc_tuple(total);
    for (i = 0; i < total; i++) {
        assert(Is_block(atoms1));
        assert(Is_block(atoms2));

        Store_field(blk1, i, Field(atoms1, 0));
        Store_field(blk2, i, Field(atoms2, 0));
        atoms1 = Field(atoms1, 1);
        atoms2 = Field(atoms2, 1);
    }
    
    ret = swap_atoms(blk1, blk2, v, 0, 1);

    CAMLreturn(ret);
}

/* Create an abstraction value <<v1>>v2. */
CAMLprim value fresh_create_abstraction(value v1, value v2)
{
    CAMLparam2(v1, v2);
    CAMLlocal1(v);

    if (valid_in_binding_pos(v1)) {
        caml_invalid_argument("Invalid value in binding position");
    }

    v = caml_alloc(2, Abst_tag);
    Store_field(v, 0, v1);
    Store_field(v, 1, v2);

    CAMLreturn(v);
}

