/**
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *	Copyright 2011 Andrew Rice (acr31@cam.ac.uk)
 */
package uk.ac.cam.acr31.geomessagingserver;

import java.io.Writer;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

public class XMLWriter {

	private XMLStreamWriter w;
	private LinkedList<String> stack;
	private SimpleDateFormat formatter = new SimpleDateFormat();

	public XMLWriter(Writer result) {
		try {
			this.w = XMLOutputFactory.newInstance().createXMLStreamWriter(
					result);
		} catch (XMLStreamException e) {
			throw new Error("Failed to construct XMLStreamWriter", e);
		} catch (FactoryConfigurationError e) {
			throw new Error("Failed to create XMLOutputFactory", e);
		}
		stack = new LinkedList<String>();
	}

	private void writeAttributes(Object... attributes)
			throws XMLStreamException {
		if (attributes != null) {
			for (int i = 0; i < attributes.length; i += 2) {
				if (attributes[i + 1] != null) {
					String key = (String) attributes[i];
					String value;
					if (attributes[i + 1] instanceof Date) {
						value = formatter.format((Date) attributes[i + 1]);
					} else {
						value = attributes[i + 1].toString();
					}
					w.writeAttribute(key, value);
				}
			}
		}
	}

	public XMLWriter raw(String xml) {
		try {
			w.writeCharacters(xml);
		} catch (XMLStreamException e) {
			throw new Error("Error in raw");
		}
		return this;
	}

	public XMLWriter open(String name, Object... attributes) {
		try {
			w.writeStartElement(name);
			stack.addLast(name);
			writeAttributes(attributes);
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in open " + name, e);
		}
	}

	public XMLWriter empty(String name, Object... attributes) {
		try {
			w.writeStartElement(name);
			writeAttributes(attributes);
			w.writeEndElement();
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in empty " + name, e);
		}
	}

	public XMLWriter dtd(String name) {
		try {
			w.writeDTD(name);
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in dtd " + name, e);
		}
	}

	public XMLWriter pi(String name, Object... attributes) {
		try {
			String data = "";
			for (int i = 0; i < attributes.length; i += 2) {
				if (attributes[i + 1] != null) {
					data += attributes[i].toString() + "=\""
							+ attributes[i + 1].toString() + "\" ";
				}
			}
			if (data.equals("")) {
				w.writeProcessingInstruction(name);
			} else {
				w.writeProcessingInstruction(name, data);
			}
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in pi " + name);
		}
	}

	public XMLWriter text(String name) {
		try {
			w.writeCharacters(name);
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in text " + name);
		}
	}

	public XMLWriter close() {
		try {
			w.writeEndElement();
			stack.removeLast();
			return this;
		} catch (XMLStreamException e) {
			throw new Error(e);
		}
	}

	public XMLWriter close(String tag) {
		try {
			do {
				w.writeEndElement();
			} while (!stack.removeLast().equals(tag));
			return this;
		} catch (XMLStreamException e) {
			throw new Error("Error in close " + tag);
		}
	}

	public XMLWriter textElement(String element, String text,
			Object... attributes) {
		try {
			w.writeStartElement(element);
			writeAttributes(attributes);
			w.writeCharacters(text);
			w.writeEndElement();
			return this;
		} catch (XMLStreamException e) {
			throw new Error(e);
		}
	}

	public XMLWriter beanElement(String element, Object bean,
			Object... attributes) {
		try {
			w.writeStartElement(element);
			writeAttributes(attributes);
			for (Field f : bean.getClass().getDeclaredFields()) {
				if (!Modifier.isStatic(f.getModifiers())) {
					f.setAccessible(true);
					Object v = f.get(bean);
					w.writeStartElement(f.getName());
					if (v != null) {
						if (v instanceof Date)
							w.writeCData(formatter.format((Date) v));
						else
							w.writeCData(v.toString());
					}
					w.writeEndElement();
				}
			}
			w.writeEndElement();
		} catch (SecurityException e) {
			throw new Error(e);
		} catch (IllegalArgumentException e) {
			throw new Error(e);
		} catch (XMLStreamException e) {
			throw new Error(e);
		} catch (IllegalAccessException e) {
			throw new Error(e);
		}
		return this;
	}

	public void flush() {
		try {
			w.flush();
		} catch (XMLStreamException e) {
			throw new Error(e);
		}
	}

}
