/**
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *	Copyright 2009 Andrew Rice (acr31@cam.ac.uk) and Vytautas Vaitukaitis (vv236@cam.ac.uk)
 */

package uk.ac.cam.acr31.android.geomessaging;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import uk.ac.cam.acr31.android.geomessaging.servertools.Message;
import uk.ac.cam.acr31.android.geomessaging.servertools.ServerHelper;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.IBinder;

public class MessagingService extends Service
{
	protected static boolean mRunning = false;
	
	private LocationManager mLocManager;
	private LocationListener mLocListener;
		
	private NotificationManager mNotificationManager;
	private int mNotificationID = 0;
	
	protected static final double PROXIMITY_MILES = 0.05;
	
	private Set<Message> notifiedMessages = new HashSet<Message>();
	
	@Override
	public void onStart(Intent intent, int startId)
	{
		super.onStart(intent, startId);
		
		// register location listener
		mLocManager = (LocationManager) this.getSystemService(Context.LOCATION_SERVICE);
		
		mLocListener = new LocationListener(){

			private boolean mProcessingLocation;

			@Override
			public void onLocationChanged(final Location location)
			{
				if (location != null) {
					if (!mProcessingLocation) {
						mProcessingLocation = true;
						new Thread() {
							@Override
							public void run() {
								processLocationUpdate(location);
								mProcessingLocation =false;
							}	
						}.start();
					}
				}
			}		

			@Override
			public void onProviderDisabled(String provider)
			{}

			@Override
			public void onProviderEnabled(String provider)
			{}

			@Override
			public void onStatusChanged(String provider, int status, Bundle extras)
			{}};

			
		String provider = LocationManager.GPS_PROVIDER;
		if (mLocManager.getProvider("OldTracks") != null) provider = "OldTracks";
		mLocManager.requestLocationUpdates(provider, 0, 0, mLocListener);
		
		mNotificationManager = (NotificationManager) this.getSystemService(Context.NOTIFICATION_SERVICE);
		
		mRunning = true;
	}

	@Override
	public IBinder onBind(Intent arg0)
	{
		return null;
	}

	@Override
	public void onDestroy()
	{
		super.onDestroy();
		mLocManager.removeUpdates(mLocListener);
		
		mRunning = false;
	}
	
	private void processLocationUpdate(Location location)
	{
		ArrayList<Message> messages = ServerHelper.getMessages(location.getLatitude(),location.getLongitude(),PROXIMITY_MILES);
		
		for (Message m : messages)
		{
			// if we haven't already notified about this message at some point earlier
			
			if (!notifiedMessages.contains(m))
			{
				
				// create a notification for each of the messages that have just arrived
				Notification notification = new Notification(R.drawable.status_message, m.getMessage(), System.currentTimeMillis());
				
				// create an intent that will be launched when clicked on the notification
				Intent intent = new Intent(this, ShowMessageActivity.class);
				intent.putExtra(ShowMessageActivity.INTENT_MESSAGE, m.getMessage());
				if (m.getPictureUri() != null)
				{
					intent.putExtra(ShowMessageActivity.INTENT_PICTURE_URI, m.getPictureUri());				
				}
				
				// note that requestCode has to be different, otherwise, the same PendingIntent is reused
				PendingIntent pendingIntent = PendingIntent.getActivity(this, mNotificationID, intent, 0);
				
				notification.flags = notification.flags | Notification.FLAG_AUTO_CANCEL;
				notification.setLatestEventInfo(this, getString(R.string.app_name), m.getMessage(), pendingIntent);
				
				mNotificationManager.notify(mNotificationID++, notification);
				
				// add this message to the notified map
				notifiedMessages.add(m);
			}
		}
	}
	
	
	
}
