/**
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *	Copyright 2009 Andrew Rice (acr31@cam.ac.uk) and Vytautas Vaitukaitis (vv236@cam.ac.uk)
 */

package uk.ac.cam.acr31.android.geomessaging;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;

import uk.ac.cam.acr31.android.geomessaging.servertools.ServerHelper;
import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.provider.MediaStore;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup.LayoutParams;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

public class LeaveMessageActivity extends Activity
{
	private final static int REQUEST_CAMERA = 0;

	private LinearLayout mImageContainer;
	private ImageButton mTakePictureButton;
	private ImageView mImageView;
	
	private Bitmap mPicture = null;
	private EditText mMessageField;
	

	private Location mLocation = null;

	private LocationManager mLocManager;
	private LocationListener mLocListener;

	private TextView mTextLocation;
	
	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.leavemessagewindow);

		// "wire up" all the fields and buttons

		// get the container of the "take a picture" button so that we can
		// replace the button when the picture is taken
		mImageContainer = (LinearLayout) this.findViewById(R.id.imageContainer);

		// get the take an image button
		mTakePictureButton = (ImageButton) this.findViewById(R.id.takePictureButton);
		mTakePictureButton.setOnClickListener(new View.OnClickListener()
		{

			@Override
			public void onClick(View v)
			{
				// fire off an intent to get a picture from camera
				Intent intent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);

				// note that we could add some extra data to the intent, e.g.
				// tell where we want the picture to be saved at, etc.

				startActivityForResult(intent, REQUEST_CAMERA);
			}
		});

		mMessageField = (EditText) this.findViewById(R.id.messageField);

		Button postButton = (Button) this.findViewById(R.id.postButton);
		postButton.setOnClickListener(new View.OnClickListener()
		{
			@Override
			public void onClick(View v)
			{
				postMessage();
			}
		});

		mTextLocation = (TextView)findViewById(R.id.TextViewLocation);
		
		// set up the location updates
		mLocManager = (LocationManager) getSystemService(Context.LOCATION_SERVICE);

		mLocListener = new LocationListener()
		{
			@Override
			public void onLocationChanged(Location location)
			{
				mLocation = location;
				mTextLocation.setText("lon="+location.getLongitude()+"\nlat="+location.getLatitude());
			}

			@Override
			public void onProviderDisabled(String provider)
			{}

			@Override
			public void onProviderEnabled(String provider)
			{}

			@Override
			public void onStatusChanged(String provider, int status, Bundle extras)
			{}
		};
		
		// start the messaging service if it is not running
		if (!MessagingService.mRunning)
		{
			Intent intent = new Intent(this, MessagingService.class);
			startService(intent);
		}

	}

	@Override
	protected void onPause()
	{
		mLocManager.removeUpdates(mLocListener);
		super.onPause();
	}

	@Override
	protected void onResume()
	{
		String provider = LocationManager.GPS_PROVIDER;
		if (mLocManager.getProvider("OldTracks") != null) { provider = "OldTracks"; }
		mLocManager.requestLocationUpdates(provider, 0, 0, mLocListener);
		super.onResume();
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data)
	{
		super.onActivityResult(requestCode, resultCode, data);

		// just a quick check to see whether it was a camera request and if the
		// picture was actually taken
		if (requestCode == REQUEST_CAMERA && resultCode == RESULT_OK)
		{
			// the picture is returned in extras field of the intent - things
			// like this need to be looked up in the Android Reference
			mPicture = (Bitmap) data.getExtras().get("data");
			
			// make the button invisible
			mImageContainer.removeView(mTakePictureButton);
			
			// note how all the layout components can be created from the code
			// as well
			mImageView = new ImageView(this);
			mImageView.setImageBitmap(mPicture);

			// add this to the container telling that the image should fill the
			// parent both in terms of width and height
			mImageContainer.addView(
					mImageView,
					new LinearLayout.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.FILL_PARENT));
		}
	}

	private void postMessage()
	{
		if (mLocation != null)
		{
			// get what's in the message
			final String message = mMessageField.getEditableText().toString();
			final String tempFileName = (mPicture == null) ? null : ("/sdcard/"
					+ System.currentTimeMillis() + ".jpg");

			if (tempFileName != null)
			{
				// compress a picture into a temporary file
				File f = new File(tempFileName);
				try
				{
					OutputStream pictureStream = new FileOutputStream(f);
					mPicture.compress(CompressFormat.JPEG, 90, pictureStream);
				}
				catch (FileNotFoundException e)
				{
					// log the error
					Log.e("LeaveMessageActivity", e.getMessage());
				}
			}

			// show progress dialog and start doing things in a separate thread
			final ProgressDialog progressDialog = new ProgressDialog(this);
			progressDialog.setTitle(R.string.postingMessage);
			progressDialog.setMessage(getString(R.string.sendingMessage));
			progressDialog.show();

			// handler message codes
			final int HANDLER_FAILED = 0, HANDLER_MESSAGE_SENT = 2;
			final Handler handler = new Handler()
			{

				@Override
				public void handleMessage(Message msg)
				{
					progressDialog.dismiss();
					switch (msg.what)
					{
					case HANDLER_MESSAGE_SENT:
						// tell the user that the message was posted
						Toast.makeText(LeaveMessageActivity.this, R.string.messagePosted,
								Toast.LENGTH_SHORT).show();

						resetFields();
						break;
					case HANDLER_FAILED:
						Toast.makeText(LeaveMessageActivity.this,
								R.string.messagePostingUnsuccessful, Toast.LENGTH_LONG).show();
						break;
					}
					super.handleMessage(msg);
				}
			};

			Thread postThread = new Thread(new Runnable()
			{

				@Override
				public void run()
				{
					if (ServerHelper.sendMessage(message, tempFileName, mLocation.getLatitude(),mLocation.getLongitude()))
						handler.sendEmptyMessage(HANDLER_MESSAGE_SENT);
					else
						handler.sendEmptyMessage(HANDLER_FAILED);
				}
			});
			postThread.start();
		} else
		{
			// location is probably not available yet
			Toast.makeText(this, R.string.locationUnavailable, Toast.LENGTH_LONG).show();
		}
	}

	private void resetFields()
	{
		mMessageField.getEditableText().clear();
		
		// remove the image view if it is not null
		if (mImageView != null)
		{
			mImageContainer.removeView(mImageView);
			
			mImageView = null;
			mPicture = null;
			
			// set the button to be visible again
			mImageContainer.addView(mTakePictureButton);
		}
	}


	@Override
	protected void onDestroy()
	{
		super.onDestroy();
	}
}