/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;

import java.util.Arrays;

public class OggPage {

	public static final int HEADER_CONTINUATION = 0x1;
	public static final int HEADER_BEGINNING_OF_STREAM = 0x2;
	public static final int HEADER_END_OF_STREAM = 0x4;

	// 0-3 Capture pattern
	// 4 Version

	// 5 Header type (1 byte)
	private int headerType;

	// 6-13 Granule position (8 bytes)
	private long granulePosition;

	// 14-17 Bitstream serial number (4 bytes)
	private int bitstreamSerialNumber;

	// 18-21 Page Sequence Number (4 bytes)
	private int pageSequenceNumber;

	// 22-25 Checksum (4 bytes)
	private int checksum;

	// 26 Page segments (1 byte)
	private int pageSegments;

	// 27- Segment table
	private int[] segmentLengths;

	public OggPage(OggStream is) throws OggFormatException,
			EndOfStreamException, InvalidStreamException {
		/*
		 * 0 0x4f 'O' 1 0x67 'g' 2 0x67 'g' 3 0x53 'S'
		 */
		int char1 = is.read8();
		int char2 = is.read8();
		int char3 = is.read8();
		int char4 = is.read8();

		if (char1 != 0x4f || //
				char2 != 0x67 || //
				char3 != 0x67 || //
				char4 != 0x53) {
			throw new OggFormatException("Invalid capture pattern (0x"
					+ Integer.toHexString(char1) + ",0x"
					+ Integer.toHexString(char2) + ",0x"
					+ Integer.toHexString(char3) + ",0x"
					+ Integer.toHexString(char4) + ")");
		}

		if (is.read8() != 0x00) {
			throw new OggFormatException("Invalid version number");
		}

		this.headerType = is.read8();
		this.granulePosition = is.read64();
		this.bitstreamSerialNumber = is.read32();
		this.pageSequenceNumber = is.read32();
		this.checksum = is.read32();
		this.pageSegments = is.read8();
		this.segmentLengths = new int[this.pageSegments];
		for (int i = 0; i < pageSegments; ++i) {
			segmentLengths[i] = is.read8();
		}
	}

	public boolean isContinuation() {
		return (this.headerType & HEADER_CONTINUATION) != 0;
	}

	public boolean isBeginningOfStream() {
		return (this.headerType & HEADER_BEGINNING_OF_STREAM) != 0;
	}

	public boolean isEndOfStream() {
		return (this.headerType & HEADER_END_OF_STREAM) != 0;
	}

	public long getGranulePosition() {
		return granulePosition;
	}

	public int getBitstreamSerialNumber() {
		return bitstreamSerialNumber;
	}

	public int getPageSequenceNumber() {
		return pageSequenceNumber;
	}

	public int getChecksum() {
		return checksum;
	}

	public int[] getSegmentLengths() {
		return segmentLengths;
	}

	public int getSegmentCount() {
		return pageSegments;
	}

	@Override
	public String toString() {
		return "OggPage [bitstreamSerialNumber=" + bitstreamSerialNumber
				+ ", checksum=" + checksum + ", granulePosition="
				+ granulePosition + ", headerType=" + headerType
				+ ", pageSegments=" + pageSegments + ", pageSequenceNumber="
				+ pageSequenceNumber + ", segmentLengths="
				+ Arrays.toString(segmentLengths) + "]";
	}

}
