/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;


public class BufferedOggStream extends OggStream implements BufferListenerable {

	private BlockingRingBuffer buffer;
	private OggStream source;
	private byte[] buf = new byte[1024];
	private long readCounter = 0;
	private long writeCounter = 0;
	private boolean closed = false;
	private BufferStatusListener bufferStatusListener;
	private int bufSize;
	
	private class FillThread implements Runnable {
		@Override
		public void run() {
			try {
				while (true) {
					source.read(buf, 0, buf.length);
					buffer.blockingWrite(buf, 0, buf.length);
					writeCounter += buf.length;
					if (bufferStatusListener != null) {
						bufferStatusListener.updateBytesReadable((int)(writeCounter-readCounter),bufSize);
					}
				}
			} catch (EndOfStreamException e) {
			} catch (InvalidStreamException e) {
			}
			buffer.setEndOfStream();
			closed = true;
		}
	}

	public BufferedOggStream(OggStream source, int bufSize) {
		this.source = source;
		this.buffer = new BlockingRingBuffer(bufSize);
		this.bufSize = bufSize;
		Thread t = new Thread(new FillThread());
		t.setPriority(Thread.MIN_PRIORITY);
		t.start();
	}

	/* (non-Javadoc)
	 * @see uk.ac.cam.acr31.oggdecoder.BufferListenerable#setBufferStatusListener(uk.ac.cam.acr31.oggdecoder.BufferStatusListener)
	 */
	public void setBufferStatusListener(BufferStatusListener l) {
		this.bufferStatusListener = l;
	}
	
	private void throwEOS() throws EndOfStreamException {
		if (this.bufferStatusListener != null) this.bufferStatusListener.endOfStream();
		throw new EndOfStreamException();
	}
	
	@Override
	public void read(byte[] buf, int off, int len) throws EndOfStreamException,
			InvalidStreamException {
		while (len > 0) {
			if (closed && readCounter == writeCounter) throwEOS();
			int read = buffer.blockingRead(buf, off, len);
			if (read == -1) {
				throwEOS();
			}
			off += read;
			len -= read;
			readCounter += read;
		}
		if (bufferStatusListener != null) {
			bufferStatusListener.updateBytesReadable((int)(writeCounter-readCounter),bufSize);
		}
	}
	
	public long getAvailable() {
		return writeCounter - readCounter;
	}

	@Override
	public void close() {
		source.close();
	}

}
