/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;

import java.nio.ByteBuffer;

public class RingBuffer {

	private int bufferSize;
	private ByteBuffer buffer;

	private int readPointer;
	private int writePointer;

	private int bytesRemaining;
	private int bytesAvailable;


	public RingBuffer(int bufferSize) {
		this.bufferSize = bufferSize;
		this.buffer = ByteBuffer.allocateDirect(bufferSize);
		readPointer = 0;
		writePointer = 0;
		bytesRemaining = bufferSize;
		bytesAvailable = 0;
	}

	public int getBytesRemaining() {
		return bytesRemaining;
	}

	public int getBytesAvailable() {
		return bytesAvailable;
	}
	
	public void write(byte[] data, int off, int len)
			throws BufferOverflowException {
		if (bytesRemaining < len)
			throw new BufferOverflowException();
		buffer.position(writePointer);
		int toEnd = bufferSize - writePointer;
		if (toEnd > len) {
			buffer.put(data, off, len);
			writePointer += len;
		} else {
			buffer.put(data, 0, toEnd);
			buffer.position(0);
			buffer.put(data, toEnd, len - toEnd);
			writePointer = len - toEnd;
		}
		bytesAvailable += len;
		bytesRemaining -= len;
	}

	public int read(byte[] data, int off, int len)
			throws BufferUnderflowException {
		if (bytesAvailable == 0)
			throw new BufferUnderflowException();
		if (bytesAvailable < len)
			len = bytesAvailable;
		buffer.position(readPointer);
		int toEnd = bufferSize - readPointer;
		if (toEnd > len) {
			buffer.get(data, off, len);
			readPointer += len;
		} else {
			buffer.get(data, off, toEnd);
			buffer.position(0);
			buffer.get(data, toEnd, len - toEnd);
			readPointer = len - toEnd;
		}
		bytesAvailable -= len;
		bytesRemaining += len;
		return len;
	}
}
