(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * (Pretty?) printer for expression trees.                        *
 *                                                                *
 * Version: $Id: pretty.mli,v 1.531 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* Pretty interface file *)

type printer_state

type printer_mode
(* used to be concrete, but now we use pm_user and pm_hash elsewhere it doesn't need to be *)

val set_ignore_defns : printer_state -> int option -> printer_state

val pm_user  : int -> int -> bool -> printer_mode
val pm_hash  : printer_mode
val pm_lib_hack  : printer_mode
val pm_marshal  : unit -> printer_mode
val pm_debug : unit -> printer_mode
val pm_objectcode : int -> printer_mode

val notrail_HACK : printer_state -> printer_state

val initial_printer_state : (Ast.internal_ident -> string) option -> printer_mode -> printer_state
(* the first argument to initial_printer_state is an option:
   Some f , where f is usually Econst.string_of_ident for the printing of external constants
   None   , no such printing

  the second is a printer_mode; see above
*)

val printer_state_after_source_definitions : printer_state -> Ast.source_definitions -> printer_state

val printer_state_after_definitions : printer_state -> Ast.definitions -> printer_state

val wrap : (printer_state -> Format.formatter -> 'a -> unit) -> printer_state -> 'a -> string
val wrap' : (printer_state -> Format.formatter -> 'a -> printer_state) -> printer_state -> 'a -> (string * printer_state)

val null_formatter : Format.formatter

val pp_concat : Format.formatter -> (Format.formatter -> 'a -> unit) -> 'a -> (Format.formatter -> 'b -> unit) -> 'b list -> unit
val pp_concat_ps : printer_state -> Format.formatter -> (Format.formatter -> 'a -> unit) -> 'a -> (printer_state -> Format.formatter -> 'b -> printer_state) -> 'b list -> printer_state

val add_ident_binder : printer_state -> Ast.internal_ident -> printer_state
val add_ident_binders : printer_state -> Ast.internal_ident list -> printer_state

val add_typname_binder : printer_state -> Ast.internal_typname -> printer_state
val add_typname_binders : printer_state -> Ast.internal_typname list -> printer_state

val add_modname_binder : printer_state -> Ast.modname -> printer_state

(* hash and abstract names construction *)
val mkHashM : (Ast.internal_ident -> string) -> Ast.external_modname * Ast.mod_hash_body -> Ast.hash
val mkHashI : (Ast.internal_ident -> string) -> Ast.external_modname * Ast.import_hash_body -> Ast.hash
val hash_of_hash : Ast.hash -> Digest.t
val hash_of_int : int -> Digest.t

val mkAbstrNameMvf : Ast.hash * Ast.external_ident * Ast.typ -> Ast.name_value
val mkAbstrNameTs : Ast.typ * string -> Ast.name_value
val mkAbstrNameHts : Ast.typ * string * Ast.name_value -> Ast.name_value

(* generic functions taking a formatter for outputting onto *)

val pp_print_ident_use : printer_state -> Format.formatter -> Ast.internal_ident -> unit
val pp_print_typname_use : printer_state -> Format.formatter -> Ast.internal_typname -> unit
val pp_print_modname : printer_state -> Format.formatter -> Ast.modname -> unit
val pp_print_name_use : printer_state -> Format.formatter -> Ast.abstract_name -> unit
val pp_print_name_value : printer_state -> Format.formatter -> Ast.name_value -> unit

val pp_print_ident_ext : printer_state -> Format.formatter -> Ast.external_ident -> unit
val pp_print_typname_ext : printer_state -> Format.formatter -> Ast.external_typname -> unit

val pp_print_ident_bind : printer_state -> Format.formatter -> Ast.ident -> unit
val pp_print_typname_bind : printer_state -> Format.formatter -> Ast.typname -> unit

val pp_print_program : printer_state -> Format.formatter -> Ast.program -> unit

val pp_print_signature : printer_state -> Format.formatter -> Ast.signature -> unit
val pp_print_structure : printer_state -> Format.formatter -> Ast.structure -> unit

(* NB: you can't print multiple signature items by calling
print_signature_item multiple times unless you update the
printer_state: otherwise later items won't reflect earlier binders and
the printing will fail with Not_found *)
val pp_print_signature_item : printer_state -> Format.formatter -> Ast.signature_item -> unit

val pp_print_atomic_version : printer_state -> Format.formatter -> Ast.atomic_version -> unit
val pp_print_version : printer_state -> Format.formatter -> Ast.version -> unit
val pp_print_version_constraint : printer_state -> Format.formatter -> Ast.version_constraint -> unit
val pp_print_withspec : printer_state -> Format.formatter -> Ast.withspec -> unit
val pp_print_likespec : printer_state -> Format.formatter -> Ast.likespec -> unit
val pp_print_resolvespec_item : printer_state -> Format.formatter -> Ast.resolvespec_item -> unit
val pp_print_resolvespec : printer_state -> Format.formatter -> Ast.resolvespec -> unit

(* print_typeenv
Note: the environment comes in ``reverse'' order: later entries in the
list bind earlier ones.

Note: it will invent names for all the binders: the only reason to
pass in a non-trivial printer_state is if there are other bound names.

Note: bound idents that are from the dom of Econst are automatically
recognised and their correct name is printed.
*)

val pp_print_nameenv_entry : printer_state -> Format.formatter -> Ast.nameenv_entry -> unit
val pp_print_nameenv : printer_state -> Format.formatter -> Ast.nameenv -> unit
val pp_print_typeenv_ps : printer_state -> Format.formatter -> Ast.typeenv -> printer_state
val pp_print_nametypeenv_ps : printer_state -> Format.formatter -> Ast.nametypeenv -> printer_state
val pp_print_store : printer_state -> Format.formatter -> Ast.store -> unit

val pp_print_esubst : printer_state -> Format.formatter -> Ast.esubst -> unit
val pp_print_tysubst : printer_state -> Format.formatter -> Ast.tysubst -> unit
val pp_print_etysubst : printer_state -> Format.formatter -> Ast.etysubst -> unit


val pp_print_typ : printer_state -> Format.formatter -> Ast.typ -> unit
val pp_print_kind : printer_state -> Format.formatter -> Ast.kind -> unit

val canonicalise_eqs : (Ast.eqs -> Ast.eqs) option ref  (* ugh, avoid recursive dependency *)
val pp_print_eq : printer_state -> Format.formatter -> Ast.eqn -> unit
val pp_print_eqs : printer_state -> Format.formatter -> Ast.eqs -> unit

val pp_print_prim_expr : printer_state -> Format.formatter -> Ast.prim_expr -> unit
val pp_print_expr : printer_state -> Format.formatter -> Ast.expr -> unit
val pp_print_prim_expb : printer_state -> Format.formatter -> Ast.prim_expr -> unit
val pp_print_mtch : printer_state -> Format.formatter -> Ast.mtch -> unit

(* NB: you can't print a pattern and the expression it binds in by
calling print_pat and then print_expr unless you update the
printer_state: otherwise names bound by the pattern won't be available
when printing the expression and the printing will fail with Not_found *)
val pp_print_pat : printer_state -> Format.formatter -> Ast.pat -> unit
val pp_print_pat_ps : printer_state -> Format.formatter -> Ast.pat -> printer_state

val pp_print_source_definition : printer_state -> Format.formatter -> Ast.source_definition -> unit
val pp_print_source_definition_ps : printer_state -> Format.formatter -> Ast.source_definition -> printer_state

(* print_source_definitions{._ps} expects the definitions to already be freshened *)
val pp_print_source_definitions : printer_state -> Format.formatter -> Ast.source_definitions -> unit
val pp_print_source_definitions_ps : printer_state -> Format.formatter -> Ast.source_definitions -> printer_state

val pp_print_definition : printer_state -> Format.formatter -> Ast.definition -> unit
val pp_print_definition_ps : printer_state -> Format.formatter -> Ast.definition -> printer_state

(* print_definitions{._ps} expects the definitions to already be freshened *)
val pp_print_definitions : printer_state -> Format.formatter -> Ast.definitions -> unit
val pp_print_definitions_ps : printer_state -> Format.formatter -> Ast.definitions -> printer_state

val pp_print_location : printer_state -> Format.formatter -> Ast.location -> unit
val pp_print_source_loc : printer_state -> Format.formatter -> Location.t -> unit

val pp_print_compilation_unit : printer_state -> Format.formatter -> 'a Ast.compilation_unit -> printer_state

val pp_print_configuration : printer_state -> Format.formatter -> Ast.configuration -> unit
(* val pp_print_smallstep_inner : printer_state -> Format.formatter -> Ast.smallstep_inner -> unit *)
val pp_print_threaded_smallstep_inner :  printer_state -> Format.formatter -> Ast.threaded_smallstep_inner -> unit
val pp_print_thread_info :  printer_state -> Format.formatter -> Ast.thread_info -> unit

val pp_print_smallstep_outer : printer_state -> Format.formatter -> Ast.smallstep_outer -> unit

(* val pp_print_conf_or_pseudo_conf_pair : printer_state -> Format.formatter ->
  ((Ast.smallstep_outer * Ast.threaded_smallstep_inner), (Ast.pseudo_configuration * Ast.smallstep_inner_info)) Util.sum
     -> unit*)


(* print_hash and print_marshalled_body take a printer_state since
their arguments may contain econst idents and the
printer_state.econst_string_of_ident field tells them how to render
these idents.  A well-typed hash or marshalled_body will have no other
free variables of any flavour so the other printer_state fields are
not used. *)

val pp_print_hash : printer_state -> Format.formatter -> Ast.hash -> unit
val pp_print_marshalled_value : printer_state -> Format.formatter -> Ast.marshalled_value -> unit


(* and specialised string versions of all the above *)

val print_ident_use : printer_state -> Ast.internal_ident -> string
val print_typname_use : printer_state -> Ast.internal_typname -> string
val print_name_use : printer_state -> Ast.abstract_name -> string
val print_name_value : printer_state -> Ast.name_value -> string
val print_modname : printer_state -> Ast.modname -> string
val print_ident_ext : printer_state -> Ast.external_ident -> string
val print_typname_ext : printer_state -> Ast.external_typname -> string
val print_ident_bind : printer_state -> Ast.ident -> string
val print_typname_bind : printer_state -> Ast.typname -> string
val print_program : printer_state -> Ast.program -> string
val print_signature : printer_state -> Ast.signature -> string
val print_signature_ps : printer_state -> Ast.signature -> string * printer_state
val print_structure : printer_state -> Ast.structure -> string
val print_signature_item : printer_state -> Ast.signature_item -> string
val print_atomic_version : printer_state -> Ast.atomic_version -> string
val print_version : printer_state -> Ast.version -> string
val print_version_constraint : printer_state -> Ast.version_constraint -> string
val print_withspec : printer_state -> Ast.withspec -> string
val print_likespec : printer_state -> Ast.likespec -> string
val print_resolvespec_item : printer_state -> Ast.resolvespec_item -> string
val print_resolvespec : printer_state -> Ast.resolvespec -> string
val print_nameenv : printer_state -> Ast.nameenv -> string
val print_typeenv_ps : printer_state -> Ast.typeenv -> string * printer_state
val print_nametypeenv_ps : printer_state -> Ast.nametypeenv -> string * printer_state
val print_store : printer_state -> Ast.store -> string
val print_esubst : printer_state -> Ast.esubst -> string
val print_tysubst : printer_state -> Ast.tysubst -> string
val print_etysubst : printer_state -> Ast.etysubst -> string
val print_typ : printer_state -> Ast.typ -> string
val print_kind : printer_state -> Ast.kind -> string
val print_eq : printer_state -> Ast.eqn -> string
val print_eqs : printer_state -> Ast.eqs -> string
val print_prim_expr : printer_state -> Ast.prim_expr -> string
val print_expr : printer_state -> Ast.expr -> string
val print_expr_loc : printer_state -> Ast.expr -> string
(*val print_prim_mtch : printer_state -> Ast.prim_mtch -> string*)
val print_pat : printer_state -> Ast.pat -> string
val print_pat_ps : printer_state -> Ast.pat -> string * printer_state
val print_source_definition : printer_state -> Ast.source_definition -> string
val print_source_definition_ps :
  printer_state -> Ast.source_definition -> string * printer_state
val print_source_definitions : printer_state -> Ast.source_definitions -> string
val print_source_definitions_ps :
  printer_state -> Ast.source_definitions -> string * printer_state
val print_definition : printer_state -> Ast.definition -> string
val print_definition_ps :
  printer_state -> Ast.definition -> string * printer_state
val print_definitions : printer_state -> Ast.definitions -> string
val print_definitions_ps :
  printer_state -> Ast.definitions -> string * printer_state
val print_location : printer_state -> int -> string
val print_compilation_unit : printer_state -> 'a Ast.compilation_unit -> string
val print_compiled_unit : printer_state -> Ast.compiled_unit -> string
val print_configuration : printer_state -> Ast.configuration -> string
(* val print_smallstep_inner : printer_state -> Ast.smallstep_inner -> string *)
val print_smallstep_outer : printer_state -> Ast.smallstep_outer -> string
val print_threaded_smallstep_inner : printer_state -> Ast.threaded_smallstep_inner -> string
val print_thread_info : printer_state -> Ast.thread_info -> string

val print_hash : printer_state -> Ast.hash -> string
val print_marshalled_value : printer_state -> Ast.marshalled_value -> string
