(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * Base constructors (see also Baseconty)                         *
 *                                                                *
 * Version: $Id: basecon.ml,v 1.6 2004/12/31 11:49:37 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)



(* Everything to do with base constructors is defined in these files; to
   add a new one, just extend the relevant definitions. *)

open Format


(* == Type constructors == *)

type tycon0 =
  | TInt | TBool | TString | TUnit | TChar       (* basic types *)
  | TVoid                                        (* empty type *)
  | TExn                                         (* exception type *)
  | TThread | TMutex | TCVar                     (* concurrency: gadgets *)
  | TThunkifymode | TThunkkey | TThunklet        (* concurrency: thunkification *)
  | TUnixErrorCode

type tycon1 =
  | TList                                        (* list type *)
  | TOption                                      (* option type *)
  | TRef                                         (* reference type *)
  | TName                                        (* type of names *)
  | TTie                                         (* type of ties *)
  | TTree                                        (* type of tree FZ *)

(* == Constructor support types == *)

type unixerrorcode =
  | E2BIG
  | EACCES
  | EADDRINUSE
  | EADDRNOTAVAIL
  | EAFNOSUPPORT
  | EAGAIN
  | EWOULDBLOCK (* only if EWOULDBLOCK <> EAGAIN *)
  | EALREADY
  | EBADF
  | EBADMSG
  | EBUSY
  | ECANCELED
  | ECHILD
  | ECONNABORTED
  | ECONNREFUSED
  | ECONNRESET
  | EDEADLK
  | EDESTADDRREQ
  | EDOM
  | EDQUOT
  | EEXIST
  | EFAULT
  | EFBIG
  | EHOSTUNREACH
  | EIDRM
  | EILSEQ
  | EINPROGRESS
  | EINTR
  | EINVAL
  | EIO
  | EISCONN
  | EISDIR
  | ELOOP
  | EMFILE
  | EMLINK
  | EMSGSIZE
  | EMULTIHOP
  | ENAMETOOLONG
  | ENETDOWN
  | ENETRESET
  | ENETUNREACH
  | ENFILE
  | ENOBUFS
  | ENODATA
  | ENODEV
  | ENOENT
  | ENOEXEC
  | ENOLCK
  | ENOLINK
  | ENOMEM
  | ENOMSG
  | ENOPROTOOPT
  | ENOSPC
  | ENOSR
  | ENOSTR
  | ENOSYS
  | ENOTCONN
  | ENOTDIR
  | ENOTEMPTY
  | ENOTSOCK
  | ENOTSUP
  | ENOTTY
  | ENXIO
  | EOPNOTSUPP
  | EOVERFLOW
  | EPERM
  | EPIPE
  | EPROTO
  | EPROTONOSUPPORT
  | EPROTOTYPE
  | ERANGE
  | EROFS
  | ESPIPE
  | ESRCH
  | ESTALE
  | ETIME
  | ETIMEDOUT
  | ETXTBSY
  | EXDEV
  | ESHUTDOWN
  | EHOSTDOWN
  | EUNKNOWN_UNIX_ERROR


(* == Basic constructors of arity 0 (no type arguments) == *)

type basecon0 =
  | Unit
  | Int of int
  | Bool of bool
  | Char of char
  | String of string
  | MarshalFail
  | ResolveFail
  | NotFound
  | EndOfFile
  | DivisionByZero
  | SysBlockedIO
  | NonExistentThread
  | NonExistentMutex
  | NonExistentCVar
  | ExistentName
  | MutexEPERM
  | ThunkifyEINTR
  | ThunkifySelf
  | ThunkifyKeylistMismatch
  | ThunkifyThreadInDefinition
  | Interrupting  (* thunkifymode *)
  | Blocking      (* thunkifymode *)
  | UnixErrorCode of unixerrorcode


(* == Basic constructors of arity 1 (no type arguments, no type inference required) == *)

type basecon1 =
  | Fail
  | InvalidArgument
  | SysError
  | MatchFail
  | UnmarshalFail
  | Thread
  | Mutex
  | CVar
  | ThunkedThread
  | ThunkedMutex
  | ThunkedCVar
  | UnixError (* of unixerrorcode * string * string *)
  | LibraryError

(* == Constructor -> type constructor correspondence == *)

let tyconof_B0 : basecon0 -> tycon0
  = fun c0
 -> match c0 with
    | Unit      -> TUnit
    | Int(_)    -> TInt
    | Bool(_)   -> TBool
    | Char(_)   -> TChar
    | String(_) -> TString
    | ResolveFail | MarshalFail
    | SysBlockedIO | DivisionByZero | EndOfFile | NotFound    -> TExn
    | Blocking | Interrupting                                 -> TThunkifymode
    | ThunkifyThreadInDefinition | ThunkifyKeylistMismatch
    | ThunkifySelf | ThunkifyEINTR
    | ExistentName
    | NonExistentCVar | NonExistentMutex | NonExistentThread
    | MutexEPERM                                                   -> TExn
    | UnixErrorCode _ -> TUnixErrorCode

let tyconof_B1 : basecon1 -> tycon0
  = fun c1
 -> match c1 with
    | Fail            -> TExn
    | InvalidArgument -> TExn
    | SysError        -> TExn
    | MatchFail       -> TExn
    | UnmarshalFail   -> TExn
    | ThunkedCVar     -> TThunklet
    | ThunkedMutex    -> TThunklet
    | ThunkedThread   -> TThunklet
    | CVar            -> TThunkkey
    | Mutex           -> TThunkkey
    | Thread          -> TThunkkey
    | UnixError       -> TExn
    | LibraryError    -> TExn
(* NB: argument types for B1 are given in Baseconty *)


(* == Printing tycons == *)

let pp_print_tycon0 ppf tc0 =
  let s =
    match tc0 with
    | TInt           -> "int"
    | TBool          -> "bool"
    | TString        -> "string"
    | TUnit          -> "unit"
    | TChar          -> "char"
    | TVoid          -> "void"
    | TExn           -> "exn"
    | TThread        -> "thread"
    | TMutex         -> "mutex"
    | TCVar          -> "cvar"
    | TThunkifymode  -> "thunkifymode"
    | TThunkkey      -> "thunkkey"
    | TThunklet      -> "thunklet"
    | TUnixErrorCode -> "unix_error_code"   (* hmm, parse error? *)
  in
  pp_print_string ppf s

let pp_print_tycon1 ps ppf tc1 =
  let s = match tc1 with
  | TList   -> "list"
  | TOption -> "option"
  | TRef    -> "ref"
  | TName   -> "name"
  | TTie    -> "tie"
  | TTree   -> "tree"
  in
  pp_print_string ppf s


(* == Printing constructors == *)

let pp_print_unix_error ppf e =
  let s =
    match e with
    | E2BIG          -> "E2BIG"
    | EACCES         -> "EACCES"
    | EADDRINUSE     -> "EADDRINUSE"
    | EADDRNOTAVAIL  -> "EADDRNOTAVAIL"
    | EAFNOSUPPORT   -> "EAFNOSUPPORT"
    | EAGAIN         -> "EAGAIN"
    | EWOULDBLOCK    -> "EWOULDBLOCK"  (* only if EWOULDBLOCK <> EAGAIN *)
    | EALREADY       -> "EALREADY"
    | EBADF          -> "EBADF"
    | EBADMSG        -> "EBADMSG"
    | EBUSY          -> "EBUSY"
    | ECANCELED      -> "ECANCELED"
    | ECHILD         -> "ECHILD"
    | ECONNABORTED   -> "ECONNABORTED"
    | ECONNREFUSED   -> "ECONNREFUSED"
    | ECONNRESET     -> "ECONNRESET-"
    | EDEADLK        -> "EDEADLK"
    | EDESTADDRREQ   -> "EDESTADDRREQ"
    | EDOM           -> "EDOM"
    | EDQUOT         -> "EDQUOT"
    | EEXIST         -> "EEXIST"
    | EFAULT         -> "EFAULT"
    | EFBIG          -> "EFBIG"
    | EHOSTUNREACH   -> "EHOSTUNREACH"
    | EIDRM          -> "EIDRM"
    | EILSEQ         -> "EILSEQ"
    | EINPROGRESS    -> "EINPROGRESS"
    | EINTR          -> "EINTR"
    | EINVAL         -> "EINVAL"
    | EIO            -> "EIO"
    | EISCONN        -> "EISCONN"
    | EISDIR         -> "EISDIR"
    | ELOOP          -> "ELOOP"
    | EMFILE         -> "EMFILE"
    | EMLINK         -> "EMLINK"
    | EMSGSIZE       -> "EMSGSIZE"
    | EMULTIHOP      -> "EMULTIHOP"
    | ENAMETOOLONG   -> "ENAMETOOLONG"
    | ENETDOWN       -> "ENETDOWN"
    | ENETRESET      -> "ENETRESET"
    | ENETUNREACH    -> "ENETUNREACH"
    | ENFILE         -> "ENFILE"
    | ENOBUFS        -> "ENOBUFS"
    | ENODATA        -> "ENODATA"
    | ENODEV         -> "ENODEV"
    | ENOENT         -> "ENOENT"
    | ENOEXEC        -> "ENOEXEC"
    | ENOLCK         -> "ENOLCK"
    | ENOLINK        -> "ENOLINK"
    | ENOMEM         -> "ENOMEM"
    | ENOMSG         -> "ENOMSG"
    | ENOPROTOOPT    -> "ENOPROTOOPT"
    | ENOSPC         -> "ENOSPC"
    | ENOSR          -> "ENOSR"
    | ENOSTR         -> "ENOSTR"
    | ENOSYS         -> "ENOSYS"
    | ENOTCONN       -> "ENOTCONN"
    | ENOTDIR        -> "ENOTDIR"
    | ENOTEMPTY      -> "ENOTEMPTY"
    | ENOTSOCK       -> "ENOTSOCK"
    | ENOTSUP        -> "ENOTSUP"
    | ENOTTY         -> "ENOTTY"
    | ENXIO          -> "ENXIO"
    | EOPNOTSUPP     -> "EOPNOTSUPP"
    | EOVERFLOW      -> "EOVERFLOW"
    | EPERM          -> "EPERM"
    | EPIPE          -> "EPIPE"
    | EPROTO         -> "EPROTO"
    | EPROTONOSUPPORT-> "EPROTONOSUPPORT"
    | EPROTOTYPE     -> "EPROTOTYPE"
    | ERANGE         -> "ERANGE"
    | EROFS          -> "EROFS"
    | ESPIPE         -> "ESPIPE"
    | ESRCH          -> "ESRCH"
    | ESTALE         -> "ESTALE"
    | ETIME          -> "ETIME"
    | ETIMEDOUT      -> "ETIMEDOUT"
    | ETXTBSY        -> "ETXTBSY"
    | EXDEV          -> "EXDEV"
    | ESHUTDOWN      -> "ESHUTDOWN"
    | EHOSTDOWN      -> "EHOSTDOWN"
    | EUNKNOWN_UNIX_ERROR -> "EUNKNOWN_UNIX_ERROR"
  in
  pp_print_string ppf s

let pp_print_basecon0 ppf c =
  match c with
  | Unit                       -> pp_print_string ppf "()"
  | Int i                      -> pp_print_int ppf i
  | Bool b                     -> pp_print_bool ppf b
  | Char c                     -> pp_print_string ppf ("'" ^ (Char.escaped c) ^ "'")
  | String s                   -> pp_print_string ppf ("\"" ^  String.escaped s ^ "\"")
  | MarshalFail                -> pp_print_string ppf "Marshal_failure"
  | ResolveFail                -> pp_print_string ppf "Resolve_failure"
  | NotFound                   -> pp_print_string ppf "Not_found"
  | EndOfFile                  -> pp_print_string ppf "End_of_file"
  | DivisionByZero             -> pp_print_string ppf "Division_by_zero"
  | SysBlockedIO               -> pp_print_string ppf "Sys_blocked_IO"
  | NonExistentThread          -> pp_print_string ppf "Nonexistent_thread"
  | NonExistentMutex           -> pp_print_string ppf "Nonexistent_mutex"
  | NonExistentCVar            -> pp_print_string ppf "Nonexistent_cvar"
  | ExistentName               -> pp_print_string ppf "Existent_name"
  | MutexEPERM                 -> pp_print_string ppf "Mutex_EPERM"
  | ThunkifyEINTR              -> pp_print_string ppf "Thunkify_EINTR"
  | ThunkifySelf               -> pp_print_string ppf "Thunkify_self"
  | ThunkifyKeylistMismatch    -> pp_print_string ppf "Thunkify_keylists_mismatch"
  | ThunkifyThreadInDefinition -> pp_print_string ppf "Thunkify_thread_in_definition"
  | Interrupting               -> pp_print_string ppf "Interrupting"
  | Blocking                   -> pp_print_string ppf "Blocking"
  | UnixErrorCode e            -> pp_print_unix_error ppf e

let pp_print_basecon1 ppf c =
  let s =
    match c with
    | Fail            -> "Failure"
    | InvalidArgument -> "Invalid_argument"
    | SysError        -> "Sys_error"
    | MatchFail       -> "Match_failure"
    | UnmarshalFail   -> "Unmarshal_failure"
    | Thread          -> "Thread"
    | Mutex           -> "Mutex"
    | CVar            -> "CVar"
    | ThunkedThread   -> "Thunked_thread"
    | ThunkedMutex    -> "Thunked_mutex"
    | ThunkedCVar     -> "Thunked_cvar"
    | UnixError       -> "UnixError"
    | LibraryError    -> "Library_error"
  in
  pp_print_string ppf s


(* == Lexing base constructors == *)

(* Parsing for Unit, Int, Bool, Char, and String is special-cased in
   parser.mlyp.  All others are given here. *)

let basecon0_lex_alist = [
    "Blocking"                     , Blocking;
    "Division_by_zero"             , DivisionByZero;
    "End_of_file"                  , EndOfFile;
    "Existent_name"                , ExistentName;
    "Interrupting"                 , Interrupting;
    "Marshal_failure"              , MarshalFail;
    "Mutex_EPERM"                  , MutexEPERM;
    "Nonexistent_cvar"             , NonExistentCVar;
    "Nonexistent_mutex"            , NonExistentMutex;
    "Nonexistent_thread"           , NonExistentThread;
    "Not_found"                    , NotFound;
    "Resolve_failure"              , ResolveFail;
    "Sys_blocked_io"               , SysBlockedIO;
    "Thunkify_EINTR"               , ThunkifyEINTR;
    "Thunkify_keylist_mismatch"    , ThunkifyKeylistMismatch;
    "Thunkify_self"                , ThunkifySelf;
    "Thunkify_thread_in_definition", ThunkifyThreadInDefinition;

    "E2BIG"                 , UnixErrorCode E2BIG;
    "EACCES"                , UnixErrorCode EACCES;
    "EADDRINUSE"            , UnixErrorCode EADDRINUSE;
    "EADDRNOTAVAIL"         , UnixErrorCode EADDRNOTAVAIL;
    "EAFNOSUPPORT"          , UnixErrorCode EAFNOSUPPORT;
    "EAGAIN"                , UnixErrorCode EAGAIN;
    "EWOULDBLOCK"           , UnixErrorCode EWOULDBLOCK;
    "EALREADY"              , UnixErrorCode EALREADY;
    "EBADF"                 , UnixErrorCode EBADF;
    "EBADMSG"               , UnixErrorCode EBADMSG;
    "EBUSY"                 , UnixErrorCode EBUSY;
    "ECANCELED"             , UnixErrorCode ECANCELED;
    "ECHILD"                , UnixErrorCode ECHILD;
    "ECONNABORTED"          , UnixErrorCode ECONNABORTED;
    "ECONNREFUSED"          , UnixErrorCode ECONNREFUSED;
    "ECONNRESET"            , UnixErrorCode ECONNRESET;
    "EDEADLK"               , UnixErrorCode EDEADLK;
    "EDESTADDRREQ"          , UnixErrorCode EDESTADDRREQ;
    "EDOM"                  , UnixErrorCode EDOM;
    "EDQUOT"                , UnixErrorCode EDQUOT;
    "EEXIST"                , UnixErrorCode EEXIST;
    "EFAULT"                , UnixErrorCode EFAULT;
    "EFBIG"                 , UnixErrorCode EFBIG;
    "EHOSTUNREACH"          , UnixErrorCode EHOSTUNREACH;
    "EIDRM"                 , UnixErrorCode EIDRM;
    "EILSEQ"                , UnixErrorCode EILSEQ;
    "EINPROGRESS"           , UnixErrorCode EINPROGRESS;
    "EINTR"                 , UnixErrorCode EINTR;
    "EINVAL"                , UnixErrorCode EINVAL;
    "EIO"                   , UnixErrorCode EIO;
    "EISCONN"               , UnixErrorCode EISCONN;
    "EISDIR"                , UnixErrorCode EISDIR;
    "ELOOP"                 , UnixErrorCode ELOOP;
    "EMFILE"                , UnixErrorCode EMFILE;
    "EMLINK"                , UnixErrorCode EMLINK;
    "EMSGSIZE"              , UnixErrorCode EMSGSIZE;
    "EMULTIHOP"             , UnixErrorCode EMULTIHOP;
    "ENAMETOOLONG"          , UnixErrorCode ENAMETOOLONG;
    "ENETDOWN"              , UnixErrorCode ENETDOWN;
    "ENETRESET"             , UnixErrorCode ENETRESET;
    "ENETUNREACH"           , UnixErrorCode ENETUNREACH;
    "ENFILE"                , UnixErrorCode ENFILE;
    "ENOBUFS"               , UnixErrorCode ENOBUFS;
    "ENODATA"               , UnixErrorCode ENODATA;
    "ENODEV"                , UnixErrorCode ENODEV;
    "ENOENT"                , UnixErrorCode ENOENT;
    "ENOEXEC"               , UnixErrorCode ENOEXEC;
    "ENOLCK"                , UnixErrorCode ENOLCK;
    "ENOLINK"               , UnixErrorCode ENOLINK;
    "ENOMEM"                , UnixErrorCode ENOMEM;
    "ENOMSG"                , UnixErrorCode ENOMSG;
    "ENOPROTOOPT"           , UnixErrorCode ENOPROTOOPT;
    "ENOSPC"                , UnixErrorCode ENOSPC;
    "ENOSR"                 , UnixErrorCode ENOSR;
    "ENOSTR"                , UnixErrorCode ENOSTR;
    "ENOSYS"                , UnixErrorCode ENOSYS;
    "ENOTCONN"              , UnixErrorCode ENOTCONN;
    "ENOTDIR"               , UnixErrorCode ENOTDIR;
    "ENOTEMPTY"             , UnixErrorCode ENOTEMPTY;
    "ENOTSOCK"              , UnixErrorCode ENOTSOCK;
    "ENOTSUP"               , UnixErrorCode ENOTSUP;
    "ENOTTY"                , UnixErrorCode ENOTTY;
    "ENXIO"                 , UnixErrorCode ENXIO;
    "EOPNOTSUPP"            , UnixErrorCode EOPNOTSUPP;
    "EOVERFLOW"             , UnixErrorCode EOVERFLOW;
    "EPERM"                 , UnixErrorCode EPERM;
    "EPIPE"                 , UnixErrorCode EPIPE;
    "EPROTO"                , UnixErrorCode EPROTO;
    "EPROTONOSUPPORT"       , UnixErrorCode EPROTONOSUPPORT;
    "EPROTOTYPE"            , UnixErrorCode EPROTOTYPE;
    "ERANGE"                , UnixErrorCode ERANGE;
    "EROFS"                 , UnixErrorCode EROFS;
    "ESPIPE"                , UnixErrorCode ESPIPE;
    "ESRCH"                 , UnixErrorCode ESRCH;
    "ESTALE"                , UnixErrorCode ESTALE;
    "ETIME"                 , UnixErrorCode ETIME;
    "ETIMEDOUT"             , UnixErrorCode ETIMEDOUT;
    "ETXTBSY"               , UnixErrorCode ETXTBSY;
    "EXDEV"                 , UnixErrorCode EXDEV;
    "ESHUTDOWN"             , UnixErrorCode ESHUTDOWN;
    "EHOSTDOWN"             , UnixErrorCode EHOSTDOWN;
    "EUNKNOWN_UNIX_ERROR"   , UnixErrorCode EUNKNOWN_UNIX_ERROR;
]

let basecon1_lex_alist = [
    "Failure"          , Fail;
    "Invalid_argument" , InvalidArgument;
    "Library_error"    , LibraryError;
    "Match_failure"    , MatchFail;
    "Sys_error"        , SysError;
    "UnixError"        , UnixError;
    "Unmarshal_failure", UnmarshalFail;
    "CVar"             , CVar;
    "Mutex"            , Mutex;
    "Thread"           , Thread;
    "Thunked_cvar"     , ThunkedCVar;
    "Thunked_mutex"    , ThunkedMutex;
    "Thunked_thread"   , ThunkedThread;
]

(* === GENLIB SUPPORT === *)

(* == Rendering Acute tycons as themselves == *)

let rec dump_tycon0 (tc0 : tycon0) =
  match tc0 with
  | TInt          -> "TInt"
  | TBool         -> "TBool"
  | TString       -> "TString"
  | TUnit         -> "TUnit"
  | TChar         -> "TChar"
  | TVoid         -> "TVoid"
  | TExn          -> "TExn"
  | TThread       -> "TThread"
  | TMutex        -> "TMutex"
  | TCVar         -> "TCVar"
  | TThunkifymode -> "TThunkifymode"
  | TThunkkey     -> "TThunkkey"
  | TThunklet     -> "TThunklet"
  | TUnixErrorCode -> "TUnixErrorCode"

let rec dump_tycon1 (tc1 : tycon1) =
  match tc1 with
  | TList   -> "TList"
  | TOption -> "TOption"
  | TRef    -> "TRef"
  | TName   -> "TName"
  | TTie    -> "TTie"
  | TTree   -> "TTree"


(* == Runtime type errors that *should* have been caught by the typesystem... == *)

exception RT_type


(* == Embedding/projecting Acute tycons into FreshOCaml == *)

let tycon0_embedding_fun (tc0 : tycon0) =
  match tc0 with
  | TInt           -> "ep_int"
  | TBool          -> "ep_bool"
  | TChar          -> "ep_char"
  | TString        -> "ep_string"
  | TUnit          -> "ep_unit"
  | TUnixErrorCode -> "ep_unixerrorcode"
  | _              -> assert false        (* this should never happen *)

let ep_toInt           x = Int x
let ep_toBool          x = Bool x
let ep_toChar          x = Char x
let ep_toString        x = String x
let ep_toUnit         () = Unit
let ep_toUnixErrorCode x = UnixErrorCode x

let ep_fromBool          = function Bool   x -> x  | _ -> raise RT_type
let ep_fromChar          = function Char   x -> x  | _ -> raise RT_type
let ep_fromInt           = function Int    x -> x  | _ -> raise RT_type
let ep_fromString        = function String x -> x  | _ -> raise RT_type
let ep_fromUnit          = function Unit     -> () | _ -> raise RT_type
let ep_fromUnixErrorCode = function UnixErrorCode x -> x | _ -> raise RT_type

let ep_int           = (TInt          , ep_toInt          , ep_fromInt          )
let ep_bool          = (TBool         , ep_toBool         , ep_fromBool         )
let ep_char          = (TChar         , ep_toChar         , ep_fromChar         )
let ep_string        = (TString       , ep_toString       , ep_fromString       )
let ep_unit          = (TUnit         , ep_toUnit         , ep_fromUnit         )
let ep_unixerrorcode = (TUnixErrorCode, ep_toUnixErrorCode, ep_fromUnixErrorCode)

