
/*
A genome is a sequence of bases specified by the letters A, C, G and
T.  This progam converts a given sequence into a 3D coloured path
following a route based on the sequence of bases and a set of angle
specifying how the genome bends at each base pair junction. The
program in interactive allowing the angles and viewing parameters to
be varied.


Implemented by Martin Richards (c) 23 October 2020

The current plan is to display an image of the genome as a sequence as
coloured strips using the BCPL OpenGL interface. The path along the
genome will rotate by an amount depending on the pair of bases
involved. These rotations can be thought of as adjustments made by
elevator, aileron and rudder controls. The 16 sets of three angles can
be adjusted interactively causing the resulting genome image to change
shape. The genome sequence is provided by the FROM file.  Only the
characters A, C, G and T not in comment lines are used. The initial
setting of the rotational angles is given by a data file. 
The current set of angles can be saved to file using the W command..

History

16/11/2020
Started to make a substantial modification causing the genome model
to remain fixed in space but allowing the position and orientation
to be moved.
N/n move the eye forward/back in direction N.
E/e move the eye forward/back in direction E.
H/h move the eye forward/back in direction H.
F/f move the eye forward/back in the direction of view.
The eye orientation can be controlled by the arrow keys and < and >
which behave like the joystick and rudder peddles of an aircraft.


28/06/2020
Started initial implementation.



Usage

dnaplot "FROM,ANGLES,-d/s,-t/s"

FROM     gives the filename of the file of genome letters. The default
         file name is "seq3" a version of the corona virus genome.
         The only significant letters used are A, C, G and T not appearing
         in text between # and the end of the line.

ANGLES   Specifies a file that replaces the default setting of the
         rotational angles of each possible base pair. There are
         16 pair AA, AC,..., TG, TT and each has 3 angles. These are
         floating point numbers giving angles in degrees. The default
         filename is "angles".

SEED=Y/n Set the random number seed.

X/s      Add random values to the w and l angles of every base pair.

-d/s     Turns on some debugging output.


Controls: ( Not yet implemented )

Scc Select base pair cc for later modification
    cc is one of AA, AC,..., TG or TT

T   Select axis T of the currently selected base pair
L   Select axis L of the currently selected base pair
W   Select axis W of the currently selected base pair

n   A floating point number giving the rotational angle in degrees for the
    currently selected base pair and axis. Example angles: 36 -4.35 +19.0

I   Increment the currently selected angle by 0.1 
D   Decrement the currently selected angle by 0.1 

<NL> Newline. Recompute the genome shape if any angles have changed since
     the last newline.

P  Display the current rotation angles for each base pair

PF <filename> Output these angles to a file 

G <filename>  Load the set of angles from a specified file in the format
              generated by the PF command

RN n  Set the rate of rotation about axis N
RW n  Set the rate of rotation about axis N
RH n  Set the rate of rotation about axis N

R     Stop/start rotating the genome image

Right/left arrow Increase/decrease rotation rate about direction N.
Up/Down arrow    Increase/decrease rotation rate about direction W.
>  <             Increase/decrease rotation rate about direction H.


Vc n  Set the N, W or H coordinate of the view point.
      n is a floating point number and c is N, W, H.

F     Move the eye further from the viewpoint
N     Move the eye nearer to the viewpoint

Q  Quit

The world axes are:

      N      Direction north
      W      Direction west
      H      Direction up

The axes used  when constructing the genome model are

      t      The forward direction alone the genome
             (the direction of thrust if the path of the genome is
             thought of as an aircraft flying along the genome)
      w      Left direction (The direction of the left wing of
             the aircraft)
      l      The direction orthogonal to t and w (the direction of lift
             of the aircraft)

Initially every base pair has a rotation of 36.0 degrees about axis t,
and no rotation about the other two axes. This generates a straight
helix in direction N. The initial viewpoint is (N,W,H)=(0,0,0).

The genome model is made up of a sequence of segments between bases..
Each segment causes the (t,w,l) axes as if the aircraft was flying at
constant speed with setting of the ailerons, elevator and rudder
depending on the previous and current base letters of the segment. The
(t,w,l) axes of the first segment of the genome are aligned with
(N,W,H). Note that both (N,W,H) and (t,w,l) are right handed axes.

The first segment is in the direction of the N axis and it starts at
(N,W,H)=(0,0,0). The initial position of the viewpoint is
(0,0,0). Unit distance in the model corresponds to a real world
distance of 1 nm.


Properties of B-type DNA

It is a right handed double helix.
The distance between bases is  0.34 nm.
The rate of rotation is 10 bases per 360 degrees ie 36 degrees per base.
The diameter of the genome   1.9 nm.


The creation of the model.

The model can be thought of as the path taken by an aircraft
travelling at constant speed. The axes as t (the direction of thrust),
w (the direction of the left wing) and (l the direction of lift). The
bending of the path is assumed to bend at the juctions between each
pair of adjacent bases. For each base pair the bend is specified by
angles of ratation about each of the axes t, w and l.

The genome is represented by a sequence of triangles in 3D pointing
along the sequence and its orientation depends on the orientation of
the previous triangle and the angles of rotation for the the two bases
involved. Each triangle is given a colour determined by which of the
four bases it represents. The genome is displayed using the OpenGL
library. The user can control its rate of rotation interactively and
can also adjust the angles of rotation for any base pair. If an
interesting genome shape is found its 48 base pair angles can be
written to file.

*/

GET "libhdr"
GET "gl.h"
GET "gl.b"          // Insert the library source code
.
GET "libhdr"
GET "gl.h"

GLOBAL {
  stdin:ug
  stdout

  FLT pi
  FLT f0 // To hold 0.0
  FLT f1 // To hold 1.0
  FLT incrementfactor  // A value of the form 2^n represented exactly
                       // as a floating point number.
  
  fromfilename
  anglesfilename

  tracing
  debug

  spacev; spacep; spacet  // Used by newvec

  blklist  // List of blocks of work space
  blkp
  blkt
 
  done
  stepping
  
  glprog
  Vshader
  Fshader

  // Attribute variable locations
  VertexLoc
  ColorNoLoc

  // Uniform variable locations
  MatrixLoc
  ColorLmatrixLoc
  ColorRmatrixLoc

  FLT rtdot; FLT rwdot; FLT rldot // Anti-clockwise rotation rates
                                  // about the t, w and l axes
 
  FLT eyex; FLT eyey; FLT eyez    // Coordinates of a point on the
                                  // line of sight from to eye to
                                  // the origin (0.0, 0.0, 0.0).

  FLT eyexdot                  // Eye velocity
  FLT eyeydot
  FLT eyezdot

  FLT eyertdot                  // Eye rates of rotation
  FLT eyerwdot
  FLT eyerldot

  FLT etx; FLT ety; FLT etz    // Eye orientation matrix
  FLT ewx; FLT ewy; FLT ewz
  FLT elx; FLT ely; FLT elz
  
  FLT eyedistance              // The distance between the eye
                               // and the origin.
  FLT Fincrement
  FLT Nincrement
  FLT Eincrement
  FLT Hincrement
  prevch
  
  FLT cent; FLT cenw; FLT cenl 

  VertexBuffer// =0 or the name of the buffer that holds all the vertex
              // data we ever need.
  IndexBuffer // =0 or the name of the buffer that holds all the index
              // data we ever need.

  projectionMatrix // is the matrix used by the vertex shader
                   // to transform the vertex coordinates to
                   // screen coordinates.
  workMatrix       // is used when constructing the projection matrix.

  colorLmatrix     // 4x4 matrix of left colours
  colorRmatrix     // 4x4 matrix of right colours
  
  // Globals used to construct the model

  // The orientation of the current base in the genome
  FLT gtn; FLT gtw; FLT gth        // Direction cosines of direction t, col 1
  FLT gwn; FLT gww; FLT gwh        // Direction cosines of direction w, col 2
  FLT gln; FLT glw; FLT glh        // Direction cosines of direction l, col 3

  // Note that [ a0,a1,a3, a3,a4 a5, a6,a7,a8 ] represents the matrix
  //
  //    { a0  a3  a6 )
  //    { a1  a4  a7 )
  //    { a2  a5  a8 )


  anglesv     // Points to 16 angle triples
  matrixv     // Vector of the 3x3 rotation matrices for the 16 base pairs
  
  genomev     // This holds the genome base numbers in
              // genomev!1 to genomev!upb where upb is genomev!0.
              // eg -> [ 4, 1,2,3,4 ] for sequence A C G T


  // Variables are used by mkGenomeModel to represent the genome model.

  vertexv      // This will hold the vertex data in vertexv!1 to vertexv!upn
               // where upb is vertexv!0

  indexv       // This will hold the indices in indexv!1 to indexv!upb
               // where upb is indexv!0

  displayv     // This will hold the vertex data in vertexv!1 to vertexv!upn
               // where upb is vertexv!0

  indexp       // The index number of the next vertex to be created. The vertex with
               // index number 0 will start at position 1 in vvec.
  
  // The global functions
  newvec

  puskval
  concatstr
  letno2ch
  allocatevectors
  setangles
  mat3set
  mat3mulmmm
  mat3mulmvv
  prmat3
  mat4set
  prmat4
  prpairs
  prpair
  setrotationmatrix
  drawSegment
  drawTriangle
  rdGenome
  mkGenomeModel
  rdangles
  standardiseV3
  adjustortho
  axis2ch
  checkfor
  doincrement
  drawmodel
  error
  get16
  get32
  initImageWindow
  deg2rad
  rad2deg

  XY2str
  prv
  push16
  push32
  pushval
  put16
  put32
  rotate
  setcurraxis
  setcurrpair
  setinitstate
  tok2str
  wrcurrangle
  
  Compileshader
  processevents
  
  // The user can adjust DNA segment angles interactively.
  
  FLT incrementdegrees
  FLT incrementradians
  
  incrementstate    // =-1 or 0 to 3 after first letter of a pair
  incrementing      // =TRUE after command AA,AC,...,TG,TT

  currpair          // = 0 to 15 for AA to TT
  curraxis          // 0=t 1=w 2=l
  modelchanged      // =TRUE if the model needs to be rebuilt and
                    // sent to GL

  setrandomangles
  
  FLT genomeradius  // Typically = 0.95 nms (=1.90/2)
  FLT genomesep     // Typically = 0.34 nms

}

MANIFEST {
  blkupb = 1000000
  
  col_white = -1
  col_black = -2

  // Pair subscripts
  AA=0; AC; AG; AT; CA; CC; CG; CT; GA; GC; GG; GT; TA; TC; TG; TT
}

LET newvec(n) = VALOF
{ LET p = blkp
  blkp := p+n+1
  
  IF blkp>=blkt DO
  { LET v = getvec(blkupb) // Get some more space
//writef("newvec: allocation block %n upb %n*n", v, blkupb)
    UNLESS v & n<blkupb DO
    { LET out = output()
      selectoutput(stdout)
      writef("*nSystem error: newvec failure*n")
      selectoutput(out)
      abort(999)
    }
    
    v!0 := blklist
    blklist := v
    blkt := v+blkupb
    p    := v+1
    blkp := p+n+1
  }
//writef("newvec: allocated p=%n n=%i4 blklist=%n*n",
//         p, n, blklist)
  RESULTIS p
}

LET pushval(sxv, val) BE
{ // sxv is a self expanding vector
  LET upb = sxv!0      // =0 or the upb of v
  LET v   = sxv!1      // =0 or a getveced vector
  // If v is not zero, v!0 will be the subscript of its latest element in v.
  // If the vector is full, pushval will allocate another
  // larger and copy the existing elements into it before
  // pushing x.

  LET p = v -> v!0, 0 // Position of the previous element, if any.
//writef("pushval: %n*n", val)

  // The upb of v grows as needed.

  // Initially upb, v, and p are all zero

  IF p>=upb DO
  { // We must allocate a new larger vector
    LET newupb = 3*upb/2 + 10
    LET newv = getvec(newupb)
//writef("pushval: sxv=%n allocating new vector at %i6 oldupb %n newupb %n*n",
//        sxv, newv, upb, newupb)
//abort(2222)
    UNLESS newv DO
    { writef("More memory needed for pushval*n")
      abort(999)
      RETURN
    }
    sxv!0 := newupb
    sxv!1 := newv
    
    // Copy the existing elements
    FOR i = 0 TO upb DO newv!i := v!i
    // Pad with zeroes
    FOR i = upb+1 TO newupb DO newv!i := 0
    // Free the old vector if it existed.
    IF v DO freevec(v)

    IF FALSE & debug DO
    {  writef("pushval: replacing v=%i6 upb=%i6 with newv=%i7 upb=%i6 p=%n*n",
               v, upb, newv, newupb, p)
      abort(6666)
    }
    v := newv
  }
  p := p+1
  v!0, v!p := p, val
}

LET concatstr(s1, s2, s3) = VALOF
{ LET len1, len2 = s1%0, s2%0
  LET pos = 0
  FOR i = 1 TO len1 DO { pos := pos+1; s3%pos := s1%i }
  FOR i = 1 TO len2 DO { pos := pos+1; s3%pos := s2%i }
  s3%0 := pos
  RESULTIS s3
}

AND letno2ch(letno) = VALOF SWITCHON letno INTO
{ DEFAULT: RESULTIS '?'
  CASE 0:  RESULTIS 'A'
  CASE 1:  RESULTIS 'C'
  CASE 2:  RESULTIS 'G'
  CASE 3:  RESULTIS 'T'
}

AND letno2colno(letno) = VALOF SWITCHON letno INTO
{ DEFAULT: RESULTIS 0.0
  CASE 0:  RESULTIS 0.0 // Colour number for base A
  CASE 1:  RESULTIS 1.0 // Colour number for base C
  CASE 2:  RESULTIS 2.0 // Colour number for base G
  CASE 3:  RESULTIS 3.0 // Colour number for base T
}

AND mat3set(m, a0, a1, a2, a3, a4, a5, a6, a7, a8) BE
{ // Set the elements of matrix m.
  m!0, m!3, m!6 := a0, a3, a6
  m!1, m!4, m!7 := a1, a4, a7
  m!2, m!5, m!8 := a2, a5, a8
}

AND mat3mulmmm(m, p, q) BE
{ // m -> [ m0,m1,m2,  m3,m4,m5,  m6,m7,m8 ]
  // p -> [ p0,p1,p2,  p3,p4,p5,  p6,p7,p8 ]
  // r -> [ r0,r1,r2,  r3,r4,r5,  r6,r7,r8 ]
  //
  // This multiplies matrices m and p together placing the result in q
  // m, p and q need not be distinct.
  //
  //  { m0  m3  m6)     {p0, p3, p6 )      ( q0  q3  q6 )
  //  { m1  m4  m7)  x  {p1, p4, p7 )  =>  ( q1  q4  q7 )
  //  { m2  m5  m8)     {p2, p5, p8 )      ( q2  q5  q8 )
  
  LET FLT m0, FLT m1, FLT m2 = m!0, m!1, m!2
  LET FLT m3, FLT m4, FLT m5 = m!3, m!4, m!5
  LET FLT m6, FLT m7, FLT m8 = m!6, m!7, m!8
  
  LET FLT p0, FLT p1, FLT p2 = p!0, p!1, p!2
  LET FLT p3, FLT p4, FLT p5 = p!3, p!4, p!5
  LET FLT p6, FLT p7, FLT p8 = p!6, p!7, p!8

  mat3set(q, m0*p0 + m3*p1 + m6*p2,  // q0
              m1*p0 + m4*p1 + m7*p2,  // q1
	      m2*p0 + m5*p1 + m8*p2,  // q2
	      
              m0*p3 + m3*p4 + m6*p5,  // q3
	      m1*p3 + m4*p4 + m7*p5,  // q4
	      m2*p3 + m5*p4 + m8*p5,  // q5
	      
              m0*p6 + m3*p7 + m6*p8,  // q6
	      m1*p6 + m4*p7 + m7*p8,  // q7
	      m2*p6 + m5*p7 + m8*p8   // q8
	   )
}

AND mat3mulmvv(m, p, q) BE
{ // m -> [ m0,m1,m2,  m3,m4,m5,  m6,m7,m8 ]
  // p -> [ p0,p1,p2 ]
  // q -> [ q0,q1,q2 ]
  //
  // This multiply matrix m by vector p placing the result in vector q
  // m, p and q need not be distinct.
  //
  //  ( m0  m3  m6 )     ( p0 )      ( q0  )
  //  ( m1  m4  m7 )  x  ( p1 )  =>  ( q1  )
  //  ( m2  m5  m8 )     ( p2 )      ( q2  )
  
  LET FLT m0, FLT m1, FLT m2 = m!0, m!1, m!2
  LET FLT m3, FLT m4, FLT m5 = m!3, m!4, m!5
  LET FLT m6, FLT m7, FLT m8 = m!6, m!7, m!8
  
  LET FLT p0, FLT p1, FLT p2 = p!0, p!1, p!2

  q!0 := m0*p0 + m3*p1 + m6*p2
  q!1 := m1*p0 + m4*p1 + m7*p2
  q!2 := m2*p0 + m7*p1 + m8*p2
}

AND setangles(XY, FLT an, FLT aw, FLT ah) BE
{ // Set the angles for the pair XY and also compute the angles
  // for pair YX. The angles are given in radians.
  LET X, Y = XY>>2 & 3, XY & 3
  LET YX = Y<<2 | X
  
  LET XYav = anglesv!XY
  LET YXav = anglesv!YX
  LET XYm  = matrixv!XY
  LET YXm  = matrixv!YX
  
  LET FLT bn, FLT bw, FLT bh = f0, f0, f0
  LET FLT c,  FLT s = ?, ?
  
  LET FLT mtn, FLT mtw, FLT mth = f0, f0, f0
  LET FLT mwn, FLT mww, FLT mwh = f0, f0, f0
  LET FLT mln, FLT mlw, FLT mlh = f0, f0, f0
  
  LET m  = @mtn  // A temp rotation matrix
  LET tm = VEC 8 // Another temp matrix
//writef("setangles: %s %7.1f %7.1f %7.1f*n",
//        XY2str(XY), rad2deg(an), rad2deg(aw), rad2deg(ah))
  XYav!0 := an
  XYav!1 := aw
  XYav!2 := ah

  setrotationmatrix(XY)
  RETURN
  IF X=Y RETURN
  
  // X and Y are different bases so we must calculate the rotation
  // angles for YX.


  // The rotation matrix for the transition from base X to Y depends
  // on it three axis rotation angles. The order in which these
  // axis rotations are done in the order N, W and H. If the t, w and
  // l axes of the segment for X are aligned in directions  N, W and
  // H the the columns of the rotation matrix will give the
  // directions of these axes for segment Y. Clearly the rotation
  // caused by the transition from segment Y to X will be the inverse
  // of the rotation from X to Y. The inverse matrix is easy to calculate
  // since for rotation matrices the inverse is just the transpose.
  // However what we need are the rotation angles bn, bw and bh of the
  // inverse transformation.  Assuming columns of the XY rotation matrix
  // are:
  //         ( mtn )   ( mwn )   ( mln )
  //         ( mtw )   ( mww )   ( mlw )
  //         ( mth )   ( mwh )   ( mlh )
  // These are unit vector representing the t, w and l directions of
  // the Y segment. We can easily find the angle of rotation about the
  // N axis to cause the l axis to lie in the NOH plane. This angle is
  // arctan(mlw/mlh). To avoid overflow this angle is taken to be zero
  // if mlw is very small. If c and s are the cosine and sine of this
  // angle the rotation aboyt N is represented by
  //      ( 1   0   0 )
  //      ( 0   c   s )
  //      ( 0  -s   c )
  // We can now multiply the orientation matrix by this one to obtain
  // one in which axis t is in the plane NOH. The angles of rotation
  // for the axes W and L are easily obtained by the same method.
  //abort(9186)
  FOR i = 0 TO 8 DO m!i := XYm!i  // Copy the XY rotation matrix
//writef("Rotation matrix for %s is:*n", XY2str(XY))
  //prmat3(m)

  // Find the rotation angle about axis N
  //writef("Rotate about N*n")
  UNLESS -1e-8 < mlw < 1e-8 DO
  { bn := sys(Sys_flt, fl_atan2, mlw, mlh)
    c, s := sys(Sys_flt, fl_cos, bn), sys(Sys_flt, fl_sin, bn)
    //writef("Angle about N is %6.1f => c=%7.4f  s= %7.4f*n", rad2deg(bn), c, s)
    // Rotate clockwise about axis N by bn radians
    //writef("mlw=%8.5f mlh=%8.5f => bn=%7.1f*n", mlw, mlh, rad2deg(bn))
    mat3set(tm,        f1,    f0,    f0,     f0,  c,  s,    f0, -s,  c)
    mat3mulmmm(tm,m,m)
  }
        
  //prmat3(m)

  // Find the rotation angle about axis W
  //writef("Rotate about W*n")
  UNLESS -1e-8 < mln < 1e-8 DO
  { bw := sys(Sys_flt, fl_atan2, -mln, mlh)
    c, s := sys(Sys_flt, fl_cos, bw), sys(Sys_flt, fl_sin, bw)
    //writef("Angle about W is %6.1f => c=%7.4f  s= %7.4f*n", rad2deg(bw), c, s)
    // Rotate clockwise about axis W by bw radians
    //writef("mln=%8.5f mlh=%8.5f => bn=%7.1f*n", mln, mlh, rad2deg(bw))
    mat3set(tm,       c,    f0, -s,     f0,    f1,    f0,  s,    f0,  c)
    mat3mulmmm(tm,m,m)
  }

  //prmat3(m)

  // Find the rotation angle about axis H
  //writef("Rotate about H*n")
  UNLESS -1e-8 < mtw < 1e-8 DO
  { bh := sys(Sys_flt, fl_atan2, -mtw, mtn)
    c, s := sys(Sys_flt, fl_cos, bh), sys(Sys_flt, fl_sin, bh)    
    //writef("Angle about H is %6.1f => c=%7.4f  s= %7.4f*n", rad2deg(bh), c, s)
    // Rotate clockwise about axis N by bn radians
    //writef("mtw=%8.5f mtn=%8.5f => bn=%7.1f*n", mtw, mtn, rad2deg(bh))
    mat3set(tm,       c,  s,    f0,  -s,  c,    f0,    f0,    f0,    f1)
    mat3mulmmm(tm,m,m)
  }

  //prmat3(m)

  YXav!0 := bn
  YXav!1 := bw
  YXav!2 := bh

  setrotationmatrix(YX)
  // Check that the product of the XY and YX rotation matrices
  // is approximatly the identy matrix.
  //writef("*nCheck product of XYm and YXm*n")
  //mat3mulmmm(XYm, YXm, m)
  //prmat3(m)
  //abort(2233)
}

AND setrotationmatrix(XY) BE
{ LET av = anglesv!XY  // The rotation angles for pair XY
  LET mv = matrixv!XY  // The 3x3 rotation matrix for pair XY
  
  LET FLT at, FLT aw, FLT al = av!0, av!1, av!2 // The angles for pair XY
  
  LET m = VEC 8   // A temp 3x3 matrix
   
 // Rotate clockwise about axis N by t radians
  mat3set(mv,        f1,    f0,    f0,     f0,  cost,  sint,    f0, -sint,  cost)

  // Rotate clockwise about axis W by w radians
  mat3set(m,       cosw,    f0, -sinw,     f0,    f1,    f0,  sinw,    f0,  cosw)
  mat3mulmmm(m,mv,mv)
 
  // Rotate clockwise about axis H by l radians
  mat3set(m,       cosl,  sinl,    f0,  -sinl,  cosl,    f0,    f0,    f0,    f1)
  mat3mulmmm(m,mv,mv)

  //writef("setrotationmatrix: at=%9.5f aw=%9.5f al=%9.5f =>*n", at, aw, al)
  //prmat3(mv)
  modelchanged := TRUE
}


AND drawTriangle(vsxv, isxv, a, b, c, colno) BE
{ // vsxv is the self expanding vector for the vertex data
  // isxv is the self expanding vector for the indices
//abort(1001)
//writef("%i5: %12.5f  %12.5f  %12.5f %3.1f*n",
//        indexp, a!0, a!1, a!2, colno)
  pushval(vsxv, a!0)
  pushval(vsxv, a!1)
  pushval(vsxv, a!2)
  pushval(vsxv, colno)
  pushval(isxv, indexp)
  indexp := indexp+1
  
//writef("%i5: %12.5f  %12.5f  %12.5f %3.1f*n",
//        indexp, b!0, b!1, b!2, colno)
  pushval(vsxv, b!0)
  pushval(vsxv, b!1)
  pushval(vsxv, b!2)
  pushval(vsxv, colno)
  pushval(isxv, indexp)
  indexp := indexp+1
  
//writef("%i5: %12.5f  %12.5f  %12.5f %3.1f*n",
//        indexp, c!0, c!1, c!2, colno)
  pushval(vsxv, c!0)
  pushval(vsxv, c!1)
  pushval(vsxv, c!2)
  pushval(vsxv, colno)
  pushval(isxv, indexp)
  indexp := indexp+1
}


AND rdGenome(filename) = VALOF
{ // Read the genome into a self expanding vector and, if successful,
  // returns its getvec'd vector of letter numbers 1, 2, 3 and 4. The
  // zeroth element is the subscript of the last base of the genome.
  // The result is zero on failure.

  LET vupb, v = 0, 0 // The self expanding vector for the letter numbers.
  LET sxv = @vupb
  
  LET instream = findinput(filename)

  UNLESS instream DO
  { writef("Trouble with file %s*n", filename)
    RESULTIS FALSE
  }

  selectinput(instream)

  { LET ch = capitalch(rdch())
    SWITCHON ch INTO
    { DEFAULT:  LOOP

      CASE endstreamch:
        pushval(sxv, -1)
	endstream(instream)
	//abort(1004)
	RESULTIS v // v is zero or a getvec's vector holding the genome.

      CASE '#': // Ignore a comment line
        UNTIL ch='*n' | ch=endstreamch DO ch := rdch()
	LOOP

      CASE 'A': pushval(sxv, 0); LOOP
      CASE 'C': pushval(sxv, 1); LOOP
      CASE 'G': pushval(sxv, 2); LOOP
      CASE 'T': pushval(sxv, 3); LOOP
    }
  } REPEAT
}


AND mkGenomeModel() = VALOF
{ // This function contructs the genome model from the sequence in genomev
  // It sets the following
  // vertexv  to hold the vertex data, vertexv!0 holds its upb
  // iindexv  to hold the indices, indexv!0 holds its upb
  // displayv to hold the display items, displayv!0 holds its upb

  LET vupb, vvec = 0, 0 // Self expanding vectors for vertices
  LET iupb, ivec = 0, 0 //   for index values
  LET dupb, dvec = 0, 0 //   for display values

  LET vsxv = @vupb
  LET isxv = @iupb
  LET dsxv = @dupb

  // vsxv is a self expanding vector for the vertices [x y z colourno].
  // isxv  is a self expanding array for the triangles.
  // dsxv  is a self expanding array for the display elements

  // Coords of the mid point of the leading edge of the first segment
  LET FLT  x, FLT  y, FLT  z = f0, f0, f0
  // Coords of the mid point of the leading edge of the next segment, if any
  LET FLT nx, FLT ny, FLT nz =  ?,  ?,  ?

  // Declare the matrix representing the orientation of the next
  // segment of the genome.
  LET FLT stx, FLT sty, FLT stz = f1, f0, f0  // Direction t
  LET FLT swx, FLT swy, FLT swz = f0, f1, f0  // Direction w
  LET FLT slx, FLT sly, FLT slz = f0, f0, f1  // Direction l
  LET m = @stx  // The 3x3 orientation matrix
  
  indexp := 0

  // Draw the initial rectangle to mark the start of the genome.
  
  { LET FLT lx0, FLT ly0, FLT lz0 = -2.0,  genomeradius, f0
    LET FLT rx0, FLT ry0, FLT rz0 = -2.0, -genomeradius, f0
    LET FLT lx1, FLT ly1, FLT lz1 =   f0,  genomeradius, f0
    LET FLT rx1, FLT ry1, FLT rz1 =   f0, -genomeradius, f0

    // Draw the rectangle as two triangles
    drawTriangle(vsxv, isxv, @rx0, @lx1, @lx0, col_white)  // Left  triangle
    drawTriangle(vsxv, isxv, @rx0, @lx1, @rx1, col_white)  // Right triangle
    //drawTriangle(vsxv, isxv, @stx, @swx, @slx, 3.0)
  }
  
  // Now draw coloured triangles for the bases of the dna sequence
  
  FOR i = 1 TO genomev!0 DO
  { // x, y, z are the coords of the mid point of the leading edge
    // of this triangle and its orientation is given by matrix m.
    LET FLT lx = x + genomeradius*swx  // Left vertex of the triangle 
    LET FLT ly = y + genomeradius*swy 
    LET FLT lz = z + genomeradius*swz 

    LET FLT rx = x - genomeradius*swx  // Right vertex of the triangle 
    LET FLT ry = y - genomeradius*swy 
    LET FLT rz = z - genomeradius*swz 

    LET X = genomev!i // The base number of the current base
    LET Y = ?         // Next base number, if any
    LET XY = ?
    
    nx := x + genomesep*stx     // The mid point of the leading 
    ny := y + genomesep*sty     // edge of the next triangle
    nz := z + genomesep*stz

    drawTriangle(vsxv, isxv, @lx, @rx, @nx, letno2colno(X))

    x, y, z := nx, ny, nz

    IF i = genomev!0 BREAK // Just drawn the last triangle

    // Update the the orientation matrix for the next triangle

    Y  := genomev!(i+1)
    XY := X<<2 | Y

    // Multiply the orientation matrix by the XY rotation matrix.
    mat3mulmmm(matrixv!XY, m, m)
  }

  // Draw the final white rectangle

  { LET FLT lx0 =  x + genomeradius*swx  // (swx,swy,swz) is the unit
    LET FLT ly0 =  y + genomeradius*swy  // vector in direction w.
    LET FLT lz0 =  z + genomeradius*swz

    LET FLT rx0 =  x - genomeradius*swx
    LET FLT ry0 =  y - genomeradius*swy
    LET FLT rz0 =  z - genomeradius*swz

    LET FLT x1 = x + 2.0*stx     // The mid point of the trailing
    LET FLT y1 = y + 2.0*sty     // edge of the final rectangle.
    LET FLT z1 = z + 2.0*stz

    LET FLT lx1 = x1 + genomeradius*swx
    LET FLT ly1 = y1 + genomeradius*swy
    LET FLT lz1 = z1 + genomeradius*swz

    LET FLT rx1 = x1 - genomeradius*swx
    LET FLT ry1 = y1 - genomeradius*swy
    LET FLT rz1 = z1 - genomeradius*swz

    // Draw the rectangle as two triangles
    drawTriangle(vsxv, isxv, @rx0, @lx1, @lx0, col_white)  // Left  triangle
    drawTriangle(vsxv, isxv, @rx0, @lx1, @rx1, col_white)  // Right triangle
  }
  

  pushval(@dupb, 4)
  pushval(@dupb, ivec!0) // The number of indices
  pushval(@dupb, 0)      // The first indicex is at position zero
                         // in the index data sent to GL
  
  vertexv  := vvec
  indexv   := ivec
  displayv := dvec

  IF debug DO
  { // Output the vertex and index data
    // as a debugging aid
    writef("*nVertex data*n")
    FOR i = 1 TO vertexv!0 DO
    { IF i MOD 4 = 1 DO writef("*n%i3: ", i-1)
      writef(" %8.3f", vertexv!i)
    }
    writef("*n*nIndex data*n")
    FOR i = 1 TO indexv!0 DO
    { IF i MOD 10 = 1 DO writef("*n%i6: ", i-1)
      writef(" %i5", indexv!i)
    }
    writef("*n*nDisplay data items*n")
    FOR i = 1 TO displayv!0 - 1 BY 3 DO
      writef(" %i5  %i5  %i5*n",
             displayv!i, displayv!(i+1), displayv!(i+2)) // mode n offset
    newline()
  }

  RESULTIS TRUE
}

AND rdangles(filename) BE
{ LET FLT an, FLT aw, FLT ah = 0, 0, 0
  LET XYav = ?
  LET instream = findinput(filename)
  UNLESS instream RETURN
writef("rdangles: reading file %s*n", filename)
  selectinput(instream)

  { LET ch = rdch()
    LET XY = -1
    LET FLT t, FLT w, FLT l = 1.0, 1.0, 1.0

    SWITCHON ch INTO
    { DEFAULT: LOOP

      CASE '#':
         ch := rdch() REPEATUNTIL ch='*n' |
	                          ch=endstreamch
         UNLESS ch=endstreamch LOOP
	 // Fall through
      CASE endstreamch:
        endstream(instream)
	RETURN

      CASE 'A': CASE 'a': XY :=  0; ENDCASE
      CASE 'C': CASE 'c': XY :=  4; ENDCASE
      CASE 'G': CASE 'g': XY :=  8; ENDCASE
      CASE 'T': CASE 't': XY := 12; ENDCASE
    }
    ch := rdch()
    SWITCHON ch INTO
    { DEFAULT: LOOP

      CASE endstreamch:
        endstream(instream)
	RETURN


      CASE 'A': CASE 'a':               ENDCASE
      CASE 'C': CASE 'c': XY := XY + 1; ENDCASE
      CASE 'G': CASE 'g': XY := XY + 2; ENDCASE
      CASE 'T': CASE 't': XY := XY + 3; ENDCASE
    }
    XYav := anglesv!XY
    
//abort(7166)
    
    an := deg2rad(readflt()) // Angle about N
    aw := deg2rad(readflt()) // Angle about W
    ah := deg2rad(readflt()) // Angle about H
    writef("rdangles: %s %7.1f, %7.1f, %7.1f*n",
            XY2str(XY), rad2deg(an), rad2deg(aw), rad2deg(ah))
    setangles(XY, an, aw, ah)
    //abort(9166)
  } REPEAT
}

LET prpairs() BE FOR XY = 0 TO 15 DO prpair(XY)

AND prpair(XY) BE
{ LET av = anglesv!XY
  LET mv = matrixv!XY
  writef("%s angles %7.1f %7.1f %7.1f*n",
          XY2str(XY), rad2deg(av!0), rad2deg(av!1), rad2deg(av!2))
  writef("%s matrix", XY2str(XY))
  FOR i = 0 TO 8 DO writef(" %5.3f", mv!i)
  newline()
}

LET setinitstate() BE
{ blklist, blkp, blkt := 0, 0, 0

  pi := 3.141592653
  f0 := 0.0
  f1 := 1.0

  anglesv := newvec(15) // anglesv!XY-> [an,aw,ah ]  Rotation angles
  matrixv := newvec(15) // matrixv!XY-> [m0,m1,m2,  m3,m4,m5, m6,m7,m8 ]
writef("anglesv=%n matrixv=%n*n", anglesv, matrixv)
	
  projectionMatrix := newvec(15)
  workMatrix       := newvec(15)
  colorLmatrix     := newvec(15)
  colorRmatrix     := newvec(15)
writef("projectionMatrix=%n workMatrix=%n*n",
        projectionMatrix, workMatrix)
writef("colorLmatrix=%n colorRmatrix=%n*n",
        colorLmatrix, colorRmatrix)

  // Setup the initial values of anglesv and matrixv.
  FOR XY = 0 TO 15 DO
  { LET av = newvec(2) // -> [3 rotation angles]
    LET mv = newvec(8) // -> [9 rotation matrix elements]
    anglesv!XY, matrixv!XY := av, mv
    av!0, av!1, av!2 := f0, f0, f0
    mat3set(mv, f1,f0,f0, f0,f1,f0, f0,f0,f1)
    //writef("setinitstate: %s av=%n mv=%n*n", XY2str(XY), av, mv)
  }

  //abort(1000)
  // Set the default segment angles

  setangles(AA, deg2rad(0.0),   f0,   f0)
  setangles(AC, deg2rad(0.0),   f0,   f0)
  setangles(AG, deg2rad(0.0),   f0,   f0)
  setangles(AT, deg2rad(0.0),   f0,   f0)

  setangles(CC, deg2rad(0.0),   f0,   f0)
  setangles(CC, deg2rad(0.0),   f0,   f0)
  setangles(CG, deg2rad(0.0),   f0,   f0)
  setangles(CT, deg2rad(0.0),   f0,   f0)

  setangles(GC, deg2rad(0.0),   f0,   f0)
  setangles(GC, deg2rad(0.0),   f0,   f0)
  setangles(GG, deg2rad(0.0),   f0,   f0)
  setangles(GT, deg2rad(0.0),   f0,   f0)

  setangles(TC, deg2rad(0.0),   f0,   f0)
  setangles(TC, deg2rad(0.0),   f0,   f0)
  setangles(TG, deg2rad(0.0),   f0,   f0)
  setangles(TT, deg2rad(0.0),   f0,   f0)
//prpairs()
//abort(4478)
  genomeradius := 0.95 // In nms (=1.90/2)
  genomesep    := 0.34 // In nms
  genomeradius := 0.25
  genomesep    := 1.000

  currpair := AA             // Current base pair is AA
  curraxis := 0              // Current axis is t
  
  incrementdegrees := 0.0625 // The amount in degrees that angles are changed
                             // by the # and ~ commands.
  incrementradians := deg2rad(incrementdegrees)
  
  incrementstate  := -1      // =0 to 3 for the first letter of a segment pair
                             // has just been pressed.

  VertexBuffer := 0          // No vertex buffer yet.
  IndexBuffer  := 0          // No index buffer yet.
}

LET start() = VALOF
{ LET argv = VEC 50

  stdin := input()
  stdout := output()

  UNLESS rdargs("FROM,ANGLES,SEED=X/n, X/s,-d/s", argv, 50) DO
  { writef("Bad arguments for dnaplot*n")
    RESULTIS 0
  }

  fromfilename   := "testdata"
  fromfilename   := 0 //"seq3"
  anglesfilename := "angles"
  
  IF argv!0 DO                       // FROM     Read the genome sequence from file
    fromfilename := argv!0
  IF argv!1 DO                       // ANGLES   Read file of angle settings
    anglesfilename := argv!1
  setseed(12345)
  IF argv!2 DO setseed(!(argv!2))    // SEED=Y/n Seed for choosing randome angles
  debug   := argv!4                  // -d/s     Output debugging info

  setinitstate()
   
  IF anglesfilename DO rdangles(anglesfilename)
  //prpairs()
  IF argv!3 DO setrandomangles()     // X/s
  
  TEST fromfilename
  THEN { genomev := rdGenome(fromfilename)
         UNLESS genomev DO
         { writef("Failed to read the genome file %s*n", fromfilename)
           GOTO fin
         }
       }
  ELSE { LET v = TABLE 17,
         // A test sequence containing every base pair just once.
         // A A C C G G T T A T C T G A G C A
            0,0,1,1,2,2,3,3,0,3,1,3,2,0,2,1,0
	 genomev := getvec(v!0) // So that freevec can be called later
         UNLESS genomev DO
         { writef("Failed to read the genome file %s*n", fromfilename)
           GOTO fin
         }
	 FOR i = 0 TO v!0 DO genomev!i := v!i
       }	 

  // genomev!1 to genomev!upb now hold the base numbers where
  // upb is in genomev!0

  IF debug DO
  { LET len = genomev!0               // Subscript of the last element in v
    IF len>500 DO len := 500
    
    writef("Start of genome: %s*n", fromfilename=0 -> "test data", fromfilename)
    FOR i = 1 TO len DO
    { LET letno = genomev!i
      IF letno<0 BREAK
      wrch(letno2ch(letno))
      IF i MOD 50 = 0 DO newline()
    }
    newline()
//    abort(6664)
  }

  initImageWindow()

  // Initialise the state

  done     := FALSE
  stepping := TRUE

  // Set the initial orientation matrix of the eye that maps from
  // model (t,w,l) coords to screen coords (x,y,z).
  // 
  // t maps to -z       ie into the screen
  // w maps to -x       ie to the left
  // l maps to  y       ie up

  // Set the initial direction cosines to place t, w and l in
  // directions -z, -x and y, ie viewing along the genome from behind.

  etx, ctw, cth :=   0.0,  0.0, -1.0  // ?? Change to ctx ctw ctl etc
  cwn, cww, cwh :=  -1.0,  0.0,  0.0
  cln, clw, clh :=   0.0,  1.0,  0.0

  // This is the orientation of the genome when displayed and is only changed
  // by rotation when stepping is TRUE.
  
  // x is to the right on the screen
  // y is up on the screen
  // z is away from the screen
  
  //ctx, cwx, clx :=   0.0, -1.0,  0.0
  //cty, cwy, cly :=   0.0,  0.0,  1.0
  //ctz, cwz, clz :=  -1.0,  0.0,  0.0

  eyertdot, eyerwdot, eyerldot := 0.0000, 0.0005, 0.0000 // No initial rotation

  eyex, eyey, eyez :=  0.0,  0.0,  1.0 // Direction of view

  Fincrement := 0  // These are used by the commands F/f N/n E/e H/h
  Nincrement := 0
  Eincrement := 0
  Hincrement := 0
  prevch     := 0

  UNTIL done DO
  {
    //writef("Calling processevents()*n")
    processevents()
    //writef("Returned from processevents()*n")
    //abort(8761, modelchanged)

    // If the angles have changed we must reconstruct the model and send
    // it to GL
    WHILE modelchanged DO
    { //writef("Building the model with new angles*n")
      modelchanged := FALSE

      // Then create the DNA model of 3D triangles in user space
      mkGenomeModel()

      // vertexv holds the vertex data, vertev!0 holds its upb
      // indexv holds the indices, indexv!0 holds its upb
      // displayv holds the display triples, displayv!0 holds its upb

      // Copy the vertices [x, y, z, colno] to GL space

      // Setup the model using OpenGL objects in the graphics server's
      // memory.
      // But delete previous vertex and index buffers if they exist.

      IF VertexBuffer DO
      { //writef("Deleting buffer %n*n", VertexBuffer)
        sys(Sys_gl, gl_DeleteBuffer, VertexBuffer)
      }
      IF IndexBuffer  DO
      { //writef("Deleting buffer %n*n", IndexBuffer)
        sys(Sys_gl, gl_DeleteBuffer, IndexBuffer)
      }
      
      //writef("start: Number of vertex elements %n*n", vertexv!0)
      VertexBuffer := sys(Sys_gl, gl_GenVertexBuffer, vertexv!0, vertexv+1)
      //writef("genVertexBuffer vertexv!0=%n vertexv+1=%n*n", vertexv!0, vertexv+1)
      // VertexBuffer is the name (a positive integer) of the vertex buffer.
      //writef("dnaplot: VertexBuffer=%n*n", VertexBuffer)
      //abort(1001)

      // Tell GL the positions in vvec of the xyz fields,
      // ie the first 3 words of each 4 word item in vvec
      //writef("start: Calling gl_EnableVertexAttribArray  VertexLoc==%n*n",
      //        VertexLoc)
      // VertexLoc is the location of the variable g_vVertex used
      // by the vertex shader.
      sys(Sys_gl, gl_EnableVertexAttribArray, VertexLoc);
      sys(Sys_gl, gl_VertexData,
                  VertexLoc,     // Attribute number for xyz data
                  3,             // 3 floats for xyz
                  4,             // 4 floats per vertex item in vertexData
                  0)             // Offset in 32 bit words of the xyz data

      //writef("start: VertexData xyz data copied to graphics object %n*n",
      //       VertexBuffer)

      // Tell GL the positions in the vertax data of the colno field,
      // ie one word word at position 3  of each 4 word item in vertexv
      sys(Sys_gl, gl_EnableVertexAttribArray, ColorNoLoc);
      sys(Sys_gl, gl_VertexData,
                  ColorNoLoc,    // Attribute number of the colour nnumber field
                  1,             // 1 floats for colour data
                  4,             // 4 floats per vertex item
                  3)             // Offset in words of the colour nnumber field

      //writef("start: Colour numbers copied to graphics object %n*n",
      //        VertexBuffer)

      freevec(vertexv) // Free vertexv since all its elements have
                       // been sent to the graphics server.
      vertexv := 0

      // Copy the indices to GL space

      //writef("start: Number of indices is %n*n", indexv!0)
      //writef("start: Calling GenIndexBuffer*n")
      IndexBuffer  := sys(Sys_gl, gl_GenIndexBuffer, indexv+1, indexv!0)

      //writef("start: IndexData copied to graphics memory object %n*n", IndexBuffer)

      freevec(indexv) // Free ivec since all its elements have
                      // been sent to the graphics server.
      indexv := 0
    }

    // Only rotate the object if stepping
    IF stepping DO
    { // If not stepping adjust the orientation of the model.
      //writef("Calling rotate(%9.3f, %9.3f, %9.3f)*n", eyertdot, eyerwdot, eyerldot)
      //rotate(eyertdot, eyerwdot, eyerldot)
    }

    // Change the model coords to make them them relative to the
    // eye position.

    mat4set( projectionMatrix,
              1.0,    0.0,   0.0,   0.0,  // column 1
              0.0,    1.0,   0.0,   0.0,  // column 2
              0.0,    0.0,   1.0,   0.0,  // column 3
              -eyex, -eyey, -eyez,  1.0)  // column 4


    // Now rotate the model coordinates about the eye position
    // to cause the model directions t, w and l to lie in screen
    // directions -z, -x and y.
    mat4set( workMatrix,
                     etx,   ety,  etz,  0.0,  // column 1
                     ewx,   ewy,  ewz,  0.0,  // column 2
                     elx,   ely,  elz,  0.0,  // column 3
                     0.0,   0.0,  0.0,  1.0)  // column 4

    // Note that the matrix
    
    //       { etx  ewx  elx }
    //       { ety  ewy  ely }
    //       { etz  ewz  elz }
    
    // will transform unit vectors in directions t, w and l as follows
    
    //  ( 1 )    ( etx )      ( 0 )    ( ewx )      ( 0 )    ( elx )
    //  ( 0 ) => ( ety )      ( 1 ) => ( ewy )      ( 0 ) => ( ely )
    //  ( 0 )    ( etz )      ( 0 )    ( ewz )      ( 1 )    ( elz )
    
    // So any point in the model will be transformed to its corresponding
    // position in screen coordinates.

    sys(Sys_gl, gl_M4mulM4, workMatrix, projectionMatrix, projectionMatrix)


    // Rotate the model and eye until the eye is on the z axis

    { LET FLT ex, FLT ey, FLT ez = eyex, eyey, eyez
      LET FLT oq = glRadius2(ex, ez) 
      LET FLT op = glRadius3(ex, ey, ez)
      LET FLT cos_theta = ez / oq 
      LET FLT sin_theta = ex / oq 
      LET FLT cos_phi   = oq / op 
      LET FLT sin_phi   = ey / op 

      // Rotate anti-clockwise about Y axis by angle theta
      mat4set( workMatrix,
                 cos_theta, 0.0, sin_theta, 0.0,   // column 1
                       0.0, 1.0,       0.0, 0.0,   // column 2
                -sin_theta, 0.0, cos_theta, 0.0,   // column 3
                       0.0, 0.0,       0.0, 1.0    // column 4
               )

      sys(Sys_gl, gl_M4mulM4, workMatrix, projectionMatrix, projectionMatrix)

      // Rotate clockwise about X axis by angle phi
      mat4set( workMatrix,
                1.0,     0.0,       0.0, 0.0,    // column 1
                0.0, cos_phi,  -sin_phi, 0.0,    // column 2
                0.0, sin_phi,   cos_phi, 0.0,    // column 3
                0.0,     0.0,       0.0, 1.0)    // column 4

      sys(Sys_gl, gl_M4mulM4, workMatrix, projectionMatrix, projectionMatrix)

      // Change the origin to the eye position on the z axis by
      // moving the model eyedistance in the negative z direction.
      mat4set( workMatrix,
                1.0, 0.0,           0.0, 0.0, // column 1
                0.0, 1.0,           0.0, 0.0, // column 2
                0.0, 0.0,           1.0, 0.0, // column 3
                0.0, 0.0,  -eyedistance, 1.0  // column 4
              )

      sys(Sys_gl, gl_M4mulM4, workMatrix, projectionMatrix, projectionMatrix)
    }

    { // Define the truncated pyramid for the view projection
      // using the frustrum transformation.
      LET FLT n, FLT f = 0.1, 5000.0
      LET FLT fan, FLT fsn = f+n, f-n
      LET FLT n2 = 2.0*n
      LET FLT l,   FLT r   = -0.5, 0.5
      LET FLT ral, FLT rsl =  r+l, r-l
      LET FLT b,   FLT t   = -0.5, 0.5 
      LET FLT tab, FLT tsb =  t+b, t-b

      LET FLT aspect = FLOAT screenxsize / FLOAT screenysize
      LET FLT fv = 2.0 / 0.5  // Half field of view at unit distance
      mat4set( workMatrix,
                fv/aspect,  0.0,             0.0,  0.0, // column 1
                      0.0,   fv,             0.0,  0.0, // column 2
                      0.0,  0.0,     (f+n)/(n-f), -1.0, // column 3
                      0.0,  0.0, (2.0*f*n)/(n-f),  0.0  // column 4
               )

      // This perspective matrix could be set more conveniently using
      // glSetPerspective library function defined in g/gl.b
      //glSetPerspective(workMatrix,
      //                     aspect, // Aspect ratio
      //                        0.5, // Field of view at unit distance
      //                        0.1, // Distance to near limit
      //                     5000.0) // Distance to far limit

      sys(Sys_gl, gl_M4mulM4, workMatrix, projectionMatrix, projectionMatrix)
    }


    // Send the resulting matrix to the uniform variable "matrix" for
    // use by the vertex shader.
    sys(Sys_gl, gl_UniformMatrix4fv, MatrixLoc, glprog, projectionMatrix)
    //writef("projectionMatrix is now defined and sent to GL*n")


    // Send the uniform variable "colorLmatrix" to GL for
    // use by the vertex shader.
    // Set the triangle colours
    mat4set( colorLmatrix,
                1.0,   0.0,  0.0, 1.0,  // column 1  Left  colour for A
                0.0,   1.0,  0.0, 1.0,  // column 2  Left  colour for C
                0.0,   0.0,  1.0, 1.0,  // column 3  Left  colour for G
                1.0,   0.0,  1.0, 1.0)  // column 4  Left  colour for T

    sys(Sys_gl, gl_UniformMatrix4fv, ColorLmatrixLoc, glprog, colorLmatrix)
    //writef("colorLmatrixatrix is now defined and sent to GL*n")

    // Send the uniform variable "ColorRmatrix" to GL for
    // use by the vertex shader.  -- No longer used.
    mat4set( colorRmatrix,
                0.5,   0.0,  0.0, 1.0,  // column 1  Right colour for A
                0.0,   0.5,  0.0, 1.0,  // column 2  Right colour for C
                0.0,   0.0,  0.5, 1.0,  // column 3  Right colour for G
                0.5,   0.0,  0.5, 1.0)  // column 4  Right colour for T

    sys(Sys_gl, gl_UniformMatrix4fv, ColorRmatrixLoc, glprog, colorRmatrix)
    //writef("colorRmatrixatrix is now defined and sent to GL*n")



    // Draw a new image
    sys(Sys_gl, gl_ClearBuffer) // Clear colour and depth buffers
    //writef("Calling clearcolour*n")
    sys(Sys_gl, gl_ClearColour,
        FLOAT 160/255, FLOAT 160/255, FLOAT 250/255, 1.0)

  sys(Sys_gl, gl_Enable, GL_DEPTH_TEST) // This call is neccessary
  sys(Sys_gl, gl_DepthFunc, GL_LESS)    // This is the default

  // Positive Z is into the screen, so a pixel is written with
  // depth < buffer depth takes precedence. ????
  // Remember that the depth test is performed after all other
  // transformations have been done.


//writef("Calling drawmodel()*n")
    drawmodel()

//writef("Calling swapbuffers*n")
    sys(Sys_gl, gl_SwapBuffers)
    delay(0_020) // Delay for 1/50 sec
//abort(1003)
  }

  sys(Sys_gl, gl_DisableVertexAttribArray, VertexLoc)
  sys(Sys_gl, gl_DisableVertexAttribArray, ColorNoLoc)

fin:
  IF genomev DO freevec(genomev)

  // Free the self expanding vector space.
  IF vertexv  DO freevec(vertexv)
  IF indexv   DO freevec(indexv)
  IF displayv DO freevec(displayv)
  
  // Free all the block in the blklist.
  WHILE blklist DO
  { LET blk = blklist
    blklist := !blk
//writef("start: freeing blklist blk %n*n", blk)
    freevec(blk)
  }

//  freevec(dvec) // Free the display items vector.
  delay(0_050)
  sys(Sys_gl, gl_Quit)

  RESULTIS 0
}//xxx




AND initImageWindow() = VALOF
{ LET str  = VEC 20 // For the window title

  UNLESS glInit() DO
  { writef("*nOpenGL not available*n")
    GOTO fin
  }

  //writef("start: calling glMkScreen*n")
  // Create an OpenGL window
  screenxsize := glMkScreen(concatstr("Genome from file: ",
                                       fromfilename -> fromfilename,
				                       "Test Genome",
				                       
				       str),
                            800, 680)
  screenysize := result2
  UNLESS screenxsize DO
  { writef("*nUnable to create an OpenGL window*n")
    RESULTIS 0
  }
  //writef("Screen Size is %n x %n*n", screenxsize, screenysize)
//abort(1000)
  glprog := sys(Sys_gl, gl_MkProg)
  //writef("dnaplot: glprog=%n*n", glprog);

  IF glprog<0 DO
  { writef("*nUnable to create a GL program*n")
    RESULTIS 0
  }

  // Read and Compile the vertex shader
  //writef("glstst: calling Compileshader(%n,TRUE,*"dnaplotVshader.sdr*")*n",glprog)
  Vshader := Compileshader(glprog, TRUE, "dnaplotVshader.sdr")
  //writef("dnaplot: Vshader=%n*n", Vshader)
//abort(8344)

// Read and Compile the fragment shader
  //writef("dnaplot: calling Compileshader(%n,FALSE,dnaplotFshader.sdr)*n",glprog)
  Fshader := Compileshader(glprog, FALSE, "dnaplotFshader.sdr")
  //writef("dnaplot: Fshader=%n*n", Fshader)

  // Link the program
  //writef("dnaplot: calling glLinkProg(%n)*n", glprog)
  UNLESS sys(Sys_gl, gl_LinkProgram, glprog) DO
  { writef("*ndnaplot: Unable to link a GL program*n")
    RESULTIS 0
  }
//abort(8345)

  //writef("dnaplot: calling glUseProgram(%n)*n", glprog)
  sys(Sys_gl, gl_UseProgram, glprog)

  // Get attribute locations after linking 
  VertexLoc  := sys(Sys_gl, gl_GetAttribLocation, glprog, "g_vVertex")
  ColorNoLoc := sys(Sys_gl, gl_GetAttribLocation, glprog, "g_vColorNo")

  //writef("dnaplot: VertexLoc=%n*n",   VertexLoc)
  //writef("dnaplot: ColorNoLoc=%n*n",  ColorNoLoc)

  // Get uniform locations after linking
  MatrixLoc := sys(Sys_gl, gl_GetUniformLocation, glprog, "matrix")
  //writef("dnaplot: MatrixLoc=%n*n",  MatrixLoc)

  ColorLmatrixLoc := sys(Sys_gl, gl_GetUniformLocation, glprog, "colorLmatrix")
  //writef("dnaplot: ColorLmatrixLoc=%n*n",  ColorLmatrixLoc)

  ColorRmatrixLoc := sys(Sys_gl, gl_GetUniformLocation, glprog, "colorRmatrix")
  //writef("dnaplot: ColorRmatrixLoc=%n*n",  ColorRmatrixLoc)

fin:
  RESULTIS 0
}

AND Compileshader(prog, isVshader, filename) = VALOF
{ // Create and compile a shader whose source code is
  // in a given file.
  // isVshader=TRUE  if compiling a vertex shader
  // isVshader=FALSE if compiling a fragment shader
  LET oldin = input()
  LET oldout = output()
  LET buf = 0
  LET shader = 0
  LET ramstream = findinoutput("RAM:")
  LET instream = findinput(filename)
  UNLESS ramstream & instream DO
  { writef("Compileshader: Trouble with i/o streams*n")
    RESULTIS -1
  }

  //Copy shader program to RAM:
  selectoutput(ramstream)
  selectinput(instream)

  { LET ch = rdch()
    IF ch=endstreamch BREAK
    wrch(ch)
  } REPEAT

  wrch(0) // Place the terminating byte

  selectoutput(oldout)
  endstream(instream)
  selectinput(oldin)

  buf := ramstream!scb_buf

  shader := sys(Sys_gl,
                (isVshader -> gl_CompileVshader, gl_CompileFshader),
                prog,
                buf)

  endstream(ramstream)
  RESULTIS shader
}

AND drawmodel() BE
{ // Draw the primitives using vertex and index data held in
  // graphics objects as specified by the display items in dvec.
  FOR p = 1 TO displayv!0 BY 3 DO
  { LET d = @displayv!p
    LET mode   = d!0  // Points, Lines, Linestrip, etc.
    LET size   = d!1  // Number of index elements.
    LET offset = d!2  // Offset in the index vector.

//writef("drawmodel: p=%n mode=%n, size=%n offset=%n*n", p, mode, size, offset)

    sys(Sys_gl, gl_DrawElements,
                mode,     // 1=points, 2=lines, 3=linestrip, etc
                size,     // Number of index elements to use.
                4*offset) // The start position (bytes) in the index vector.
	
  }
}

AND processevents() BE WHILE getevent() SWITCHON eventtype INTO
{ 
  DEFAULT:
    //writef("processevents: Unknown event type = %n*n", eventtype)
    LOOP

  CASE sdle_quit:             // 12
    writef("QUIT*n")
    sys(Sys_gl, gl_Quit)
    LOOP

  CASE sdle_keydown:
  { LET ch = eventa2
    //writef("*n%i3 ", eventa2)
    //IF 32 <= eventa2 < 127 DO writef("'%c'*n", eventa2)
    SWITCHON ch INTO
    { DEFAULT:  LOOP

      CASE '*n':
      CASE '*c':
                //writef("*nENTER pressed*n")
		incrementstate := -1
                LOOP

      CASE '?':
        newline()
        newline()
        writef("?      Output this help info*n")
        writef("Q      Quit*n")
        writef("AA AC AG AT CA CC etc to TG TT  Select base pair*n")
	writef("X      Select next axis*n")
	writef("F/f    Move eye forward/back in the direction of view*n")
	writef("N/n    Move eye forward/back in direction N*n")
	writef("E/e    Move eye forward/back in direction E*n")
	writef("H/h    Move eye forward/back in direction H*n")
	writef("P      Output the current orientation and other values*n")
	writef("W name Write the angles data to file*n")
	writef("S      Start/Stop the rotation*n")
	writef("#/~    Increase/Decrease the currently selected angle*n")
	writef("|/\    Double/Halve the increment angle*n")
	writef("Y      Randomly increment to the w and l angles of all base pairs*n")
	writef("Z      Set w and l angles to zero for all 16 base pairs*n")
	writef(">/<    Increase/decrease the eye rotation about l*n")
        writef("Up/Down arrow      Increase/decrease the eye rotation about w*n")
        writef("Left/Right arrow   Increase/decrease the eye rotation about t*n")
	newline()
		
	LOOP
		
      CASE 'Q': done := TRUE
                LOOP

      CASE 'A': wrch(ch); deplete(cos)
                TEST incrementstate<0
                THEN incrementstate := 0 // First letter of a pair
		ELSE setcurrpair(0)
		LOOP

      CASE 'C': wrch(ch); deplete(cos)
                TEST incrementstate<0
                THEN incrementstate := 1 // First letter of a pair
		ELSE setcurrpair(1)
		LOOP

      CASE 'G': wrch(ch); deplete(cos)
                TEST incrementstate<0
                THEN incrementstate := 2 // First letter of a pair
		ELSE setcurrpair(2)
		LOOP

      CASE 'T': wrch(ch); deplete(cos)
                TEST incrementstate<0
                THEN incrementstate := 3 // First letter of a pair
		ELSE setcurrpair(3)
		LOOP

      CASE 'Y': setcurraxis((curraxis+1) MOD 3) // Select axit 0, 1 or 2
                LOOP

      CASE 'F': // Move eye foward in direction t
        TEST prevch=ch THEN Fincrement := 2.0 * Fincrement
	               ELSE Fincrement := 1.0
        prevch := ch
        eyex := eyex + etx*Fincrement
        eyey := eyey + ety*Fincrement
        eyez := eyez + etz*Fincrement
        LOOP

      CASE 'f': // Move eye back in direction t
        TEST prevch=ch THEN Fincrement := 2.0 * Fincrement
	               ELSE Fincrement := 1.0
        prevch := ch
        eyex := eyex - etx*Fincrement
        eyey := eyey - ety*Fincrement
        eyez := eyez - etz*Fincrement
        LOOP

      CASE 'N': // Move eye foward in direction N
        TEST prevch=ch THEN Nincrement := 2.0 * Nincrement
	               ELSE Nincrement := 1.0
        prevch := ch
        eyex := eyex + Nincrement
        LOOP

      CASE 'n': // Move eye foward in direction N
        TEST prevch=ch THEN Nincrement := 2.0 * Nincrement
	               ELSE Nincrement := 1.0
        prevch := ch
        eyex := eyex - Nincrement
        LOOP

      CASE 'E': // Move eye foward in direction E
        TEST prevch=ch THEN Nincrement := 2.0 * Nincrement
	               ELSE Nincrement := 1.0
        prevch := ch
        eyey := eyey - Nincrement
        LOOP

      CASE 'e': // Move eye backward in direction E
        TEST prevch=ch THEN Nincrement := 2.0 * Nincrement
	               ELSE Nincrement := 1.0
        prevch := ch
        eyey := eyey + Nincrement
        LOOP

      CASE 'H': // Move eye foward in direction H
        TEST prevch=ch THEN Nincrement := 2.0 * Hincrement
	               ELSE Nincrement := 1.0
        prevch := ch
        eyez := eyez - Hincrement
        LOOP

      CASE 'P': // Print direction cosines and other data
                newline()
                writef("eyet   %9.6f %9.6f %9.6f eyertdot=%9.6f*n",
                               etn,  etw,  eth,  eyertdot)
                writef("eyew   %9.6f %9.6f %9.6f eyerwdot=%9.6f*n",
                               ewn,  eww,  ewh,  eyerwdot)
                writef("eyel   %9.6f %9.6f %9.6f eyerldot=%9.6f*n",
                               eln,  elw,  elh,  eyerldot)
                newline()
                writef("eyepos %9.3f %9.3f %9.3f*n",
                               eyex, eyey, eyez)
                LOOP

      CASE 'W': wranglesdata()
                LOOP

      CASE 'S': stepping := ~stepping
                LOOP

      CASE '0': eyex, eyez :=  0.000,  1.000; LOOP
      CASE '1': eyex, eyez :=  0.707,  0.707; LOOP
      CASE '2': eyex, eyez :=  1.000, -0.000; LOOP
      CASE '3': eyex, eyez :=  0.707, -0.707; LOOP
      CASE '4': eyex, eyez :=  0.000, -1.000; LOOP
      CASE '5': eyex, eyez := -0.707, -0.707; LOOP
      CASE '6': eyex, eyez := -1.000,  0.000; LOOP
      CASE '7': eyex, eyez := -0.707,  0.707; LOOP

      CASE '=':
      CASE '+': eyedistance := eyedistance * 1.1; LOOP

      CASE '_':
      CASE '-': IF eyedistance >= 1.0 DO
                   eyedistance := eyedistance / 1.1
                LOOP

      CASE '>':CASE '.':    rldot := rldot + 0.0005
                            IF rldot> 0.0060 DO rldot :=  0.0060
                            LOOP
      CASE '<':CASE ',':    rldot := rldot - 0.0005
                            IF rldot<-0.0060 DO rldot := -0.0060
                            LOOP
      CASE '#':             doincrement(incrementradians)
                            LOOP
      CASE '~':             doincrement(-incrementradians)
                            LOOP
      CASE '\':             incrementdegrees := 0.5*incrementdegrees
                            incrementradians := deg2rad(incrementdegrees)
                            writef("*nincrementvdegrees=%9.6f*n", incrementdegrees)
                            LOOP
      CASE '|':             incrementdegrees := 2.0*incrementdegrees
                            incrementradians := deg2rad(incrementdegrees)
                            writef("*nincrementvdegrees=%9.6f*n", incrementdegrees)
                            LOOP
			    
      CASE sdle_arrowdown:
        rwdot := rwdot + 0.0005
        IF rwdot> 0.0060 DO rwdot :=  0.0060
        LOOP
	
      CASE sdle_arrowup:
        rwdot := rwdot - 0.0005
        IF rwdot<-0.0060 DO rwdot := -0.0060
        LOOP

      CASE sdle_arrowleft:
        rtdot := rtdot + 0.0005
        IF rtdot> 0.0060 DO rtdot :=  0.0060
        LOOP

      CASE sdle_arrowright:
        rtdot := rtdot - 0.0005
        IF rtdot<-0.0060 DO rtdot := -0.0060
        LOOP

      CASE 'X':
        // Add random values in the range -incrementdegrees to +incrementdegrees
	// to the w and l angles of every base pair.
        setrandomangles()
        LOOP
    }
  }
}

AND wranglesdata() BE
{ LET filename = VEC 100
  LET len = 0
  LET outstream = 0
  LET ch = 0
  writef("*nWrite angles data to file: ")
  deplete(cos)
  selectinput(findinput("**"))
  ch := rdch() REPEATWHILE ch='*s'
  WHILE 'A' <= ch <= 'Z' |
        'a' <= ch <= 'z' |
	'0' <= ch <= '9' |
	ch='/' | ch='.' | ch='**'DO
	{ len := len+1
	  filename%0, filename%len := len, ch
	  ch := rdch()
	}

  writef("*nAngle data written to file: %s*n", filename)

  UNLESS ch='*n' DO
  { writef("*nThe file name can only contain letters, digits and*n")
    writef("'/', '.' or '**'*n")
    RETURN
  }
  outstream := findoutput(filename)
  UNLESS outstream DO
  { writef("*nUnable to write to file: %s*n", filename)
    RETURN
  }
  selectoutput(outstream)
  FOR XY = AA TO TT DO
  { LET av = anglesv!XY
    writef("%s %9.6f %9.6f %9.6f*n",
            XY2str(XY), rad2deg(av!0), rad2deg(av!1), rad2deg(av!2))
  }
  endstream(outstream)
  RETURN
}

AND setrandomangles() BE FOR XY = AA TO TT DO
{ LET av = anglesv!XY
  FOR a = 1 TO 2 DO
  { LET FLT r = FLOAT(randno(2000001) - 1000000)
    LET FLT x = incrementdegrees * r / 1000000.0
    av!a := av!a + x
  }
  setrotationmatrix(XY)

}

AND setcurrpair(letno) BE
{ currpair := incrementstate*4 + letno
  incrementstate := -1
  // Use the same axis as before.
  wrcurrangle()
}

AND setcurraxis(axis) BE
{ curraxis := axis
  newline()
  wrcurrangle()
}

AND doincrement(FLT val) BE
{ LET av = anglesv!currpair
  av!curraxis := av!curraxis + val
  writef("doincrement: val=%7.3f currpair=%s axis=%c  new angle=%7.3f*n",
          val, XY2str(currpair), axis2ch(curraxis), av!curraxis)
  modelchanged := TRUE
  wrcurrangle()
  setrotationmatrix(currpair)
  prpair(currpair)
}

AND wrcurrangle() BE
{ LET v = anglesv!currpair
  writef("*c%s%c=%6.3f  incrementdegrees=%6.3f   *n",
           XY2str(currpair), axis2ch(curraxis), v!curraxis, incrementdegrees)
  deplete(cos)
}

AND deg2rad(FLT angle) = angle*pi/180

AND rad2deg(FLT angle) = angle*180.0/pi

AND XY2str(XY) = VALOF SWITCHON XY INTO
{ DEFAULT:  RESULTIS "??"

  CASE  0:  RESULTIS "AA"
  CASE  1:  RESULTIS "AC"
  CASE  2:  RESULTIS "AG"
  CASE  3:  RESULTIS "AT"

  CASE  4:  RESULTIS "CA"
  CASE  5:  RESULTIS "CC"
  CASE  6:  RESULTIS "CG"
  CASE  7:  RESULTIS "CT"

  CASE  8:  RESULTIS "GA"
  CASE  9:  RESULTIS "GC"
  CASE 10:  RESULTIS "GG"
  CASE 11:  RESULTIS "GT"

  CASE 12:  RESULTIS "TA"
  CASE 13:  RESULTIS "TC"
  CASE 14:  RESULTIS "TG"
  CASE 15:  RESULTIS "TT"
}

AND axis2ch(axis) = "twl"%(axis+1)

AND rotate(FLT t, FLT w, FLT l) BE
{ // Rotate the genome orientation by angles given in radians.
  // t, w and l cause rotation about axes t, w and l. Positive
  // values cause clockwise rotations.

  LET FLT tx =    etx -  l*cwn + w*cln
  LET FLT wx =  l*etx +    cwn - t*cln
  LET FLT lx = -w*etx +  t*cwn +   cln

  LET FLT ty =    ctw -  l*cww + w*clw
  LET FLT wy =  l*ctw +    cww - t*clw
  LET FLT ly = -w*ctw +  t*cww +   clw

  LET FLT tz =    cth -  l*cwh + w*clh
  LET FLT wz =  l*cth +    cwh - t*clh
  LET FLT lz = -w*cth +  t*cwh +   clh

  etx, ctw, cth := tx, ty, tz
  cwn, cww, cwh := wx, wy, wz
  cln, clw, clh := lx, ly, lz

  standardiseM3(@etx)
}

AND standardiseV3(v) BE
{ // Make v a vector of unit length
  LET FLT r = glRadius3(v!0, v!1, v!2)
  v!0 := v!0 / r
  v!1 := v!1 / r
  v!2 := v!2 / r
}

AND standardiseM3(m) BE
{ // Make the three vectors in matrix m of unit length
  // and mutually orthogonal.
  LET v1, v2, v3 = @m!0, @m!3, @m!6
  standardiseV3(v1)
  crossprodV3(v1, v2, v3)
  standardiseV3(v2)
  crossprodV3(v3, v1, v2)
}

AND crossprodV3(v1, v2, v3) BE
{ LET FLT x1, FLT y1, FLT z1 = v1!0, v1!1, v1!2
  LET FLT x2, FLT y2, FLT z2 = v2!0, v2!1, v2!2
  v2!0 := y1*z2 - y2*z1
  v2!1 := z1*x2 - z2*x1
  v2!2 := x1*y2 - x2*y1
}

AND adjustortho(a, b) BE
{ // Attempt to keep the unit vector b orthogonal to a
  LET FLT a0, FLT a1, FLT a2 = a!0, a!1, a!2
  LET FLT b0, FLT b1, FLT b2 = b!0, b!1, b!2
  LET FLT corr = a0*b0 + a1*b1 + a2*b2
  b!0 := b0 - a0 * corr
  b!1 := b1 - a1 * corr
  b!2 := b2 - a2 * corr
}

AND mat4set(m,  a0,  a1,  a2,  a3,
                a4,  a5,  a6,  a7,
                a8,  a9, a10, a11,
               a12, a13, a14, a15) BE
{ // Set the elements of matrix m.
  m!0, m!4,  m!8, m!12 := a0, a4,  a8, a12
  m!1, m!5,  m!9, m!13 := a1, a5,  a9, a13
  m!2, m!6, m!10, m!14 := a2, a6, a10, a14
  m!3, m!7, m!11, m!15 := a3, a7, a11, a15
}


AND prmat3(m) BE
{ // m is a 3x3 matrix as a sequence of columns.
  writef("( %9.5f %9.5f %9.5f )*n", m!0, m!3, m!6)
  writef("( %9.5f %9.5f %9.5f )*n", m!1, m!4, m!7)
  writef("( %9.5f %9.5f %9.5f )*n", m!2, m!5, m!8)
}

AND prmat4(m) BE
{ // m is a 4x4 matrix as a sequence of columns.
  writef("( %8.3f %8.3f %8.3f %8.3f )*n", m!0, m!4, m! 8, m!12)
  writef("( %8.3f %8.3f %8.3f %8.3f )*n", m!1, m!5, m! 9, m!13)
  writef("( %8.3f %8.3f %8.3f %8.3f )*n", m!2, m!6, m!10, m!14)
  writef("( %8.3f %8.3f %8.3f %8.3f )*n", m!3, m!7, m!11, m!15)
}

AND prv(v) BE
{ // v is a vector of four elements.
  writef(" %8.3f %8.3f %8.3f %8.3f*n", v!0, v!1, v!2, v!3)
}


