/*

This will eventually be a BCPL implementation of an emulator of the
Z80 microprocessor. Its human interaction will be based on the
interactive debugger of the BCPL Cintcode system. This emulator has
features that the debugging of code generated by the BCPL to Z80
compiler z80bcpl.b. Only implements the Z80 instructions needed for
this purpose. Additional instructions can easily be added if needed.

Implemented by Martin Richards (c) 5 Oct 2020

Usage: FROM,TO/K,GLBSIZE,STKSIZE,d/S
FROM    The file name of an optional Intel Hex file to load.
TO      Output to file rather than the screen
GLBSIZE The size in words of the root global vector, default size 768.
STKSIZE The size in words of the root stack, default size 4000.
-d      Turn on debugging output


All memory locations are first initialised with the value #xDEAD.
The global vector elements are initialised with #xC000+gn where gn is
the global number, except global zero is initialise with GLBSIZE.
All stack elements are initialised with the value #xABCD.
If FROM is specified the register PC is set to the value of global 1.

IY is normally set to the byte address of global 64.
IX is normally set to the byte address of word 64 relative to the
   current BCPL P pointer.

The BCPL function stack frame is as follows:

          ----------------------------------------------------
         |   P' |  L | A1 | A2 | A3 | more arguments and locals  
          ----------------------------------------------------
             ^
             |
             P

P     The BCPL stack frame pointer of the current function
P'    The BCPL stack frame pointer of the enclosing function or zero.
L     The return byte address.
A1, A2,... The function arguments and locals
      A1 is passed in HL'  ie in the alternate register set
      A2 is passed in DE'
      A3 is passed in BC'

HL is the function entry address

The first instruction of a function is:
RST #x38 SETLINK0   if there are no arguments
RST #x38 SETLINK1   if there awas one argument
RST #x38 SETLINK2   if there were two arguments
RST #x38 SETLINK3   if there were three or more arguments

The function result is returned in HL

History

09/09/2022
Making extensive changes, still in process.

05/10/2020
Started the initial implementation.

*/

GET "libhdr"

/*
// All the possible instructions compiled by bcplcgz80
// Clearly z80emu.b must implement all of them.

        CASE i_inc    :  RESULTIS  "    INC   %S"
        CASE i_dec    :  RESULTIS  "    DEC   %S"
        CASE i_and    :  RESULTIS  "    AND   %S"
        CASE i_or     :  RESULTIS  "    OR    %S"
        CASE i_xor    :  RESULTIS  "    XOR   %S"
        CASE i_sbchl  :  RESULTIS  "    SBC   HL,%S"
        CASE i_addhl  :  RESULTIS  "    ADD   HL,%S"
        CASE i_ldrr   :  RESULTIS  "    LD    %S,%S"
        CASE i_cpl    :  RESULTIS  "    CPL"
        CASE i_ldrn   :  RESULTIS  "    LD    %S,#X%X2"
        CASE i_ldrnn  :  RESULTIS  "    LD    %S,#X%X4"
        CASE i_ldrll  :  RESULTIS  "    LD    %S,L%N"
        CASE i_push   :  RESULTIS  "    PUSH  %S"
        CASE i_pop    :  RESULTIS  "    POP   %S"
        CASE i_ret    :  RESULTIS  "    RET"
        CASE i_jpihl  :  RESULTIS  "    JP    (HL)"
        CASE i_exx    :  RESULTIS  "    EXX"
        CASE i_jr     :  RESULTIS  "    JR    L%N-$"
        CASE i_jrz    :  RESULTIS  "    JR    Z,L%N-$"
        CASE i_jrnz   :  RESULTIS  "    JR    NZ,L%N-$"
        CASE i_jrc    :  RESULTIS  "    JR    C,L%N-$"
        CASE i_jrnc   :  RESULTIS  "    JR    NC,L%N-$"
        CASE i_jp     :  RESULTIS  "    JP    L%N"
        CASE i_jpz    :  RESULTIS  "    JP    Z,L%N"
        CASE i_jpnz   :  RESULTIS  "    JP    NZ,L%N"
        CASE i_jpc    :  RESULTIS  "    JP    C,L%N"
        CASE i_jpnc   :  RESULTIS  "    JP    NC,L%N"
        CASE i_jpp    :  RESULTIS  "    JP    P,L%N"
        CASE i_jpm    :  RESULTIS  "    JP    M,L%N"
        CASE i_ldri   :  RESULTIS  "    LD    %S,(%S%S%N)"
        CASE i_ldir   :  RESULTIS  "    LD    (%S%S%N),%S"
        CASE i_ldhll  :  RESULTIS  "    LD    HL,(L%N)"
        CASE i_ldrl   :  RESULTIS  "    LD    %S,(L%N)"
        CASE i_srl    :  RESULTIS  "    SRL   %S"
        CASE i_rr     :  RESULTIS  "    RR    %S"
        CASE i_ldlhl  :  RESULTIS  "    LD    (L%N),HL"
        CASE i_ldlr   :  RESULTIS  "    LD    (L%N),%S"
        CASE i_nop    :  RESULTIS  "    NOP"
        CASE i_cpir   :  RESULTIS  "    CPIR"

        CASE i_rst38  :  RESULTIS  "    RST   #X38      " // MR 20/08/2022

// RST arguments used by bcplcgz80.b
        CASE m_apply      :  RESULTIS  "M_APPLY"
        CASE m_fixup      :  RESULTIS  "M_FIXUP"
        CASE m_loadix     :  RESULTIS  "M_LOADIX"
        CASE m_loadiy     :  RESULTIS  "M_LOADIY"
        CASE m_storeix    :  RESULTIS  "M_STOREIX"
        CASE m_storeiy    :  RESULTIS  "M_STOREIY"  
        CASE m_setlink0   :  RESULTIS  "M_SETLINK0"
        CASE m_setlink1   :  RESULTIS  "M_SETLINK1"
        CASE m_setlink2   :  RESULTIS  "M_SETLINK2"
        CASE m_setlink3   :  RESULTIS  "M_SETLINK3"
        CASE m_return     :  RESULTIS  "M_RETURN"
        CASE m_finish     :  RESULTIS  "M_FINISH"
        CASE m_loadlvix   :  RESULTIS  "M_LOADLVIX"
        CASE m_loadlviy   :  RESULTIS  "M_LOADLVIY"
        CASE m_multiply   :  RESULTIS  "M_MULTIPLY"
        CASE m_divide     :  RESULTIS  "M_DIVIDE"
        CASE m_rem        :  RESULTIS  "M_REM"
        CASE m_lshift     :  RESULTIS  "M_LSHIFT"
        CASE m_rshift     :  RESULTIS  "M_RSHIFT"
        CASE m_eq         :  RESULTIS  "M_EQ"
        CASE m_ne         :  RESULTIS  "M_NE"
        CASE m_ls         :  RESULTIS  "M_LS"
        CASE m_gr         :  RESULTIS  "M_GR"
        CASE m_le         :  RESULTIS  "M_LE"
        CASE m_ge         :  RESULTIS  "M_GE"
        CASE m_rdivide    :  RESULTIS  "M_RDIVIDE"
        CASE m_rrem       :  RESULTIS  "M_RREM"
        CASE m_rlshift    :  RESULTIS  "M_RLSHIFT"
        CASE m_rrshift    :  RESULTIS  "M_RRSHIFT"
        CASE m_abs        :  RESULTIS  "M_ABS"
        CASE m_linsearch  :  RESULTIS  "M_LINSEARCH"

        CASE m_default    :  RESULTIS  "M_DEFAULT"
*/

MANIFEST
$( // Taken from com/bcplcgz80.b to help show all the Z80 istructions
   // that the compiler might generate.
/*
    dupl0                 =  #B000000000 
    dupl1                 =  #B100000000 

    i_inc                 =  #B00000011 // INC   %S
    i_dec                 =  #B00001011 // DEC   %S
    i_and                 =  #B10100000 // AND   %S
    i_or                  =  #B10110000 // OR    %S
    i_xor                 =  #B10101000 // XOR   %S
    i_sbchl               =  #B01000010 // SBC   HL,%S
    i_addhl               =  #B00001001 // ADD   HL,%S
    i_ldrr                =  #B01000000 // LD    %S,%S
    i_cpl                 =  #B00101111 // CPL
    i_ldrn                =  #B00000110 // LD    %S,#X%X2
    i_ldrnn               =  #B00000001 + dupl0  // LD    %S,#X%X4
    i_ldrll               =  #B00000001 + dupl1  // LD    %S,L%N
    i_push                =  #B11000101 // PUSH  %S
    i_pop                 =  #B11000001 // POP   %S
    i_ret                 =  #B11001001 // RET
    i_jpihl               =  #B11101001 // JP    (HL)
    i_exx                 =  #B11011001 // EXX
    i_jr                  =  #B00011000 + dupl0  // JR    L%N-$
    i_jrz                 =  #B00101000 // JR    Z,L%N-$
    i_jrnz                =  #B00100000 // JR    NZ,L%N-$
    i_jrc                 =  #B00111000 + dupl0  // JR    C,L%N-$
    i_jrnc                =  #B00110000 // JR    NC,L%N-$
    i_jp                  =  #B11000011 // JP    L%N
    i_jpz                 =  #B11001010 // JP    Z,L%N
    i_jpnz                =  #B11000010 // JP    NZ,L%N
    i_jpnc                =  #B11010010 // JP    NC,L%N
    i_jpc                 =  #B11011010 // JP    C,L%N"
    i_jpp                 =  #B11110010 // JP    P,L%N"
    i_jpm                 =  #B11111010 // JP    M,L%N"
    i_ldri                =  #B01000110 // LD    %S,(%S%S%N)
    i_ldir                =  #B01110000 // LD    (%S%S%N),%S
    i_ldhll               =  #B00101010 // LD    HL,(L%N)
    i_ldrl                =  #B01001011 // LD    %S,(L%N)
    i_srl                 =  #B00111000 + dupl1  // SRL   %S
    i_rr                  =  #B00011000 + dupl1  // RR    %S
    i_ldlhl               =  #B00100010 // LD    (L%N),HL
    i_ldlr                =  #B01000011 // LD    (L%N),%S
    i_cpir                =  #B10110001 // CPIR
    i_nop                 =  #B00000000 // NOP
    i_rst38               =  #B11111111 // RST   #X38
*/
    m_jpentrysize         =  3

    m_apply               =  0  * m_jpentrysize
    m_fixup               =  1  * m_jpentrysize
    m_loadix              =  2  * m_jpentrysize
    m_loadiy              =  3  * m_jpentrysize
    m_storeix             =  4  * m_jpentrysize
    m_storeiy             =  5  * m_jpentrysize
    m_setlink0            =  6  * m_jpentrysize
    m_setlink1            =  7  * m_jpentrysize
    m_setlink2            =  8  * m_jpentrysize
    m_setlink3            =  9  * m_jpentrysize        
    m_return              =  10 * m_jpentrysize         
    m_finish              =  11 * m_jpentrysize         
    m_loadlvix            =  12 * m_jpentrysize        
    m_loadlviy            =  13 * m_jpentrysize        
    m_multiply            =  14 * m_jpentrysize        
    m_divide              =  15 * m_jpentrysize        
    m_rem                 =  16 * m_jpentrysize        
    m_lshift              =  17 * m_jpentrysize        
    m_rshift              =  18 * m_jpentrysize        
    m_eq                  =  19 * m_jpentrysize        
    m_ne                  =  20 * m_jpentrysize        
    m_ls                  =  21 * m_jpentrysize        
    m_gr                  =  22 * m_jpentrysize        
    m_le                  =  23 * m_jpentrysize        
    m_ge                  =  24 * m_jpentrysize        
    m_rdivide             =  25 * m_jpentrysize        
    m_rrem                =  26 * m_jpentrysize        
    m_rlshift             =  27 * m_jpentrysize        
    m_rrshift             =  28 * m_jpentrysize        
    m_abs                 =  29 * m_jpentrysize        
    m_linsearch           =  30 * m_jpentrysize

    m_sys                 =  31 * m_jpentrysize // Added for z80emu
$)

GLOBAL {
stdin:ug
stdout
debug       // = TRUE or FALSE
watchaddr
watchval

z80memupbb  // Typically = #xFFFF
z80memupb   // Typically = #x7FFF
mem         // A vector holding the 64K bytes of the Z80 memory

glbsize
stksize
glbupb
stkupb
rootgvec
rootstack
codevec
codep

gptr
pptr

pflag         // =TRUE for printing, =FALSE for executing
doz80instr    // (a)   Print or execute one instruction
prz80instr    // (pc)  Print the instruction at pc

faultcode     // Value returned by doz80instr
indentpos     // Used by prstate()

evenparity

zerror
notyet        // (str,a,b,c) the instruction at pc is not yet implemented
              // It returns the size of the offending instruction.
r2str
cc2str

// The variables holding the Z80 registers
// These alway hold sign extended 8 bit value.
r_a;  r_b;  r_c;  r_d;  r_e;  r_f;  r_h;  r_l   // The main registers
r_a1; r_b1; r_c1; r_d1; r_e1; r_f1; r_h1; r_l1  // The alternate set
// These always hold unsigned extended 16 bit value.
r_ix; r_iy; r_sp; r_pc
// These always hold unsigned extended 8 bit value.
r_i;  r_r   // Interrupt page register and the memory refresh register

// Bits in F
r_fval; r_fval1     // Used to detemine S, Z
r_fh; r_fh1         // Holds 0 or 1
r_fpval; r_fpval1   // Holds a value whose parity is the value of the P/V bit
r_fn; r_fn1         // Holds 0 or 1
r_fc; r_fc1         // Hold 0 or 1
fromfilename
tofilename
checkaddr
cont
error
bkval

getbb
getsbb
getwb
getswb
getww
getsww

putbb
putwb
putww

signextb
signextw

getr
putr

praddr

rdval
rch

bpt        // The current breakpoint number or -1
bpt_addr   // Vector of breakpoint PC values
bpt_instr  // Vector of breakpoint first bytes (op codes)
brkstep    // =TRUE when resuming from a breakpoint
singlestep // =TRUE when single stepping (\ command)
tracing

gptr       // Currently selected G pointer = (IY-128 >> 2
pptr       // Currently selected P pointer = (IX-128) >> 2

count      // A count decremented every time a Z80 instruction is executed.
           // Fault 2 occurs when count reaches zero.

trcount    // Set by the TR command to trace the exeution of a number
           // of instructions. When not tracing its value is -1

finval     // The value to be returned by start

finp       // Finish label
finl

recp       // Recovery label
recl

retp       // The label to cause a return from sadebug.
retl       // This will cause the next Z80 instruction to
           // be executed.

style
fullstatus
val        // Current value
vars       // Vector of variables (V1 .. V9)

ch         // Current charater, capitalised
lch        // Current charater

loadhex    // (filename)  Load an Intel Hex file and set the globals defined.

z80dumpmem // Dump the Z80 registers and memory to Z80DUMP in a compacted
           // form using rn length encoding.

prflags    // (f) print the flag bits

// Execution functions that print or execute the instruction
// at a specified location depending on the setting of pflag.
// The result is the size in bytes of the instruction.

doz80instrCB   // (a)   Print or execute one instruction with op BC
doz80instrED   // (a)   Print or execute one instruction with op EB
doz80instrXD   // (a)   Print or execute one instruction with op DD or FD
doz80instrXDCB // (a)   Print or execute one instruction with op XD CB

// Fuctions to print or execute specific instructions depending
// on the setting of pflag.

// The op is in capitals, followed by w for 16 bit operations
// and operands f the form:
//    r    r_a, r_b etc
//    ir   (BC), (DE) (HL), (SP)
//    ird  (IX+d) (IY+d)
//    n    n in range 0 to 255
//    nn   nn in range 0 to 65535

/*
z80ADCrizd
z80ADCrn
z80ADCrr
z80ADCr
z80ADDrizd
z80ADDrn
z80ADDrr
z80ADD2rr
z80ADDwrr
z80ADDr
z80ANDizd
z80ANDn
z80ANDrr
z80ANDr
z80BITbizd
z80BITbr
z80BITbro
z80CALLnn
z80CCF
z80CCP
z80CPD
z80CPDR
z80CPI
z80CPIR
z80CPL
z80CPizd
z80CPn
z80CPr
z80DAA
z80DAAr
z80DECizd
z80DECr
z80DEC16r
z80DI
z80DJNZo
z80EI
z80EXX
z80EXrr
z80HALT
z80IM
z80IMk
z80INCizd
z80INCr
z80INC16r
z80IND
z80INDR2
z80INI
z80INIR
z80INrin
z80INrr
z80JPr
z80JPnn
z80JPccnn
z80JPccd
z80JPd
z80JPir
z80JRra
z80LDD
z80LDDR
z80LDI
z80LDIR
z80LDizdr
z80LDinnr
z80LDrizd
z80LDrinn
z80LDrn
z80LDrr
z80LD16rnn
z80LDwr
z80MULUWrr
z80NEG2
z80NOP
z80ORn
z80ORr
z80OTDR
z80OTIR
z80OUTD
z80OUTI
z80OUTrr
z80POP16r
z80PUSH16r
z80RCCizd
z80RRCr
z80RESbizd
z80RESbr
z80RET
z80RETI
z80RETN
z80RETr
z80RLA
z80RLCA
z80RLCizd
z80RLC2r
z80RLD
z80RLizd
z80RLr
z80RRA
z80RRCA
z80RRCizd
z80RRD
z80RRizd
z80RRr
z80RSTn
z80SBC2rn
z80SBCrr
z80SBCr
z80SCF
z80SETbizd
z80SETbr
z80SLAizd
z80SLAr
z80SRAizd
z80SRAr
z80SRLizd
z80SRLr
z80SUBn
z80SUBrn
z80SUBrr
z80SUBr
z80XORn
z80XORr
*/

// Execution functions (do not change r_pc)
// All 8 bit operations unless 16 is specified.

x_adc
x_adcw
x_add
x_addw
x_and
x_neg
x_or
x_rl
x_rlc
x_rr
x_rra
x_rrc
x_sbc
x_sbc16
x_sla
x_srl
x_sub
x_xor

x_exx


}

MANIFEST
{ // Register numbers
  n_b=0     // B
  n_c       // C
  n_d       // D
  n_e       // E
  n_h       // H
  n_l       // L
  n_f       // F
  n_a       // A


  n_af      // AF
  n_bc      // BC
  n_de      // DE
  n_hl      // HL
  n_sp      // SP
  n_pc      // PC
  n_ix      // IX
  n_iy      // IY

  n_af1     // AF'
  n_bc1     // BC'
  n_de1     // DE'
  n_hl1     // HL'

  n_ic      // (C)    Used by IN and OUT
  n_ibc     // (BC)
  n_ide     // (DE)
  n_ihl     // (HL)
  n_isp     // (SP)
  n_inn     // (nn)   used by eg LD A,(nn)

  n_nn      // nn     used by eg LD A,nn
  n_n       // n      used by eg LD A,n
  n_i       // I
  n_r       // R

// The flag bit positions
  fb_s = #b_1000_0000
  fb_z = #b_0100_0000
  fb_h = #b_000h_0000
  fb_p = #b_0000_0100
  fb_v = #b_0000_0100
  fb_n = #b_0000_0010
  fb_c = #b_0000_0001
  
  n_a1;  n_b1; n_c1; n_d1; n_e1; n_f1; n_h1; n_l1  // The alternate set

  // Condition codes
  cc_c=1     // C       These are needed by eg RET M  or RET NZ
  cc_m       // M
  cc_nc      // NC
  cc_nz      // NZ
  cc_p       // P
  cc_pe      // PE
  cc_po      // PO
  cc_z       // Z

  f_brk = #xEF     // RST #x28 The instruction used for breakpoints
  f_sys = #xF7     // RST #x30 The instruction used the BCPL sys function

  tagbits    = #b_1111_1100_0000_0000
  nontagbits = #b_0000_0011_1111_1111
  globword   = #b_1010_0000_0000_0000
}


LET start() = VALOF
{ LET argv = VEC 50
  finp  := level()   // Label to be used by longjump to cause
  finl  := fin       // z80emu to terminate.

  writef("*nZ80 Emulator for BCPL code compiled by z80bcpl*n")
  writef("Type ? for help*n")

  stdin   := input()
  stdout  := output()
  debug   := FALSE
  watchaddr := 0
  watchval := 0
  
  z80memupbb := #xFFFF                       // The byte upb of the Z80 memory
  z80memupb  := z80memupbb>>1                // The word upb of the Z80 memory
  mem     := getvec(z80memupbb/bytesperword) // 64K bytes of the Z80 memory

  // Initialise the whole Z80 memory
  FOR a = 0 TO z80memupbb-1 BY 2 DO putwb(a, #xDEAD)

  UNLESS rdargs("FROM,TO/K,GLBSIZE/K/N,STKSIZE/K/N,-s/S,-d/S", argv, 50) DO
  { writef("Bad arguments for z80emu*n")
    RESULTIS 0
  }

  fromfilename := 0
  glbsize := 1000  // Default sizes
  stksize := 4000
  
  IF argv!0 DO fromfilename := argv!0                    // FROM

  IF argv!1 DO tofilename := argv!1                      // TO/K

  IF argv!2 DO glbsize := !(argv!2)                      // GLBSIZE/K/N
  glbupb := glbsize-1

  IF argv!3 DO stksize := !(argv!3)                      // GLBSIZE/K/N
  stkupb := stksize-1

  singlestep := TRUE
  IF argv!4 DO singlestep := ~singlestep                 // -s/S
  tracing := singlestep
  
  IF argv!5 DO debug := ~debug                           // -d/S

  vars := 10 // word address of the variables vector inZ80 memory
  FOR i = 0 TO 9 DO putww(vars+i, 0) // Set all variables to zero
  
  bpt_addr   := TABLE 0,0,0,0,0,0,0,0,0,0
  bpt_instr  := TABLE 0,0,0,0,0,0,0,0,0,0
  style      := 'X'                   // Default printing style
  fullstatus := TRUE
  val        := 0                     // Current value
  brkstep    := FALSE
  //singlestep, tracing := FALSE, FALSE
  faultcode  := 0
  
  // Set the default values of the z80 registers.
  r_a,  r_b,  r_c,  r_d,  r_e,  r_h,  r_l  := 0, 0, 0, 0, 0, 0, 0
  r_a1, r_b1, r_c1, r_d1, r_e1, r_h1, r_l1 := 0, 0, 0, 0, 0, 0, 0
  r_ix, r_iy, r_sp, r_pc := 0, 0, 0, 0
  r_i,  r_r := 0, 0  // Interrupt page register and the memory refresh register

  // Individual bits of registers F and F' are determined by the following.
  r_fval,  r_fpval,  r_fh,  r_fn,  r_fc  := 0, 1, 0, 0, 0
  r_fval1, r_fpval1, r_fh1, r_fn1, r_fc1 := 0, 1, 0, 0, 0

  bkval := 0 // Not yet breaking on LD r,nn
  
  count := 1_000_000_000 // Decremented every time a Z80 intruction is executed.
  trcount := -1
  
// Allocate and initialise the root global vector
  rootgvec := 1000   // The root global vector starts at word addr 1000
  gptr := rootgvec
  putww(gptr, glbupb)
  FOR gn = 1 TO glbupb DO putww(gptr+gn, globword+gn)

  // Allocate and initialise the root stack
  rootstack := rootgvec+glbsize
    FOR a = 0 TO stkupb DO putww(rootstack+a, #xEEEE)

  // Set the location for code loaded by loadcode.
  codevec := rootstack+stksize
  codep := codevec<<1 // Byte address of where to load Intel Hex
  FOR i = 0 TO 1000 DO putww(codevec+i, #xC0DE)
  writef("gvec=%n stackbase=%n codebase=%n*n",
          rootgvec, rootstack, codevec)
  
  IF fromfilename DO loadhex(fromfilename)

  r_iy := (rootgvec <<1) + 128
  pptr := rootstack
  r_ix := (pptr<<1) + 128
  r_sp := 64000-2
  putwb(r_sp, 0)  // Set return address zero to cause a return from
                  // the z80emu command.
//  abort(4444)
  r_pc := 100          // Execution starts at byte address 100
  
  putbb(100, #xFD)     // LD L,(IY-124)    Set HL to G1 ie start
  putbb(101, #x6E)
  putbb(102, #x82)
  putbb(103, #xFD)     // LD H,(IY-125)
  putbb(104, #x66)
  putbb(105, #x83)
  putbb(106, #xFF)     // Apply 2          Enter start with return
  putbb(107, m_apply)  //                  address zero.
  putbb(108, #x02)
  putbb(109, #xFF)     // Exit from z80emu
  putbb(110, m_finish)
  
  putbb(111, #x00)
  putbb(112, #x00)

  putww(gptr+3, 113)       // Initialise sys
  
  putbb(113, #xFF)
  putbb(114, m_setlink3)   // SETLINK3
  putbb(115, #xFF)
  putbb(116, m_sys)        // SYS          Calls z80sys()
  putbb(117, #xFF)
  putbb(118, m_return)     // RETURN
  
  putbb(119, #x00)
  putbb(120, #x00)
  putbb(121, #x00)
  putbb(122, #x00)
  putbb(123, #x00)

  // Start execution in the interactive debugger.
  faultcode := 0
  brkstep := FALSE
  //singlestep, tracing := TRUE, TRUE  // Normally set by the -s option
  val := 0
  pflag := FALSE
  
  writef("*nEntering sadebug before executing the *
         *first Z80 instruction of the program*n*n")
  indentpos := 0
//db("Just after initialisation")

  UNLESS singlestep GOTO execute
  
  { sadebug()

    // singlestep is TRUE if single stepping
    // brkstep is TRUE if executing the instruction at a breakpoint.
    // If neither are TRUE call doz80instr repeatedly while faultcode is zero
execute:
//abort(6598)

    { // Execute Z80 intructions until the watch address has a new
      // val, or count or trcount becomes zero or the instruction
      // causes a non zero fault code.

      //writef("*nwatchaddr=%n watchval=%x4 getww(watchaddr)=%X4*n",
      //        watchaddr, watchval, getww(watchaddr))
      
      IF watchaddr UNLESS getww(watchaddr)=watchval DO
      { writef("*nWatch address %n changed from %x4 to %x4*n",
                 watchaddr, watchval, getww(watchaddr))
        watchval := getww(watchaddr)
        indentpos := 0
        prstate()
        brkstep := FALSE
        tracing := TRUE
        singlestep := TRUE
        BREAK
      }
  
      IF count=0 DO
      { faultcode := 2 // Zero count fault
        BREAK
      }
      IF trcount=0 DO
      { faultcode := 18
        BREAK
      }
      faultcode := execz80instr(r_pc)
      IF tracing DO prstate() // Possibly output the current state.
      // Note a count or trcount of -1 is treated as infinity.
      IF count>0   DO count   := count -   1
      IF trcount>0 DO trcount := trcount - 1
    } REPEATUNTIL faultcode | singlestep
  } REPEAT
  
fin:
  newline()
  IF mem DO freevec(mem)
}

AND evenparity(w) = w -> ~evenparity(w&(w-1)),
                         TRUE

AND nsize(n) = VALOF
{ LET res = 1
  IF n<0 DO { res := res+1; n := -n }
  WHILE n > 9 DO { res := res+1; n := ABS(n/10) }
  RESULTIS res
}

AND wrd(n, d) BE
{ FOR i = nsize(n)+1 TO d DO wrch(' ')
  wrn(n)
}

AND wrn(n) BE
{ IF n<0 DO { wrch('-'); n := -n }
  wrpn(n)
}

AND wrpn(n) BE
{ IF n > 9 DO wrpn((n>>1)/5)   // Unsigned division by 10
  wrch((n + 10) MOD 10 + '0')  // Unsigned MOD 10
}

AND sadebug() BE
// Enter the standalone debugger. What it does depends on the settings
// of the global variables faultcode trcount brkstep and singlestep.

// faultcode holds the fault code returned doz80instr when it last
//           executed an instruction. If non zero the fault is reported
//           and the execution state output followed by the debugger
//           prompt.

// brkstep   is only TRUE when the instruction at a breakpoint has just
//           been executed when resuming execution after a breakpoint.
//           If faultcode is non zero the fault is reported and the
//           execution state output followed by the debugger prompt.
//           If faultcode is zero and singlestep is FALSE, normal
//           execution of Z80 instructions resumes.

// singlstep If TRUE the execution state output followed by the
//           debugger prompt. If singlestep is FALSE only the
//           debugger prompt is output.

// When the debugger prompt has been output this function reads and
// performs debugging commands from the user. It only returns when
// another Z80 instruction needs to be executed.

{ recp := level()    // For longjump to goto the place where the 
  recl := recover    // debugger command prompt is output.

  retp := level()    // For longjump to use to cause a return from
  retl := ret        // sadebug causing another Z80 instruction to
                     // be executed.
	
  bpt   := -1        // No breakpoint number yet

  unsetbreakpoints()

  IF brkstep & faultcode=0 & ~singlestep DO
  { // We are resuming normal execution after the first instruction at
    // a breakpoint has just been executed.
    setbreakpoints()
    brkstep := FALSE
    RETURN
  }

  IF faultcode=18 DO
  { faultcode := 0
  }

  IF faultcode DO singlestep := TRUE

  // Now report the fault if faultcode is non zero
  
  IF faultcode=2 DO  // Fault caused by the BRK instruction
    FOR i = 0 TO 9 IF bpt_addr!i=val DO
    { bpt := i
      brkstep := TRUE // This must be TRUE when resuming from a breakpoint.
      writef("*n## BPT %n*n", bpt)
      indentpos := 0
      faultcode := 0
      prstate()
      //abort(999)
      BREAK
    }

  // Attempt to give a description of the current fault
  IF faultcode DO
  { LET mess =  VALOF SWITCHON faultcode INTO
    { DEFAULT:  RESULTIS "Unknown fault"
    
      CASE   1: RESULTIS "Illegal instruction"
      CASE   2: RESULTIS "BRK instruction"
      CASE   3: RESULTIS "Zero count"
      CASE   4: RESULTIS "pc out of range"
      CASE   5: RESULTIS "Division by zero"
      CASE  11: RESULTIS "Watch addr: %+%i7 value: %i8"
      CASE  12: RESULTIS "Indirect address out of range: %+%+%+%n"
      CASE  13: RESULTIS "SIGINT received"
      CASE  15: RESULTIS "PC out of range"
      CASE  16: RESULTIS "P pointer out of range"
      CASE  17: RESULTIS "SIGSEGV occurred"
      CASE  18: RESULTIS "Zero trcount"
      CASE  25: RESULTIS "Break after instruction of the form: LD r,#x%x4"
      CASE  99: RESULTIS "User requested"
      CASE 110: RESULTIS "Callco fault"
      CASE 111: RESULTIS "Resumeco fault"
      CASE 112: RESULTIS "Deleteco fault"
      CASE 186: RESULTIS "Bad input stream"
      CASE 187: RESULTIS "Bad output stream"
      CASE 188: RESULTIS "Unable to replenish input"
      CASE 189: RESULTIS "Wrch fault"
      CASE 190: RESULTIS "Endread fault"
      CASE 191: RESULTIS "Endwrite fault"
      CASE 197: RESULTIS "Store chain fault"
    }
    IF faultcode=2 DO
    { sawritef("*n## Breakpoint at PC=%n*n", r_pc)
      singlestep := TRUE
      brkstep := TRUE // When resuming execution at a breakpoint
                      // if brkstep is TRUE the next instruction
                      // to be executed with all breakpoints
                      // removed and with singlestep set to TRUE.
		      // If brkstep is TRUE and singlestep is FALSE
		      // after an instruction has been executed
		      // normal execution of instructions resumes
		      // with all current breakpoint set.
		      // brkstep is always set to FALSE except when
		      // resuming execution after a breakpoint.
       GOTO recover
    }
    sawritef("*n## ABORT %n: ", faultcode)
    sawritef(mess, bkval,0,0,0)
    sawrch('*n')
    indentpos := 0
    IF faultcode=25 DO
    { prstate()
      singlestep := TRUE
      tracing := TRUE
    }
  }

recover:
  ch := '*n'
nxt:                       // Main loop for debug commands
  IF ch='*n' DO sawritef("$ ")
  rch()
sw:
  singlestep := FALSE // Only set to TRUE by the '\' command.
  SWITCHON ch INTO

  { DEFAULT: error()

    CASE endstreamch:
    CASE 'Q': sawritef("*n"); sys(Sys_quit, 0)   // Quit
         
    CASE '*s':
    CASE '*t':
    CASE '*n': GOTO nxt

    CASE '?':
      newline()
      writes("?           Print this list of debug commands*n")
      writes("L name      Load an Intel Hex program into memory*n")
      writes("123 #o377 #FF03 'c  Constants*n")
      writes("V Vn        Variables vector and the variables 0 to 9*n")
      writes("G Gn        Global vector and global variables*n")
      writes("P Pn        Stack frame pointer and local variables*n")
      writes("R<reg>      Where <reg> is one of*n")
      writes("            A B C D E F H L AF BC DE HL SP PC IX IY*n")
      writes("!e          Subscription*n")
      writes("**e /e %e +e -e      Arithmetic operators*n")
      writes("|e &e ^e    Bit pattern operators OR, AND and XOR*n")
      writes("< >         Logical shift left/right one place*n")
      writes("=           Print current value*n")
      writes("F           Toggle the fullstatus flag*n")
      writes("$b $c $d $e $f $o $s $u $x Set the print style*n")
      writes("Sn SR<reg>  Store val in word address n or a register*n")
      writes("SPn SGn SVn Store val in a local, global or variable*n")
      writes("SXW SXB     Sign extend word or byte*n")
      writes("UW  UB      Eqivalent to &#FFFF and &#FF*n")
      writes("TRn         Trace the next n instructions*n")
      writes("TBn         Print n bytes from byte address val*n")
      writes("TS          Print the registers and the current instruction*n")
      writes("TF          Print the flags register F*n")
      writes("Tn          Print n words from word address val*n")
      writes("I           Print the instruction at byte address val and*n")
      writes("              increment val by the size of the instruction*n")
      writes("D           Dump all of Z80 memory to DUMP.mem*n")
      writes("Q           Quit -- leave z80emu*n")
      writes("M           Set/Reset memory watch word address*n")
      writes("B 0Bn eBn   List, Unset or Set breakpoints*n")
      writes("BK          If val is non zero cause a break when the next*n")
      writes("              instruction of the form; LD r,nn is encountered*n")
      writes("              with nn=val. If val is zero do not break on such*n")
      writes("              an instruction*n")
      //writes("X           Execute code until the next function return at this*n")
      //writes("              stack level*n")
      writes("C           Continue normal execution*n")
      writes("\           Execute one Z80 instruction*n")
      GOTO recover

    CASE 'L':  // L <filename>
    { LET len = 0
      LET name = VEC 50
      rch() REPEATWHILE ch='*s'
      WHILE 'A'<=ch<='Z' |
            '0'<=ch<='9' |
	    ch='-' | ch='_' | ch='.' | ch='/' DO
      { len := len+1
        IF len <= 50 * bytesperword DO name%len := lch
	rch()
      }
      name%0 := len
      writef("Loading Intel Hex file: %s*n", name)
      loadhex(name)
      GOTO sw
    }
    
    CASE '0': CASE '1': CASE '2':
    CASE '3': CASE '4': CASE '5':
    CASE '6': CASE '7': CASE '8':
    CASE '9': CASE '#': CASE '*'':
    CASE 'A': CASE 'G': CASE 'P': CASE 'R':
    CASE 'V':
              val := rdval();                     GOTO sw

    CASE 'U':
              rch()
	      IF ch='W' DO
	      { val := val & #xFF;
                newline()
		GOTO recover
	      }
	      IF ch='B' DO
	      { val := val & #xFF;
                newline()
		GOTO recover
	      }
	      writef("*nUW or UB was expected*n")
              GOTO recover

    CASE '!': rch(); val := cont(val + rdval());  GOTO sw
    CASE '+': rch(); val :=      val + rdval();   GOTO sw
    CASE '-': rch(); val :=      val - rdval();   GOTO sw
    CASE '**':rch(); val :=      val * rdval();   GOTO sw
    CASE '/': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val / a
                       GOTO sw
                     }
    CASE '%': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val REM a
                       GOTO sw
                     }
    CASE '|': rch(); val := val  |  rdval();  GOTO sw
    CASE '&': rch(); val := val  &  rdval();  GOTO sw
    CASE '^': rch(); val := val XOR rdval();  GOTO sw

    CASE '<': val := val << 1;                GOTO nxt
    CASE '>': val := val >> 1;                GOTO nxt

    CASE '=': prW(val); newline();            GOTO recover

    CASE '$': rch()
              UNLESS ch='B' | ch='C' | ch='D' | ch='F' |
                     ch='O' | ch='E' | ch='S' | ch='U' |
                     ch='X' | ch='Y' DO
              { writef("Valid style letters are: BCDEFOSUXY*n")
                GOTO recover
              }
              style := ch
              GOTO nxt

    CASE 'S': // Sn  SR<reg>  SPn  SGn  SVn  SXB  SXW
      rch()
      SWITCHON ch INTO
      { DEFAULT:
          GOTO sw

	CASE '0': CASE '1': CASE '2': CASE '3': CASE '4':
	CASE '5': CASE '6': CASE '7': CASE '8': CASE '9':
        { LET a = rdn()
          UNLESS 0<=a<z80memupb DO error() 
          putww(a, val)
          GOTO sw
        }

        CASE 'B': rch()
	{ LET n = rdn()
          UNLESS 0<=n<z80memupbb DO error() 
          putbb(n, val)
          GOTO sw
	}
	
        CASE 'R': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}
	
	CASE 'P': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'G': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'V': rch()
	{ LET n = rdn()
          UNLESS 0<=n<=9 DO error() 
          putww(vars+n, val)
          GOTO sw
        }

        CASE 'X':
	{ rch()
	  IF ch='W' DO
	  { val := signextw(val)
	    newline()
	    GOTO recover
	  }
	  IF ch='B' DO
	  { val := signextb(val)
	    newline()
	    GOTO recover
	  }
	  writef("*nSXW or SXB was expected*n")
          GOTO recover
        }
      }

    CASE '\':               // Single step execution.
      //db("CASE '\':")
      indentpos := 3        // Used by prstate()
      singlestep := TRUE
      tracing := TRUE
      RETURN                // Execute one Z80 intruction

    CASE 'X':  // Xn   Trace n instruction executions
      error()

    CASE 'H':
      writef("*nAF'=%x4 BC'=%x4 DE'=%x4 HL'=%x4*n",
              getr(n_af1), getr(n_bc1), getr(n_de1), getr(n_hl1))
	      GOTO recover

    CASE 'F':
      fullstatus := ~fullstatus
      writef(" Full Status is %s*n", fullstatus -> "on", "off")
      GOTO recover
      
    CASE 'T': // TS  TF  TRn  TBn  Tn
      rch()
      SWITCHON ch INTO
      { DEFAULT:
          error()

        CASE 'S':         // TS     Print the current state
        { newline()
          prstate()
          GOTO recover
        }

        CASE 'F':         // TF     Print the flags
        { writef(" ")
          prflags()
	  newline()
          GOTO recover
        }

        CASE 'R': rch()    // TRn    Trace n instructions
        { trcount := rdn()
          tracing := TRUE
	  singlestep := FALSE
	  GOTO ret         // Resume execution while
	                   // trcount>0 or faultcode~=0
			   // tracing each instruction.
        }

        CASE 'B': rch()    // TBn    Print n bytes from byte address val
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO writef("*n%i5:   ", val+i)
            prB(mem%(val+i))
	    wrch(' ')
          }
	  newline()
	  GOTO sw
        }

        CASE '0':CASE '1':CASE '2':CASE '3':CASE '4': // Tn
        CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO praddr(val+i)
            prW(getww(val+i))
	    wrch(' ')
          }
	  newline()
          GOTO recover
        }
      }

    CASE 'D': writef("*nCintcode memory dumped to DUMP.mem*n")
              sys(Sys_dumpmem, 6) // Dump the memory (context=6)
              GOTO recover

    CASE 'I': writef(" %i5: ", val)
              val := val+prz80instr(val)
 	      newline()
	      GOTO recover
 
    CASE 'Z':  // Equivalent to P1B9C
      error()
      //val := pptr!1  // set breakpoint 9 to current return address

    caseb:
    CASE 'B':  // Set, clear or display breakpoints.
    { LET comch = ch
      TEST comch='B' THEN rch()       // For B
                     ELSE ch := '9'   // For X or Z
      IF '0'<=ch<='9' DO
      { LET n = ch - '0'  // Set or Clear break point n.
        bpt_addr!n := 0
        IF val=0 GOTO nxt
        checkaddr(val>>B2Wsh)
	// Clear any breakpoint for this address
        FOR i = 0 TO 9 IF bpt_addr!i=val DO bpt_addr!i := 0
	// Set breakpoint n for this address
        bpt_addr!n, bpt_instr!n := val, mem%val
        GOTO comch='B' -> nxt, resume
      }
      IF ch='K' DO
      { // Set bkval to break on instructons o the form LD r,nn
	IF val DO
	{ writef("*nSetting a break after instruction form: LD r,#x%x4*n", val)
          bkval := val
	  GOTO recover
	}
	IF bkval DO
	{ writef("*nUnsetting a break after instruction form: LD r,#x%x4*n", bkval)
          bkval := 0
	  GOTO recover
	}
	writef("*nUnsetting a break after instruction form: LD r,nn*n")
	GOTO recover
      }
      // B is not followed by a digit so list the breakpoints.
      UNLESS ch='*n' DO newline()
      FOR i = 0 TO 9 DO
      { LET ba=bpt_addr!i
        IF ba DO
        { writef("%n:  ", i)
          writearg(ba)
          newline()
        }
      }
      GOTO recover
    }

    CASE 'M':  // Set or clear memory watch address
      checkaddr(val)
      TEST val THEN writef("*nWatch word address: %n*n", val)
               ELSE writef("*nWatch unset*n")
      watchaddr := val
      watchval  := getww(watchaddr)
      GOTO recover

    CASE 'C': // Continue normal execution.
      setbreakpoints()
      IF getbb(r_pc)=f_brk DO
      { brkstep := TRUE       // Resuming from a breakpoint
        unsetbreakpoints()
	RETURN                // Obey one instruction before restoring
	                      // the breakpoints.
      }
      brkstep := FALSE
      singlestep := FALSE
      tracing := FALSE
      newline()
      //db("CASE 'C':")
      RETURN

resume:
    { LET pc = r_pc
      newline()
      FOR i = 0 TO 9 IF pc=bpt_addr!i DO
      { // We are resuming at a break point
        writef("Resuming after breakpoint %n*n", i)
        brkstep := TRUE       // Execute one instruction before reinstating
        RETURN                // the BRK instructions
      }
      writef("Resuming normal execution*n")
    }
  }

ret:
  UNLESS brkstep DO setbreakpoints()
  RETURN
}

AND setbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO
  { bpt_instr!i := getbb(ba) // Save the previous instruction at the
    putbb(ba, f_brk)         // breakpoint. and replace it by BRK (=RST #x28)
  }
}

AND unsetbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO putbb(ba, bpt_instr!i) // Restore the original instruction.
}

AND prprompt() BE
{ 
  writef("** ")  // Standalone prompt
}


AND writearg(n) BE
{ LET tag = n & tagbits
//abort(2398)
  IF tag=globword DO
  { //abort(9898)
    writef("  #G%z3#", n & nontagbits)
    RETURN
  }
  IF n=#xC0DE DO
  { writef("  #x%x4", n)
    RETURN
  }
  writef("  %6i", n)
}

AND rdn() = VALOF
{ LET res = 0
  WHILE '0'<=ch<='9' DO { res := res*10 + ch - '0'; rch() }
  RESULTIS res
}

AND rdreg() = VALOF
{ // Read a register name and return its register number.
  // The result is -1 if the register in not recognised.
  // On entry ch is the first character of the name.
  // On exit ch is the character following the name.
  // The registers are:
  //  B C D E H L F A AF BC DE HL SP PC IX IY
  
  SWITCHON ch INTO
  { DEFAULT:
      rch(); RESULTIS -1
    CASE 'A':
      rch()
      UNLESS ch='F' RESULTIS n_a
      rch(); RESULTIS n_af
    CASE 'B':
      rch()
      UNLESS ch='C' RESULTIS n_b
      rch(); RESULTIS n_bc
    CASE 'C':
      rch(); RESULTIS n_c
    CASE 'D':
      rch()
      UNLESS ch='E' RESULTIS n_d
      rch(); RESULTIS n_de
    CASE 'E':
      rch(); RESULTIS n_e
    CASE 'H':
      rch()
      UNLESS ch='L' RESULTIS n_h
      rch(); RESULTIS n_hl
    CASE 'L':
      rch(); RESULTIS n_l
    CASE 'F':
      rch(); RESULTIS n_f
    CASE 'S':
      rch()
      UNLESS ch='P' RESULTIS -1
      rch(); RESULTIS n_sp
    CASE 'P':
      rch()
      UNLESS ch='C' RESULTIS -1
      rch(); RESULTIS n_pc
    CASE 'I':
      rch()
      IF ch='X' DO { rch(); RESULTIS n_ix }
      IF ch='Y' DO { rch(); RESULTIS n_iy }
      RESULTIS -1
  }
}

AND rdargval(str) = VALOF
{ // str is either a decimal number ddd..d
  // or        a hexadecimal number #hh..h
  LET res = 0
  LET radix = 0
  FOR i = 1 TO str%0 DO
  { LET ch = capitalch(str%i)
    SWITCHON ch INTO
    { DEFAULT:
      CASE '#': UNLESS i=1 RESULTIS -1
                radix := 16
		LOOP

      CASE '0':CASE '1':CASE '2':CASE '3':CASE '4':
      CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
                res := res*radix + ch - '0'
		LOOP

      CASE 'A':CASE 'B':CASE 'C':CASE 'D':CASE 'E':CASE 'F':
                UNLESS radix=16 RESULTIS -1
                res := res*radix + ch - 'A'
		LOOP

    }
  }
  RESULTIS res
  
}

AND rdval() = VALOF
{ // Read a val from the current input stream with first
  // character in ch.
  // Possible values:
  // V Vn G Gn P Pn P
  // 'c #<hex> #O<oct> n
  // RA  RB  RC  RD  RE  RF  RH  RL
  // RIX RIY RSP RPC RAF RBC RDE RHL
  LET res, radix = 0, 10

  WHILE ch='*s' DO rch()
  
  SWITCHON ch INTO
  { DEFAULT:   error()

    CASE 'V':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         UNLESS 0<=n<=9 DO
		 { writef("*nVariable number %n out of range*n", n)
		   RESULTIS 0
		 }
	         RESULTIS getww(vars+n)
	       }
               RESULTIS vars // Return the word address of the variables vector

    CASE 'A':  rch()        // A n   Return the contents of word address n
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(n)
	       }
               RESULTIS 0

    CASE 'G':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(gptr+n)
	       }
               RESULTIS gptr

    CASE 'P':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(pptr+n)
	       }
               RESULTIS pptr

    CASE '*'': rch(); res := lch; rch();  RESULTIS res

    CASE '#':  radix := 16
               rch()
               IF ch='O' DO { radix := 8; rch() }

    CASE '0': CASE '1': CASE '2': CASE '3': CASE '4': 
    CASE '5': CASE '6': CASE '7': CASE '8': CASE '9': 
             { LET d = 100
               IF '0'<=ch<='9' DO d := ch-'0'
               IF 'A'<=ch<='F' DO d := ch-'A'+10
               IF d>=radix RESULTIS res
               res := res*radix+d
               rch()
             } REPEAT

    CASE 'R':
      rch()
      RESULTIS getr(rdreg()) 
  }
}

AND praddr(a) BE
{ LET type, base = 'A', 0
  LET fsize = 100
  IF pptr <= a <= pptr+fsize & a <= pptr+100  DO type, base := 'P', pptr
  IF gptr <= a <= gptr+getww(gptr+g_globsize) DO type, base := 'G', gptr
  IF vars <= a <= vars+9                      DO type, base := 'V', vars
  writef("*n%c%i5: ", type, a-base)
}

AND prch(ch) BE
{ IF 32 < ch <= 127 DO
  { writef("   %c", ch)
    RETURN
  }
  SWITCHON ch INTO
  { DEFAULT:   writef(" %i3", ch); RETURN
    CASE '*s': writef("  SP");     RETURN
    CASE '*b': writef("  BS");     RETURN
    CASE '*n': writef("  NL");     RETURN
    CASE '*c': writef("  CR");     RETURN
    CASE '*p': writef("  FF");     RETURN
  }
}

AND prB(b) BE
{ // Print a byte in the current style
  b := b & #xFF
  SWITCHON style INTO
  { DEFAULT:   error();              RETURN
    CASE 'C':  prch(b);              RETURN
    CASE 'B':  writef("%8b", b);     RETURN
    CASE 'F':
    CASE 'S':
    CASE 'D':  writef("%4i", signextb(b));     RETURN
    CASE 'O':  writef("%3o", b);     RETURN
    CASE 'U':  writef("%3u", b);     RETURN
    CASE 'X':  writef("%2x", b);     RETURN
    CASE 'Y':  writef("%2x", b);     RETURN
  }

}

AND prW(w) BE
{ // Print a word in the current style
  SWITCHON style INTO
  { DEFAULT:   error();                 RETURN
    CASE 'C':  { LET ch = w & #xFF
                 //writes(" ")
                 wrch(32<=ch<=127 -> ch, '?')
	         ch := w>>8 & #xFF
                 wrch(32<=ch<=127 -> ch, '?')
                 RETURN
               }
    CASE 'B':  writef("%16b", w);           RETURN
    CASE 'D':  writef("%6i",  signextw(w)); RETURN
    CASE 'F':  writearg(w);                 RETURN
    CASE 'O':  writef("%6o",  w&#xFFFF);    RETURN
    CASE 'S':  checkaddr(w)
               writef("%S",   w);           RETURN
    CASE 'U':  writef("%6u",  w&#xFFFF);    RETURN
    CASE 'X':  writef("%4x",  w);           RETURN
    CASE 'Y':  writef("%2x",  w);           RETURN
  }
}

AND checkaddr(a) = VALOF // a is a BCPL pointer in the Z80 memory
{ UNLESS 0<=a<=z80memupb DO error()
  RESULTIS a
}

AND cont(a) = getww(a) // a is a BCPL pointer

AND error() BE { writes("  ??*n"); longjump(recp, recl) }

AND prstate() BE
{ // indentpos is the current indentation position
  LET prevpflag = pflag // pflag must be preserved
  IF fullstatus DO
  { FOR i = indentpos TO 3 DO wrch(' ')
    writef("P=%i5: %i5 %i5", pptr, getww(pptr), getww(pptr+1))
    FOR i = 2 TO 5 DO { wrch(' '); prW(getww(pptr+i)) }
    writef("  SP->["); prW(getwb(r_sp)); wrch(' '); prW(getwb(r_sp+2))
    writef("]  "); prflags()
    newline()
    indentpos := 0
  }
  FOR i = indentpos TO 3 DO wrch(' ')
  writef("A="); prB(r_a)
  writef(" BC="); prW(getr(n_bc))
  writef(" DE="); prW(getr(n_de))
  writef(" HL="); prW(getr(n_hl))
  writef(" IX="); prW(r_ix)
  writef(" IY="); prW(r_iy)
  writef(" sp=%i5 %i5:", r_sp, r_pc)
  prz80instr(r_pc)
  newline()
  pflag := prevpflag
  //abort(7687)
}

AND getbb(a) = mem%a

AND getsbb(a) = signextb(mem%a)

AND getwb(a) = mem%a | mem%(a+1)<<8

AND getswb(a) = signextw(mem%a | mem%(a+1)<<8)

AND getww(a) = getwb(a+a)

AND getsww(a) = signextw(getwb(a+a))

AND putbb(a, val) BE mem%a := val

AND putwb(a, val) BE mem%a, mem%(a+1) := val, val>>8

AND putww(a, val) BE putwb(a+a, val)

AND signextb(byte) = VALOF
{ LET res = byte & #x7F
  IF (byte & #x80)>0 DO res := res - #x80
  RESULTIS res
}

AND signextw(word) = VALOF
{ LET res = word & #x7FFF
  IF (word & #x8000)>0 DO res := res - #x8000
  RESULTIS res
}

AND putreg(n, val) BE SWITCHON n INTO
{ DEFAULT:
    error()
    RETURN

  CASE  n_a: r_a  := signextb(val);    RETURN  
  CASE  n_b: r_b  := signextb(val);    RETURN  
  CASE  n_c: r_c  := signextb(val);    RETURN  
  CASE  n_d: r_d  := signextb(val);    RETURN  
  CASE  n_e: r_e  := signextb(val);    RETURN  
  CASE  n_f: r_f  := signextb(val);    RETURN  
  CASE  n_h: r_h  := signextb(val);    RETURN  
  CASE  n_l: r_l  := signextb(val);    RETURN  
  CASE n_ix: r_ix := val & #xFFFF;     RETURN  
  CASE n_iy: r_iy := val & #xFFFF;     RETURN  
  CASE n_sp: r_sp := val & #xFFFF;     RETURN  
  CASE n_pc: r_pc := val & #xFFFF;     RETURN  
  CASE n_af: r_a  := signextb(val>>8)
             setflags(val);            RETURN  
  CASE n_bc: r_b, r_c := signextb(val>>8), signextb(val); RETURN
  CASE n_de: r_d, r_e := signextb(val>>8), signextb(val); RETURN
  CASE n_hl: r_h, r_l := signextb(val>>8), signextb(val); RETURN
}

AND rch() BE { lch := sardch(); ch := capitalch(lch) }

AND loadhex(filename) = VALOF
{ // Load BCPL Intel Hex from the given file starting at byte address codep.
  // Globals defined in this file update entries in the global vector
  // with word address rootgvec.
  // Relocation data is encoded by lines starting with $02000004 followed by
  // 4 byte hex values giving the addrsses of the words to relocate and
  // terminated by the checksum byte.


  LET prevstream = input()
  LET hexstream = findinput(filename)
  UNLESS hexstream DO
  { writef("Cannot open file: *"%s*"*n", filename)
    RESULTIS FALSE
  }
  selectinput(hexstream)

  { // Read an Intel Hex record
    LET ch = rdch()
//IF ch>0 DO writef("ch='%c'*n", ch)
//abort(1001)

    IF ch=endstreamch BREAK

    IF ch='$' DO
    { // Either a relocation record or an ingnorable record
      LET n    = rdhexbyte()    // Number of addresses in this record, or ignored
      LET ch1  = rdhexbyte()    // low byte of load address, ignored
      LET ch2  = rdhexbyte()    // low byte of load address, ignored
      LET type = rdhexbyte()    // The record's type, =04 for a relocation block
      LET addr = (ch1<<8) + ch2 // Big ender ordering, ignored
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

      IF type=4 DO
      {
//writef("$%x2%x2%x2%x2*n", n, ch1,ch2, type)
//writef("loadhex: type=4 ch=%c n=%n ch1=%n ch2=%n addr=%n*n",
//                        ch,   n,   ch1,   ch2,   addr)
        FOR i = 1 TO n DO
        { LET p = codep//+6
	  ch1 := rdhexbyte()
          ch2 := rdhexbyte()
	  addr := (ch1<<8) + ch2
	  p := p + addr             // Need not be 2 byte aligned.
          sum := sum+ch1+ch2
          putwb(p, codep+getwb(p))

//writef("codep=%n*n",codep)
//writef("Word at %n changed from %n to %n*n", p, getwb(p), codep+getwb(p))
//abort(1009)
        }
      
//        newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
      
      UNTIL ch='*n' | ch=endstreamch DO ch := rdch()
      IF ch='*n' LOOP
    }


    IF ch=':' DO
    { // Format: : nn aaaa tt   followed by bytes to load  and the checksum byte
      LET n    = rdhexbyte()  // Number of code bytes in this record
      LET ch1  = rdhexbyte()  // low byte of load address
      LET ch2  = rdhexbyte()  // low byte of load address
      LET type = rdhexbyte()  // The record's type
      LET addr = ch1<<8 | ch2 // Big ender ordering
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

//      writef("n=%i2 addr=%i5 type=%n*n", n, addr, type)
      
      IF type=0 DO
      { FOR i = 1 TO n DO
        { LET byte = rdhexbyte()
          sum := sum+byte
          putbb(codep+addr, byte)
//writef(" %x2", byte)
          addr := addr+1
        }
      
        //newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
    }
    
    IF ch=endstreamch BREAK
    ch:=ch // To get round a compiler bug
  } REPEAT


//writef("Init the globals*n")
//abort(1003)
  // Initialise the globals
  
  // BCPL sections start with bytes 'B' 'C''P' 'L' followed by 2 bytes
  // giving the size of the section in bytes.
  { LET size = getwb(codep+4)
    LET p = codep+size-6
//    writef("codep=%n size=%n*n", codep, size)
//    FOR a = 0 TO size*2 -1 DO writef("%x2: %x2*n", a, getbb(codep+a))
//    newline()


    { LET gn    = getwb(p)>>1
      LET entry = getwb(p+2)
      IF entry=0 | p < codep BREAK
      putww(rootgvec+gn, entry)
      //writef("gn=%i3 entry=%n*n", gn, entry)
      p := p-4
    } REPEAT
    
    //abort(1234)
//    FOR a = codep TO codep+size-2 BY 2 DO writef("%i5: %x4*n", a, getwb(a))
  }
  
  IF hexstream DO endstream(hexstream)
  selectinput(prevstream)
  RESULTIS TRUE
}

AND rdhexbyte() = VALOF
{ // Read tw hex characters and retun the specified byte
  LET res = hexval(rdch())
  RESULTIS res<<4 | hexval(rdch())
}

AND hexval(ch) = '0'<=ch<='9' -> ch-'0',
                 'A'<=ch<='F' -> ch-'A'+10,
		 'a'<=ch<='f' -> ch-'a'+10,
		                 -1

AND z8dumpmem() BE
{ LET dumpstream = findoutput("Z80DUMP")
  selectoutput(dumpstream)
  writef("AF=%x4*n", r_a<<8 | r_f)
  writef("BC=%x4*n", r_c<<8 | r_b)
  writef("DE=%x4*n", r_e<<8 | r_d)
  writef("HL=%x4*n", r_l<<8 | r_h)
  writef("IX=%x4*n", r_ix)
  writef("IY=%x4*n", r_iy)
  writef("SP=%x4*n", r_sp)
  writef("PC=%x4*n", r_pc)
  writef("MEM=*n")
  FOR i = 0 TO z80memupbb DO
  { IF i MOD 32 = 0 DO newline()
    writef("%x2", getbb(i))
  }
  newline()
  endstream(dumpstream)
}

AND prizd(z, d) BE // eg (IX-114)  or  (IY+12)
{ writef("(%s", r2str(z))
  IF d>0 DO writef("+%n", d)
  IF d<0 DO writef("-%n", -d)
  writef(")")
}

AND f2flags() BE
{ // Use r_f to set r_fval. r_fh. r_fpval, r_fn and r_fc
  r_fval := 1
  IF (r_f & fb_s)>0 DO r_fval := -1
  IF (r_f & fb_s)>0 DO r_fval :=  0
  r_fh    := (r_f & fb_h)>0 -> 1, 0
  r_fpval := (r_f & fb_p)>0 -> 0, 1
  r_fn    := (r_f & fb_n)>0 -> 1, 0
  r_fc    := (r_f & fb_c)>0 -> 1, 0
}

AND flags2f() BE
{ // Use  r_fval. r_fh. r_fpval, r_fn and r_fc to set r_f
  r_f := 1
  IF r_fval<0            DO r_f := r_f | fb_s
  IF r_fval=0            DO r_f := r_f | fb_z
  IF r_fh                DO r_f := r_f | fb_h
  IF evenparity(r_fpval) DO r_f := r_f | fb_p
  IF r_fn                DO r_f := r_f | fb_n
  IF r_fc                DO r_f := r_f | fb_c
}

AND prflags(flags) BE
{ writef("F=")
  IF r_fval<0            DO wrch('S')
  IF r_fval=0            DO wrch('Z')
  IF r_fh                DO wrch('H')
  IF evenparity(r_fpval) DO wrch('V')
  IF r_fn                DO wrch('N')
  IF r_fc                DO wrch('C')
}

AND prz80instr(pc) = VALOF
{ // Print the Z80 instruction at location pc
  // and return its size in bytes.
  pflag := TRUE
  RESULTIS doz80instr(pc)
}

AND execz80instr(pc) = VALOF
{ // Execute the Z80 instruction at location pc =r_pc
  // and return zero or a non zero fault code.
  IF count=0   RESULTIS 1    // Zero count fault
  IF count>0 DO count := count-1
  IF trcount=0 RESULTIS 18   // Zero trcunt fault
  IF trcount>0 DO trcount := count-1
  
  pflag := FALSE
  RESULTIS doz80instr(pc)
}
// Functions whose name start with z80 usually print or execute
// instructions depending of the value of pflag. The fuction names
// have the following convention. After z80 there is the function
// code mnemonic in capitals followed by w if the instruction involves
// 16 bit quantities. Next is a digit 2, 3 or 4 giving the size of the
// instruction unless its size is one. The first argument is pc, the
// address of the instruction. If the instruction is being executed
// (ie if pflag=FALSE), this first aregument is equal to r_pc. The
// extra arguments, if any, are described by the remaining letters of
// the function name as follows.

// r   a register                    z80DECr(pc, n_a)         DEC A
// ir  an indirect register          z80DECir(pc,n_hl)        DEC (HL)
// iz  an indirect register          z80JP2iz(pc,n_ix)        JP (IX)
// izd an indexed address            z80LD3rizd(n_a,n_ix,d))  LD A,(IX+d)
// d   a signed byte                 z80JRd(pc,d)             JR d
// n   an unsigned byte              z80LD2rn(pc,n_a,n)       LD A,n
// nn  an integer on the range 65535 z80JP3nn(pc,nn)          JP nn
// inn an indirect address           z80LD3rnn(pc,n_a,nn)     LD A,(nn)
// cc  an condition code             z80JP2d(pc,cc_nz,d)      JR NZ,d
// b   a bit position 0-7            z80BIT2bir(pc,b,n_hl)    BIT b,(HL)

AND doz80instr(pc) = VALOF SWITCHON getbb(pc) INTO
{ // If pflag=TRUE output the instruction at location pc and
  //               return its size in bytes.
  // If pflag=FALSE execute the instruction at location pc (=r_pc) and
  // return zero if the instruction does not cause a fault or
  // return a non zero fault code.
  

  CASE #x00: RESULTIS z80NOP()                           // NOP
  CASE #x01: RESULTIS z80LDw3rnn(pc, n_bc, getswb(pc+1)) // LD BC,nn
  CASE #x02: RESULTIS z80LDirr(pc, n_bc, n_a)            // LD (BC),A
  CASE #x03: RESULTIS z80INCwr(pc, n_bc)                 // INC BC
  CASE #x04: RESULTIS z80INCr(pc, n_b)                   // INC B
  CASE #x05: RESULTIS z80DECr(pc, n_b)                   // DEC B
  CASE #x06: RESULTIS z80LD2rn(pc, n_b, getbb(pc+1))     // LD B,n
  CASE #x07: RESULTIS z80RLCA(pc)                        // RLCA
  CASE #x08: RESULTIS z80EXwrr(pc, n_af, n_af1)          // EX AF,AF'
  CASE #x09: RESULTIS z80ADDwrr(pc, n_hl, n_bc)          // ADD HL,BC
  CASE #x0A: RESULTIS z80LDrir(pc, n_a, n_bc)            // LD A,(BC)
  CASE #x0B: RESULTIS z80DECwr(pc, n_bc)                 // DEC BC
  CASE #x0C: RESULTIS z80INCr(pc, n_c)                   // INC C
  CASE #x0D: RESULTIS z80DECr(pc, n_c)                   // DEC C
  CASE #x0E: RESULTIS z80LD2rn(pc, n_c, getbb(pc+1))     // LD C,n
  CASE #x0F: RESULTIS z80RRCA(pc)                        // RRCA

  CASE #x10: RESULTIS z80DJNZ2d(pc, getsbb(pc+1))        // DJNZ d
  CASE #x11: RESULTIS z80LDw3rnn(pc, n_de, getswb(pc+1)) // LD DE,nn
  CASE #x12: RESULTIS z80LDirr(pc, n_de, n_a)            // LD (DE),A
  CASE #x13: RESULTIS z80INCwr(pc, n_de)                 // INC DE
  CASE #x14: RESULTIS z80INCr(pc, n_d)                   // INC D
  CASE #x15: RESULTIS z80DECr(pc, n_d)                   // DEC D
  CASE #x16: RESULTIS z80LD2rn(pc, n_d, getbb(pc+1))     // LD D,n
  CASE #x17: RESULTIS z80RLA(pc)                         // RLA
  CASE #x18: RESULTIS z80JR2d(pc, getsbb(pc+1))          // JR d
  CASE #x19: RESULTIS z80ADDwrr(pc, n_hl, n_de)          // ADD HL,DE
  CASE #x1A: RESULTIS z80LDrir(pc, n_a, n_de)            // LD A,(DE)
  CASE #x1B: RESULTIS z80DECwr(pc, n_de)                 // DEC DE
  CASE #x1C: RESULTIS z80INCr(pc, n_e)                   // INC E
  CASE #x1D: RESULTIS z80DECr(pc, n_e)                   // DEC E
  CASE #x1E: RESULTIS z80LD2rn(pc, n_e, getbb(pc+1))     // LD E,n
  CASE #x1F: RESULTIS z80RRA(pc)                         // RRA

  CASE #x20: RESULTIS z80JR2ccd(pc, cc_nz, getsbb(pc+1)) // JR NZ,d
  CASE #x21: RESULTIS z80LDw3rnn(pc, n_hl, getswb(pc+1)) // LD HL,nn
  CASE #x22: RESULTIS z80LDw3innr(pc, getwb(pc+1), n_hl) // LD (nn),HL
  CASE #x23: RESULTIS z80INCwr(pc, n_hl)                 // INC HL
  CASE #x24: RESULTIS z80INCr(pc, n_h)                   // INC H
  CASE #x25: RESULTIS z80DECr(pc, n_h)                   // DEC H
  CASE #x26: RESULTIS z80LD2rn(pc, n_h, getbb(pc+1))     // LD H,n
  CASE #x27: RESULTIS z80DAA(pc)                         // DAA
  CASE #x28: RESULTIS z80JR2ccd(pc, cc_z, getsbb(pc+1))  // JR Z,d
  CASE #x29: RESULTIS z80ADDwrr(pc, n_hl, n_hl)          // ADD HL,HL
  CASE #x2A: RESULTIS z80LDw3rinn(pc, n_hl, getwb(pc+1)) // LD HL,(nn)
  CASE #x2B: RESULTIS z80DECwr(pc, n_hl)                 // DEC HL
  CASE #x2C: RESULTIS z80INCr(pc, n_l)                   // INC L
  CASE #x2D: RESULTIS z80DECr(pc, n_l)                   // DEC L
  CASE #x2E: RESULTIS z80LD2rn(pc, n_l, getbb(pc+1))     // LD L,n
  CASE #x2F: RESULTIS z80CPL(pc)                         // CPL

  CASE #x30: RESULTIS z80JR2ccd(pc, cc_nc, getsbb(pc+1)) // JR NC,d
  CASE #x31: RESULTIS z80LDw3rnn(pc, n_sp, getswb(pc+1)) // LD SP,nn
  CASE #x32: RESULTIS z80LD3innr(pc, getwb(pc+1), n_a)   // LD (nn),A
  CASE #x33: RESULTIS z80INCwr(pc, n_sp)                 // INC SP
  CASE #x34: RESULTIS z80INCir(pc, n_hl)                 // INC (HL)
  CASE #x35: RESULTIS z80DECir(pc, n_hl)                 // DEC (HL)
  CASE #x36: RESULTIS z80LD2irn(pc, n_hl, getbb(pc+1))   // LD (HL),n
  CASE #x37: RESULTIS z80SCF(pc)                         // SCF
  CASE #x38: RESULTIS z80JR2ccd(pc, cc_c, getsbb(pc+1))  // JR C,d
  CASE #x39: RESULTIS z80ADDwrr(pc, n_hl, n_sp)          // ADD HL,SP
  CASE #x3A: RESULTIS z80LD3rinn(pc, n_a, getwb(pc+1))   // LD A,(nn)
  CASE #x3B: RESULTIS z80DECr(pc, n_sp)                  // DEC SP
  CASE #x3C: RESULTIS z80INCr(pc, n_a)                   // INC A
  CASE #x3D: RESULTIS z80DECr(pc, n_a)                   // DEC A
  CASE #x3E: RESULTIS z80LD2rn(pc, n_a, getbb(pc+1))     // LD A,n
  CASE #x3F: RESULTIS z80CCF(pc)                         // CCF

    CASE #x40: RESULTIS z80LDrr(pc, n_b, n_b)      // LD B,B
    CASE #x41: RESULTIS z80LDrr(pc, n_b, n_c)      // LD B,C
    CASE #x42: RESULTIS z80LDrr(pc, n_b, n_d)      // LD B,D
    CASE #x43: RESULTIS z80LDrr(pc, n_b, n_e)      // LD B,E
    CASE #x44: RESULTIS z80LDrr(pc, n_b, n_h)      // LD B,H
    CASE #x45: RESULTIS z80LDrr(pc, n_b, n_l)      // LD B,L
    CASE #x46: RESULTIS z80LDrir(pc, n_b, n_hl)    // LD B,(HL)
    CASE #x47: RESULTIS z80LDrr(pc, n_b, n_a)      // LD B,A
    CASE #x48: RESULTIS z80LDrr(pc, n_c, n_b)      // LD C,B
    CASE #x49: RESULTIS z80LDrr(pc, n_c, n_c)      // LD C,C
    CASE #x4A: RESULTIS z80LDrr(pc, n_c, n_d)      // LD C,D
    CASE #x4B: RESULTIS z80LDrr(pc, n_c, n_e)      // LD C,E
    CASE #x4C: RESULTIS z80LDrr(pc, n_c, n_h)      // LD C,H
    CASE #x4D: RESULTIS z80LDrr(pc, n_c, n_l)      // LD C,L
    CASE #x4E: RESULTIS z80LDrir(pc, n_c, n_hl)    // LD C,(HL)
    CASE #x4F: RESULTIS z80LDrr(pc, n_c, n_a)      // LD C,A

    CASE #x50: RESULTIS z80LDrr(pc, n_d, n_b)      // LD D,B
    CASE #x51: RESULTIS z80LDrr(pc, n_d, n_c)      // LD D,C
    CASE #x52: RESULTIS z80LDrr(pc, n_d, n_d)      // LD D,D
    CASE #x53: RESULTIS z80LDrr(pc, n_d, n_e)      // LD D,E
    CASE #x54: RESULTIS z80LDrr(pc, n_d, n_h)      // LD D,H
    CASE #x55: RESULTIS z80LDrr(pc, n_d, n_l)      // LD D,L
    CASE #x56: RESULTIS z80LDrir(pc, n_d, n_hl)    // LD D,(HL)
    CASE #x57: RESULTIS z80LDrr(pc, n_d, n_a)      // LD D,A
    CASE #x58: RESULTIS z80LDrr(pc, n_e, n_b)      // LD E,B
    CASE #x59: RESULTIS z80LDrr(pc, n_e, n_c)      // LD E,C
    CASE #x5A: RESULTIS z80LDrr(pc, n_e, n_d)      // LD E,D
    CASE #x5B: RESULTIS z80LDrr(pc, n_e, n_e)      // LD E,E
    CASE #x5C: RESULTIS z80LDrr(pc, n_e, n_h)      // LD E,H
    CASE #x5D: RESULTIS z80LDrr(pc, n_e, n_l)      // LD E,L
    CASE #x5E: RESULTIS z80LDrir(pc, n_e, n_hl)    // LD E,(HL)
    CASE #x5F: RESULTIS z80LDrr(pc, n_e, n_a)      // LD E,A

    CASE #x60: RESULTIS z80LDrr(pc, n_h, n_b)      // LD H,B
    CASE #x61: RESULTIS z80LDrr(pc, n_h, n_c)      // LD H,C
    CASE #x62: RESULTIS z80LDrr(pc, n_h, n_d)      // LD H,D
    CASE #x63: RESULTIS z80LDrr(pc, n_h, n_e)      // LD H,E
    CASE #x64: RESULTIS z80LDrr(pc, n_h, n_h)      // LD H,H
    CASE #x65: RESULTIS z80LDrr(pc, n_h, n_l)      // LD H,L
    CASE #x66: RESULTIS z80LDrir(pc, n_h, n_hl)    // LD H,(HL)
    CASE #x67: RESULTIS z80LDrr(pc, n_h, n_a)      // LD H,A
    CASE #x68: RESULTIS z80LDrr(pc, n_l, n_b)      // LD L,B
    CASE #x69: RESULTIS z80LDrr(pc, n_l, n_c)      // LD L,C
    CASE #x6A: RESULTIS z80LDrr(pc, n_l, n_d)      // LD L,D
    CASE #x6B: RESULTIS z80LDrr(pc, n_l, n_e)      // LD L,E
    CASE #x6C: RESULTIS z80LDrr(pc, n_l, n_h)      // LD L,H
    CASE #x6D: RESULTIS z80LDrr(pc, n_l, n_l)      // LD L,L
    CASE #x6E: RESULTIS z80LDrir(pc, n_l, n_hl)    // LD L,(HL)
    CASE #x6F: RESULTIS z80LDrr(pc, n_l, n_a)      // LD L,A

    CASE #x70: RESULTIS z80LDirr(pc, n_hl, n_b)    // LD (HL),B
    CASE #x71: RESULTIS z80LDirr(pc, n_hl, n_c)    // LD (HL),C
    CASE #x72: RESULTIS z80LDirr(pc, n_hl, n_d)    // LD (HL),D
    CASE #x73: RESULTIS z80LDirr(pc, n_hl, n_e)    // LD (HL),E
    CASE #x74: RESULTIS z80LDirr(pc, n_hl, n_h)    // LD (HL),H
    CASE #x75: RESULTIS z80LDirr(pc, n_hl, n_l)    // LD (HL),L
    CASE #x76: RESULTIS z80HALT(pc)                // HALT
    CASE #x77: RESULTIS z80LDirr(pc, n_hl, n_a)    // LD (HL),A
    CASE #x78: RESULTIS z80LDrr(pc, n_a, n_b)      // LD A,B
    CASE #x79: RESULTIS z80LDrr(pc, n_a, n_c)      // LD A,C
    CASE #x7A: RESULTIS z80LDrr(pc, n_a, n_d)      // LD A,D
    CASE #x7B: RESULTIS z80LDrr(pc, n_a, n_e)      // LD A,E
    CASE #x7C: RESULTIS z80LDrr(pc, n_a, n_h)      // LD A,H
    CASE #x7D: RESULTIS z80LDrr(pc, n_a, n_l)      // LD A,L
    CASE #x7E: RESULTIS z80LDrir(pc, n_a, n_hl)    // LD A,(HL)
    CASE #x7F: RESULTIS z80LDrr(pc, n_a, n_a)      // LD A,A

    CASE #x80: RESULTIS z80ADDrr(pc, n_a, n_b)           // ADD A,B
    CASE #x81: RESULTIS z80ADDrr(pc, n_a, n_c)           // ADD A,C
    CASE #x82: RESULTIS z80ADDrr(pc, n_a, n_d)           // ADD A,D
    CASE #x83: RESULTIS z80ADDrr(pc, n_a, n_e)           // ADD A,E
    CASE #x84: RESULTIS z80ADDrr(pc, n_a, n_h)           // ADD A,H
    CASE #x85: RESULTIS z80ADDrr(pc, n_a, n_l)           // ADD A,L
    CASE #x86: RESULTIS z80ADDrir(pc, n_a, n_hl)         // ADD A,(HL)
    CASE #x87: RESULTIS z80ADDrr(pc, n_a, n_a)           // ADD A,A
    CASE #x88: RESULTIS z80ADCrr(pc, n_a, n_b)           // ADC A,B
    CASE #x89: RESULTIS z80ADCrr(pc, n_a, n_c)           // ADC A,C
    CASE #x8A: RESULTIS z80ADCrr(pc, n_a, n_d)           // ADC A,D
    CASE #x8B: RESULTIS z80ADCrr(pc, n_a, n_e)           // ADC A,E
    CASE #x8C: RESULTIS z80ADCrr(pc, n_a, n_h)           // ADC A,H
    CASE #x8D: RESULTIS z80ADCrr(pc, n_a, n_l)           // ADC A,L
    CASE #x8E: RESULTIS z80ADCrir(pc, n_a, n_hl)         // ADC A,(HL)
    CASE #x8F: RESULTIS z80ADCrr(pc, n_a, n_a)           // ADC A,A

    CASE #x90: RESULTIS z80SUBr(pc, n_b)           // SUB B
    CASE #x91: RESULTIS z80SUBr(pc, n_c)           // SUB C
    CASE #x92: RESULTIS z80SUBr(pc, n_d)           // SUB D
    CASE #x93: RESULTIS z80SUBr(pc, n_e)           // SUB E
    CASE #x94: RESULTIS z80SUBr(pc, n_h)           // SUB H
    CASE #x95: RESULTIS z80SUBr(pc, n_l)           // SUB L
    CASE #x96: RESULTIS z80SUBir(pc, n_hl)         // SUB (HL)
    CASE #x97: RESULTIS z80SUBr(pc, n_a)           // SUB A
    CASE #x98: RESULTIS z80SBCrr(pc, n_a, n_b)           // SBC A,B
    CASE #x99: RESULTIS z80SBCrr(pc, n_a, n_c)           // SBC A,C
    CASE #x9A: RESULTIS z80SBCrr(pc, n_a, n_d)           // SBC A,D
    CASE #x9B: RESULTIS z80SBCrr(pc, n_a, n_e)           // SBC A,E
    CASE #x9C: RESULTIS z80SBCrr(pc, n_a, n_h)           // SBC A,H
    CASE #x9D: RESULTIS z80SBCrr(pc, n_a, n_l)           // SBC A,L
    CASE #x9E: RESULTIS z80SBCrir(pc, n_hl)        // SBC A,(HL)
    CASE #x9F: RESULTIS z80SBCrr(pc, n_a, n_a)           // SBC A,A

    CASE #xA0: RESULTIS z80ANDr(pc, n_b)           // AND B
    CASE #xA1: RESULTIS z80ANDr(pc, n_c)           // AND C
    CASE #xA2: RESULTIS z80ANDr(pc, n_d)           // AND D
    CASE #xA3: RESULTIS z80ANDr(pc, n_e)           // AND E
    CASE #xA4: RESULTIS z80ANDr(pc, n_h)           // AND H
    CASE #xA5: RESULTIS z80ANDr(pc, n_l)           // AND L
    CASE #xA6: RESULTIS z80ANDir(pc, n_hl)         // AND (HL)
    CASE #xA7: RESULTIS z80ANDr(pc, n_a)           // AND A
    CASE #xA8: RESULTIS z80XORr(pc, n_b)           // XOR B
    CASE #xA9: RESULTIS z80XORr(pc, n_c)           // XOR C
    CASE #xAA: RESULTIS z80XORr(pc, n_d)           // XOR D
    CASE #xAB: RESULTIS z80XORr(pc, n_e)           // XOR E
    CASE #xAC: RESULTIS z80XORr(pc, n_h)           // XOR H
    CASE #xAD: RESULTIS z80XORr(pc, n_l)           // XOR L
    CASE #xAE: RESULTIS z80XORir(pc, n_hl)         // XOR (HL)
    CASE #xAF: RESULTIS z80XORr(pc, n_a)           // XOR A

    CASE #xB0: RESULTIS z80ORr(pc, n_b)            // OR B
    CASE #xB1: RESULTIS z80ORr(pc, n_c)            // OR C
    CASE #xB2: RESULTIS z80ORr(pc, n_d)            // OR D
    CASE #xB3: RESULTIS z80ORr(pc, n_e)            // OR E
    CASE #xB4: RESULTIS z80ORr(pc, n_h)            // OR H
    CASE #xB5: RESULTIS z80ORr(pc, n_l)            // OR L
    CASE #xB6: RESULTIS z80ORir(pc, n_hl)          // OR (HL)
    CASE #xB7: RESULTIS z80ORr(pc, n_a)            // OR A
    CASE #xB8: RESULTIS z80CPr(pc, n_b)            // CP B
    CASE #xB9: RESULTIS z80CPr(pc, n_c)            // CP C
    CASE #xBA: RESULTIS z80CPr(pc, n_d)            // CP D
    CASE #xBB: RESULTIS z80CPr(pc, n_e)            // CP E
    CASE #xBC: RESULTIS z80CPr(pc, n_h)            // CP H
    CASE #xBD: RESULTIS z80CPr(pc, n_l)            // CP L
    CASE #xBE: RESULTIS z80CPir(pc, n_hl)          // CP (HL)
    CASE #xBF: RESULTIS z80CPr(pc, n_a)            // CP A

    CASE #xC0: RESULTIS z80RETcc(pc, cc_nz)        // RET NZ
    CASE #xC1: RESULTIS z80POPwr(pc, n_bc, 1)      // POP BC
    CASE #xC2: RESULTIS z80JP3ccnn(pc, cc_nz, getwb(pc+1)) // JP NZ,nn
    CASE #xC3: RESULTIS z80JP3nn(pc, getwb(pc+1))  // JP nn
    CASE #xC4: RESULTIS prbadop(pc)
    CASE #xC5: RESULTIS z80PUSHwr(pc, n_bc)        // PUSH BC
    CASE #xC6: RESULTIS z80ADDrr(pc, n_a, n_n)     // ADD A,n
    CASE #xC7: RESULTIS z80RSTn(pc, #x00)          // RST #x00
    CASE #xC8: RESULTIS z80RETcc(pc, cc_z)         // RET Z
    CASE #xC9: RESULTIS z80RET(pc)                 // RET
    CASE #xCA: RESULTIS z80JP3ccnn(pc, cc_z, getwb(pc+1)) // JP Z,nn
    CASE #xCB: RESULTIS doz80instrCB(pc)           // CB instructions
    CASE #xCC: RESULTIS prbadop(pc)
    CASE #xCD: RESULTIS prbadop(pc)
    CASE #xCE: RESULTIS z80ADC2rr(pc, n_a, getbb(pc+1)) // ADC A,n
    CASE #xCF: RESULTIS z80RSTn(pc, #x08)          // RST #x08

    CASE #xD0: RESULTIS z80RETcc(pc, cc_nc)        // RET NC
    CASE #xD1: RESULTIS z80POPwr(pc, n_de, 1)     // POP DE
    CASE #xD2: RESULTIS z80JP3ccnn(pc, cc_nc, getwb(pc+1)) // JP NC,nn
    CASE #xD3: RESULTIS prbadop(pc)
    CASE #xD4: RESULTIS prbadop(pc)
    CASE #xD5: RESULTIS z80PUSHwr(pc, n_de)        // PUSH DE
    CASE #xD6: RESULTIS z80SUB2n(pc,getbb(pc+1))   // SUB n
    CASE #xD7: RESULTIS z80RSTn(pc, #x10)          // RST #x10
    CASE #xD8: RESULTIS z80RETcc(pc, cc_c)         // RET C
    CASE #xD9: RESULTIS z80EXXw(pc)                // EXX
    CASE #xDA: RESULTIS z80JP3ccnn(pc, cc_c, getwb(pc+1)) // JP C,nn
    CASE #xDB: RESULTIS prbadop(pc)
    CASE #xDC: RESULTIS prbadop(pc)
    CASE #xDD: RESULTIS doz80instrXD(pc, n_ix)     // DD and FD instructions
    CASE #xDE: RESULTIS z80SBC2rn(pc, n_a, getbb(pc+1)) // SBC A,n
    CASE #xDF: RESULTIS z80RSTn(pc, #x18)          // RST #x18

    CASE #xE0: RESULTIS z80RETcc(pc, cc_po)        // RET PO
    CASE #xE1: RESULTIS z80POPwr(pc, n_hl)         // POP HL
    CASE #xE2: RESULTIS z80JP3ccnn(pc, cc_po, getwb(pc+1)) // JP PO,nn
    CASE #xE3: RESULTIS z80EXwirr(pc, n_sp, n_hl)   // EX (SP),HL
    CASE #xE4: RESULTIS prbadop(pc)
    CASE #xE5: RESULTIS z80PUSHwr(pc, n_hl)        // PUSH HL
    CASE #xE6: RESULTIS z80ANDr(pc, n_n)           // AND n
    CASE #xE7: RESULTIS z80RSTn(pc, #x20)          // RST #x20
    CASE #xE8: RESULTIS z80RETcc(pc, cc_pe)        // RET PE
    CASE #xE9: RESULTIS z80JPir(pc, n_hl)          // JP (HL)
    CASE #xEA: RESULTIS z80JP3ccnn(pc, cc_pe, getwb(pc+1)) // JP PE,nn
    CASE #xEB: RESULTIS z80EXwrr(pc, n_de, n_hl)   // EX DE,HL
    CASE #xEC: RESULTIS prbadop(pc)
    CASE #xED: RESULTIS doz80instrED(pc)
    CASE #xEE: RESULTIS z80XOR2n(pc, getbb(pc+1))  // XOR n
    CASE #xEF: RESULTIS z80RSTn(pc, #x28)          // RST #x28 (BRK instruction)

    CASE #xF0: RESULTIS z80RETcc(pc, cc_p)         // RET P
    CASE #xF1: RESULTIS z80POPwr(pc, n_af)         // POP AF
    CASE #xF2: RESULTIS z80JP3ccnn(pc, cc_p, getwb(pc+1)) // JP P,nn
    CASE #xF3: RESULTIS z80DI(pc)                  // DI  Disable INT
    CASE #xF4: RESULTIS prbadop(pc)
    CASE #xF5: RESULTIS z80PUSHwr(pc, n_af)        // PUSH AF
    CASE #xF6: RESULTIS z80OR2n(pc,getbb(pc+1))    // OR n
    CASE #xF7: RESULTIS z80RSTn(pc, #x30)          // RST #x30
    CASE #xF8: RESULTIS z80RETcc(pc, cc_m)         // RET M
    CASE #xF9: RESULTIS z80LDwrr(pc, n_sp, n_hl)   // LD SP,HL 
    CASE #xFA: RESULTIS z80JP3ccnn(pc, cc_m, getwb(pc+1)) // JP M,nn
    CASE #xFB: RESULTIS z80EI(pc)                  // EI  Enable INT
    CASE #xFC: RESULTIS prbadop(pc)
    CASE #xFD: RESULTIS doz80instrXD(pc, n_iy)     // DD and FD instructions
    CASE #xFE: RESULTIS z80CP2n(pc, getbb(pc+1))   // CP n
    CASE #xFF: RESULTIS z80RSTn(pc, #x38)          // RST #X38
}

AND doz80instrXD(pc, z) = VALOF // For op DD or FD
{ // Deal wilh DD and FD instructions,
  // If op=DD z = n_ix
  // If op=FD z = n_iy
  SWITCHON getbb(pc+1) INTO
  { DEFAULT:   RESULTIS prbadop(pc)
    CASE #x09: RESULTIS z80ADDwrr(pc, z, n_bc)               // ADD z,BC
    CASE #x19: RESULTIS z80ADDwrr(pc, z, n_de)               // ADD z,DE
    CASE #x21: RESULTIS z80LDw3rnn(pc, z, getwb(pc+1))       // LD z,nn
    CASE #x22: RESULTIS z80LD4innr(pc, getwb(pc+2), z)       // LD (nn),z
    CASE #x23: RESULTIS z80INCr(pc, z)                       // INC z
    CASE #x29: RESULTIS z80ADDw2rr(pc, z, z)                 // ADD z,z
    CASE #x2A: RESULTIS z80LDw4rnn(pc, z, getwb(pc+2))       // LD z,(nn)
    CASE #x2B: RESULTIS z80DEC2r(pc, z)                      // DEC z
    CASE #x34: RESULTIS z80INC3izd(pc, z, getsbb(pc+2))      // INC (z+d)
    CASE #x35: RESULTIS z80DEC3izd(pc, z, getsbb(pc+2))      // DEC (z+d)
    CASE #x36: RESULTIS z80LD4izdn(pc,z,getsbb(pc+2),getbb(pc+3)) // LD (z+d),n
    CASE #x39: RESULTIS z80ADDw2rr(pc, z, n_sp)              // ADD z,SP
    CASE #x43: RESULTIS z80LDrr(pc, n_inn, n_bc)             // LD (nn),BC
    CASE #x46: RESULTIS z80LD3rizd(pc, n_b, z, getsbb(pc+2)) // LD B,(z+d)
    CASE #x4B: RESULTIS z80LDw4rnn(pc, n_bc, getwb(pc+2))    // LD BC,(nn)
    CASE #x4E: RESULTIS z80LD3rizd(pc, n_c, z, getsbb(pc+2)) // LD C,(z+d)
    CASE #x53: RESULTIS z80LD4innr(pc,getwb(pc+2),n_de)      // LD (nn),DE
    CASE #x56: RESULTIS z80LD3rizd(pc, n_d, z, getsbb(pc+2)) // LD D,(z+d)
    CASE #x57: RESULTIS z80LDI2rr(pc, n_a,n_i)               // LD A.I
    CASE #x5B: RESULTIS z80LDw4rinn(pc, n_de, getwb(pc+2))   // LD DE,(nn)
    CASE #x5E: RESULTIS z80LD3rizd(pc, n_e, z, getsbb(pc+2)) // LD E,(z+d)
    CASE #x5F: RESULTIS z80LDRr(pc, n_a)                     // LD A.R
    CASE #x66: RESULTIS z80LD3rizd(pc, n_h, z, getsbb(pc+2)) // LD H,(z+d)
    CASE #x6E: RESULTIS z80LD3rizd(pc, n_l, z, getsbb(pc+2)) // LD L,(z+d)
    CASE #x70: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_b) // LD (z+d),B
    CASE #x71: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_c) // LD (z+d),C
    CASE #x72: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_d) // LD (z+d),D
    CASE #x73: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_e) // LD (z+d),E
    CASE #x74: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_h) // LD (z+d),H
    CASE #x75: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_l) // LD (z+d),L
    CASE #x77: RESULTIS z80LD3izdr(pc, z, getsbb(pc+2), n_a) // LD (z+d),A
    CASE #x7E: RESULTIS z80LD3rizd(pc, n_a, z, getsbb(pc+2)) // LD A,(z+d)
    CASE #x86: RESULTIS z80ADD3rizd(pc, n_a, z, getsbb(pc+2)) // ADD A,(z+d)
    CASE #x8E: RESULTIS z80ADC3rizd(pc, n_a, z, getsbb(pc+2)) // ADC A,(z+d)
    CASE #x96: RESULTIS z80SUB3rizd(pc, n_a, z, getsbb(pc+2)) // SUB A,(z+d)
    CASE #x9E: RESULTIS z80SBC3rizd(pc, n_a, z, getsbb(pc+2)) // SBC A,(z+d)
    CASE #xA2: RESULTIS z80INI2(pc)                           // INI
    CASE #xAA: RESULTIS z80IND2(pc)                           // IND
    CASE #xA6: RESULTIS z80AND3izd(pc, z, getsbb(pc+2))       // AND (z+d)
    CASE #xAE: RESULTIS z80XOR3izd(pc, z, getsbb(pc+2))       // XOR (z+d)
    CASE #xB2: RESULTIS z80INIR2(pc)                          // INIR
    CASE #xB6: RESULTIS z80OR3izd(pc,z,getsbb(pc+2))          // OR (z+d)
    CASE #xBA: RESULTIS z80INDR2(pc)                          // INDR
    CASE #xBE: RESULTIS z80CP3izd(pc,z,getsbb(pc+2))          // CP (z+d)
    CASE #xCB: RESULTIS doz80instrXDCB(pc, z)                 // XD CB instructions
    CASE #xE1: RESULTIS z80POPw2r(pc, z)                      // POP z
    CASE #xE3: RESULTIS z80EXw2irz(pc, n_sp, z)               // EX (SP),z
    CASE #xE5: RESULTIS z80PUSHw2r(pc, z)                     // PUSH z
    CASE #xE9: RESULTIS z80JP2ir(pc, z)                       // JP (z)
    CASE #xF9: RESULTIS z80LDw2rr(pc, n_sp, z)                // LD SP,z
  }
  RESULTIS 0
}

AND doz80instrED(pc) = VALOF
{ SWITCHON getbb(pc+1) INTO
  { DEFAULT:   RESULTIS prbadop(pc)
    CASE #x40: RESULTIS z80INrr(pc, n_b, n_ic)       // IN B,(C)
    CASE #x41: RESULTIS z80OUTrr(pc, n_ic, n_b)      // OUT (C),B
    CASE #x42: RESULTIS z80SBCw2rr(pc, n_hl, n_bc)   // SBC HL,BC
    CASE #x43: RESULTIS z80LDw4innr(pc, getwb(pc+2), n_bc) // LD (nn),BC
    CASE #x44: RESULTIS z80NEG2(pc)                  // NEG
    CASE #x45: RESULTIS z80RETN(pc)                  // RETN
    CASE #x46: RESULTIS z80IMk(pc, 0)                // IM 0
    CASE #x47: RESULTIS z80LDrr(pc, n_i, n_a)        // LD I,A
    CASE #x48: RESULTIS z80INrr(pc, n_c, n_ic)       // IN C,(C)
    CASE #x49: RESULTIS z80OUTrr(pc, n_ic, n_b)      // OUT (C),C
    CASE #x4A: RESULTIS z80ADCw2rr(pc, n_hl, n_bc)   // ADC HL,BC
    CASE #x4B: RESULTIS z80LDw4rinn(pc, n_bc, getwb(pc+2)) // LD BC,(nn)
    CASE #x4D: RESULTIS z80RETI(pc)                  // RETI
    CASE #x4F: RESULTIS z80LDrr(pc, n_r,n_a)         // LD R,A
    CASE #x50: RESULTIS z80INrr(pc, n_d, n_ic)       // IN D,(C)
    CASE #x51: RESULTIS z80OUTrr(pc, n_ic, n_d)      // OUT (C),D
    CASE #x52: RESULTIS z80SBCw2rr(pc, n_hl, n_de)   // SBC HL,DE
    CASE #x53: RESULTIS z80LDw4innr(pc, getwb(pc+2), n_de) // LD (nn),DE
    CASE #x56: RESULTIS z80IMk(pc, 1)                // IM 1  Set IN Mode 1
    CASE #x57: RESULTIS z80LDrr(pc, n_a, n_i)        // LD A,I
    CASE #x58: RESULTIS z80INrr(pc, n_e, n_ic)       // IN E,(C)
    CASE #x59: RESULTIS z80OUTrr(pc, n_ic, n_e)      // OUT (C),E
    CASE #x5A: RESULTIS z80ADCw2rr(pc, n_hl, n_de)   // ADC HL,DE
    CASE #x5B: RESULTIS z80LDw4rinn(pc, n_de, getwb(pc+2)) // LD DE,(nn)
    CASE #x5E: RESULTIS z80IMk(pc, 2)                // IM 2  Set INT Mode 2
    CASE #x5F: RESULTIS z80LDrr(pc, n_a, n_r)        // LD A,R
    CASE #x60: RESULTIS z80INrr(pc, n_h, n_ic)       // IN H,(C)
    CASE #x61: RESULTIS z80OUTrr(pc, n_ic, n_h)      // OUT (C),H
    CASE #x62: RESULTIS z80SBCw2rr(pc, n_hl, n_hl)   // SBC HL,HL
    CASE #x67: RESULTIS z80RRD2(pc)                  // RRD
    CASE #x68: RESULTIS z80INrr(pc, n_l, n_ic)       // IN L,(C)
    CASE #x69: RESULTIS z80OUTrr(pc, n_ic, n_l)      // OUT (C),L
    CASE #x6A: RESULTIS z80ADCw2rr(pc, n_hl, n_hl)   // ADC HL,HL
    CASE #x6F: RESULTIS z80RLD2(pc)                  // RLD
    CASE #x70: RESULTIS z80INrr(pc, n_f, n_ic)       // IN F,(C)
    CASE #x72: RESULTIS z80SBCw2rr(pc, n_hl, n_sp)   // SBC HL,SP
    CASE #x73: RESULTIS z80LDw4innr(pc, getwb(pc+2), n_sp) // LD (nn),SP
    CASE #x79: RESULTIS z80OUTrr(pc, n_ic, n_a)      // OUT (C),A
    CASE #x7A: RESULTIS z80ADCw2rr(pc, n_hl, n_sp)   // ADC HL,SP
    CASE #x7B: RESULTIS z80LDw4rinn(pc, n_sp, getwb(pc+2)) // LD SP,(nn)
    CASE #xA0: RESULTIS z80LDI2(pc)                  // LDI
    CASE #xA1: RESULTIS z80CPI(pc)                   // CPI
    CASE #xA2: RESULTIS z80INI2(pc)                  // INI
    CASE #xA3: RESULTIS z80OUTI2(pc)                 // OUTI
    CASE #xA8: RESULTIS z80LDD2(pc)                  // LDD
    CASE #xA9: RESULTIS z80CPD(pc)                   // CPD
    CASE #xAA: RESULTIS z80IND2(pc)                  // IND
    CASE #xAB: RESULTIS z80OUTD2(pc)                 // OUTD
    CASE #xB0: RESULTIS z80LDIR2(pc)                 // LDIR
    CASE #xB1: RESULTIS z80CPIR(pc)                  // CPIR
    CASE #xB2: RESULTIS z80INIR2(pc)                 // INIR
    CASE #xB3: RESULTIS z80OTIR2(pc)                 // OTIR
    CASE #xB8: RESULTIS z80LDDR2(pc)                  // LDDR
    CASE #xB9: RESULTIS z80CPDR(pc)                  // CPDR
    CASE #xBA: RESULTIS z80INDR2(pc)                  // INDR
    CASE #xBB: RESULTIS z80OTDR2(pc)                  // OTDR
  //CASE #xC1: RESULTIS z80MULUBrr(pc, n_a, n_b)     // MULUB A,B
                                                     // note C1+8*r
  //CASE #xC3: RESULTIS z80MULUWrr(pc, n_hl, n_bc)   // MULUW HL,BC
  //CASE #xF3: RESULTIS z80MULUWrr(pc, n_hl, n_sp)   // MULUW HL,SP
  }
  RESULTIS 0
}

AND doz80instrCB(pc) = VALOF
{ LET op = getbb(pc+1)
  LET bbb = op>>3 & 7   // bbb = 0 - 7
  LET rrr = op & 7      // rrr = B C D E H L (HL) A
  // 
  // CB 00+rrr        RLC rrr
  // CB 08+rrr        RRC rrr
  // CB 10+rrr        RL  rrr
  // CB 18+rrr        RR  rrr
  // CB 20+rrr        SLA rrr    
  // CB 28+rrr        SRA rrr
  // CB 30+rrr        Error
  // CB 38+rrr        SRL rrr

  // CB 01_bbb_rrr    BIT bbb,rrr
  // CB 10_bbb_rrr    RES bbb,rrr
  // CB 11_bbb_rrr    SET bbb,rrr

//writef("*ndoz80instrCB: op=%b8 bbb=%b3 rrr=%b3*n", op, bbb, rrr)
  SWITCHON op>>6 INTO
  { DEFAULT:
    CASE #b00:
      SWITCHON op & #b11111_000 INTO
      { DEFAULT:    RESULTIS zerror(pc, 2)

        CASE #x00:  RESULTIS z80RLC2r(pc, rrr)  // RLC rrr
        CASE #x08:  RESULTIS z80RRC2r(pc, rrr)  // RRC rrr
        CASE #x10:  RESULTIS z80RL2r (pc, rrr)  // RL  rrr
        CASE #x18:  RESULTIS z80RR2r (pc, rrr)  // RR  rrr
        CASE #x20:  RESULTIS z80SLA2r(pc, rrr)  // SLA rrr
        CASE #x28:  RESULTIS z80SRA2r(pc, rrr)  // SRA rrr
        CASE #x30:  RESULTIS zerror (pc, 2)
        CASE #x38:  RESULTIS z80SRL2r(pc, rrr)  // SRL rrr
      } 
    CASE #b01: RESULTIS z80BITbr(pc, bbb, rrr) // BIT bbb,rrr
    CASE #b10: RESULTIS z80RESbr(pc, bbb, rrr) // RES bbb,rrr
    CASE #b11: RESULTIS z80SETbr(pc, bbb, rrr) // SET bbb,rrr
  }
}

AND doz80instrXDCB(pc, r) = VALOF
{ // Deal with instructions:
  // DD CB d op=00-4F Shifts, r = n_ix
  // FD CB d op=00-4F Shifts, r = n_iy


LET d  = getsbb(pc+2)
  LET op = getbb(pc+3)
  LET bbb = op>>3 & #b111
  LET rrr = op    & #b111

  // XD CB d 06                 RLC (r+d)        r = IX or IY
  // XD CB d 0E                 RRC (r+d)        r = IX or IY
  // XD CB d 16                 RL  (r+d)        r = IX or IY
  // XD CB d 1E                 RR  (r+d)        r = IX or IY
  // XD CB d 26                 SLA (r+d)        r = IX or IY
  // XD CB d 2E                 SRA (r+d)        r = IX or IY
  // XD CB d 30                 S?? 
  // XD CB d 3E                 SRL (r+d)        r = IX or IY

  // XF CB d op=40-7F =40+8*b+r BIT b,(r+d)      r = IX or IY
  // XF CB d op=80-BF =80+8*b+r RES b,(r+d)      r = IX or IY
  // XF CB d op=C0-FF =C0+8*b+r SET b,(r+d)      r = IX or IY

  SWITCHON op>>6 INTO
  { DEFAULT:
    CASE #00:
      SWITCHON op INTO
      { DEFAULT:   RESULTIS zerror(pc, 2)
        CASE #x06: // XD CB d 06
	           RESULTIS z80RLCizd(pc, r, d)   // RLC (r+d)
        CASE #x0E: // XD CB d 0E
	           RESULTIS z80RRC4izd(pc, r, d)  // RRC (r+d)
        CASE #x16: // XD CB d 16
	           RESULTIS z80RL4izd(pc, r, d)   // RL (r+d)
        CASE #x1E: // XD CB d 1E
	           RESULTIS z80RR4izd(pc, r, d)   // RR (r+d)
        CASE #x26: // XD CB d 26
	           RESULTIS z80SLA4izd(pc, r, d)  // SLA (r+d)
        CASE #x2E: // XD CB d 2E
	           RESULTIS z80SRA4izd(pc, r, d)  // SRA (r+d)
        CASE #x36: // XD CB d 36
	           RESULTIS zerror(pc, 4)         // ???
        CASE #x3E: // XD CB d 3E
	           RESULTIS z80SRL4izd(pc, r, d)  // SRL (r+d)
      } 
    CASE #b01: RESULTIS z80BIT4bizd(pc, bbb, r, d) // BIT b,(r+d)
    CASE #b10: RESULTIS z80RES4bizd(pc, bbb, r, d) // RES b,(r+d)
    CASE #b11: RESULTIS z80SET4bizd(pc, bbb, r, d) // SET b,(r+d)
  }
}

//xxx

AND z80DJNZ2d(pc,d) = VALOF
{ TEST pflag
  THEN { writef("DJNZ %n", d)
         RESULTIS 2
       }
  ELSE { notyet("JNZ2d")
	 r_pc := pc+2
         RESULTIS 0
       }
}

AND z80JR2ccd(pc,cc,d) = VALOF
{ TEST pflag
  THEN { writef("JR %s,%n", cc2str(cc), d)
         RESULTIS 2
       }
  ELSE { r_pc := evalcc(cc) -> pc+2+d, pc+2
         RESULTIS 0
       }
}

AND z80JP3nn(pc,nn) = VALOF
{ LET nn = getwb(pc+1)
  TEST pflag
  THEN { writef("JP %n", nn)
         RESULTIS 3
       }
  ELSE { //notyet("JP3nn")
         r_pc := nn
         RESULTIS 0
       }
}

AND z80JP3ccnn(pc,cc,nn) = VALOF
{ TEST pflag
  THEN { writef("JP %s,%n", cc2str(cc), nn)
         RESULTIS 3
       }
  ELSE { r_pc := evalcc(cc) -> nn, pc+3
         RESULTIS 0
       }
}

AND z80JR2d(pc, d) = VALOF
{ TEST pflag
  THEN { writef("JR %n", d)
         RESULTIS 2
       }
  ELSE { r_pc := pc+2+d
         RESULTIS 0
       }
}

AND z80JPir(pc,r) = VALOF TEST pflag // r is always n_hl
THEN { writef("JP (%n)", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("JPir")
       r_pc := getr(r)
       RESULTIS 0
     }
     
AND z80JP2ir(pc,r) = VALOF TEST pflag // r is either n_ix or n_iy
THEN { writef("JP (%n)", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("JP2ir")
       r_pc := getr(r)
       RESULTIS 0
     }
     
AND z80RSTn(pc, addr) = VALOF SWITCHON addr INTO
{ DEFAULT:
    TEST pflag
    THEN { writef("RSTn   RST #x%x2", addr)
           RESULTIS 1
	 }
    ELSE { notyet("RSTn")
           RESULTIS 0
         }


  CASE #x28:
    // This lists or executes the RST #x28 instruction.
    // This is the BRK instruction used in the implementation
    // of breakpoints.
    // When executed it sets the faultcode to 2 and returns
    // without advancing r_pc.
    TEST pflag
    THEN { writef("BRK")
           RESULTIS 1
	 }
    ELSE { RESULTIS 2    // Cause fault code 2
         }

  CASE #x38:
  { // This lists or executes the RST instruction that causes a
    // subroutine jump to location addr in page 0. If pflag=TRUE
    // it lists the instruction returning its size in byte.
    // If pflag=FALSE it executes the instruction pointed toby r_pc
    // and return a fault code. If the instruction did not generate
    // fault the result is zero.
    // If addr is #x38 this subroutine looks at the bytes following
    // the RST #x38 instruction to decide what to do. The inline
    // arguments given to this subroutine are in the subsequent bytes.
    // Each inline argument is either an 8 bit value encoded by a
    // single byte, a 16 bit value encoded by a pair of bytes or an
    // unsigned 16 bit even numbers encoded by one or two bytes as
    // follows:

    // 0bbbbbbb          =>  00000000 0bbbbbbb
    // 1bbbbbbb cccccccc =>  0ccccccc cbbbbbbb

    // This encoding was done in bcplcgz80.b as follows:
  
    //  LET low7   =  (value)       &  #B0000000001111111
    //  LET high8  =  (value << 1)  &  #B1111111100000000

    //  TEST  high8 = 0  THEN  code_n_1( low7 )
    //  ELSE
    //  $(
    //      code_n_1( low7 + #B10000000 )
    //      code_n_1( high8 >> 8 )
    //  $)

    // Other arguments are placed in the stack and results may also be
    // placed there.
    // The first inline argument is a byte specifying which operation to
    // be perform. These operations are needed by the BCPL compiled code.
    // Some examples are:      
    //    0: m_apply
    //    3: m_fixup
    //    6: m_loadix
    //    9: m_loadiy

    LET mtype = getbb(pc+1)

    SWITCHON mtype INTO
    { DEFAULT: zerror("RST #x38 bad op=%n*n", mtype)
               RESULTIS 2
	     
      CASE  m_apply:
      { LET size = 3
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, arg & #x7F | getbb(pc+3)<<7 
        TEST pflag
        THEN { writef("APPLY %n", arg)
               RESULTIS size
	     }
        ELSE { //notyet("RST #x38 APPLY")
	       pptr := pptr + arg // The new P pointer
               putww(pptr, r_ix)
	       putww(pptr+1, r_pc+size)
	       r_ix := (pptr<<1)+128
	       r_pc := getr(n_hl)
	       RESULTIS 0
             }
      }

      CASE  m_fixup: TEST pflag
        THEN { writef("FIXUP")
	       RESULTIS 2
	     }
	ELSE { notyet("RST #x38 FIXUP")
	       r_pc := r_pc+2
               RESULTIS 0
	     }
      
      CASE  m_loadix:
      { // Push P!(arg+128)
        LET size = 3
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
      
        TEST pflag
        THEN { writef("LOADIX P%n", arg+128)
               RESULTIS size
	     }
        ELSE { notyet("RST #x38 LOADIX")
	       r_sp := r_sp-2
               putwb(r_sp, getww(pptr+arg+128))
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE  m_loadiy:
      { // Push G!arg
        LET size = 3
        LET arg = getbb(pc+2)
        //writef("*nloadlvix: size=%n arg=%n*n", size, arg)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
      
        TEST pflag
        THEN { writef("LOADIY G%n", arg+128)
               RESULTIS size
	     }
        ELSE { //notyet("RST #x38 LOADIY")
	       r_sp := r_sp-2
               putwb(r_sp, getww(gptr+arg+128))
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE m_storeix:
      { // This is used when storing a value in local 128 and above.
        // The distance in words between P128 and the destination local
	// is encoded after the RST #x38 STOREIX instruction and the
	// value to store is on the top of the SP stack.
	LET size = 3 // Will hold the byte size of the STOREIV instruction.
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
	       
        TEST pflag
        THEN { writef("STOREIX P%n", arg+128)
	       RESULTIS size
	     }
	ELSE { LET val = getwb(r_sp)
	       //notyet("RST #x38 STOREIX")
	       r_sp := r_sp+2
               //writef("*nstoreix: size=%n arg=%n*n", size, arg)
               //writef("ie store: %x4 in local P%n*n", val, arg+128)
	       putww(pptr+arg+128, val)
	       r_pc := r_pc+size
               RESULTIS 0
	     }
      }

      CASE m_storeiy:
      { // This is used when storing a value in global 128 and above.
        // The distance in words between G128 and the destination global
	// is encoded after the RST #x38 STOREIY  instruction and the
	// value to store is on the top of the SP stack.
	LET size = 3 // Will hold the byte size of the STOREIV instruction.
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
	       
        TEST pflag
        THEN { writef("STOREIY G%n", arg+128)
	       RESULTIS size
	     }
	ELSE { LET val = getwb(r_sp)
	       //notyet("RST #x38 STOREIY")
	       r_sp := r_sp+2
               //writef("*nstoreiy: size=%n arg=%n*n", size, arg)
               //writef("ie store: %x4 in global %n*n", val, arg+128)
	       putww(gptr+arg+128, val)
	       r_pc := r_pc+size
               RESULTIS 0
	     }
      }

      CASE m_setlink0:
        TEST pflag
        THEN { writef("SETLINK0")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 SETLINK0")
	       x_exx()
               r_pc := r_pc+2
               RESULTIS 0
	     }

      CASE m_setlink1:
        TEST pflag
        THEN { writef("SETLINK1")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 SETLINK1")
	       x_exx()
               putww(pptr+2, getr(n_hl))
               r_pc := r_pc+2
               RESULTIS 0
	     }
        RESULTIS 2
      
      CASE m_setlink2:
        TEST pflag
        THEN { writef("SETLINK2")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 SETLINK2")
	       x_exx()
               putww(pptr+2, getr(n_hl))
               putww(pptr+3, getr(n_de))
               r_pc := r_pc+2
               RESULTIS 0
	     }
        RESULTIS 2
      
      CASE m_setlink3:
        TEST pflag
        THEN { writef("SETLINK3")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 SETLINK3")
	       x_exx()
               putww(pptr+2, getr(n_hl))
               putww(pptr+3, getr(n_de))
               putww(pptr+4, getr(n_bc))
               r_pc := r_pc+2
               RESULTIS 0
	     }

      CASE m_return:
        TEST pflag
        THEN { writef("RETURN")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 RETURN")
	       r_ix := getww(pptr)
	       r_pc := getww(pptr+1)
	       pptr := (r_ix-128)>>1
	       IF r_pc=0 DO
	       { //writef("Returning from start with result %n*n", getr(n_hl))
	         longjump(finp, finl)
	       }
	       //writef("doRSTn returning zero*n")
               RESULTIS 0
	     }

      CASE m_finish:
        TEST pflag
        THEN { writef("FINISH")
               RESULTIS 2
             }
        ELSE { //notyet("RST #x38 FINISH")
	       longjump(finp, finl) // Exit from z80emu
	     }
      
      CASE m_loadlvix:
      { // Push ((IX-128)>>1) + arg
        LET size = 3
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
      
        TEST pflag
        THEN { writef("LOADLVIX @P%n", arg)
               RESULTIS size
	     }
        ELSE { //notyet("RST #x38 LOADLVIX")
	       r_sp := r_sp - 2
               putwb(r_sp, pptr+arg)
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE m_loadlviy:
      { // Push ((IY-128)>>1) + arg
        LET size = 3
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
      
        TEST pflag
        THEN { writef("LOADLVIY @G%n", arg)
               RESULTIS size
	     }
        ELSE { //notyet("RST #x38 LOADLVIY")
	       r_sp := r_sp - 2
               putwb(r_sp, gptr+arg)
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE m_multiply:
        TEST pflag
        THEN { writef("MULTIPLY")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 MULTIPLY")
	       //writef("%n ** %n => %n*n", a, b, a * b)
	       putwb(r_sp, a*b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_divide:
        TEST pflag
        THEN { writef("DIVIDE");
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 DIVIDE")
	       putwb(r_sp, a/b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_rem:
        TEST pflag
        THEN { writef("REM")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 REM")
	       putwb(r_sp, a REM b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_lshift:
        TEST pflag
        THEN { writef("LSHIFT")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 LSHIFT")
	       putwb(r_sp, (a<<b)&#xFFFF)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_rshift:
        TEST pflag
        THEN { writef("RSHIFT")
               RESULTIS 2
	     }
        ELSE { LET sh = getwb(r_sp)
	       LET w  = getwb(r_sp+2)
	       //notyet("RST #x38 RSHIFT")
	       r_sp := r_sp+2
	       putwb(r_sp, w>>sh)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_eq:
        TEST pflag
        THEN { writef("EQ")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 EQ")
	       putwb(r_sp, a=b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_ne:
        TEST pflag
        THEN { writef("NE")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 NE")
	       putwb(r_sp, a~=b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_ls:
        TEST pflag
        THEN { writef("LS")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 LS")
	       putwb(r_sp, a<b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_gr:
        TEST pflag
        THEN { writef("GR")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       //writef("m_gr: a=%n b=%n a>b => %n*n", a, b, a>b)
	       r_sp := r_sp+2
	       //notyet("RST #x38 GR")
	       putwb(r_sp, a>b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_le:
        TEST pflag
        THEN { writef("LE")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 LE")
	       putwb(r_sp, a<=b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_ge:
        TEST pflag
        THEN { writef("GE")
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 GE")
	       putwb(r_sp, a>=b)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_rdivide:
        TEST pflag
        THEN { writef("RDIVIDE");
               RESULTIS 2
	     }
        ELSE { LET b = signextw(getwb(r_sp))
	       LET a = signextw(getwb(r_sp+2))
	       r_sp := r_sp+2
	       //notyet("RST #x38 RDIVIDE")
	       putwb(r_sp, b/a)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
       
      CASE m_rrem:
        TEST pflag
        THEN { writef("RREM")
               RESULTIS 2
	     }
        ELSE { notyet("RST #x38 RREM")
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_rlshift:
        TEST pflag
        THEN { writef("RLSHIFT")
               RESULTIS 2
	     }
        ELSE { LET b = getwb(r_sp)
	       LET a = getwb(r_sp+2)
	       //notyet("RST #x38 RLSHIFT")
	       r_sp := r_sp+2
	       putwb(r_sp, b<<a)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_rrshift:
        TEST pflag
        THEN { writef("RRSHIFT")
               RESULTIS 2
	     }
        ELSE { LET b = getwb(r_sp)
	       LET a = getwb(r_sp+2)
	       //notyet("RST #x38 RRSHIFT")
	       r_sp := r_sp+2
	       putwb(r_sp, b>>a)
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
      CASE m_abs:
        TEST pflag
        THEN { writef("ABS")
               RESULTIS 2
	     }
        ELSE { LET a = getwb(r_sp)
	       //notyet("RST #x38 ABS")
	       putwb(r_sp, ABS(signextw(a)))
	       r_pc := r_pc+2
	       RESULTIS 0
             }
      
        RESULTIS 2

      CASE m_linsearch: //abort(1002)
      { LET size = 3
        LET n = getbb(pc+2)
        IF n > #x7F DO size, n := 4, (n & #x7F) + (getbb(pc+3)<<7) 
        //       n K1 L1 ... Kn Ln Ldefault
        //       n is represented by one or two bytes but the case constants
	//       and label including the default label are two bytes long.
        //       The total size of the instruction is size+4n+2.
//UNLESS pflag DO
//{ writef("*nCalling abort at CASE m_linsearch*n"); abort(1000) }

        TEST pflag
        THEN { writef("LINSEARCH")
               RESULTIS size+4*n+2
	     }
        ELSE { LET p = r_pc+size
	       LET de = getr(n_de)
	       //notyet("RST #38    LINSEARCH n=%n*n", n)
	       //writef("LINSEARCH: n=%n de=%n pc=%n*n", n, de, r_pc)
	       //FOR i = 1 TO n DO
	       //{ writef("p=%i6 %i6  %i6*n", p, getwb(p), getwb(p+2))
		// p := p+4
	       //}
	       //writef("p=%i6 default %i6*n", p, getwb(p))
	       //abort(1001)
	       //p := r_pc+size
	       //abort(1000)
	       FOR i = 1 TO n DO
	       { //writef("Comparing %i6 with %i6 at %i6*n", de, getwb(p), p)
	         IF de=getwb(p) DO
	         { r_pc := getwb(p+2)
		   //writef("Jumping to a CASE label at %n*n", r_pc)
		   RESULTIS 0
		 }
		 p := p+4
	       }
	       r_pc := getwb(p)
               //writef("Jump to DEFAULT: at %n*n", r_pc)
	       RESULTIS 0
             }
      }

      CASE m_sys:  // This is only used in the body of the function sys
                   // which is:
	  	   //    RST #x38  m_setlink3
		   //    RST #x38  m_sys
		   //    RST #x38  m_return
		   // It calls z80sys() with pptr, gptr setup and the
		   // first three arguments, if present, are accessible
		   // by getr(n_hl), getr(n_de) and getr(n_bc).
        TEST pflag
        THEN { writef("SYS")
               RESULTIS 2
	     }
        ELSE { //notyet("RST #x38 SYS")
	       z80sys()
               RESULTIS 0
	     }

    }
  }
}

AND z80sys() BE SWITCHON getr(n_hl) INTO
{ DEFAULT:
    writef("*nERROR in z80sys: args %n %n %n %n*n",
         getr(n_hl), getr(n_de), getr(n_bc), getww(pptr+5))
    r_pc := r_pc+2
    RETURN

  CASE Sys_sawrch: // sawrch
    wrch(getr(n_de))
    r_pc := r_pc+2
    RETURN

  CASE Sys_tracing: // Start single stepping
    r_pc := r_pc+2
    writef("*nStarting single stepping*n")
    indentpos := 0
    brkstep := FALSE
    singlestep := TRUE
    tracing := TRUE
    RETURN
}

AND prbadop(pc) = VALOF
{ prstate()
  writef("*nInstruction at pc=%n not yet implemented: %x2 %x2 %x2 %x2*n",
           pc, getbb(pc), getbb(pc+1), getbb(pc+2), getbb(pc+3))
  val :=val+1
  longjump(recp, recl)
}

AND r2str(r) = VALOF SWITCHON r INTO
{ DEFAULT:    writef("r2str: Bad r=%n*n", r)
abort(999)
              RESULTIS "??"

  CASE n_a:   RESULTIS "A"
  CASE n_f:   RESULTIS "F"
  CASE n_b:   RESULTIS "B"
  CASE n_c:   RESULTIS "C"
  CASE n_d:   RESULTIS "D"
  CASE n_e:   RESULTIS "E"
  CASE n_h:   RESULTIS "H"
  CASE n_l:   RESULTIS "L"
  CASE n_af:  RESULTIS "AF"
  CASE n_bc:  RESULTIS "BC"
  CASE n_de:  RESULTIS "DE"
  CASE n_hl:  RESULTIS "HL"
  CASE n_sp:  RESULTIS "SP"
  CASE n_pc:  RESULTIS "PC"
  CASE n_af1: RESULTIS "AF'"
  CASE n_bc1: RESULTIS "BC'"
  CASE n_de1: RESULTIS "DE'"
  CASE n_hl1: RESULTIS "HL'"
  CASE n_ic:  RESULTIS "(C)"
  CASE n_ibc: RESULTIS "(BC)"
  CASE n_ide: RESULTIS "(DE)"
  CASE n_ihl: RESULTIS "(HL)"
  CASE n_isp: RESULTIS "(SP)"
  CASE n_ix:  RESULTIS "IX"
  CASE n_iy:  RESULTIS "IY"
}

AND cc2str(cc) = VALOF SWITCHON cc INTO
{ DEFAULT:    writef("Unknown condition code %n*n", cc)
              RESULTIS "??"
  CASE cc_c:   RESULTIS "C"
  CASE cc_m:   RESULTIS "M"
  CASE cc_nz:  RESULTIS "NZ"
  CASE cc_nc:  RESULTIS "NC"
  CASE cc_p:   RESULTIS "P"
  CASE cc_pe:  RESULTIS "PE"
  CASE cc_po:  RESULTIS "PO"
  CASE cc_z:   RESULTIS "Z"
}

AND getr(rn) = VALOF SWITCHON rn INTO
{ DEFAULT:    writef("*ngetr: Bad register number %n, %s*n", rn, r2str(rn))
              RESULTIS 0

  CASE n_a:   RESULTIS r_a & #xFF
  CASE n_b:   RESULTIS r_b & #xFF
  CASE n_c:   RESULTIS r_c & #xFF
  CASE n_d:   RESULTIS r_d & #xFF
  CASE n_e:   RESULTIS r_e & #xFF
  CASE n_h:   RESULTIS r_h & #xFF
  CASE n_l:   RESULTIS r_l & #xFF
  CASE n_f:   RESULTIS r_f & #xFF
  
  CASE n_bc:  RESULTIS r_b<<8 | r_c & #xFF
  CASE n_de:  RESULTIS r_d<<8 | r_e & #xFF
  CASE n_hl:  RESULTIS r_h<<8 | r_l & #xFF
  CASE n_af:  RESULTIS r_a<<8 | r_f & #xFF
  
  CASE n_bc1: RESULTIS r_b1<<8 | r_c1 & #xFF
  CASE n_de1: RESULTIS r_d1<<8 | r_e1 & #xFF
  CASE n_hl1: RESULTIS r_h1<<8 | r_l1 & #xFF
  CASE n_af1: RESULTIS r_a1<<8 | r_f1 & #xFF
    
  CASE n_ibc: RESULTIS getbb(r_b<<8 | r_c & #xFF)
  CASE n_ide: RESULTIS getbb(r_d<<8 | r_e & #xFF)
  CASE n_ihl: RESULTIS getbb(r_h<<8 | r_l & #xFF)
  
  CASE n_ix:  RESULTIS r_ix
  CASE n_iy:  RESULTIS r_iy
  CASE n_sp:  RESULTIS r_sp
  CASE n_pc:  RESULTIS r_pc
  CASE n_isp: RESULTIS getbb(r_sp)
}

AND putr(rn, val) BE SWITCHON rn INTO
{ DEFAULT:    writef("*nBad register number %n, %s*n", rn, r2str(rn))
              RETURN

  CASE n_a:   r_a := val; RETURN
  CASE n_b:   r_b := val; RETURN
  CASE n_c:   r_c := val; RETURN
  CASE n_d:   r_d := val; RETURN
  CASE n_e:   r_e := val; RETURN
  CASE n_h:   r_h := val; RETURN
  CASE n_l:   r_l := val; RETURN
  CASE n_f:   r_f := val; f2flags();  RETURN
  
  CASE n_af:  r_a, r_f := val>>8, val & #xFF; f2flags(); RETURN
  CASE n_bc:  r_b, r_c := val>>8, val & #xFF; RETURN
  CASE n_de:  r_d, r_e := val>>8, val & #xFF; RETURN
  CASE n_hl:  r_h, r_l := val>>8, val & #xFF; RETURN
  
  
  CASE n_ibc: putwb(r_b<<8 | r_c & #xFF, val); RETURN
  CASE n_ide: putwb(r_d<<8 | r_d & #xFF, val); RETURN
  CASE n_ihl: putwb(r_h<<8 | r_l & #xFF, val); RETURN
  
  CASE n_ix:  r_ix := val;      RETURN
  CASE n_iy:  r_iy := val;      RETURN
  CASE n_sp:  r_sp := val;      RETURN
  CASE n_isp: putwb(r_sp, val); RETURN
}

AND zerror(pc, n) = VALOF
{ writef("*nUnimplemented instruction at %n: ", pc)
  FOR i = 0 TO 2 DO writef(" %x2", getbb(pc+i))
  //pflag := TRUE
  //n := prz80instr(pc)
  newline()
  abort(999)
  RESULTIS n
}

AND notyet(str,a,b,c) BE
{ // This prints an error message and
  // returns the size of the offending insruction.
  // It is only called when trying to execute the instruction
  // at location r_pc.
  LET size = 0
  writef("*nInstruction is not yet implemented at %n: ", r_pc)
  FOR i = 0 TO 3 DO writef("%x2 ", getbb(r_pc+i))
  writef("   ")
  size := prz80instr(r_pc)
  writef("*nin function z80"); writef(str, a,b,c)
  newline()
  r_pc := r_pc+size
  faultcode := 99
  singlestep := TRUE
  tracing := TRUE
  //writef("*nCalling   longjump(recp=%n, recl=%n)*n",recp, recl)
//  longjump(recp, recl)
}


AND z80ADC2rr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADC2rr")
       putr(r, x_adc(getr(r), getr(s), r_fc))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80ADCw2rr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADCw2rr")
       putr(r, x_adc(getr(r), getr(s), r_fc))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADC3rizd(pc,r,z, d) = VALOF TEST pflag
THEN { writef("ADC %s,(%s+%n)", r2str(r), r2str(z), d)
       RESULTIS 3
     }
ELSE { notyet("ADC3rizd")
       putr(r, x_adc(getr(r), getbb(getr(z)+d), r_fc))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80ADDrn(pc, r, n) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), n)
       RESULTIS 1
     }
ELSE { notyet("ADDrn")
       putr(r, x_add(getr(r), n))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADDrr")
       putr(r, x_add(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADDrir(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,(%s)", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADDrir")
       putr(r, x_add(getr(r), getbb(getr(s))))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADDwrr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { //notyet("ADDwrr")
       putr(r, x_addw(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADDw2rr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { notyet("ADDw2rr")
       putr(r, x_addw(getr(r), getr(s)))
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80ADD3rizd(pc, r, z, d) = VALOF TEST pflag
THEN { writef("ADD %s,", r2str(r)); prizd(z,d)
       RESULTIS 3
     }
ELSE { notyet("ADD33izd")
       putr(r, x_add(getr(r), getbb(getr(z)+d)))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80ANDr(pc,r) = VALOF TEST pflag
THEN { writef("AND %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("ANDr")
       r_a := x_and(r_a, getr(r))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ANDir(pc,r) = VALOF TEST pflag
THEN { writef("AND (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("ANDir")
       r_a := x_and(r_a, getr(r))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80AND3izd(pc, z, d) = VALOF TEST pflag
THEN { writef("AND "); prizd(z,d)
       RESULTIS 3
     }
ELSE { notyet("AND3izd")
       putr(z, x_and(r_a, getbb(getr(z)+d)))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80XORr(pc,r) = VALOF TEST pflag
THEN { writef("XOR %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("XORr")
       r_a := x_xor(r_a, getr(r))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80XORir(pc,r) = VALOF TEST pflag
THEN { writef("XOR (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("XORir")
       r_a := x_xor(r_a, getbb(getr(r)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80XOR2n(pc,n) = VALOF TEST pflag
THEN { writef("XOR %n", n)
       RESULTIS 2
     }
ELSE { notyet("OR2n")
       putr(n_a, x_xor(r_a, n))
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80XOR3izd(pc,z) = VALOF
{ LET d = getbb(pc+2)
  TEST pflag
  THEN { writef("XOR "); prizd(z, d)
         RESULTIS 3
       }
  ELSE { notyet("XOR3izd")
         putr(n_a, x_xor(r_a, getwb(getr(z)+d)))
         r_pc := r_pc+3
         RESULTIS 0
       }
}

AND z80CCF(pc) = VALOF TEST pflag
THEN { writef("CCP")
       RESULTIS 1
     }
ELSE { notyet("CCF")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80CCP(pc) = VALOF TEST pflag
THEN { writef("CCP")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("CCP")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80CPL(pc) = VALOF TEST pflag
THEN { writef("CPL")
       RESULTIS 1
     }
ELSE { //notyet("CPL")
       r_a := (~r_a) & #xFF
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80CPr(pc, r) = VALOF TEST pflag
THEN { writef("CP %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("CPr")
       x_sub(getr(r), r_a)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80CPir(pc, r) = VALOF TEST pflag
THEN { writef("CP (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("CPir")
       x_sub(getbb(getr(r)), r_a)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80DAA(pc) = VALOF TEST pflag
THEN { writef("DAA")
       RESULTIS 1
     }
ELSE { notyet("DAA")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80DAAr(pc, r) = VALOF TEST pflag
THEN { writef("DAA %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("DAAr")
       r_pc := pc+1
       RESULTIS 0
     }

AND z80DECr(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("DECr")
       putr(r, getr(r)-1)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80DEC2r(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("DEC2r")
       putr(r, getr(r)-1)
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80DECir(pc, r) = VALOF TEST pflag
THEN { writef("DEC (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("DECir")
       putbb(getr(r), getbb(getr(r))-1)
       RESULTIS 0
     }

AND z80DEC3izd(pc, r) = VALOF TEST pflag
THEN { writef("DEC (%s)", r2str(r))
       RESULTIS 3
     }
ELSE { notyet("DEC3izd")
       putbb(getr(r), getbb(getr(r))-1)
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80DECwr(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("DECwr")
       putr(r, getr(r)-1)
       r_pc := pc+1
       RESULTIS 0
     }

AND z80DI(pc) = VALOF TEST pflag
THEN { writef("DI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("DI")
       RESULTIS 0
     }

AND z80EI(pc) = VALOF TEST pflag
THEN { writef("EI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("EI")
       RESULTIS 0
     }

AND z80EXXw(pc) = VALOF TEST pflag
THEN { writef("EXX")
       RESULTIS 1
     }
ELSE { x_exx(pc)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80EXwrr(pc, r, s) = VALOF TEST pflag
THEN { writef("EX %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { LET t = getr(r)
       // This implements one of
       // EB:      EX DE,HL
       // 08:      EX AF,AF'
       notyet("EXwrr")
       putr(s, getr(r))
       putr(r, t)
       r_pc := r_pc+1
       RESULTIS 0
     }
     
AND z80EXwirr(pc, r, s) = VALOF TEST pflag
THEN { writef("EX (%s),%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { LET t = getr(r)
       // This implements
       // E3:      EX (SP),HL
       notyet("EXwirr")
       putr(s, getr(r));
       putr(r, t)
       r_pc := r_pc+1
       RESULTIS 0
     }
     
AND z80EXw2irz(pc, r, z) = VALOF TEST pflag
THEN { writef("EX (%s),%s", r2str(r), r2str(z))
       RESULTIS 2
     }
ELSE { LET t = getr(r)
       // This implements one of
       // DD E3:   EX (SP),IX
       // FD E3:   EX (SP),IY     
       notyet("EX2irz")
       putr(z, getr(r));
       putr(r, t)
       r_pc := r_pc+2
       RESULTIS 0
     }
     
AND z80HALT(pc) = VALOF TEST pflag
THEN { writef("HALT")
       RESULTIS 1
     }
ELSE { notyet("HALT")
       writef("*nHALT instructions encountered*n")
       abort(999)
       r_pc := pc+1
       RESULTIS 0
     }

AND z80INCr(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("INCr")
       putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80INCir(pc, r) = VALOF TEST pflag
THEN { writef("INC (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { notyet("INCir")
       putbb(getr(r), getbb(getr(r)) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80INCwr(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("INCwr")
       putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80LDrir(pc, r, s) = VALOF
{ TEST pflag THEN { writef("LD %s,(%s)", r2str(r), r2str(s))
                    RESULTIS 1
		  }
             ELSE { //notyet("LDrir")
                    putr(r, getbb(getr(s)))
		    r_pc := r_pc+1
                    RESULTIS 0
		  }
}

AND z80LDirr(pc, r, s) = VALOF
{ TEST pflag THEN { writef("LD (%s),%s", r2str(r), r2str(s))
                    RESULTIS 1
		  }
             ELSE { //notyet("LDirr")
                    putbb(getr(r), getr(s))
		    r_pc := r_pc+1
                    RESULTIS 0
		  }
}


AND z80LDD2(pc) = VALOF
{ TEST pflag THEN { writef("LDD")
                    RESULTIS 2
		  }
             ELSE { notyet("LDD2")
	            r_pc := r_pc+2
                    RESULTIS 0
		  }
}


AND z80LDDR2(pc) = VALOF
{ TEST pflag THEN { writef("LDDR")
                    RESULTIS 2
		  }
             ELSE { notyet("LDDR2")
	            r_pc := r_pc+2
                    RESULTIS 0
		  }
}


AND z80LDI2(pc) = VALOF
{ TEST pflag THEN { writef("LDI")
                    RESULTIS 2
		  }
             ELSE { notyet("LDI2")
	            r_pc := r_pc+2
                    RESULTIS 0
		  }
}


AND z80LDIR2(pc) = VALOF
{ TEST pflag THEN { writef("LDIR")
                    RESULTIS 2
		  }
             ELSE { notyet("LDIR")
	            r_pc := r_pc+2
                    RESULTIS 0
		  }
}


AND z80LD2irn(pc, r, n) = VALOF
{ TEST pflag THEN { writef("LD (%s),%n", r2str(r), n)
                    RESULTIS 2
		  }
             ELSE { //notyet("LD2irn")
                    putbb(getr(r), n)
	            r_pc := r_pc+2
                    RESULTIS 0
		  }
}


AND z80LD4innr(pc, nn, z) = VALOF
{ TEST pflag THEN { writef("LD (%n),%s", nn, r2str(z))
                    RESULTIS 4
		  }
             ELSE { notyet("LD4innr")
                    putwb(nn, getr(z))
		    r_pc := r_pc+4
                    RESULTIS 0
		  }
}

     
AND z80LD3innr(pc, nn, z) = VALOF
{ TEST pflag THEN { writef("LD (%n),%s", nn, r2str(z))
                    RESULTIS 3
		  }
             ELSE { notyet("LD3innr")
                    putwb(nn, getr(z))
		    r_pc := r_pc+3
                    RESULTIS 0
		  }
}

     

AND z80LDw3ir(pc, nn, r) = VALOF
{ TEST pflag THEN { writef("LD (%n),%s", nn, r2str(r))
                    RESULTIS 3
		  }
             ELSE { notyet("LDw3ir")
                    putwb(nn, getr(r))
                    RESULTIS 0
		  }
}

AND z80LD4izdn(pc, z, d, n) = VALOF
{ // Instruction: LD (IX+d),n  or  LD (IY+d),n
  // Code:        DD 36  d  n  or  FD 36  d  n
  // z=n_ix or n_iy
  TEST pflag
  THEN { writef("LD "); prizd(z,d); writef(",%n", n)
         RESULTIS 4
       }
  ELSE { LET a = getr(z)+d
         notyet("LD4izdn")
         putbb(a, n)
	 r_pc := pc+4
	 RESULTIS 0
       }
}

AND z80LDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, getr(s))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80LDwrr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("LDwrr")
       putr(r, getr(s))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80LDw2rr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { notyet("LDwrr")
       putr(r, getr(s))
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80LDw4rr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 4
     }
ELSE { notyet("LDw4rr")
       putr(r, getr(s))
       r_pc := r_pc+4
       RESULTIS 0
     }

AND z80LDw4innr(pc, nn, r) = VALOF TEST pflag
THEN { writef("LD (%n),%s", nn, r2str(r))
       RESULTIS 4
     }
ELSE { //notyet("LDw4innr")
       putwb(nn, getr(r))
       r_pc := r_pc+4
       RESULTIS 0
     }

AND z80LD2rn(pc, r, n) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), n)
       RESULTIS 2
     }
ELSE { //notyet("LD2rn")
       putr(r, n)
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80NEG2(pc) = VALOF TEST pflag
THEN { writef("NEG")
       RESULTIS 2
     }
ELSE { notyet("NEG2")
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80NOP(pc) = VALOF TEST pflag
THEN { writef("NOP")
       RESULTIS 1
     }
ELSE { notyet("NOP")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80POPwr(pc, r) = VALOF TEST pflag
THEN { writef("POP %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("POPwr")
       putr(r, getwb(r_sp))
       r_sp := r_sp+2
       r_pc := pc+1
       RESULTIS 0
     }

AND z80POPw2r(pc, r) = VALOF TEST pflag
THEN { writef("POP %s", r2str(r))
       RESULTIS 2
     }
ELSE { putr(r, getwb(r_sp))
       r_sp := r_sp+2
       r_pc := pc+2
       RESULTIS 0
     }

AND z80RR4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("RR "); prizd(z, d)
       RESULTIS 4
     }
ELSE { notyet("RR4izd")
       r_pc := r_pc+4
       RESULTIS 0
     }

AND z80RRA(pc) = VALOF TEST pflag
THEN { writef("RRA")
       RESULTIS 1
     }
ELSE { notyet("RRA")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80RRD2(pc) = VALOF TEST pflag
THEN { writef("RRD")
       RESULTIS 2
     }
ELSE { notyet("RRD2")
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80RRC2r(pc, r) = VALOF TEST pflag
THEN { writef("RRC %s", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("RRC2r")
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80RRC4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("RRC "); prizd(z, d)
       RESULTIS 4
     }
ELSE { notyet("RRC4izfr")
       r_pc := r_pc+4
       RESULTIS 0
     }

AND z80RRCA(pc) = VALOF TEST pflag
THEN { writef("RRCA")
       RESULTIS 1
     }
ELSE { notyet("RRCA")
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80RET(pc) = VALOF TEST pflag
THEN { writef("RET")
       RESULTIS 1
     }
ELSE { notyet("RET")
       r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETI(pc) = VALOF TEST pflag
THEN { writef("RETI")
       RESULTIS 2
     }
ELSE { notyet("RETI")
       r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETcc(pc, cc) = VALOF TEST pflag
THEN { writef("RET")
       RESULTIS 1
     }
ELSE { notyet("RETcc")
       TEST evalcc(cc)
       THEN r_pc := getwb(r_sp)
       ELSE r_pc := pc+1
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETN(pc) = VALOF TEST pflag
THEN { writef("RETN")
       RESULTIS 1
     }
ELSE { notyet("RETN")
       r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80SUBr(pc, r) = VALOF
{ TEST pflag
  THEN { writef("SUB %s", r2str(r))
         RESULTIS 1
       }
  ELSE { notyet("SUBr")
         r_a := x_sub(r_a, getr(r))
         r_pc := pc+1
         RESULTIS 0
       }
}

AND z80SUBir(pc, r) = VALOF
{ TEST pflag
  THEN { writef("SUB (%s)", r2str(r))
         RESULTIS 1
       }
  ELSE { notyet("SUBir")
         r_a := x_sub(r_a, getbb(getr(r)))
         r_pc := pc+1
         RESULTIS 0
       }
}

AND z80SUB2n(pc, n) = VALOF
{ TEST pflag
  THEN { writef("SUB %n", n)
         RESULTIS 2
       }
  ELSE { notyet("SUB2n")
         r_a := x_sub(r_a, n)
         r_pc := pc+2
         RESULTIS 0
       }
}

AND z80SUB3rizd(pc, r, z, d) = VALOF TEST pflag
THEN { writef("SUB %s,", r2str(r)); prizd(z,d)
       RESULTIS 3
     }
ELSE { notyet("SUB3rizd")
       putr(r, x_sub(getr(r), getbb(getr(z)+d)))
       r_pc := r_pc+3
       RESULTIS 0
     }


AND z80SBC3rizd(pc, r, z, d) = VALOF TEST pflag
THEN { writef("SBC %s,", r2str(r)); prizd(z,d)
       RESULTIS 3
     }
ELSE { notyet("SBC3rizd")
       putr(r, x_sub(getr(r), getbb(getr(z)+d)))
       r_pc := r_pc+3
       RESULTIS 0
     }


AND z80SBCw2rr(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { //notyet("SBCw2rr")
       putr(r, x_sbcw(getr(r), getr(s), r_fc))
       r_pc := pc+2
       RESULTIS 0
     }

AND z80SBCrr(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("SBCrr")
       putr(r, x_sbc(getr(r), getr(s), r_fc))
       r_pc := pc+1
       RESULTIS 0
     }

AND z80SBCrir(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,(%s)", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("SBCrir")
       putr(r, x_sbc(getr(r), getbb(getr(s)), r_fc))
       r_pc := pc+1
       RESULTIS 0
     }

AND z80SBC2rn(pc, r, n) = VALOF TEST pflag
THEN { writef("SBC %s,%n", r2str(r), n)
       RESULTIS 2
     }
ELSE { notyet("SBC2rn")
       putr(r, x_sbc(getr(r), n, r_fc))
       r_pc := pc+2
       RESULTIS 0
     }

AND z80SBCw4rr(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,%s", r2str(r), r2str(s))
       RESULTIS 4
     }
ELSE { notyet("SBCw4rr")
       putr(r, x_sbcw(getr(r), getr(s), r_fc))
       r_pc := r_pc+4
       RESULTIS 0
     }

AND z80ANDn(pc) = VALOF TEST pflag
THEN { writef("AND %n", getbb(pc+1))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("ANDn")
       RESULTIS 0
     }

AND z80ANDrir(pc,r, s) = VALOF TEST pflag
THEN { writef("AND %s,(%s)", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ANDrir")
       putr(n_a, x_and(r_a, getbb(getr(r))))
       r_pc := pc+1
       RESULTIS 0
     }

AND z80BITbr(pc, b, r) = VALOF TEST pflag
THEN { writef("BIT %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("BITbr")
       RESULTIS 0
     }

AND z80BIT4bizd(pc, b, z, d) = VALOF TEST pflag
THEN { writef("BIT %n,"); prizd(z, d)
       RESULTIS 4
     }
ELSE { notyet("BIT4bizd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80SET4bizd(pc, b, z, d) = VALOF TEST pflag
THEN { writef("SET %n,"); prizd(z, d)
       RESULTIS 4
     }
ELSE { notyet("SET4bizd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80RES4bizd(pc, b, z, d) = VALOF TEST pflag
THEN { writef("RES %n,"); prizd(z, d)
       RESULTIS 4
     }
ELSE { notyet("RES4bizd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80CPD(pc) = VALOF TEST pflag
THEN { writef("CPD")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("CPD")
       RESULTIS 0
     }

AND z80CPDR(pc) = VALOF TEST pflag
THEN { writef("CPDR")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("CPDR")
       RESULTIS 0
     }

AND z80CPI(pc) = VALOF TEST pflag
THEN { writef("CPI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("CPI")
       RESULTIS 0
     }

AND z80CPIR(pc) = VALOF TEST pflag
THEN { writef("CPIR")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet("CPIR")
       RESULTIS 0
     }

AND z80CP3izd(pc, r, z, d) = VALOF
  TEST pflag
  THEN { writef("LD "); prizd(z,d)
         RESULTIS 3
       }
  ELSE { notyet("CP3izd")
         x_cp(r, getwb(getr(z)+d))
         r_pc := r_pc+3
         RESULTIS 0
       }

AND z80CP2n(pc) = VALOF TEST pflag
THEN { writef("CP %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { notyet("CP2n")
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80DJNZo(pc) = VALOF TEST pflag
THEN { writef("DJNZo %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("DJNZo")
       RESULTIS 0
     }

AND z80IMk(pc, k) = VALOF TEST pflag
THEN { writef("IM %n", k)
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("IMk")
       RESULTIS 0
     }

AND z80IND2(pc) = VALOF TEST pflag
THEN { writef("IND")
       RESULTIS 2
     }
ELSE { notyet("IND2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80INDR2(pc) = VALOF TEST pflag
THEN { writef("INDR")
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("INDR")
       RESULTIS 0
     }

AND z80INI2(pc) = VALOF TEST pflag
THEN { writef("INI")
       RESULTIS 2
     }
ELSE { notyet("INT2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80INIR2(pc) = VALOF TEST pflag
THEN { writef("INIR")
       RESULTIS 2
     }
ELSE { notyet("INTR2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80INC3izd(pc, r) = VALOF TEST pflag
THEN { writef("INC "); prizd(r, getsbb(pc+2))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("INC3izd")
       RESULTIS 0
     }

AND z80INrr(pc, r, s) = VALOF TEST pflag
THEN { writef("IN %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("INrr")
       RESULTIS 0
     }

AND z80LDrn(pc, r) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), getbb(pc+1))
       RESULTIS 2
     }
ELSE { notyet("LDrn")
       putr(r, getbb(pc+1))
       r_pc := r_pc+2 & #xFFFF
       RESULTIS 2
     }

AND z80LDw3rnn(pc, r, nn) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), nn)
       RESULTIS 3
     }
ELSE { //notyet("LDw3rnn")
       putr(r, nn)
       r_pc := r_pc+3
       IF bkval & bkval=nn RESULTIS 25
       RESULTIS 0
     }

AND z80LDw3innr(pc, nn, r) = VALOF TEST pflag
THEN { writef("LD (%n),%s", nn, r2str(r))
       RESULTIS 3
     }
ELSE { //notyet("LDw3innr")
       putwb(nn, getr(r))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80LDw3rinn(pc, r, nn) = VALOF TEST pflag
THEN { writef("LD %s,(%n)", r2str(r), nn)
       RESULTIS 3
     }
ELSE { putr(r,getwb(nn))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80ORr(pc, r) = VALOF TEST pflag
THEN { writef("OR %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("ORr")
       r_a := x_or(r_a, getr(r))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ORir(pc, r) = VALOF TEST pflag
THEN { writef("OR (%s)", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("ORir")
       r_a := x_or(r_a, getbb(getr(r)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80OR2n(pc, n) = VALOF TEST pflag
THEN { writef("OR %n", n)
       RESULTIS 1
     }
ELSE { notyet("OR2n")
       r_a := x_or(r_a, n)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80OR3izd(pc, z) = VALOF
{ LET d = getsbb(pc+2)
  TEST pflag
  THEN { writef("OR "); prizd(z, d)
         RESULTIS 3
       }
  ELSE { notyet("ORizd")
         r_a := x_or(r_a, getbb(getr(z)+d))
         r_pc := r_pc+3
         RESULTIS 0
       }
}

AND z80OTDR2(pc) = VALOF TEST pflag
THEN { writef("OTDR")
       RESULTIS 2
     }
ELSE { notyet("OTDR2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80OTIR2(pc) = VALOF TEST pflag
THEN { writef("OTIR")
       RESULTIS 2
     }
ELSE { notyet("OTIR2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80OUTrr(pc, r, s) = VALOF TEST pflag
THEN { writef("OUT %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("OUTrr")
       RESULTIS 0
     }

AND z80OUTI2(pc) = VALOF TEST pflag
THEN { writef("OUTI")
       RESULTIS 2
     }
ELSE { notyet("OUTI2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80OUTD2(pc) = VALOF TEST pflag
THEN { writef("OUTD")
       RESULTIS 2
     }
ELSE { notyet("OUTD2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80PUSHr(pc, r) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("PUSHr")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80PUSHwr(pc, r) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 1
     }
ELSE { //notyet("PUSHwr")
       // For PUSH AF BC DE and HL
       r_sp := r_sp-2
       putwb(r_sp, getr(r))
       r_pc := pc+1
       RESULTIS 0
     }

AND z80PUSHw2r(pc, r) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 2
     }
ELSE { //notyet("PUSHw2r"
       // For PUSH IX and PUSH IY
       r_sp := r_sp-2
       putwb(r_sp, getr(r))
       r_pc := pc+2
       RESULTIS 0
     }

AND z80RRCr(pc, r) = VALOF TEST pflag
THEN { writef("RRC %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("RRCr")
       RESULTIS 0
     }

AND z80RESbr(pc, b, r) = VALOF TEST pflag
THEN { writef("RES %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("RESbr")
       RESULTIS 0
     }

AND z80RLC2r(pc, r) = VALOF TEST pflag
THEN { writef("RLC %s", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("RLC2r")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80RLCA(pc) = VALOF TEST pflag
THEN { writef("RLCA")
       RESULTIS 1
     }
ELSE { notyet("RLCr")
       r_pc := pc+1
       RESULTIS 0
     }

AND z80RL2r(pc, r) = VALOF TEST pflag
THEN { writef("RL %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("RL2r")
       RESULTIS 0
     }

AND z80RLA(pc) = VALOF TEST pflag
THEN { writef("RLA")
       RESULTIS 1
     }
ELSE { notyet("RLA")
       r_pc := pc+1
       RESULTIS 0
     }

AND z80RLD2(pc) = VALOF TEST pflag
THEN { writef("RLD")
       RESULTIS 2
     }
ELSE { notyet("RLD2")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80RL4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("RL "); prizd(z,d)
       RESULTIS 4
     }
ELSE { notyet("RLD4izd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80RR2r(pc, r) = VALOF TEST pflag
THEN { writef("RR %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET res = 0
       //notyet("RR2r pc=%n r=%s =%x2", pc, r2str(r), getr(r))
       res := x_rr(getr(r))
//writef("z80RR2r: res=%x2 after putr(r,res) getr(r)=%x2*n", res, getr(r));
//abort(7723)
       putr(r, res)
//writef("z80RR2r: After putr(%n,%x2) getr(r)=%x2*n", r, res, getr(r)); 
       //indentpos := 0
       //prstate()
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80SLA2r(pc, r) = VALOF TEST pflag
THEN { writef("SLA %s*n", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("SLA2r")
       r_pc := 2
       RESULTIS 0
     }

AND z80SLA4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("SLA "); prizd(z,d)
       RESULTIS 4
     }
ELSE { notyet("SLA4izd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80SCF(pc) = VALOF TEST pflag
THEN { writef("SLF")
       RESULTIS 1
     }
ELSE { notyet("SLF")
       r_pc := pc+1
       RESULTIS 0
     }

AND z80SRA2r(pc, r) = VALOF TEST pflag
THEN { writef("SRA %s", r2str(r))
       RESULTIS 2
     }
ELSE { notyet("SRAr")
       r_pc := pc+2
       RESULTIS 0
     }

AND z80SRA4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("SRA "); prizd(z,d)
       RESULTIS 4
     }
ELSE { notyet("SRA4izd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80SRL2r(pc, r) = VALOF TEST pflag
THEN { writef("SRL %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET w = getr(r)
       //notyet("SRL2r")
       r_fc := w&1
       w := (w>>1)&#x7F
       r_fval := w
       r_fh := 0
       r_fpval := w
       r_fn := 0
       putr(r, w)
       r_pc := r_pc+2
       RESULTIS 0
     }


AND z80SRL4izd(pc, z, d) = VALOF TEST pflag
THEN { writef("SRL "); prizd(z,d)
       RESULTIS 4
     }
ELSE { notyet("SRL4izd")
       r_pc := pc+4
       RESULTIS 0
     }

AND z80SETbr(pc, b, r) = VALOF TEST pflag
THEN { writef("SET %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet("SETbr")
       RESULTIS 0
     }


AND z80ADCrizd(pc, r, x, d) = VALOF TEST pflag
THEN { writef("ADC %s,", r2str(r)); prizd(x, d)
       RESULTIS 3
     }
ELSE { LET a = getr(x) + d
       notyet("ADCrizd")
       putr(r, x_adc(getr(r), getbb(a), r_fc))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80ADCrr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADCrr")
       putr(r, x_adc(getr(r), getr(s), r_fc))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADCrir(pc, r, s) = VALOF TEST pflag
THEN { writef("ADC %s,(%s)", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet("ADCrir")
       putr(r, x_adc(getr(r), getbb(getr(s)), r_fc))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80LD3izdr(pc, r, d, s) = VALOF TEST pflag
THEN { writef("LD "); prizd(r, d); writef(",%s", r2str(s)); 
       RESULTIS 3
     }
ELSE { LET a = getr(r)+d
       putbb(a, getr(s))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80LDrizd(pc, r, s, d) = VALOF TEST pflag
THEN { writef("LD %s,", r2str(r)); prizd(s, d)
       RESULTIS 3
     }
ELSE { LET a = getbb(getr(s)+d)
//db("Loading reg %s with (%s+%n) value %n*n", r2str(r), r2str(s), d, a)
       putr(r, a)
       r_pc := pc+3
       //db("Returning from z80LDrizd: r_pc=%n*n", r_pc)
       RESULTIS 0
     }

AND z80LD3rinn(pc, r, nn) = VALOF
{ // For eg: LD A,(nn)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), nn)
         RESULTIS 3
       }
  ELSE { notyet("LD3rinn")
         putr(r, getwb(nn))
         r_pc := pc+3
         RESULTIS 0
       }
}

AND z80LD3rizd(pc, r, z, d) = VALOF
  TEST pflag
  THEN { writef("LD %s,", r2str(r)); prizd(z,d)
         RESULTIS 3
       }
  ELSE { //notyet("LD3rizd")
         putr(r, getbb(getr(z)+d))
         r_pc := r_pc+3
         RESULTIS 0
       }

AND z80LDw4rinn(pc, r, nn) = VALOF
{ // For eg: DD 2A nn nn       LD IX,(nn)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), nn)
         RESULTIS 4
       }
  ELSE { //notyet("LDw4rinn")
         putr(r, getwb(nn))
         r_pc := r_pc+4
         RESULTIS 0
       }
}

AND z80LDw4rnn(pc, r, nn) = VALOF
{ // For eg: DD 2A nn nn       LD IX,(nn)
  TEST pflag
  THEN { writef("LD %s,#x%x4", r2str(r), nn)
         RESULTIS 4
       }
  ELSE { notyet("LDw4rnn")
         putr(r, nn)
         r_pc := r_pc+4
         RESULTIS 0
       }
}

AND z80LDI2rr(pc, r, s) = VALOF
{ TEST pflag
  THEN { writef("LDI %s,%s", r2str(r), r2str(s))
         RESULTIS 2
       }
  ELSE { notyet("LDIr")
         putr(r, r_i)
         r_pc := r_pc+2
         RESULTIS 0
       }
}

AND z80LDRr(pc, r) = VALOF
{ TEST pflag
  THEN { writef("LDI %s", r2str(r))
         RESULTIS 3
       }
  ELSE { notyet("LDRr")
         putr(r, r_r)
         r_pc := r_pc+2 & #xFFFF
         RESULTIS 0
       }
}

AND z80RLCizd(pc, r, d) = VALOF TEST pflag
THEN { writef("RLC "); prizd(r, d)
       RESULTIS 4
     }
ELSE { r_pc := pc+notyet("RLCizd")
       RESULTIS 0
     }

// Arithemetic and logical functions

AND evalcc(cc) = VALOF SWITCHON cc INTO
{ DEFAULT:    writef("evalcc: Bad cc=%n*n", cc)
              abort(999)
	      RESULTIS FALSE
  CASE cc_m:  RESULTIS r_fval<0
  CASE cc_c:  RESULTIS r_fc>0
  CASE cc_nc: RESULTIS r_fc=0
  CASE cc_nz: RESULTIS r_fval~=0
  CASE cc_p:  RESULTIS r_fval>=0
  CASE cc_pe: RESULTIS evenparity(r_fpval)=0
  CASE cc_po: RESULTIS evenparity(r_fpval)=0
  CASE cc_z:  RESULTIS r_fval=0

}

AND x_adc(a, b) = VALOF
{ LET res = a + b
  IF r_fc DO res := res + 1
  RESULTIS res
}

AND x_adcw(a, b) = VALOF
{ LET res = a + b
  IF r_fc DO res := res + 1
  RESULTIS res
}

AND x_add(a, b) = VALOF
{ LET res = a + b
  
  RESULTIS res
}

AND x_addw(a, b) = VALOF
{ LET res = signextw(a + b)
  r_fval := res
  r_fh := (a&#xFFF)  + (b&#xFFF) >#xFFF  -> 1, 0 // Carry out of bit 11
  r_fc := (a&#xFFFF) + (b&#xFFFF)>#xFFFF -> 1, 0 // Carry out of bit 15
  RESULTIS res
}

AND x_sub(a, b) = VALOF
{ LET res = signextb(a - signextb)
  r_fval := res
  r_fh := ((a&#xF)  - (b&#xF) >#x10) >0 -> 1, 0 // Carry out of bit 7
  r_fc := ((a&#xFF) - (b&#xFF)>#x100)>0 -> 1, 0 // Carry out of bit 7
  RESULTIS res
}

AND x_and(a, b) = VALOF
{ LET res = a & b
  r_fval := res
  r_fh := 0
  r_fpval := res
  r_fn := 0
  r_fc := 0
  RESULTIS res
}

AND x_cp(a, b) = VALOF
{ x_sub(a, b)
  RESULTIS 0
}

AND x_or(a, b) = VALOF
{ LET res = (a | b) & #xFF
  r_fval := signextb(res)
  r_fh := 0
  r_fpval := res
  r_fn := 0
  r_fc := 0
  RESULTIS res
}

AND x_rr(a) = VALOF
{ LET res = (a>>1) & #x7F
  IF r_fc DO res := res+#x80
  r_fval := signextb(res)
  r_fh := 0
  r_fpval := res
  r_fn := 0
  r_fc := a&#x01
  RESULTIS r_fval
}

AND x_sbc(a, b, c) = VALOF
{ r_fval := signextb(a) - signextb(b) - c
  TEST -#x80 <= r_fval <= #x7F
  THEN r_fpval := 1                           // No overflow
  ELSE r_fval, r_fpval := signextb(r_fval), 0 // Overflow 
  r_fh := 0
  r_fn := 0
  r_fc := (a&#xFF) - (b&#xFF) - c < 0 -> 1, 0
  RESULTIS r_fval
}

AND x_sbcw(a, b, c) = VALOF
{ r_fval := signextw(a) - signextw(b) - c
  TEST -#x8000 <= r_fval <= #x7FFF
  THEN r_fpval := 1                           // No overflow
  ELSE r_fval, r_fpval := signextw(r_fval), 0 // Overflow 
  r_fh := 0
  r_fn := 0
  r_fc := (a&#xFFFF) - (b&#xFFFF) - c < 0 -> 1, 0
  RESULTIS r_fval
}

AND x_xor(a, b) = VALOF
{ LET res = signextw(a XOR b)
  r_fval := res
  r_fh := 0
  r_fpval := res
  r_fn := 0
  r_fc := 0
  RESULTIS res
}

AND x_exx(pc) BE
{ LET t = ?
  t := r_a1; r_a1:=r_a; r_a:=t 
  t := r_f1; r_f1:=r_f; r_f:=t 
  t := r_b1; r_b1:=r_b; r_b:=t 
  t := r_c1; r_c1:=r_c; r_c:=t 
  t := r_d1; r_d1:=r_d; r_d:=t 
  t := r_e1; r_e1:=r_e; r_e:=t 
  t := r_h1; r_h1:=r_h; r_h:=t 
  t := r_l1; r_l1:=r_l; r_l:=t
}

AND db(mes, a,b,c,d,e) BE
{ writef("*ndb: "); writef(mes, a,b,c,d,e)
  writef("*nr_pc=%n pptr=%n faultcode=%n singlestep=%n brkstep=%n mem=%n pflag=%n",
            r_pc,   pptr,   faultcode,   singlestep,   brkstep,   mem,   pflag)
  writef("*n  100: ")
  FOR a = 100 TO 118 DO writef(" %x2", getbb(a))
  FOR i = 0 TO 19 DO
  { IF i MOD 10 = 0 DO writef("*n%i5: ", pptr+i)
    writef(" %x4", getww(pptr+i))
  }
  newline()
  indentpos := 0
  prstate()
  abort(1000)
}
